/*-----------------------------------------------------------------------------
*  Copyright 2007-2010 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
*  
*  
*  Author  Jon Cox  <jcox@alfresco.com>
*  File    HrefBearingRequestPathNameMatcher.java
*----------------------------------------------------------------------------*/

package org.alfresco.repo.avm.util;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;

import org.alfresco.util.NameMatcher;

/**
 *  NameMatcher that matches a list of extensions (case insensitively)
 *  associated with the request path portion of of a URL.    For example,
 *  Given the URL:   "http://example.com/moo/cow.html", the extention is
 *  "html" (note: the '.' character is elided).
 */
public class HrefBearingRequestPathNameMatcher implements NameMatcher, Serializable 
{

    /**
     * The extensions to match.
     */
    HashMap<String,String> ext_;
    
    /**
     * Default constructor.
     */
    public HrefBearingRequestPathNameMatcher()
    {
        ext_ = new HashMap<String,String>();
    }
    
    /**
     * Set the extensions case insensitively.  
     * The empty string corresponds to files no extention.
     *
     * @param extensions
     */
    public void setExtensions(List<String> extensions)
    {
        // Be careful to ignore '.' within "extension"

        for (String extension : extensions)
        { 
            int bad_dot_index = extension.lastIndexOf('.');
            if ( bad_dot_index >=0)
            {
                // TODO: warn here?

                extension = extension.substring( bad_dot_index + 1);
            }

            ext_.put( extension.toLowerCase(), null );
        }
    }
    
    /**
    *  Returns true if the URL requst path (e.g.:  /moo/cow.html)
    *  contains an extension (e.g.:  "html") included in the list
    *  of matching extensions.  
    */
    public boolean matches(String path)
    {
        String ext  = "";

        int dir_index = path.lastIndexOf('/');

        if ( dir_index >= 0)
        {
            String leaf   = path.substring(dir_index+1);
            int ext_index = leaf.lastIndexOf('.');

            if ( ext_index >=0 )
            {
                ext = leaf.substring(ext_index + 1).toLowerCase();
            }
        }

        return ext_.containsKey( ext );
    }
}


/*

#!/usr/bin/perl -w
#------------------------------------------------------------------------------
# Copyright 2007-2010 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
# 
# 
# Author  Jon Cox  <jcox@alfresco.com>
# File    get_filext_info
#
# See also   http://filext.com/alphalist.php
#
# Use  Scrape file-to-program association info from filext.com, in
#      order to help determine when GET vs HEAD should be performed.
#      Many lists of file-to-type associations exist, but to my
#      knowledge, http://filext.com is the most comprehensive and
#      accurate.  In in the end, this list is just used to create
#      an "include list" of what file types will be fetched via GET.
#
# Background:
#      For performance reasons, it's important to be able to figure out 
#      whether to do HEAD or a GET when validating links.  When you
#      don't need embedded links, you can always do HEAD, but if the result
#      might contain links you need to validate, the alternatives are:
#
#          [A]  Do GET when unsure  + make configurable GET exclude list
#          [B]  Do HEAD when unsure + make configurable GET include list
#          [C]  User-based config regarding what to do when unsure.
#
#      Analysis:
#
#         Option [A] is more complete, but the exclude list would be 
#         very hard to manage.   For example, as of 2007 an exclude 
#         list would need to contain over 20k entries to keep up with 
#         the association data at filext.com (see list below).   Also, 
#         when a previously unknown non text/html media type is hit,
#         the file size could easily be quite large... thereby needlessly
#         impairing network performance for everyone on the lan, and
#         delaying overall linkvalidation info for a snapshot considerably.
#
#         Option [B] has the potential for giving the user a false sense
#         of security by leading them to believe that a page has been 
#         validated when in fact it has not (due to not being on the
#         "include" list).   It might take a long time for someone to
#         notice.
#
#         Option [C] clearly provides the most flexibility, but with
#         that comes extra training cost, docs, and the potential for
#         making a mistake if the "right answer" is fairly clear-cut.
#
#         Option [B] make the most sense because [A] seems to have great 
#         potential for causing unnoticed & needless system-wide network 
#         performance problems, and delaying average-case validation checks.
#         Because [B] seems to dominate [A] so strongly, [C] is pointless.
#
#
# Aside:  How many lines would the equivalent Java program have been? :)
#         Cheers, Larry!  Much joy!
#
#------------------------------------------------------------------------------
use Unicode::String;
foreach $i ( "A" .. "Z"  )
{
    my $url  = "http://filext.com/alphalist.php?extstart=%5E" . $i;
    my $data = `lwp-request \'$url\'`;
    $data = Unicode::String::latin1( $data ); 
    $data    =~ s/.*?Thank you//s;            # See "ethical use" comment below
    $data    =~ s{</table>.*$}{}s;

    my @tuples = 
       $data =~ m{<td [^>]*><a [^>]*>([^>]*)</a></td>.*?<td>(.*?)</td>}gms;

    # must use an array, not a hash -- some files have multiple associations!
    for (my $i = 0; $i < @tuples; $i+=2)
    {
        printf ("%-20s %s\n", $tuples[$i], $tuples[$i+1]);
    }
}
#------------------------------------------------------------------------------
#
# Ethical/legal/politeness issue:
#
#         The folks at filext.com don't want this list displayed elsehere.
#         They say:
#
#                  Please honor the copyright on this list. 
#                  Do not copy it and display it elsewhere. 
#                  Thank you.
#
#         Presumably, this is so they can get ad revenue for the free
#         service they provide.  That seems quite reasonable.  While 
#         compendiums, tables, lists, charts, etc., are not subject
#         to copyright law, I'd still like to be respectful of the 
#         "sweat of their brow", and cause them no harm.  
#
#         Is inclusion of file association data in the source code of 
#         a program considered "a display" in the sense they intended?  
#         I don't think so.   Nobody in their right mind would choose to 
#         look at *this* file as the easiest way to get file/type data.
#         For example, none of the supplementary text is here, just the 
#         raw ext-to-file data.  Hence, I doubt I'll drive anybody from 
#         them, so no ad revenue will be lost, and no harm will be done. 
#
#         If there's a real objection from filext.com, I'd be happy to
#         remove the output of the program below from this file, out of 
#         courtesy.   Again, thanks for the outstanding website site!
#
#------------------------------------------------------------------------------

  Here are the results of the last time this program was run
  (Tue Jun 26 12:58:18 EDT 2007).  
    

    A                    Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    A                    ADA Program 
    A                    Assembly Source Code (Macintosh) 
    A                    UNIX Library 
    A                    Free Pascal Archive File for Linux or DOS Version (FPC Development Team)
    A00                  Archive Section 
    A01                  ARJ Multi-volume Compressed Archive (can be 01 to 99) (also see .000) (can be 01 to 99) (also see .000) 
    A01                  OzWin CompuServe E-mail/Forum Access SYSOP File 
    A01                  Archive Section 
    A02                  Archive Section 
    A02                  OzWin CompuServe E-mail/Forum Access SYSOP File 
    A03                  Archive Section 
    A03                  annotare ava 04 Project File (annotare.net)
    A03                  OzWin CompuServe E-mail/Forum Access SYSOP File 
    A04                  OzWin CompuServe E-mail/Forum Access SYSOP File 
    A04                  Archive Section 
    A05                  OzWin CompuServe E-mail/Forum Access SYSOP File 
    A05                  Archive Section 
    A06                  Archive Section 
    A06                  Lotto Pro 2002 Smart Number Ticket 
    A06                  OzWin CompuServe E-mail/Forum Access SYSOP File 
    A07                  OzWin CompuServe E-mail/Forum Access SYSOP File 
    A07                  Archive Section 
    A08                  Archive Section 
    A08                  OzWin CompuServe E-mail/Forum Access SYSOP File 
    A09                  Archive Section 
    A09                  OzWin CompuServe E-mail/Forum Access SYSOP File 
    A1                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    A1                   Free Pascal Archive File for GO321v1 Platform (FPC Development Team)
    A10                  OzWin CompuServe E-mail/Forum Access SYSOP File 
    A11                  AOL Instant Messenger (AIM) Graphic (America Online, Inc.)
    A2                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    A21                  Soundweb Firmware File (BSS Audio)
    A2A                  APLASCII EISPACK 
    A2A                  IBM Voice Type Language Vocabels File 
    A2A                  Sharp APL Workspace Export File 
    A2B                  A2B Player Playlist (DigitalPhono.com)
    A2BPS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    A2C                  Alice Scene File (Carnegie Mellon University)
    A2DOM                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    A2M                  AdLib Tracker 2 
    A2M                  AudioToMidi Settings 
    A2PROBPS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    A2W                  Alice Program (World) File (Carnegie Mellon University)
    A3                   Unrelated Inventions Audiotools Show Log 
    A3                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    A31                  Authorware Ver. 3.x Library (Adobe Systems Incorporated)
    A3D                  Amapi 3D Modeling (Eovia)
    A3K                  Yamaha A3000 Sampler File (Yamaha Corporation of America)
    A3L                  Authorware Ver. 3.x Library (Adobe Systems Incorporated)
    A3M                  Authorware Unpackaged Macintosh FIle (Adobe Systems Incorporated)
    A3W                  Authorware Unpackaged Windows File (Adobe Systems Incorporated)
    A41                  Authorware Ver. 4.x Library (Adobe Systems Incorporated)
    A4L                  Authorware Ver. 4.x Library (Adobe Systems Incorporated)
    A4M                  Authorware Unpackaged Macintosh File (Adobe Systems Incorporated)
    A4P                  Authorware File Without Runtime (Adobe Systems Incorporated)
    A4R                  Authorware Packaged Course File (Adobe Systems Incorporated)
    A4W                  Authorware Unpackaged Windows File (Adobe Systems Incorporated)
    A51                  Authorware Ver. 5.x Library (Adobe Systems Incorporated)
    A52                  Atari 5200 Game Image 
    A5L                  Authorware Ver. 5.x Library (Adobe Systems Incorporated)
    A5W                  Alpha Five Web Publishing Page (Alpha Software, Inc.)
    A5W                  Authorware Unpackaged Windows File (Adobe Systems Incorporated)
    A60                  Roland Garros Tennis MVTS File 
    A64                  Artist 64 
    A65                  Authorware v6.5 File (Adobe Systems Incorporated)
    A6P                  Authorware Application (Adobe Systems Incorporated)
    A78                  Atari 7800 ROM Image 
    A8                   Cubicomp PictureMaker 24bit 3D Animation 
    A86                  A86 Assembler Source Code 
    AA                   Audio Book 
    AA                   PROGNOSIS Automated Analyst Document File 
    AAA                  Softphrase Automation File 
    AAA                  App Cam Automation 
    AAA                  Xtal Backup 
    AAA                  Sybase SQLAnywhere Temp File 
    AAB                  Authorware Binary (Adobe Systems Incorporated)
    AAC                  MPEG-2 Advanced Audio Coding File 
    AAF                  Advanced Authoring Format File 
    AAM                  Authorware Shocked File (Map) (Adobe Systems Incorporated)
    AAO                  America's Army Map 
    AAP                  Apollo Advanced Playlist 
    AAP                  LexNavigator Information Update (UltraTech Group)
    AAPKG                ArchestrA IDE Package (Invensys)
    AAR                  Axis Archive (Apache Software Foundation)
    AAS                  Star Trek Elite Force Bot Support File (Raven Software)
    AAS                  Audible Words File (Audible, Inc.)
    AAS                  Movie Clip; Autodesk Animation Setup; used by Compton's Reference Collection 
    AAS                  Authorware Shocked Packet (Segment) (Adobe Systems Incorporated)
    AAS                  Bong Script File 
    AAT                  Arcinfo Line Data Attribute Data 
    AATREND              ActiveFactory Trend File (Invensys Systems, Inc.)
    AB                   Applix Builder 
    AB$                  AutoCAD Spooled Plot (Autodesk, Inc.)
    AB1                  Chromas Lite Nucleotide Sequence Analysis Results (Technelysium Pty Ltd)
    AB1                  ABIF Applied Biosystems Inc. Format (Applied Biosystems)
    AB2                  Parson's Address Book 
    AB3                  PhotoImpact Album File 
    AB4                  ENIAC Computing AB4 Accounting & Business File 
    AB6                  AB Stat Data 
    AB8                  AB Stat Data 
    ABA                  Palm Address Book File (Palm, Inc.)
    ABAP                 ABAP Source Code (SAP AG)
    ABC                  LithTech Game 3D Model (Touchdown Entertainment, Inc.)
    ABC                  ABC Programming Language 
    ABC                  ACT! E-mail Address Book File (Best Software CRM Division)
    ABC                  Athena 16 Address Book 
    ABC                  Pro-Act 
    ABC                  FlowCharter Data File (iGrafx (a division of Corel Inc.))
    ABC                  Musical Notation Language File 
    ABD                  Adventure Builder Database 
    ABD                  The Bat! Address Book (RITLABS)
    ABD                  AmBiz Bonus Calculator Data File 
    ABF                  Binary Screen Font (Adobe)
    ABFS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ABG                  Fifa World Cup Game Data File 
    ABI                  dnaLIMS ABI Trace File (dnaTools)
    ABI                  Chromas Nucleotide Sequence Analysis Results (Technelysium Pty Ltd)
    ABI                  ABI- CODER (ABI- Software Development)
    ABI                  AOL 6 Organizer (America Online, Inc.)
    ABK                  Need for Speed: Underground Audio File (Electronic Arts, Inc.)
    ABK                  Corel Draw AutoBackup 
    ABK                  HP-95LX Appointment Book File 
    ABK                  AMOS Basic Data (SPSS Inc.)
    ABK                  AMOS Music Bank 
    ABK                  Any Automatic Backup 
    ABK                  Ability Office Write Document Backup (Ability Software)
    ABL                  Maxagrid Avails Data File 
    ABL                  Xilinx Integrated Software Environment ABEL Hardware Description Language (Xilinx, Inc.)
    ABL                  ADONIS Application Library Export (BOC Group)
    ABM                  HitPlayer Audio Album File 
    ABM                  Photo Impressions Album (ArcSoft, Inc.)
    ABM                  Album 
    ABM                  Audition On-line Dance Battle Music (T3 Entertainment Co., LTD.)
    ABO                  Applix Builder Turbo 
    ABOUT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ABOUTIT              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ABR                  Photoshop Brush (Adobe)
    ABS                  Absolute Database Database (ComponentAce)
    ABS                  Abscissa Data File 
    ABS                  MPEG Audio Sound File 
    ABS                  AB SuDoKu Sudoku Save Game (Andrei Baronov)
    ABS                  Absolute Packager Configuration File (Xceed Software Inc.)
    ABS                  Abstracts 
    ABS                  Optocat 
    ABS                  PC GNU Compiler Output 
    ABT                  The Arbiter Sports Official Schedule 
    ABT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ABU                  ACT! E-mail Address Book File (Best Software CRM Division)
    ABW                  AbiWord Document File (The AbiSource Community)
    ABX                  WordPerfect Address Book File (Corel Corporation)
    ABY                  AOL (America Online, Inc.)
    ABZ                  AbszOrb 
    AC                   AC3D Geometry File (Inivis)
    AC                   CaseWare Working Papers Client File 
    AC                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AC$                  AutoCAD Undo Info (Autodesk, Inc.)
    AC1                  Weaver Spider Software (Weaver Spider Software)
    AC2                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AC3                  AC3 Audio File Format 
    ACA                  Agent Character (Microsoft)
    ACA                  Project Manager Workbench File 
    ACAD                 AutoCAD Database (Autodesk, Inc.)
    ACB                  Photoshop Color Book (Adobe Systems Inc.)
    ACB                  AOL Cab Launcher (America Online, Inc.)
    ACB                  ACMB Graphic Image 
    ACC                  Composer File (Microsoft)
    ACC                  DR-DOS Viewmax File 
    ACC                  Graphic Accounts Accounts Data (FKJ Software)
    ACC                  Atari ST Executable 
    ACCDB                Access 2007 Database File (Microsoft Corporation)
    ACCESS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ACCESS               Java Access Control File (Sun)
    ACCESSPRIV           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ACCOLADE             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ACCORD               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ACCT                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ACD                  RSLogix 5000 Project File (Rockwell Automation)
    ACD                  Sonic Foundry Acid Music File (Sony Corporation of America)
    ACD                  Chromeleon Channel Raw Data 
    ACD                  Nascar Racing 2002 Arcade Car Setup (Sierra)
    ACD                  allCLEAR Flowcharting File 
    ACD                  Alan Game 
    ACD                  Agent Character Definition (Microsoft)
    ACE                  Jupiter Ace Snapshot 
    ACE                  WinAce Compressed File (e-merge GmbH)
    ACE                  Train Simulator Texture File (Microsoft Corporation)
    ACE                  Aces200 
    ACE                  Casting Electronic Enterprise CV Archive (Casting Electronic)
    ACE                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ACF                  AdviceCalculator (JRK Software)
    ACF                  Photoshop Custom Filter (Adobe)
    ACF                  Spectramap Application Colors File (Coloritto BV)
    ACF                  DB/TextWorks Database Access Control File 
    ACF                  Agent Character (Microsoft)
    ACF                  X-Plane Plane Description File (Laminar Research)
    ACG                  Agent Preview (Microsoft)
    ACG                  Age of Wonders Saved Game (Triumph Studios)
    ACGD                 ACG Case-Mix System ACG Predictive Model (Johns Hopkins University)
    ACGI                 AppleSearch CGI Interface 
    ACGL                 ACG Case-Mix System License File (Johns Hopkins University)
    ACH                  Speccy Snapshot 
    ACH                  Rockwell Software Logix 5 File 
    ACH                  Medlin Accounting Automated Clearing House File (Medlin Accounting)
    ACI                  ACI Development Appraisal 
    ACIDBAT2             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ACL                  Access Control List (Microsoft)
    ACL                  RaidenFTPD Rights File 
    ACL                  Corel Draw 6 Keyboard Accelerator 
    ACL                  ACL For Windows 
    ACL                  Office AutoCorrect List (Microsoft)
    ACLOCK               Amazing clock (Amazing-Clock.com)
    ACM                  ACBM Image File 
    ACM                  Interplay Compressed Sound File 
    ACM                  Movie File in some games 
    ACM                  American Civil War File (Interactive Magic)
    ACM                  Audio Compression Manager Driver (Microsoft)
    ACM                  Infinity Game Engine Sound (BioWare Corp.)
    ACM                  Windows System File 
    ACM                  Photoshop Command Button (Adobe)
    ACMB                 Graphics File 
    ACMD                 Apache Commander Project (Zecos Software)
    ACO                  ArCon Project (TRIXL GmbH)
    ACO                  Photoshop Color Swatch File (Adobe)
    ACO                  Fast-Talk Acoustic Model File (Fast-Talk Communications, Inc.)
    ACP                  Action! Document (Macromedia)
    ACP                  Acrobat Paper Capture Temporary File (Adobe)
    ACP                  Office Assistant Preview File (Microsoft)
    ACP                  ACB Compressed File Archive 
    ACQ                  AcqURL 
    ACQ                  AcquTach+ (Acquisition Systems)
    ACR                  Dicom 
    ACR                  American College of Radiology (American College of Radiology)
    ACR                  Medlin Accounting Accrual Method Adjustments (Unpaid Invoices) (Medlin Accounting)
    ACS                  Agent Character (Microsoft Corporation)
    ACT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ACT                  Action! Presentation (Macromedia)
    ACT                  Animation:Master Action (Hash)
    ACT                  DataWave Physiological Data File 
    ACT                  FoxPro Documenting Wizard Action Diagram (Microsoft)
    ACT                  ADPCM Compressed File Recorded by Some MP3 Players/Voice Recorders 
    ACT                  Office Actor Program File (Microsoft)
    ACT                  Photoshop Color Table (Adobe)
    ACT                  Train Simulator Activity File (Microsoft Corporation)
    ACTH                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ACTH2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ACTH3                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ACTION               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ACTIONNDA            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ACTIONS              Photoline4 Defaults File 
    ACTPROJ              Visual Studio .NET ACT Project (Microsoft)
    ACU                  AppleLink-PE Compression Format 
    ACU                  ACUCOBOL-GT Intermediate Object File (Acucorp, Inc.)
    ACU                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ACV                  Photoshop Saved Curve (Adobe)
    ACV                  OS/2 Audio Drivers 
    ACW                  AutoCaption Worksheet File (Image Logic Corporation)
    ACW                  American Civil War Game Save File (Interactive Magic)
    ACW                  Accessibility Wizard Settings (Microsoft)
    ACX                  Atari ST Executable 
    ACX                  FOCUS/WebFOCUS File Description 
    ACX                  Rational XDE (IBM)
    AC_                  CaseWare Working Papers Compressed Client File 
    AC_                  Creativ compressed Sb16 sbide file 
    AD                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AD                   AppleDouble Encoded File (Apple Computer, Inc.)
    AD                   Ascend Password Protocol 
    AD                   After Dark Screen Saver 
    AD                   AxSys Script File (Genomic Solutions)
    AD!                  OzWin CompuServe E-mail/Forum Access QS-Special - Get Marked Online 
    AD2                  Compressed Voice File 
    AD3                  Compressed Voice File 
    ADA                  ADA Program 
    ADA                  Advanced Digital Audio Compressed Audio 
    ADA                  ActiveDocs Answer Set 
    ADB                  Ability Office Database (Ability Software)
    ADB                  Advanced Integrated Maintenance Support System (AIMSS) Runtime Project Configuration File (Raytheon Company)
    ADB                  ACT! Activity Data File (Best Software CRM Division)
    ADB                  Alpha Five Database Information File (Alpha Software, Inc.)
    ADB                  Alphacam Diecut VB Macro (Planit)
    ADB                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ADB                  HP 100LX Organizer Appointment Database 
    ADB                  ALLDATA 
    ADB                  CorelCENTRAL Address Book (Corel Corporation)
    ADB                  ADA Package Body 
    ADB1                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ADB2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ADBA                 Advanced Integrated Maintenance Support System (AIMSS) Author Project Configuration File (Raytheon Company)
    ADBLOCK              Alpha Five Temporary Locking File (Alpha Software, Inc.)
    ADC                  Chromeleon Demo Data File 
    ADC                  ActiveDocs Clipping Catalog 
    ADC                  Systems Management Server (SMS) Collection Evaluator Addition (Microsoft Corporation)
    ADC                  ADC Data Acquistion 
    ADC                  Lingvo Dictionary Text File 
    ADC                  Scanstudio 16-color Bitmap Graphic File 
    ADC                  TI/MIT ADC Format Sound 
    ADC                  XMap ImageReg Output File (Delorme)
    ADC                  ClearSight Analyzer Trace File (ClearSight Networks, Inc.)
    ADD                  Alphacam Diecut Drawing (Planit)
    ADD                  Advantage Database Server Data Dictionary 
    ADD                  OS/2 Adapter Device Driver 
    ADD                  PageMaker (Adobe)
    ADDEND               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ADDENDUM             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ADDRESS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ADDRESSES            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ADE                  Access Project Extension (Microsoft)
    ADE                  ADC Audio File 
    ADEX                 ADEX Corporation ChromaGraph Graphics Card Bitmap Graphic 
    ADF                  ReliaSoft ALTA 1 
    ADF                  Wyatt Technology ASTRA(r)Chromatography 
    ADF                  Grand Theft Auto Vice City Radio Station (Take-Two Interactive Software, Inc.)
    ADF                  ArcView ARC/INFO Coverage Data File (ESRI)
    ADF                  MicroSim PCBoard Photoplot Aperture Definition File 
    ADF                  Dog Creek QC Mask File 
    ADF                  Amiga Disk File (Amiga)
    ADF                  Admin Config File 
    ADF                  Adapter Description File 
    ADF                  Actual Drawing File 
    ADF                  I-DEAS Associated Data File (EDS)
    ADI                  Amateur Data Interchange Format (ADIF) File 
    ADI                  Advantage Database Server Database Index File 
    ADI                  AutoCAD Plotter File (Autodesk, Inc.)
    ADINEKIRN            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ADJ                  LISCAD Plus Adjustment File 
    ADJUST               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ADK                  OS/2 Saved SKF Format File (IBM)
    ADL                  OzWin CompuServe E-mail/Forum Access QS-Special Get Marked 
    ADL                  ADONIS Models and Model Groups Export From Database (BOC Group)
    ADL                  DESQview (Symantec)
    ADL                  QEMM MCA Adapter Description Library 
    ADM                  SCSI Driver 
    ADM                  Addict Compiled Dictionary 
    ADM                  Portfolio Admin-level Acces Flag (Extensis, Inc.)
    ADM                  Windows Policy Template (Microsoft)
    ADM                  Alphacam Diecut Parameter Macro (Planit)
    ADM                  After Dark Screen Saver Module 
    ADM                  Advantage Database Server Database Memo File 
    ADN                  1-2-3 Add-in Utility (IBM)
    ADN                  Access Blank Project Template (Microsoft)
    ADO                  Stata Program 
    ADO                  Photoshop Duotone Options (Adobe)
    ADOBENET             Studio Snapshot Tool Application (Adobe)
    ADOC                 Authentica Secure Office Protected Word Document (Authentica)
    ADOS                 ADOS Compound Document (ADOS Corporation)
    ADP                  AOL Server Dynamic Page (America Online, Inc.)
    ADP                  Astound Dynamite 
    ADP                  FaxWorks Modem Setup File 
    ADP                  Alphacam Diecut Post (Planit)
    ADP                  Access Project (Microsoft)
    ADPCM                Intel/DVI Adaptive Differential Pulse Code Modulation Format Sound 
    ADPCM                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ADR                  Opera Web Browser Bookmark File (Hotlist) (Opera Software ASA)
    ADR                  XAIRON-Addy Address Manager Data File 
    ADR                  Smart Address Address Book 
    ADR                  After Dark Random Screen Saver Module 
    ADR                  Address Plus Database 
    ADR                  Address Book 
    ADRESSES             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ADS                  ADA Package Specification 
    ADS                  ADS Applications File (Autodesk, Inc.)
    ADT                  Casio Travel Phrase Database 
    ADT                  Lingvo Dictionary File 
    ADT                  Alphacam Diecut Tools (Planit)
    ADT                  ACT! Document Template (Best Software CRM Division)
    ADT                  MediSoft Database Dictionary File 
    ADT                  AdTech Fax 
    ADT                  Advantage Database Server Database File 
    ADT                  AutoCAD Audit Report (Autodesk, Inc.)
    ADT                  HP NewWave Card Application Data File 
    ADV                  GUS Virtual Device Driver 
    ADV                  TAPCIS Advanced Commands (User Entered) 
    ADVENTURE            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ADW                  ADHAWIN Multilingual Transliteration Scheme 
    ADX                  Archetype Designer Document 
    ADX                  Approach Index File (IBM)
    ADX                  Dreamcast Audio File (Sega)
    ADX                  Dynazip Active Delivery Script 
    ADZ                  Amiga GZ-Packed Disk Image (Amiga)
    AE                   SoftQuad Author/Editor File 
    AEB                  Alphacam Wire EDM VB Macro (Planit)
    AEC                  DataCAD Double Precision Drawing (DATACAD LLC)
    AED                  Alphacam Wire EDM Drawing (Planit)
    AEF                  ELECTRICAL Designer 
    AEG                  Netmino File 
    AEH                  iPer Advanced Embedded Hypertext (Visual Vision)
    AEM                  Alphacam Wire EDM Parameter Macro (Planit)
    AEP                  ArcExplorer Project (Environmental Systems Research Institute, Inc. (ESRI))
    AEP                  Alphacam Wire EDM Post (Planit)
    AEP                  Homeworld (Sierra Entertainment, Inc.)
    AEP                  After Effects Project File (Adobe)
    AER                  Atmosphere File (Adobe Systems Inc.)
    AES                  After Effects Stablilzer Settings (Adobe)
    AET                  After Effects Tracker Settings (Adobe)
    AEU                  AutoRoute Express Europe Route (Microsoft Corporation)
    AEX                  Pretty Good Privacy (PGP) Armored Extracted Public Encryption Key (PGP Corporation)
    AEX                  Art Explosion Greeting Card Factory Deluxe Graphic (Nova Development Corporation)
    AEX                  Alpha Five Compiled Add-in Library Script (Alpha Software, Inc.)
    AEXPK                Pretty Good Privacy (PGP) Armored Extracted Public Key (PGP Corporation)
    AF                   3DPlus2 Model (Serif)
    AF2                  ABC Flowchart 
    AF3                  FlowCharter Ver. 3 & 4 (iGrafx (a division of Corel Inc.))
    AFB                  Alphacam Flame VB Macro (Planit)
    AFC                  Apple Sound (Apple Computer, Inc.)
    AFC                  ArisFlow Commander 
    AFD                  Alphacam Flame Drawing (Planit)
    AFD                  Prismo Graphics ActiveFile Data 
    AFD                  ArisFlow Diagram 
    AFF                  AnyForm Form 
    AFFAIR               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AFG                  Guitar Musik Trainer File 
    AFI                  Truevision Bitmap Graphic 
    AFL                  Lotus Font (IBM)
    AFL                  X-Plane Airfoil (Laminar Research)
    AFL                  AnimaFlex RubberFlex Animation 
    AFL                  XMap (Delorme)
    AFM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AFM                  Outline Font Metric (Adobe Systems Inc.)
    AFM                  HP NewWave Cardfile Application 
    AFM                  LOUT File 
    AFM                  Alphacam Flame Parameter Macro (Planit)
    AFM                  DESQview (Symantec)
    AFM                  FormMaster Editor Form Template (AT SYSTEMS S.R.L.)
    AFM                  Abassis Finance Manager Data File (SmartMedia Informatica)
    AFO-XVID             RAR-compressed Video 
    AFP                  Androsa FileProtector Encrypted File (AndrosaSoft)
    AFP                  Prismo Graphics ActiveFile Proxy 
    AFP                  Papyrus Document Storage Format 
    AFP                  Gangsters Game File 
    AFP                  Alphacam Flame Post (Planit)
    AFP                  IBM Transactional Data Format (Advanced Function Print) 
    AFP                  Flowchart Symbol Palette 
    AFP                  XMap Flight Plan or Flight Plan Statistic File (Delorme)
    AFS                  Filter Factory Photoshop Filter 
    AFS                  Sega Dreamcast Compressed Audio 
    AFS                  Type Manager Font Set (Adobe)
    AFS3                 Advanced File Security Encrypted File (Osborne Software - Eric Pörsel)
    AFT                  Ancestry Family Tree Data File (MyFamily.com Inc.)
    AFT                  AnyForm Template 
    AFT                  Flowchart Template 
    AFTER                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AFW                  Flowchart Work Area 
    AFW                  Industrial Aspect Integrator Platform (ABB Ltd)
    AFX                  Auto FX Photographic Edges (Auto FX Software)
    AFZ                  Ancestry Family Tree Backup Data File (MyFamily.com Inc.)
    AG                   Agenda Application (IBM)
    AG                   Argus Spreadsheet 
    AG                   Applix Graphic 
    AG4                  Access G4 File (Microsoft)
    AGC                  AfterGRASP Compiled Script (John Bridges)
    AGE3SCN              Age of Empires 3 Scenario (Microsoft Corporation)
    AGF                  Atlas GeoFile 
    AGG                  AggFlow Plant Flow Diagram (BedRock Software Inc.)
    AGG                  Graphic 
    AGI                  Asterisk Gateway Interface File (Digium, Inc.)
    AGM                  DOTAGM.JAR Packed Audio File (agm digital arts GmbH)
    AGN                  Psion Series 3 Agenda (Psion PLC)
    AGO                  Autolog Script (Soft Machines)
    AGP                  Aspen Graphics Pages 
    AGP                  AppGini MySQL Code Generator Project (BigProf Software)
    AGR                  Aepryus Graph Image or Screen Saver File (Aepryus Software)
    AGR                  Grace Project File 
    AGR                  APM Graph Vector Graphic (APM)
    AGREEMNT             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AGS                  AfterGRASP Script (John Bridges)
    AGS                  AppGini Saved Style Sheet (BigProf Software)
    AGT                  PeerAgent Typed Executable E-mail 
    AGW                  Aspen Graphics Windows 
    AGX                  Rational XDE (IBM)
    AGZ                  AggFlow Aggregate Plant Flow Diagram (BedRock Software Inc.)
    AH                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AHB                  AIM High Briefcase Report File (Cosby Clifton Computer Consultants Ltd)
    AHD                  Asus Probe 
    AHF                  ASP Header 
    AHK                  AutoHotkey Script (AutoHotkey)
    AHOD                 Homeworld (Sierra Entertainment, Inc.)
    AHP                  AutoCAD Help (Autodesk, Inc.)
    AHQ                  AudioHQ Plug-in Module 
    AHS                  Photoshop Halftone Screens (Adobe)
    AHTM                 HTML File 
    AHTML                HTML File 
    AHUFF                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AHX                  WinAHX THX Tracker Music (Abyss)
    AI                   Rise of Nations AI File (Microsoft)
    AI                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AI                   Ai Archiver Archive 
    AI                   Corel Trace Drawing 
    AI                   EARS Component File 
    AI                   Illustrator Vector Graphic (Adobe Systems Inc.)
    AI                   Progress Database After Image File (Progress Software Corporation)
    AI                   Advantage Database Server 
    AI3                  Generic Encapsulated PostScript Graphic Format 
    AI4                  Generic Encapsulated PostScript Graphic Format 
    AI5                  Generic Encapsulated PostScript Graphic Format 
    AI6                  Generic Encapsulated PostScript Graphic Format 
    AI7                  Generic Encapsulated PostScript Graphic Format 
    AIC                  Advanced Image Coding Bitmap 
    AIF                  OS/2 Program Information File (IBM)
    AIF                  SymbianOS Application Information File 
    AIF                  Knowledgeware Setup Information 
    AIF                  Audio Interchange File 
    AIFC                 Audio Interchange File 
    AIFF                 Audio Interchange File 
    AIG                  Design Your Own Home Architecture Drawing File (theliquidateher Software, An MCS Investments, Inc. Company)
    AIH                  ArcView Optional Index File (ESRI)
    AII                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AIJ                  Oracle After-Image Journal 
    AIL                  AutoImager 
    AIM                  Asm Text Mode Image File (The Ultimate Draw) 
    AIM                  AOL Instant Messenger (AIM) (America Online, Inc.)
    AIML                 Artificial Intelligence Markup Language 
    AIN                  ArcView Optional Index File (ESRI)
    AIN                  AIN Compressed Archive 
    AINSO                VAX CRYSTALS Chemical Crystallogaphy File 
    AIO                  APL File Transfer Format 
    AIP                  Star Trek Armada Arficicial Intelligence Parameters 
    AIP                  Illustrator Plug-in (Adobe)
    AIP                  Audiosoft Parameter File 
    AIQ                  AzureIQ Client 
    AIR                  Flight Simulator Aircraft Performance Info File (Microsoft)
    AIR                  Align It! Resource 
    AIR                  Automatic Image Registration 
    AIR                  SeeYou Airspace (Naviter)
    AIRPORT              RealFlight Radio Control Flying Site (Knife Edge Software)
    AIS                  ACDSee Image Sequence File (ACD Systems Ltd.)
    AIS                  Velvet Studio Instruments 
    AIS                  Array of Intensity Samples 
    AIV                  Stronghold Crusader Computer Player AI (Firefly Studios Ltd.)
    AIW                  Formula 1 car race 2001 Season01 School France file 
    AIX                  HP NewWave Cardfile Application Data 
    AJA                  AJC Active Backup Backup Revision Control Archive (AJC Software)
    AJB                  Alphacam Water Jet VB Macro (Planit)
    AJD                  Alphacam Water Jet Drawing (Planit)
    AJG                  AJC Grep Project File (AJC Software)
    AJL                  ARCserve Backup Journal File (Computer Associates International, Inc.)
    AJM                  Alphacam Water Jet Parameter Macro (Planit)
    AJP                  Alphacam Water Jet Post (Planit)
    AJP                  Anfy Applet Generator Saved File (Anfy Team)
    AJZ                  Jukebox MP3 Player Firmware Update (Archos)
    AK                   Audio Utility Tuningwrench Compressed File 
    AKF                  Acrobat Key File (Adobe)
    AKM                  Aksharamala Binary Keymap File (Deshweb.com Pvt. Ltd.)
    AKM                  Microsoft Operation Manager (MOM) Management Pack (Binary) (Microsoft Corporation)
    AKS                  Aksharamala Ver 2.0 Binary Keymap File (Deshweb.com Pvt. Ltd.)
    AKT                  Aksharamala Source Keymap File (Deshweb.com Pvt. Ltd.)
    AKT                  AKT Compressed Archive 
    AKW                  RoboHELP Help Project Index File 
    AL                   ALAW Format Sound 
    AL                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AL                   Oracle File 
    AL$                  Cubase Compressed File (Steinberg)
    ALA                  Sharp Fax File 
    ALA                  Alan Adventure Language File 
    ALANIS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ALARMZ               Alarmz Alarm File (Fichier d'alarmes) (Francis Gagné)
    ALAW                 ALAW Format Sound 
    ALB                  Cubase or VST Backup Song File (Steinberg)
    ALB                  Photo Soap2 File 
    ALB                  Raduga Playlist (WoLoSoft)
    ALB                  Alphacam Laser VB Macro (Planit)
    ALB                  Image Commander Album (Corel Corporation)
    ALB                  Pro/DESKTOP Album (PTC)
    ALB                  Chief Architect Library File (Chief Architect)
    ALB                  Web Easy Album (VCOM/Avanquest Publishing USA, Inc.)
    ALB                  Alpha Five Data Dictionary (Alpha Software, Inc.)
    ALBINON              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ALBM                 webAlbum Photo Album (Goopers - Logan Mueller)
    ALBM                 Photosmart Photo Printing Album (Hewlett-Packard Development Company, L.P.)
    ALC                  Alchemy Chemical Format (CambridgeSoft Corporation)
    ALC                  MP3 Embedded Dynamic Lyrics 
    ALC                  Norton Internet Security Ad Server File 
    ALC                  ACUCOBOL-GT License File (Acucorp, Inc.)
    ALD                  Alphacam Laser Drawing (Planit)
    ALD                  Aladdin Interactive DSP Document (Hitech Development AB)
    ALDOC                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ALDRIN               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ALE                  Avid Log Exchange 
    ALERT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ALF                  Cadence Ambit BuildGates 
    ALF                  Color Quick Label 
    ALF                  VPCom Remote User Configuration File 
    ALF                  AdventNet Look and Feel Standards 
    ALF                  Aventail Connect License 
    ALF                  ACT! User Logon Tracking (Best Software CRM Division)
    ALF                  LANDesk Client Manager Configuration File 
    ALF                  ALF Logic 
    ALF                  Abacus Law (Abacus Data Systems, Inc.)
    ALF                  Airpro 2000 Landfile 
    ALG                  ARCSOLO Activity Log 
    ALG                  InRoads Survey Alignment File (Bentley Systems, Incorporated)
    ALG                  Netmino File 
    ALH                  Aladdin Interactive DSP Model Header File (Hitech Development AB)
    ALI                  Inspect3D Aligned Image Stack (FEI Company)
    ALI                  SAP Proprietary Format Document File 
    ALI                  OM2 OMALI Native OMALI Document 
    ALI                  DESQview (Symantec)
    ALIAS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ALIAS                Alias Image 
    ALIASES              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ALIEN                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ALIEN1               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ALIGN                Insight II Sequence Alignment File (Accelrys Software Inc.)
    ALL                  FRQView File List of All Files 
    ALL                  Cubase or VST Song File (Steinberg)
    ALL                  OS/2 Program List (IBM)
    ALL                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ALL                  Arts & Letters Symbols and Characters 
    ALL                  1-2-3 Always Working Page Format File (IBM)
    ALL                  WordPerfect Printer Info (Corel Corporation)
    ALLIANCE             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ALLOCM               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ALM                  AL-Mail Mailbox File (AL-Mail)
    ALM                  Alphacam Laser Parameter Macro (Planit)
    ALM                  Alpha Five Database Information File (Alpha Software, Inc.)
    ALM                  Aley's Module v1.0, 1.1, 1.2 
    ALN                  LexNavigator Database Update (UltraTech Group)
    ALN                  Grlib Saoimage File 
    ALO                  Almanac Support File 
    ALONE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ALP                  Alphacam Laser Post (Planit)
    ALPHA                Raw Alpha Bytes Bitmap 
    ALPHA                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ALPHAB               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ALR                  VirusScan Alert File (Network Associates)
    ALR                  ADRIFT Language Resource (ADRIFT)
    ALS                  Arcade Lines Graphic Resource File (Lobstersoft)
    ALS                  Pro/ENGINEER Temporary File (PTC)
    ALS                  Live Project (Ableton AG)
    ALS                  Alias Image 
    ALS                  Calmira Shortcut File 
    ALT                  ReliaSoft ALTA 6 
    ALT                  WordPerfect Menu File (Corel Corporation)
    ALT                  Aladdin Interactive DSP Message Text (Hitech Development AB)
    ALTO                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ALV                  Vbox Available Languages File (Aladdin Knowledge Systems)
    ALV                  Photoshop Levels (Adobe)
    ALX                  Alpha Five Library Index (Alpha Software, Inc.)
    ALX                  OM2 OMALI Plug-in Extension 
    ALX                  ActiveX Layout File (Microsoft Corporation)
    ALYSIS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ALZ                  ALZip Compressed File (ESTsoft, Corp.)
    ALZ                  Game File 
    AL_                  Audio Utility Juker Compressed File 
    AM                   Anark Studio Media File (Anark)
    AM                   Advantage Database Server 
    AM                   Applix SHELF Macro 
    AM                   Music File 
    AM1                  Adventure Maker Project File (Giovanni Albani)
    AM2                  AutoPlay Menu Studio Ver. 2 Data File (Indigo Rose Software)
    AM3                  AutoPlay Menu Studio Ver. 3 Data File (Indigo Rose Software)
    AM4                  AutoPlay Menu Studio Ver. 4 Data File (Indigo Rose Software)
    AM5                  AutoPlay Menu Studio Ver. 5 Data File (Indigo Rose Software)
    AM6                  AutoPlay Menu Studio Ver. 6 Data File (Indigo Rose Software)
    AMA                  Fifa2000 Feart File 
    AMAT                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AMAYA                Amaya 
    AMB                  Radiance Ambient File Format 
    AMB                  Alphacam 3D Mill, 2D Mill VB Macro (Planit)
    AMC                  Ant Movie Catalog 
    AMC                  Wireless Analyzer Captured Data (AirMagnet)
    AMD                  ABBYY Finereader 5.0 Pro File (ABBYY Software House)
    AMD                  ASCET XML Model Description (ETAS GmbH)
    AMD                  Alphacam 3D Mill, 2D Mill Drawing (Planit)
    AMD                  AMUSIC AdLib Tracker 
    AME                  ACT! E-mail System Library (Best Software CRM Division)
    AMERICA              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AMF                  Autolog Macro (Soft Machines)
    AMF                  ADRIFT Module File (ADRIFT)
    AMF                  Advanced Module Format Music 
    AMF                  Amiga Metafile (Amiga)
    AMF                  DSMIA/Asylum Module Music 
    AMFF                 Amiga Metafile (Amiga)
    AMG                  AMGC Compressed Archive 
    AMG                  AMG ACTOR System Image 
    AMG                  Netmino File 
    AMI                  Amiga Paint File (Amiga)
    AMI                  Cocreate SolidDesigner Annotation File 
    AMI                  AMI BIOS Firmware Update File (American Megatrends Inc.)
    AMIGA                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AMIGA                Amiga (Amiga)
    AML                  AutoMage 
    AML                  SAS Activity-Based Management Model File (SAS Institute Inc.)
    AML                  Mio Technology Application Markup Language (Mioplanet Technologies, Inc.)
    AMM                  AMM 
    AMM                  Alphacam 3D Mill, 2D Mill Parameter Macro (Planit)
    AMM                  Orchida Knitting System (Orchida Soft)
    AMM                  Microphone Modeler Model File (Antares Audio Technologies)
    AMODEL               ActiveModeler Professional Project File (KAISHA-Tec Co. Ltd.)
    AMOS                 AMOS Basic Code 
    AMP                  Alphacam 3D Mill, 2D Mill Post (Planit)
    AMP                  PlayMedia AMP MP3 Audio File 
    AMP                  Photoshop Arbitrary Map Settings (Adobe)
    AMP                  Connectivity Memory Model AMPA and NMDA Parameters Input File 
    AMPER                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AMR                  Adaptive Multi-Rate ACELP Codec 
    AMS                  Velvet Studio Module 
    AMS                  Extreme's Tracker Module Format 
    AMS                  Photoshop Monitor Setup (Adobe)
    AMT                  ABBYY Finereader 5.0 Pro (ABBYY Software House)
    AMT                  Alphacam 3D Mill, 2D Mill Tool (Planit)
    AMT                  ArtMoney Table (ArtMoney Team)
    AMV                  MTV Movie File 
    AMW                  WinScript Show Data (Alcorn McBride Inc.)
    AMW                  AMOS Data 
    AMX                  MetaMOD Half-Life Plug-in 
    AMX                  After Effects Motion Exchange File (Adobe)
    AMXX                 AMX Mod X Plug-in (AMX Mod X Dev Team)
    AMY                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AMZ                  The Amazon Trail Saved Game 
    AN                   Groundworks Text File 
    AN                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AN1                  XMap Draw File (Delorme)
    AN1                  Street Atlas Draw File (Delorme)
    AN1                  WWG DA-30 Capture File 
    AN8                  Anim8or 3D Animation 
    ANAL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANALASIS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANAT                 XNBC V8 Biological Neural Network Simulation Workstation Network Anatomy 
    ANB                  Analyst's Notebook Analytical Chart (i2 Ltd.)
    ANC                  Canon Computer Pattern Maker File 
    ANC                  Alphacam NC Program (Planit)
    ANCESTRY             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AND                  Collins Electronic Dictionary Data (Windows) / Concise Oxford Dictionary (DOS) 
    ANF                  AcuteNotes 
    ANG                  AIMSUN NG Network (TSS-Transport Simulation Systems)
    ANGELES              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANI                  NEOchrome Animation 
    ANI                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANI                  Windows Animated Cursor (Microsoft)
    ANI                  NeroMix File (Nero AG)
    ANIM                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANIM                 Homeworld (Sierra Entertainment, Inc.)
    ANIM                 Amiga IFF Animation (Amiga)
    ANIM3                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANIM5                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANIM50               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANIMAL               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANIMALS1             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANIMALS2             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANIMALS3             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANIMASIA             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANIMATION            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANIMDEMO             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANIMTOOL             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANL                  Annotis Mail Language File (Emeris Technologies)
    ANL                  Eisenbahn.exe Professionell Railroad Track Layout (Software Untergrund)
    ANL                  Project Analyzer Saved Analysis 
    ANL                  Alphacam Nest List (Planit)
    ANM                  Jazz Jackrabbit 2 Title Sets Images (Epic Games, Inc.)
    ANM                  Infinity Equation Script (MathRevolt)
    ANM                  DeluxPaint Animation 
    ANM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANM                  Animatic Animation File 
    ANN                  King-FIX 
    ANN                  Help Annotation 
    ANNOT                Amaya Annotation 
    ANO                  Annotation 
    ANP                  Annotis Mail (Emeris Technologies)
    ANR                  XMap Route File (Delorme)
    ANR                  Street Atlas Route File (Delorme)
    ANS                  HotDocs Answer File (Matthew Bender & Company, a member of the LexisNexis Group)
    ANS                  Word Text and Layout (Microsoft Corporation)
    ANS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANS                  ANSI Graphic/Text 
    ANS2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ANSR                 AnswerTool Question/Answer Document (DTLink LLC)
    ANT                  calcAnt Saved Calculation (SFR GmbH)
    ANT                  SimAnt for Windows Saved Game File 
    ANTIFRAG             DC++ Incomplete Download 
    ANV                  AirNav 
    ANX                  HotDocs Answer File (Matthew Bender & Company, a member of the LexisNexis Group)
    AO                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AO                   Free Pascal Archive File for OS/2 Version (FPC Development Team)
    AO                   ActionOutline Data File (Green Parrots Software)
    AO                   AOL Install Archive (America Online, Inc.)
    AOB                  DVD Audio File 
    AOF                  Acorn Object Format 
    AOI                  Art of Illusion Scene File (Peter Eastman)
    AOL                  AOL (America Online, Inc.)
    AOM                  Download Manager Online Manager Shortcut (Adobe Systems Inc.)
    AOM                  After Effects Output Settings (Adobe)
    AOMVER               Adobe Download Manager Version File (Adobe Systems Incorporated)
    AOR                  Searchlight AR3000/AR3000A Radio Scanner Frequency Listing (AOR UK LTD)
    AOS                  Gmini Firmware Update (ARCHOS)
    AOS                  Nokia Phone Add-on Software (Nokia)
    AOT                  Novell snAppShot Application Binary Object Template File 
    AP                   WHAP-Compressed Amiga Archive 
    AP                   ArmPack Archive File 
    AP                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AP                   Datalex EntryPoint 90 Data File 
    AP                   Applix Presents 
    AP0                  Windows APW_data File 
    APA                  AlfaPad Notes Organizer Document (AlfaAlfa Software)
    APA                  ArcPad Ver. 6 Applet 
    APB                  Avram's PC Backup File List (QuadraSoft)
    APB                  Alphacam Punch VB Macro (Planit)
    APC                  AiroPeek 
    APC                  Cryo Interactive APC Audio 
    APC                  APAC Compressed Audio 
    APC                  Lotus Printer Driver Characters (IBM)
    APC                  Gupta Team Developer Compiled Application File 
    APD                  PageMaker Printer Description (Adobe)
    APD                  Alphacam Punch Drawing (Planit)
    APD                  Gupta Team Developer Dynamic Application Library File 
    APD                  Lotus Printer Driver (IBM)
    APE                  Monkey's Audio Lossless Audio Compression Format (Matthew T. Ashland)
    APE                  Winamp Plug-ins AVS File 
    APF                  Lotus Printer Driver Fonts (IBM)
    APF                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    APF                  MightyFax File 
    APF                  Fax2000 File 
    APF                  Acrobat Profile File (Adobe)
    APF                  Speedy Printed Circuit Board Design Software (Mania Technologie)
    APF                  Homesite Project File 
    APF                  ArcPad Ver. 5 Profile 
    APF2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    APG                  EasyBase Database File (Pg SoftWorkS')
    APG                  Active Page Generator APG Script (Webgecko Software)
    APG                  Graphic 
    APG                  PowerAda Aprobe File (OC Systmes)
    APH                  Ability Office Photo Album (Ability Software)
    API                  1st Reader Passed Parameter File 
    API                  Acrobat Plug-in (Adobe)
    API                  Application Program Interface 
    API                  Lotus Printer Driver Info (IBM)
    API                  Photoshop Ink Colors Setup (Adobe)
    API                  WebObjects Interface (Apple Computer, Inc.)
    APIN                 InDesign Import/Export Filter (Adobe Systems Inc.)
    APJ                  ARM Project Manager 
    APJ                  Angel Debug Monitor Project 
    APJ                  Music Screen Saver File 
    APK                  GameSpy Arcade Service 
    APK                  Train Simulator Packaged Activity (Microsoft Corporation)
    APK                  Active Tutor Project (4C Media, Inc.)
    APL                  APL Workspace/Source 
    APL                  Gupta Team Developer Application Library File 
    APL                  ArcPad Ver. 6 Layer 
    APL                  Monkey's Audio Image Link File (Matthew T. Ashland)
    APLN                 InDesign 3rd Party Plug-in (Adobe Systems Inc.)
    APLWRKS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    APLWRKSGS            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    APM                  Aldus Placeable Metafiles (Adobe)
    APM                  ArcPad Ver. 6 
    APM                  AutoPlay Menu Studio Configuration (Indigo Rose Software)
    APM                  Alphacam Punch Parameter Macro (Planit)
    APO                  Apollo Scripts 
    APO                  Paperless Office (Abacre)
    APP                  Until Application 
    APP                  OrbWorks PocketC CE Application 
    APP                  ArcPad Ver. 5 Project 
    APP                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    APP                  SymbianOS Application (Symbian Ltd.)
    APP                  Lotus Symphony Add-in Application 
    APP                  Alphacam Punch Post (Planit)
    APP                  Executable Application 
    APP                  dBASE Application Object File 
    APP                  R:Base, Symphony, DR-DOS, FoxPro (or other) Application 
    APP                  Clarion for Windows Application (SoftVelocity)
    APP                  Video CD on CD-i Application 
    APP                  Psion Oval Application (Psion PLC)
    APP                  Ca Visual Objects Platform for Developer (CAVO) Project File (CA)
    APP                  Gupta Team Developer Normal Mode Application File 
    APP                  VTech Helio Application (VTech Holdings Ltd.)
    APPL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    APPLE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    APPLE2               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    APPLESHARE           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    APPLESOF             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    APPLEWKS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    APPM                 APPM Gereedschapskist Document (TPM-webapplicaties)
    APPS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    APPT                 Authentica Secure Office Protected PowerPoint Document (Authentica)
    APR                  Various Appraisal Software Appraisal File (PRIMEDIA Business Magazines & Media Inc.)
    APR                  ArcView Project File (ESRI)
    APR                  Approach 97 File (IBM)
    APR                  MicroSim PCBoard Aperture Information Report 
    APR                  Employee Appraiser Performance Review File 
    APRJ                 ApacheConf Project (Zecos Software)
    APS                  Advanced Patching Systems with Error Checking 
    APS                  ArcPad Ver. 5 Symbology 
    APS                  Flash 
    APS                  Visual C++ File (Microsoft)
    APT                  Alphacam Punch Tool (Planit)
    APT                  Adaptive Prediction Tree APT Encoded Bitmap (John Robinson)
    APT                  Nimbus 7 SAMS Data File 
    APT                  AutoPlay Menu Studio Project File (Indigo Rose Software)
    APT                  appTranslator Project (appTranslator)
    APT                  Approach Data View File (IBM)
    APT                  Gupta Team Developer Text Mode Application File 
    APV                  ApiViewer API Database (ActiveVB)
    APV                  Speedy Printed Circuit Board Design Software (Mania Technologie)
    APW                  Authorware Packaged FIle (Adobe Systems Incorporated)
    APW                  Any Password Password File (RomanLab)
    APW                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    APWKS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    APX                  C++ Appexpert Database (Borland Software Corporation)
    APX                  ArcPad Ver/ 6 XML File 
    APX                  Approach Paradox-specific Information File (IBM)
    APX                  Ability Office Photopaint Image (Ability Software)
    APX                  TheTomb Browser APX Webfile (Apoltix)
    APZ                  AutoPlay Menu Studio Shared Project File (Indigo Rose Software)
    AP_                  C Poet Compressed Disk1 File 
    AQ                   Applix Data 
    AQ                   Ancestral Quest Genealogy File (Incline Software, LC)
    AQ1                  AptiQuiz Test File (RJ Software)
    AQA                  AptiQuiz Test Result (RJ Software)
    AQF                  Aquifer Content Management System Aquifer-Generated Web Page (Liquid Development, LLP)
    AQL                  AOL Windows DLL (America Online, Inc.)
    AQM                  CP/M Information File 
    AQQ                  AQQ Plug-in (AQQ Team)
    AR                   Midtown Madness City Add-on (Microsoft)
    AR                   Javasoft JRE 1.3 Library File 
    AR                   AIX Small Indexed Archive 
    ARA                  ATI Radeon Video Driver 
    ARB                  Alphacam Router VB Macro (Planit)
    ARC                  Schedule+ 1.0 File (Microsoft)
    ARC                  Insight II Cartesian Coordinate Archive File (Accelrys Software Inc.)
    ARC                  Archive 
    ARCADE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ARCANE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ARCH                 Illum Source Dotfiles File 
    ARCH00               F.E.A.R. Game Archive (Monolith Productions, Inc.)
    ARCHIE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ARCHIMEDES           Speccy Snapshot 
    ARCHITECT            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ARCHITECT            London Architect Design File (BSS Audio)
    ARD                  Alphacam Router Drawing (Planit)
    ARE                  Infinity Game Engine Area Description (BioWare Corp.)
    ARF                  AOL Instant Messenger (AIM) Resource File (America Online, Inc.)
    ARF                  Active Tutor Data (4C Media, Inc.)
    ARF                  Automatic Response File 
    ARG                  AutoCAD Profile Export (Autodesk, Inc.)
    ARG                  HDF Fort_ps File 
    ARH                  ArHymo Hydrologic Simulation 
    ARH                  Archivers Definitions File (DN) 
    ARI                  Compressed Archive 
    ARI                  Aristotle Audio 
    ARJ                  Robert Jung ARJ Compressed Archive 
    ARJ                  Compressed Archive 
    ARK                  CP/M Information File 
    ARK                  QuArk Compressed File Archive 
    ARK                  Managing Your Money Achive 
    ARK3                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ARKCHEAT             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ARL                  AOL Organizer (America Online, Inc.)
    ARLOLD               AOL 6 Organizer (America Online, Inc.)
    ARM                  Alphacam Router Parmeter Macro (Planit)
    ARM                  Acorn ROM/Disk Image File 
    ARM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ARM                  Armadillo Software Protection System 
    ARMADA               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ARMOURFONT           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ARMX                 ASP.NET Source File (Microsoft Corporation)
    ARMY                 Impossible Creatures Collection of Created Creatures (Microsoft Corporation)
    ARN                  XMap Flight Plan or Flight Plan Statistic File (Delorme)
    ARN                  Astronomical Research Network 
    ARO                  SteelArrow Web Application Server Script (Tomahawk Technologies)
    ARP                  Alphacam Router Post (Planit)
    ARP                  Audition Play Data (Adobe Systems Inc.)
    ARP                  Ahsomme Accounting Package Report Layout Template (Simple Solutions)
    ARPX8                Techno Toys Arp-X8 MIDI Arpeggiator File 
    ARQ                  Remedy Action Request System Client Macro 
    ARQ                  Compressed Archive 
    ARR                  MicroImages Value Array (MicroImages, Inc.)
    ARR                  Remedy Action Request System Client Report 
    ARR                  Amber ARR Bitmap Image 
    ARR                  Atari Cubase Arrangement (Steinberg)
    ARS                  Artifax Report Editor 
    ARS                  After Effects Render Settings (Adobe)
    ART                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ART                  ArtCAM Pro and ArtCAM Insignia Model Data (Delcam plc)
    ART                  Blood Archive 
    ART                  Vort image 
    ART                  Xara Studio Drawing (Xara)
    ART                  pfs:Art Publisher 
    ART                  Alphacam Router Tool (Planit)
    ART                  Clip Art File 
    ART                  Bernina Artista 
    ART                  First Publisher Raster Graphic 
    ART                  ARTiSAN Real-time Studio 
    ART                  Art Director Image 
    ART                  Canon Crayola Art File 
    ART                  AOL Johnson-Grace Compressed File (America Online, Inc.)
    ART                  Central Point Scrapbook 
    ART                  Compressed Archive 
    ART                  Canon Crayola Art File 
    ART                  Another Ray Tracer Format 
    ART                  Duke Nukem 3D Archive 
    ARTGALLERY           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ARTICLE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ARTICLE1             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ARV                  Arsiv 
    ARV                  AutoRoute User Information (Microsoft Corporation)
    ARV                  Eschalon Setup Archive (Eschelon Development (24-2979 Panorama Dr., Coquitlam BC V3E 2W8, Canada))
    ARW                  Sony RAW Bitmap File (Sony Corporation of America)
    ARX                  AutoCAD Runtime Extension (Autodesk, Inc.)
    ARX                  ARX Compressed Archive 
    ARY                  Compaq SmartStart Scripting Toolkit File 
    AS                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AS                   Macromedia Flash Action Script 
    AS                   AppleSingle Encoded File (Apple Computer, Inc.)
    AS                   Applix Spreadsheet 
    AS                   Dispatch Alaska/Horizon Timetable File (Scott Andrew Borton)
    AS3                  Alternate Sources Data 
    AS3                  APL*PLUS II/386 and III/Windows Source Code 
    AS4                  AS/400 Client Access 
    ASA                  Visual InterDev File (Microsoft)
    ASA                  Active Server Document 
    ASAX                 ASP.NET Source File (Microsoft)
    ASB                  Alphacam Stone VB Macro (Planit)
    ASC                  3D Point File 
    ASC                  ASCII Text 
    ASC                  Pretty Good Privacy (PGP) Armored Encrypted File (PGP Corporation)
    ASC                  3D Studio 3D Scene (Autodesk, Inc.)
    ASC                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ASC                  ADONIS AdoScript File (BOC Group)
    ASCII                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ASCX                 Web User Control (Microsoft)
    ASD                  Astound Presentation 
    ASD                  Advanced Streaming Format Description (Microsoft Corporation)
    ASD                  Word Automatic Backup (Microsoft Corporation)
    ASD                  Live Audio Analysis File (Ableton AG)
    ASD                  Lotus Screen Driver (IBM)
    ASD                  Alphacam Stone Drawing (Planit)
    ASD                  ASD Archiver Compressed Archive 
    ASD                  Origins Embroidery Design (Origins Software Technologies, Inc.)
    ASE                  Adobe Swatch Exchange (Adobe Systems Incorporated)
    ASE                  Alice Scene Export File (Carnegie Mellon University)
    ASE                  Velvet Studio Sample 
    ASE                  3ds Max ASCII Export File (Autodesk, Inc.)
    ASF                  Lotus Screen Font (IBM)
    ASF                  Advanced Streaming Format (Microsoft Corporation)
    ASF                  APL*Plus /PC APL Shared File 
    ASF                  FIFA Football 2004 Audio File (Electronic Arts, Inc.)
    ASF                  Electronic Arts Music File (Electronic Arts, Inc.)
    ASF                  STATGRAPHICS Data File (Manugistics, Inc.)
    ASG                  Airport Tycoon Saved Game (Global Star Software)
    ASH                  Sony DSC-F828 Firmware Upgrade File (Sony)
    ASH                  TASM Assembler Header 
    ASHDISC              Burning Studio Disc Image File (Ashampoo GMBH & Co.)
    ASHPRJ               Burning Studio Project (Ashampoo GMBH & Co.)
    ASHX                 ASP.NET Web Handler File 
    ASI                  Assembler Include (Borland Software Corporation)
    ASIC                 ASIC Language Source Code 
    ASJ                  Aldfaer 
    ASK                  Ad-Aware Skin File (Lavasoft)
    ASK                  askSam Database 
    ASK                  Astrotite Skin (Fantiusen Software)
    ASK                  ALShow Media Player Skin (ESTsoft)
    ASL                  Photoshop Layer Style (Adobe)
    ASL                  AppSight Console Log (Identify Software Ltd.)
    ASL                  Quest Adventure Game (Axe Software)
    ASL                  Asmodo 9s 
    ASM                  Solid Edge Document (UGS PLM Solutions)
    ASM                  Alphacam Stone Parameter Macro (Planit)
    ASM                  Assembler Source Language 
    ASM                  Pro/ENGINEER Assembly File (PTC)
    ASM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ASM1                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ASM2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ASM3                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ASMVERS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ASMX                 ASP.NET Webservices Source 
    ASN                  Chemical Interchange Format (Old Form) 
    ASN1                 Asn Specs 
    ASO                  Assembler Object 
    ASO                  Astound Dynamite Objects 
    ASP                  Text Data 
    ASP                  Alphacam Stone Post (Planit)
    ASP                  Aspen/2 Data Manager (Tekton Software, Inc.)
    ASP                  Terminal Program 
    ASP                  Software Publishing Corp. 
    ASP                  Photoshop Separation Setup (Adobe)
    ASP                  Procomm Plus ASPECT Program 
    ASP                  ASAP WordPower WebShow Presentation 
    ASP                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ASP                  Photoshop Photo File Index (Adobe)
    ASP                  EROSION 3D Aspect File 
    ASP                  Astound Presentation 
    ASP                  Active Server Page 
    ASPHTML              ASP HTML (Microsoft Corporation)
    ASPI                 Audio Catalyst 
    ASPR                 ASProtect (ASPACK SOFTWARE)
    ASPX                 ASP.NET Source File (Microsoft)
    ASR                  Streets & Trips Route (Microsoft)
    ASR                  Photoshop Scratch Area (Adobe)
    ASS                  Assistance for the Blind 
    ASS                  Aegisub Advanced SubStation Alpha Subtitle File (Rodrigo Braz Monteiro)
    ASS                  Media Player Classic Subtitle 
    ASS                  Assembler/Disassembler 68HC11 
    ASS                  Flash 
    ASS                  ASS I/O 
    ASSIGNED             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ASSOC                Associate This File Association Database (Spearit)
    AST                  Technics Sx KN 6000 Keyboard All Custom Memory File 
    AST                  Ability Office Spreadsheet (Ability Software)
    AST                  Photoshop Separation Tables (Adobe)
    AST                  Need for Speed: Underground Audio File (Electronic Arts, Inc.)
    AST                  Claris Works Assistant File 
    AST                  Astound Multimedia 
    AST                  AstroGrav Simulation Data File (AstroGrav Astronomy Software)
    AST2                 Astrotite (Fantiusen Software)
    ASTROGS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ASV                  QuarkXpress Auto Save File (Quark, Inc.)
    ASV                  MATLAB Autosave File (The MathWorks, Inc.)
    ASV                  DataCAD Autosave File (DATACAD LLC)
    ASV                  Train Simulator Computed Activity File (Microsoft Corporation)
    ASV                  Photoshop Selective Color (Adobe)
    ASV                  Pegasus Mail Auto Saved Message (David Harris)
    ASV                  NotePad 2004 (Finale)
    ASWCS                avast! Antivirus Skin (ALWIL Software)
    ASWS                 avast! Antivirus Skin (ALWIL Software)
    ASX                  Advanced Stream Redirector File (Microsoft Corporation)
    ASX                  Alpha Five Temporary Index (Alpha Software, Inc.)
    ASX                  Cheyenne Backup Script 
    ASX                  ActiveSite Extension (Jonathon Rossi)
    ASXML                AudioPlayer XML Music Playlist (Adrian Wright)
    ASY                  LTspice/SwitcherCAD III Circuit Diagram (Linear Technology)
    AS_                  Winhelp Compressed File 
    AT                   Rescue Disk file 
    AT2                  WinControl DCX-AT3000 Motion Control Command Language File (Precision MicroControl Corp.)
    AT2                  Persuasion Ver. 2.0 Auto Template (Adobe)
    AT3                  WinControl DCX-AT3000 Motion Control Command Language File (Precision MicroControl Corp.)
    AT3                  Airport Tycoon Version 3 Saved Game (Global Star Software)
    AT3                  PSP Audio File (Sony)
    AT5                  NauticPath Marine Electronic Chart (Lowrance)
    ATB                  TechMed Annotation ToolBar (TechMed)
    ATB                  Alphacam Lathe VB Macro (Planit)
    ATC                  AutoCAD Catalog (Autodesk, Inc.)
    ATC                  AceText Collection (JGsoft - Just Great Software)
    ATC                  Sniffer Capture File (Network Associates Technology, Inc.)
    ATD                  Alphacam Lathe Drawing (Planit)
    ATD                  MS Systems Management Server SMSADDIN.EXE tool registration 
    ATF                  Sharp APL Workspace Export File 
    ATF                  Photoshop Transfer Function (Adobe)
    ATF                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ATF                  GenePix Axon Text File (Molecular Devices Corporation)
    ATG                  Netmino File 
    ATH                  Alphacam Lathe Thread (Planit)
    ATHEME               Amazing clock Theme (Amazing-Clock.com)
    ATK                  Andrew Toolkit Raster Object Bitmap 
    ATL                  Hal/Supernova/Lunar/LunarPlus Screen Readers/Magnifiers 
    ATL                  PROGNOSIS Atlas Document File 
    ATL                  ATP Assembler Source (Brancepeth Computer Publications)
    ATLT2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ATM                  Terragen Light and Atmosphere (Planetside Software)
    ATM                  .nugthat Player (Nuggets Unlimited, Inc.)
    ATM                  The Universal 3D Model/Texture File (The Universal Development Team)
    ATM                  Type Manager Data File (Adobe)
    ATM                  Alphacam Lathe Parameter Macro (Planit)
    ATMN                 Automation Anywhere Automation Macro (Tethys Solutions, LLC.)
    ATMSPHR              TerragGen Atmosphere (Planetside Software)
    ATN                  Photoshop Action File (Adobe)
    ATO                  Just Burn File (Sonic Solutions)
    ATOM                 Atom Feed 
    ATOOL                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ATP                  Visual Studio .NET ACT Project (Microsoft)
    ATP                  Anti Tracks Plug-in (RIGHT Utilities, Inc.)
    ATP                  Alphacam Lathe Post (Planit)
    ATP                  ATRAC Coded File 
    ATP                  Automise Project File (VSoft Technologies Pty Ltd.)
    ATPZ                 Automise Compressed Project File (VSoft Technologies Pty Ltd.)
    ATR                  MicroSim PCBoard Attributes Report 
    ATR                  Lightscape Material Library 
    ATR                  Atari Disk Image File (Atari)
    ATRAM                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ATROM                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ATS                  Unrelated Inventions Audiotools Plug-in 
    ATS                  Attityd Swedish Survey Program 
    ATT                  ZipLip Secure E-mail 
    ATT                  SeeYou Waypoint (Naviter)
    ATT                  AT&T Group 4 Bitmap (Fax) 
    ATT                  Web Form Data (POSTDATA.ATT) 
    ATT                  Alphacam Lathe Tool (Planit)
    ATW                  Attityd Swedish Survey Program 
    ATW                  AnyTime Deluxe PIM File 
    ATX                  Alphasoft Trueterm 2001 Dictionaries 
    ATX                  A ZIP-formatted Compressed Archive 
    ATZ                  Aston Compiled Theme (Gladiators Software)
    AT_                  Audio Utility Winatb Compressed File 
    AU                   uLaw/AU Audio File 
    AU                   Audacity Audio Block 
    AU3                  AutoIt Ver. 3 Script (HiddenSoft)
    AUCTIONS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AUD                  Audio File Used by Westwood Studios for Various Games 
    AUD                  DTS Audio Data Stream (Digital Theater Systems Inc.)
    AUD                  Winamp Media File (Nullsoft)
    AUDIO                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AUF                  Alphacam User Font (Planit)
    AUF                  ASAUpdaterV2 Update File (ASA Datec Datensystem GmbH)
    AUP                  ePocrates File 
    AUP                  Audacity Project File 
    AUR                  AutoREALM Map 
    AUR                  Sheffield Thesaurus file 
    AUS                  AutoREALM Symbols 
    AUS                  Agnitum Outpost Firewall Configuration 
    AUT                  Authentication File 
    AUT                  AutoIt (pre Ver. 3) Script (HiddenSoft)
    AUT                  Descent Manager Mission File 
    AUT                  GPSMan-autoMapic File 
    AUT                  Interactive Pictures iPIX Format 
    AUT                  PocketWear Car Lease Kit Vehicle Data File 
    AUT                  Signwave Auto-Illustrator File 
    AUT                  Xitami Webserver Admin Password File 
    AUT                  AUTHOR eLearning Module (Microcraft)
    AUT                  Autostart Adware Malware 
    AUT                  TLG Workplace CD Search File 
    AUTOCONF             UNIX File 
    AUTOCOPY             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AUTOLAUNCH           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AUTOREG              Mozilla (Netscape) Auto Registration Clue (Mozilla.org)
    AUX                  TeX/LaTeX Auxilliary Reference File 
    AUX                  Geospatial Imaging ERDAS IMAGINE and LPS Data File (Leica Geosystems LLC)
    AUX                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AUX                  ChiWriter Auxilliary Dictionary File 
    AUX                  MegaHAL (Ai Research)
    AV                   Final Draft AV Document File (Final Draft, Inc.)
    AVA                  Avagio Publication 
    AVA                  Photoshop Variations File (Adobe)
    AVB                  Inoculan Anti-Virus Virus Infected File 
    AVB                  Chat Character File (Microsoft Corporation)
    AVB                  Avid Bin File (Avid Technology, Inc.)
    AVC                  Kaspersky Anti-virus Toolkit Database (Kaspersky Lab)
    AVD                  Label Pro Data (Avery)
    AVD                  DOS7 File 
    AVE                  ArcView Avenue Script (ESRI)
    AVF                  AccountsVision Workbook (AccountsVision Ltd)
    AVG                  AVG Virus Information Database (GRISOFT, s.r.o.)
    AVG                  Grlib Urt Get Getami File 
    AVI                  Audio Video Interleave File 
    AVL                  Video Fun File 
    AVL                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AVL                  ArcView Theme Legend File (ESRI)
    AVM                  AVG Incremental Persistent File (GRISOFT, s.r.o.)
    AVM                  AVISO Geographic Information System Map File (Wojciech Pomianowski)
    AVN                  altaVENTE Application User Notes (altaVENTE)
    AVP                  ArcView Pallet File (ESRI)
    AVP                  Avid Project 
    AVR                  Audio Visual Research Sound 
    AVS                  Avid Settings 
    AVS                  Avisynth 
    AVS                  Application Visualization System Format 
    AVS                  Stardent AVS X Image 
    AVS                  Winamp Advanced Visualization Studio (Nullsoft)
    AVS                  Animation 
    AVT                  A3E Avatar-coding File 
    AVX                  ArcView Extension File (ESRI)
    AVX                  Avid Extensions 
    AW                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AW                   Free Pascal Archive File for Windows Version (FPC Development Team)
    AW                   Answer Wizard (Microsoft)
    AW                   Applix Word 
    AW                   HP AdvanceWrite Text File 
    AW2                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AW3                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AWA                  ActiveWords Application (ActiveWord Systems, Inc.)
    AWA                  Animation Works Accelerated Movie 
    AWARE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AWB                  Ad-Aware Backup File (Lavasoft)
    AWB                  ActiveWords WordBase (ActiveWord Systems, Inc.)
    AWB                  Adaptive Multi-Rate Wideband ACELP CODEC 
    AWBS                 Automated Weight and Balance System (AWBS) Data Storage File (Lockheed Martin Corporation)
    AWD                  Artweaver Painting (Boris Eyrich)
    AWD                  Award BIOS 
    AWD                  AWK Source Code 
    AWD                  FaxView Document (Microsoft)
    AWE                  Acrobat Bookmark XML File (Adobe)
    AWF                  Who Wants to be a Millionaire Archive 
    AWF                  Qui veut gagner des millions Archive 
    AWGS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AWGS1                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AWGS2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AWH                  ActiveWorlds Browser Help File (Activeworlds, Inc.)
    AWK                  AWK Script 
    AWK                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AWL                  Ad-Aware Language File (Lavasoft)
    AWM                  RenderWare Language Pack (Criterion Software)
    AWM                  Animation Works Movie 
    AWM                  AllWebMenus 
    AWP                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AWP                  MS Fax Key Viewer 
    AWP                  Akimbot Waypoint File 
    AWP                  ActiveWords Productivity Log (ActiveWord Systems, Inc.)
    AWP                  Ability Office Write Document Template (Ability Software)
    AWPSTRINGS           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AWR                  Ad-Aware Reference File (Lavasoft)
    AWR                  Telsis Digital Audio File 
    AWS                  AutoCAD Work Space (Autodesk, Inc.)
    AWS                  Ability Office Spreadsheet (Ability Software)
    AWS                  ActiveWords System File (ActiveWord Systems, Inc.)
    AWSHORTCUT           ActiveWords Shortcut (ActiveWord Systems, Inc.)
    AWT                  AbiWord Document File (The AbiSource Community)
    AWT                  AlbumWeb 
    AWW                  AWW WML Website Administrator Configuration File 
    AWW                  Ability Office Write Document (Ability Software)
    AWX                  Custom Application Wizard 
    AWX                  Fax 
    AWX                  DirectX 8.1 (Microsoft)
    AWZIP                ActiveWords Compressed Solution (ActiveWord Systems, Inc.)
    AW_                  Quartz Sets Compressed File 
    AX                   Modula-3 Linker Information for Libraries 
    AX                   ATI DA Interface Handler (ATI Technologies Inc.)
    AX                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    AX                   DirectShow Filter 
    AX                   MPEG-4 DVD Filter 
    AX2                  Apex Ver. 2 Sketch File (Apex Software)
    AX3                  Apex Ver. 3 Sketch File (Apex Software)
    AXD                  Actrix Technical 2000 (Autodesk, Inc.)
    AXD                  Label Pro Re-Index (Avery)
    AXE                  AutoRoute Export File (Microsoft Corporation)
    AXE                  ChordMaster 
    AXE                  Paradigm C++ Integrated Debugger File 
    AXE                  Pchord Powpro Compressed File 
    AXF                  Mira File List (Mirametrics, Inc.)
    AXF                  Apex Ver. 1 Sketch File (Apex Software)
    AXG                  AutoRoute Trip File (Microsoft Corporation)
    AXI                  MetaToolsFile Kai's Power Goo Picture Library 
    AXI                  Axiom Composite File Management System 
    AXL                  ArcIMS XML Project 
    AXL                  ASCET Archived XML (ETAS GmbH)
    AXLS                 Authentica Secure Office Protected Excel Document (Authentica)
    AXP                  AppGini MySQL Code Generator Project (BigProf Software)
    AXP                  AnvilCad CAD File (Manufacturing and Consulting Services (MCS), Inc.)
    AXP                  GraphStore eCADLite File 
    AXQ                  Ada ADA Editor File 
    AXR                  Telsis Digital Audio File 
    AXR                  AXIS Diplomat Report File (Systems AXIS Ltd.)
    AXR                  ElcomSoft Advanced Archive Password Recovery 
    AXS                  MindAvenue AXEL Stream File 
    AXS                  HTML; ActiveX Script 
    AXT                  Photoshop Replace Color/Color Range (Adobe)
    AXT                  ZENworks Desktop Management snAPPshot ASCII Application Object Template (Novell, Inc.)
    AXX                  AxCrypt Encrypted File (Axon Data)
    AY                   Dispatch Finnair Timetable File (Scott Andrew Borton)
    AY                   AY Chip Sound (Project AY)
    AYA                  RPG-Maker2000 Project Launcher Saved Game File 
    AYM                  Z80 Music Code With AY Music 
    AYS                  AY-Paste Paste File 
    AYU                  AY-Paste Uninstall Information 
    AZ                   WinDVD File 
    AZM                  CP/M Disk Fix File 
    AZZ                  AZZ Cardfile Personal Information Manager 
    A_T                  Atrain 
    B                    Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    B                    Modula-3 Base Program 
    B                    Limbo Implementation File 
    B                    Molconn-Z Format File 
    B                    Grand Theft Auto IIISave File (Take-Two Interactive Software, Inc.)
    B                    IRIS Editor Shortened Format File (eurobraille)
    B                    Brainf*ck Program 
    B                    Befunge Program 
    B                    BASIC Language Source 
    B                    Applause Batch List 
    B!K                  Flight Simulator Scenery File (Microsoft)
    B&W                  1st Reader Mono Binary Screen Image 
    B&W                  Image Lab 
    B&W                  Atari Black and White Graphics 
    B02                  ADP Payroll 
    B1                   Progress Database File (Progress Software Corporation)
    B1                   21st Century Mahjong 
    B16                  PCO Graphic 
    B1J                  BookJones 
    B1N                  1st Reader Binary Screen Image 
    B1S                  BookSmith 
    B2                   Progress Database File (Progress Software Corporation)
    B2D                  Brutus2D Source Code File (Pewter Software)
    B2Z                  B2Z dVS Model/Object 
    B30                  Ventura Publisher Printer Font 
    B32                  MS-DOS Bcc file 
    B3D                  BDE Multipath Movie Digital Viewer 
    B3D                  Blitz3D/Max/Plus Texture or Mesh File (Blitz Research Ltd.)
    B3D                  Boso View Express Structure Mesh Diagram (Mackoy)
    B3D                  3D Builder 
    B4                   Helix Nuts and Bolts File 
    B4S                  Winamp 3+ Playlist 
    B4U                  docPointer Visual ReadMe File (docPointer)
    B5                   CEDICT Chinese/English Dictionary (Erik E. Peterson)
    B5I                  IsoBuster CD/DVD Image File (Container) (Smart Projects)
    B5I                  BlindWrite Ver. 5 Image File (VSO-Software)
    B5I                  DAEMON Tools 
    B5S                  Numerical Controlled Machine File 
    B5T                  DAEMON Tools 
    B5T                  BlindWrite Ver. 5 Table of Contents File (VSO-Software)
    B64                  Base 64 MIME-encoded File 
    B6I                  BlindWrite Ver. 6 Image File (VSO-Software)
    B6T                  BlindWrite Ver. 6 Table of Contents File (VSO-Software)
    B8                   Piclab Raw Graphic File 
    B8                   PictureMaker Blue Channel Image Data (Cubicomp)
    BA$                  MS Compressed BAS Decompress with UNPACK.EXE. 
    BAB                  Babble! Data File (Korenthal Associates, Inc, 230 W 13th St, NYC 10011)
    BABAR                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BABER                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BAC                  Backup 
    BACH                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BACHTOLD             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BACK                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BACKGAMON            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BACKGRND             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BACKGROUND           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BACKPACK             Backpack Word List (Mammoet Software)
    BACKUP               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BACKUP               pgAdmin Database Backup (pgAdmin Project)
    BACKUP               ISAPI Loader Backup of Current ISAPI Application (EGGcentric)
    BACKUP               Ad-Aware Reference File (Lavasoft)
    BACLOOP7             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BAD                  Oracle BAD File 
    BAD                  Rime Mailer Address File 
    BAD                  Brutus Application Definition File 
    BAD                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BADONGO              BaDonGo File Information (BaDonGo.com)
    BAF                  Infinity Game Engine Script (BioWare Corp.)
    BAG                  OS/2 Netfinity Manager Sysinfo File 
    BAG                  Bag Archive File 
    BAG                  NoX 
    BAG                  AOL 6 Organizer (America Online, Inc.)
    BAG                  Emperor: Battle for Dune Archive 
    BAG                  PMMail 2000 Mail Index File (Blueprint Software Works)
    BAG                  PMMail/2 Mail Index File (V.O.I.C.E. (Virtual OS/2 International Consumer Education))
    BAG                  AOL Instant Messenger (AIM) Buddy List (America Online, Inc.)
    BAH                  Infinity Game Engine (BioWare Corp.)
    BAK                  Backup 
    BAK                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BAL                  Ballade Music Score 
    BAL                  BAL Borland Programming Language Source 
    BALANCE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BALL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BALL2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BAM                  Bob's Adlib Music 
    BAM                  Infinity Game Engine Animation and Multi-frame Graphics (BioWare Corp.)
    BAM                  Bravo! Archived Media (Thought Studios, Inc.)
    BAM                  OHRRPGCE Music File (Hamster Republic)
    BAM                  Infinity Game Engine Zlib Compressed Animation and Multi-frame Graphics (BioWare Corp.)
    BAN                  Creatacard Banner Project (Broderbund)
    BAN                  MegaHAL Non-Keyword List (Ai Research)
    BAN                  Sierra Print Artist Banner 
    BAND                 GarageBand Song (Apple Computer, Inc.)
    BANGER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BANJ                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BANK                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BANKSW               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BANNER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BAR                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BAR                  Age of Mythology 
    BAR                  dBASE Application Generator Horizontal Menu Object 
    BAS                  VBDOS Pro 1.0 Source 
    BAS                  BASIC Source Code 
    BAS                  e.Report Designer Actuate Basic Source (Actuate Corporation)
    BAS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BAS                  QuickBASIC Extended v7.1 Source 
    BAS                  HBasic Source Code 
    BAS2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BAS7                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BASE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BASKETBALL           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BASS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BASS1                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BASS2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BASSD                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BASSDRU2             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BASSG                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BAT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BAT                  Batch Processing (Microsoft)
    BATCH                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BATH                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BAV                  AVG The Bat! Add-in (GRISOFT, s.r.o.)
    BAV                  The Bat! E-mail Client Anti-Virus File (RITLABS)
    BAVANT               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BAY                  Kodak/Roper Bayer Picture Sequence 
    BB                   Starfleet Battles DOS Game 
    BB                   Papyrus Database Backup 
    BB                   Blitz3D/Max/Plus Source Code (Blitz Research Ltd.)
    BB                   BrowserBob Project File (BrowserBob weblications)
    BB                   Art-lantis Billboard Management File (ABVENT)
    BB2                  XBBS-OS/2, XGROUP, AV and ZTree Archiver Control File 
    BBC                  BBC BASIC for Windows File (R. T. Russell)
    BBC                  Babya Photo Workshop Color Collection (Babya Software Group)
    BBE                  Brainbloom Document Exchange File (Gael Limited)
    BBF                  FRAMES Body Burden Concentration File 
    BBG                  CyberSports for Basketball Game File (CyberSports, Inc.)
    BBK                  WWTBAM Game Component 
    BBL                  TeX/BibTeX Bibliographic Reference File 
    BBM                  Deluxe Paint Image File 
    BBM                  BDBZM (Project BDBZM)
    BBMA                 Bonzibuddy 
    BBNNG                BBN BitGraph Terminal Display Pixel Data (DPD) Sequence Bitmap 
    BBP                  BitBeamer Split File 
    BBP                  Bezier Surface File 
    BBQ                  BitBeamer Transfer Queue File 
    BBRKPUZZLE           BrainsBreaker Puzzle (Juan Trujillo)
    BBRKSAVE             BrainsBreaker Started Puzzle Status (Juan Trujillo)
    BBS                  FTN Software Hudson-style Messagebase 
    BBS                  Bulletin Board System Text 
    BBS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BBT                  CyberSports for Basketball Team File (CyberSports, Inc.)
    BBZ                  Bitblaze Audio File 
    BC                   BitComet Incomplete Download (BitComet.com)
    BC                   Low Level Virtual Machine (LLVM) Bytecode (Compressed) (University of Illinois at Urbana-Champaign Computer Science Department)
    BC                   Low Level Virtual Machine (LLVM) Bytecode (University of Illinois at Urbana-Champaign Computer Science Department)
    BC                   Mikmak 203 file 
    BC                   Sierra Print Artist Business Card 
    BC!                  BitComet Incomplete Download (BitComet.com)
    BC3                  BCC 3 Make File 
    BC3                  Fiebdc-3 Format File 
    BCA                  Railroad Tycoon 3 Building (PopTop Software Inc.)
    BCA                  Allplan 2005 Backup Catalog File (Nemetschek A.G.)
    BCB3                 BC3 Make File 
    BCC                  C++ File/Makefile (Borland Software Corporation)
    BCC                  Calendar Creator 8 Collection 
    BCD                  OfficePrinter Document (Moss Bay Software)
    BCD                  Turbo Pascal DOS File (Borland Software Corporation)
    BCD                  Binary Cartographic Data File 
    BCD                  Cosmi Business Card Maker 
    BCF                  The Sims 3D Body Mesh File (Electronic Arts (Maxis))
    BCF                  ConfigSafe Snapshot Index 
    BCF                  Business Card Designer Pro Business Card Design (Belltech Systems, LLC)
    BCH                  Datalex Entry Point 90 Data File (Phoenix Software)
    BCH                  dBASE Application Generator Batch Process Object 
    BCI                  Advisor Computer Profile Report (Belarc)
    BCK                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BCK                  VMS/VAX BaCKup 
    BCK                  Backup 
    BCKP                 Ad-Aware Quarantine File (Lavasoft)
    BCM                  Works Communications File (Microsoft)
    BCM                  Compaq Easy Access Keyboard Driver 
    BCMR                 BC Framer (Boise Cascade)
    BCN                  Business Card Pro Design 
    BCO                  MicroSim PCBoard Backward Engineering Change Order 
    BCO                  Bitstream Fontware 
    BCP                  C++ Makefile (Borland Software Corporation)
    BCP                  Business Card Designer Plus Design File (CAM Development)
    BCS                  Infinity Game Engine Compiled Script (BioWare Corp.)
    BCS                  Yamaha Basic Style (Yamaha Corporation of America)
    BCS                  Windows95 Browse Information 
    BCS                  Brian's Compression Shell (BCS) Image Animation/Video Shell 
    BCT                  Clarion for Windows Backup Dictionary (SoftVelocity)
    BCT                  Business Card Designer Template 
    BCW                  C++ Version 4.5 Environment Settings (Borland Software Corporation)
    BCX                  Beam Report Standard Definition File (SYSTEAM)
    BD                   Solaris Browser Data File (Sun Microsystems, Inc.)
    BD                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BD4                  West Point Bridge Designer Ver. 4 Bridge Design 
    BDB                  CHASE Evaluation and Audit System Executable (Health And Safety Technology And Management Ltd.)
    BDB                  Works Database File (Microsoft)
    BDBZM                BDBZM (Project BDBZM)
    BDC                  West Point Bridge Designer 
    BDC                  Lingvo dictionary 
    BDD                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BDE                  Pro/ENGINEER Temporary File (PTC)
    BDE                  Borland Database Engine (Borland Software Corporation)
    BDF                  Egret Data File 
    BDF                  Brother BES-100E Embroidery Software File (Brother Industries, Ltd.)
    BDF                  Bupdate Binary Update File 
    BDF                  Backup To CD-RW Backup Definition File 
    BDF                  RD Birthday Reminder Data File (Christian Weiske)
    BDF                  TXRDWIN X-ray Diffraction Pattern (Omni Scientific Instruments, Inc.)
    BDF                  Quartus II Block Design File (Altera Corporation)
    BDF                  UNIX Font File 
    BDF                  Glyph Bitmap Distribution Format (Adobe)
    BDF                  NuGraf Scene file 
    BDF                  X11 Bitmap Distribution Format 
    BDF                  West Point Bridge Designer 
    BDF                  E-Mail Detective Binary Data Fragment (Hot Pepper Technology, Inc.)
    BDG                  Badge Designer 2000 Photo ID 
    BDG                  BudgetExpress Data File (De Marque Inc.)
    BDI                  Pro/ENGINEER Temporary File (PTC)
    BDK                  OS/2 Saved SKF Format File (IBM)
    BDL                  OneSpace Designer Bundle (CoCreate)
    BDM                  Pro/ENGINEER Temporary File (PTC)
    BDMV                 Blu-ray Information File (Blu-ray Disc Association)
    BDR                  Publisher Border (Microsoft)
    BDRUM1               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BDSPROJ              Developer Studio Project File (Borland Software Corporation)
    BDT                  Lingvo Dictionary 
    BDT                  TAC Browser Script (Swoot)
    BDW                  Budget For Windows Data File (Snowmint Creative Solutions LLC)
    BDX                  FUSE Science Pipeline Bad Data File 
    BDX                  JavaBib Index File 
    BDX                  BarBack Drink Exchange File 
    BE                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEA                  Beamer Song File (The Beamer Project)
    BEACHBALL            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEACON               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEAM                 Erlang Compiled Erlang Code (Telefonaktiebolaget LM Ericsson)
    BEANS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEAT                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEATWAVE             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEC                  Break Even Calculator 
    BED                  InfoSnorkel Blue Elephant Definition 
    BED                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEE                  Beeline-AIR System File 
    BEE                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEE                  The Bee Archiver Compressed Archive 
    BEEP                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEF                  Boomi Integration Platform Setup Information (Boomi, Inc.)
    BEGIN                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEGINNERS            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEGINNING            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEHEADER             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEHINDTHEWH          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEJEWELED2DELUXESAVEDGAME Bejeweled Ver 2 Saved Game File (PopCap Games, Inc.)
    BEL                  Beamer Layout File (The Beamer Project)
    BEL                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEL                  Belon Compressed Archive 
    BEL                  HPXView Binary Event Log Archive (Haliplex Communication Systems)
    BELL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BELL2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BELLS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BELOW                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEN                  Syssoft Sandra File 
    BEP                  Beamer Playlist (The Beamer Project)
    BER                  German Bericht Report File 
    BES                  MOBY Database (Bitter End Systems)
    BETWEEN              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEX                  Kashmir 3D Transparent Color Definition 
    BEX                  British National Bibliography File 
    BEX                  BOOM Extensions To Dehacked 
    BEX                  Pretty Good Privacy (PGP) Binary Extracted Public Key (PGP Corporation)
    BEXPK                Pretty Good Privacy (PGP) Binary Extracted Public Key (PGP Corporation)
    BEYOND               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEYOND2000           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BEZ                  Bitstream Fontware 
    BEZ                  Bezier Surface File 
    BF                   Beyond Good & Evil Game Archive (Ubi Soft Entertainment)
    BF                   Befunge Program 
    BF                   Brainf*ck Program 
    BF2                  Bradford 2 Font 
    BF2                  Bidfax 2000 SE Database (Management Computer Controls, Inc.)
    BFA                  Blowfish-encrypted File 
    BFB                  Ralcgm File 
    BFC                  Windows 95 Briefcase File 
    BFE                  bcrypt Blowfish Encrypted File (Johnny Shelley)
    BFF                  AIX Backup File Format 
    BFF                  WorldToolKit Neutral File Format 
    BFFM                 B17 Flying Fortress Game File 
    BFL                  Colin McRae Rally 2 Data File 
    BFL                  Big Flexible Line Interpretation (C64) Bitmap 
    BFL                  yBook eBook (Spacejock Software)
    BFLI                 Big Flexible Line Interpretation (C64) Bitmap 
    BFM                  UNIX Font Metrics File 
    BFN                  Brother Embroidery Software Font (Brother Industries, Ltd.)
    BFR                  NI Observer Capture File 
    BFS                  Tivoli Storage Manager File (IBM)
    BFX                  BitFax Document (Bitware)
    BG                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BG                   Lotus Agenda File 
    BG                   Backgammon for Windows Game File (Microsoft)
    BGA                  OS/2 Bitmap Graphic Array (IBM)
    BGB                  Chat Background Graphic (Microsoft)
    BGDB                 Global Virtual Accademy E-Learning File 
    BGI                  Borland Graphic Interface (Borland Software Corporation)
    BGI                  BGInfo Configuration File (Bryce Cogswell)
    BGL                  Babylon Glossary File (Babylon.com Ltd.)
    BGL                  Flight Simulator Scenery File (Microsoft)
    BGM                  Jane's Combat Simulations F-15 
    BGN                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BGO                  BigOven Recipe Card (sdmventures)
    BGT                  Quicken 2002 Internet Common File (Intuit Inc.)
    BGT                  Graphic Accounts Budget Data (FKJ Software)
    BGV                  BonanzaGraf 
    BGW                  Final Fantasy XI Music (Square Enix Co., LtdSquare Enix Co., Ltd)
    BGX                  BC45 BGI File 
    BH                   BlackHole Archive Format File 
    BH                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BH3                  Rise of Nations Unit Graphic (Microsoft Corporation)
    BHA                  Rise of Nations Unit Graphic (Microsoft Corporation)
    BHF                  pcAnywhere Host File (Symantec)
    BHI                  Partminer Lib file 
    BHL                  Partminer Lib file 
    BHS                  Rise of Nations Script (Microsoft)
    BHTML                BabuHTML Embedded Software File 
    BHX                  BinHex Compressed File ASCII Archive 
    BI                   Quick Basic or Visual Basic for DOS Include File (Microsoft)
    BI                   Progress Database Before Image File (Progress Software Corporation)
    BI                   Binary File 
    BI$                  MS Compressed BI Decompress with UNPACK.EXE. 
    BI7                  Rukus 110 File 
    BIB                  TeX/BibTeX Literature Database 
    BIB                  Papyrus Database 
    BIB                  Bibliography 
    BIB                  JavaBib BibTeX File 
    BIBTEX               BibTeX Document 
    BIC                  Neverwinter Nights Character Sheet (Bioware)
    BIC                  Civilization III Scenario 
    BID                  BidMaker 2002 
    BIE                  JBIG File 
    BIF                  BioWare Infinity Engine Archive of Resources (BioWare Corp.)
    BIF                  GroupWise Initialization File 
    BIF                  Image Capture Board Binary Black & White Image 
    BIF                  Binary Information File 
    BIF                  Baldur's Gate Archive 
    BIF                  UltraISO Boot Information File (EZB Systems, Inc.)
    BIF                  MagicISO Boot Information File (MagicISO  Inc.)
    BIF                  byLight Graphic 
    BIF                  Planescape: Torment Archive 
    BIFF                 XLITE 3D file 
    BIFF                 BioWare Infinity Engine Archive of Resources (BioWare Corp.)
    BIG                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BIG                  Home World 2 SGA Archive (Relic Entertainment Inc.)
    BIG                  Electronic Arts Game MOD File (Electronic Arts, Inc.)
    BIG                  Homeworld Compressed Data Archive (Sierra Entertainment, Inc.)
    BIG                  Fifa World Cup Game File 
    BIG                  Chinese Text 
    BIG                  Cartes Michelin 
    BIG5                 Chinese Text 
    BIGBLUEFONT          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BIK                  Bink and Smacker Game Video File Codec (RAD Game Tools, Inc.)
    BIK                  BioCharter Profile Backup 
    BIL                  ArcView Image File (ESRI)
    BILL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BIMD                 Blue Ink Meta-Data (Automated Architecture, Inc.)
    BIN                  CDRWin CD Image File 
    BIN                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BIN                  Linux Executable 
    BIN                  Perceptron Scanner File 
    BIN                  AVG Update File (GRISOFT Inc.)
    BIN                  Binary File 
    BIN                  TenCORE Plug-in Helper Application (TenCORE)
    BIN                  MacBinary Archive Format Often used on Macintosh. 
    BIN                  Micrografx Designer 7 Project Image 
    BIN                  SGI Powerflip 
    BINS3                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BINS4                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BIO                  OS/2 Bios File 
    BIO                  BioCharter Profile 
    BIO                  Infinity Game Engine Character Biography (BioWare Corp.)
    BIO                  BioArc Compressed Archive 
    BIP                  Blue Ink Project (Automated Architecture, Inc.)
    BIP                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BIP                  ArcView Image File (ESRI)
    BIRD                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BIT                  Worms Armageddon Imported Map 
    BIT                  X11 Bitmap Image 
    BIT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BIT                  Xilinx Integrated Software Environment Bitstream File (Xilinx, Inc.)
    BIT                  Worms World Party Imported Map 
    BITMAP               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BIU                  BidMaker 2002 Update 
    BIX                  Civilization III Scenario 
    BIX                  Index to a .BIB BibTeX Bibliography Database 
    BIX                  BIX Archiver Compressed Archive 
    BIZ                  Print Shop Business Card File (Broderbund)
    BIZ                  Creatacard Business Card Project (Broderbund)
    BIZ                  BIZ dVS Model/Object 
    BIZ                  ShowBiz Project File (ArcSoft, Inc.)
    BIZ                  PrintMaster Gold Document (Broderbund)
    BIZZARELOVETRIANGLE  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BJF                  NTI Backup NOW! File 
    BJG                  DESQview (Symantec)
    BJK                  Scanner Recognita file 
    BJL                  BJL Business Systems Active Translation Document (The BJL Group)
    BJO                  Tabrite Banjo Tablature (Janet Davis Music)
    BK                   JetFax Faxbook File 
    BK                   FrameMaker BookFile Document (Adobe)
    BK                   LinguAssist File 
    BK                   Backup 
    BK                   Movie Magic Screenwiter Screenwriter Document Backup (Write Brothers, Inc.)
    BK!                  Backup 
    BK$                  Backup 
    BK1                  Bach Preludes and Fugues MIDI akmi Source, Book 1 
    BK1                  Backup 
    BK1                  VirtuaGirl Graphics Container (Totem Entertainment)
    BK1                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BK2                  VirtuaGirl Graphics Container (Totem Entertainment)
    BK2                  Bach Preludes and Fugues MIDI akmi Source, Book 2 
    BK2                  Backup 
    BK3                  VirtuaGirl Graphics Container (Totem Entertainment)
    BK3                  Backup 
    BK4                  Backup 
    BK5                  Backup 
    BK6                  Backup 
    BK7                  Backup 
    BK8                  Backup 
    BK9                  Backup 
    BKC                  Collectorz.com Book Collector Book Collection Data (Collectorz.com)
    BKC                  Backup4all Backup Catalog File (Softland)
    BKF                  Moto Racer Archive 
    BKF                  WindowsXP Backup Utility Backup Set (Microsoft)
    BKG                  Background File 
    BKG                  UWXAFS Binary Format Data File 
    BKGND                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BKI                  IBM BookManager Backup Index 
    BKI                  Backup Index 
    BKI                  IBM Softcopy Reader (Bookmanager) Bookshelf (and Book) Index File (IBM)
    BKP                  Backup 
    BKR                  ReplaceEm Fileset (Custom Fit Software)
    BKR                  Bruker Aspect NMR Data File 
    BKS                  BookManager Read Bookshelf (IBM)
    BKS                  Works Spreadsheet Backup (Microsoft)
    BKS                  Windows 2000 Scheduled Backup Script (Microsoft)
    BKS                  IBM Softcopy Reader Bookmanager Book File (IBM)
    BKT                  Zortech 3 tools Samples File 
    BKU                  Timeslips Backup (Peachtree Software)
    BKW                  FontEdit Fontset Mirror Image 
    BKY                  bookkey Bookmark/Keyword Data (Michael Valentiner-Branth)
    BL                   Binary Linework TIFF Bitmap 
    BLA                  Black Color Separation (Adobe)
    BLA                  Blat E-mail 
    BLA                  Blaise Data Model (Statistics Netherlands)
    BLACK                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLACKMORE            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLANK                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLAST                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLASTER              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLASTERBALL3SAVEDGAME BlasterBall Ver. 3 Saved Game (WildTangent, Inc.)
    BLB                  ACT! Database File (Best Software CRM Division)
    BLB                  Blorb Game Resource Package 
    BLB                  DreamWorks Resource Archive 
    BLC                  BIACORE Instruments and BIAlite Project (Biacore)
    BLD                  Weinberger Picture Sequence 
    BLD                  Yoot Tower Saved (Game) Building (Sega)
    BLD                  EnergyPro (EnergySoft, LLC)
    BLD                  MegaPaint 
    BLD                  Developer Studio MAK (Microsoft)
    BLD                  RealFlight Radio Control Airplane Description (Knife Edge Software)
    BLD                  VersaPro Block Build Info 
    BLD                  My Personal Programmer Project File (M:Pact Technologies)
    BLD                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLD                  BASIC Bload Graphics 
    BLD                  ACT! (Best Software CRM Division)
    BLD2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLDITAL              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLE                  Photoline4 Blends file 
    BLE                  BIAevaluation Ver 2.x Project (Biacore)
    BLEND                Blender 3D File 
    BLEND                Blender Publisher Data File (Stichting Blender Foundation)
    BLF                  Beast 2.02 Trojan File 
    BLF                  DriveBlockEditor Function Block Programming Data (Rockwell Automation)
    BLF                  Rally Online Car Texture File (GameSpy Industries)
    BLF                  Telerate Workspace File 
    BLF                  BIACORE Instruments and BIAlite Project (Biacore)
    BLF                  BancTec Scanner Driver (Kofax)
    BLF                  ObjectStore Distribution File (Franz Inc.)
    BLF                  Advanced Computer Architecture Car Simulation 
    BLG                  ARCSOLO Backup Log 
    BLG                  BibTeX Log 
    BLG                  MicroSim PCBoard Backward ECO Log File 
    BLG                  Windows Binary Performance Log 
    BLI                  BLINK Archiver Archive (De Trans Software)
    BLINK                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLK                  Amiga IFF (Amiga)
    BLK                  Lightscape Block Library 
    BLK                  VersaPro Block 
    BLK                  Wavefront Image (Alias)
    BLK                  WordPerfect Temporary File (Corel Corporation)
    BLK                  ProWORX Nxt Block Data (Schneider Electric)
    BLK                  Win32Forth Block Data 
    BLL                  VBS/European-A Worm 
    BLL                  INPRISE Format Language Library (Borland Software Corporation)
    BLN                  Rise of Nations Campaign Media File (Microsoft)
    BLOB                 Cosmic Blobs Model (SolidWorks Corporation)
    BLOCK                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLOCKS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLOSSOMS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLP                  Blizzard Game Picture (Blizzard Entertainment)
    BLR                  BIACORE Instruments and BIAlite Project (Biacore)
    BLS                  Brush Lettering Saved Brush Lettering Style (EK Success, Ltd., pcCrafter, Inc.)
    BLS                  BIACORE Instruments and BIAlite Project (Biacore)
    BLT                  Wordperfect for DOS (Corel Corporation)
    BLT                  CATC Merlin Bluetooth Trace Files (Computer Access Technology Corporation (CATC))
    BLT                  AOL Instant Messenger (AIM) Saved Buddy List File (America Online, Inc.)
    BLU                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLU                  blue Database (walkingtoyou.com)
    BLUE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLUEBERRY            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLUEESMERALDA        Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLUEMONDAY           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLURB                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BLW                  ArcView World File For BIL Image (ESRI)
    BLY                  Fifa World Cup Game Art Loadgame file. 
    BLZ                  Bitmap Graphic 
    BLZ                  Emblaze.Creator File 
    BLZ                  Serial Number File 
    BLZ                  Twinkle Bulbs Bulb File 
    BLZ                  Blazon Heraldic Shield 
    BLZ                  N-Gage Game File (Nokia)
    BM                   X Windows System Bitmap 
    BM                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BM$                  Modem Bitware Fax disk2 file 
    BM1                  Apogee BioMenace Data 
    BM2                  Subspace/Continuum Graphic 
    BMAP                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BMB                  SLICS Bounding Box Information 
    BMB                  Pro/ENGINEER Temporary File (PTC)
    BMC                  IBM Voice Type languages Newuser File 
    BMC                  Embroidery Graphic File 
    BMD                  Mu Online Data File (WEBZEN, Inc.)
    BME                  Beatmania Music Simulator File (Konami Broadcasting)
    BMF                  ByteMap Font 
    BMF                  BrainMine Mind Map (Neural Matters Limited)
    BMF                  Corel Flow Image File (Corel)
    BMF                  BMF Bitmap 
    BMF                  FloorPlan Project File (IMSI (International Microcomputer Software Inc.))
    BMG                  Bert's Dinosaurs File 
    BMI                  3ds Max Executable (Autodesk, Inc.)
    BMI                  Buzz Instrument 
    BMI                  Blue Martini Import File 
    BMK                  BMC Software Patrol Icon File 
    BMK                  Help Bookmark 
    BML                  BPM Studio Skin Bitmap (ALCATech)
    BML                  Bean Scripting Framework (BSF) File 
    BML                  Alpha Five Image Library File (Alpha Software, Inc.)
    BMM                  Tacmi Pixia Palette File 
    BMP                  Windows OS/2 Bitmap Graphics 
    BMP                  Infinity Game Engine Palette Information (BioWare Corp.)
    BMP24                Bitmap Graphic 
    BMPENX               egis Encrypted BMP File (HiTRUST Inc.)
    BMS                  BeMusic Script 
    BMS                  Beatmania Music Simulator File (Konami Broadcasting)
    BMS                  Backup Magic Backup Set 
    BMT                  Ami Pro Button Image (IBM)
    BMT                  Alpha Five Image Library File (Alpha Software, Inc.)
    BMTP                 Bome's Midi Translator Project File (bome.com)
    BMU                  Infinity Game Engine BioWare Music Unit (BioWare Corp.)
    BMW                  Buzz Music File with Waves 
    BMW                  W32/Liji-A 
    BMX                  BMXdesign Extended Bitmap Graphic (Container) (HEIDENHAIN)
    BMX                  Alpha Five Image Library File (Alpha Software, Inc.)
    BMX                  Siemens Mobile Phone Animation (Siemens)
    BMX                  Buzz Music File 
    BMZ                  Compressed BMP File 
    BMZ                  Demon Stalkers: Raid on Doomfane 
    BM_                  WebCD Fread File 
    BM_                  Mumath 30 File 
    BN                   AdLib Instrument Bank 
    BN2                  Bojanov Encryptor Encrypted File (Bojanov Software)
    BN7                  Ox Object-Oriented Matrix Programming Language PcGive 7 Data File 
    BNC                  Bojanov Encryptor Encrypted File (Bojanov Software)
    BND                  DB2 Bind File 
    BND                  Flight Simulator Panels File (Microsoft)
    BND                  Typequick File 
    BNF                  IBM Voice Type Languages Map File 
    BNG                  RANGER.EXE 
    BNK                  ExamView Question Bank (FSCreations, Inc.)
    BNK                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BNK                  SimCity Game File (Electronic Arts, Inc.)
    BNK                  Bink Compressed Video/Audio File 
    BNK                  AdLib Instrument Bank 
    BNL                  MicroSim PCBoard Netlist File For The Board Layout 
    BNR                  Binary News Reader File 
    BNR                  Banner - Poster Graphic Banner 
    BNSKIN               BeatNik Clock Skin (Something Decent)
    BN_                  Ntgraph Ze32v215 file 
    BOA                  BOA Constrictor Archiver Compressed Archive 
    BOARD                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOB                  Bob Raytracer 
    BOB                  TAC Browser URL (Swoot)
    BOB                  BobDown Downloading Program 
    BOCHSRC              Bochs Configuration File (The Bochs Project)
    BOD                  Data Designer Ddbac File 
    BOD                  Roland Garros Tennis Terrain File 
    BOE                  Outlook Express Backup File (Microsoft)
    BOF                  IBM Voice Type Language Task Enroll File 
    BOI                  Botje Bot-file (TNHteam)
    BOL                  Booasm Compressed Archive Library (Microsoft)
    BOLD                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOLD6                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOLF6                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOM                  MicroSim PCBoard Bill of Materials 
    BOM                  OrCAD Schematic Capture Bill of Materials File (Cadence Design Systems, Inc.)
    BOM                  Pro/ENGINEER Temporary File (PTC)
    BOM                  Bill of Materials File 
    BOM                  Softshare Delta Business Object Model 
    BOMBER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BONES                xfit Ridgeline File 
    BONGO                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BONK                 BONK Lossless/Lossy Audio Compressor 
    BOO                  Image File 
    BOO                  IBM BookManager Book 
    BOO                  Kermit Protocol Binary Bootstrap File 
    BOO                  Compressed Archive File 
    BOO                  Book 
    BOOK                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOOK                 ICE Book Reader Book File (ICE Graphics)
    BOOK                 FrameMaker Book (Adobe Systems Inc.)
    BOOKS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOOST                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOOT                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOOT                 InstallShield (InstallShield Software Corporation)
    BOOT0                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOOTSKIN             BootSkin Windows Boot Skin (Stardock Corporation, Inc.)
    BOP                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOR                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BORDERS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BORED                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BORLAND              C Makefile (Borland Software Corporation)
    BOS                  Brotherhood of Steel Compressed File 
    BOSSFONT             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOT                  Robot Arena: Design & Destroy Robot (Gabriel Group, Inc.)
    BOT                  Frogbot Editor 
    BOT                  Soldat Bot Information File (Michal Marcinkowski)
    BOT                  LinkBot document 
    BOULDER              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOULESHP             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOUTON               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOW                  Back Orifice 2K Workspace 
    BOWL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOX                  MediaShout Presentation Script (MediaComplete)
    BOX                  Notes Mailbox 
    BOX                  Lemmings Revolution Archive 
    BOX                  VTrain (Vocabulary Trainer) Box File (Paul Raedle)
    BOX                  Calypso Email Mailbox (Rose City Software)
    BOX                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOX                  Foxmail E-Mail Box 
    BOX                  Systems Management Server (SMS) Temporary Holding Area (Microsoft Corporation)
    BOX                  ThrottleBox Multimedia Viewer 
    BOX                  Vbox Helper Application File (Aladdin Knowledge Systems)
    BOXES                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOXR                 Boxerjam Interactive Game System 
    BOY                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BOZ                  BZIP Over ZIP Compressed File Archive 
    BP                   BankPerfect Accounts File (Fabio Chelly)
    BP                   Binary Picture TIFF Bitmap 
    BPC                  Business Plan Toolkit Chart 
    BPC                  BuildPro File 
    BPD                  BizInt Smart Charts for Patents Data Transfer Format (BizInt Solutions)
    BPD                  DTLplot Beadpull Data File 
    BPD                  FARSITE Burn Period File (The Fire Sciences Lab (FiSL))
    BPD                  Cash Compass Plan (Palo Alto Software, Inc.)
    BPD                  Business Plan Pro Business Plan Data (Palo Alto Software, Inc.)
    BPD                  Data Designer Database Backup File 
    BPD                  PCIScope PCI and ISA PnP Subsystem Information (APSoft)
    BPD                  Pscript Binary PostScript Printer Description File 
    BPD                  catDB Template Property Dictionary 
    BPD                  Büro Plus NexT FlashFiler Database File (microtech GmbH)
    BPEL                 Fiorano BPEL Project File (Fiorano Software Inc.)
    BPF                  F-Secure Base Policy File (F-Secure)
    BPF                  Cash Compass Table Formatting (Palo Alto Software, Inc.)
    BPG                  Borland Project Group (Borland Software Corporation)
    BPI                  IBM Voice Type languages Newuser file 
    BPJ                  Bravo! Project (Thought Studios, Inc.)
    BPK                  C++ Builder Package File (Borland Software Corporation)
    BPKEY                Blender Publisher License Key (Stichting Blender Foundation)
    BPL                  AutoCAD R2000 Batch Plot List (Autodesk, Inc.)
    BPL                  Delphi Library (Borland Software Corporation)
    BPL                  Winamp Ver 3 Playlist File (Nullsoft)
    BPLIB                BrilliantPhoto Photo Library File (Brilliant Labs, Inc.)
    BPM                  Brazil r/s Photon Map Storage File (SputterFish, LLC)
    BPM                  Battle Planner Module (Battle Planner, LLC.)
    BPP                  Clarion for Windows Backup Application (SoftVelocity)
    BPP                  Taskmaster Batch Pilot Project (Datacap Inc.)
    BPP                  MUSICMATCH Burner Plus Project 
    BPR                  Reference Count 01 File 
    BPR                  BackupXpressPro Project File 
    BPR                  C++ Builder 6 Project (Borland Software Corporation)
    BPS                  Works Document (Microsoft)
    BPS                  XMap Flight Plan or Flight Plan Statistic File (Delorme)
    BPS                  Business Plan Toolkit 
    BPS                  Bulletproof FTP Settings 
    BPT                  Corel Bitmap Master File 
    BPT                  Belarc Product Key (Belarc)
    BPV                  BallPark Business Valuation Valuation Scenario File (BulletProof Business Plans)
    BPW                  ArcView World File For BIP Or BMP Images (ESRI)
    BPX                  BusiPlan File (Manager-Pro)
    BPX                  Truevision Targa Bitmap 
    BPX                  Lumena BigPix 
    BQW                  ArcView World File For BSQ Image (ESRI)
    BQY                  BrioQuery File 
    BR                   The Playa File 
    BR                   Bridge Script 
    BR                   Omnis7 Data (Omnis Software)
    BR1                  Wincmd Fle 
    BR3                  Bryce 3D Scene File (Corel)
    BR4                  Bryce 4 (Corel)
    BR5                  Bryce 5 (Corel)
    BRAIN                Msdev Common Addins File 
    BRAINCHIME           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BRAS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BRC                  Bryce 2 Scene File (Corel)
    BRC                  Onset Design Briefcase Briefcase (Onset Design)
    BRCC                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BRD                  Winning Bridge Saved Hand (Gerald Wilson)
    BRD                  WinBoard Project (Ivex Design International)
    BRD                  RPG Toolkit Board Design File (Christopher Matthews)
    BRD                  BizInt Smart Charts for Pharmaceuticals Data Transfer Format (BizInt Solutions)
    BRD                  EAGLE Layout Editor Board File (CadSoft Computer GmbH)
    BRD                  Allegro (Cadence Design Systems)
    BRE                  Breuckmann Scanner File 
    BREAK                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BREAST               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BREATH               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BREP                 Boundary Representation File Format 
    BREP                 Open CASCADE 3D Model File (Open CASCADE Company)
    BRF                  Brief 
    BRF                  Braille ASCII File 
    BRF                  Duxbury Braille Translator Cross Platform Braille File (Duxbury Systems, Inc.)
    BRI                  FirstClass 
    BRI                  Basic Rate Interface File 
    BRITANNIA            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BRK                  The Brake! Mailer REXX Script 
    BRK                  Brooktrout Fax-Mail File 
    BRL                  Ballistic Research Laboratory CAD 
    BRLCAD               Ballistic Research Laboratory CAD 
    BRN                  yBook Compiled eBook (Spacejock Software)
    BRN                  MegaHAL Brain File (Ai Research)
    BRN                  PersonalBrain Data (Brain) File (TheBrain Technologies Corp.)
    BRN                  BrainStorm Data File (Brainstorm Software Ltd)
    BRO                  Tree Professional Broadleaf Creator File 
    BRO                  Creatacard Booklet/Brochure Project (Broderbund)
    BRO                  Embroidery File 
    BRO                  BR-532 Digital Studio 
    BRP                  Tree Professional Broadleaf Creator Image 
    BRS                  Batchrun Batch File (Outer Technologies)
    BRS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BRS                  Corel Painter Brush File 
    BRS                  RoboHELP Browse Sequence 
    BRSH                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BRT                  Micrografx System4 Browser file 
    BRT                  PeoplePC Install Script (PeoplePC, Inc.)
    BRT                  Brooktrout Fax-Mail File 
    BRU                  Brutus Restore File 
    BRU                  Photoline4 Brushes File 
    BRUSH                Xerox Doodle Brush File 
    BRUSH                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BRUSH2               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BRW                  Application File Associated with financial institution(s) loan applications. 
    BRX                  Multimedia Browsing Index 
    BRX                  Beam Report Document (SYSTEAM)
    BRZ                  DbBRZ File 
    BS                   Printfox/Pagefox Bitmap 
    BS                   Infinity Game Engine Compiled Script (BioWare Corp.)
    BS1                  Apogee Blake Stone Data 
    BS2                  Basic Stamp 2 Code File 
    BSA                  BSARC Compressed Archive 
    BSB                  SWAT Sub-basin Output File 
    BSB                  MapInfo Sea Chart 
    BSC                  BinSCII Apple II Compressed Archive 
    BSC                  Source Browser Database 
    BSC                  Boyan Script 
    BSC                  Developer Studio Browser File (Microsoft)
    BSC                  Fortran Pwbrmake Object 
    BSCS                 Strategy2Act Balanced Scorecard File (AKS-Labs)
    BSCT                 Strategy2Act Balanced Scorecard Strategy File (AKS-Labs)
    BSD                  FacetWin Bitwise Backup Set (FacetCorp)
    BSD                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BSD                  Crossfire File 
    BSE                  CRiSP Harvest File 
    BSF                  Bridge-Soft Quarterly File 
    BSF                  BadukSearch/BadukTracer Selected Game File 
    BSF                  Rainbow Six: Rogue Spear Buildspace File (Red Storm Entertainment, Inc.)
    BSF                  JCanyon: Grand Canyon for Java DEM File 
    BSH                  BeanShell Script 
    BSH                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BSI                  BSPlayer Configuration 
    BSK                  Bryce 4 File (Corel)
    BSL                  BSPlayer Configuration 
    BSM                  SQLMatcher Project File (Berryware)
    BSN                  BSA Packing Program Archive (PhysTechSoft Ltd.)
    BSN                  MIDI File 
    BSP                  Star Trek Elite Force Map File (Raven Software)
    BSP                  Valve Game Map File (Valve Corporation)
    BSP                  Quake Map (Id Software, Inc.)
    BSP                  Half-life/TFC/CS Map 
    BSP                  Getic 3D Scene/Model File 
    BSP                  arahnoWare Enterprise Sever 6 BSP Code Editor 
    BSP                  BestAddress HTML Editor Site Project File (Multimedia Australia Pty. Ltd.)
    BSP                  Steam Source SDK Compiled Game Map File (Valve Corporation)
    BSQ                  ArcView Image File (ESRI)
    BSQ                  BCDown Download 
    BSQ                  Apple II BINSCII-encoded (Apple Computer, Inc.)
    BSQ                  Oracle Control File 
    BST                  Star Trek: Birth of the Federation Structure/Ship Info File 
    BST                  BibTeX Style File 
    BSU                  Brindys BriTray 
    BSV                  BASIC Bsave Graphics 
    BSW                  Office Business Scorecard Manager Workspace File (Microsoft Corporation)
    BSY                  VTrain (Vocabulary Trainer) Learning Cardfile (Paul Raedle)
    BSY                  FTN Soft Busy Flag 
    BSZ                  BSplayer Compressed Skin File (BST2K Networks)
    BS_                  Bookshelf Find Menu Shell Extension (Microsoft)
    BT                   VTBuilder Binary Terrain 
    BT3                  Control Runner Button Configuration File (MomSoft)
    BTAB                 Base Guitar Tabulature 
    BTC                  Becker Tools Compressed Archive 
    BTF                  Brother Scanner File 
    BTG                  FlightGear Scenery File (The FlightGear Crew)
    BTH                  Bathy Recorder Data File (Jeppesen Marine, Inc.)
    BTL                  American Civil War File (Interactive Magic)
    BTL                  BEATO Lite Single Assessment Data (Unisys)
    BTL                  Pokémon NetBattle Battle Replay File (Smogon)
    BTL                  INPRISE Format Language Library (Borland Software Corporation)
    BTL                  Inmos BooTabLe File 
    BTM                  4DOS Batch To Memory Batch File 
    BTN                  buttonFile File (ButtonWare Inc.)
    BTN                  JustButtons Animated Bitmap (Lincoln Beach Software)
    BTNDATA              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BTO                  Baytex Organix! 2001 Language Kit 
    BTOA                 Binary-to-ASCII Format 
    BTPC                 BTPC Encoded Bitmap (John Robinson)
    BTR                  Btrieve Database File (Pervasive Software Inc.)
    BTR                  FrontPage Binary-Tree Index (Microsoft)
    BTS                  Creative Digital Blaster Digital VCR File 
    BTT                  ICQ Find Users File (ICQ Inc.)
    BTU                  BEATO Unite Multiple Assessment Data (Unisys)
    BTW                  Steuer2001 File 
    BTW                  BarTender Barcode Label (Seagull Scientific, Inc.)
    BTX                  DB/TextWorks Database Term and Word Indexes 
    BTX                  SEEDIS Index 
    BTX                  BibTeX Document 
    BTY                  Railroad Tycoon 3 Building (PopTop Software Inc.)
    BU                   Pegasus Mail Temporary File (David Harris)
    BUB                  Ipix Spherical Panorama 
    BUB                  Photobubble File 
    BUBBLES              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUC                  Photoshop Elements Backup File (Adobe Systems Inc.)
    BUD                  Quicken Backup (Intuit Inc.)
    BUDGET               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUDGETS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUDGETX              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUF                  WinXL File 
    BUFFER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUFR                 Binary Universal Form for the Representation 
    BUFR                 Meteorological Data 
    BUG                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUG                  Bug (Problem) File 
    BUGFIXES             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUGS                 Generally Bugs File 
    BUGS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUILD                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUILDER              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUL                  Capella Report Card Data File (Castor & Pollux Software Inc.)
    BUL                  Holiday Lights Screen Saver Bulb File 
    BUM                  Poser Bump 
    BUM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUN                  Cakewalk Audio Bundle File (Cakewalk)
    BUN                  Monkey Island 3 
    BUNDLE               iMovie 3 Plug-in Bundle (Apple Computer, Inc.)
    BUNDLE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUNNY                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUNNY2               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUP                  PC-Bibliothek Update File 
    BUP                  Softcode Tracker Contact Manager Backup 
    BUP                  DVD Info File Backup 
    BUP                  Brother Embroidery System File 
    BUP                  CD Indexer 
    BUREAU               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BURGER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BURNING              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BURST                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUS                  Netmino File 
    BUSH                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUSYBOX              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUT                  Buttons! Button Definition 
    BUTTONS4             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BUY                  Movie Data File 
    BUYING               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BV1                  WordPerfect Overflow File (Corel Corporation)
    BV2                  WordPerfect Overflow File (Corel Corporation)
    BV3                  WordPerfect Overflow File (Corel Corporation)
    BV4                  WordPerfect Overflow File (Corel Corporation)
    BV5                  WordPerfect Overflow File (Corel Corporation)
    BV6                  WordPerfect Overflow File (Corel Corporation)
    BV7                  WordPerfect Overflow File (Corel Corporation)
    BV8                  WordPerfect Overflow File (Corel Corporation)
    BV9                  WordPerfect Overflow File (Corel Corporation)
    BVC                  IBM Voice Type Language Newuser File 
    BVH                  Biovision Motion File 
    BVI                  IBM Voice Type Language Newuser File 
    BVL                  Micrografx Picture Publisher 8 Textures File 
    BVS                  BVS Solitaire Collection 
    BW                   Silicon Graphics Raw Black and White Bitmap (Silicon Graphics, Inc.)
    BW                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BW1                  Byteworx FMEA² FMEA Database (Byteworx Inc.)
    BWA                  BlindWrite BWA Builder Physical CD Characteristics File (VSO-Software)
    BWB                  Visual Baler Spreadsheet File 
    BWC                  BeadWizard Color Palette (Black Hills Software)
    BWD                  Rally Online Car Description File (GameSpy Industries)
    BWG                  BrainWave Generator Binaural Beat File (Noromaa Solutions)
    BWI                  BlindWrite Ver. 4 Image File (VSO-Software)
    BWM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BWP                  BeadWizard Pattern (Black Hills Software)
    BWP                  Book Writer Project 
    BWR                  Kermit Beware Bug List 
    BWS                  BlindWrite Ver. 4 Subchannel Data File (VSO-Software)
    BWT                  CDmage Table of Contents 
    BWT                  BlindWrite Ver. 4 Table of Contents File (VSO-Software)
    BWU                  BlindWrite (VSO-Software)
    BWV                  Business Wave File 
    BWW                  Bagpipe Notation 
    BWZ                  WinImage Batch Configuration File 
    BXB                  BasicX Compiled Project Bytecode (NetMedia, Inc.)
    BXD                  Broken Cross Disk Manager Database (Kevin Fucik)
    BXF                  Man's Best Friend Software Breeder Exchange Format 
    BXM                  BX-24 Basic Express Compiler Map 
    BXP                  BootXP File 
    BXP                  BX-24 Basic Express Compiler Module List 
    BXRC                 Bochs Configuration File (The Bochs Project)
    BXU                  PictureGear Studio File (Sony Electronics Inc.)
    BXX                  blaxxun Contact 
    BXY                  ShrinkIt Apple II Compressed Archive 
    BY                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BY                   Spice Pspice File 
    BYO                  CartoExploreur Cartographic Map (Bayo)
    BYU                  Movie File 
    BZ                   Bzip UNIX Compressed File 
    BZ                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BZ1                  WinTOTAL Automatic Backup (a la mode, inc.)
    BZ2                  Bzip 2 UNIX Compressed File 
    BZ2                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    BZ2                  ZPoc Christian Music Networks User File List (ZPoc)
    BZA                  BZA Compressed Archive 
    BZABW                AbiWord (The AbiSource Community)
    BZF                  Textures and Other Information 
    BZIP2                Compressed Archive Format 
    BZN                  Star Trek Armada Map File 
    BZPATCH              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    B_T                  Eru/erd File 
    B_W                  Image Lab 
    B_W                  1st Reader Mono Binary Screen Image 
    B_W                  Atari/Macintosh Black and White Graphic 
    B~K                  Backup 
    C                    Check Point Site Configuration for Secure Remote 
    C                    UNIX Compact File Achive 
    C                    Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    C                    Desktop Color Separation Specification Cyan Layer 
    C                    C/C++ Program File 
    C                    ADS Source Code (Autodesk, Inc.)
    C                    Origin C Program File (OriginLab)
    C V                  Cvip File 
    C!_                  Audio Utility Midimach Compressed File 
    C#=                  C# (Microsoft)
    C++                  C++ Source Code 
    C--                  Sphinx C-- Source 
    C0                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    C00                  WinAce Compressed File Split Portion of Compressed File (e-merge GmbH)
    C00                  Ventura Print File 
    C01                  Genesis 2000 (Genesis 2000, Inc.)
    C01                  WinAce Compressed File Split Portion of Compressed File (e-merge GmbH)
    C02                  (C02-C38) Driveimage Ace File 
    C02                  WinAce Compressed File Split Portion of Compressed File (e-merge GmbH)
    C03                  WinAce Compressed File Split Portion of Compressed File (e-merge GmbH)
    C04                  WinAce Compressed File Split Portion of Compressed File (e-merge GmbH)
    C05                  WinAce Compressed File Split Portion of Compressed File (e-merge GmbH)
    C06                  WinAce Compressed File Split Portion of Compressed File (e-merge GmbH)
    C07                  WinAce Compressed File Split Portion of Compressed File (e-merge GmbH)
    C08                  WinAce Compressed File Split Portion of Compressed File (e-merge GmbH)
    C08                  Lord Of Destruction Game File 
    C09                  WinAce Compressed File Split Portion of Compressed File (e-merge GmbH)
    C1                   Pov-ray for Windows v3.1 Scenes Advanced Newltpot File (Persistence of Vision Raytracer Pty Ltd.)
    C10                  WinAce Compressed File Split Portion of Compressed File (e-merge GmbH)
    C13                  Charlie Packer 
    C2                   Pov-ray for Windows v3.1 Scenes Advanced Newltpot File (Persistence of Vision Raytracer Pty Ltd.)
    C2C                  CopyToDVD Project File (VSO-Software)
    C2D                  WinOnCD CD Image (Sonic Solutions)
    C2E                  NJStar Chinese Word Processor Dictionary File Index (NJStar Software Corp.)
    C2I                  Tt File 
    C2I                  Driver File 
    C2L                  Cequadrat WinonCD File 
    C2X                  WinonCD File 
    C3                   Pov-ray for Windows v3.1 Scenes Advanced Newltpot File (Persistence of Vision Raytracer Pty Ltd.)
    C39                  (C39-C56) Quicken Ace File 
    C3D                  Chem3D Chemical File (CambridgeSoft Corporation)
    C3D                  Coordinate 3D 
    C3D                  Cult3D Designer Intermediate File 
    C3D                  Micrografx 
    C3E                  Cype 
    C3T                  Chem3D Chemical Template File 
    C4                   Joint Engineering Data Management (JEDMICS) DoD Engineering Data Format 
    C4                   Edmics 
    C4                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    C4D                  Clonk Object Definition (RedWolf Design GmbH)
    C4D                  Cinema 4D 3D Model (MAXON Computer GmbH)
    C4F                  Clonk Round File (RedWolf Design GmbH)
    C4G                  Clonk System File (RedWolf Design GmbH)
    C4P                  Clonk Player File (RedWolf Design GmbH)
    C4S                  Clonk Scenario (RedWolf Design GmbH)
    C4V                  Clonk Video (RedWolf Design GmbH)
    C6                   Ashlar-Vellum 
    C64                  Commodore 64 Game ROM 
    C86                  Computer Innovation C86 C Program 
    C9                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CA                   Telnet Server Initial Cache Data File 
    CA                   FlexPlus Data File (DATAIR Employee Benefit Systems, Inc.)
    CA                   CLAN Conversion Analysis (Child Language Data Exchange System)
    CA0                  Installer Packed and Split File (Borland Software Corporation)
    CA1                  STABCAL (stability calculation for aqueous systems) File 
    CA1                  OzWin CompuServe E-mail/Forum Access Catalog File 
    CA1                  Delphi Install Disk11 File 
    CA2                  Delphi Install Disk12 File 
    CA2                  STABCAL (stability calculation for aqueous systems) File 
    CA3                  Delphi Install Disk9 File 
    CA4                  STABCAL (stability calculation for aqueous systems) File 
    CA4                  Delphi Install Disk10 File 
    CA4                  Electronics Workbench Circuit 
    CA5                  Delphi Install Disk11 File 
    CA7                  Beta 44 Job File 
    CA7                  STABCAL (stability calculation for aqueous systems) File 
    CAB                  Cabinet File (Microsoft Corporation)
    CAB                  Install Shield v5.x or 6.x Compressed File (InstallShield Software Corporation)
    CAB                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CAC                  Collage Alarm Clock Alarm File (Collage Alarm Clock)
    CAC                  dBASE IV Executable File 
    CAC                  Infoview (Microsoft)
    CACHE                Snacc-1.3 VDA File 
    CACHE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CAD                  Inago Rage Compact Arena Definition (Dejobaan Games)
    CAD                  Softdesk Drafix CAD File 
    CAD                  QuickCAD CAD Drawing (Autodesk, Inc.)
    CADET                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CAF                  Southern Company Terrestrial Data Acq 
    CAF                  Class Action Gradebook Secondary/College Edition Data File (CalEd Software)
    CAG                  Capella Gallery Data File (Capella Software)
    CAG                  Clip Gallery Catalog File (Microsoft)
    CAG                  CADfix ASCII Geometry File (ITI TranscenData)
    CAI                  SeeYou Flight Data (Naviter)
    CAISSE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CAJ                  CAJViewer 
    CAKEWALKWINDOWLAYOUT Cakewalk Window Layout (Cakewalk)
    CAL                  Creatacard Calendar Project (Broderbund)
    CAL                  PhotoSmart 850 Digital Camera Firmware Update (Hewlett-Packard Development Company, L.P.)
    CAL                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CAL                  CALS Raster 
    CAL                  Cakewalk Application Language Script (Cakewalk)
    CAL                  CyberAudioLibrary Link 
    CAL                  DOS Calendar Creator File 
    CAL                  Radiance Function File Format 
    CAL                  QV Map Geographic Coordinates (Touratech)
    CAL                  HTML Calendar Maker Pro 
    CAL                  Schedule Data (Microsoft)
    CAL                  SuperCalc Worksheet 
    CAL                  Calendar File 
    CALC                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CALCULATOR           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CALENDAR1994         Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CALL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CALS                 CALS Raster 
    CAM                  Calibrated Camera File Photomodeller 
    CAM                  Camtasia Studio Project/Producer Files (TechSmith Corporation)
    CAM                  Mill Data File (GraphiTech Ltd.)
    CAM                  Minolta Scanner 3D File (Minolta)
    CAM                  NASCAR Racing 2002 Replay Camera Settings (Sierra)
    CAM                  HouseCall AUTOEXEC.BAT Backup (Trend Micro Inc.)
    CAM                  Casio QV Digital Camera Image (Casio)
    CAMPROJ              Camtasia Studio Project File (TechSmith Corporation)
    CAMREC               Camtasia Studio Recorder File (TechSmith Corporation)
    CAMS                 Driver (Atari)
    CAMSHR               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CAMTHTR              Camtasia Studio Theater File (TechSmith Corporation)
    CAN                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CAN                  Canon Navigator Fax 
    CAP                  Code Architect Project File (Embedded Systems Academy)
    CAP                  Generic Network Capture Document 
    CAP                  WWG LinkView PRO Capture File 
    CAP                  WWG Domino Capture File 
    CAP                  Capella Music Sheet Data File (Capella Software)
    CAP                  Fluke Protocol Inspector Capture File 
    CAP                  Sniffer Capture File (Network Associates Technology, Inc.)
    CAP                  Agilent LAN Analyzer Capture File 
    CAP                  ProComm/Telix session Capture File 
    CAP                  Shomiti Surveyor Capture File 
    CAP                  Snoop Capture File 
    CAP                  Telix Session Capture 
    CAP                  Ventura Caption 
    CAP                  TTC FireBerd 500 PC Capture File 
    CAP                  NetMon Capture File (Microsoft)
    CAP                  PacketView Pro Packet Trace Data (Klos Technologies, Inc.)
    CAPITAL              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CAPROJ               ChartAble Project File (MRDC Ltd)
    CAPS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CAPS                 Visimetrics Digital CCTV Recording 
    CAR                  AtHome Assistant 
    CAR                  Carrara Environment 
    CAR                  Creatacard Quarter-Fold Card Project (Broderbund)
    CAR                  CardMaker Ver. 1.0-2.0 Card File (Avery)
    CAR                  NASCAR Racing 2002 Car Description File (Sierra)
    CAR                  NeoBook Cartoon (NeoSoft Corp.)
    CAR                  Viper Racing Car Description File 
    CAR                  PSA Cards Address Book (PSA Software)
    CAR                  Design-Your-CD Data File (Avery)
    CAR                  Insight II Cartesian Coordinate File (Accelrys Software Inc.)
    CAR                  SeeYou Airspace (Naviter)
    CAR                  Railroad Tycoon 3 Car Properties (PopTop Software Inc.)
    CAR                  Cellsprings/Web or DT Rule File (J.M.G. Elliott)
    CAR                  Carnivores / Ice Age Resource File (Action Forms, Ltd.)
    CAR                  CAR Archive Compressed Archive (MylesHi! Software)
    CARAMBA              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CARD                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CARD                 Funkiball Adventure Auto-Saved Game File (RealNetworks Inc.)
    CARDS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CAROL                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CAROLES              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CAS                  Quest Compiled Adventure Game (Axe Software)
    CAS                  FLUENT Case (Fluent Incorporated)
    CAS                  Atari Cassette Image File (Atari)
    CAS                  Videoton TVC Computer Tape File 
    CAS                  i2's Case Notebook Timeline Analysis 
    CAS                  Cascade DTP Ver. 3 Layout File (Price Media)
    CAS                  Casio Program 
    CAS                  Comma-delimited ASCII File 
    CAS4                 Cascade DTP Ver. 4 Layout File (Price Media)
    CASCII               CACTVS Chemical ASCII Format 
    CASE                 EnSight Results File (Computational Engineering International (CEI))
    CASE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CASECONVERT          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CASTLE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CAT                  TAPCIS Forum Library Catalog 
    CAT                  dBASE Catalog 
    CAT                  Advanced Disk Catalog Disk Catalog (Elcomsoft)
    CAT                  Casio Calculator Program (Casio)
    CAT                  Cognos Impromptu Database Report 
    CAT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CAT                  TablEdit Chord Catalog 
    CAT                  Catz Cat Description File (http://petz.ubi.com/)
    CAT                  Quicken IntelliCharge Categorization File (Intuit Inc.)
    CAT                  OzWin CompuServe E-mail/Forum Access Catalog File 
    CAT                  PhotoStudio (ArcSoft, Inc.)
    CAT                  Todd Osborne Directory Catalog 
    CAT                  Security Catalog (Microsoft)
    CAT                  Rational Rose Logical Package (IBM)
    CATALOG              CATIA Catalog (Dassault Systèmes)
    CATALOG              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CATANALYSIS          CATIA (Dassault Systèmes)
    CATDRAWING           CATIA Drawing (Dassault Systèmes)
    CATEGORIES           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CATFCT               CATIA (Dassault Systèmes)
    CATMATERIAL          CATIA Material Specification (Dassault Systèmes)
    CATPART              CATIA Part Description (Dassault Systèmes)
    CATPROCESS           CATIA Process Description (Dassault Systèmes)
    CATPRODUCT           CATIA Product Description (Dassault Systèmes)
    CATPRODUCT           CATIA (Dassault Systèmes)
    CATSHAPE             CATIA Shape (Dassault Systèmes)
    CATSWL               CATIA (Dassault Systèmes)
    CATSYSTEM            CATIA (Dassault Systèmes)
    CAULDRON             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CAVE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CAZ                  Computer Associates Archive 
    CA_                  Cakepro Compressed Audio File 
    CB                   Clean Boot File (Microsoft)
    CB                   Gigascape Virtual Town Project (Third Dimension Limited)
    CB                   Brief Macro Source Code 
    CB                   Crystal Button File 
    CB                   COMBAS Source Code (Living Byte Software GmbH)
    CB3                  CliBench Mk III SMP 0.7.10 
    CBC                  CubiCalc Fuzzy Logic System File 
    CBCP                 CodeBaby Production Studio Character Package (CodeBaby Corp.)
    CBD                  System DLL Catroot File 
    CBD                  WordPerfect Dictionary File (Corel Corporation)
    CBF                  Infinity Game Engine Archive of Resources (BioWare Corp.)
    CBF                  Calendar Builder Saved Calendar (RKS Software)
    CBF                  CoffeeCup Button Factory Button (CoffeeCup Software, Inc.)
    CBH                  ChessBase Chess Database File 
    CBI                  IBM Mainframe Column Binary Formatted File 
    CBIN                 CACTVS Chemical Binary Format 
    CBK                  National Construction Estimator 32 Costbook 
    CBK                  System DLL Catroot File 
    CBK                  CardScan Backup File (Corex Technologies)
    CBK                  Kreon Scanner File 
    CBK                  Giza Specifier Drawing Backup File (20-20 Technologies)
    CBL                  COBOL Program 
    CBL                  CD Box Labeler Pro (Green Point Software UK Ltd.)
    CBL                  Contextual Browsing Language Viewer 
    CBL                  RM/COBOL Development System Source File (Liant)
    CBLE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CBM                  Fuzzy Bitmap 
    CBM                  XLib Compiled Bitmap 
    CBMV                 Comic Book Millennium Comic Book Collection (Beaver Valley Software Company)
    CBN                  CBN Selector Smart Image (CBN Systems)
    CBN                  PaperMaster Cabinet (j2 Global Communications, Inc.)
    CBO                  COMBAS Object File / Compiled Program (Living Byte Software GmbH)
    CBO                  Interactive Training Bookmark (Microsoft Corporation)
    CBP                  Conlab Project (Apix Concrete Solutions)
    CBP                  CentraBuilder (CentraLearning International)
    CBP                  CD Box Labeler Pro (Green Point Software UK Ltd.)
    CBR                  CDisplay RAR Archived Comic Book File 
    CBS                  Borland Computer Based Training (Borland Software Corporation)
    CBS                  Computer Based Training (Microsoft)
    CBS                  Calendar Builder Style Sheet (RKS Software)
    CBS                  Cardbox Macro Script 
    CBS                  MasterWord Button Bar Configuration 
    CBT                  Computer Based Training 
    CBU                  Conlab Update Information (Apix Concrete Solutions)
    CBV                  ChessBase Archive File 
    CBW                  Cardbox Workspace 
    CBX                  Rational XDE (IBM)
    CBZ                  CDisplay ZIP Archived Comic Book File 
    CC                   Trojan Spymaster.A Text File 
    CC                   Picture Window's Color Curve File 
    CC                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CC                   Visual dBASE Custom Class File 
    CC                   dB2K Custom Class File 
    CC                   CC Language Source Code 
    CC                   C++ Program File 
    CC                   ADS Source Code (Autodesk, Inc.)
    CC                   KODAK Color Control Profile (KODAK)
    CC1                  Chem3D Cartesian Coordinates 1 
    CC1                  Briggs Softworks Calendar Commander 
    CC2                  ClubControl 2 
    CC2                  Calendar Commander Calendar File (Briggs Softworks)
    CC2                  Chem3D Cartesian Coordinates 2 
    CC3                  CUEcards Professional Database (Marcus Humann)
    CC5                  Calendar Creator 5.x 6.x File 
    CCA                  Multimedia Fusion File (Clickteam)
    CCA                  Common Component Architecture (CCA) Component (Dept. of Energy Laboratories)
    CCA                  CC:Mail Archive File (IBM)
    CCAD                 ClarisCAD data 
    CCB                  Visual Basic Animated Button Configuration 
    CCC                  WordPerfect Office Calendar File (Corel Corporation)
    CCC                  Curtain Call Native Bitmap Graphic 
    CCC                  Internet Chat (Comic Chat) Conversation/Chatroom Log (Microsoft)
    CCC                  FLEXnet Manager (Macrovision Corporation)
    CCD                  Vector CAD Program File 
    CCD                  CloneCD Control File (SlySoft Inc.)
    CCD                  IsoBuster Data Recovery File (Smart Projects)
    CCDOC                Rational CCDoctor Output (IBM)
    CCE                  Calendar Creator 2 Event File 
    CCF                  CommView Capture File (TamoSoft, Inc.)
    CCF                  CryptLoad Container File (CryptLoad)
    CCF                  GPS Pathfinder Office Configuration File (Trinble Navigation Limited)
    CCF                  Cumulus Database (Canto)
    CCF                  Cannondale Motorcycle ECM System Engine Calibration File (Cannondale Corp.)
    CCF                  Remote Control File 
    CCF                  OS/2 Multimedia Viewer Configuration File 
    CCF                  Card Studio Card File Template (Hallmark)
    CCF                  Symphony Communications Configuration File 
    CCF                  AudioReQuest Control File 
    CCH                  Fabasoft Components 
    CCH                  Corel Chart 
    CCH                  Forgotten Realms Unlimited Adventures (FRUA) Character Reference File 
    CCH                  Photomodeler Photochip Thumbnail File (OES Systems)
    CCH                  PhotoModeler 
    CCI                  CCITT Group 3 and Group 4 Encoding 
    CCITT                CCITT Group 3 and Group 4 Encoding 
    CCJ                  Crossword Compiler Compiled Crossword Applet (Antony Lewis)
    CCK                  Corel Clipart Format (Corel)
    CCL                  Intalk Communication Command Language 
    CCL                  CCtools Levelset (Translucent Dragon)
    CCM                  CC:Mail Mailbox (IBM)
    CCN                  IMSI Multimedia Fusion Express File 
    CCN                  Vitalize! Game File 
    CCN                  Calendar Constructer Calendar Data (Aspire Software)
    CCO                  CyberChat Data File 
    CCO                  Command Console Command File (DigitallyCreated)
    CCO                  XBTX BTX Graphics File 
    CCO                  MicroSim PCBoard Autorouter Session Output 
    CCP                  Two Brothers Software Encrypted File Information (Two Brothers Software)
    CCP                  CaptureCAM-PRO Screen Recording (Click-N-Learn PTY LTD)
    CCP                  CloneCD (SlySoft Inc.)
    CCP                  CalComp Color Palette File 
    CCP                  C Converter Profiles 
    CCPR                 CaptureCAM-PRO (Click-N-Learn PTY LTD)
    CCR                  Systems Management Server (SMS) Client Configuration Request (Microsoft Corporation)
    CCR                  Internet Chat Room Shortcut (Microsoft)
    CCR                  HealthFrame ASTM Continuity of Care Record (Records For Living, Inc.)
    CCRF                 Calcomp Raster Bitmap 
    CCS                  CCS-lib File 
    CCS                  CONKY CeZaR Stribas International 
    CCT                  DesignWorks Schematic File (Capilano Computing)
    CCT                  Macromedia Director Shockwave Cast (Adobe Systems Incorporated)
    CCT                  MicroSim PCBoard Design Information and Rules For Autorouting 
    CCT                  SIL Hebrew Font System Transliteration Table 
    CCT                  Railroad Tycoon 3 Car Type Description (PopTop Software Inc.)
    CCT                  LogicWorks Schematic File (Capilano Computing)
    CCV                  Crystal Vision Report (CTMS)
    CCX                  Total Annihilation: Counter-Strike 
    CCX                  Corel Compressed Exchange File 
    CCX                  CorelDraw File 
    CCX                  Multimedia Fusion Extension (Clickteam)
    CC_                  Audio Utility Midimach Compressed File 
    CD                   CD-i OptImage CD Image 
    CD                   cdTree Content List 
    CD1                  Cafeteria Document System Data File (DATAIR Employee Benefit Systems, Inc.)
    CD150                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CD2                  Click'N Design 3D CD Label (Avery Dennison Corporation)
    CD2                  Cafeteria Document System Custom Text File (DATAIR Employee Benefit Systems, Inc.)
    CDA                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CDA                  Giza Specifier Typical File (20-20 Technologies)
    CDA                  CD Audio Track Shortcut 
    CDAT                 Internet Security and Acceleration (ISA) Server Cache File (Microsoft)
    CDB                  SymbianOS Contact Database File 
    CDB                  PowerDesigner Conceptual Model Backup File 
    CDB                  Giza Specifier Design File (20-20 Technologies)
    CDB                  The Cleaner Trojan Database (Moosoft Development LLC.)
    CDB                  Turbo C Database (Borland Software Corporation)
    CDB                  Pocket Access Database (Microsoft)
    CDB                  CardScan Database (Corex Technologies)
    CDB                  Clipboard File 
    CDB                  One Page from a ROM With a View Project File 
    CDB                  Nokia 9210/9290 Phone Contact Database (Nokia)
    CDB                  SeeYou Waypoint (Naviter)
    CDBK                 SPSS Database File 
    CDC                  Canvastic Documnet/Drawing (Canvastic)
    CDC                  CLAN Debug File (Child Language Data Exchange System)
    CDC                  Chromeleon Driver Configuration Plug-in 
    CDC                  Nero CD Cover File (Nero AG)
    CDD                  Claris Draw v 1.2 Document 
    CDD                  Balder Multiboot Configuration File (Hans Christian Ihlen)
    CDD                  Chromeleon Device Driver 
    CDD                  CD-Architect Driver File (Sonic Foundry)
    CDD                  Concept Draw Drawing/Diagram (Computer Systems Odessa, Corp.)
    CDD                  Cadifra Diagram (Cadifra Inc.)
    CDD                  CADAM Drawing (Dassault Systemes)
    CDDA                 AIFF Audio 
    CDDS                 Midtown Madness 3 Data 
    CDE                  Honeywell Hybrid Control Designer 
    CDE                  CADKEY Dynamic Extension (Kubotek USA Inc.)
    CDEV                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CDEVSAMPLES          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CDF                  Channel Definition Format (Microsoft Corporation)
    CDF                  ANDI/netCDF Mass Spectrometry Data Interchange format 
    CDF                  MicroSim PCBoard Component Description File For Bill Of Materials 
    CDF                  Unidata NetCDF Graphic File (University Corporation for Atmospheric Research)
    CDF                  ASCII Format Describing VRML Worlds 
    CDF                  InstallShield Components Definition File (InstallShield Software Corporation)
    CDF                  Cyberspace Description Format 
    CDF                  Comma Delimited Format 
    CDG                  DART Karaoke Studio CD+G File 
    CDI                  Disc Juggler CD Image File 
    CDI                  CD Anywhere Virtual CD (V Communications, Inc.)
    CDI                  Phillips Compact Disk Interactive Format 
    CDJ                  Disc Juggler File 
    CDK                  Calamus Document (invers Software)
    CDK                  CD Katalog (KRISoftware)
    CDK                  OS/2 Saved SKF Format File (IBM)
    CDKEY                Protocol Used in Communicating from MediaRemote Server to PC MediaRemote Plug-in 
    CDL                  SignLab Vector Graphic 
    CDL                  NetCDF CDL Metadata (Unidata)
    CDL                  CADKEY Advanced Design Language (CADL) (Kubotek USA Inc.)
    CDL                  CaseWare Working Papers Document Link 
    CDM                  Minolta Scanner 3D File (Minolta)
    CDM                  PowerDesigner Conceptual Model File 
    CDM                  Novell NetWare Disk Drivers NPA 
    CDM                  Compressed Music Format 
    CDM                  NTI CD&DVD-Maker Image File (NewTech Infosystems)
    CDM                  Visual dBASE Custom Data Module 
    CDN                  Click'N Design 3D CD Label (Avery Dennison Corporation)
    CDO                  Cdo Navigator 
    CDO                  CADAM Overlay (Dassault Systemes)
    CDP                  Ca Visual Objects Platform for Developer (CAVO) (CA)
    CDP                  CD/Spectrum Pro 
    CDPRJ                CD Menu Creator Project File (Drackon Tech)
    CDQ                  CD Indexer 
    CDR                  Final Cut Pro CD/DVD Image File (Apple Computer, Inc.)
    CDR                  GameJack Virtual CD or DVD Image (S.A.D. GmbH.)
    CDR                  Raw Audio-CD Data 
    CDR                  Sound File 
    CDR                  Elite Plus Commander Saved Game 
    CDR                  Corel Vector Graphic Drawing (Corel Corporation)
    CDRZIP               DICOM Zipped Image Archive 
    CDS                  United States Postal Service Computerized Delivery Sequence 
    CDS                  ChemDraw Stationery Document 
    CDS                  Delphi TClientDataset Data (Borland Software Corporation)
    CDSC                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CDSG                 DrawMusic Chord Song (Conrad Albrecht)
    CDT                  STABCAL (stability calculation for aqueous systems) File 
    CDT                  CorelDraw Template (Corel)
    CDT                  MRS-802 Album Data (Zoom Corporation)
    CDT                  ProWORX Nxt Temporary Long Comment Data (Schneider Electric)
    CDU                  CDU Paint 
    CDV                  CatDV Media Catalog (Square Box Systems Ltd.)
    CDV                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CDW                  CeledyDraw Image 
    CDW                  CADKEY Organized Dialog File (CADKEY)
    CDX                  Active Server Document 
    CDX                  Visual Foxpro Index (Microsoft)
    CDX                  MicroStation Cell Library Index (Bentley Systems, Incorporated)
    CDX                  NovaBACKUP Backup Database Index (NovaStor Corporation)
    CDX                  Corel Draw Compressed Drawing 
    CDX                  Alpha Five Table Index (Alpha Software, Inc.)
    CDX                  ChemDraw Chemical Structure Exchange File (CambridgeSoft Corporation)
    CDX                  Compound Index 
    CDXL                 Old Amiga Movie Format 
    CDXML                ChemDraw XML ChemDraw Format 
    CDZ                  ROM With a View Project File 
    CE                   FarSide Computer Calendar File 
    CE                   Computer Eyes, Digital Vision 
    CE                   CA Clipper Workbench Application 
    CE1                  Computer Eyes Raw Low Resolution Image 
    CE2                  Computer Eyes Raw Medium Resolution Image 
    CE3                  Calendar Creator 3.x 4.x Event List 
    CEB                  Apabi eBook 
    CEF                  CenturionMail Encrypted File (CenturionSoft)
    CEF                  CruzerLock Encrypted File (SanDisk Corporation)
    CEF                  CA Clipper Workbench Application 
    CEF                  Spaix PumpSelector Catalog Exchange Format (VSX - VOGEL SOFTWARE GMBH)
    CEF                  Class Action Gradebook Elementary Edition Data File (CalEd Software)
    CEG                  Tempra Show Bitmap Graphic 
    CEG                  Continuous Edge Graphic Bitmap (Edsun)
    CEI                  CRiSP Harvest File 
    CEL                  Lumena CEL Bitmap (Time Arts Software)
    CEL                  MicroStation Cell Library (Bentley Systems, Incorporated)
    CEL                  Animator Graphic (Autodesk, Inc.)
    CEL                  Audition Loop File (Adobe Systems Inc.)
    CEL                  KiSS Paper Doll File 
    CEL                  Celestia 3D Rendering Script 
    CEL                  CIMFast Event Language File 
    CEL2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CELL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CELLAR               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CELTIC               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CELX                 Celestia 3D Rendering Script 
    CEM                  Computer Graphics Metafile 
    CEM                  Empire Earth Game File 
    CEN                  Master Tracker Financial Analysis 
    CEO                  Extension associated with Winevar Worm 
    CEP                  CostOS Estimating Project Database File (Nomitech Unlimited)
    CER                  Sierra Print Artist Certificate 
    CER                  MSI Cerius II Chemical Format File (Accelrys Software Inc.)
    CER                  Cerius2 (Accelrys Software Inc.)
    CER                  Creatacard Certificate Project (Broderbund)
    CER                  Lahey Fortran Compilation Error Messages 
    CER                  Internet Security Certificate File 
    CES                  Sheffield Dictionary File 
    CEV                  LOUT Character Encoding File 
    CEV                  Software License Tracking Tool 
    CEX                  The Currency Exchanger Rate File 
    CEX                  ThumbsPlus File 
    CEX                  CLAN Output File (Child Language Data Exchange System)
    CEX                  INMOS Transputer Development System Occam User Program 
    CF                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CF                   Sendmail Configuration File 
    CF                   Imake Configuration File 
    CF1                  Common File Format 1 
    CF2                  Common File Format 2 
    CF4                  Catfish File Manager Support File 
    CFB                  Comptons Multimedia File 
    CFB                  Inmos Binary Configuration File 
    CFC                  Cold Fusion File 
    CFD                  CryptoForge Document (Ranquel Technologies)
    CFDG                 Context Free Design Grammar 
    CFE                  CryptoForge Encrypted File (Ranquel Technologies)
    CFF                  Capabilities File for FOUNDATION Fieldbus H1 
    CFF                  BoomTracker 4.0 
    CFF                  Common File Format 
    CFG                  Pro/ENGINEER Configuration (PTC)
    CFG                  Configuration 
    CFG                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CFG                  Proxomitron Configuration File (Scott R. Lemmon (deceased))
    CFH                  Capabilities File for FOUNDATION Fieldbus HSE 
    CFI                  CleverForm Script File (Code Development Ltd.)
    CFIG                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CFK                  Kardplay 1.62 File 
    CFK                  MS Developer Studio Diagram 
    CFL                  Compressed File Library File (Compressed File Container) 
    CFL                  Developer Studio Diagram (Microsoft)
    CFL                  Corel Flowchart File 
    CFM                  Cold Fusion Template File 
    CFM                  Visual dBASE Windows Customer Form 
    CFM                  Corel FontMaster 
    CFML                 Cold Fusion Markup Language 
    CFN                  Calamus Font Data (invers Software)
    CFO                  Turbo C (Borland Software Corporation)
    CFO                  TCU Turbo C Utilities C Form Object 
    CFP                  Complete Fax Portable Fax File 
    CFR                  NRC Inspection Manual CFR Guidance 
    CFR                  IBM Configurator Report 
    CFS                  Signal Data (Cambridge Electronic Design Limited)
    CFS                  Inmos Configuration File 
    CFT                  Flow Chart; Corel Flow Template 
    CFT                  Animation Studio CFast Graphics File (Disney)
    CFT                  Cumberland Family Tree 
    CFW                  ChemFinder Form 
    CFX                  Creative DSP File 
    CFX                  Flow-Cal File 
    CG                   Chord Groover Guitar Song (Guitartab.co.uk)
    CG                   Norton Crashguard File 
    CG3                  Dungeons & Dragons Character File (Wizards of the Coast, Inc.)
    CGA                  Ventura CGA Screen Characters 
    CGA                  CGA Resolution BMP Bitmap 
    CGA                  Color Graphics Adapter Graphics Data 
    CGB                  Gameboy Color 
    CGF                  ProWORX Nxt CODEGEN Batch File (Schneider Electric)
    CGF                  Egrid32 Grid Format File (Sweetheartgames.com)
    CGF                  Asm Objasm 
    CGF                  Family Tree Creator Deluxe (Broderbund)
    CGI                  Common Gateway Interface Script 
    CGL                  CodeGuard Error Log 
    CGM                  Computer Graphics Metafile [Old Style] 
    CGM                  Computer Graphics Metafile [New Style] 
    CGO                  Railroad Tycoon 3 Car Properties (PopTop Software Inc.)
    CGP                  NetCaptor CaptorGroup (Stilesoft, Inc.)
    CGR                  Quest3D Program Part (Act-3D B.V.)
    CH                   OS/2 Configuration File 
    CH                   Clipper Header 
    CH1                  Freelance CHart 
    CH3                  Harvard Graphics Chart (Harvard Graphics)
    CH4                  Charisma 4.0 Presentation 
    CHA                  Character Data 
    CHA                  PageMaker Kerning Data (Adobe)
    CHA                  Challenger Encrypted File (it-com4)
    CHA                  CLAN Frequency Sample File (Child Language Data Exchange System)
    CHA                  mIRC Chat File 
    CHANG                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHANGE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHANGELOG            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHANGER              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHANGES              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHANGES              Squeak Change File 
    CHAR                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHAR1                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHAR2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHAR3                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHAR4                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHARSET              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHART                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHART2               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHAT                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHAT                 PIRCH98 Chat File 
    CHB                  Oledb Checkbook File 
    CHD                  Ares Font Chameleon Font Descriptor 
    CHD                  MAME Compressed Hard Disk File (MAME Team)
    CHD                  ChartFX File 
    CHEAT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHEATS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHECK                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHECKER              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHESSREV             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHESSTITANSSAVE-MS   Chess Titans Saved Game File (Microsoft Corporation)
    CHF                  pcAnywhere Remote Control File 
    CHG                  Quicken On-line Data File (Intuit Inc.)
    CHG                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHGR                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHH                  C++ Header 
    CHI                  ChiWriter Document 
    CHI                  Help File Index (Microsoft)
    CHIME                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHIMES               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHJ                  Help Composer Project 
    CHK                  Sophos Intercheck File 
    CHK                  EpiData Check File (EpiData Association)
    CHK                  SSConvert Multilanguage Video Subtitle Data (Screen Subtitling Systems Ltd.)
    CHK                  3D Movie Maker Interface Data File (Microsoft)
    CHK                  CHKDSK/SCANDISK Output 
    CHK                  Starcraft Campaign Map 
    CHK                  WordPerfect Temporary File (Corel Corporation)
    CHKEXP               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHKJAIL              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHL                  Chromeleon Channel Raw Data 
    CHL                  RealPlayer Live Channel 
    CHL                  Internet Explorer Channel Information (Microsoft)
    CHL                  Configuration History Log 
    CHM                  HTML Help Compiled Help File (Microsoft Corporation)
    CHM                  Ares Font Chameleon Source Font - Font Outline 
    CHM                  ChemDraw Chemical Structure (CambridgeSoft Corporation)
    CHN                  Ethnograph Data File 
    CHO                  Ares Font Chameleon Output Font 
    CHO                  Animation:Master Choreography (Hash)
    CHO                  Chord Pro Song Format (Mussoft)
    CHOICE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHOOCHOO             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHOPIN               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHOPPER              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHOPRO               Chord Pro Song Format (Mussoft)
    CHOR                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHORALE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHORD                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHORDHARM            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHORDPIANO           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHORES               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHORUS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHP                  DVD Chapter File 
    CHP                  BizInt Smart Charts for Patents Document (BizInt Solutions)
    CHP                  CodeHealer Project File (SOCK Software, LLC)
    CHP                  Ventura Publisher Chapter (Corel Corporation)
    CHP                  FLOWer (Pallas Athena)
    CHQ                  Help Combined Full-text Search File (Microsoft)
    CHR                  Character or Font File 
    CHR                  Borland Language Stroke Font File (Borland Software Corporation)
    CHR                  Infinity Game Engine Exported Character Info (BioWare Corp.)
    CHR                  Wormhole 2.0/ Liero Graphics Editor Liero Graphics Mod 
    CHRISTMAS            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHROMAFONT           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHS                  ChessRally Saved Game/E-mail Attached Game (Ingenuware, Ltd.)
    CHS                  ATI Radeon Video Driver 
    CHS                  Corel WP Chart Style 
    CHT                  ChartFX Template File 
    CHT                  Navigator Vector Chart File (Tecepe)
    CHT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHT                  ZSNES Cheat Code (ZSNES Team)
    CHT                  Novagraph Chartist File 
    CHT                  BizInt Smart Charts for Drug Pipelines Document (BizInt Solutions)
    CHT                  ChartViewer dBASE Interface File 
    CHT                  Cheat Machine Data File 
    CHT                  Harvard Graphics Vector File (Harvard Graphics)
    CHT                  Sierra Generations Family Tree Chart (Sierra Entertainment, Inc.)
    CHT                  Chartist from Novagraph 
    CHT                  ICQ Saved Chat Session (ICQ Inc.)
    CHU                  Infinity Game Engine GUI Elements File (BioWare Corp.)
    CHURCH               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CHUZZLEDELUXESAVEDGAME Chuzzle Deluxe Saved Game (PopCap Games, Inc.)
    CHW                  HTML Help General Index Funtionally comparable to .GID. 
    CHZ                  ChArc Compressed File Archive 
    CI                   CitectSCADA File 
    CI2                  Calin Lawyer's Interest Calculator Ver 2 Data File (Lawtech (Aust) Pty Ltd.)
    CI3                  Calin Lawyer's Interest Calculator Data File (Lawtech (Aust) Pty Ltd.)
    CIA                  Sheffield Dictionary File 
    CID                  CallerID File (AnalogX)
    CID                  Navigator Chart Image Description (Tecepe)
    CIF                  pcAnywhere Caller File 
    CIF                  Ventura Chapter Information 
    CIF                  Colombo Ipermedia File 
    CIF                  Chip Layout Information 
    CIF                  Crystallographic Information File 
    CIF                  Ventura Converted Image File 
    CIF                  Ariba Buyer Catalog Information Format (Ariba Inc.)
    CIF                  CalTech Intermediate Graphic 
    CIF                  Creator Document File (MultiAd, Inc.)
    CIF                  Easy CD Creator Image (Sonic Solutions)
    CIG                  Fastback Plus Compressed Backup 
    CIK                  Norton CleanSweep Installation Log (Symantec)
    CIK                  Corel Graphics Ver. 10 Custom Dual Tone File (Corel)
    CIL                  Clip Gallery Download Package (Microsoft Corporation)
    CIL                  Norton CleanSweep Installation Log (Symantec)
    CIM                  The Ultimate Draw C Text Mode Image File 
    CIM                  SimCity 2000 (Electronic Arts, Inc.)
    CIM                  GE Industrial Systems CIMPLICITY HMI Screen 
    CIM                  CIMEX Cimpack Design Drawing File 
    CIM                  CompuApps Drive Backup Image 
    CIN                  Photo-Sonics Picture Sequence 
    CIN                  Cinetitler Digital Screen Subtitling 
    CIN                  Cineon Bitmap (Kodak)
    CIN                  Digital Moving Picture Exchange Bitmap (The Society of Motion Picture and Television Engineers)
    CIN                  Cinemania (Microsoft)
    CIN                  Quake II Cinematic Video (id Software)
    CIN                  OS/2 Change Control File 
    CIN96LM              See .CIN 
    CIN96UPD             See .CIN 
    CIP                  CryptoBuddy Encrypted File (Research Triangle Software)
    CIP                  Ghost Keylogger File 
    CIPO                 Internet Same as .CPI. 
    CIQ                  CableIQ Test Data (Fluke Networks)
    CIR                  PSpice File (OrCAD)
    CIR                  SuperMax E-CAD Save-File Command 
    CIR                  real-PCB Netlist (Crocodile Clips Ltd)
    CIRCLES              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CIS                  TAPCIS Parameter/Address Book/Rates File 
    CIS                  CheckInbox File 
    CIT                  ClockIt Employee Duty Roster (ES-Software.net)
    CIT                  Intergraph Scanned Image 
    CIT                  Norton CleanSweep Installation Log (Symantec)
    CIT                  SeeYou Vector Maps (Naviter)
    CITY                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CIV                  Empire Earth Custom Civilization Information (Sierra Entertainment, Inc.)
    CIX                  TCU Turbo C Utilities Database Index 
    CIX                  Rational XDE (IBM)
    CJB                  cleaner XL Job (discreet (an Autodesk Company))
    CJP                  WinOnCD Project File (Sonic Solutions)
    CK1                  iD/Apogee Commander Keen 1 Data File 
    CK2                  iD/Apogee Commander Keen 2 Data File 
    CK2                  iD/Apogee Commander Keen 1 Data File 
    CK3                  iD/Apogee Commander Keen 1 Data File 
    CK3                  iD/Apogee Commander Keen 3 Data File 
    CK4                  iD/Apogee Commander Keen 1 Data File 
    CK4                  iD/Apogee Commander Keen 4 Data File 
    CK5                  iD/Apogee Commander Keen 5 Data File 
    CK5                  iD/Apogee Commander Keen 1 Data File 
    CK6                  iD/Apogee Commander Keen 6 Data File 
    CK6                  iD/Apogee Commander Keen 1 Data File 
    CKB                  C++ Keyboard Mapping File (Borland Software Corporation)
    CKD                  CADKEY Design File (Kubotek USA Inc.)
    CKI                  Off by One Browser Persistent HTTP Cookies (Off By One Productions, a division of Home Page Software Inc.)
    CKO                  KnockOut 2 Image Mask (Corel)
    CKP                  Ingres Checkpoint File (Computer Associates International, Inc.)
    CKP                  Valise Secure Archive (CertainKey, Inc.)
    CKT                  B2 Spice Curcuit Design (Beige Bag Software, Inc.)
    CKT                  CircuitMaker Schematic (Altium Limited)
    CKT                  CADKEY Template File (CADKEY)
    CKT                  Crocodile Clips 
    CL                   ArtIcons Pro Cursor Library (Aha-soft)
    CL                   Generic LISP Source Code 
    CL3                  Easy CD Creator Layout File (Sonic Solutions)
    CL3                  Bruker Aspect NMR Data File 
    CL4                  Easy CD Creator Ver 4 Layout File (Sonic Solutions)
    CL5                  Easy CD Creator Layout File (Project) (Sonic Solutions)
    CLA                  Java Class File (Sun)
    CLA                  Clarion for DOS Source Code (SoftVelocity)
    CLAP                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CLASS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CLASS                Java Bytecode (Sun Microsystems, Inc.)
    CLASS                Java Class File (Sun Microsystems, Inc.)
    CLB                  cdrLabel Compact Disc Label 
    CLB                  Corel Library 
    CLB                  Super NoteTab ClipBook Template 
    CLB                  Total Club Manager Single Club Info (Electronic Arts Inc.)
    CLB                  ICQ Contact List (ICQ Inc.)
    CLB                  COM+ Catalog File 
    CLB                  Office XP Developer Code Librarian (Microsoft Corporation)
    CLC                  Schedule+ 7-Habits Mission Statement (Microsoft)
    CLC                  UltimaCalc Calculation Data File (UltimaCalc)
    CLC                  Corel Catalog 
    CLD                  CA Clipper Debugger Configuration File 
    CLD                  Terragen Cloud Layer File (Planetside Software)
    CLD                  Skerryvore Software Simply Calenders 
    CLD                  CD-Label Print Label Layout File (Canon, Inc.)
    CLDLYR               Terragen Cloud Layer File (Planetside Software)
    CLE                  Icon Lock-iT Locked File Indicator (Moon Valley Software, Inc.)
    CLEAN                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CLEANER              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CLEAR                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CLEARANCE            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CLF                  ListPro Database (Ilium Software, Inc.)
    CLG                  Disk Catalog Database 
    CLG                  CollageMaker Graphic (Galleria Software)
    CLG                  Maxagrid CTM Log File 
    CLI                  Client Management System Customer File (HawkSoft, Inc.)
    CLICK                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CLIP                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CLIP1                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CLIP2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CLK                  R.A.V.E. Animation File (Corel)
    CLKK                 Clicker Keyboard (Crick Software Ltd.)
    CLKT                 Clicker Template (Crick Software Ltd.)
    CLKW                 Clicker Word Bank (Crick Software Ltd.)
    CLKX                 Clicker Grid Set (Crick Software Ltd.)
    CLL                  OzWin CompuServe E-mail/Forum Access Custom Library Search 
    CLL                  Clicker (Crick Software Ltd.)
    CLM                  Micrografx Picture Publisher 7 Macro 
    CLN                  Backup Configuration File 
    CLO                  SPSS Chart Look (SPSS, Inc.)
    CLO                  Cloe Ray-Tracer 
    CLOCK                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CLOE                 Cloe Ray-Tracer 
    CLOSED               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CLOUD                Homeworld Cloud Data (Sierra Entertainment, Inc.)
    CLOUD                GOM Atos Scanner File 
    CLP                  GoDot 
    CLP                  Windows Clipboard/Picture 
    CLP                  Clipper 5 Compiler Script 
    CLP                  PC Paint/Pictor Picture 
    CLP                  CLIPS/wxCILPS File 
    CLP                  Quattro Pro Clip Art (Corel)
    CLPI                 Blu-ray Clip AV Stream File (Blu-ray Disc Association)
    CLR                  WinEdit Colorization Word List 
    CLR                  FileMaker Spelling Dictionary 
    CLR                  WinMX Color Scheme (Frontcode Technologies)
    CLR                  Boxer/2 Color Scheme 
    CLR                  1st Reader Binary Color Screen Image 
    CLR                  PhotStyler Color Definition 
    CLR                  Flash Color Table File (Macromedia)
    CLR                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CLS                  MS VB-4 Class Module Add-In 
    CLS                  MS VisC++/Visual Basic Class Definition 
    CLS                  JavaScript Class 
    CLS                  C++ Class Definition 
    CLS                  ArcView Geocoding Classification File (ESRI)
    CLS                  Visual Basic Class Module 
    CLT                  VisualBoyAdvance Cheat List (Forgotten and VBA Team)
    CLU                  Clusse Node File (Heikki Hannikainen)
    CLU                  Phantasmagoria 2 8-bit Playback Pallete File 
    CLU                  IBM PCI Cryptographic Coprocessor Code File (IBM)
    CLU                  Broken Sword 2 Dialog File (THQ Inc.)
    CLU                  XNBC V8 Biological Neural Network Simulation Workstation Cluster Names 
    CLUB                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CLUBS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CLV                  Vbox Current Language File (Aladdin Knowledge Systems)
    CLV                  Yamaha Cool Voices (Yamaha Corporation of America)
    CLV                  Clovis for Outlook Schema Settings (Steelhead Systems)
    CLW                  Visual C++ Class Wizard File (Microsoft)
    CLW                  Clarion for Windows Source Code (SoftVelocity)
    CLX                  Acrobat (Adobe)
    CLX                  Rational XDE (IBM)
    CLX                  Macro ToolsWorks Macro 
    CLY                  ACT! Layout (Best Software CRM Division)
    CLY                  Corel Graphics Ver.10 Custom Layouts File (Corel)
    CL_                  C Poet compressed Disk1 File 
    CM                   Craftman Data File 
    CM                   Unix Puzzle 8-bit Bitmap 
    CM1                  CaseMap Version 1.x Case Analysis File (LexisNexis)
    CM2                  Curious Labs Poser Camera File 
    CM2                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CM2                  CaseMap Version 2.x Case Analysis File (LexisNexis)
    CM3                  CaseMap Version 3.x Case Analysis File (LexisNexis)
    CM4                  Championship Manager 4 Saved Game (Sports Interactive Ltd.)
    CM4                  ClipMate Data File 
    CM4                  CaseMap Version 4.x Case Analysis File (LexisNexis)
    CM5                  CaseMap Version 5.x Case Analysis File (LexisNexis)
    CM6                  CaseMap Version 6.x Case Analysis File (LexisNexis)
    CM7                  CaseMap Version 7.x Case Analysis File (LexisNexis)
    CMA                  Filetopia Collection Manager List (Filetopia)
    CMA                  Camomile Library File 
    CMA                  Applix TM1 Database 
    CMAP                 CmapTools Concept Map (IHMC)
    CMB                  Xtree for Windows Button Bar 
    CMB                  Reason Combinator Instrument Patch (Propellerhead Software)
    CMB                  Chromeleon Backup Archive 
    CMC                  Collectorz.com Comic Collector Collection Data (Collectorz.com)
    CMC                  PowerFront File 
    CMD                  Command File for Windows NT (Microsoft)
    CMD                  dBASE II Program File 
    CMD                  OS/2 
    CMD                  OS/2 REXX Batch File 
    CMD                  Summit Contact Manager Contact Database (MIE Software Pty Ltd)
    CMD                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CMD                  DOS CP/M Command File 
    CMD                  1st Reader External Command Menu 
    CMD5                 CrystalMaker Data (CrystalMaker Software Ltd.)
    CMDF                 CrystalMaker Data (CrystalMaker Software Ltd.)
    CMDR                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CME                  Crazy Machines Level File (Pepper Games)
    CMF                  Corel Metafile 
    CMF                  CopyMate Data File (GraphiTech Ltd.)
    CMF                  Connection Manager Fixer File 
    CMF                  CaseMap File Viewer Definition (LexisNexis)
    CMF                  Creative Music File 
    CMG                  Chessmaster Saved Game (Ubisoft Entertainment)
    CMG                  cramfs ROM Filesystem Package 
    CMI                  Camomile Interface File 
    CMI                  CD Mate CD Burning Image File (An Chen Computers Co. Ltd.)
    CMK                  Card Shop Plus File 
    CML                  PADGen Company Info File 
    CML                  Chemical Markup Language Document 
    CML                  ProWORX Nxt Comment Data (Schneider Electric)
    CML                  Crazy Machines Saved Laboratory File (Pepper Games)
    CML                  Filetopia Collection Manager List (Filetopia)
    CML                  Cheat Machine Library File 
    CMM                  CEnvi (ScriptEase:Desktop) Batch File 
    CMMF                 CrystalMaker Data (CrystalMaker Software Ltd.)
    CMMP                 Camtasia Studio Menu Maker Project (TechSmith Corporation)
    CMMTPL               Camtasia Studio Menu Maker Template (TechSmith Corporation)
    CMN                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CMN                  Systems Management Server (SMS) Collection Membership Update (Microsoft Corporation)
    CMO                  Spanish Whiz What Did You Say? Sound File (GB Blanchard)
    CMO                  Virtools Behavioral Server Composition 
    CMP                  Route 66 Address Document 
    CMP                  Leadview (LEAD) Bitmap 
    CMP                  CaseMap Send To TimeMap Plug-in (LexisNexis)
    CMP                  Freelancer Physical Model Layout (Microsoft)
    CMP                  Photofinish Calibration MaP 
    CMP                  Word for DOS User Dictionary (Microsoft)
    CMP                  PKWare Compression Library Compressed Data 
    CMP                  JPEG Bitmap 
    CMP                  Internet Explorer Custom Menu Plug-in (Microsoft)
    CMP                  Generic CADD Component File 
    CMP                  CraftMan Data 
    CMP                  CorelDRAW Postscript Printer Header 
    CMP                  Chord Pro Song Format (Mussoft)
    CMP                  Address Document 
    CMP                  SWAT 3 Compressed Sound File 
    CMP                  HotDocs Template Component File (Matthew Bender & Company, a member of the LexisNexis Group)
    CMP                  Discorp CMP Image 
    CMP                  Open Access File 
    CMP                  Technics Keyboard Composer File 
    CMP                  Giza Specifier Color Map (20-20 Technologies)
    CMPRB                CaseMap ReportBook Plug-in (LexisNexis)
    CMPT                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CMQ                  Culturemetrics 
    CMR                  MediaPlayer Movie 
    CMR                  SeeYou Raster Maps (Naviter)
    CMR                  CribMaster File 
    CMRBK                CaseMap ReportBook Definition (LexisNexis)
    CMRT                 CraftMan Data 
    CMS                  TrialDirector Compound Media Storage 
    CMS                  CaseMap Case Script (LexisNexis)
    CMT                  Culturemetrics 
    CMT                  Corel Draw 
    CMT                  Chinon ES-1000 Digital Camera 
    CMT                  e-Sword Commentary (Rick Meyers)
    CMTD                 Cool Moon To Do To Do List (Cool Moon Software, LLC.)
    CMU                  Carnegie Mellon Univeristy Stereo Raster Image 
    CMU                  CMU Window Manager Bitmap 
    CMUWM                Carnegie Mellon University (CMU) Window Manager Bitmap 
    CMV                  CorelDRAW 4.0 Movie Animation (Corel Corporation)
    CMW                  COM-Watch 
    CMW                  Custom Maintenance Wizard File (Microsoft)
    CMX                  Rational XDE (IBM)
    CMX                  Presentation Exchange Image (Corel)
    CMX                  Patch File 
    CMX                  The Sims 3D Body Mesh Data (Electronic Arts (Maxis))
    CMX                  Apple Viewer (Apple Computer, Inc.)
    CMXA                 Camomile Packed Library File 
    CMY                  Corel40 Custom File 
    CMYK                 Raw Cyan, Magenta, Yellow, and Black Samples 
    CNB                  Pegasus Mail Mail Message in Noticeboard (David Harris)
    CNC                  CNC General Program Data 
    CNC                  Click 'n Create File (Clickteam)
    CND                  Embroidery Design File 
    CND                  ControlDraw File 
    CNF                  Conference Link 
    CNF                  Configuration File Used by Telnet, Windows, and other applications with varying internal formats. 
    CNF                  1-2-3 Configuration (IBM)
    CNF                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CNF                  Inor MinIpaq Software File 
    CNF                  Windows SpeedDial (Microsoft)
    CNF                  KiSS Paper Doll Set 
    CNM                  NoteMap Outline File (LexisNexis)
    CNM                  WinRoute Electronic Mail File (Kerio Technologies Inc.)
    CNM                  Pegasus Mail New Mail Message (David Harris)
    CNM                  Windows Application Menu Options and Setup File 
    CNQ                  WebShop Platinum Project File (Boomerang Software, Inc.)
    CNQ                  Compuworks Design Shop File 
    CNR                  Pegasus Mail Mail Message in Systemwide Folder (David Harris)
    CNR                  CRiSP Harvest File 
    CNREG                EULANDA Warenwirtschaft (ERP) Configuration Information 
    CNREP                EULANDA ERP System Report Template 
    CNS                  Windows2000 Client Connection Manager Export File (Microsoft)
    CNT                  Help File Contents 
    CNT                  Rational Rose 98 Contents Tab File (IBM)
    CNV                  DB2 Conversion File 
    CNV                  Canvas Versions 6, 7, 8, 9 Graphic File (ACD Systems Ltd.)
    CNV                  Word Data Conversion File (Microsoft Corporation)
    CNV                  WordPerfect Temporary File (Corel Corporation)
    CNV                  WS_FTP Pro Upload Conversion File Data 
    CNX                  Rational XDE (IBM)
    CN_                  Regeditx File 
    CO                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CO                   Cult3D ActiveX Player Application 
    CO$                  MS Compressed COM Decompress with UNPACK.EXE. 
    COA                  Medlin Accounting Chart of Accounts With Beginning of Year Balances (Medlin Accounting)
    COA                  TrueSpace 3D Object (Caligari Corporation)
    COB                  Total Annihilation Unit Script (Cavedog Entertainment)
    COB                  RM/COBOL Runtime System Application File (Liant)
    COB                  TrueSpace2 Object 
    COB                  COBOL Program File 
    COB                  Creatures Compiled Object File (Gameware Development Ltd.)
    COB                  TrueSpace Object (Caligari Corporation)
    COBRA                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COCO                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COD                  Multiplan Data File (Microsoft)
    COD                  dBASE Application Generator Template Source File 
    COD                  Charset Code File 
    COD                  Code List 
    COD                  Boxer/2 printer Code Definition File 
    COD                  Blackberry Compiled Java Code (Research In Motion Limited)
    COD                  dBASE Template Source File 
    COD                  Forecast Plus 
    COD                  FORTRAN Compiled Code 
    COD                  C Compiler Output (Microsoft)
    COD                  Compiler Program Code 
    COD                  UUPC Code Definition Table 
    COD                  StatPac Gold Datafile 
    COD                  SWAT Input Control File 
    COD                  Video Text File 
    COD                  Agent Charmap (Forté)
    COD                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CODE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CODE                 Graal and Related Servers Encrypted Cache (Cyberjoueurs)
    CODE0                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CODE1                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CODE2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CODE3                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CODE4                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CODEC                Rockbox CODEC (The Rockbox Crew)
    CODES                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COF                  Days Of Death Program/Media Saved Information (ColemanSystems)
    COK                  Carmageddon Patch File 
    COKE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COL                  MPEG Audio Collection Collection (The MAC Team)
    COL                  Grand Theft Auto III Collision Data (Take-Two Interactive Software, Inc.)
    COL                  HTML Help Collection File (Microsoft)
    COL                  Internet Color Wizard Color Scheme 
    COL                  Notes Import Column Description File (IBM)
    COL                  Movie Maker Collection File (Microsoft)
    COL                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COL                  Autodesk Color Palette (Autodesk, Inc.)
    COL                  Multiplan Spreadsheet (Microsoft)
    COLA                 Cola Source Code (The Perl Foundation)
    COLLAPSE             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COLLECTION           Shareaza Collection (Shareaza Pty. Ltd.)
    COLOGNELA            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COLOR                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COLOR                Amaya Configuration 
    COLOR2               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COLORS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COM                  Common Object Module 
    COM                  DOS or CP/M Executable 
    COM                  Command 
    COM                  AOL (America Online, Inc.)
    COM                  EICAR Anti-Virus Test File (EICAR)
    COM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COM                  Psion Organiser CommsLink Setup File (Psion PLC)
    COM2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COMBAT               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COMFYCAKESSAVE-MS    Purble Place Saved Game (Microsoft Corporation)
    COMICSCARTOON        Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COMING               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COMMANDLINE          Commandline Powertoy File 
    COMMANDS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COMMANDS2            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COMMENTS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COMMON               MATLAB Install File (The MathWorks, Inc.)
    COMMS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COMPACT              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COMPANY              Chris Sawyers Locomotion Company Owner Face (Atari Interactive, Inc.)
    COMPARE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COMPAT               Likely a Text File 
    COMPAT               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COMPILERS            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COMPLICATIONS        Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COMPOSER             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COMPPIANO            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COMPRESSOR           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CON                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CON                  WordPerfect Document Concordance (Corel Corporation)
    CON                  Tree Professional Conifer Creator File 
    CON                  Train Simulator Train/Model Identification File (Microsoft Corporation)
    CON                  Simdir Configuration File 
    CON                  Knowledgeware Consolidation File 
    CON                  LucasArts Container File (LucasArts Entertainment Company LLC)
    CON                  Battlefield 1942 Configuration File (Electronic Arts, Inc.)
    CON                  Unreal 2 Audio File (Epic Games, Inc.)
    CON                  Trace Data File (GraphiTech Ltd.)
    CONCERTO             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONF                 IRC Related Configuration File 
    CONF                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONF                 Palace Server Server Configuration (Communities.com)
    CONF                 Linux Configuration File 
    CONFERENCE           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONFIDENT2           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONFIDENTIAL         Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONFIG               NET Application Configuration Information (Microsoft Corporation)
    CONFIG               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONFIG               CRiSP Harvest File 
    CONFIG               Rockbox Configuration File (The Rockbox Crew)
    CONFIG               WatchDirectory Task Information (GdP Software)
    CONFIGDBLINK         Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONFIGS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONFIGURE            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONNECTION           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONSEQUEN            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONSOLE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONSOLE              WinNT Console File 
    CONT                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONTENTS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONTENTSONCLIPBOARD  Contents on Clipboard Powertoy File 
    CONTROL              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONTROLPANEL         Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONV                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONV                 Stylus Studio Data Conversion Template (DataDirect Technologies, a division of Progress Software Corporation)
    CONVERSIONS          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONVERT              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONVERT2             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CONVERTER            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COO                  Fabasoft Components 
    COO                  Cookie File 
    COOKIES              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COOL                 QB - Text Analyzer Archive (Sierra Vista Software)
    COP                  Tree Professional Conifer Creator Image File 
    COPIER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COPIEREC             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COPPER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COPY                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COPYING              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COPYR                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COR                  WMOVIEC Input File 
    COR                  Web GPS Correction Server Export File 
    COR                  posem Correction File (Nimal Solutions Ltd.)
    COR                  Insight II Output Coordinate File (Accelrys Software Inc.)
    COR                  Protein Structure File 
    COR                  Cokriging Software COREG Output File 
    CORE                 Core Software Tech CORE IDC Bitmap 
    CORELPROJECT         Photo Album Project (Corel Corporation)
    CORELSHOW            MediaOne Enhanced Slideshow File (Corel Corporation)
    CORRECT              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CORRECTION           As-U-Type Correction FIle (Fanix Software)
    CORTOUCHKA           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COS                  Creatures Object Source (Gameware Development Ltd.)
    COS                  Scarlett Expert System Compiler Query File 
    COS3                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COSMO                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COSMO2               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COSMO3               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COSMO4               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COSMO5               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COSTS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    COUNT                Gen Counters File 
    COUNTER              Netobjects Fusion Components Bbscomp Message File 
    COV                  Fax Cover Page (Microsoft)
    COX                  Multimedia Fusion Plug-In (Clickteam)
    CP                   CompuMedic Database 
    CP                   Sound Shared File 
    CP1                  Compel Show Ver 1 Presentation (Click2learn, Inc.)
    CP2                  Compel Show Ver 2 Presentation (Click2learn, Inc.)
    CP8                  CP8 256 Gray Scale Image 
    CPA                  Colasoft Capsa 
    CPA                  Compass program of the Takahashi 
    CPA                  Premiere Conformed Audio File (Adobe Systems Inc.)
    CPA                  Cimrex Prog (Beijer Electronics)
    CPAN                 Tucows Info File (Tucows Inc.)
    CPAO                 3dize Cool Page Objects Counter File 
    CPC                  CPC Lite Cartesian Perceptual Compressed File 
    CPD                  Fax Coversheet 
    CPD                  Complaints Desk Script 
    CPD                  Corel PrintOffice File 
    CPE                  Fax Cover Sheet (Microsoft)
    CPF                  ProWORX Nxt Comment Pointer And Length File (Schneider Electric)
    CPF                  Complete PC Fax File 
    CPF                  CaseMap Portable Format (LexisNexis)
    CPF                  Canvastic Preferences (Canvastic)
    CPF                  O&O CleverCache Profile (O&O Software GmbH)
    CPF                  CPF Editor Contract Preparation Form Submission File (CORDIS)
    CPF                  CircuitCAM Manufacturing Process Definition File (Aegis Industrial Software Corporation)
    CPG                  3dize Cool Page File Tutorial 
    CPH                  Corel PrintHouse Image 
    CPI                  Calamus Graphic (invers Software)
    CPI                  Sietronics CPI XRD Document (Sietronics Pty Ltd)
    CPI                  Cartesian Products Compressed Internet Document 
    CPI                  Windows International Code Page (Microsoft Corporate)
    CPI                  ColorLab Processed Image 
    CPICS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CPIO                 UNIX CPIO Archive 
    CPJ                  Commotion File 
    CPJ                  CeQuadrant CD Project 
    CPK                  Sega FILM/CPK File Format (Sega Corporation)
    CPL                  Windows Control Panel Extension (Microsoft)
    CPL                  Corel Color Palette (Corel)
    CPL                  Compel Show Ver 1 Presentation (Click2learn, Inc.)
    CPM                  Cisco Secure Policy Manager File 
    CPM                  ConceptHDL Project File 
    CPM                  Turbo Pascal DOS File (Borland Software Corporation)
    CPM                  InterComm Project Manager File 
    CPM                  CP/M Executable 
    CPN                  Rise of Nations Campaign File (Microsoft)
    CPN                  Age of Empires I and II Campaign File (Microsoft Corporation)
    CPN                  CPN Tools CPN Model File (CPN Group at the University of Aarhus)
    CPO                  Corel Print House File 
    CPP                  Xcode Core C Program (Apple Computer, Inc.)
    CPP                  CA-Cricket Presents Presentation 
    CPP                  Maya OpenGL 3D Scene (Alias Systems, a division of Silicon Graphics Limited)
    CPP                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CPP                  C++ Builder 6 (Borland Software Corporation)
    CPP                  Visual C++ Source Code File (Microsoft)
    CPQ                  Exam Essentials Exam File (Cyber Pass Inc.)
    CPR                  PV-WAVE Compiled Procedure File 
    CPR                  Cubase Project File (Steinberg)
    CPR                  INMOS Transputer Development System Occam Program Code Fold 
    CPR                  Corel Presents Presentation 
    CPR                  CPC Plus Cartridge Image 
    CPR                  ConTEXT Project File (Eden Kirin)
    CPR                  Knowledge Access Graphics 
    CPR                  Patrician II 
    CPR                  Port Royale 
    CPS                  Photo House Image (Corel Corporation)
    CPS                  Central Point PC Tools Backup 
    CPS                  Colored PostScript File 
    CPS                  Anti-Virus Checksum 
    CPS                  QEMM Backup of Startup Files 
    CPS                  Classroom Performance System File 
    CPT                  After Shot (Corel Corporation)
    CPT                  Compact Pro Archive (Cyclos)
    CPT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CPT                  CCRYPT Encrypted File 
    CPT                  dBASE Encrypted Memo 
    CPT                  CA Cricket Presents Template 
    CPT                  Corel Photo-Paint Image (Corel)
    CPT6                 Corel Photo-Paint Image 
    CPU                  Sysoft Sandra File 
    CPX                  Code Page Translation File (Microsoft)
    CPX                  Rise of Nations Campaign File (Microsoft)
    CPX                  Age of Empires I and II Campaign File (Microsoft Corporation)
    CPX                  CryptaPix Encrypted Image 
    CPX                  Control Panel Applet 
    CPX                  Corel Presentation Exchange Compressed Drawing 
    CPY                  Atari Ramdrive CPY Commands 
    CPY                  eCopy Scanned Document (eCopy, Inc.)
    CPY                  Flexus COBOL Definitions 
    CPY                  RM/COBOL Development System Include File (Liant)
    CPY                  Copy Books Data File 
    CPY                  Cobol Include 
    CPZ                  CP Shrink Archive (Symantec)
    CPZ                  COMPOZ Music Text 
    CQ                   CP/M SYSGEN Replacement 
    CQL                  XDCC Catcher Saved Queue List (SiHiDa and balduz (Developers))
    CQQ                  Cool Quiz 
    CQS                  CADfix Quality Standard (ITI TranscenData)
    CQZ                  Cool Quiz 
    CR                   Crack File 
    CR                   Iris Printer CT Format Bitmap 
    CR                   CRiSP Source Code 
    CR$                  BASIC VB Kartei File 
    CR2                  Canon Digital Camera Raw Image Format (Canon Inc.)
    CR2                  Curious Labs Poser Character File 
    CR4                  STABCAL (stability calculation for aqueous systems) File 
    CR5                  Tripod Data Systmes Survey Pro Coordinate File 
    CR7                  STABCAL (stability calculation for aqueous systems) File 
    CRA                  Personal Craft Factory File 
    CRA                  Advanced Crack File (usually text) 
    CRACK                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CRACKING             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CRACKPOTY            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CRACKZBYMAB          YaHoE File 
    CRAM                 Compressed ROM File System File 
    CRASH                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CRASH2               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CRATE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CRAW                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CRAZE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CRAZY                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CRC                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CRC                   JPEG Sheriff CRC Info File (engineering bureau Office Automation)
    CRC                  Total Commander CRC File (Christian Ghisler, C. Ghisler & Co.)
    CRC                  Pro/ENGINEER Temporary File (PTC)
    CRC16                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CRD                  Spanish Whiz Card Game Sound File (GB Blanchard)
    CRD                  ColdRED Script File 
    CRD                  Windows Cardfile 
    CRD                  Guitar Chord File 
    CRDEL                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CRE                  LiveUpdate Crescendo Catapult File 
    CRE                  Infinity Game Engine Monster Description (BioWare Corp.)
    CREATOR              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CREDITS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CREG                 CrazayReg Database (Crazy-Team)
    CRF                  Database Cross-Reference File 
    CRF                  CHARTrunner Chart Definition (PQ Systems)
    CRF                  Thief: The Dark Project Archive 
    CRF                  Thief 2: The Metal Age Archive 
    CRF                  System Shock 2 Archive 
    CRF                  Sierra Print Artist Craft 
    CRF                  Programming Cross-reference 
    CRF                  OM2Cipher Recipient File 
    CRF                  Calcomp Raster Bitmap 
    CRF                  MASM Cross-Reference File 
    CRF                  Zortech C++ Cross-Reference File 
    CRG                  Calamus Graphic (invers Software)
    CRH                  Links Games Course File 
    CRH                  Golf Image File (Microsoft)
    CRK                  Crack File 
    CRL                  Certificate Revocation List 
    CRLD                 Clicker Crick Resource Library Data File (Crick Software Ltd.)
    CRM                  CHARTrunner Multi-Chart Definition (PQ Systems)
    CRM                  Netmino File 
    CRM                  Capital Research Vendor Bid System 
    CRO                  Webalizer.01 Lang File 
    CRONTAB              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CROPS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CROSS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CROWN                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CRP                  Visual dBASE Custom Report 
    CRP                  Corel Presents Run-Time Presentation 
    CRP                  dBASE Encrypted Database 
    CRS                  Dance With Intensity 
    CRS                  Gold Game Course File 
    CRS                  Links Games Course File 
    CRS                  WordPerfect 5.1 for Windows File Conversion Resource (Corel Corporation)
    CRT                  Certificate File 
    CRT                  Oracle Terminal Settings Info 
    CRT                  UNIX Crontab File 
    CRT                  Soft Concert Karaoke File 
    CRU                  CRUSH Compressed Archive 
    CRUNCH               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CRUNCHER             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CRV                  Corel Graphics Ver. 10 Custom Gradients File (Corel)
    CRW                  CHARTrunner Workspace Definition (PQ Systems)
    CRW                  Cadrail Train Terrain Layout (Sandia Software)
    CRW                  Canon Digital Camera Raw Image Format (Canon, Inc.)
    CRW                  Crossword Genius 
    CRX                  Links Games Course File 
    CRY                  Crypto Chat II Encrypted Notebook-content 
    CRYPTOMITE           CryptoMite Encrypted Archive (BAxBEx Software)
    CRYSTALHAMMERREMIX   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CRZ                  Links Games Course File 
    CRZ                  FScruiser Cruise Record (USDA Forest Service)
    CS                   Torque Game Engine Script (GarageGames)
    CS                   Fractal Lab Data/Parameter File (Mystic Fractal)
    CS                   Visual C#.NET Source Code (Microsoft)
    CS                   Tribes 2 Scripting File (Sierra)
    CS                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CS-CNF               Currency Server Software Configuration (Cloanto Corporation)
    CS1                  CaptureShop 1-Shot Image (Sinar)
    CS3                  Casio WK-1800 Keyboard Output File 
    CS4                  CaptureShop 4-Shot Image (Sinar)
    CSA                  Ultimate Ride Roller Coaster 
    CSA                  InterComm ASCII Sheet File 
    CSA                  Comma Deliminated Text 
    CSB                  CircuitSoft Viewer 
    CSB                  Corel Photo-Paint Script (Corel)
    CSB                  Pci Ncr3030 Ncrflash File 
    CSC                  CrystalControl LCD Control Software Screen (Crystalfontz)
    CSC                  Corel Script (Corel)
    CSC                  Wizard 
    CSD                  Embroidery Design File 
    CSD                  Bitstream Fontware 
    CSD                  Csound Unified File Format for Orchestras and Scores (Dr. Richard Boulanger)
    CSD                  Chemsoft MSDS Database File 
    CSD                  EasyLink Measurement Database (Damalini AB)
    CSE                  GPS Pathfinder Office Coordinate System (Trinble Navigation Limited)
    CSE                  Cardinal Studio Template File 
    CSF                  [Unknown] 
    CSF                  Brava Reader Content Sealed Format (Informative Graphics Corp)
    CSF                  Van Dyke's CRT/SecureCRT Script File 
    CSF                  Uefa Champions League Logis Cutscene File 
    CSF                  Photoshop Color Settings (Adobe Systems Inc.)
    CSF                  GeoMedia Coordinate System File 
    CSFDB                Open CASCADE 3D Model File (Open CASCADE Company)
    CSG                  Statistica Graph File 
    CSH                  CARA-Script-Interpreter-Engine Script 
    CSH                  Hamilton Labs C Shell Script File 
    CSH                  UNIX csh Shell Script 
    CSH                  csh Script 
    CSH                  Photoshop Custom Shape (Adobe)
    CSH                  Cubase Offline Processing Information (Steinberg Media Technologies GmbH.)
    CSHRC                UNIX C Shell Environment File 
    CSI                  Cyberautograph Signed Item 
    CSK                  Claris Works 
    CSK                  StoryWeaver Template (Storymind Store)
    CSK                  Copernic 2001 Skin (Copernic)
    CSL                  AOL Modem Script (America Online, Inc.)
    CSL                  Designer Symbol Library (Corel)
    CSL                  TestQuest Pro Test Script (TestQuest, Inc.)
    CSLM                 Zeiss CSLM Bitmap 
    CSM                  C++ Symbol File (Borland Software Corporation)
    CSM                  C++ Ver. 4.5 Precompiled Header (Borland Software Corporation)
    CSM                  Kodak DC265 Camera Script 
    CSM                  Chemical Style Markup Language 
    CSM                  COOLSoft Model Database (Wright Williams & Kelly, Inc.)
    CSM                  CASL Compiled PalmPilot Program (Feras Information Technologies)
    CSML                 Chemical Style Markup Language 
    CSN                  CopyMate Raw Points Cloud (GraphiTech Ltd.)
    CSN                  Dreamweaver MX's Snippet 
    CSO                  CISO Compressed ISO CD Image 
    CSP                  PC Emcee On-Screen Image 
    CSP                  Draw Service Bureau Profiler (Corel)
    CSP                  Crystal Enterprise Crystal Server Page 
    CSP                  CaseWare Working Papers Script Page 
    CSP                  Cardinal Studio Project File 
    CSP                  Intersystems Caché Server Page File 
    CSP                  CASLwin Program (Feras Information Technologies)
    CSP                  ChaoSynth Parameter File (Nyrsound)
    CSP                   AudioZip Encoded Audio 
    CSPROJ               Visual Studio .NET C# Project (Microsoft)
    CSQ                  Foxpro Query (Microsoft)
    CSS                  Citadel SafStor User Script File 
    CSS                  InterComm Part Body Definition 
    CSS                  Hypertext Cascading Style Sheet 
    CSS                  Statistica Datasheet 
    CSS                  Stats+ Datafile 
    CSS1                 Cascading Style Sheet See .CSS. 
    CST                  Macromedia Director Cast File 
    CST                  CHARTrunner Chart Style Definition (PQ Systems)
    CST                  Medbasin Model Calibration File (National Technical University of Athens, Laboratory of Reclamation Works & Water Resources Management)
    CST                  Technics Sx KN 6000 Keyboard Custom Style File 
    CSV                  Comma-Separated Variables 
    CSV                  CompuShow Adjusted EGA/VGA Palette 
    CSV                  Boso View Express Railway and Structure File (Mackoy)
    CSVENX               egis Encrypted CSV File (HiTRUST Inc.)
    CSW                  Compressed Square Wave 
    CSW                  WordPerfect Setup Info File (Corel Corporation)
    CSY                  EPOC Polymorphic DLL 
    CS_                  Creativ Compressed Sb16 Sbide File 
    CT                   Continuout Tone TIFF Bitmap 
    CT                   Iris CT Graphic 
    CT                   Scitex CT Handshake Continuous-Tone Bitmap 
    CT                   Cheat Engine Cheat Table (Dream.media)
    CT0                  Maxagrid CTM File Being Processed 
    CT0                  Systems Management Server (SMS) Master Site Control File (Microsoft Corporation)
    CT1                  Systems Management Server (SMS) Proposed Site Control File (Microsoft Corporation)
    CT1                  Maxagrid CTM File Successfully Processed 
    CT2                  CTRAN/W Graphical Layout File (Geo-Slope International)
    CT2                  Maxagrid CTM File Unsuccessfully Processed 
    CT3                  CaseMap Version 3.x Case Analysis Template File (LexisNexis)
    CT3                  Systems Management Server (SMS) Proposed Site Control File (Microsoft Corporation)
    CT3                  CTRAN/W Sketch Lines and Text File (Geo-Slope International)
    CT4                  CaseMap Version 4.x Case Analysis Template File (LexisNexis)
    CT5                  CaseMap Version 5.x Case Analysis Template File (LexisNexis)
    CT6                  CaseMap Version 6.x Case Analysis Template File (LexisNexis)
    CT7                  CaseMap Version 7.x Case Analysis Template File (LexisNexis)
    CTAB                 CACTVS Chemical Table Format 
    CTB                  AutoCAD Color-dependent Plot Style Table (Autodesk, Inc.)
    CTB                  NoteTab Clipbar 
    CTC                  SeeYou Flight Data (Naviter)
    CTC                  PC Installer Control 
    CTD                  Simpsons Cartoon Studio Export File 
    CTD                  Cardtable File 
    CTD                  Cobra Track Dump 
    CTD                  Marine Data File 
    CTE                  Classical Text Editor Document (Stefan Hagel, Vienna)
    CTF                  CoffeeCup HTML Editor Table File (CoffeeCup Software, Inc.)
    CTF                  TIFF Compressed File 
    CTF                  AVG Update Control File (GRISOFT, s.r.o.)
    CTF                  Symphony Character Code Translation (IBM)
    CTF                  WhereIsIt Catalog 
    CTF                  PhotoSuite Album File (Sonic Solutions)
    CTF                  HP-95LX Character Translation File for Datacomm 
    CTF                  Calculator Text Format 
    CTF                  BrightStor ARCserve Backup Backup File (Computer Associates International, Inc.)
    CTG                  BMC Software Patrol File 
    CTG                  Powershot Pro 70 Info File (Canon, Inc.)
    CTG                  Cartridge Definition File 
    CTG                  ChessBase Opening Book 
    CTI                  Collectorz.com Template Info File (Bitz & Pixelz)
    CTK                  mSuite License File (CommonTime Limited)
    CTL                  Setup Information 
    CTL                  User Control 
    CTL                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CTM                  SDR99 Speech Recognition Task Time Marked Conversation Input 
    CTM                  Maxagrid CTM File 
    CTM                  Java Charge Transaction Matrix File 
    CTM                  Crazytalk Talking Animation File 
    CTM                  Content Delivery Service 
    CTM                  CGMetafile 
    CTM                  FreeBSD Content Management File 
    CTN                  CADTERNS Sloper File (CADTERNS Custom Clothing Inc.)
    CTP                  American Greetings CreataCard (Broderbund)
    CTP                  BestAddress HTML Editor Combined Template File (Multimedia Australia Pty. Ltd.)
    CTP                  ChemDraw Template 
    CTR                  CATALYST Text Parsing Rules File (Alchemy Software Development Ltd.)
    CTR                  ERGO Receiver Configuration File (Creative Express Corporation)
    CTR                  Corel40 Trace File 
    CTR                  CTRAN/W DEFINE Data File (Geo-Slope International)
    CTR                  Counter File 
    CTR                  The Cookie Counter! Hit Count File (CLIQ Services Coop)
    CTRL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CTS                  ABC Programming Language Permanent Location Contents 
    CTS                  ClipTrakker Clipboard Collection (SoftwareOnline)
    CTT                  MSN Messenger Saved Contact List (MSN)
    CTT                  CAIL Suite Terminal Emulation Data (CAIL Systems Inc.)
    CTT                  LabView File 
    CTV                  Citavi Project (Academic Software Zurich GmbH)
    CTW                  Context Tree Weighting Compressed File (Eindhoven University of Technology)
    CTX                  Alphacam Compiled Text (Planit)
    CTX                  Gasteiger Group CTX Chemical File 
    CTX                  Visual Basic User Control Binary File 
    CTX                  Nokia PC Suite Backup Contact File (Nokia)
    CTX                  Windows Terminal Server INI Backup File (Microsoft)
    CTX                  Pretty Good Privacy (PGP) Ciphertext File (PGP Corporation)
    CTX                  CTRAN/W DEFINE Compressed Data File (Geo-Slope International)
    CTX                  Chinese Character Input File 
    CTX                  Compressed Text 
    CTX                  Online Course Text (Microsoft)
    CTX                  GE Industrial Systems CIMPLICITY Text Version HMI Screen 
    CTY                  Fog Creek CityDesk Template 
    CTY                  SimCity City File (Electronic Arts, Inc.)
    CTY                  Railroad Tycoon 3 Cargo (PopTop Software Inc.)
    CU1                  Photoline4 Curves Curve File 
    CU3                  Photoline4 Curves Clut File 
    CUB                  PowerPlay File (Cognos Incorporated)
    CUB                  Cubloc Studio PLC Information File (Comfile Technology Inc.)
    CUB                  OLAP Cube File (Microsoft)
    CUB                  MDL Molfile Gaussian Cube (Wavefunction) Format (MDL Information Systems, Inc.)
    CUB                  SeeYou Airspace (Naviter)
    CUBE                 Cubicomp PictureMaker Bitmap 
    CUBE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CUBE                 Gaussian Cube File 
    CUBI                 Cubicomp PictureMaker Bitmap 
    CUC                  SeeYou Contest File (Naviter)
    CUE                  BPM Studio Cue Points (ALCATech)
    CUE                  CUEcards 2000 Database (Marcus Humann)
    CUE                  CDRWin and Others Description File for a CD-Image 
    CUE                  Cue Cards Data (Microsoft)
    CUF                  Turbo C Utilities Form Definition (Borland Software Corporation)
    CUL                  IconForge/ImageForge Cursor Library 
    CUL                  Cubloc Studio PLC Information File (Comfile Technology Inc.)
    CUP                  OzWin CompuServe E-mail/Forum Access Catalog Library Update 
    CUP                  Roland Garros Tennis Terrain File 
    CUP                  SeeYou Waypoint (Naviter)
    CUR                  Windows Cursor (Microsoft Corporation)
    CUREXSCHEME          CursorXP Theme (Stardock Corporation, Inc.)
    CURL                 URL Links Curl 
    CURSOR               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CURSOR               Sun Icon/Cursor 
    CURVE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CURXPTHEME           CursorXP Theme (Stardock Corporation, Inc.)
    CUS                  AutoCAD Custom Dictionary (Autodesk, Inc.)
    CUS                  Medlin Accounting Customer Data (Medlin Accounting)
    CUS                  ConciseURI Syntax (Clear Methods, Inc.)
    CUSTOM               Personal Paint Custom Language User Interface File (Amiga)
    CUT                  CLAN Disambiguation Rules (Child Language Data Exchange System)
    CUT                  Utilitaires 
    CUT                  INMOS Transputer Development System Occam Utility Package 
    CUT                  Dr. Halo Bitmap Graphic 
    CUT                  DataWave Physiological Data File 
    CUT                  BPM Studio Cue Points and Marker Information (ALCATech)
    CUTLIST              Asfbin Assistant Cut Data 
    CUZ                  CropWalker 
    CU_                  Winhelp Compressed File 
    CV                   Versions Archive (Corel)
    CV                   CodeView Information (Microsoft)
    CV3                  Civilization III Game High Score 
    CV4                  CodeView Colors (Microsoft)
    CV5                  CCS-lib.2 JPEG Stream File 
    CV5                  Canvas Version 5 
    CVA                  Compaq Diagnostics (Hewlett-Packard Development Company, L.P.)
    CVB                  Borland BDE File (Borland Software Corporation)
    CVD                  Clam AntiVirus Database (Clam AntiVirus Team)
    CVD                  Calamus Vector Graphic (invers Software)
    CVD                  Yamaha Custom Drums (Yamaha Corporation of America)
    CVD                  CVTracker Dictionary File (Netmap Software Inc.)
    CVF                  SuperStor Compressed Volume 
    CVF                  Train Simulator Cab View File (Microsoft Corporation)
    CVG                  Calamus Vector Graphic (invers Software)
    CVG                  CVTracker Personal Goal File (Netmap Software Inc.)
    CVH                  Sound File 
    CVH                  CVTracker Proprietary Help File (Netmap Software Inc.)
    CVIP                 CVIPlab File 
    CVJ                  Cabinet Vision Solid Job Document (Planit Solutions, Inc.)
    CVL                  Coastal Vector List Text Format 
    CVL                  VLBI and Tied Array Cover Letter Tasking Document 
    CVN                  Yamaha Custom Voices (Yamaha Corporation of America)
    CVP                  WinFax Cover Page (Symantec)
    CVP                  Portrait 
    CVP                  Kodak File 
    CVQ                  Coastal Vector List Compressed Format 
    CVR                  WinFax Cover Page (Symantec)
    CVR                  ElectraSoft Fax Cover Sheet 
    CVR                  Bitware Fax File 
    CVS                  Outlook Express File 
    CVS                  Sound File 
    CVS                  Canvas Drawing File 
    CVSAUTO              Call Management System (CMS) Automatic Script (Avaya Inc.)
    CVSIGNORE            Snort File 
    CVT                  CVTracker Position and Company Template File (Netmap Software Inc.)
    CVT                  Convert Data File (Joshua F. Madison)
    CVT                  Follett Conversion Tool 
    CVT                  dBASE Converted Database Backup 
    CVW                  Giza Specifier Carera View (20-20 Technologies)
    CVW                  CodeView Colors (Microsoft)
    CVX                  Canvas Versions 6, 7, 8, 9 Graphic File (ACD Systems Ltd.)
    CW                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CW2                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CW3                  Crossword Construction Kit Puzzle Group 
    CW3                  Claris Works Win 3 Document 
    CW3                  Chem Draw Molecular Structure 
    CW6                  COMPRESS Windows Version Pressure Vessel Data File (Codeware Inc.)
    CWA                  Castelware Inventory Hardware/Software Audit File (Castelware GmbH)
    CWA                  See .CLC 
    CWB                  WebNote Document 
    CWC                  CADfix Wizard Configuration File (ITI TranscenData)
    CWD                  Cardwar Card Deck (Stefan Sarbok)
    CWD                  StarDraft CWAD File 
    CWDOC                CopyWrite Project Document (Bartas Technologies)
    CWE                  Crossword Express (AUS-PC-SOFT)
    CWEB                 C Web 
    CWF                  CodeWatch for Windows Workspace File (Liant)
    CWK                  ClarisWorks Data File (FileMaker, Inc.)
    CWK                  CattleMax Data File (Cattlesoft, Inc.)
    CWK                  GCPrevue Work File (Graphicode)
    CWL                  ClarisWorks Library 
    CWP                  Cakewalk SONAR Audio Project (Cakewalk)
    CWPB                 CopyWrite Project Binder (Bartas Technologies)
    CWR                  Wrapcandy Candy Wrapper (Wrapcandy.com)
    CWR                  Crystal Reports Report (Crystal Decisions)
    CWS                  combustion Workspace (Discreet)
    CWS                  Claris Works Template 
    CWW                  Crossword Weaver File 
    CWZ                  CropWalker File 
    CW_                  Corel Graphics Ver. 10 Workspace (Corel)
    CX                   CodeMapper Script 
    CX3                  click.EXE 3.0 Script File (ASXperts Inc.)
    CXB                  ComicGuru (Fay Solutions)
    CXB                  Custody X Change Data File (Custody X Change, Inc.)
    CXC                  Custody X Change Data File (Custody X Change, Inc.)
    CXD                  SimplePCI Combined Graphics and Data (Compix Inc., Imaging Systems)
    CXE                  Common XML Envelope 
    CXF                  Chemical Abstracts Exchange Format 
    CXH                  Custody X Change Data File (Custody X Change, Inc.)
    CXL                  Business Integrator Extract Schema (Pervasive Software Inc.)
    CXM                  XYZ-Company.com Proprietary File 
    CXM                  QuadColor Color Correction Master 
    CXO                  CXInsight Export File (CodeMatrix Ltd.)
    CXP                  Core Media Player XML-based Playlist File (CoreCodec, Inc.)
    CXP                  CX-Programmer PLC Program File (Binary) (Omron Electronics LLC)
    CXP                  XYZ-Company.com Proprietary File 
    CXS                  ConciseXML Format File (Clear Methods, Inc.)
    CXT                  CX-Programmer PLC Program File (Text) (Omron Electronics LLC)
    CXT                  Vegas Jackpot Gold Game Data File 
    CXT                  Crocodile Technology Circuit (Crocodile Clips Ltd)
    CXT                  Macromedia Director Protected Cast File 
    CXV                  Custody X Change Data File (Custody X Change, Inc.)
    CXX                  Visual C++ Source Code File (Microsoft)
    CXX                  Zortech C++ Program 
    CXX                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CX_                  C Poet Compressed Disk1 File 
    CY                   CCS-lib.2 JPEG-v4 File 
    CYA                  Cyan Color Separation (Adobe)
    CYC                  CYCAS Design Project (Verlag Frese)
    CYCLE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    CYG                  Pollux Data (CyGem, ltd.)
    CYL                  Ribbons Molecular Cylinder 
    CYM                  Callus Game Sound File (Bloodlust Software)
    CYN                  Cynex Root55 Program Extension 
    CYP                  Crocodile Physics Simulation File (Crocodile Clips Ltd.)
    CYP                  Cypherus Encrypted Archive 
    CYT                  Crocodile Technology Simulation (Crocodile Clips Ltd.)
    CZE                  Personal Paint Czech Language User Interface File (Amiga)
    CZE                  NHL Ice Hockey 2002 Required File 
    CZIP                 ZipGenius Compressed Archive (M.Dev Software)
    CZP                  Cloze Pro Archive (Crick Software Ltd.)
    C_                   Winhelp Compressed File 
    C_M                  Eru/erd File 
    C__                  C++ Source; C__ is C++ in Win32 
    C~G                  Windows 3.x System File 
    D                    D Programming Language Source Code File 
    D                    Dialect Source Code File 
    D                    SPPACK File 
    D                    Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    D                    GBG DraftMaker Drawing File (think3, Inc.)
    D$W                  MindStor Driver 
    D00                  Blaster Master Pro File 
    D00                  Ganbatte! Winamp Plug-in 
    D00                  Edlib Tracker Mod Compressed 
    D00                  AdLib Format File 
    D01                  Tonline DB File 
    D01                  MicroSim PCBoard NC Drill Data File (Cadence)
    D01                  Ganbatte! Winamp Plug-in 
    D02                  Tonline DB File 
    D02                  MicroSim PCBoard NC Drill Data File (Cadence)
    D03                  Tonline DB File 
    D03                  MicroSim PCBoard NC Drill Data File (Cadence)
    D04                  MicroSim PCBoard NC Drill Data File (Cadence)
    D05                  MicroSim PCBoard NC Drill Data File (Cadence)
    D06                  MicroSim PCBoard NC Drill Data File (Cadence)
    D07                  MicroSim PCBoard NC Drill Data File (Cadence)
    D08                  MicroSim PCBoard NC Drill Data File (Cadence)
    D09                  MicroSim PCBoard NC Drill Data File (Cadence)
    D1                   Progress Database File (Progress Software Corporation)
    D1                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    D10                  MicroSim PCBoard NC Drill Data File (Cadence)
    D11                  Macro Mania Data File 
    D15                  Beat Master 62 File 
    D2                   Progress Database File (Progress Software Corporation)
    D2                   Diablo II Character Data File (Blizzard Entertainment)
    D2                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    D2D                  3D Fassade Plus 2D/3D Object 
    D2H                  Diablo II Hack It Module File 
    D2I                  Diablo II Item File 
    D2J                  d2jsp Script File (Paul Taulborg and The d2jsp Team)
    D2R                  Dazzle Dazzle Realization 
    D2S                  Diablo II Character and Equipment File (Blizzard Entertainment)
    D2V                  DVD Ripper File 
    D2WMODEL             WebObjects Web Script (Apple Computer, Inc.)
    D2X                  Diablo II Stash File 
    D30                  Driver 
    D32                  Visualisation (IBM)
    D3D                  Direct3D File 
    D3D                  CorelDream 3D Drawing File (Corel Corporation)
    D3F                  MechCommander 2 Encyclopedia (Microsoft)
    D3T                  Doom Texture (Id Software, Inc.)
    D4                   Dataphor D4 Script (Alphora)
    D4C                  Dataphor D4 Catalog File (Alphora)
    D5P                  Dragon UnPACKer 5 Language Pack (Dragon Software)
    D64                  Commodore 64 Disk Image/Game ROM 
    D67                  C64 Emulator Disk Image 
    D70                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    D71                  C64 Emulator Disk Image 
    D80                  C64 Emulator Disk Image 
    D81                  C64 Emulator Disk Image 
    D82                  C64 Emulator Disk Image 
    D83                  GAEB 1990 File (Bundesamt für Bauwesen und Raumordnung)
    D8A                  AmBiz Productivity Pak Catalog 
    DA                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DA                   DynApp Dynamic Application Template File ( e-Synaptix (DynApp Inc.))
    DA!                  Dinfo Data File 
    DA$                  MS Compressed DAT Use UNPACK.EXE. 
    DA0                  Star Trek Generations Saved Games 
    DA0                  Windows 95/98 Registry Backup (Microsoft Corporation)
    DA1                  Registry Backup 
    DA1                  Star Trek Generations Saved Games 
    DA2                  Windows Dancer (Microsoft)
    DA2                  DeepAnalysis 2 eBay Auction Data (HammerTap)
    DA2                  Star Trek Generations Saved Games 
    DA3                  Star Trek Generations Saved Games 
    DA4                  Star Trek Generations Saved Games 
    DA4                  SeeYou Waypoint (Naviter)
    DA5                  Green Line File 
    DA5                  Star Trek Generations Saved Games 
    DA6                  Star Trek Generations Saved Games 
    DA7                  Star Trek Generations Saved Games 
    DA8                  Star Trek Generations Saved Games 
    DA9                  Star Trek Generations Saved Games 
    DAA                  PowerISO Direct-Access-Archive (PowerISO Computing, Inc.)
    DAC                  Pclhandler Image File 
    DAC                  Sound File 
    DAD                  Chromeleon Channel Raw Data 
    DAE                  COLLADA (COLLAborative Design Activity) 3D Digital Asset (The Khronos Group)
    DAEMONSCRIPT         DaemonTools Script (The Daemons Home)
    DAF                  dKart Office Cartographic Data Exchange Format (HydroService AS)
    DAF                  Download Accelerator (SpeedBit Ltd.)
    DAF                  Mobius Document Archive 
    DAF                  OM2 911 Xchange Version 3 & 6 Document 
    DAF                  Pelesoft DAF 
    DAG                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DAG                  Aralia Format Fault Tree 
    DAI                  DAIProcessor 
    DAILY                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DAL                  SpeedBit Download Accelerator Plus DAP List 
    DAL                  DVD-lab Project File (MediaChance)
    DAMS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DAN                  Winphone Address Book 
    DAN                  Analysis System Patient File (DanMedical Ltd.)
    DAN                  ATI Radeon Video Driver 
    DAN                  Personal Paint Danish Language User Interface File (Amiga)
    DANCE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DAO                  Duplicator CD File 
    DAO                  Windows Registry Backup 
    DAP                  Access Data Access Page (Microsoft)
    DAP                  Omnis5 Application File (Raining Data Corporation)
    DAP                  Download Accelerator Temporary File During Download (SpeedBit Ltd.)
    DARBY                TheWord & Bible Companion Darby's Bible Translation Bible 
    DARTS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DAS                  Download Accelerator File List (SpeedBit Ltd.)
    DAS                  PackRat Index 
    DAS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DAT                  Nascar Racing Archive 
    DAT                  Unfinished FastTrack Download 
    DAT                  Nero CD Speed Results File (Nero AG)
    DAT                  SPOT Graphic 
    DAT                  SHARP MZ-series Emulator RAM-Disk File 
    DAT                  RDXR020305.DAT is Morpheus Adware 
    DAT                  Novell Message File 
    DAT                  Terrasoft Dataset Definition File 
    DAT                  Runtime Software Disk Image (Runtime Software)
    DAT                  Shareaza Thumbnail (Shareaza Development Team)
    DAT                  SeeYou Waypoint (Naviter)
    DAT                  Peachtext List Manager (Peachtree Software (Sage Software SB, Inc.))
    DAT                  Chuzzle Saved Game 
    DAT                  LithTech Game Compiled Level File (Touchdown Entertainment, Inc.)
    DAT                  My Personal Programmer Distributed Project (M:Pact Technologies)
    DAT                  VMS Data File 
    DAT                  WordPerfect Merge Data (Corel Corporation)
    DAT                  Z80 Spectrum Emulator Snap/File Format 
    DAT                  MS Word for DOS File 
    DAT                  Windows Registry Hive (Microsoft Corporation)
    DAT                  Video CD MPEG or MPEG1 Movie 
    DAT                  VCD and/or SVCD File 
    DAT                  Mitsubishi DJ-1000 and Photorun Native Format 
    DAT                  Clarion DOS Database (SoftVelocity)
    DAT                  PestPatrol Data/Scan Strings (PestPatrol, Inc.)
    DAT                  EasyRecovery Saved Recovery State (Kroll Ontrack Inc.)
    DAT                  Oric Atmos Snapshot 
    DAT                  Allegro Generic Data 
    DAT                  Windows 9x Registry Hive (Microsoft Corporation)
    DAT                  BLUEWAVE File 
    DAT                  Data 
    DAT                  MapInfo Native Data Format (Mapinfo Corporation)
    DAT                  Internet Explorer Cache File (Microsoft)
    DAT                  Walkman NW-S23 MP3 File Storage Container (Sony)
    DAT                  Gunlok Archive 
    DAT                  Commodore 64 Audio 
    DAT                  Exchange Server Error Message (WINMAIL.DAT) (Microsoft Corporation)
    DAT                  Digital Audio Tape 
    DATA                 SID Tune 
    DATA                 Font2D Resources File 
    DATA                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DATA1                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DATA2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DATA3                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DATA4                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DATA5                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DATACAT              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DATAGRABBER          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DATALINK             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DATALINK2            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DATALINKHELP         Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DATAS1               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DATAS2               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DATE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DATEBOOK             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DAVE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DAVES                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DAWN                 Experimental File Format 
    DAX                  Daxaif Compressed Audio 
    DAX                  DAX Compressed CD Image 
    DAY                  Journal 
    DAYS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DAZ                  Poser 3D File 
    DB                   SQL Anywhere Database (iAnywhere Solutions, Inc.)
    DB                   Progress Database File (Progress Software Corporation)
    DB                   Mozilla (Netscape) Client Certificate or Security Module Database (Mozilla.org)
    DB                   WindowsXP Thumbnail Database (Microsoft Corporation)
    DB                   dBASE IV or dBFast Configuration 
    DB                   MSWorks 
    DB                   data by Synopsys Design Compiler, dbVista, Paradox, Smartware, XTreeGold 
    DB                   Paradox Ver 7 Table Database (Borland Software Corporation)
    DB                   ArcView Object Database File (ESRI)
    DB                   ANSYS Database 
    DB                   Oracle Database 
    DB                   DIGIBooster Module/Song 
    DB                   Solid Database 
    DB                   Novell GroupWise Addressbook Database 
    DB                   SmartWare Office Pac File 
    DB                   MultiEdit Configuration 
    DB$                  dBASE Temporary File 
    DB$                  Clarion for Modula-2 Debug Info (SoftVelocity)
    DB0                  dBASE Initialization File 
    DB1                  Adressmn 
    DB1                  ImageDatabase Index to Image Database (A.I.Soft)
    DB1                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DB2                  ImageDatabase Image Database (A.I.Soft)
    DB2                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DB2                  dBfast File 
    DB2                  dBASE II 
    DB3                  Dr Brain 3 
    DB3                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DB3                  dBASE III File 
    DB4                  dBASE 4 Data 
    DB4                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DB5                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DB6                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DBA                  DataEase Data File 
    DBA                  DarkBASIC 
    DBA                  Palm DateBook File (Palm)
    DBA                  Turbo Prolog Database (Borland Software Corporation)
    DBAS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DBB                  ANSYS Database Backup 
    DBB                  Skype User Data (Profile and Contacts) (Skype Limited)
    DBB                  Mopheus Music File 
    DBB                  DeBabelizer BatchList 
    DBC                  CANdb Network File (Vector Informatik GmbH)
    DBC                  OrCAD Capture CIS Database Configuration (Cadence Design Systems, Inc.)
    DBC                  PROGNOSIS Database Collection Document File 
    DBC                  Visual Foxpro Database Container (Microsoft)
    DBD                  DemoShield Project File (InstallShield Software Corporation)
    DBD                  Business Insight Data 
    DBD                  Psion (Psion PLC)
    DBD                  Clarion for Modula-2 Debug Info (SoftVelocity)
    DBD                  Oracle Record Type 
    DBE                  ActiveSync Backup File (Hewlett-Packard Development Company, L.P.)
    DBE                  Database Engine File 
    DBF                  Oracle 8.x Tablespace File 
    DBF                  Psion Series 3 Database (Psion PLC)
    DBF                  Abacus Law Data (Abacus Data Systems, Inc.)
    DBF                  JetForm FormFlow Data File 
    DBF                  Database 
    DBF                  ArcView Shapefile Attribute Table File (ESRI)
    DBF                  NovaBACKUP Backup Database (NovaStor Corporation)
    DBF                  ACT! Main Database File (Best Software CRM Division)
    DBF                  Alpha Five Table Data File (Alpha Software, Inc.)
    DBF                  The Network Diary Database (CF Systems Ltd)
    DBG                  Paradox File 
    DBG                  MS Visual C++ CodeView COFF-format Debugger information 
    DBG                  Debugger Script Watcom Debugger 
    DBG                  C++ Symbolic Debugging Info (Borland Software Corporation)
    DBG                  ArcView Problem Debug Log File (ESRI)
    DBG                  PFE File 
    DBG                  Watcom Debugger Script 
    DBG                  Oracle Generator File 
    DBI                  Database Explorer Information (Borland Software Corporation)
    DBI                  Isearch Database Information 
    DBI                  Photo Soap2 File 
    DBI                  Fifa World Cup Game Data Comin fe File 
    DBK                  dBASE Database Backup 
    DBK                  OrCAD Schematic Capture Backup File (Cadence Design Systems, Inc.)
    DBKIT                Doggiebox Drum Kit File (Zygoat Creative Technical Services)
    DBL                  LineWriter File 
    DBL                  WindowsXP Product Activation File (Microsoft Corporation)
    DBM                  OzWin CompuServe E-mail/Forum Access SYSOP File 
    DBM                  DataEase / DataEase Express Data File 
    DBM                  DataBoss Menu Template 
    DBM                  DigiBooster Pro Tracker MOD File (APC & TCP)
    DBM                  Cold Fusion Template 
    DBNML                NML Language Database Extension (NevoSoft)
    DBO                  DB/TextWorks Database Directory 
    DBO                  dBASE IV Compiled Program File 
    DBOOK                Pocket Diary Text 
    DBP                  SignIQ Data Broker Project (ImpulseLogic)
    DBP                  Visual Studio .NET Database Project (Microsoft)
    DBPRO                DarkBASIC Professional Project File (The Game Creators Ltd.)
    DBQ                  QuickLink Database 
    DBQ                  Paradox Memo 
    DBQ                  AutoCAD DB Query (Autodesk, Inc.)
    DBR                  Golf Courses File 
    DBR                  TCM - MRP Software Compliled Executable (WorkWise, Inc.)
    DBR                  DeepBurner Burn File List (ASTONSOFT Co.)
    DBR                  Comdial VMMI Database 
    DBR                  DB/TextWorks Database 
    DBS                  ProDas - SQL Windows Data 
    DBS                  Word Printer Description File (Microsoft Corporation)
    DBS                  Works Printer Description File (Microsoft)
    DBS                  PRODAS Data File 
    DBS                  DB/TextWorks Database Textbase Structure File 
    DBS                  Managing Your Money Database 
    DBS                  SQLBase Database (Gupta)
    DBS                  GPS Data File 
    DBS                  DeBabelizer Script 
    DBS                  SQL Anywhere Database (iAnywhere Solutions, Inc.)
    DBS                  Structured Query Language (SQL) Format Database 
    DBSONG               Doggiebox Song File (Zygoat Creative Technical Services)
    DBT                  FoxPro, Foxbase+ Style Memo 
    DBT                  SeeYou Waypoint (Naviter)
    DBT                  OpenInsight Database File (Revelation Software)
    DBT                  Abacus Law Ver. 9 Data (Abacus Data Systems, Inc.)
    DBT                  dBFast Memo Text for Database 
    DBT                  Foxbase Memo 
    DBT                  Database Text File 
    DBT                  AutoCAD R2000 DB Template (Autodesk, Inc.)
    DBT                  JetForm FormFlow Data File 
    DBT                  Oracle Template 
    DBTOOLS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DBV                  Abacus Law Ver. 10 Data (Abacus Data Systems, Inc.)
    DBV                  Flexfile Memo Field File 
    DBW                  DataBoss Database File 
    DBX                  Visual Foxpro Table (Microsoft)
    DBX                  Formula Graphics Project 
    DBX                  Database Index 
    DBX                  DataBeam Image 
    DBX                  Outlook Express E-mail Folder (Microsoft Corporation)
    DB_                  CAD File 
    DC                   DesignCAD CAD File 
    DC                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DC                   Dialect Compiled Script 
    DC                   Spice DC Output 
    DC                   Systems Management Server (SMS) Collection Evaluator Deletion (Microsoft Corporation)
    DC$                  Ntgraph Visual C Wizzard File 
    DC+                  DataCAD Plus Drawing (DATACAD LLC)
    DC1                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DC2                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DC2                  WinControl DC2-PC Motion Control Command Language File (Precision MicroControl Corp.)
    DC2                  DesignCAD CAD File 
    DC2                  DC25 Digital Camera File (Kodak)
    DC3                  Diamond Caves Version 3 Level Group File (Diamond Productions)
    DC5                  DataCAD Drawing File (DATACAD LLC)
    DCA                  D-PAS Portfolio Manager File 
    DCA                  DCA Archiver Compressed Archive 
    DCA                  IBM DisplayWrite Document Content Architecture Text File (IBM)
    DCA                  Visual Basic X DateBook File (Microsoft)
    DCB                  Concordance Database Control Block 
    DCB                  Digital Ceremonial Bugle Configuration (Profile) File (S & D Consulting Int. Ltd.)
    DCC                  WordExpress2.0 Dictionary File 
    DCD                  Dynamic-CD-Wizard Project File 
    DCD                  FORTRAN Data File 
    DCD                  Document Content Description File 
    DCD                  INMOS Transputer Development System Occam Object Code 
    DCE                  DriveCam Video (DriveCam, Inc.)
    DCE                  DCE AutoEnhance Settings File (MediaChance)
    DCE                  AutoCAD R2000 Dialog Error Log (Autodesk, Inc.)
    DCF                  Taskmaster Batch Pilot Form (Datacap Inc.)
    DCF                  HP-95LX Datacomm Configuration 
    DCF                  WordExpress2.0 Dictionary 
    DCF                  ProWORX Nxt Database Configuration (Schneider Electric)
    DCF                  Disk Image File 
    DCF                  DynSite Configuration File 
    DCF                  Dyadic Data File 
    DCF                  Design Rule for Camera File Systems File 
    DCH                  DraftChoice Drawing (TRIUS, Inc.)
    DCI                  AOL HTML Mail (America Online, Inc.)
    DCIM                 Digital Imaging and Communications in Medicine 
    DCK                  Resolume Deck (Edwin de Koning and Bart van der Ploeg)
    DCK                  ZX Spectrum-Emulator 
    DCL                  AutoCAD Dialog Control Language Description (Autodesk, Inc.)
    DCL                  DumpSec 
    DCL                  Delphi Component Library (Borland Software Corporation)
    DCL                  DesignComposer Addition to Layout Knowledge Base 
    DCLST                DC and DC++ Direct Connect 
    DCM                  Atari Disk Image Format (Atari)
    DCM                  Sound 
    DCM                  Digital Imaging and Communications in Medicine (DICOM) Image 
    DCN                  VerdiCash Digital Cash Note (VerdiCash Inc.)
    DCP                  Delphi Compiled Packages (Borland Software Corporation)
    DCP                  Dynamic Cone Penetrometer Data File (Pavement Technology Ltd.)
    DCP                  ArcView Default Codepage File (ESRI)
    DCP                  Terminal Program; DynaComm Script 
    DCP                  OS/2 Data CodePage (IBM)
    DCPIL                Delphi Compiler Symbolic Information (Borland Software Corporation)
    DCR                  Delphi Component Resource (Borland Software Corporation)
    DCR                  Kodak Digital Camera Raw Image File (Kodak)
    DCR                  RIF Picture 
    DCR                  Shockwave Movie (Macromedia, Inc.)
    DCS                  Desktop Color Separation File 
    DCS                  Quark Desktop Color Separation Specification 
    DCS                  Kodak Professional Digital Camera 
    DCS                  AcaStat DataCalc Data Files (AcaStat Software)
    DCS                  Color Separated EPS Format 
    DCS                  Bitmap Graphic 
    DCS                  ACT! Activity Data File (Best Software CRM Division)
    DCS                  1st Reader 
    DCT                  Dictionary 
    DCT                  Delphi Component Template (Borland Software Corporation)
    DCT                  Visual Foxpro Database Container (Microsoft)
    DCT                  ArcView Geocoding Dictionary File (ESRI)
    DCT                  Clarion for Windows Data Dictionary (SoftVelocity)
    DCT                   DALiM LiTHO Continuous Tone Bitmap (Blanchard Systems Inc.)
    DCT                  StartSpanish Dictation Sound File (GB Blanchard)
    DCT                  GIS (and many others) Dictionary 
    DCTMP                DC++ Incomplete Download File 
    DCU                  Delphi Compiled Unit (Borland Software Corporation)
    DCUIL                Delphi Compiler Symbolic Information (Borland Software Corporation)
    DCV                  DriveCrypt Volume 
    DCW                  Draft Choice for Windows 
    DCX                  ElectraSoft Fax 
    DCX                  Graphics Multipage PCX Bitmap 
    DCX                  Macros 
    DCX                  PC-Paintbrush File (Zsoft)
    DCX                  Visual Foxpro Database Container (Microsoft)
    DCX                  DesignCAD (UpperSpace.com)
    DC~                  Ntgraph Visual C Wizzard File 
    DD                   Macintosh Compressed Disk Doubler Archive 
    DD                   C64 Doodle 
    DD                   C64 Image Doodle 
    DD2                  Dungeon Designer 2 Map Design (ProFantasy Software Ltd.)
    DD3E                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DDATA                Pocket Diary Text File 
    DDB                  ProWORX Nxt Descriptor Data (Schneider Electric)
    DDB                  Protel 99SE Design Database File 
    DDB                  INMOS Transputer Development System Occam Debugging Info 
    DDB                  Device-Dependent Bitmap Graphics File 
    DDB                  DProfiler Cost Database (Beck Technology)
    DDB                  ACT! Sales Data File (Best Software CRM Division)
    DDB                  Design Database File 
    DDC                  DivX Descriptor DVD Description File (Daren-Softwares)
    DDC                  ddoc Print and Preview Print Preview Document (Don Dickinson)
    DDD                  TIS-Office Digital Tachograph Data File (Siemens AG)
    DDD                  BullsEye Style Sheet 
    DDD                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DDD                  Medlin Accounting Direct Deposit Information (Medlin Accounting)
    DDD                  Fuji Xerox 2D CAD Data 
    DDD                  Acrobat Distiller (Adobe)
    DDD                  ColdFusion Verity Engine Fields Definition (Adobe Systems Incorporated)
    DDD                  Alpha Five Table Data Dictionary File (Alpha Software, Inc.)
    DDD                  ARC+ Drawing (ACA)
    DDD                  Diagram Designer Project Data File (MeeSoft)
    DDE                  Dynamic Data Exchange 
    DDE                  TIMTEL File 
    DDF                  Grand Theft Auto III File (Take-Two Interactive Software, Inc.)
    DDF                  Diamond Directive File (MIcrosoft)
    DDF                  Doom Definition File 
    DDF                  Btrieve Data Dictionary File (Pervasive Software Inc.)
    DDF                  ACDSee Use Database (Photo Disc Data File) (ACD Systems Ltd.)
    DDF                  Music Library File 
    DDF                  ACT! Database Definition File (Best Software CRM Division)
    DDF                  GPS Pathfinder Office Data Dictionary (Trinble Navigation Limited)
    DDF                  Bitstream Fontware 
    DDF                  My Personal Diary Entry Storage File (CAM Development)
    DDF                  ProWORX Nxt Descriptor Data (Schneider Electric)
    DDI                  DiskDupe Image 
    DDI                  Disk Doubler Image 
    DDIF                 Digital Equipment DIGITAL Document Interchange Format 
    DDJ                  SilkRoad Alchemy Page (Joymax. Co., Ltd.)
    DDK                  OS/2 Saved SKF Format File (IBM)
    DDL                  SQL Data Definition Language File 
    DDM                  Alpha Five Table Objects (Alpha Software, Inc.)
    DDM                  Intelliware Data Modeller 
    DDNA                 Miramar Systems Desktop DNA 
    DDO                  Tank3d Game File 
    DDOC                 DigiDoc Digital Signature (OpenXAdES Project)
    DDP                  Inor DSoft Software File 
    DDP                  OS/2 Device Driver Profile File 
    DDP                  DDAdd Dinkey Dongle Protection Parameters (Microcosm Ltd.)
    DDP                  Delphi Diagram Portfolio (Borland Software Corporation)
    DDPOKERSAVE          DD Tournament Poker Save File (Donohoe Digital LLC)
    DDR                  Systems Management Server (SMS) Discovery Data Record (Microsoft Corporation)
    DDR                  DDRemote Parameter File (Microcosm Ltd.)
    DDR                  FileMaker Pro Database Design Report (FileMaker, Inc.)
    DDS                  XMap XData Dataset (Delorme)
    DDS                  Orbiter Texture File (Martin Schweiger)
    DDS                  DirectX (Multiple Versions) DirectDraw Surface (Microsoft)
    DDS                  Photoshop Compressed Textures (Adobe)
    DDS                  INMOS Transputer Development System Occam Compiler Descriptor 
    DDS                  Battlefield 1942 Unit/Object Texture File (Electronic Arts, Inc.)
    DDT                  Linux Configuration File 
    DDT                  Neurological Data for Statistical Analysis 
    DDT                  Jill Game File 
    DDT                  Diagram Designer Template (MeeSoft)
    DDV                  Xbase Dbfast Example Ivadbsp File 
    DDW                  Datadisc Data 
    DDW                  CSPro File 
    DDX                  Alpha Five Dictionary Index File (Alpha Software, Inc.)
    DDX                  Device Data Exchange Format File (Get Organised Our Dissemination of Die Information in Europe)
    DDX                  Juno Get File 
    DE                   MetaProducts Download Express Incompletely Downloaded File 
    DE                   Win Ace 204 File 
    DE$                  Modem Bitware Fax Disk2 File 
    DEAD                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEAR                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEB                  Debian Linux Package 
    DEB                  DOS Debug Script 
    DEBIAN               Debian Make File 
    DEBUG                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEC                  EasyConverter Project File (Dolphin Computer Access Ltd.)
    DEC                  Decoded File 
    DEC                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEC                  VersaPro Declaration File 
    DEC                  Magic: The Gathering Deck of Cards (Wizards of the Coast, Inc., a subsidiary of Hasbro, Inc.)
    DEC                  ArcView UNIX Hyperhelp Supporting File (ESRI)
    DEC                  DEC DX, WPS Plus Document 
    DECK                 DeckTools Deck Project (dotDeck File) (DeckTools)
    DECOR                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEE                  Creator Simulator Network Description File (Tufts University)
    DEELITED             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEF                  Geoworks Assembly Header File 
    DEF                  AniSprite (CompuPhase)
    DEF                  Data Entry/Review Module (DATAIR Employee Benefit Systems, Inc.)
    DEF                  ADS Source Code (Autodesk, Inc.)
    DEF                  Defaults 
    DEF                  ZDP ZEBU's Directory Printer Definitions of User-defined Output Formats (ZEBU Informatics)
    DEF                  Definitions 
    DEF                  Modula-2 Definition Module File 
    DEF                  ArcView Defaults File (ESRI)
    DEF                  SGML Tag Definition 
    DEF                  SmartWare II Data File 
    DEF                  Bert's Dinosaurs File 
    DEF                  C++ Definition 
    DEF                  Module Definition/Configuration File 
    DEF                  TLink - Wlink - Others Linker Definition 
    DEF                  3D Fassade Plus Define Module 
    DEF                  OPTLINK Digital Mars Module Definition File (Digital Mars)
    DEF                  Visual C++ Definition 
    DEF                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEFAULT              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEFAULTS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEFENSE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEFI                 Oracle Ver. 7 De-install Script 
    DEFS                 Xcode Core MIG Program (Apple Computer, Inc.)
    DEH                  Dehacked File 
    DEI                  Gerber Design Engineered Interface 
    DEJAVU-THEME-1       Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEK                  Memorize-It Multimedia Flashcard Deck (Side-Eight Software)
    DEL                  Creator Simulator Compiled Module (Tufts University)
    DEL                  Data 
    DEL                  Deleted Data List 
    DEL                  PCTracker Undelete Tracking File 
    DELAY                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DELETE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DELTA                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEM                  Vista Pro Graphic 
    DEM                  Unreal Tournament Demonstration File (Epic Games, Inc.)
    DEM                  Descent Demonstration 
    DEM                  Turing (Holt Software Associates, Inc.)
    DEM                  Demetra Automated Project (European Commission (Eurostat))
    DEM                  ArcView Digital Elevation Model File (ESRI)
    DEM                  USGS US Geological Survey National Mapping Division Topo30 3D File (USGS SDTS)
    DEM                  Quake/Hexen II .DEM File Format 
    DEM                  Half-Life Demo File (Sierra)
    DEM                  Vista Digital Elevation Model File 
    DEM                  Delphi Edit Mask (Borland Software Corporation)
    DEM                  Creator Simulator Temporary File (Tufts University)
    DEM                  Raven Shield In-Game Recording (Ubi Soft Entertainment)
    DEMO                 Demo 
    DEMO                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEMO2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEMOPIC              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEO                  Creator Simulator Compiled Module (Tufts University)
    DEP                  Visual Basic Setup Wizard Dependency (MIcrosoft)
    DEP                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEPLOY               DeployMaster Installation Project File (JGsoft - Just Great Software)
    DEPR                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEPRO                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEPROTECT            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEPROTECTS           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DER                  DER Encoded X509 Certificate 
    DES                  File Investigator Description File (RobWare)
    DES                  Tribes 2 Game File (Sierra)
    DES                  Interscope BlackBox File 
    DES                  Pro/DESKTOP File 
    DES                  Delphi Description File 
    DES                  Description Text 
    DES                  QuickBooks Forms Template (Intuit, Inc.)
    DES                  Medlin Accounting Sales Code Information (Medlin Accounting)
    DES                  nProtect GameGuard Game Security Plug-in (INCA Internet Co, Ltd.)
    DES                  Creator Simulator Mozart Source File (Tufts University)
    DES                  Designer Technical Illustration (Corel)
    DES                  ACT! (Best Software CRM Division)
    DESC                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DESC                 Description 
    DESCR                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DESIGN               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DESK                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DESK                 Card Deck File 
    DESKEDIT             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DESKLINK             Windows Explorer Desktop Controls (Microsoft)
    DESKLINK             Desklink Powertoy File (Microsoft)
    DESKTOP              DesktopX Theme (Stardock Corporation, Inc.)
    DESKTOP              Desktop Entry Standard Desktop Entry File 
    DESKTOP              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DESKTRCKR            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DET                  StageII Detector Properties File (FITS Format) 
    DET                  Dart File 
    DET                  ACT! Saved E-mail File (Best Software CRM Division)
    DETAILS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DETECTOR             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEU                  Personal Paint German Language User Interface File (Amiga)
    DEU                  German File (possibly) 
    DEU                  ATI Radeon Video Driver 
    DEV                  Dev-C++ Project File (Bloodshed Software)
    DEV                  LaTeX File 
    DEV                  Device Independent TeX 
    DEV                  Device Driver 
    DEV                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEV                  e-Sword Devotions (Rick Meyers)
    DEV                  London Architect Custom Device File (BSS Audio)
    DEVEL                Developper File 
    DEVIATION            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEVLIST              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEWF                 General Sound 
    DEWF                 Macintosh SoundCap/SoundEdit Recorded Instrument 
    DEX                  WinGlucofacts Data File (Bayer HealthCare LLC)
    DEX                  Excel File 
    DEX                  DEXIS Digital X-ray (ProVision Dental Systems, Inc.)
    DEZ                  DeZign for databases Project (Datanamic)
    DEZ                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DEZ                  DES Encrypted Zip File 
    DEZ                  Data Ease version 4.x 
    DE_                  Doko Compressed Install File 
    DF                   XtalView Double .FIN File (The Computational Center for MacroMolecular Structures (CCMS))
    DF                   Paradox Table Filter 
    DF                   NCSA Hierarchical Data File 
    DF                   Lightscape Parameters 
    DF                   Data File 
    DF$                  Ntgraph Visual C Wizzard File 
    DF1                  Omnis7 and Omnis Studio Database File (Omnis Software)
    DF2                  Omnis7 and Omnis Studio Database File (Omnis Software)
    DF3                  Omnis7 and Omnis Studio Database File (Omnis Software)
    DF3                  Pov-ray for Windows (Persistence of Vision Raytracer Pty Ltd.)
    DF4                  Omnis7 and Omnis Studio Database File (Omnis Software)
    DF5                  Omnis7 and Omnis Studio Database File (Omnis Software)
    DF6                  Omnis7 and Omnis Studio Database File (Omnis Software)
    DF7                  Omnis7 and Omnis Studio Database File (Omnis Software)
    DF8                  Omnis7 and Omnis Studio Database File (Omnis Software)
    DF9                  Omnis7 and Omnis Studio Database File (Omnis Software)
    DFA                  Amateur Strip Poker Data 
    DFB                  Data Flask Data Formula File (Interscape)
    DFC                  Voicemail File 
    DFCLASS              Together ControlCenter Development Tool 
    DFCOMPONENT          Together ControlCenter Development Tool 
    DFD                  CWK File 
    DFD                  File Flow Diagram Graphic 
    DFD                  Prosa Data Flow Diagram Graphic 
    DFD                  ABC Programming Language Dyadic Function 
    DFF                  Digital Animation Format 
    DFF                  Grand Theft Auto III Export Format (Take-Two Interactive Software, Inc.)
    DFF                  Dynamic File Format Resource Database (Kreative Korporation)
    DFG                  Data Flask Grid File (Interscape)
    DFI                  DiskFactory32 Diskette Image (Accurate Technologies)
    DFI                  Digifont Outline Font Description 
    DFL                  Signature Default Program Settings 
    DFL                  Fresh Download List File (Freshdevices.com)
    DFM                  Digital-FM 
    DFM                  Delphi Form (Borland Software Corporation)
    DFM                  File Flow Diagram Model 
    DFM                  Prosa Data Flow Diagram Model 
    DFM                  Wintidy File 
    DFM                  Datafax Image 
    DFM                  C++ Builder 6 Form (Borland Software Corporation)
    DFN                  Definition File 
    DFN                  DFNWarrior Itemlist 
    DFN                  Delphi Integrated Translation Environment (Borland Software Corporation)
    DFN                  TalkAssist Icon String Meaning 
    DFN                  Winyaps Style 
    DFONT                Macintosh OS X Data Fork Font File (Apple)
    DFP                  Data Flask Package File (Interscape)
    DFP                  Digital Fusion Plug-in 
    DFP                  Zoids Commando Player Data File (D9 Software)
    DFP                  Stabilogram Diffusion Analysis File 
    DFPACKAGE            Together ControlCenter Development Tool 
    DFS                  Swirlique Data/Parameter File (Mystic Fractal)
    DFS                  AutoCAD Utility Defaults (Autodesk, Inc.)
    DFS                  Delight Sound File 
    DFSEQUENCE           Together ControlCenter Development Tool 
    DFT                  PC Draft File 
    DFT                  cncKad2002 Drafting Tool (Metalix)
    DFT                  BullsEye Style Sheet 
    DFT                  Fakt2000 File 
    DFT                  SolidEdge CAD File 
    DFT                  WaveMaker File 
    DFT                  Workshare Synergy File 
    DFV                  GYYR DVMS Digital CCTV Recording 
    DFV                  Word Print Format Template (Microsoft Corporation)
    DFW                  Derive Math Package (Texas Instruments Inc.)
    DFWEBAPPLICATION     Together ControlCenter Development Tool 
    DFX                  AutoCAD 3D Vector Graphics (Autodesk, Inc.)
    DFX                  Micrografx Designer Effects 
    DG                   Xsteel Drawing File (Tekla)
    DG                   Auto-trol Graphics 
    DG-KEY               DigiGuide Key File (GipsyMedia Limited)
    DGC                  DigiGuide Channel Extension (GipsyMedia Limited)
    DGC                  Digital G Codec Archiver 
    DGC                  TurboTax (Intuit)
    DGC                  IBM DataGLANce Capture File 
    DGF                  Acrobat (Adobe)
    DGF                  GPS Pathfinder Office Geoid Grid File (Trinble Navigation Limited)
    DGK                  Delcam Powershape/Powermill 
    DGL                  DigiGuide Language File (GipsyMedia Limited)
    DGL                  DynaGeo License File (EUKLID)
    DGM                  Freelance Graphics Diagram (IBM)
    DGM                  Digital Geospatial Metadata File 
    DGM                  Arcland Flow Chart Diagram 
    DGMARKER             DigiGuide Marker File (GipsyMedia Limited)
    DGN                  ArcView Design Drawing File (ESRI)
    DGN                  Dragon 32 Emulator 
    DGN                  EdG Design File 
    DGN                  Intergraph Graphics 
    DGN                  MicroStation Design File (Bentley Systems, Incorporated)
    DGO                  Dimitrius Settings File 
    DGP                  Creator Simulator Tab Fault List (Tufts University)
    DGPRINT              DigiGuide Print Template (GipsyMedia Limited)
    DGR                  PhoneTools Internal Graphic Format (BVRP Software UK)
    DGR                  MicroStation 95 CAD Design File (Bentley Systems, Incorporated)
    DGR                  Fax Page 
    DGR                  Pivot Graphic Fax 
    DGR                  DART Pro 98 File Group Details 
    DGR                  Keeptool ER Diagrammer 
    DGS                  Dagesh Document (GalTech Software)
    DGS                  AT Advanced Diagnostics 
    DGSCHEME             DigiGuide Scheme (GipsyMedia Limited)
    DGSCHEME-WEB         DigiGuide Scheme (GipsyMedia Limited)
    DGSS                 DigiGuide Smart Search File (GipsyMedia Limited)
    DGT                  Digital Gallery 
    DGTOOLBAR            DigiGuide Toolbar (GipsyMedia Limited)
    DGW                  Digi-Watcher Proprietary Video Format 
    DGX                  Rational XDE (IBM)
    DH                   Geoworks Dependency Information File 
    DH2                  Pursuit CRM Report Viewer File (Pursuit Services Ltd.)
    DHE                  Visual Basic Dialog Box Help Editor Document 
    DHE                  Help Workshop Dialog Box Help Editor Document (Microsoft)
    DHF                  Help Maker File 
    DHG                  Direct Hit Graphics Engine Graphic Output (Direct Hit Systems, Inc.)
    DHP                  Dr. Halo PIC 
    DHR                  Image Apple IIe / Apple IIc, Double Hi-Res Writes 2 colors. 
    DHR                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DHRP                 Image Apple IIe / Apple IIc, Double Hi-Res 2 colors compressed. 
    DHT                  Ox Object-Oriented Matrix Programming Language Gauss Data File 
    DHT                  Gauss Data File 
    DHT                  DanTemplate Templated HTML File (DanSoft Australia)
    DI                   Sound 
    DI                   Digital Illusion Format 
    DI                   Atari Disk Image (Atari)
    DIA                  Diaporama 
    DIA                  Computer Support Corp Diagraph Graphics File 
    DIA                  DIA Drawing 
    DIA                  Psion Organiser Diary File (CM/XP Format) (Psion PLC)
    DIABLOFONT           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIALLIST             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIALOG               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIALOGUE             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIAMONDS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIB                  Device-Independent Bitmap Graphic 
    DIC                  Timeslips Personal Dictionary (Peachtree Software)
    DIC                  Mozilla Dictionary File (Mozilla)
    DIC                  Microsoft Business Solutions--Great Plains Program Source Code (Microsoft)
    DIC                  DICOM Digital Imaging and Communications in Medicine Format Bitmap 
    DIC                  Dictionary 
    DIC                  Notes Dictionary (IBM)
    DIC                  WordExpress2.0 Dictionary 
    DICM                 DICOM Digital Imaging and Communications in Medicine 
    DICOM                DICOM Digital Imaging and Communations in Medicine Format Bitmap 
    DICPROOF             Dictionary Proofing File (Microsoft)
    DICT                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DICT                 Dicts File 
    DICTIONARY           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DID                  BullsEye File 
    DID                  Acrobat Distiller (Adobe)
    DIE                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIE-AGAIN            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIESEL               Bandits - Phoenix Rising (GRIN)
    DIF                  Wright Design's Design Image Format 
    DIF                  Spreadsheet 
    DIF                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIF                  Raytheon Raster Bitmap 
    DIF                  ProWORX Nxt Data Interchange File (Schneider Electric)
    DIF                  Output from diff Command script for Patch command. 
    DIF                  OS/2 Display Information File 
    DIF                  Excel Data Interchange Format (Microsoft)
    DIF                  DVCPRO or DV Video File 
    DIF                  Data Interchange Format 
    DIFF                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIFF                 Difference File 
    DIG                  Sound Designer Audio File 
    DIG                  Text Document 
    DIG                  Digilink Format 
    DIGI                 DIGIBooster Module Format 
    DII                  Summation Document Image Information Load File (CT Summation, Inc.)
    DIL                  DesignComposer Input DIL Script 
    DIL                  Delphi File 
    DIL                  Dialog LOTUS or HighEdit Import-Export Images 
    DIL                  WordExpress2.0 File 
    DIM                  AutoCAD Dimension File (Autodesk, Inc.)
    DIM                  XMap Symbol File (Delorme)
    DIME                 MyPublisher BookMaker Book File (MyPublisher)
    DIN                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIN                  ADC Labs eWave File 
    DIN                  DataEase File 
    DINERDASHSAVEDGAME   Diner Dash Saved Game (Gamelab)
    DINGBATS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIP                  DVDINFOPro Saved Data Graph (Nic Wilson & Jonathan Wilson)
    DIP                  Watcom Debugger Debug Info Processor 
    DIR                  Procomm Plus Dialing Directory 
    DIR                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIR                  Macromedia Director Movie (Macromedia)
    DIR                  Grand Theft Auto III (Take-Two Interactive Software, Inc.)
    DIR                  Directory Indicator (Microsoft)
    DIR                  DEC VAX Directory 
    DIR                  CPS Backup 
    DIR                  ArcView INFO Directory Manager File (ESRI)
    DIR                  Worms Armageddon Archive 
    DIR                  VAX - CPS Backup Directory File 
    DIR                  WebSTAR Directory Indexer Plugin File 
    DIRECT               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIRECTORY            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIRLIST              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIS                  Dashboard of Sustainability Indicator Set (International Institute for Sustainable Development and JRC)
    DIS                  VAX Mail Distribution File 
    DIS                  Corel Draw Thesaurus 
    DIS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIS                  Mobius Distribution Database 
    DIS                  DKB Ray Tracer File 
    DIS                  Data Import Specification Layout File (DATAIR Employee Benefit Systems, Inc.)
    DIS                  Soundweb Firmware File (BSS Audio)
    DIS                  Oracle Discoverer Workbook (Oracle)
    DISAP                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DISASM               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DISCO                NET Web Service Discovery File (Microsoft)
    DISHONOR             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DISK                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIST2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIST4                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIT                  Windows2000 Active Directory Schema (Microsoft)
    DIT                  DRAWiT Save File (Poster Software)
    DIV                  DivX Movie (DivXNetworks, Inc.)
    DIVX                 DivX Movie (DivXNetworks, Inc.)
    DIVX                 Movie Encoded with DivX-codec 
    DIX                  Stonevoice Translator Extended Dictionary (Stonevoice)
    DIY                  HomeStudio Project File (Do-It-Yourself Portraits)
    DIZ                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DIZ                  Description In Zip File 
    DI_                  Doko Compressed Install File 
    DJE                  MattBatt iAM-player Track List (MattBatt Productions)
    DJIA                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DJTA                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DJUA                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DJV                  DjVu File 
    DJVU                 DjVu File (LizardTech, Inc.)
    DJW                  Datadisc Explore 
    DJX                  Dundjinni Art (Fluid Entertainment)
    DK                   Danemark 
    DKB                  DKBTrace Ray-Traced Graphics File 
    DKB                  DataKeeper Backup File (Symantec)
    DKD                  Submarine Titans Battle Map (Ellipse Studios)
    DKI                  Doppel Kopf 
    DKP                  Dreamkeys Plug-in 
    DKS                  DataKeeper Configuration File (Symantec)
    DKT                  DktBot File 
    DKX                  Submarine Titans Battle Map (Ellipse Studios)
    DKZ                  Description In Zip File 
    DL                   MAC Image Format 
    DL                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DL                   FLEXnet Manager Debug Log File (Macrovision Corporation)
    DL                   Masked .DLL File 
    DL                   Animation 
    DL$                  System DLLbackup File 
    DL0                  ABBYY Finereader 4.0 Sprint (ABBYY Software House)
    DL4                  Lindab Dimsilencer File 
    DLB                  Spectrum Analyzer Spectrum Data File 
    DLB                  TigerSHARC Processor VisualDSP++ 3.5 C/C++ Compiler Run-time Library File 
    DLB                  SmartBook 
    DLB                  SSL Trojan Data File 
    DLC                  Echelon HQ Configuration File 
    DLC                  3ds Max Animation Controller File (Autodesk, Inc.)
    DLC                  Model Dislocation File 
    DLC                  DIGILINEAR Compressed Archive (MURASE INDUSTRIES INC.(DIGILINEAR))
    DLC                  Disclib Library Database (Lyra Software Ltd.)
    DLD                  1-2-3 (IBM)
    DLD                  DLLdetective 
    DLE                  Macromedia Designer 3D Translator File 
    DLF                  Belkin Router Firmware Update 
    DLF                  DataCAD Double Precision Layer (DATACAD LLC)
    DLG                  Infinity Engine Game Dialog File 
    DLG                  C++ Dialogue Script 
    DLG                  ArcView Digital Line Graph File (ESRI)
    DLG                  Digital Line Graph 
    DLG                  Dialog Resource Script 
    DLG                  Infinity Game Engine Inter-Character Dialog (BioWare Corp.)
    DLG                  MicroSim PCBoard NC Drill Descriptive File 
    DLI                  Beast 2.02 Trojan File 
    DLK                  INMOS Transputer Development System Occam Compiler Link Info 
    DLK                  Xprotect License Code List (Milestone Systems)
    DLL                  Dynamic Link Library 
    DLL                  CorelDRAW Export/Import Filter 
    DLL                  CorelDraw Export/Import Filter 
    DLM                  3ds Max (Autodesk, Inc.)
    DLM                  Dirty Little Helper Cheat File 
    DLM                  FileMaker Pro Data 
    DLM                  ASCII Delimited File 
    DLM                  Dynamic Link Module 
    DLN                  Scanner Recognita File 
    DLO                  3ds Max Plug-in (Autodesk, Inc.)
    DLOG                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DLR                  TIB Browser 
    DLS                  Norton DiskLock Setup File 
    DLS                  DLS/32 Supply Chain Planning (Adapta)
    DLS                  dataLive Database Engine File 
    DLS                  Blood 2 
    DLS                  Downloadable Sounds 
    DLT                  VCTEditor Data File (LEONI Wiring Systems UK Ltd.)
    DLU                  Dirty Little Helper Update File 
    DLV                  CATIA Export File (Dassault Systèmes)
    DLW                   DALiM LiTHO Line Work Bitmap (Blanchard Systems Inc.)
    DLX                  Complete Works (Toplevel Computing)
    DLX                  Premiere Movie (Adobe Systems Inc.)
    DLY                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DLY                  Complete Works Wordprocessing Template (Toplevel Computing)
    DLZ                  Compressed Data File 
    DL_                  Compressed DLL File 
    DL_                  The Land Of UM 
    DM                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DM                   Paradox Data Model (Borland Software Corporation)
    DM2                  Quake 2 Demo File 
    DM2                  Demetra Detailed Analysis Project (European Commission (Eurostat))
    DM3                  Quake 3 Arena Demo File 
    DMA                  Direct Memory Access Programming File 
    DMB                  DHTML Menu Builder Project File (xFX JumpStart)
    DMB                  BYOND Byte-Code (BYOND)
    DMC                  Medical Manager DML System Compiled Script 
    DMD                  OS/2 Warp Device Manager Driver 
    DMD                  Corel Data Modeling Desktop 
    DMD                  dB2K/dQuery Web DataModule 
    DMD                  Visual dBASE Data Module 
    DME                  Medical Manager DML System Data Merge 
    DMF                  Delusion/XTracker Digital Music File 
    DMF                  Packed Amiga Disk Image 
    DMF                  Windows Disk Map File (Microsoft)
    DMF                  Macintosh Disk Image 
    DMF                  DemoForge Software Demo File (DemoForge)
    DMF                  Digitals/Delta Map File (Geosystem (Ukrainian Mapping Agency))
    DMG                  Macintosh OS X Disk Copy Disk Image File (Apple Computer, Inc.)
    DMG                  Oracle Binary Format Dump File (Oracle)
    DMJ                  Dustman Selection for Delete 
    DMK                  Digimask File 
    DML                  Darn! Shopping! Mall File 
    DML                  Drillbench Drilling Engineering Markup Language (Scandpower Petroleum Technology)
    DML                  DynaScript Markup Language (Dynascript Technologies, Inc.)
    DML                  Medical Manager DML System Script (Emdeon Corporation)
    DMN                  SAS Stat Studio Metadata (SAS Institute Inc.)
    DMO                  The Land Of UM Demo File 
    DMO                  S3M Compressed Module 
    DMO                  Twin TrackPlayer 
    DMO                  Game Demo File 
    DMO                  Duke Nukem 3D/Redneck Rampage Recorded Games 
    DMO                  Derive Demo 
    DMP                  ORACLE File 
    DMP                  Precision Guesswork LANWatch Capture File 
    DMP                  Screen or Memory Dump 
    DMP                  INMOS Transputer Development System Occam Core Dump 
    DMP                  Windows Error Dump (Microsoft Corporation)
    DMS                  Exigen Workflow Imaging File (Exigen, Inc.)
    DMS                  Packrat Database Index 
    DMS                  Amiga DISKMASHER Compressed Archive 
    DMS                  BYOND Script File (BYOND)
    DMSD                 VideoWave DVD Project (Roxio, a division of Sonic Solutions)
    DMT                  XMap Transfer File (Delorme)
    DMT                  Delphi Menu Template (Borland Software Corporation)
    DMT                  Delcam Machining Triangles 
    DMT                  Street Atlas Transfer File (Delorme)
    DMU                  Digital Mugician Song/Module 
    DMV                  MPEG-1 File 
    DMV                  Acrobat Catalog (Adobe)
    DMW                  DrawMe Vector Image 
    DMW                  Datamass Ag 
    DMX                  ProTrader Database 
    DMX                  Medical Manager DML System Index for Custom .DME Files 
    DMY                  Container File 
    DMY                  Dummy 
    DMZ                  DMesh 3D Model 
    DM_1                 Call of Duty Demo File (Activision)
    DM_2                 Call of Duty Demo File (Activision)
    DM_3                 Call of Duty Demo File (Activision)
    DM_4                 Call of Duty Demo File (Activision)
    DM_48                Quake 3 Arena Demo File 
    DM_57                Return to Castle Wolfenstein Demo File (Id Software, Inc.)
    DM_59                Return to Castle Wolfenstein Demo File (Id Software, Inc.)
    DM_66                Quake 3 Arena Demo File 
    DM_67                Quake 3 Arena Demo File (ID Software)
    DN2                  Windows Dancer (Microsoft)
    DN8                  Blue Martini UTF-8 Encoded DNA File 
    DNA                  Virtual Woman Digital DNA (CyberPunk Software)
    DNA                  Gene Pool Swimmer Data (JJ Ventrella)
    DNA                  NewMoon Distributed Network Application Information 
    DNA                  Desktop DNA Data Storage File 
    DNA                  Blue Martini DNA File 
    DNA.XML              Blue Martini XDNA File 
    DNASYM               Desktop DNA Compiled Application Script 
    DNAX                 Desktop DNA Exclusion List 
    DNC                  Machine Tool Communications 
    DNC                  Compressed Dictionary File 
    DND                  ClustaW Tree File 
    DNE                  Darn Ver. 5+ Windows Events List 
    DNE                  Netica Bayes Net File (Norsys Software Corp.)
    DNG                  Dungeon File 
    DNG                  Digital Negative Format (Adobe Systems Inc.)
    DNI                  realMyst 3D Archive 
    DNK                  Killad File 
    DNL                  netMod Modem Firmware Upgrade File 
    DNL                  DigitalWebBook Electronic Book 
    DNP                  DRM File 
    DNP                  NeatImage Ver. 2.2 Free Ed. Profile File (ABSoft)
    DNS                  DynSite Plug-in File 
    DO                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DO                   ModelSim Script File (Mentor Graphics Corp.)
    DO                   DigitalOutrage Hosting Control Panel Info 
    DO                   Stata Batch Analysis File 
    DO                   MicroSim PCBoard File With Autorouting Control Information 
    DO$                  Modem Bitware Fax Disk2 File 
    DO0                  Sysinfo Sa51swe File 
    DO1                  Sysinfo Sa51swe File 
    DO2                  Sysinfo Sa51swe File 
    DOB                  Visual Basic User Document Form File (Microsoft)
    DOC                  Word DOT File (Microsoft Corporation)
    DOC                  DisplayWrite 4 File 
    DOC                  Word Document (Microsoft Corporation)
    DOC                  Document or Documentation 
    DOC                  FrameMaker/FrameBuilder Document (Adobe)
    DOC                  Interleaf Document Format 
    DOC                  Mastercam Material Files (CNC Software, Inc.)
    DOC                  BIFF File (Microsoft)
    DOC                  Perfect Office Document (Perfect Office)
    DOC                  Palm Pilot DOC File Format 
    DOC                  Maybe RTF 
    DOC1                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOC2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOC3                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOC4                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOC5                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOC6                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOC7                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOC8                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOCENX               egis Encrypted Word DOC File (HiTRUST Inc.)
    DOCGS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOCHTML              Word HTML Document (Microsoft Corporation)
    DOCKLET              ObjectDock Docklet (Stardock Corporation)
    DOCKPACK             ObjectDock Theme Pak (Stardock Corporation)
    DOCKTHEME            ObjectDock Theme (Stardock Corporation)
    DOCM                 Word Microsoft Office Open XML Format Document with Macros Enabled (Microsoft Corporation)
    DOCMENX              egis Encrypted DOCM (Word 2007) File (HiTRUST Inc.)
    DOCMHTML             Word HTML Document (Microsoft Corporation)
    DOCS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOCTOR               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOCU                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOCUMENTATION        Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOCX                 Word Microsoft Office Open XML Format Document (Microsoft Corporation)
    DOCXENX              egis Encrypted Word Open XML DOCX File (HiTRUST Inc.)
    DOE                  Arena Model File (Rockwell Automation, Inc.)
    DOF                  Delphi Option File (Borland Software Corporation)
    DOF                  Racer 3D Geometry File (Ruud van Gaal, Dolphinity)
    DOF                  WinTidy File 
    DOG                  Laughing Dog Screen Saver 
    DOG                  Dogz Dog Description File (Ubisoft Entertainment)
    DOG                  PAFEC Phase 2 CAD File (SER Systems Ltd.)
    DOH                  Geoworks Dependency Information File 
    DOK                  German or Dutch Text 
    DOK                  Dokument, often ASCII Text 
    DOL                  GameCube Executable File (Nintendo)
    DOL                  D-PAS Portfolio Manager Flexible Benchmarking Analysis File 
    DON                  Textur Editor File 
    DON                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOO                  Vibrants Music 
    DOO                  Atari Doodle 
    DOP                  Digital Orchestrator File 
    DOQ                  Digital Orthophoto Quadrangle 
    DOQQ                 Digital Orthophoto Quarter Quadrangle 
    DORE                 Dore Raster File Format [Steve Hollasch] 
    DOS                  DOS 7 System Files. Win 95 Boot up in DOS. 
    DOS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOS                  General Network Driver File 
    DOS                  Text File/DOS Specification Info 
    DOS                  1st Reader External Command File 
    DOS                  Network Driver File 
    DOS                  NDIS Network Card Driver 
    DOSERRS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOT                  Corel Lines-Definition 
    DOT                  Ribbons Molecular Dot Surface 
    DOT                  Word Document Template (Microsoft Corporation)
    DOT                  Bert's Dinosaurs File 
    DOTHTML              Word HTML Document Template (Microsoft Corporation)
    DOTMENX              egis Encrypted DOTM (Word 2007) File (HiTRUST Inc.)
    DOTXENX              egis Encrypted DOTX (Word 2007) File (HiTRUST Inc.)
    DOUBLEWIDE           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOV                  Temporary File 
    DOVOICE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOW                  TAPCIS Download Command List 
    DOWN                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOWNLOAD             Mozilla Partial Download File (Mozilla)
    DOWNLOADHOST         MSN Explorer Download View (Microsoft Corporation)
    DOX                  Visual Basic User Document Binary Form File (Microsoft)
    DOX                  Word Text Document (Microsoft Corporation)
    DOX                  MultiMate Document 
    DOX                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOX                  Text File 
    DOX1                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOX2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOX3                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DOZ                  VENDINFO Description Out of Zip File 
    DP                   THOR Database Primary Data 
    DP                   Common Group 
    DP                   Portable Document; Common Ground Digital Paper 
    DP                   Daily Planner Calendar File 
    DP                   Text 
    DP                   DataPhile Data File 
    DP$                  Ntgraph Visual C Wizzard File 
    DPA                  Serif DrawPlus Animation 
    DPA                  Archive 
    DPB                  Dataplore Signals 
    DPB                  DataPilot Database (Susteen, Inc.)
    DPB                  ProWORX Nxt Descriptor Pointer File (Schneider Electric)
    DPB                  Fax Master File 
    DPC                  DiscPlay Collection File 
    DPC                  Delphi Package Collection File (Borland Software Corporation)
    DPC                  OrCAD Design Rule Check Output (Cadence Design Systems, Inc.)
    DPC                  Uefa Champions League Data 
    DPD                  Ovation Pro Document (David Pilling)
    DPD                  Document Processor DESIGNER Project File (Upload@24 IT Consulting GmbH)
    DPD                  OmniServer File 
    DPD                  ABC Programming Language Dyadic Predicate 
    DPD                  Dish Network DVD Recorder File (EchoStar Satellite L.L.C.)
    DPD                  Dekart Private Disk Encrypted Disk Image (Dekart)
    DPF                  Dynamic Process Format Database (Barco)
    DPF                  F-Secure Default Policy File (F-Secure)
    DPF                  ProWORX Nxt Descriptor Pointer (Schneider Electric)
    DPF                  Document Processor FLOW Project File (Upload@24 IT Consulting GmbH)
    DPF                  Dictaphone Walkabout Express 
    DPF                  COMPRESS Vessel Drafting Program File 
    DPF                  Dataplore Figure 
    DPF                  DeltaDOS Delta Patch File 
    DPF                  DanProgrammer Script (DanSoft Australia)
    DPF                  Speedy Printed Circuit Board Design Software (Mania Technologie)
    DPG                  Deleted program group; Windows Applications Manager 
    DPG                  DPGraph Mathematical Graph 
    DPGRAPH              DPGraph Mathematical Graph 
    DPI                  Pointline Bitmap 
    DPJ                  Delphi Project (Borland Software Corporation)
    DPJ                  DeskProto CAM Project File 
    DPK                  Lemon Interactive Starmageddon File 
    DPK                  Delphi Package File (Borland Software Corporation)
    DPK                  Deleted Package, Windows Applications Manager 
    DPL                  Digital Photo Librarian Database 
    DPL                  Dasher Playlist File (ChristmasCave)
    DPL                  Delphi Package Library (Borland Software Corporation)
    DPM                  Darkplaces Engine Game Movie Information (LordHavoc)
    DPM                  DataPilot RingTone (Susteen, Inc.)
    DPMI                 DOS Protected-Mode Interface Programming 
    DPO                  DECAdry Express Business Cards Business Card File (DECAdry)
    DPO                  Delphi Object Repository (Borland Software Corporation)
    DPP                  Serif DrawPlus Drawing 
    DPP                  Document Processor CORE Project File (Upload@24 IT Consulting GmbH)
    DPQ                  PCX format 
    DPR                  Desk Project 
    DPR                  C++ Default Project (Borland Software Corporation)
    DPR                  Wintidy File 
    DPR                  Dance Paranoia 
    DPS                  dpsVelocity Split Video File (Leitch Technology Corporation)
    DPS                  DPS Reality Video Editing File 
    DPS                  DivX XP Skin (DivXNetworks, Inc.)
    DPT                  Desktop DNA Template 
    DPT                  Publish-It! Publication File 
    DPT                  Better File Rename Droplet (Frank Reiff - PublicSpace.net)
    DPV                  PrintView Raw Text (DATAIR Employee Benefit Systems, Inc.)
    DPX                  Animation 
    DPX                  Digital Moving Picture Exchange Bitmap 
    DPX                  Kodak Cineon Raster Bitmap (Kodak)
    DPY                  PROGNOSIS Display Document File 
    DPY                  Realpolitik Saved Game 
    DP~                  BC31 Bin File 
    DQA                  itaz doQument File 
    DQB                  D-PAS Portfolio Manager File 
    DQC                  CP/M Disk Information File 
    DQD                  itaz Default doQument Database 
    DQL                  DataEase Database 
    DQY                  Excel ODBC Query File (Microsoft)
    DR                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DR$                  Modem Bitware Fax Disk2 File 
    DR9                  Directory File 
    DRA                  Dr. Abuse Saved File (Barres & Boronat)
    DRA                  Dragon Naturally Speaking 
    DRAFT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DRAG                 EdGCM Data File (The EdGCM Cooperative Project of Columbia University)
    DRAGON               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DRAGONSFUNK          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DRAT                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DRAW                 2D Graphic 
    DRAW                 Acorn Object-based Vector Graphic 
    DRAW                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DRB                  DrBeam 
    DRC                  OrCAD Schematic Capture Design Rules Check Report File (Cadence Design Systems, Inc.)
    DRC                  MicroSim PCBoard DRC Errors Report 
    DRC                  DriveCrypt Container File 
    DRC                  Delphi Compiled Resource File (Borland Software Corporation)
    DREAM                DreamScene Dream File (Microsoft Corporation)
    DREAM                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DREAMING             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DREAMS2              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DRES                 Diagnosis Results 
    DRF                  Photoline Drawing Filter 
    DRF                  Hummingbird DOCS Open, PowerDOCS, DM 
    DRF                  DrFrame 
    DRF                  3D Studio Scene (Autodesk, Inc.)
    DRF                  Nicolet DataViewer Distributed NRF Files (Nicolet Instrument Technologies, Inc.)
    DRG                  Dyno2000 Car Design (Motion Software)
    DRG                  VLBI and Tied Array Drudge Tasking Document 
    DRI                  Speedy Printed Circuit Board Design Software (Mania Technologie)
    DRIFT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DRINDX               Indexer Index of FIles (DotRational Software)
    DRIVE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DRIVER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DRIVERS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DRIVES               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DRJ                  Deraj Data File 
    DRL                  DocObject Resource Locator 
    DRM                  Yamaha Drum Kits (Yamaha Corporation of America)
    DRM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DRM                  Pro/ENGINEER (PTC)
    DRM                  MRS-802 Drum Kit Data (Zoom Corporation)
    DRM                  Cubase Drum File (Steinberg)
    DRML                 Protected Digital Content (Digital Rights Management Limited)
    DRN                  Darn for DOS 
    DRO                  D-Robots Robot 
    DRS                  Text 
    DRS                  Age of Empires II Archive (Microsoft Corporation)
    DRS                  BMP Bitmap 
    DRS                  DrawPerfect Ver. 1.0 Font 
    DRS                  WordPerfect Driver Resource (Corel Corporation)
    DRT                  Dispatch Route File 
    DRT                  DataPilot RingTone (Old Style) (Susteen, Inc.)
    DRT                  Director MX Shockwave Export (Macromedia, Inc.)
    DRT                  CADAM Export Format (Dassault Systemes)
    DRT                  egseing Direct Mode 
    DRT                  PCB Designer with PSpice Custom Derating File (OrCAD)
    DRT                  EPOS Data Recovery Tool Working File (EPOS Ltd.)
    DRT                  DirectRT Styles File (Empirisoft)
    DRT                  Windows Draw Template (Corel Corporation)
    DRU                  Wfm384s Demo File 
    DRUM                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DRUMS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DRV                  Program Overlay 
    DRV                  Device Driver 
    DRV                  Traverse PC Desktop Drawing (Traverse PC, Inc.)
    DRVR                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DRW                  Micrografx Vector Graphic 
    DRW                  Drawing 
    DRW                  Pro/ENGINEER Drawing (PTC)
    DRW                  Freelance Graphics Version 2 for DOS Drawing/Presentation (IBM)
    DRW                  Personal Designer (4Design) CAD Drawing Database (4D Graphics, Inc.)
    DRWHO                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DRX                  Photoshop Tutorial (Adobe)
    DRY                  XAIRON Organizer Diary File 
    DRY                  Net2000 Executable 
    DRZ                  Draz Paint 
    DR_                  Compressed DRV File of VFW 
    DS                   TWAIN Data Source 
    DS                   Dialog Studio 
    DS                   HP Officejet T Series All-in-One TWAIN File (Hewlett-Packard Development Company, L.P.)
    DS                   PhotoMax Drive 
    DS                   SQL Server Data Source (Microsoft Corporation)
    DS                   Chemisty Software 
    DS                   Dynamics Solver Problem Definition (Juan M. Aguirregabiria)
    DS                   Desktop KornShell (DtKsh) File 
    DS4                  Micrografx Designer Ver. 4 Graphic (iGrafx (Corel Corporation))
    DSA                  DasyTec DASYLab File 
    DSA                  PKCS7 Signature, DSA 
    DSASAVE              Dungeon Siege File (Microsoft)
    DSB                  Orchida Embroidery System (Orchida Soft)
    DSB                  Embroidery File 
    DSB                  DasyTec DASYLab File 
    DSB                  Rayman2 Play 
    DSB                  DiscSafe Backup (Engelmann Media GmbH)
    DSC                  Nikon Disk Identification File (Nikon Inc.)
    DSC                  PostScript Language Document Structuring Conventions Specification Version 3.0 
    DSC                  Description File 
    DSC                  Oracle Discard File 
    DSC                  Belief Network (Microsoft)
    DSC                  DPKG-source 
    DSC                  SoftImage Scene File 
    DSC                  MIME Tag Type 
    DSCE                 Diagnosis Scenario 
    DSCMD                Desktop Sidebar Command File (Idea2 s.c.)
    DSCRIP               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DSD                  DataShaper Database File 
    DSD                  Document Structure Definition File 
    DSDLL                Dungeon Siege Resource Library (Microsoft Corporation)
    DSES                 Diagnosis Session 
    DSF                  PC-TRUST Document Signer 
    DSF                  DataCAD Double Precision Symbol File with Layers (DATACAD LLC)
    DSF                  Vicon Kollector 2.0 CCTV Recording 
    DSF                  ProWORX Nxt Data Setup (Schneider Electric)
    DSF                  MS Belief Network Document Containing Infix Formula Declarations 
    DSF                  Borland C++ or Turbo Pascal Project Desktop Settings File 
    DSF                  Designer Vector Graphic (Corel)
    DSF                  DriveSurf picture 
    DSF                  Delusion/XTracker Digital Sample 
    DSG                  Doom Saved Game (Id Software, Inc.)
    DSH                  Dasher Grid File (ChristmasCave)
    DSH                  Dashboard PDA Wallpaper 
    DSI                  DESI Labeling System Telephone Label File (DESI Telephone Labels, Inc.)
    DSI                  Digital Sound Module Tracker Format 
    DSI                  R-Studio Hard Disk Scan Results File 
    DSK                  Generic Disk Image 
    DSK                  Extended CPCEMU Style Disk Image 
    DSK                  Standard CPCEMU Style Disk Image 
    DSK                  ZX Spectrum-Emulator 
    DSK                  WinImage Disk Image File 
    DSK                  SHARP MZ-series Emulator File 
    DSK                  NetWare Disk Driver (Novell)
    DSK                  Notes Design Elements (IBM)
    DSK                  Simple IDs Database (DSKE Inc.)
    DSK                  MSX DOS Disk Image 
    DSK                  Windows XP System Restore Backup (Microsoft)
    DSK                  Apple II or Macintosh Game ROM/Disk Image (Apple Computer, Inc.)
    DSK                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DSK                  Desk 
    DSK                  Delphi Project Desktop File (Borland Software Corporation)
    DSL                  DSSSL Style Sheet 
    DSL                  Product Compass Suite Product Launcher (Deadline Software)
    DSL                  Lingvo User Feature Dictionary 
    DSM                  Developer Studio Macro (Microsoft)
    DSM                  Delphi Symbol Module (Borland Software Corporation)
    DSM                  Digital Sound Module Tracker Format 
    DSM                  Media Player Classic DSM Streaming Container 
    DSM                  Dynamic Studio Music Module 
    DSMAP                Dungeon Siege File (Microsoft)
    DSN                  CD Stomper Pro 
    DSN                  SureThing Office Labeler Document Formerly DesignExpress. 
    DSN                  OrCAD Schematic Capture Schematic File (Cadence Design Systems, Inc.)
    DSN                  Object System Designer Design 
    DSN                  ODBC Data Source 
    DSN                  OLE DB Provider for ODBC (Microsoft)
    DSN6                 Ribbons FRODO Map 
    DSO                  Tribes 2 Compiled Script (Sierra)
    DSP                  Signature Display Parameters 
    DSP                  Dynamic Studio Professional Module 
    DSP                  Dr. Halo Graphic Screen Driver 
    DSP                  Dr. Halo Graphic Display Driver 
    DSP                  Norton Viewer DLL File 
    DSP                  Developer Studio Project (Microsoft)
    DSPACKAGE            Desktop Sidebar Skin/Panel (Idea2 s.c.)
    DSPARTY              Dungeon Siege File (Microsoft)
    DSPLUGIN             Desktop Sidebar Plug-in (Idea2 s.c.)
    DSQ                  DASYLab Worksheet Sequence 
    DSQ                  Corel QUERY File 
    DSQSAVE              Dungeon Siege File (Microsoft)
    DSR                  WordPerfect Driver (Corel)
    DSR                  Visual Basic Active Designer File (Microsoft)
    DSR                  Micrografx Designer Graphic 
    DSR                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DSRES                Dungeon Siege Resource File (Microsoft)
    DSS                  HEC-DSS Hydrologic Database (Hydrologic Engineering Center)
    DSS                  DCC Active Designer File 
    DSS                  Digital Speech Standard 
    DSS                  Digital Soup Digital Sound File 
    DSSAVE               Dungeon Siege Saved Game (Microsoft)
    DSSL                 DSSL SGML Style Sheet 
    DSSLIDESHOW          Desktop Sidebar Slideshow Panel (Idea2 s.c.)
    DSST                 ConsumerPoint File 
    DST                  Tajima Embroidery Machine Stitch File 
    DST                  ViPNet Products Encryption Key Set (Infotecs)
    DST                  PC-RDist Distribution File (Pyzzo)
    DST                  Micrografx Designer Template 
    DST                  Home Embroidery Format 
    DST                  CubicPlayer file 
    DST                  C++ Desktop Settings (Borland Software Corporation)
    DST                  MicroImages Priroda Lens Correction Photo-grammetric Distortion Values (radius/distortion pairs) Specification (MicroImages, Inc.)
    DST                  Orchida Embroidery System (Orchida Soft)
    DST                  WAIN (Scanner Spec) Data Source DLL 
    DSU                  ProWORX Nxt Data Setup (Schneider Electric)
    DSV                  SQL Server Data Source View (Microsoft Corporation)
    DSW                  C++ Desktop Settings (Borland Software Corporation)
    DSW                  Visual Studio Workspace (Microsoft)
    DSX                  Visual Basic Active Designer Binary File (Microsoft)
    DSX                  Micrografx Designer Graphic 
    DSY                  AJC Directory Synchronizer Project File (AJC Software)
    DSY                  PC Draft Symbol Library 
    DSZ                  Orchida Embroidery System (Orchida Soft)
    DSZ                  Embroidery File 
    DSZ                  DynaMaSZ Data/Parameter File (Mystic Fractal)
    DSZ                  Win Help Related File 
    DS_STORE             Macintosh OSX Folder Information (Apple Computer, Inc.)
    DT                   DAT Backup 
    DT                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DT                   Macintosh File Data Fork 
    DT0                  DTED Level 0 (National Geospatial-Intelligence Agency)
    DT1                  DTED Level 1 (National Geospatial-Intelligence Agency)
    DT2                  DTED Level 2 (National Geospatial-Intelligence Agency)
    DT5                  Brother's Keeper File 
    DTA                  Hidden & Dangerous Archive 
    DTA                  Stata Binary Saved Datasets 
    DTA                  World Bank's STARS Data 
    DTA                  Data 
    DTA                  EZ-Forms Data File in Standard EZ-Forms Format (EZ-Forms Company)
    DTAUS                Misc 
    DTAZV                Misc 
    DTB                  DesignTools PCB Design and Make 
    DTB                  Cybermedia Uninstaller 
    DTB                  After Dark 
    DTC                  DYTEC HEADQUARTER 
    DTC                  Windows Applog Journal 
    DTD                  ArcView UNIX Hyperhelp Supporting File (ESRI)
    DTD                  DesignTools 2D Design 
    DTD                  SGML Document Definition File 
    DTE                  Win Applog File 
    DTEA                 Diagnosis Template Archive 
    DTED                 Digital Terrain Elevation Data 
    DTF                  DanTemplate Template File (DanSoft Australia)
    DTF                  ICEM CAD Data 
    DTF                  pfs:Q&A Database 
    DTF                  ACDSee Use Database (ACD Systems Ltd.)
    DTF                  Symantec Q&A Relational Database Data File (Symantec Corporation)
    DTF                  Solid Edge Document (UGS PLM Solutions)
    DTF                  Exchange Header File (Microsoft)
    DTG                  Windows Applog File 
    DTH                  Windows Applog File 
    DTI                  Windows Applog File 
    DTI                  Delphi Design Time Information (Borland Software Corporation)
    DTJ                  Windows Applog File 
    DTL                  Hal/Supernova/Lunar/LunarPlus Screen Readers/Magnifiers 
    DTL                  Windows Applog File (Microsoft)
    DTL                  Pro/ENGINEER (PTC)
    DTM                  DigiTrekker Module 
    DTM                  Softshare Delta Translation Map 
    DTM                  Surpac Vision Data Repository (Surpac Minex Group)
    DTM                  Topocad (Chaos Systems AB)
    DTM                  DeFy Adlib Tracker 
    DTN                  TunnelCAD Import/Export File (IQSoft)
    DTP                  Timeworks Publisher Text Document 
    DTP                  SecurDesk! Desktop 
    DTP                  Text Document of Timeworks Publisher 3.x 
    DTP                  PageMagic - Publish-It! - Publisher3 Publication 
    DTP                  Page Express Desktop Publishing Document 
    DTP                  Publish-It! Publication 
    DTP                  Pressworks Template File 
    DTP                  ProComm Program Patch 
    DTQ                  Database Tools Query 
    DTR                  DiagTransfer Chess Diagram (alain Blaisot)
    DTR                  DTREG Project File (Phillip H. Sherrod)
    DTS                  Digital Surround Audio File Format 
    DTS                  Data Transformation Services (Microsoft)
    DTX                  E-Book File 
    DTX                  DocuTech Print6 Engine File 
    DTX                  LaTeX Source and Documentation Archive 
    DTX                  pfs:Q&A Index 
    DTX                  LithTech Game Model Texture File (Touchdown Entertainment, Inc.)
    DT_                  Macintosh Data File Fork 
    DT_                  Mac-ette 
    DUA                  Fifa 2001 Data Feart Legal Screeen 
    DUB                  Audio Utility Tuningwrench Tuning Compressed File 
    DUD                  DirectUpdate Dynamic DNS Service Description (William Levra-Juillet)
    DUEL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DUET                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DUF                  Beckman Instrument's Spectrophotometer Data Archive 
    DUM                  Ada ADA Tutor File 
    DUN                  Dunce File 
    DUN                  Dial-Up Networking Export File 
    DUNGEON              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DUP                  Duplicate Backup 
    DUPLICATEZAP         Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DUR                  DURILCA Compressed Archive 
    DUS                  Readiris Font Dictionary 
    DUSTCLOUD            Homeworld Dust Cloud Data (Sierra Entertainment, Inc.)
    DUT                  Dutch Text File 
    DV                   Digital Video File 
    DV                   DESQview Script (Symantec)
    DVA                  dpsVelocity Split Audio File (Leitch Technology Corporation)
    DVB                  AutoCAD VBA Source Code (Autodesk, Inc.)
    DVB                  DVBLine Document 
    DVB                  Visual Basic for Applications Macro Project (Microsoft)
    DVC                  1-2-3 (IBM)
    DVD                  DOS Device Driver File 
    DVD                  DVDMaestro Project File (Spruce Technologies)
    DVD                  Animation 
    DVDREGIONCSSFREE     DVD Region+CSS Free (Fengtao Software Inc.)
    DVDRIP               Ripped DVD File 
    DVDRIP-INFO          dvd::rip Technical Information 
    DVF                  DV Studio Camcorder Graphics File 
    DVF                  Sony Compressed Voice File (Sony Corporation of America)
    DVG                  GraphicWorks Vector Drawing 
    DVI                  TeX Device Independent Document 
    DVI                  GlossTeX Installation Instructions 
    DVI                  Action Media II Digital Video Interactive Movie 
    DVM                  DVM Movie File Format 
    DVO                  DESQview (Symantec)
    DVP                  DESQview Program Information (Symantec)
    DVP                  AutoCAD Device Parameter (Autodesk, Inc.)
    DVPRJ                Déjà Vu X Translation Project File (ATRIL Language Engineering S.L.)
    DVR                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DVR                  DVR-Studio Stream (Haenlein-Software)
    DVR                  Program Overlay 
    DVR                  Device Driver 
    DVR                  DESQview (Symantec)
    DVR-MS               Media Center Recorded Television File (Microsoft)
    DVSD                 DVMaster Pro Video 
    DVT                  DepoView Digital Video Transcript (inData Corporation)
    DVU                  DVBControl 
    DVU                  Documat Adaptive Binary Optimization Format File (MatrixView Pte. Ltd.)
    DVW                  Digital Voice Recorder Voice File (Dektron)
    DVX                  DivX Movie (DivXNetworks, Inc.)
    DW                   David Whitaker Format Music File 
    DW1                  Deathwatch WebCrawler Encrypted Log File (Deathwatch Development Group)
    DW2                  Deathwatch WebCrawler Crawl Log (Deathwatch Development Group)
    DW2                  DesignCAD Drawing File 
    DW3                  Deathwatch WebCrawler Temporary Data Storage (Deathwatch Development Group)
    DW4                  Visio/DisplayWrite 4 Text 
    DWA                  Project Dogwaffle Animation (TheBest3D.com)
    DWARF                DWARF 
    DWB                  Coryphaeus Software Designer Workbench 
    DWC                  Compressed Archive 
    DWC                  NetOp Connection File 
    DWD                  DiamondWare Digitized File (DiamondWare, Ltd.)
    DWD                  Davka Writer File 
    DWE                  DataWarehouse Explorer (CNS International)
    DWE                  Creator Simulator (Tufts University)
    DWF                  Autodesk Design Web Format (Autodesk, Inc.)
    DWG                  Older Generic CADD Drawing Format 
    DWG                  Drafix Drawing 
    DWG                  2D Graphic 
    DWG                  AutoCAD Drawing Database (Autodesk, Inc.)
    DWG                  BravoDRAFT, Detailer 
    DWG                  Pro/ENGINEER Drawing (PTC)
    DWI                  Dance With Intensity (SimWolf)
    DWI                  Dependency Walker Image (Steve P. Miller)
    DWJ                  Dynamic Submission 2000 V6 File 
    DWK                  Locked DWG File 
    DWK                  DADiSP DADiSP Worksheet File (DSP Development Corporation)
    DWL                  AutoCAD Drawing Database File Locking Information (Autodesk, Inc.)
    DWL                  Textur Editor Compressed File 
    DWM                  Dwstk File 
    DWM                  DESQview (Symantec)
    DWN                  Experimental File Format 
    DWN                  Cisco Download File 
    DWN                  Firmware Update File 
    DWP                  DeScribe Document File 
    DWP                  Dweep Game Levels 
    DWP                  SharePoint Portal Server Dashboard Web Part File (Microsoft Corporation)
    DWP                  Darwin Pond 
    DWR                  WordEpress2.0 File 
    DWR                  NetOp Recording File 
    DWS                  Dreamweaver Site Definition File (Adobe Systems Incorporated)
    DWS                  NetOp Script File 
    DWS                  Creator Simulator Waveform (Tufts University)
    DWS                  Dyadic Workspace File 
    DWS                  AutoCAD Standarization (Autodesk, Inc.)
    DWT                  GoLive Web Template (Adobe)
    DWT                  SharePoint Designer (Microsoft Corporation)
    DWT                  FrontPage Dynamic Web Template (Microsoft)
    DWT                  AutoCAD Template/Prototype (Autodesk, Inc.)
    DWT                  Demon's World Game Texture File 
    DWT                  Dreamweaver Template File (Adobe Systems Incorporated)
    DWV                  WAV 
    DWX                  CADLock Locked AutoCAD Drawing (CADLock, Inc.)
    DWY                  Dwyco Video Clip 
    DWZ                  Compressed AutoCAD Drawing File 
    DWZ                  Designer Projects Template File 
    DX                   Auto-trol Raster Bitmap 
    DX                   Document Imaging File 
    DX                   MultiMate Word Processor Document 
    DX                   THOR Database Cross-reference Data 
    DX                   J-CAMP Spectroscopic Data Exchange Format 
    DX                   Deus Ex Map File (Eidos Interactive Ltd.)
    DX                   JCAMP Chemical Spectroscopic Data Exchange Format (MDL Information Systems, Inc.)
    DX                   Data Explorer Document 
    DX                   Digital Electric Corporation (DEC) Data Exchange File 
    DXB                  Duxbury Braille Translator Braille File (Duxbury Systems, Inc.)
    DXB                  AutoCAD Drawing Exchange Format (Autodesk, Inc.)
    DXB                  Drawing Interchange Binary 
    DXB                  Database 
    DXD                  Diogenes Extended Document (Dimensione CAD)
    DXE                  Amiga Emulator 
    DXF                  XMap Exported Draw File (Delorme)
    DXF                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DXF                  Image PSP, Fh7 
    DXF                  2D Graphic 
    DXF                  AutoCAD Drawing Interchange Format (Autodesk, Inc.)
    DXF                  3D Graphic 
    DXF                  Micrografx Picture Publisher Special Effect 
    DXG                  QuickTac Duxbury Graphic File (Duxbury Systems, Inc.)
    DXG                  Duxbury Braille Translator File (Duxbury Systems, Inc.)
    DXI                  PayWindow Payroll Database (ZPAY Payroll Systems, Inc.)
    DXL                  Duxbury Braille Translator Label (Braille) (Duxbury Systems, Inc.)
    DXL                  DOORS Extension Language 
    DXL                  Domino XML File (IBM)
    DXL                  Dapyx MP3 Explorer 
    DXM                  Music File 
    DXMAX                Oozic File 
    DXN                  Fujitsu dexNet Fax Document 
    DXP                  Piles'O'Tiles Tile Image File (F1 Software)
    DXP                  Duxbury Braille Translator Duxbury Print File (Duxbury Systems, Inc.)
    DXP                  Dynamic XML Page (Neolectric)
    DXPACK               DesktopX Object Package (Stardock Corporation, Inc.)
    DXPACKPRO            DesktopX Pro Document Package (Stardock Corporation, Inc.)
    DXR                  Acrobat (Adobe)
    DXR                  AEC DesignWare Model Parameter File 
    DXR                  Dependable Strengths Administrator Resources 
    DXR                  Green Building Advisor File 
    DXR                  Vegas Jackpot Gold Game File 
    DXR                  Macromedia Director Protected Movie File (Macromedia)
    DXS                  Deus Ex Saved Game 
    DXSTUDIO             DX Studio Document (Worldweaver Ltd.)
    DXTHEME              DesktopX Theme (Stardock Corporation, Inc.)
    DXTHEMEPRO           DesktopX Pro Theme (Stardock Corporation, Inc.)
    DXU                  DivX File (DivXNetworks, Inc.)
    DXV                  DocXchanger Vector File (Condrey Corporation)
    DXX                  AutoCAD Drawing Interchange Attribute File (Autodesk, Inc.)
    DY22                 Dyaxis Sound Format 
    DY44                 Dyaxis Sound Format 
    DYA                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DYAINSTS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DYC                  ICUII Videochat File 
    DYL                  EPOC16 Dynamic Link Library 
    DYLAN                Dylan Source File 
    DYN                  1-2-3 (IBM)
    DYN                  Dyno2000 Engine Simulation (Motion Software)
    DYNAMO               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    DYS                  SwordSearcher Daily Devotional Data File (StudyLamp Software)
    DZ                   Firefly Data/Parameter File (Mystic Fractal)
    DZ                   Dzip Compressed File 
    DZS                  Character File 
    DZS                  Download Accelerator Skin (SpeedBit Ltd.)
    D_                   Seqmax Seqprest Compressed Audio File 
    D_1                  Polytext Ma'archot Meida Co. S"TaM 
    D_3                  Polytext Ma'archot Meida Co. S"TaM 
    D_T                  Eru/erd File 
    D~L                  Creative DLL Copy 
    D~V                  Windows 3.x System File 
    E                    Euphoria Include File (Rapid Deployment Software)
    E                    EiffelStudio Source File (Eiffel Software Inc.)
    E                    Epsilon Editor EEL Macro Language 
    E                    Specman Testbench E-program Source File 
    E                    Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    E00                  ArcInfo Coverage Export (Environmental Systems Research Institute, Inc. (ESRI))
    E01                  Encase Forensic Image File (Guidance Software, Inc.)
    E2                   thinkdesign CAD Design 
    E2C                  NJStar Chinese Word Processor Dictionary File Index (NJStar Software Corp.)
    E2P                  PonyProg Device File (Lanconelli Open Systems)
    E3                   thinkdesign CAD Design (think3 Inc.)
    E32                  Inno Setup 1.3 File 
    E3D                  Macromedia Extreme3D Object 
    E3D                  Instant3D Project 
    E48                  Emu48 HP48 Emulator File 
    E4C                  e4Clicks Project Estimator Import File (4Clicks Solutions, LLC)
    E78                  IBM 3270 Terminal Emulator Screen Layout Definition 
    E99                  Steuer99 Daten File 
    EA                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EA3                  Fifa 2001 Environment Data 
    EAC                  EmEditor Auto Completion File 
    EAF                  FRAGSTATS*ARC File 
    EAF                  Aframe 
    EAF                  MicroEmacs Abbreviation File Format 
    EAL                  Enfocus Action List 
    EAM                  France98 Football Game Audio File 
    EAP                  Enterprise Architect Project (Sparx Systems Pty Ltd.)
    EAR                  Java Enterprise Application Packaging Unit 
    EAR                  ETI Camcorder Pro Audio File (Eyemail Technology, Inc.)
    EARTH                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EAS                  OS/2 Extended File Attributes 
    EAS                  EAS 2004 Applicant Reference Database (UCAS)
    EAS                  Elite Visual Basic API Spy 
    EASA                 EASA Application Development File (AEA Technology plc)
    EASAP                EASA Archive (AEA Technology plc)
    EASM                 eDrawings File 
    EATF                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EAZ                  Express Assist File 
    EBA                  Mobile Phone Data Manager 
    EBAK                 Ideal Summary Sheets Encrypted Backup (Conquer Internet Ltd.)
    EBB                  Empower Business Bulletin (Metapraxis Ltd.)
    EBD                  Empower Business Dashboard (Metapraxis Ltd.)
    EBF                  Pocket PC WindowsCE Form File (Microsoft)
    EBF                  EARS Definition Backup 
    EBJ                  Geoworks Error Checking Object File 
    EBK                  Email Saver Xe Backup File (Amic Games)
    EBK                  EARS Database Backup 
    EBK                  Embiid Reader eBook (Embiid Inc.)
    EBM                  Attachmate Extra! Macro 
    EBO                  Reader eBook Format (Microsoft Corporation)
    EBP                  Jes-Soft Sports Program Play Description File (Jes-Soft)
    EBP                  Pocket PC WindowsCE Project File (Microsoft)
    EBP                  Relic Entertainment Game File (Relic Entertainment Inc.)
    EBS                  EyeBatch Command Script (Atalasoft, Inc.)
    EBS                  E-Scripter Script (CMS)
    EBS                  Rational Rose 98 Script Source (IBM)
    EBS                  WindowsXP Scanner File 
    EBU                  Exact Globe 2000/2003 Database Backup (Exact Software)
    EBUILD               Gentoo Linux Ebuild Application File (Gentoo Foundation, Inc.)
    EBX                  Electronic Book Exchange eBook (Electronic Book Exchange (EBX) Working Group)
    EBX                  Rational Rose 98 Compiled Script (IBM)
    EC                   Preprocessed GOC Source Code 
    EC3                  Print CD CD Label Image (Epson America, Inc.)
    ECA                  electroCAD File 
    ECC                  Essential Taceo Crypto Container (Essential Security Software, Inc.)
    ECC                  Ecchi Role-playing Format 
    ECD                  Questionnaire Specification Language File 
    ECD                  LabelMaker CD/DVD Label File (Memorex Products, Inc.)
    ECF                  Outlook Add-in File (Microsoft)
    ECF                  Micrografx Media Manager 
    ECF                  Eventcorder Recorded Events and Clickviews (CMS)
    ECF                  WinFax Office Add-in File (Symantec)
    ECG                  Preprocessed GOC Source Code 
    ECG                  ECG Viewer (Nihon Kohden Corporation)
    ECHO                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ECHO6                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ECHOING              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ECHR                 Explorations RPG System Character/Map System Sprites (Tyrone W. Lee)
    ECL                  Eclipse (Active Media)
    ECM                  Cmpro Examples File 
    ECO                  NetManage ECCO File 
    ECP                  eCourse Planner eCourse Template (ICS Corporation)
    ECR                  Ecrypt E-mail File (Email Connection)
    ECRAN                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ECS                  Sony Ericsson Phone Backup File 
    ECS                  Encrypted Compressed GIS Software Geographic Shape File 
    ECT                  Visual C++ Exception Console Template (Microsoft Corporation)
    ECW                  EclipseCrossword Crossword Puzzle 
    ECW                  Ensoniq Waveset Format (E-MU Systems)
    ECW                  Enhanced Compressed Wavelet 
    ECW2002PUZZLE        EclipseCrossword Puzzle 
    ECW2002WORDS         EclipseCrossword Word List 
    ED                   LithTech Game Level File (Touchdown Entertainment, Inc.)
    ED                   Tmx Editor File 
    ED                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ED                   EasyDraw CAD File 
    ED2K                 eDonkey Download Link File (MetaMachine)
    ED5                  EDMICS Graphics File 
    ED6                  EDMICS Graphics File 
    EDA                  Ensoniq ASR Disk Image (E-MU Systems)
    EDB                  Euphoria Database System (EDS) Database File (Rapid Deployment Software)
    EDB                  EzyMailManPro (Welshware)
    EDB                  ACT! E-mail Data File (Best Software CRM Division)
    EDB                  ROOTS3 Geneological Data 
    EDB                  Business Assessment Manager (E2S n.v.)
    EDB                  EstImage Database (Stelvio Inc.)
    EDB                  Exchange Server Property Store (Microsoft)
    EDC                  Kryptel (light) Decoder/Encoder 
    EDD                  FrameMaker Element Definition Document (Adobe)
    EDE                  Ensoniq EPS Disk Image (E-MU Systems)
    EDF                  Drumbeat Component 
    EDF                  Workshare Synergy Document Change Management 
    EDG                  EdG Command File 
    EDG                  EDGE Diagrammer Diagram (Pacestar Software)
    EDH                  xcontur XEDH Electron-Density Histogram 
    EDH                  EditHelp Source File 
    EDI                  EDI Interchange File (Budget EDI)
    EDI                  EDI Engine EDI File (123 EDI, Inc.)
    EDIT                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EDIT0                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EDITHELP             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EDITOR               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EDK                  OS/2 Saved SKF Format File (IBM)
    EDK                  Ensoniq KT Disk Image (E-MU Systems)
    EDL                  Edit Decision List 
    EDL                  ASAP Entity Definition Language File (Hewlett-Packard Development Company, L.P.)
    EDM                  CAD Contouring Parameter Data File 
    EDM                  Setters III 3 Map File 
    EDM                  Extinction Corrected Differential Photometry 
    EDM                  Eclipse Data Module (Active Media)
    EDML                 Dreamweaver MX Extension File 
    EDN                  Acrobat Document (Adobe)
    EDN                  EDIF Netlist File 
    EDP                  Attachmate Extra! Session 
    EDPI                 ExamDiff Pro Session File (PrestoSoft)
    EDQ                  Ensoniq SQ1/SQ2/KS32 Disk Image (E-MU Systems)
    EDR                  GROMACS Portable Energy File 
    EDR                  Embird Plus Color Paletter (Embird)
    EDRW                 eDrawings File 
    EDS                  Ensoniq SQ80 Disk Image (E-MU Systems)
    EDT                  External Editor Definition 
    EDT                  VAX EDT Editor Default Settings 
    EDV                  Ensoniq VFX-SD Disk Image (E-MU Systems)
    EDX                  Editor Dictionary File (Serenity Software)
    EEA                  Attachment Decoder Encrypted Attachment (Scanlon Associates)
    EEB                  EaseBook File 
    EEB                  WordPerfect Equation Editor Button Bar (Corel Corporation)
    EEE                  3eee Triple Encryption Encrypted File (Fasttrack Ltd.)
    EEE                  EEEWin Archive (Remia s.r.o.)
    EEG                  SleepScan and Ceegraph EEG Data (Bio-logic Systems Corp.)
    EEP                  Evidence Eliminator (Robin Hood Software Ltd.)
    EEP                  EEPROM Data File 
    EER                  Lahey Fortran Execution Error Messages 
    EES                  Empire Earth Scenario File 
    EFA                  Ensoniq ASR File (E-MU Systems)
    EFA                  Ecrypt 2005 E-mail File (Email Connection)
    EFC                  Technics Sx KN 6000 Keyboard Effect Memory File 
    EFC                  FAIRCOPY Encrypted Fair Copy (FAIRCOPY)
    EFC                  EFCoins File 
    EFD                  EARS Filter Definition 
    EFDEMO               Star Trek Voyager Elite Force Demo File 
    EFE                  Ensoniq EPS-Family Instrument (E-MU Systems)
    EFF                  EverQuest Effects File 
    EFF                  Infinity Game Engine Effects File (BioWare Corp.)
    EFFECT               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EFG                  Gambit Extensive Form Representation (The Gambit Project)
    EFI                  Extensible Firmware Interface Firmware File (Intel)
    EFK                  Ensoniq KT File (E-MU Systems)
    EFP                  Exchange Forms Designer Template (Microsoft)
    EFQ                  Ensoniq SQ1/SQ2/KS32 File (E-MU Systems)
    EFS                  FlowChart; EasyFlow 6.x-8.x 
    EFS                  Ensoniq SQ80 File (E-MU Systems)
    EFT                  Finale Enigma Tansportable File (MakeMusic!)
    EFT                  ChiWriter High Res Screen Characters 
    EFT                  e-Form Form Template (Hong Kong SAR Government Communications and Technology Branch)
    EFV                  Ensoniq VFX-SD File (E-MU Systems)
    EFX                  eFax Fax Document (j2 Global Communications, Inc.)
    EG2                  eGatherer Collected System Configuration Information (IBM)
    EGA                  Ventura EGA Screen Characters 
    EGA                  Enhanced Graphics Adapter Graphics Data 
    EGA                  EGA Resolution BMP Bitmap 
    EGD                  GIS Software Encrypted Grid Shape File 
    EGG                  EGG Solution 360RealTour 
    EGG                  Ducks Add-on/Level File (Hungry Software)
    EGG                  Wer Wird Millionaer Data Screens File 
    EGG                  Explosion Graphics Generator Script (Shawn Hargreaves; maintained by Vincent Penquerc'h)
    EGISENC              egis Encrypted File (HiTRUST Inc.)
    EGISENX              egis Encrypted File (HiTRUST Inc.)
    EGL                  EagleCheck Encrypted Database Reference File (Washington State Gambling Commission)
    EGP                  Egrid32 Form Properties Template (Sweetheartgames.com)
    EGP                  Easy Grade Pro Data 
    EGR                  Egrid32 Form (Sweetheartgames.com)
    EGRP                 Explorations RPG System MapGroup Object (Tyrone W. Lee)
    EGS                  GIS Software Encrypted Grid Shape File 
    EGWARSAVE            War! Age of Imperialism Save File (Donohoe Digital LLC)
    EHP                  HelpLogic Project (Electric Butterfly, Inc.)
    EHTML                Enhanced HTML 
    EI                   Electric Image 
    EI                   EARS Component File 
    EID                  IBM ViaVoice Vocabulary File 
    EID                  Faena MyID Encrypted File 
    EIDI                 Electric Image EIDI File 
    EIF                  Eroiica Graphic 
    EIG                  Easy Index Generator Project (Easy HTools Corporation)
    EIKULE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EIN                  Skyplot Settings File (Frank P. Thielen Software Design)
    EIO                  EIOffice Document Binder (Evermore Software)
    EISLEY               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EJP                  Elite Japan Crossword Puzzle (EliteWare)
    EK5                  Simrad EK500 Scientific Echosounder Data 
    EK6                  Simrad EK60 Scientific Echosounder Data 
    EKA                  Eureka Internal Data File (Borland Software Corporation)
    EKM                  EXP: The Scientific Word Processor Macro 
    EKS                  Empower Site File (Metapraxis Ltd.)
    EL                   eMacs Lisp Source Code 
    EL1                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EL2                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EL3                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EL4                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ELA                  TradeStation EasyLanguage Archive (TradeStation Securities, Inc.)
    ELC                  eMacs Lisp Byte-compiled Source Code 
    ELD                  EARS Label Definition 
    ELD                  TradeStation EasyLanguage Document (TradeStation Securities, Inc.)
    ELECBASS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ELECGTR              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ELECPIANO            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ELEMENT              Ampoliros Platform Component Type Handler (Solarix)
    ELEMS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ELEPHANT             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ELETTERHEAD          Electronic Letterhead (Basildon Bond)
    ELF                  Black Belt Systems Extended Layer File 
    ELF                  Electronic Application Form File 
    ELF                  Tasklink for Windows Machine Control Executable and Linking Format 
    ELG                  EARS Log 
    ELG                  EnABLE Activity Log (Energy Scitech Ltd.)
    ELH                  Electronic Application Help File 
    ELI                  ELI Compressed File Archive 
    ELI                  Encyclopedia Article List 
    ELISE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ELL                  ATI Radeon Video Driver 
    ELM                  FrontPage Theme-Pack File (Microsoft)
    ELOG                 McAffee Firewall Log 
    ELP                  HotRecorder Proprietary Audio Format (Porto Ranelli, S.A.)
    ELS                  EnSight 3D Document (Scenario) (Computational Engineering International (CEI))
    ELS                  TradeStation EasyLanguage Storage (TradeStation Securities, Inc.)
    ELSE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ELSEWHERE            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ELT                  Event List Text 
    ELT                  Prosa Event List 
    ELX                  Design Pro Tools Form File (Elixir Technologies)
    ELZ                  Electronic Application Form Answer File 
    EM$                  Modem Bitware Fax Disk2 File 
    EM1                  Emax I Bank File 
    EM2                  Emax II Bank File 
    EMAIL                Outlook Express Mail Message (Microsoft)
    EMAT                 ANSYS Element Matrix 
    EMB                  Orchida Embroidery System Embroidery Pattern (Orchida Soft)
    EMB                  Everest Embedded Bank File 
    EMB                  EMBL Nucleotide Format 
    EMB                  ABT Extended Module 
    EMBL                 IEMBL Nucleotide Format 
    EMD                  Embroidery File 
    EMD                  ABT EMD Module/Song Format 
    EMD                  Micrografx System4 Media Manager File 
    EMD                  ExpressThermo Log (Eclo, Lda.)
    EMD                  POP Peeper E-mail Database (Mortal Universe Software Entertainment)
    EMDL                 Softimage XSI Exported 3D Model (Softimage)
    EMF                  Extended (Enhanced) Windows Metafile Format 
    EMF                  EZClaim HCFA-1500 Graphic Image (EZClaim)
    EMITTER              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EMK                  EMK Compress Archive (Elex Media)
    EML                  QuickMail Pro 
    EML                  Outlook Express Electronic Mail (Microsoft Corporation)
    EMLX                 OS X Tiger Mail E-mail File (Apple Computer, Inc.)
    EMM                  EARS Mail Merge Definition 
    EMN                  Pro/ENGINEER (PTC)
    EMO                  Messenger Emoticon (Smiley) (Microsoft Corporation)
    EMP                  eMusic File Format (eMusic.com, Inc.)
    EMP                  Medlin Accounting Employee Data File (Medlin Accounting)
    EMP                  Pro/ENGINEER (PTC)
    EMR                  Emrite 
    EMS                  Revolution EMSA 1.0 Spectrum File (4pi Analysis, Inc.)
    EMS                  PC Tools Enhanced Menu System Config 
    EMS                  eMachineShop CAD Design File (eMachineShop)
    EMS                  YoshimuraEMS Motorcycle Fuel Injection 
    EMT                  EverQuest (Sony)
    EMU                  Emulation 
    EMU                  BITCOM Terminal Emulation Data 
    EMUL1                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EMUL2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EMUL3                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EMUL4                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EMUL5                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EMULECOLLECTION      eMule Plus Collection (eMule Team)
    EMV                  LabView Configuration File (EuroMeV)
    EMX                  Emax System File 
    EMX                  Enigmacross Puzzle File 
    EMX                  MS-DOS Extender file 
    EMY                  Eericsson Ringtone 
    EMZ                  Windows Compressed Enhanced Metafile (Microsoft Corporation)
    EN                   Stratum Template File 
    EN$                  Cubase Compressed File (Steinberg)
    ENB                  EverNote Database (EverNote Corp.)
    ENB                  Census of Employment and Wages Flat Data File (Bureau of Labor Statistics)
    ENB                  eNotebook Saved Notebook (eNotebook)
    ENB                  eNewsletter Manager Database (e-undertown s.r.l.)
    ENC                  Census of Employment and Wages Flat Data File (Bureau of Labor Statistics)
    ENC                  Sniffer (DOS) Capture File (Network Associates Technology, Inc.)
    ENC                  Dungeon Related File 
    ENC                  Knowledgeware Encyclopedia 
    ENC                  BlackIce Evidence Log File 
    ENC                  Video File 
    ENC                  UUENCODE Encoded File 
    ENC                  My Personal Programmer Encrypted Distributed Project (M:Pact Technologies)
    ENC                  Media Safe Encrypted Data (Nova Epoc Systems)
    ENC                  Encore Musical Notation File (GVOX)
    END                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    END                  Corel Draw Arrow Definition 
    END                  Census of Employment and Wages Flat Data File (Bureau of Labor Statistics)
    ENE                  GROMACS Binary Energy File 
    ENEM                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ENEMY                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ENF                  RightClick-Encrypt File 
    ENF                  SmartScore Score (MUSITEK Corporation)
    ENF                  EndNote Filter File 
    ENFF                 Extended Neutral File Format 
    ENG                  English Documentation 
    ENG                  Train Simulator Engine File (Microsoft Corporation)
    ENG                  DreamMaker Program Source Code in English (Intelligent Systems)
    ENG                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ENG                  Sprint Dictionary Engine File 
    ENG                  Personal Paint English Language User Interface File (Amiga)
    ENG                  RealFlight Radio Control Airplane Description (Knife Edge Software)
    ENG                  Dictionary 
    ENG                  EnerGrahics Chart Graphics File 
    ENG                  MATLAB Engine Routines (The MathWorks, Inc.)
    ENGLISH              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ENH                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ENH                  CRiSP Harvest File 
    ENIGMA               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ENJOYTHESILENCE      Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ENL                  EndNote Library File 
    ENM                  eNewsletter Manager Message (e-undertown s.r.l.)
    ENOCHIAN             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ENOLAMIX             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ENOUGH               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ENP                  Entelligence Media Security Encrypted File/Folder (Entrust)
    ENR                  Enroute Toolath 
    ENS                  Envox Studio Script (Envox Group)
    ENS                  REAKTOR Ensemble (Native Instruments)
    ENS                  EndNote Style File (ISI ResearchSoft)
    ENT                  Brookhaven PDB Molecule File 
    ENT                  HyperChem Molecule File (Hypercube, Inc.)
    ENT                  egseing Engineering HK Data 
    ENT                  Entrust Entelligence Secured File 
    ENT                  SGML Entities, character mapping 
    ENTERTAINER          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ENTRTAIN             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ENTRY                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ENTRY1               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ENU                  ATI Radeon Video Driver 
    ENU                  Win Ace204 File 
    ENV                  Novell ENVoy Data 
    ENV                  HP-95 Environment; like CNF 
    ENV                  TeamQuest View Environment (TeamQuest Corporation)
    ENV                  Acrobat Spelling File (Adobe)
    ENV                  Envelope or Environment 
    ENV                  Train Simulator Environment File (Microsoft Corporation)
    ENV                  Motocross Madness 2 Track (Microsoft)
    ENV                  Sierra Print Artist Envelope 
    ENV                  Creatacard Envelope Project (Broderbund)
    ENV                  The Golem Project Environment File 
    ENV                  X-Plane Scenery File (Laminar Research)
    ENV                  WOPR Enveloper Macro 
    ENV                  WordPerfect Environment (Corel Corporation)
    ENVELOPE             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ENW                  Sierra LandDesigner 3D Terrain Modeling 
    ENX                  eNeighborhoods Document (eNeighborhoods, Inc.)
    ENX                  Rational XDE (IBM)
    ENZ                  EndNote Connection File (The Thomson Corporation)
    EOBJ                 Explorations RPG System Objects (Tyrone W. Lee)
    EOBJ                 Geoworks Object Code 
    EOMODELD             WebObjects Data Modeler (Apple Computer, Inc.)
    EOMODELED            WebObjects Data Model (Apple Computer, Inc.)
    EON                  EON Studio File 
    EOP                  EON Studio File 
    EOT                  WEFT Embedded OpenType File (Microsoft)
    EOX                  EON Studio File 
    EOZ                  EON Studio File 
    EP                   Bonas Jaquard EP Weave Format File 
    EP2                  EclipsePackager Package 
    EPA                  Award BIOS Logo 
    EPC                  Jill Game File 
    EPD                  Express Publisher Publication File 
    EPD                  EARS Printer Definition 
    EPDF                 Encapsulated Portable Document Format 
    EPE                  EON Studio Personal Edition File 
    EPF                  Entrust Entelligence Profile (Entrust Inc.)
    EPF                  WebSphere Development Studio Client Profile (IBM)
    EPF                  EdgeCAM Part Data (Pathtrace Engineering Systems)
    EPH                  NOAA Post-fit GPS Orbit Data 
    EPH                  GPS Pathfinder Office Ephemeris File (Trinble Navigation Limited)
    EPHTML               Enhanced Perl-parsed HTML File 
    EPI                  Encapsulated PostScript Interchange Format (Adobe)
    EPI                  Express Publisher Encapsulated PostScript Text and graphics. 
    EPI                  EclipsePackager Invoice 
    EPJ                  RapidSQL SCCS Project 
    EPJ                  Java-clients File 
    EPL                  Tallyman E-commerce Software 
    EPL                  Encirq \PL Programming Language Source File 
    EPM                  Destiny MPE Encoder Encrypted Portable Media File (Destiny Media Technologies Inc.)
    EPP                  EditPad Pro Project (JGsoft - Just Great Software)
    EPR                  Eiffel Project Repository 
    EPR                  Z88 Disk/ROM Image 
    EPRT                 eDrawings File 
    EPS                  Ventura Printer Font (Corel)
    EPS                  Encapsulated PostScript 
    EPS2                 Level II Encapsulated PostScript (Adobe)
    EPS3                 Level III Encapsulated PostScript (Adobe)
    EPSF                 Encapsulated PostScript (Adobe)
    EPSI                 Encapsulated PostScript Interchange Format (Adobe)
    EPSON                Epson Printer Graphics File 
    EPT                  Encapsulated PostScript Interchange TIFF Preview (Adobe)
    EPT                  Elpoint Presentation File 
    EPT                  WITE32 File 
    EPT                  Clarion for Windows Embed Points File (SoftVelocity)
    EPZ                  EON Studio Personal Edition File 
    EQD                  EquineMax Data File (Cattlesoft, Inc.)
    EQF                  Winamp2/Winamp3 Equalizer Settings File 
    EQL                  DART Pro 98 Fabric Equalization Presets 
    EQN                  HP-95LX Solver Equation File 
    EQN                  KaleidaGraph Macro File (Synergy Software)
    EQN                  Equation 
    EQN                  WordPerfect Equation (Corel Corporation)
    EQP                  BPM Studio Equalizer Presets (ALCATech)
    EQT                  ReliaSoft Weibull++ 6 
    EQU                  Assembly Language Equates 
    EQU                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EQUATES              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EQUILIBFONT          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EQUS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EQW                  SPEFO Stellar Spectra Analysis File 
    EQX                  Goldwave 402 File 
    ER                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ER                   AOL 3.0 Organizer (America Online, Inc.)
    ER1                  Visible Advantage 
    ER1                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ER1                  AllFusion ERwin Data (Computer Associates International, Inc.)
    ER2                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ER3                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ERASURE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ERC                  Ercato Engine Ercaton (Living Pages Research GmbH.)
    ERD                  Eclipse Resource Database (Active Media)
    ERD                  Prosa Entity Relationship Diagram Graphic File 
    ERD                  Entity Relationship Diagram 
    ERF                  Neverwinter Nights Toolset Exported Resource 
    ERF                  Eclipse Resource File (Active Media)
    ERF                  SGML Visual DTD Entity Reference File 
    ERG                  Arranger Audio File 
    ERI                  AllFusion ERwin Data (Computer Associates International, Inc.)
    ERI                  Entis Rasterized Image 
    ERIC                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ERM                  Bitmap Graphic 
    ERM                  Entity Relationship Diagram Model 
    ERM                  Prosa Entity Relationship Diagram Model File 
    ERP                  British Economics & Social Research Council Electronic Form 
    ERP                  NOAA Post-fit GPS Earth Rotation Parameters 
    ERR                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ERR                  Error Log 
    ERROR                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ERRORS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ERS                  Pro/ENGINEER Temporary File (PTC)
    ERS                  Eraser 
    ERS                  EARS Control File 
    ERS                  Earth Resource Mapping Satellite Image Header 
    ERS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ERX                  ERWin File 
    ER_                  Winhelp Compressed File 
    ES                   Serious Sam Entity Class (Croteam Ltd.)
    ES                   EasySIGN Drawing Sheet 
    ESA                  Enveloc Secure Archive (Enveloc, Inc.)
    ESA                  ExeemSkin Archive (Swarm Systems Inc.)
    ESA                  ExpertScan Survey Report (AutoData Systems)
    ESD                  ExpertScan Survey Document (AutoData Systems)
    ESF                  VisualAge Generator External Source Format (ESF) File (IBM)
    ESF                  Emblaze Audio File 
    ESFORM               EasySurveyor Form File (Pepto Systems AB)
    ESG                  Effect Size Generator Data File (Psytek Ltd.)
    ESH                  Extended Shell Batch File 
    ESH                  Easy Splitter Split File 
    ESI                  Diskeeper Disk Defragmenter 
    ESI                  eStore Image File (ePC)
    ESI                  Esri Plot File 
    ESJ                  eCabinet Job (Thermwood Corporation)
    ESJ                  Profile Master 2000 Archived Job Database (MAP Ltd.)
    ESK                  ESK Document Reader 
    ESK                  Longplay Vesa file 
    ESL                  FoxPro Macintosh Runtime Library 
    ESL                  Visual FoxPro Distributable Support Library (Microsoft)
    ESL                  Everest Authoring System Screen Library 
    ESM                  Enhanced Simplex 
    ESM                  Europay Security Module 
    ESM                  The Elder Scrolls Main Game File (Bethesda Softworks LLC)
    ESO                  FoxPro 
    ESP                  TESIV:Oblivion Plug-in (Bethesda Softworks LLC)
    ESP                  Ventura File 
    ESP                  Easy Setup Program Configuration File (Microscan Systems, Inc.)
    ESP                  Morrowind: The Elder Scrolls Plug-in (Bethesda Softworks LLC)
    ESP                  Ethos System Analysis File 
    ESP                  ACD Analysis File 
    ESP                  ACD/CurveManager Analysis File 
    ESP                  Edskes Software Pakket 
    ESP                  Eclipse Screensaver Project (Active Media)
    ESP                  Email Sentinel Pro Protected Email Attachments (DS Development)
    ESP                  E$Planner File 
    ESP                  Extension-Sort Packer Compressed Archive (GyikSoft & MikroLab)
    ESP                  Personal Paint Spanish Language User Interface File (Amiga)
    ESPS                 ESPS Audio File 
    ESR                  4th Dimension Database Windows Procedure (4D, Inc.)
    ESS                  EXP: The Scientific Word Processor Style Sheet 
    ESS                  EasySpreadsheet Native Format File (e-press Corp)
    EST                  ICE 2000 File 
    EST                  Streets & Trips Trip File (Microsoft)
    EST                  Webalizer.01 Lang File 
    ESTIMATE             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ESU                  LokSound Electronic Sound File (Electronic Solutions Ulm GmbH & Co. KG)
    ESX                  Xactimate File 
    ESY                  EmEditor Syntax File 
    ES_                  Audio Waveprg Sounder Compressed File 
    ET1                  Energy 10 Weather Data 
    ET2                  Electronic Tax Return Security File 
    ETA                  eTreppid Audio File 
    ETB                  Seismograph File 
    ETBL                 eTextbookViewer Licence For Packaged Electronic Textbook (Textbook Solutions Limited)
    ETBP                 eTextbookCreator Project (Textbook Solutions Limited)
    ETBZ                 eTextbookViewer Packaged Electronic Textbook (Textbook Solutions Limited)
    ETC                  Seismograph File 
    ETC                  eTreppid Compressed File 
    ETD                  Acrobat eBook Reader EBX Transfer Data File (Adobe)
    ETF                  Enriched Text File 
    ETF                  PolyEdit File 
    ETH                  HP Internet Advisor Capture File 
    ETH                  Ethnograph Document File 
    ETL                  Eclipse Transition Listing (Active Media)
    ETL                  Windows2000 Trace Log 
    ETL                  ThermaData Logger Log Data File (Electronic Temperature Instruments Ltd)
    ETM                  eTreppid Audio File 
    ETP                  Visual Studio .NET Enterprise Template Project (Microsoft)
    ETV                  eTreppid Audio File 
    ETX                  Setext (Structure Enhanced Text) 
    EU                   Euphoria Include File (Rapid Deployment Software)
    EUC                  Japanese 
    EUC                  EucliDraw Dynamic Geometry File (Logismos Inc.)
    EUG                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EUG2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EUI                  Ensoniq EPS Family Compacted Disk Image (E-MU Systems)
    EUK                  Euklides Dynamic Geometry 
    EUM                  Euklides Dynamic Geometry 
    EUS                  EDonkey2000 Serverlist Tool 
    EV                   EV8 File 
    EV                   EchoView File 
    EV2                  Java File 
    EV2                  Calendar Commander Event File (Briggs Softworks)
    EV3                  EVERLOCK Options File 
    EV6                  EV8 File 
    EVA                  MSX Video File 
    EVA                  Viz Modeler 3D Scene Archive 
    EVB                  Embedded Visual Basic File 
    EVC                  EchoView Color Scheme 
    EVE                  EmbeddedVectorEditor Diagram (Barry Kauler)
    EVE                  Summit Event Manager Event Data File (MIE Software Pty Ltd)
    EVENT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EVENTS               Homeworld Animation/Effects Data (Sierra Entertainment, Inc.)
    EVERYC               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EVI                  EchoView Index File 
    EVK                  EVERLOCK File 
    EVL                  Business Evaluation Systems Pro Spreadsheet (ValuSource, the valuation software division of John Wiley & Sons, Inc.)
    EVL                  Cokriging Software COKRIG Output File 
    EVL                  Echoview Exported Line (SonarData Pty Ltd.)
    EVL                  EchoView Line File 
    EVL                  CHASE Evaluation and Audit System Evaluation Lock File (Health And Safety Technology And Management Ltd.)
    EVO                  HD DVD Video (DVD Forum)
    EVO                  CRiSP Harvest File 
    EVO                  EnVideo Video Animation Document (Computational Engineering International (CEI))
    EVP                  Sound Envelope 
    EVR                  Cokriging Software COKRIG Output File 
    EVR                  EchoView Region File 
    EVT                  Windows NT/2000 Event Viewer Log File (Microsoft Corporation)
    EVT                  PC Tools Event File Scheduler 
    EVT                  Cameo D1 Digital CCTV System Recording Event Log 
    EVY                  Tumbleweed Software ActiveX Control A portable format. 
    EVY                  Envoy Document (Corel Corporation)
    EW                   Euphoria Include File (Rapid Deployment Software)
    EWB                  Electronics Workbench Circuit (Electronics Workbench Corporation)
    EWD                  Express Publisher for Windows Document 
    EWF                  EpicWriter Document (Epic) (Trollzsoft)
    EWL                  EclipseCrossword Word List 
    EWL                  Encarta Document (Microsoft)
    EWP                  QuickCog Project File 
    EWP                  Evidence Washer Plug-In 
    EWP                  Elastic Collaboration EC-Button (Elastic Workspace Software, Inc.)
    EWS                  QuickCog Sample Set File 
    EWT                  Elastic Collaboration EC-Toolbar (Elastic Workspace Software, Inc.)
    EWZ                  EdgeWize Compressed Encoded Data File (Qivx Inc.)
    EX                   Symantec Ghost Template File (Symantec Corporation)
    EX                   Euphoria Source Code (Rapid Deployment Software)
    EX                   Cequadrant Backup File 
    EX                   EPM Macro File 
    EX                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EX                   Install Helper File 
    EX!                  Modem Bitware Fax Disk1 File 
    EX$                  Compressed File 
    EX$                  MS Compressed EXE Decompress with UNPACK.EXE. 
    EX$                  Temporary File 
    EX1                  Renamed .EXE File 
    EX2                  PC Doctor File 
    EX2                  ExodusII File 
    EX3                  Harvard Graphics Device Driver (Harvard Graphics)
    EXA                  Examine32 Saved Search Parameters 
    EXA                  Lsys File 
    EXAML                Examine32 Search Data (Aquila Software)
    EXAMPL               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EXAMPLE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EXB                  Flash Image File 
    EXC                  QEMM Exclude File 
    EXC                  REXX Program 
    EXC                  Exces Vault (SeoSoft)
    EXC                  Mail Exchange Component 
    EXC                  Text Document 
    EXC                  Word Exclusion Dictionary (Microsoft Corporation)
    EXCEPTION            As-U-Type Exceptions File (Fanix Software)
    EXD                  Control Information Cache (Microsoft)
    EXE                  Settlers IV Saved Game 
    EXE                  Executable File (Microsoft Corporation)
    EXE                  Self-extracting Archive 
    EXE                  Playstation Executable File (Sony Corporation of America)
    EXE                  Out-of-process Code Component Used by COM Clients (Microsoft)
    EXE                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EXE2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EXE3                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EXEC                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EXECUTE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EXEME                Likely an .EXE File Be careful if you rename and run a file of this type. 
    EXEMPLE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EXF                  CostX Data Exchange File (Exactal Pty Ltd)
    EXF                  Calculus EZ-Fax Fax 
    EXF                  Fuji EXIF Format File 
    EXL                  MIMEsweeper Ver 5 Expression List (Clearswift Limited)
    EXM                  HP Palmtop 95/100/200LX System Manager Executable File (Hewlett-Packard Development Company, L.P.)
    EXM                  RapidExam Test (XStream Software)
    EXMPLE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EXO                  System File 
    EXP                  Export File 
    EXP                  ASCET Binary Model Description (ETAS GmbH)
    EXP                  Viscal C++ Export File (Microsoft)
    EXP                  STEP Express Schema 
    EXP                  QuickBooks (Intuit Inc.)
    EXP                  PharLap Protected Mode Executable Program 
    EXP                  Melco Embroidery Design File 
    EXP                  ICQ Saved Chat File (ICQ Inc.)
    EXP                  CATIA Export File (Dassault Systèmes)
    EXP                  Drawing Express CAD Drawing File (Trial Systems)
    EXPERT               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EXPLAIN              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EXPLORE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EXPLORER             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EXPLOSION            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EXR                  OpenEXR Bitmap (Lucas Digital Ltd. LLC.)
    EXS                  Emagic EXS-24 Instrument Files (Apple Computer, Inc.)
    EXS                  VideoNet Director Video File (inLighten)
    EXT                  WS-FTP ASCII Binary Transfer File 
    EXT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EXT                  E-mail Text Attachment 
    EXT                  Extinction Corrected Photometry 
    EXT                  Norton Commander Extension File 
    EXTBASE              Associate This Extension Base (Spearit)
    EXTRA                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EXTRA2               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EXU                  Euphoria Source Code (Rapid Deployment Software)
    EXW                  Euphoria Source Code (Rapid Deployment Software)
    EXX                  IBM LinkWay MsgPut Intermediate File 
    EXY                  Orchida Embroidery System (Orchida Soft)
    EX^                  Norton Live Update File 
    EX_                  Compressed EXE File 
    EYA                  eYaViewer Graphic (ESRI China (Beijing))
    EYB                  Encarta Encyclopedia Yearbook and Web Links Update File (Microsoft Corporation)
    EYE                  Renderize Project 
    EYE                  ETI Camcorder Pro Video File (Eyemail Technology, Inc.)
    EYE                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    EYE                  CubicEye Cube 
    EYE                  Glass Eye File 
    EYE                  Eyeris Encoded Audio/Video File (Eyewonder, Inc.)
    EYE                  Eyemail Video (Eyemail Technology Inc (ETI))
    EYP                  EasyFP 
    EZ2                  EZPix Secure Photo Album Photo Package (Electronic Zombie Corp.)
    EZ3                  EZPix Secure Photo Album Photo Package (Electronic Zombie Corp.)
    EZB                  EasyBookkeeper Native Format File (e-press Corp)
    EZB                  EasyBoot Image File (EZB Systems, Inc.)
    EZC                  EZScreen Screen Saver Editor 
    EZC                  EZClaim Memorized Claim File (EZClaim)
    EZC                  EZGUI Designer Control Definition (Christopher R. Boss)
    EZHEX                Harmony Advanced Universal Remote Saved Configuration File (Logitech)
    EZL                  EasyLog Well Log Record (EasySolve Software LLC)
    EZM                  Text File 
    EZN                  EZClaim Memorized Note File (EZClaim)
    EZP                  EZ-Pix Image (Xequte Software)
    EZP                  Edify Electronic Workforce Backup Utility Compressed Slips File 
    EZP                  Edify Electronic Workforce Backup Utility 
    EZS                  Sunburst Technology Easy Sheet Document 
    EZS                  EZ-R Stats for Windows Batch Submit File (EZ-R Stats, LLC)
    EZT                  Sunburst Technology Easy Sheet Template 
    EZX                  WHAP File 
    EZX                  EZ-Forms (EZ-Forms Company)
    E_E                  Eru file 
    F                    Paradox File 
    F                    Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    F                    Newing-Hall Computer Engraving Software Graphic 
    F                    Freeze Compressed File Archive 
    F                    FORTRAN Source Code 
    F                    Win32Forth Source Code 
    F                    Compressed Archive File 
    F&F                  Netmino File 
    F00                  Kanji Font 
    F01                  Perfect Fax Document 
    F06                  DOS 6-pixel Screen Font 
    F07                  DOS 7-pixel Screen Font 
    F08                  DOS 8-pixel Screen Font 
    F09                  DOS 9-pixel Screen Font 
    F1                   21st Century Mahjong 
    F10                  DOS 10-pixel Screen Font 
    F11                  DOS 11-pixel Screen Font 
    F12                  DOS 12-pixel Screen Font 
    F13                  DOS 13-pixel Screen Font 
    F14                  DOS 14-pixel Screen Font 
    F15                  DOS 15-pixel Screen Font 
    F16                  DOS 16-pixel Screen Font 
    F2                   Biosflash File 
    F2                   FLASH BIOS File 
    F2F                  Byteworx FMEA² Exported Data (Byteworx Inc.)
    F2F                  File to File 
    F2R                  Farandoyle Linear Module Format 
    F3                   FLASH BIOS File 
    F32                  Raw 32-bit IEEE Floating Point Values 
    F33                  ANSYS Plot 
    F3D                  MotoRacer 3 Bike Model 
    F3R                  Farandoyle Blocked Module Format 
    F6                   Fonts File 
    F64                  Raw 64-bit IEEE Floating Point Values 
    F7                   Fonts File 
    F77                  FORTRAN 77 Program 
    F8                   Fonts File 
    F8                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    F90                  FORTRAN Program 
    F96                  Frecom FAX96 Document 
    FA                   EROSION 3D Flow Accumulation 
    FA1                  FAS 132 System Data File (DATAIR Employee Benefit Systems, Inc.)
    FA2                  FAS 132 System Custom Text File (DATAIR Employee Benefit Systems, Inc.)
    FABLES               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FAC                  Usenix FaceSaver Graphic 
    FACE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FACE                 Usenix FaceServer Graphic 
    FACT                 Electric Image 3D Image File (Electric image, Inc.)
    FACTOR               Factor Language Source File (Factor Community)
    FACTSHEET            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FAD                  Data File 
    FAD                  Fast Atomic Density Evaluator File 
    FADE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FADEIN               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FADEOUT              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FAITH2               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FAKRACK              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FAKRACK2             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FALLING              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FAM                  Paradox TV 
    FAM                  The Sims Family File (Electronic Arts (Maxis))
    FAM                  FamicomS Nintendo Entertainment System Emulator ROM Image 
    FAM                  Maxis File 
    FAM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FAME                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FAP                  FotoAngelo Project Settings (ACD Systems Ltd.)
    FAQ                  Frequently Asked Questions 
    FAQ                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FAQ-SPEEDUPS         Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FAQT                 FAQTool XML Contents File (DTLink LLC)
    FAR                  The Sims Archive File (Electronic Arts (Maxis))
    FAR                  Visual eForms Designer Form File (Cerenade, Inc.)
    FAR                  Farandoyle Tracker Music Module 
    FARGOLASE            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FAS                  Macsyma Compiled Program 
    FAS                  AutoCAD Fast-load Auto LISP (Autodesk, Inc.)
    FAS                  3D Fassade Plus Basic Module 
    FAST                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FASTA                dnaLIMS Fasta Library (dnaTools)
    FASTER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FAT                  Zinf Audio Player Theme File 
    FATESAVEDGAME        Fate Saved Game (WildTangent, Inc.)
    FAU                  Creator Simulator Collapsed Fault List (Tufts University)
    FAV                  Outlook Bar Shortcuts (Microsoft)
    FAV                  Technics Sx KN 6000 Keyboard Favorites File 
    FAX                  Fax File 
    FAX                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FAZ                  DeskMate Generic Character Data (Oska Educational Systems Pty Limited)
    FB                   Backup 
    FB                   Slim! Compressed Archive (Serge Voskoboynikov)
    FB2                  HaaliReader eBook | FictionBook 2.0 (Mike Matsnev)
    FB5                  FlowBiz Integrator File 
    FBAP                 FinalBuilder ActionStudio Automated Build Tool Action Design (VSoft Technologies Pty Ltd.)
    FBC                  FamilyTree Compressed Backup File 
    FBD                  The Quarterback Pro Coverage Worksheet 
    FBD                  Honeywell Control Builder File 
    FBD                  Remedy Change Management Flashboard 
    FBD                  CADfix Geometry Database File (ASCII) (ITI TranscenData)
    FBD                  DART Pro 98 User-definied Complex Restoration Filters 
    FBD                  F-22 Lightning Advanced Gravis Joystick Control File 
    FBDZONES             FileBoss Drop Zones Information (The Utility Factory)
    FBF                  Free Backup Fix Backup File (Promosoft Corporation)
    FBFILESET            FileBoss File Sets (The Utility Factory)
    FBI                  Total Annihilation Main Unit Definition File (Cavedog Entertainment)
    FBK                  FamilyTree Backup File 
    FBK                  Navison Financials Backup 
    FBK                  Train Simulator Shaders Definition File (Microsoft Corporation)
    FBL                  CADfix Log File (ITI TranscenData)
    FBLZ                 FinalBuilder Log File (VSoft Technologies Pty Ltd.)
    FBM                  Fuzzy Bitmap 
    FBM                  CADfix Geometry Database File (Binary) (ITI TranscenData)
    FBN                  ArcView Spatial Index File For Read-Only Datasets (ESRI)
    FBOPTS               FileBoss Options (The Utility Factory)
    FBP3                 FinalBuilder Ver 3. Project File (VSoft Technologies Pty Ltd.)
    FBP4                 FinalBuilder Ver 4. Project File (VSoft Technologies Pty Ltd.)
    FBR                  FlashBack File (Blueberry Software)
    FBS                  File Burner Skin 
    FBS                  FastBid Plan Sheet (Builders Exchange of Washington, Inc.)
    FBWINS               FileBoss Saved Windowing Information (The Utility Factory)
    FBX                  3D Data Exchange Format 
    FBX                  ArcView Spatial Index File For Read-Only Datasets (ESRI)
    FBZ3                 FinalBuilder Ver 3. Compressed Project File (VSoft Technologies Pty Ltd.)
    FBZ4                 FinalBuilder Ver 4. Compressed Project File (VSoft Technologies Pty Ltd.)
    FC                   Harvard Graphics Dictionary (Harvard Graphics)
    FC                   Harvard Graphics Ver. 2.0 Spell Checking Dictionary (Harvard Graphics)
    FC$                  Basic Realizer Disk1 L File 
    FC2                  Curious Labs Poser Face File 
    FCA                  Omnis Web Client Form Cache (Raining Data)
    FCB                  Forest Cover Attribute Processing File 
    FCB                  FAAST Builder File 
    FCC                  PEERNET File Conversion Center Conversion Job (PEERNET Inc.)
    FCC                  Basic Realizer Careal Library File 
    FCC                  SiteMinder Security Control File (Netegrity)
    FCD                  Virtual CD-ROM 
    FCD                  FastCAD/EasyCAD Output 
    FCD                  Patton & Patton Flow Charting 3 File 
    FCE                  Need for Speed Car Shape (Electronic Arts, Inc.)
    FCE                  Need for Speed Render File 
    FCF                  Final Draft File Converter Format (Final Draft, Inc.)
    FCF                  SHELXL93 File 
    FCF                  HP-95LX Filer Configuration File 
    FCF                  Quartus II FLEX Chain File (Altera Corporation)
    FCF                  xfit Phase File 
    FCG                  FastCGI Script 
    FCG                  Mystic Photo Format 
    FCGI                 FastCGI Script 
    FCH                  Gaussian Checkpoint File (CambridgeSoft Corporation)
    FCHK                 Gaussian Checkpoint File (CambridgeSoft Corporation)
    FCI                  EROSION 3D Inflow Table 
    FCL                  Cyber Audio Library Link File 
    FCM                  FC-M Packer Song/Module 
    FCM                  Fluke Networks CableManager File 
    FCM                  Forward Compression Binary File Patch 
    FCM                  Factory Commander Model Data File (Wright Williams & Kelly, Inc.)
    FCP                  FLAMES Checkpoint Restart File (Ternion Corporation)
    FCS                  Foobar2000 Columns UI Settings 
    FCS                  Canon Zoom Browser EX File 
    FCS                  Spectrum Server Log File 
    FCS                  Virtual FlashCards Flash Card Set (Alai Productions)
    FCS                  CD Trustee File 
    FCS                  c-tree Server/Plus Data File 
    FCS                  Fantasy Football League Organizer File 
    FCS                  RapidForm Polygon Faces Data (INUS Technology, Inc.)
    FCS                  RealProducer Pro Settings 
    FCS                  Flow Cytometry Standard Format 
    FCT                  FolderClone Task List (Salty Brine Software)
    FCT                  Foxpro Catalog (Microsoft)
    FCT                  Campaign Cartographer 2 File 
    FCW                  Campaign Cartographer 2 (ProFantasy Software Ltd.)
    FCW                  FastCAD File 
    FCX                  Vax VMS Compressed File 
    FD                   DataFlex Field Compiler Offsets (Data Access Corporation)
    FD                   Final Draft Document File (Final Draft, Inc.)
    FD                   Front Door Resource File 
    FD                   FORTRAN Declarations 
    FDA                  System SYSUTIL File 
    FDA                  FF&EZ Uncompressed Data Archive (Jim Carls - Data Management Consultant)
    FDAT                 Fiasco Database Data File 
    FDB                  Portfolio Catalog File (Extensis, Inc.)
    FDB                  Fiasco Database Database 
    FDB                  FlexyTrans Database (Berlitz International, Inc.)
    FDB                  FlukeView Forms Database (Fluke Corporation)
    FDB                  Firebird Database (Firebird Foundation Incorporated)
    FDB                  Legacy Family Tree Database (Millennia Corporation)
    FDB                  FoxPro Database 
    FDB                  Art Explosion Portfolio Catalog (Nova Development Corporation)
    FDB                  Navison Financials Database 
    FDB                  FormTool Pro Form File (IMSI)
    FDB                  Ming Font File (Opaque Industries)
    FDC                  Sniffer Capture File (Network Associates Technology, Inc.)
    FDD                  FormDocs Data (Formdocs LLC)
    FDD                  FlexiDATA Database File (Flexible Software Ltd)
    FDD                  Parallels Desktop for Mac Floppy Disk Drive Image (Parallels, Inc.)
    FDE                  Fade-It for AOL 
    FDE                  FLAMES Dataset Export (Ternion Corporation)
    FDF                  Acrobat Portable Document Input Form (Adobe)
    FDF                  Format Definition File (IBM)
    FDF                  Origin Fitting Function Definition File (OriginLab)
    FDI                  Amiga Disk File (Amiga)
    FDI                  FormDocs Index (Formdocs LLC)
    FDK                  OS/2 Saved SKF Format File (IBM)
    FDL                  FormDocs Link File (Formdocs LLC)
    FDL                  Paradox 
    FDL                  XYRO Active Distribution Console File Distribution List 
    FDM                  Exchange Forms Designer Form Definition (Microsoft)
    FDM                  Floppy Disk Manager Disk Image 
    FDP                  MapSonic Memory Card Format Map File (Michelin)
    FDPART               Fresh Download Partial Download (Freshdevices Corp.)
    FDR                  SideKick 2 Note File 
    FDR                  Final Draft Document File (Final Draft, Inc.)
    FDR                  Embroidery Design File 
    FDS                  JetForm FormFlow Data File 
    FDS                  Nintendo Famicom (NES) Disk System 
    FDT                  Final Draft Script Attributes File (Final Draft, Inc.)
    FDT                  FormDocs Template (Formdocs LLC)
    FDW                  FEBDOK (NELFO)
    FDW                  F3 Design and Mapping Document Form 
    FDX                  Force Index 
    FEATURES             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FEB                  WordPerfect Figure Editor Button Bar (Corel Corporation)
    FEB94                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FEC                  Ferite Programming Language File 
    FEC                  U.S. Federal Campaign Expense Submission File 
    FEE                  Precision Fees Fee Schedule (Kalahari Software, LLC)
    FEED                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FEEL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FEF                  Steuer2001 Formular File 
    FEG                  Art Explosion Greeting Card Factory E-Card (Nova Development Corporation)
    FEI                  Geoworks Fatal Error Infotable 
    FEM                  CADRE Finite Element Mesh File 
    FEN                  FileEncryptor Encrypted File (ARXsoft Co., Ltd.)
    FENDER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FENETRE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FEP                  WEFT Embedding Project (Microsoft)
    FES                  Topicscape Fileless Occurance Export (3D-Scape Limited)
    FEV                  FLAMES Environment Variable (Ternion Corporation)
    FEX                  Focus File 
    FEZ                  FUND E-Z Backup (FUND E-Z Development Corporation)
    FF                   Intelligont FIAS Format 
    FF                   Sigmac Source File 
    FF                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FF                   AGFA CompuGraphics Outline Font Description 
    FF1                  PCLTool (Page Technology Marketing, Inc.)
    FF2                  Softrans Gmbh Hello Engines! Standard file 
    FF2                  Flash Filer 
    FF3                  FreeForm 4.1 Database (Creative Companion Europe)
    FF7                  Final Fantasy VII Saved Game (Square Enix Co., Ltd)
    FF8                  !Miracle Executable 
    FFA                  Fast Find Status File (Index) (Microsoft)
    FFD                  Flash Filer 
    FFD                  Test & Go Preferences File (QualiControl)
    FFD                  Softrans Gmbh Hello Engines! 2.1 file 
    FFDATA               ABBYY eForm Filler Data (ABBYY Software)
    FFE                  DirectInput Force Feedback Effect (Microsoft)
    FFF                  Gravis UltraSound PnP InterWave Patch 
    FFF                  Pegasus Mail Extension (David Harris)
    FFF                  Maggi Hairstyles & Cosmetics 
    FFF                  defFax Fax Document 
    FFFF                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FFI                  Atech FastFont (AllType)
    FFIVW                ASCII File Format for the Interchange of Virtual Worlds (VRML) 
    FFL                  Alien vs Predator Archive 
    FFL                  PrintMaster Gold Image File 
    FFL                  Fast Find Document List (Microsoft)
    FFN                  Fifa World Cup Game Data FE Art File 
    FFO                  Fast Find Document Properties Cache (Microsoft)
    FFP                  Corel Graphics10 Custom File 
    FFT                  Fast Fourier Transform Analysis Data 
    FFT                  Fast Find (Microsoft)
    FFT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FFT                  DisplayWrite Document 
    FFT                  DCA/FFT Final Form Text 
    FFW                  File Fission Wizard File Split Information (Gavin Enterprises)
    FFX                  Fast Find Index (Microsoft)
    FGA                  Folder Guard Attributes 
    FGD                  Digital Raster Graphic Metadata File 
    FGD                  Folder Guard Data 
    FGD                  Half-life Modification Map Configuration File 
    FGL                  Inno Setup Component Definitions file 
    FGP                  Folder Guard Passwords File 
    FGX                  Formula Graphics Project File 
    FGZ                  Formula Graphics Standalone Presentation Archive 
    FH                   Freehand Graphic (Macromedia)
    FH10                 Freehand Ver. 10 File (Macromedia)
    FH11                 FreehandMX Ver. 11 File (Macromedia)
    FH3                  Freehand Ver. 3 Drawing (Macromedia)
    FH4                  Freehand Ver. 4 Drawing (Macromedia)
    FH5                  Freehand 5 (Macromedia)
    FH6                  Freehand 6 (Macromedia)
    FH7                  Freehand 7 (Macromedia)
    FH8                  Freehand 8 (Macromedia)
    FH9                  Freehand 9 (Macromedia)
    FHC                  Freehand (Macromedia)
    FHD                  PCLTool Form File (Page Technology Marketing, Inc.)
    FHD                  Family Historian Diagram Type (Calico Pie Limited)
    FHIS                 Insight II Dynamics Trajectory History File (Accelrys Software Inc.)
    FHQ                  Family Historian Query (Calico Pie Limited)
    FHR                  Family Historian Report (Calico Pie Limited)
    FHT                  Family Historian Text Scheme (Calico Pie Limited)
    FHTML                HTML File (Macromedia)
    FHX                  Fh5 Usenglsh Xtras File 
    FHX                  DeltaV Configuration File (Emerson Process Management)
    FI                   Bitstream Intellifont 
    FI                   Flash Image 
    FI                   FORTRAN Interface File 
    FI                   Dispatch Icelandair Timetable File (Scott Andrew Borton)
    FI                   FORTRAN Interface File (Microsoft)
    FIB                  File Investigator (RobWare)
    FIC                  Sheffield Dictionary File 
    FIC                  WINDEV Hyper File Database (Native File Format) (PC SOFT)
    FICONS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FID                  Bruker Aspect NMR Data File 
    FID                  File Investigator (RobWare)
    FIDDLE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FIDX                 Fiasco Database Index 
    FIF                  SpinRite Hard Disk Fingerprint File 
    FIF                  Fractal Image Format 
    FIF                  Font Information File 
    FIF                  FIF 
    FIF                  Altamira Genuine Pixels 1.0 Compressed Format 
    FIG                  MATLAB Plot Figure File (The MathWorks, Inc.)
    FIG                  REND386/AVRIL Graphic 
    FIG                  Super Nintendo Game-console ROM Image 
    FIG                  XFIG Graphic File 
    FIG                  Lahey FORTRAN Linker Options 
    FIG                  Geometry II Plus Figure (Cabrilog)
    FIGHT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FIGURE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FIH                  File Investigator (RobWare)
    FIL                  SeeYou Flight Data (Naviter)
    FIL                  AVG Antivirus Vault File (GRISOFT, s.r.o.)
    FIL                  WordPerfect Overlay (Corel Corporation)
    FIL                  ProWORX Nxt Format Export (Schneider Electric)
    FIL                  File List 
    FIL                  ABAQUS File (ABAQUS, Inc.)
    FIL                  dBASE Files List Object 
    FIL                  Cardbox Database 
    FIL                  Application Generator File Template 
    FIL                  ACL For Windows Data 
    FILE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FILE                 File With No Extension 
    FILE                 AS/400 Physical File (IBM)
    FILEKILL             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FILEPLANETCACHE      GameSpy Aborted Download (GameSpy Industries)
    FILES                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FILM                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FILMAO               Aleph One Film (Marathon Open Source)
    FILTER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FIM                  Construsyc/Instasyc File 
    FIM                  Paul van Keep's Icon Heaven Icons Package 
    FIM                  Fractal Imaginator Data/Parameter File (Mystic Fractal)
    FIN                  Personal Paint Finnish Language User Interface File (Amiga)
    FIN                  ABAQUS File (ABAQUS, Inc.)
    FIN                  Corel Saved Find/Search Info (Corel)
    FIN                  WordPerfect Print Format Text (Corel Corporation)
    FIN                  ATI Radeon Video Driver 
    FIN                  Perfect Writer Print Formatted Text 
    FIN                  Perfect Writer/Scribble/MINCE Print-formatted Text File 
    FIN                  XtalView Basic Crystallographic Data File (The Computational Center for MacroMolecular Structures (CCMS))
    FINAL                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FIND                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FINDER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FIO                  PhotoStyler Graphics Filter (Aldus)
    FIO                  ULead Viewer Support File 
    FIO                  Image PALS Viewer 
    FIP                  FingerPost Information Processor File 
    FIP                  File Investigator (RobWare)
    FIR                  Webalizer.01 File 
    FIR                  Canon Digital Camera EOS Camera Firmware File (Canon Inc.)
    FIRE                 FireStarter Project File 
    FIRESTORMFT          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FIRMWARE             London Architect Soundweb London Firmware (BSS Audio)
    FIRST                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FIS                  Dynamite VSP Feature Interpretation Style (3am Solutions (UK) Limited)
    FISH                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FISH                 3d Fish Tank Saved Fish 
    FISHES               3d Fish Tank Saved Fishes 
    FISHTANK             3d Fish Tank Saved Fishtank 
    FIT                  Windows NT File Index Table (Microsoft)
    FIT                  FITS Graphic 
    FITS                 FITS Graphic 
    FITS                 FITS File 
    FIV                  File Investigator (RobWare)
    FIVE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FIX                  Generic Patch File 
    FIX                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FIXED                DLLbackup Root File 
    FIXER                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FIXES                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FIXKERN              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FIZ                  FITS File 
    FK                   Shockwave Streamed Media File 
    FKD                  FotoKiss Photo Set Data File (LANSRAD)
    FKY                  Foxpro Macro (Microsoft)
    FL                   Freelancer Saved Game (Microsoft)
    FL                   Floating Format Sound 
    FLA                  Free Lossless Audio Codec 
    FLA                  Flash Movie Authoring File (Macromedia)
    FLAC                 Free Lossless Audio Codec 
    FLAG                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FLAGS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FLARE                Homeworld (Sierra Entertainment, Inc.)
    FLASK                FlasKMPEG Language Definition File 
    FLB                  VersaPro Folder Backup 
    FLB                  MicroSim PCBoard Footprint Library File 
    FLB                  Papyrus Format Library 
    FLC                  FIGlet Control File 
    FLC                  FLIC Animated Picture (Autodesk, Inc.)
    FLC                  Corel Show 
    FLD                  3D Fassade Plus Field Define Module 
    FLD                  Charisma Folder 
    FLD                  Hijaak Thumbnail Folder 
    FLD                  VersaPro Folder Contents 
    FLD                  WinFlash Educator Flashcard Options/Statistics File (Open Window Software)
    FLE                  Op9630 Settings File 
    FLE                  Scanner Settings File 
    FLETCH               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FLEX                 Flexi Data Smoothing 
    FLEXRESP             Snort File 
    FLF                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FLF                  Figletized Font File 
    FLF                  Navison Financials License File 
    FLF                  Firehand Lightning Graphic Collection 
    FLF                  Corel Paradox Form 
    FLF                  FIGlet Zipped Figletized Text Font 
    FLF                  OS/2 Driver File 
    FLG                  MicroSim PCBoard Log Of Forward Engineering Change Orders Applied To The Layout 
    FLH                  FLIC Animation 
    FLI                  EmTeX TeX Font Library 
    FLI                  FLIC Animation (Autodesk, Inc.)
    FLI                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FLIGS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FLINT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FLIPCURS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FLK                  File Lock (FuJe Software)
    FLK                  MYOB Lock Files (MYOB Technology Pty Ltd.)
    FLL                  TextPipe Pro Filter 
    FLL                  Foxpro Library (Microsoft)
    FLL                  Micrografx Designer 7 Image Fill Pattern 
    FLL                  The Sims Floor File (Electronic Arts (Maxis))
    FLM                  AutoCAD Film (Autodesk, Inc.)
    FLM                  Adobe FilmStrip (Adobe Systems Inc.)
    FLM                  Filmline Document 
    FLM                  FoxPro Library 
    FLM                  Image Format on Atari 
    FLM                  Medi@Show File 
    FLM                  RealPixel Animation File 
    FLN                  XMap Flight Plan or Flight Plan Statistic File (Delorme)
    FLO                  FlowCharter (iGrafx (a division of Corel Inc.))
    FLOBY                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FLOOR                Structural Concrete Software Floor Data File 
    FLOW                 floW Structured Data Definition (ontonym.net)
    FLOWER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FLOYD                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FLP                  Family Lawyer Saved Document 
    FLP                  Floppy Disk Image File 
    FLP                  Corel Flow Project Flow Chart 
    FLP                  Fruityloops Saved File (Track) 
    FLP                  FlipAlbum File 
    FLP                  Fractal Explorer Fractal Landscape Project 
    FLP                  GIFLine V1.0 
    FLR                  Live3D File (Netscape)
    FLR                  The Sims Floor Texture File (Electronic Arts (Maxis))
    FLS                  WinFlash Educator Flashcard Source File (Open Window Software)
    FLS                  DBdemos File 
    FLS                  Creator Simulator Fault List (Tufts University)
    FLS                  Nokia Phone Flash File (Nokia)
    FLS                  Stock Prospector Data Export Format (ICLUBcentral Inc.)
    FLS                  ArcView Windows Help Supporting File (ESRI)
    FLS                  Farrukh Imposition Purlisher Filelist Document 
    FLT                  Filter 
    FLT                  Graphics Filter (Microsoft)
    FLT                  Aldus Import Filter 
    FLT                  Animator Animation (Autodesk, Inc.)
    FLT                  Asymetrix Graphics Filter Support 
    FLT                  Audition Filter (Adobe Systems Inc.)
    FLT                  FLIC Animation 
    FLT                  FileMaker Filter (FileMaker, Inc.)
    FLT                  Illustrator Filter (Adobe)
    FLT                  Lotus 1-2-3 LogAnalyzer Filter File 
    FLT                  Micrografx Picture Publisher Filter 
    FLT                  OpenFlight 3D File (MultiGen-Paradigm)
    FLT                  OS/2 Warp Filter Device Driver 
    FLT                  StarTrekker Music Module 
    FLT                  WinFlash Educator Flashcard Compiled Test File (Open Window Software)
    FLT                  XMap Flight Plan or Flight Plan Statistic File (Delorme)
    FLT                  Word Import Filter (Microsoft Corporation)
    FLT                  Corel Graphic Filter 
    FLT                  Qimage Filter (Digitalala.com)
    FLT                  EtherPeek Filter (WildPackets Inc.)
    FLT                  Flight Simulator Saved Flight In Progress (Microsoft)
    FLU                  WinFlash Educator Ver 10 Unicode Source File (Open Window Software)
    FLUTE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FLUX                 FluxTime Old Clip Format (pCode Software)
    FLV                  Flash Video File (Macromedia, Inc.)
    FLW                  Kivio Flowchart (theKompany.com)
    FLX                  MicroSurvey CAD Drawing File (MicroSurvey Software, Inc.)
    FLX                  FLIC Animation 
    FLX                  Flexstor Database Format 
    FLX                  The Bat! Folder Configuration (RITLABS)
    FLX                  DataFlex Compiled Binary File/Ordered Binder (Data Access Corporation)
    FLY                  Digiflyer E-mail Document (Digiflyer)
    FLY                  Quiz-Buddy Quiz Data (Sierra Vista Software)
    FLY                  TerraExplorer Project (Skyline Software Systems Inc.)
    FLYP                 Mapwing Tour Project (Redbug Technologies)
    FLYS                 Mapwing Streaming Tour (Redbug Technologies)
    FLZ                  Medi@Show Data File 
    FLZ                  WinFlash Educator Flashcard Compiled Study File (Open Window Software)
    FLZW                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FM                   FeatureCAM Document (Engineering Geometry Systems)
    FM                   FrameMaker Document (Adobe Systems Inc.)
    FM                   FileMaker Pro Spreadsheet 
    FM1                  1-2-3 Release 2.x Spreadsheet File (IBM)
    FM2                  Maestro Mama Demo File 
    FM3                  1-2-3 Format File (IBM)
    FM3                  Harvard Graphics Device Driver (Harvard Graphics)
    FM3                  FileMaker Ver. 3 Database Document (FileMaker, Inc.)
    FM5                  FileMaker Ver. 5 Database Document (FileMaker, Inc.)
    FMB                  WordPerfect File Manager Button Bar (Corel Corporation)
    FMB                  Oracle Binary Form Source Code (Oracle Corporation)
    FMB                  OnBelay Flash/Disk Media Backup (CompuApps, Inc.)
    FMC                  Embroidery Design File 
    FMC                  NCAD File 
    FMC                  System SYSUTIL File 
    FMD                  Open Access File 
    FMD                  MotionArtist Document (e frontier America, Inc.)
    FMD                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FMD                  Cumulate Draw Editable FMD Format (Cumulate Labs)
    FMDTOOL              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FME                  FMEAPlus Database (Ford Motor Company)
    FMF                  FaxMan Jr Fax File (Data Techniques, Inc.)
    FMF                  IBM LinkWay Font or Icon File 
    FMG                  FreeMarkets Graphics Browser 
    FMGEN                FlexiMusic Generator 
    FMK                  PowerStation FORTRAN Make File 
    FMK                  FaxMaker File 
    FMK                  FlowMake Workflow Design (Praxis. Distributed Systems Technology Centre)
    FML                  Oracle Mirror List 
    FML                  GetRight File Mirror List (Headlight Software, Inc.)
    FMMUS                FlexiMusic Producer 
    FMO                  dBASE Ordered Format 
    FMORC                FlexiMusic Orchestra 
    FMP                  AutoCAD Font Map (Autodesk, Inc.)
    FMP                  FileMaker Pro Document 
    FMP                  FLAMES Model Prototype (Ternion Corporation)
    FMP                  Profit Plan Financial Master Plan Index (Security Development Corporation)
    FMP                  FantaMorph Project (Abrosoft)
    FMR                  Family Medical Records Data File 
    FMS                  FMS32-PRO FMS Protocol File (HeiRueSoft)
    FMS                  FantaMorph Skin File (Abrosoft)
    FMS                  1-2-3 Impress Add-in (IBM)
    FMT                  Style Sheet (Sprint)
    FMT                  Schedule+ Print File (Microsoft)
    FMT                  Ox Object-Oriented Matrix Programming Language Gauss Matrix File 
    FMT                  Pro/ENGINEER (PTC)
    FMT                  1-2-3 (Lotus)
    FMT                  Format File 
    FMT                  dBASE IV Format File 
    FMT                  Cardbox Format File 
    FMT                  Visual FoxPro Csreen Format File (Microsoft)
    FMT                  Oracle Text Form Format 
    FMV                  Frame Vector Metafile 
    FMV                  FrameMaker Picture (Adobe)
    FMV                  Bink Tools Full Motion Video File (RAD Game Tools, Inc.)
    FMX                  Oracle Executable Form (Oracle Corporation)
    FMZ                  Form*Z Project File 
    FMZIP                FlexiMusic Zipper 
    FN3                  Harvard Graphics Font (Harvard Graphics)
    FN3                  Denso BHT-7000 Terminal Executable Program File 
    FND                  Informida Secure Information Database 
    FND                  Explorer Saved Search (Microsoft Corporation)
    FNG                  Font Navigator Group File 
    FNK                  FunkTracker Module 
    FNO                  Folio Infobase 
    FNP                  FlashNpack 
    FNS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FNT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FNT                  Font File 
    FNT                  FONTEDIT.EXE Raw Font 
    FNW                  FLARENET Model Data (Aspen Technology, Inc.)
    FNX                  Exact Inactive Font 
    FN_                  CAD File 
    FO$                  Ballade Compressed Audio File 
    FO1                  Turbo C Font (Borland Software Corporation)
    FO2                  Turbo C Font (Borland Software Corporation)
    FOB                  Navision Attain Object File 
    FOC                  FOCUS/WebFOCUS Database File 
    FOCF                 Folder Organizer Folder Organizer Configuration File (H-SW)
    FOD                  EZ-Forms ULTRA Relationship File (EZ-Forms Company)
    FOF                  EZ-Forms Filled Out Form (EZ-Forms Company)
    FOG                  Fontographer Font 
    FOG                  Computer Generated Forces for Action Fog Defaults 
    FOL                  1st Reader Saved Message Folder 
    FOL                  pfs:First Choice Database File 
    FOLD                 LockedFolder Encrypted File (AwesomeScripts.com)
    FOLDER               Explorer Folder (Microsoft)
    FOLDERS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FOM                  EZ-Forms Fill Out Master Form (EZ-Forms Company)
    FON                  Telix Dialing Directory 
    FON                  Font File (Microsoft)
    FON                  Procomm Plus Call Log 
    FONT                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FONT                 UNIX Font Data 
    FONT1                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FONTASM              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FONTCHART            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FONTCOMM             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FONTGEN              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FONTREPORT           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FONTRIX              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FONTS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FONTYX1              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FONTYX2              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FOO                  Kai's SuperGoo Fusion 
    FOO                  Foobar2000 Audio Metadata Database 
    FOODS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FOOTER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FOP                  Freedom of Press Bitmap Image 
    FOR                  Font Resource File 
    FOR                  FORTRAN Source 
    FOR                  WindowBase Form 
    FORCE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FORECAST             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FORM                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FORMAT               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FORMATTER            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FORMS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FORREVIEW            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FORTH                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FORUM                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FOT                  Installed TrueType Font 
    FOU                  SpiceCAD Lowpass File 
    FOUNDATION           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FOUR                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FOUR                 VAX CRYSTALS Chemical Crystallogaphy File 
    FOW                  Family Origins File 
    FOX                  FoxBase/FoxPro+ Executable File 
    FP                   FinePrint Saved or Output File (FinePrint Software, LLC)
    FP                   Homeworld (Sierra Entertainment, Inc.)
    FP                   FastPaste Project File (Dextronet)
    FP                   CRiSP Harvest File 
    FP                   FileMaker Pro File 
    FP                   FoxPro Configuration Info (Microsoft)
    FP                   Vinal Page Vector TIFF Image 
    FP                   GoldMine Business Contact Manager Report 
    FP1                  Flying Pigs for Windows Data 
    FP2                  HDF Utility File 
    FP3                  Imsi FloorPlan 3D 
    FP3                  FileMaker Pro 3.0 File 
    FP3                  Fisher Price FP3 Player Music File (Mattel, Inc.)
    FP5                  FileMaker Pro Database (FileMaker, Inc.)
    FP7                  FileMaker Pro Ver. 7+ Database Document (FileMaker, Inc.)
    FPA                  Franklin Potter Associates Business Invoice Designer 
    FPB                  Fiorano BPEL Flow Data (Fiorano Software Inc.)
    FPB                  FLAMES Playback File (Ternion Corporation)
    FPBF                 Finder Backup Burnable Archive (Apple Computer, Inc.)
    FPC                  QuoteWerks Report Template (Aspire Technologies, Inc.)
    FPC                  FoxPro Catalog File (Microsoft)
    FPD                  TMT Pascal Compiled Unit 
    FPD                  MicroSim PCBoard External ASCII Footprint Definition File 
    FPF                  IKEA Kitchen Planner Document (Inter IKEA Systems B.V.)
    FPF                  FilterPro Active Filter Design (Texas Instruments Incorporated)
    FPF                  Canadian Flight Wizard 
    FPG                  DIV Game Studio Multi Map 
    FPHTML               FrontPage HTML Document (Microsoft)
    FPIX                 FlashPix Bitmap 
    FPK                  JetForm FormFlow File (Adobe Systems Incorporated)
    FPK                  PerFORM Communicator File Packet (Adobe Systems Incorporated)
    FPL                  FrontDesigner Front Panel (ABACOM Ingenieurbüro)
    FPL                  Foobar2000 Playlist 
    FPM                  FoxPro Startup File 
    FPR                  Fruitypro Samples Humanize Presets Grooves File 
    FPR                  FLAMES Prototype File (Ternion Corporation)
    FPR                  MicroSim PCBoard Footprint Statistics Report 
    FPR                  Fiasco Database Print Report Definition 
    FPS                  Operation Flashpoint Game File (Bohemia Interactive and The Codemasters Software Company Limited)
    FPT                  FileMaker Pro File Database Memo 
    FPT                  Farandole Composer Pattern 
    FPT                  FoxPro Memo Field (Microsoft)
    FPT                  Abacus Law Data (Abacus Data Systems, Inc.)
    FPT                  Ultimate Family Tree Geneology Database 
    FPT                  Liquid Audio Player Faceplate (Theme) File (Geneva Media, LLC)
    FPT                  Alpha Five Memo Field File (Alpha Software, Inc.)
    FPT                  ACT! (Best Software CRM Division)
    FPT                  NovaBACKUP Backup Database Related Information (NovaStor Corporation)
    FPVERS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FPW                  FloorPlan Plus Floorplan Drawing 
    FPW                  FoxPro Configuration 
    FPWEB                FrontPage Disk Based Web (Microsoft)
    FPX                  FlashPix Bitmap (Kodak)
    FPX                  Compiled FoxPro Program 
    FQF                  FlashFXP Queue File 
    FQY                  FLAMES (FLARE) Command File (Ternion Corporation)
    FR3                  Fractal Explorer 3D Attractors/IFS Spots in Internal Parametrical Format 
    FR3                  dBASE IV Renamed dBASEIII+ Form 
    FR4                  Fractal Explorer Quaternion Spots in Internal Parametrical Format 
    FRA                  Fractint Fractal Graphic 
    FRA                  FrameViewer File 
    FRA                  Fifa World Cup Game Data fe Art Legalscr File 
    FRA                  Fractal Explorer Animation of Fractal Navigation Project 
    FRA                  Personal Paint French Language User Interface File (Amiga)
    FRACT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FRAME                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FRAME5               CVIP Formatted File 
    FRAMES               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FRC                  FLAMES RECORD Output File / FLARE Input File (Ternion Corporation)
    FRD                  Need for Speed Trk2000 Track File 
    FRE                  Creative Digital Blaster Digital VCR File 
    FRE                  Fifa 2002 Data Legal Screen File 
    FRE                  Male Normal CT 
    FREC                 Fiasco Database Frequencies File 
    FREECELLSAVE-MS      FreeCell Saved Game (Microsoft Corporation)
    FREEGUITAR           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FREESIDEFNT          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FREQ                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FRF                  FontMonger Font 
    FRG                  dBASE IV Uncompiled Report 
    FRI                  Firefox Bookmark Backup File 
    FRI                  Fractal Explorer IFS Spot in Internal Parametrical Format 
    FRK                  File holding Macintosh Resource Fork Information 
    FRK                  File holding Macintosh Data Fork Information 
    FRL                  GP-Forth Library 
    FRL                  Formflow Electronic Form Loader (Adobe Systems Inc.)
    FRM                  WordPerfect Merge Form (Corel Corporation)
    FRM                  FrameBuilder Document 
    FRM                  Visual Basic Form 
    FRM                  MYOB Form File (MYOB Technology Pty Ltd.)
    FRM                  Megalux Frame 
    FRM                  FrameMaker Document (Adobe)
    FRM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FRM                  Insight II Free-Format Files (Accelrys Software Inc.)
    FRM                  Oracle Executable Form 
    FRM                  Pro/ENGINEER (PTC)
    FRM                  Phoenix Visual Designer Form (Prometheus Software)
    FRM                  DataCAD Symbol Report File (DATACAD LLC)
    FRM                  CT Summation iBlaze Database Form File (CT Summation, Inc.)
    FRM                  EZ-Forms Master Form (EZ-Forms Company)
    FRM                  dBASE IV Report File 
    FRM                  Form File 
    FRM                  Fractint Formula 
    FRM                  Frame Stacks 
    FRM                  MySQL Generic Dictionary File (MySQL AB)
    FRO                  dBASE IV Compiled Report 
    FRO                  A-Robots Fighting Robot Object 
    FRO                  FormFlow File 
    FRONT242             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FRONT_242_MIX        Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FRP                  Fractal Explorer Palette 
    FRP                  FaxRush Packages 
    FRP                  PerFORM Pro Plus Form (Adobe Systems Incorporated)
    FRP                  Form Flow 2.x Data File (Adobe Systems Incorporated)
    FRP                  FormFlow 2.x Data File (Adobe Systems Incorporated)
    FRS                  WordPerfect Graphics Driver (Corel Corporation)
    FRS                  Corel Painter Pattern, Selection or Texture File 
    FRS                  Fractal Explorer Formula Spot for Fractals 
    FRT                  FoxPro Report File (Microsoft)
    FRT                  GP-Forth Language Source Code File 
    FRW                  RollerCoaster Tycoon Fireworks Source (Atari)
    FRX                  Visual Basic Binary Form File (Microsoft)
    FRX                  FoxPro Report File (Microsoft)
    FRZ                  FormFlow File 
    FRZ                  EPOC Freeze File 
    FRZ                  Formflow Electronic Form Loader (Adobe Systems Inc.)
    FS                   NetBSD File 
    FS                   Usenix FaceServer Bitmap 
    FS                   FormSaver File 
    FS                   FlexiSIGN File (Scanvec Amiable)
    FS                   PDT Feature Structure File 
    FS                   Taskmaster Paper Keyboard FormSpec (Datacap Inc.)
    FS2                  FreeSpace 2 Mission (Volition, Inc.)
    FS5                  Flight Simulator Scenery File (Microsoft)
    FS6                  Flight Simulator Panels File (Microsoft)
    FSA                  ABIF Applied Biosystems Inc. Format (Applied Biosystems)
    FSAC                 Files Search Assistant File List (AKS-Labs)
    FSC                  Worms Armageddon Fiddler File 
    FSC                  Campaign Cartographer 2 File 
    FSC                  Design Pro Tools Form File (Elixir Technologies)
    FSC                  FLAMES Scenario File (Ternion Corporation)
    FSD                  FacetWin Backup File Set (FacetCorp)
    FSD                  Sheets Document (Leaf Twenty-Five)
    FSED                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FSF                  Convert It! Factor File 
    FSF                  fPrint Audit Tool File 
    FSFIX                Internet Security Signed Product Patch File (F-Secure)
    FSG                  IBM Voice Type Language Map File 
    FSH                  Fifa World Cup Game File 
    FSH                  EA Sports Game Graphic Editor File (Electronic Arts Inc.)
    FSH                  Coolfish Encrypted File (CompSci)
    FSI                  FileSplit 
    FSKEY                Internet Security Product Key File (F-Secure)
    FSL                  Paradox Version 7 Forms (Borland Software Corporation)
    FSL                  Corel Paradox Saved Form 
    FSM                  Farandoyle Composer WaveSample Music Format 
    FSP                  Floating Point Data Files 
    FSP                  FormScape Branch File (FormScape, Ltd.)
    FSP                  fPrint Audit Tool File Format 
    FSR                  File Substring Replacement Utility 
    FSS                  Micrografx Picture Publisher 8 Register File 
    FSS                  File Split Pro 
    FSS                  Iomega Backup File Selection Set 
    FSSD                 Sound 
    FST                  FruityLoops Channel State File 
    FST                  FastStats Analyzer File 
    FST                  dbFast Linkable Program 
    FST                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FST                  MRS-802 Rhythm Song Fast Method Formula Data (Zoom Corporation)
    FSX                  FormSaver Fields 
    FSX                  1-2-3 Data (IBM)
    FSY                  FileSync Profile (FileWare)
    FSY                  PhotoFantasy Image 
    FSZ                  Fractal Zplot Data/Parameter File (Mystic Fractal)
    FT                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FT                   Notes Full Text Index (IBM)
    FT5                  FH5 File 
    FT7                  Macromedia Freehand Drawing 
    FT8                  Macromedia Freehand Drawing 
    FT9                  Macromedia Freehand Drawing 
    FTA                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FTB                  Roots3 Index File 
    FTB                  Family Tree Maker Geneology File (MyFamily.com, Inc.)
    FTBL                 PIPE-FLO Professional Fluid Data Table (Engineered Software)
    FTC                  FluxTime Clip (pCode Software)
    FTG                  ArcView UNIX Help Supporting File (ESRI)
    FTG                  Windows Help Full-text Search Group File 
    FTI                  FamiTracker Instrument (jsr)
    FTL                  FutureTense Texture 
    FTL                  Family Tree Legends Family File (Pearl Street Software)
    FTM                  Finale Template File (MakeMusic!)
    FTM                  FaceTheMusic Song/Module 
    FTM                  FamiTracker Song File (jsr)
    FTM                  MicroGrafx Font 
    FTM                  Fathom Data File (Chartwell-Yorke)
    FTM                  Family Tree Maker Family File (MyFamily.com, Inc.)
    FTN                  FORTRAN Source Code File 
    FTO                  Sistemas Administrativos (ASPEL DE MEXICO, S.A. de C.V)
    FTP                  FTP Configuration Information 
    FTP                  FTP Voyager Document (Rhino Software, Inc.)
    FTR                  Future Document (SERAFIM Limited)
    FTS                  FastTrack Schedule File (AEC Software, Inc.)
    FTS                  ArcView UNIX Help Supporting File (ESRI)
    FTS                  Borland BDE File (Borland Software Corporation)
    FTS                  Flexible Image Transport System Bitmap 
    FTS                  StateClock Export to STEP 
    FTS                  Windows Help Full-Text Search Index File (Microsoft)
    FTS                  CHASE Evaluation and Audit System Help File (CHASE.FTS) (Health And Safety Technology And Management Ltd.)
    FTV                  Forest & Trees File 
    FTW                  FormTool Pro Form File (IMSI)
    FTW                  Family Tree Maker Family Tree File (MyFamily.com, Inc.)
    FTW                  FontTwister File 
    FU                   F* You Software Key Generator 
    FUD                  FairUse Wizard Data 
    FUK                  POSTAL 2 Map File (Running With Scissors)
    FUL                  Full Foram Count File 
    FULL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FULL                 Cygwin Variable Log File 
    FUN                  Project FUN Editor Game Source Code (DigiPen Institute of Technology)
    FUN                  Koolmoves File 
    FUN                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FUP                  FairUse Wizard Project 
    FUTHARK              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FUTHARKG             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FUTURELAS            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FUZZGUITAR           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FUZZY                Fuzzy Logic File 
    FUZZY                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FVANIM               Powerbullet Presenter Animation File (DDD Pty Ltd.)
    FVF                  Fluke View 
    FVML                 formVista Document (DTLink LLC)
    FVPL                 Frozen Vertex Pair List File 
    FVS                  Forest Vegetation Simulator Tree Data File 
    FVT                  Interlock Public Computer Utility 
    FW                   Framework Database 
    FW                   Kodak Camera Firmware File (Eastman Kodak Company)
    FW2                  Framework II File 
    FW3                  Framework III File 
    FW4                  Framework IV File 
    FWB                  FileWrangler Data File Backup 
    FWD                  IMail Server Forwarded Mail (Ipswitch, Inc.)
    FWD                  RollerCoaster Tycoon Fireworks Display (Atari)
    FWEB                 Fortran WEB 
    FWF                  Xwave FWF File 
    FWI                  PhotoSmart 850 Digital Camera Firmware Update (Hewlett-Packard Development Company, L.P.)
    FWK                  Fireworkz Document (Abacus Training)
    FWL                  FileWrangler EXE Library 
    FWP                  Worms Armageddon Fiddler Weapons Module 
    FWP                  SharePoint Designer (Microsoft Corporation)
    FWP                  IMail Server Forwarded Mail (Ipswitch, Inc.)
    FWS                  FileWrangler Data File for File Splitting Configuration 
    FX                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FX                   DirectX D3DXEffect Object (Microsoft)
    FX                   WordPerfect Office Template File (Corel Corporation)
    FX                   FastLynx On-Line Guide 
    FX2                  WordPerfect Office Template File (Corel Corporation)
    FX2                  G3 Two-dimensional Working with Message Windows Fax 
    FXB                  HALion Sampler File (Steinberg Media Technologies GmbH.)
    FXD                  WinFax Sent Document (Symantec Corporation)
    FXD                  FAXit Phonebook 
    FXD                  FoxPro FoxDoc Support 
    FXF                  BigFix Fix File (BigFix, Inc.)
    FXG                  BigFix Fixlet Pool (BigFix, Inc.)
    FXI                  File-Ex Configuration File (Cottonwood Software)
    FXM                  Fuxoft AY Music Chip Language 
    FXM                  BigFix Fixlet Pool (BigFix, Inc.)
    FXM                  WinFax Fax (Symantec)
    FXO                  DirectX Pre-Compiled Pixel/Vertix Shader (Microsoft Corporation)
    FXO                  Fax Image Document 
    FXP                  FoxPro Compiled Source (Microsoft)
    FXP                  HALion Sampler File (Steinberg Media Technologies GmbH.)
    FXP                  Steinberg Plug-in Format (Steinberg)
    FXPACK               MXwendler Packed Shader (device+context)
    FXR                  WinFax Received Document (Symantec Corporation)
    FXS                  The Land Of UM 
    FXS                  WinFax Fax Transmit Graphi (Symantec)
    FXT                  Cottonwood Software File-Ex Trained Template 
    FXT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    FXT                  Finale Music Notation Software Plug-in 
    FXY                  Orchida Embroidery System (Orchida Soft)
    FY                   Fontasy Publisher Regular Font 
    FYB                  Fontasy Publisher Bold Font 
    FYI                  Flow-Cal File 
    FYI                  Fontasy Publisher Italic Font 
    FZ                   AIRLINE City Data File (Efzed Pty. Ltd.)
    FZB                  Casio FZ-1 Bank Dump 
    FZD                  AIRLINE City Data File (Efzed Pty. Ltd.)
    FZF                  Casio FZ-1 Full Dump 
    FZF                  FontZip Font Packer 
    FZP                  Fargo Primera Color Printer Dye Sub Support File 
    FZV                  Casio FZ-1 Voice Dump 
    FZX                  CP/M Fix File 
    G                    Gravity News Database 
    G                    Ashton Tate Applause Chart 
    G                    Cliq Accessories Datebook Schedule Group 
    G                    Paradox File 
    G                    ANTLR Options File (Terence Parr and Colleagues)
    G                    Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    G++                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    G01                  MicroSim PCBoard Photoplot Artwork File (Cadence)
    G02                  MicroSim PCBoard Photoplot Artwork File (Cadence)
    G03                  MicroSim PCBoard Photoplot Artwork File (Cadence)
    G04                  MicroSim PCBoard Photoplot Artwork File (Cadence)
    G05                  MicroSim PCBoard Photoplot Artwork File (Cadence)
    G06                  MicroSim PCBoard Photoplot Artwork File (Cadence)
    G07                  MicroSim PCBoard Photoplot Artwork File (Cadence)
    G08                  MicroSim PCBoard Photoplot Artwork File (Cadence)
    G09                  MicroSim PCBoard Photoplot Artwork File (Cadence)
    G10                  MicroSim PCBoard Photoplot Artwork File (Cadence)
    G16                  GoldED / DOS Compiled Configuration 
    G3                   G3 FAX File 
    G32                  PaperPort Application Link File (32-bit) (ScanSoft Inc.)
    G32                  GFA BASIC 32 Program File 
    G3D                  GenDesigner 3.x Genealogy Database 
    G3F                  Zetafax Fine Resolution TIFF File (Equisys)
    G3N                  Zetafax Normal Resolution TIFF File (Equisys)
    G3S                  Goletas 3S Triangle Studio Project (Maksim Goleta)
    G4                   Access (Microsoft)
    G4                   GTX RasterCAD File 
    G42                  DLS/32 Supply Chain Planning (Adapta)
    G4B                  Sokoban++ Level File 
    G53                  DLS/32 Supply Chain Planning (Adapta)
    G64                  C64 Emulator Disk Image 
    G8                   PictureMaker Green Channel Image Data (Cubicomp)
    G8                   PicLab Plane Three Graphics 
    G9                   Gest Data File (Arktec, S.A.)
    GAB                  WINDEV Controls' Styles Description (PC SOFT)
    GAB                  Global Address Book Export File (Dencom)
    GAB                  Connectivity Memory Model Axo-axonic Connections Onto Presynaptic Terminals Input File 
    GAC                  Global Audio Control Skin 
    GADGET               Microsoft Gadget (Microsoft Corporation)
    GAF                  Total Annihilation 
    GAK                  Win32 Shellext File 
    GAL                  GenePix Array List (Molecular Devices Corporation)
    GAL                  Corel Multimedia Manager Album 
    GAL                  Gallery of Pictures 
    GAL                  Hewlett-Packard Graphics Gallery 
    GAM                  Dark Engine Games (Looking Glass Studio)
    GAM                  MacMolPlt GAMESS Input format 
    GAM                  Astro-PC Astrology Data File (Aureas Software & Publishing)
    GAM                  Klik'n'Play Game (Clickteam)
    GAM                  The Games Factory Game (Clickteam)
    GAM                  TADS Ver. 2.x Game File 
    GAM                  Vectrex Game 
    GAM                  Rise of Nations Saved Game (Microsoft)
    GAM                  Infinity Game Engine Saved Game (BioWare Corp.)
    GAM                  Baldur's Gate Game File 
    GAM                  RPG Toolkit Game Project (Christopher Matthews)
    GAM                  GammaFax Fax Document 
    GAME                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GAMES                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GAML                 Generalized Analytical Markup Language 
    GAN                  GanttProject Project Plan File 
    GANI                 Graal Game Animation 
    GAP                  Electrical Generation Analysis and Planification 
    GARDE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GAS                  Intelligence Tracking System Data File 
    GAS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GAS                  Grab-a-Site Project File (BlueSquirrel)
    GAT                  Gator File 
    GAT                  Genetic Algorithm Timetabler (GATTer) Configuration and Information File (Chris Skardon)
    GAU                  Gaussian Input File (MDL Information Systems, Inc.)
    GAU                  Flight Simulator Gauge (Microsoft)
    GAUSS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GAUSS2               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GAX                  Rise of Nations Saved Game (Microsoft)
    GAX                  Age of Empires II: The Conquerors Expansion Saved Game (Microsoft Corporation)
    GAY                  Reported as Sometimes a Porn File 
    GB                   CEDICT Chinese/English Dictionary (Erik E. Peterson)
    GB                   Printfox/Pagefox Graphic 
    GB                   Nintendo GameBoy Emulator ROM Image File 
    GB$                  BASIC VB Beispiel Kartei File 
    GB1                  Game Maker Backup (YoYo Games Limited)
    GBA                  Nintendo Game Boy Advance ROM Image 
    GBA                  GrabIt Batch Files 
    GBC                  Nintendo GameBoy Color Emulator ROM Image File 
    GBD                  Great Budget Budget Definition File (Great Budget Software)
    GBD                  Gator Banner File 
    GBF                  Great Budget Fringe Definition File (Great Budget Software)
    GBF                  InteGrade Pro Gradebook File 
    GBFS                 GBFS Archive (Pin Eight)
    GBK                  Interbase Database Backup 
    GBL                  Visual Basic Global Definition 
    GBL                  VAXTPU Global Definitions 
    GBOOK                The Guest Book Egbook file 
    GBR                  GIMP Brush File (The GIMP Team)
    GBR                  Gerber Format File 
    GBS                  Nintendo Gameboy Music/Audio Data 
    GBS                  Gameshark Online Save/Snapshot 
    GBSCR                GreenBlue Services Script (Ecyware)
    GBT                  Photoshop (Adobe)
    GBX                  Great Budget Budget File (Great Budget Software)
    GBX                  Gerber File 
    GBX                  Nadeo Games File Format (Nadeo)
    GC                   Sierra Print Artist Greeting Card 
    GC1                  Golden Common Lisp 1.1 Source Code 
    GC3                  Golden Common Lisp 3.1 Source Code 
    GCA                  GOCA Graphics File 
    GCA                  G Compression Archiver Compressed File 
    GCB                  Guitar Chord Buster 
    GCC                  Gnu C++ File 
    GCC                  VISUAL EPR COSEUL.EXE Result 
    GCD                  GraphiCAD Data File (GraphiTech Ltd.)
    GCD                  General Content Descriptor (Sprint Nextel)
    GCD                  CD Ghost Task 
    GCD                  Generic CADD Drawing 
    GCD                  GNUe Class Definition 
    GCD                  Prassi CD Image (Prassi Technology)
    GCF                  Half-Life Game Cache File (Sierra Entertainment, Inc.)
    GCF                  Global Virtual Accademy GVA XT Author File 
    GCF                  Scream! Guralp Compressed Format (Güralp Systems Ltd.)
    GCF                  WinXComp Grouped Compressed File 
    GCF                  Graphing Calculator (Pacific Tech)
    GCG                  GCG 8 Sequence Chemical File 
    GCI                  Visual EPR COSEUL.EXE Result 
    GCI                  GTA2 Game File 
    GCI                  Sonic Adventure DX Saved Game File (Sega of America, Inc.)
    GCK                  Giants: Citizen Kabuto Customized Map (Planet Moon Studios)
    GCK                  NICET Fire Alarm Test-Prep Software Tracking File (Zenith Design Group, Inc.)
    GCM                  Group Mail CMessage Store File 
    GCM                  GeoConcept Map File (GeoConcept SA)
    GCM                  GameCube Image File (Nintendo)
    GCP                  Spectramap Group Properties (Coloritto BV)
    GCP                  Ground Control Point File 
    GCR                  GlowCode Profiling Results (Electric Software, Inc.)
    GCR                  Visual EPR COSEUL.EXE Output 
    GCT                  ApplinX Screen Scrapper 
    GD2                  GDLib Image 
    GD3                  GreatFamily Family Tree Project (GreatProgs)
    GDA                  [Unknown] 
    GDB                  Group Mail File 
    GDB                  Interbase Database 
    GDB                  ProMaster Key Manager Key Management Database (Westorn House Software)
    GDB                  GVA2000 Author Lecture (Youngsan Info & Communication Co., Ltd.)
    GDB                  Formula 1 Car Race 2001 Season01 File 
    GDB                  GardenBoss Design File (RedBog Software)
    GDB                  MapSource GPS Waypoint Database (Garmin Ltd.)
    GDB                  GetPicturesList Image Location Database (Tracking-Celebs.info)
    GDB                  ACT! Group Data File (Best Software CRM Division)
    GDB                  MoneyWorks Main Database (GenSoft Ltd.)
    GDF                  FutuRUG Group Definition File (Two Brothers Software)
    GDF                  IBM Graphics Data Format Bitmap 
    GDF                  GlossTeX Definition File 
    GDF                  GRAFIK Eye Designer Project File (Lutron Electronics, Inc.)
    GDF                  3DMove 3D Geological Model (Midland Valley Exploration)
    GDF                  GEOS Dictionary 
    GDF                  Quartus II Graphic Design File (Altera Corporation)
    GDG                  ReliaSoft RG 
    GDK                  OS/2 Saved SKF Format File (IBM)
    GDM                  Bells, Whistles, and Sound Boards Module 
    GDP                  Scrolling Game Development Kit Project File (Benjamin Marty)
    GDR                  SymbianOS Bitmap Font File 
    GDR                  GRAFIK Eye Designer User Information (Lutron Electronics, Inc.)
    GDS                  Chip Layout Information 
    GDS                  Image File 
    GDS                  McDonnell-Douglas Things 
    GDT                  Guitar & Drum Trainer Song Configuration File (Renegade Minds)
    GDW                  GedStar Pro Configuration (GHCS Software)
    GE                   Gecho Configuration File 
    GE3                  Mastercam Mill Tool Library (CNC Software, Inc.)
    GE3                  Mastercam Geometry File (CNC Software, Inc.)
    GE80                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GEA                  Fifa World Cup Game Data fe Art Legalscr File 
    GEANT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GEAR                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GEB                  DR Hardware File 
    GED                  GEDCOM Family History File 
    GED                  Game Editor Project File (Game Editor)
    GED                  Micrografx Simply 3D Geometry 
    GED                  Arts & Letters Graphics (Arts & Letters Corporation)
    GED                  Family Historian Native Data Format (GEDCOM File) (Calico Pie Limited)
    GED                  Wicat Image 
    GED                  EnerGraphics File 
    GED                  GoldED / DOS Compiled Configuration File 
    GED                  Graphic Environment Document 
    GEF                  Graphics Exchange Format 
    GEF                  GE Industrial Systems CIMPLICITY Workbench 
    GEF                  Geotechnical Exchange Format (GEONET)
    GEM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GEM                  Ventura Publisher Vector Graphics 
    GEM                  GEM File 
    GEM                  Digital Research GEM Paint 
    GEN                  Generations Embroidery Software Design File (Notcina Corporation)
    GEN                  Chromeleon Generic Driver Configuration 
    GEN                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GEN                  ToGenBank Format 
    GEN                  ArcView ARC/INFO UnGenerate Format (ESRI)
    GEN                  dBASE Application Generator Compiled Template 
    GEN                  Ventura-Generated Text File 
    GEN                  Genius Family Tree Data 
    GENERAL              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GENIE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GENKEY               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GEO                  GeoSoft Compressed Add-on 
    GEO                  Geoworks Geode 
    GEO                  GeoPaint 
    GEO                  GoldED / OS/2 Compiled Configuration File 
    GEO                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GEO                  VideoScope File 
    GEO                  VideoScape 3D Object 
    GEO                  Homeworld Geometry File (Sierra)
    GEO                  Trumpf ToPs File 
    GEO                  eXplorist GPS Geocache Manager Database File (Thales Navigation, Inc.)
    GEOG                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GER                  GerberView File 
    GER                  German Text/HTML Info File 
    GER                  Pro/ENGINEER Temporary File (PTC)
    GERMAN               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GESTION              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GET                  Pov-ray for Windows (Persistence of Vision Raytracer Pty Ltd.)
    GET                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GETBILL              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GETRIGHT             GetRight Download in Progress (Headlight Software, Inc.)
    GETVERS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GEX                  GEcho Configuration File 
    GF                   Geometric Data 
    GF                   Starfleet Command - Empires At War File 
    GF                   Pilot Logbook for Grok File 
    GF                   METAFONT Generic Font File 
    GF                   gFx UNIX Plotting Program 
    GFA                  GFA-BASIC 32 ATARI Tokenized Source Ver. 2.x (Protected) (GFA Software Technologies)
    GFA                  GFA-BASIC MS-DOS Tokenized Source (GFA Software Technologies)
    GFA                  GFA-BASIC 32 ATARI Tokenized Source Ver. 2.x (GFA Software Technologies)
    GFA                  Bitmap Graphic 
    GFB                  GIFBlast Compressed GIF Image 
    GFC                  Patton & Patton Flowcharting 4 File 
    GFD                  Menus for GNUe Forms 
    GFD                  VISUAL EPR Energy, Fields File 
    GFD                  GeForms (Gadwin Systems, Inc.)
    GFF                  SignalMap Gene-Finding Format (GFF) Data File (NimbleGen Systems Inc.)
    GFI                  Genigraphics Graphics Link Presentation 
    GFL                  GeForms (Gadwin Systems, Inc.)
    GFM                  Computer Graphics Meta-file 
    GFM                  Ashton Tate Applause CGM Picture 
    GFO                  SGI Radiosity 
    GFS                  GGFileSPlit File Fragment (GGNetworks)
    GFS                  Immersioni Underwater Decompression File (Mario Giuseppe Leonardi)
    GFT                  NeoPaint Font 
    GFT                  GeForms Form Template (Gadwin Systems, Inc.)
    GFT                  GSP Family Tree 
    GFW                  BASIC GFA File 
    GFW                  ArcView World File For GIF Image (ESRI)
    GFX                  The Games Factory Extension (Clickteam)
    GFX                  WarCraft II Image File (Blizzard Entertainment)
    GFX                  Print Artist 
    GFX                  Instant Artist image 
    GFX                  Genigraphics Graphics Link Presentation 
    GFX                  Cue Club Image File 
    GFX                  RSView Machine Edition Graphics Display (Rockwell Automation, Inc.)
    GFY                  Grafical 1.0 File 
    GG                   Sega GameGear Game ROM Image 
    GG                   Google Desktop Gadget (Google)
    GG                   Sigmac Compiled Machine Code 
    GG                   Gridgen File 
    GG                   Koala Paint Compressed File 
    GG1                  Goofy Golf Deluxe Course 
    GG2                  Goofy Golf Deluxe Saved Game 
    GGDB                 GGNetworks Database (GGNetworks)
    GGF                  GPS Pathfinder Office Geoid Grid File (Trinble Navigation Limited)
    GGI                  GGI Extension 
    GGP                  GemCom Graphic Format 
    GGR                  GIMP Gradient (The GIMP Team)
    GGS                  Gameboy Emulator File 
    GHI                  NCSA Mosaic Configuration File 
    GHO                  Ghost Disk Image File (Symantec)
    GHOST                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GHS                  Tarshare File 
    GHS                  Ghost Disk Image Span File (Symantec Corporation)
    GHS                  Lasertank High Scores 
    GI                   Audio Utility Tuningwrench Compressed File 
    GI                   Sonic RecordNow! CD Image (Hewlett-Packard Development Company, L.P.)
    GI                   Prassi or PrimoDVD Image 
    GI                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GI                   Sony CD Extreme Global Image File 
    GIB                  Graph-in-the-Box Chart 
    GID                  GiD Project Directory (CIMNE (The International Center for Numerical Methods in Engineering))
    GID                  Windows Help Index File (Microsoft)
    GIF                  Graphic Interchange Format 
    GIF                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GIF2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GIF2                 Netobjects Fusion Styles Lines File 
    GIF87                GIF87 File 
    GIF89A               Gif 89a Image File 
    GIFA                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GIFENX               egis Encrypted GIF File (HiTRUST Inc.)
    GIFF                 Graphic Interchange Format 
    GIG                  GigaStudio Sound Bank (TASCAM)
    GIH                  GIMP Animated Brush (The GIMP Team)
    GIL                  GILS Document 
    GILS                 GILS Document 
    GIM                  Genigraphics Graphics Link Presentation 
    GINT                 Gintor Download Assistant Control File 
    GIO                  Nyquist V2.10 Test File 
    GIRL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GIS                  ERDAS IMAGINE Gray-scale Bitmap Image (Leica Geosystems , LLC)
    GIW                  Graph-in-the-Box Presentation 
    GIX                  Genigraphics Graphics Link Presentation 
    GJD                  GetJet Professional Database 
    GJD                  11th Hour Game Archive 
    GJD                  7th Guest Game Archive 
    GJF                  Gaussian Input Data File 
    GJT                  GAJET Input Data File (UK Health and Safety Laboratory)
    GKA                  GKWIN (Matsushita Electric Works UK Limited)
    GKB                  General Knowledge Base Database (Baltsoft)
    GKH                  Ensoniq EPS Family Disk Image (E-MU Systems)
    GKS                  Graphics Kernel System 
    GKS                  Gravis GripKey Document 
    GL                   GRASP - Graphical System for Presentation File 
    GL                   GALink Script 
    GL3                  Green Line File 
    GLADE                Glade User Interface Design 
    GLB                  Raptor: Call Of The Shadows Support File 
    GLB                  Global Module in Basic Program 
    GLD                  Glide File 
    GLD                  The GUI Loft Design (Johan Lindström)
    GLF                  Grantleigh Labels 
    GLF                  Golf Save File 
    GLF                  AfterGRASP Font (John Bridges)
    GLG                  MicroSim PCBoard Photoplot Job Description File 
    GLH                  Wise Installer Temporary File (Wise Solutions)
    GLK                  PaperPort Application Link File (16-bit) (ScanSoft Inc.)
    GLM                  DInsight Modeling Studio 3D CAD Model (DInsight)
    GLM                  Glim Data File 
    GLO                  Instant Text Glossary File (Textware Solutions)
    GLO                  LaTeX Auxiliary for Glossary 
    GLO                  Global Module in Basic Program 
    GLOBAL               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GLOBALS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GLOSS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GLOSSARY             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GLOSSARYASP          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GLS                  Across Data 
    GLS                  Sothink Glenda Export Shape File 
    GLS                  Babylon Builder 
    GLT                  Sothink Glenda File 
    GLU                  Cyberglue File 
    GLUE                 GlueMon Song/Module 
    GLUT                 OpenGL Glut-3.6 File 
    GLUT2                OpenGL Glut-3.6 File 
    GLUT3                OpenGL Glut-3.6 File 
    GLV                  Glove Data File (New Planet Software)
    GLY                  Word Glossary (Microsoft Corporation)
    GLY                  ACT! Layout (Best Software CRM Division)
    GM                   Autologic Bitmap 
    GM                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GM0                  S.A.P.S. - Sharp Advanced Presentation Software Professional 
    GM1                  Heroes of Might & Magic III Saved Game (The 3DO Company)
    GM1                  S.A.P.S. - Sharp Advanced Presentation Software Professional 
    GM2                  S.A.P.S. - Sharp Advanced Presentation Software Professional 
    GM2                  Autologic Bitmap 
    GM2                  Heroes of Might & Magic III Saved Game (The 3DO Company)
    GM3                  Heroes of Might & Magic III Saved Game (The 3DO Company)
    GM3                  S.A.P.S. - Sharp Advanced Presentation Software Professional 
    GM4                  S.A.P.S. - Sharp Advanced Presentation Software Professional 
    GM4                  Autologic Bitmap 
    GM4                  Heroes of Might & Magic III Saved Game (The 3DO Company)
    GM5                  Heroes of Might & Magic III Saved Game (The 3DO Company)
    GM6                  Heroes of Might & Magic III Saved Game (The 3DO Company)
    GM6                  Game Maker Project File (YoYo Games Limited)
    GM7                  Heroes of Might & Magic III Saved Game (The 3DO Company)
    GM8                  Heroes of Might & Magic III Saved Game (The 3DO Company)
    GM9                  Heroes of Might & Magic III Saved Game (The 3DO Company)
    GMA                  S.A.P.S. - Sharp Advanced Presentation Software Professional 
    GMANIFEST            Google Desktop Gadget Control File (Google)
    GMAX                 gmax Game Creator (Autodesk, Inc.)
    GMB                  Great Budget Model (Template) File (Great Budget Software)
    GMB                  GoldMine Business Contact Management Backup Data (FrontRange Solutions Inc.)
    GMD                  Game Maker Stored Game Under Development (YoYo Games Limited)
    GMD                  Great Budget Model Definition for the Template (Great Budget Software)
    GMD                  IMAGINE Graphical Model (Erdas)
    GME                  Multi Game Memory Editor Main Game File (Opencoding.net)
    GME                  Interact DexDrive Playstation Memory Card Save 
    GME                  GunBound Main GunBound Executable (Softnyx)
    GMF                  Applause CGM Graphics 
    GMF                  Great Budget Fringe Definition for the Template (Great Budget Software)
    GMF                  Robot Arena: Design & Destroy Graphic (Gabriel Group, Inc.)
    GMIN                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GMK                  Game Maker Project File (YoYo Games Limited)
    GML                  Geography Markup Language 
    GML                  Game Maker Script (YoYo Games Limited)
    GML                  NetRemote XML-based Configuration File 
    GML                  Graphlet Graphscript File 
    GMLWB                Garry's Mod Lua Weapons Builder Project File (Happymax)
    GML_BBCRSSINTCLUSTER BBC Alerts (BBC)
    GMM                  Group Mail Message Log File 
    GMO                  GNU Gettext GNU Machine Object File (Free Software Foundation)
    GMP                  Geomorph Tile Map 
    GMP                  Group Mail List Information File 
    GMP                  GTA2 Game File 
    GMP                  Railroad Tycoon 3 Scenarios and Saved Games (PopTop Software Inc.)
    GMR                  Schlafhorst Automation Graphical Monitor Record 
    GMR                  Game Maker Compiled Game (YoYo Games Limited)
    GMS                  Corel Graphics10 Draw File 
    GMS                  Corel Global Macro File 
    GMT                  Scarlett Expert System Compiler Grammar File 
    GMX                  Group Mail Message File 
    GNA                  Genigraphics Graphics Link Presentation 
    GNI                  VisiDAQ Strategy File (Visidaq Solutions, Inc.)
    GNM                  Generator GNM Output Log/Music 
    GNO                  Genopro Genealogy Document File (Genogram) (GenoPro Inc.)
    GNO                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GNT                  Micro Focus Generated COBOL Code 
    GNUMERIC             Gnumeric Default XML-based File (GNOME Foundation)
    GNUPLOT              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GNX                  Genigraphics Graphics Link Presentation 
    GO                   CompuServe (CompuServe)
    GO                   GraphOn Bitmap Image 
    GO1                  Wingames Go File 
    GOA                  SpiceCAD File 
    GOAL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GOB                  Dark Forces Game Archive (Lucasfilm Entertainment Company Ltd.)
    GOB                  Indiana Jones 3D Game Archive (Lucasfilm Entertainment Company Ltd.)
    GOB                  Jedi Knights Game Archive (Lucasfilm Entertainment Company Ltd.)
    GOBJ                 Geoworks Object Code 
    GOC                  Geoworks GOC Source Code 
    GOD                  Payvision BV Gamelauncher Games on Demand Volume (Payvision)
    GODD                 G.O.D Arena Partial Downloaded Game File (Games On Demand ( Australia ) Pty. Ltd. (G.O.D))
    GOE                  McIDAS System Satellite Image Data 
    GOES                 McIDAS System Satellite Image Data 
    GOGO                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GOH                  Geoworks GOC Header 
    GOLD                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GOLDFONT             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GOLF                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GOO                  Kai's Power Goo / Kai's Supergoo 
    GOODBYE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GOODE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GOODNESS             Goodness File 
    GOODS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GOOK                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GOR                  Gorilla 
    GOTHIC               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GOULD                Gould Scanner Bitmap 
    GOV                  Netmino File 
    GOX                  The Games Factory Extension (Clickteam)
    GP                   Gofer Project 
    GP                   Geoworks Glue Geode Parameter File 
    GP1                  Gerber Printed Circuit File 
    GP1                  Street Atlas GPS Log (Delorme)
    GP3                  Guitar Pro Ver. 3 Tablature (Arobas Music)
    GP3                  CCITT Group 3 File 
    GP4                  GTX Group IV Format /CALS Picture 
    GP4                  CCITT Group 4 File 
    GP4                  Guitar Pro Ver. 4 Tablature (Arobas Music)
    GP5                  Guitar Pro Ver. 5 Tablature (Arobas Music)
    GPD                  Generic Printer Description Unidrv Minidriver (Microsoft Corporation)
    GPD                  VISUAL EPR Input Data for PARAMS.EXE 
    GPF                  Ragnarok Online Gravity Re-Packed File (Gravity Co., Ltd. & Lee Myoungjin)
    GPG                  GNU Privacy Guard (GnuPG) Public Keyring (Free Software Foundation, Inc.)
    GPH                  Freelance Graphics for OS/2 
    GPH                  1-2-3 Graph (IBM)
    GPH                  Pro/ENGINEER (PTC)
    GPH                  Stata Graph 
    GPI                  Bitware Fax File 
    GPI                  Garmin Point of Interest File (Garmin Ltd.)
    GPJ                  jGRASP Project Metadata (Auburn University)
    GPJ                  Autorun-Autoplay Tools Project File (GS Technology Works LLC dba GS Technologies)
    GPJ                  gINT Project File (gINT Software, Inc.)
    GPJ                  Grapher Worksheet Project (Golden Software, Inc.)
    GPJ                  Xpower Project (PaulMace Software Inc.)
    GPK                  WaveLab Peak File (Steinberg)
    GPK                  Geos Compressed Omnigo File Archive 
    GPK                  Omnigo Program Package 
    GPL                  GetPicturesList URL Conversion Rule (Tracking-Celebs.info)
    GPL                  GPS Log File 
    GPL                  XMap GPS Log File (Delorme)
    GPL                  GenePix Lab Book (Molecular Devices Corporation)
    GPLT                 Gnuplot Plot Files 
    GPN                  GlidePlan Map Document (GlidePlan Inc.)
    GPP                  GraphPap Graph Paper File 
    GPP                  Serif GraphicsPlus Object 
    GPR                  VISUAL EPR PARAMS.EXE Results 
    GPR                  GenePix Results (Molecular Devices Corporation)
    GPR                  GMS Project File (Environmental Modeling Systems, Inc.)
    GPS                  Tripod Data Systmes Survey Pro GPS Coordinate File 
    GPS                  GenePix Settings (Molecular Devices Corporation)
    GPS                  MicroImages GPS Log File (MicroImages, Inc.)
    GPS                  BPM Studio File/Play List Structure (ALCATech)
    GPS                  Pocket CoPilot GPS Tracks Data File 
    GPT                  MOPAC Graph File (CambridgeSoft Corporation)
    GPT                  MicroImages Text Group (MicroImages, Inc.)
    GPX                  BASIS File 
    GPX                  GraphX Document (FlexSoft)
    GPX                  GPS eXchange Format (TopoGrafix)
    GPZ                  Msdev Common Ide Utility Projects File 
    GQ                   Epson Printer Page Description Language 
    GQ                   QLFC Compressed Archive 
    GQA                  BI/Query Data Model Admin Layer 
    GQL                  BI/Query Data Model 
    GQU                  BI/Query Data Model User Layer 
    GR                   XGMML (eXtensible Graph Markup and Modeling Language) File 
    GR2                  Granny Game Content File (RAD Game Tools, Inc.)
    GR2                  Windows 3.0 Screen Driver 
    GR3                  Windows 3.0 Screen Grabber 
    GR4                  Pathloss Network File Sharing File 
    GRA                  OpenGL Object 
    GRA                  STN Express Transcript Graphics File 
    GRA                  SigmaPlot Data File 
    GRA                  Graph Chart (Microsoft)
    GRA                  Flight Simulator File (Microsoft)
    GRAAL                Graal Level (Cyberjoueurs)
    GRABBER              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GRAD                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GRADE1FONT           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GRADS                Metafile 
    GRAFFLE              OmniGraffle Drawing (The Omni Group)
    GRAFSAO              Aleph One Shapes (Marathon Open Source)
    GRAPH                Amaya 
    GRAPHICS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GRAPHLI              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GRASP                Animation 
    GRAY                 Raw Gray Samples 
    GRB                  Gridded Binary 
    GRB                  MS-DOS Shell Monitor 
    GRB                  HP-48sx Calculator GROB Graphic Object 
    GRB                  HP-48/49 GROB 
    GRB                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GRBDROPFILE          GetRight (Headlight Software, Inc.)
    GRD                  Photoshop Gradient File (Adobe)
    GRD                  Grid File 
    GRD                  Gradebook Power File 
    GRD                  Golden Software Surfer 
    GRD                  Drivers for GRX 
    GRD                  StrongDisk Encrypted Disk Image (PhysTechSoft Ltd.)
    GRD                  3D Garden Composer Design File (DiComp, Inc.)
    GRDEF                Geoworks UI resource file 
    GRE                  Webalizer.01 Language File 
    GREY                 RAW RGB 24-bit Graphic 
    GRF                  Insight II Standard Graph Definition File (Accelrys Software Inc.)
    GRF                  Ragnarok Online Game Archive (Gravity Co., Ltd. & Lee Myoungjin)
    GRF                  Stanford Image 
    GRF                  Ragnarok Packed File (Gravity Co., Ltd.)
    GRF                  Graph Plus Drawing (Corel)
    GRF                  Level Up! Site Information File (Level Up!)
    GRF                  Graph 
    GRF                  DPlot Graph File 
    GRF                  Golden Software Graph File 
    GRF                  Grafix Idea File 
    GRF                  GraphEdit Filter Graph (Microsoft Corporation)
    GRF                  DLG Graphic Format 
    GRG                  GreatFamily Graphic Gallery (GreatProgs)
    GRIB                 Gridded Binary 
    GRID                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GRK                  Gradekeeper Class Gradebook (Daniel Ethier)
    GRK                  Kreon Scanner File 
    GRL                  MATLAB Graphic Format (The MathWorks, Inc.)
    GRM                  Minos File 
    GRM                  VISUAL EPR Input Data for FIntGraf 
    GRN                  Masterpoint Green Points File (ABF Masterpoint Centre)
    GRN                  Granny 3D Rendering Program 
    GRN                  Drivers for GRX 
    GRO                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GRO                  HP-48/48sx/49 Graphic Object Bitmap (Hewlett-Packard Development Company, L.P.)
    GRO                  Warlords Battlecry Archive 
    GRO                  Serious Sam Game File (Croteam Ltd.)
    GROB                 HP-48/48sx/49 Graphic Object Bitmap (Hewlett-Packard Development Company, L.P.)
    GRP                  Windows Program Manager Group 
    GRP                  ACT! Group Data File (Best Software CRM Division)
    GRP                  Starcraft Archive 
    GRP                  BPM Studio File/Play List Archive (ALCATech)
    GRP                  Shadow Warrior Archive 
    GRP                  DESQview (Symantec)
    GRP                  Duke Nukem 3D Archive 
    GRP                  PixBase Pictures Group 
    GRPH                 ANSYS Plot 
    GRQ                  Quickstep Barco Graphics 
    GRS                  Packedge Barco Graphics 
    GRS                  GetRight Skin (Headlight Software, Inc.)
    GRT                  VISUAL EPR Calculation Results of Fields 
    GRT                  MegaHAL Greeting File (Ai Research)
    GRV                  Groove Networks File 
    GRV                  Cakewalk (Cakewalk)
    GRW                  SeeYou Waypoint (Naviter)
    GRX                  GetRight File List (Headlight Software, Inc.)
    GRXML                XML Grammar 
    GRY                  RAW RGB 24-bit Graphic 
    GRZ                  GRZip Compressed Archive (Magic Software Online)
    GRZ                  GRZip II Compressed Archive (Magic Software Online)
    GS                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GS                   Gofer Script 
    GS                   eXplorist GPS Geocache Waypoint File (Thales Navigation, Inc.)
    GS1                  GraphShow Presentation 
    GS1                  Genecyst Save State 1 
    GS2                  Genecyst Save State 2 
    GS3                  Genecyst Save State 1 
    GS4                  Genecyst Save State 4 
    GS5                  Genecyst Save State 5 
    GS6                  Genecyst Save State 6 
    GS7                  Genecyst Save State 7 
    GS8                  Genecyst Save State 8 
    GS9                  Genecyst Save State 9 
    GSA                  GeneStudio Alignment File (GeneStudio, Inc.)
    GSAGA                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GSB                  GeneSys Battery File (Psytech International)
    GSBUG                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GSC                  GeneStudio File 
    GSC                  IMail Server Web Messaging File (Ipswitch, Inc.)
    GSD                  GraphTec Vector Graphic Data (GraphTec)
    GSD                  GSplit Piece File (G.D.G. Software)
    GSD                  Professional Draw Vector Graphics 
    GSD                  GSM Internet Realtime Audio 
    GSDB25               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GSDIN8               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GSE                  IMail Server Returned Mail (Ipswitch, Inc.)
    GSE                  GeneSys Export File (Psytech International)
    GSF                  Gedemin Settings File (Golden Software of Belarus, Ltd)
    GSF                  Golf Seasons File 
    GSF                  Multi Game Memory Editor Date File (Opencoding.net)
    GSHR                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GSI                  GeneSys Remote Administration File (Psytech International)
    GSLIB                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GSM                  Raw GSM 6.10 Audio Stream 
    GSM                  ArchiCAD Library Object (Graphisoft R&D Software Development Rt.)
    GSM                  GoldSim Model File (GoldSim Technology Group)
    GSM                  US Robotics Modem File GSM Audio over MODEM File 
    GSMA                 Glove Shack Mail Archive (The Glove Shack)
    GSN                  GeneSys Norm File (Psytech International)
    GSO                  WinOrganizer or GoldenSection Notes Database (The Golden Section Labs)
    GSOS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GSP                  Giza Specifier Project File (20-20 Technologies)
    GSP                  Geometer's Sketchpad Document File (Key Curriculum Press)
    GSP                  IMail Server Returned Mail (Ipswitch, Inc.)
    GSP                  GoldSim Input Model File (GoldSim Technology Group)
    GSPLUS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GSR                  GeneSys Report File (Psytech International)
    GSR6                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GSS                  Geometer's Sketchpad Script (Key Curriculum Press)
    GSS                  Multi Game Memory Editor Memory Snapshot (Opencoding.net)
    GST                  Giza Specifier Template File (20-20 Technologies)
    GST                  Multi Game Memory Editor Template File (Opencoding.net)
    GST                  MapInfo Geoset File 
    GST                  Need for Speed 
    GSV                  GameShark SP Gameboy Advance Game Save Info (Mad Catz, Inc.)
    GSV                  Genecyst SRAM ROM Dump 
    GSW                  GraphShow Worksheet 
    GSW                  GRDSOZ4W Gerdsooz Source File 
    GSX                  Gens 2.0 Save State File 
    GSX                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GSXL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GT2                  Graoumf Tracker Music Module 
    GTAG                 Jelbert GeoTagger Database File (Jelbert Consulting)
    GTAR                 GNU tar Compressed File Archive (GNU Tape Archive) 
    GTE                  GigaTrust RM-protected File (GigaMedia Access Corporation)
    GTF                  Tom Clancey's Ghost Recon Game File (Red Storm Entertainment)
    GTF                  PGA Golf Binary File 
    GTH                  Domino Document Manager (IBM)
    GTK                  Graoumf Tracker Music Module 
    GTM                  GPS TrackMaker General Map File (Geo Studio Technologies Ltd.)
    GTO                  Quicken On-line File (Intuit Inc.)
    GTP                  Atari Executable 
    GTP                  Guitar Pro Tablature (Arobas Music)
    GTP                  MultiFS Video File 
    GTS                  Genome Software Tempo Alarm Clock 
    GTT                  Guntram's Tabletweaving Thingy 
    GTV                  GigaTrust RM-protected File (GigaMedia Access Corporation)
    GTX                  GTX Group IV Raster Graphic 
    GTX                  Genetica Texture Pack (Spiral Graphics Inc.)
    GTY                  Gibson Tray File 
    GUI                  Doom User Interface Information (Id Software, Inc.)
    GUIDE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GUIDE                Amigaguide Hypertext Document (Amiga, Inc.)
    GUITAR               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GUL                  JungUm Global Viewer Korean Language File (Samsung Electronics)
    GUN                  Tom Clancey's Ghost Recon Gun Description File (Red Storm Entertainment)
    GUN                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GUN2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GUP                  PopMail Data 
    GUR                  Gutsch & Exner Software Update Undo 
    GUT                  ASSISTANT Data File (GUT GmbH)
    GUX                  Gutsch & Exner Software Update 
    GV                   Grandview 
    GV                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GVI                  Google Video File (Google)
    GVP                  Google Video Playlist (Google)
    GW3                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GWC                  Golfwits Course Map (Siscosoft, Inc.)
    GWF                  Benthic Golden 
    GWI                  Groupwise File 
    GWK                  GraphiCode Printed Circuit Board Drawing 
    GWM                  Epson Creative Studio 30 Example File 
    GWP                  Greetings WorkShop File 
    GWP                  GoodWay Flight Planner Flight Plan File (GoodWay)
    GWS                  GeoMedia Geoworkspace File 
    GWV                  Groundwater Vistas Packaged Model File (Environmental Simulations International Ltd (ESI))
    GWX                  Genigraphics Graphics Link Presentation 
    GWZ                  Genigraphics Graphics Link Presentation 
    GX                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    GX1                  Show Partner Graphics File 
    GX2                  Show Partner Graphics File 
    GXA                  General CADD Pro Attributes (General CADD Products, Inc.)
    GXC                  General CADD Pro Component (General CADD Products, Inc.)
    GXD                  General CADD Pro Drawing File (General CADD Products, Inc.)
    GXD                  GX-Reports File 
    GXD                  Jeol EX Spectrometer Data File 
    GXF                  Grid Exchange Format 
    GXF                  General CADD Pro Font (General CADD Products, Inc.)
    GXL                  Genus Graphics Library 
    GXM                  General CADD Pro Macro Script (General CADD Products, Inc.)
    GXP                  Jeol EX Spectrometer Data File 
    GXR                  Genexus Report Viewer 
    GXR                  GX-Reports File 
    GXS                  GrafiXML Native Format (Universite Catholique de Louvain-la-Neuve)
    GXT                  Grand Theft Auto II/III Text (Take-Two Interactive Software, Inc.)
    GYM                  Geoworks Generic Symbol File 
    GYM                  Sega Genesis/Mega Drive Music Logged Format (Sega)
    GZ                   Gzip Compressed Archive 
    GZ                   GIMP Image File (The GIMP Team)
    GZA                  GZA Compressed Archive 
    GZF                  Webcopier File 
    GZIP                 Gzip Compressed Archive 
    GZIP                 GNU Zip Compressed Archive 
    GZL                  Go!Zilla File 
    GZP                  Giants: Citizen Kabuto 
    H                    ADS Include File (Autodesk, Inc.)
    H                    Header 
    H                    Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    H!                   Flambeaux Tutorial Database 
    H++                  C++ Header File 
    H--                  Sphinx C-- Header File 
    H0                   Magix Musstu File 
    H1                   Magix Musstu File 
    H1                   Bruker Aspect NMR Data File 
    H16                  VC98 Include 16-bit File 
    H19                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    H2O                  Liquid Entertainment H2O Compressed Archive (Liquid Entertainment)
    H2O                  Water Web Program File (Clear Methods, Inc.)
    H32                  BC45 32-bit Include File 
    H3M                  Game Map File 
    H4C                  Heroes of Might and Magic Map File 
    H4R                  Possible Container for .MP3 Files 
    H5                   HDF5 File (The National Center for Supercomputing Applications)
    H5B                  SuperHeat 5 Building File (Survey Design Associates Ltd.)
    H6C                  SPP File 
    HA                   Compressed Archive 
    HAD                  MixlleniumEngine (Haydee) Web Server Module Style File 
    HAF                  HEAT Archive File 
    HAI                  NTgraph Nt09b-2b EMX Library File 
    HAIRY                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HAK                  GameHack File 
    HAK                  Neverwinter Nights File 
    HAL                  Mortyr Archive 
    HAL                  Hyper Access Lite OS/2 Data File 
    HALOGEN              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HAM                  Novell NetWare Disk Drivers NPA 
    HAM                  Amiga Hold and Modify Image (Amiga, Inc.)
    HAMMER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HAN                  Viewpoint Headline Studio 
    HAND                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HANOI                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HAP                  Hamarsoft HAP 3.x Compressed Archive 
    HAPTAR               People Putty Haptek Character (Haptek, Inc.)
    HARP                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HARPSAC              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HARPSAC2             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HARRIER              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HAT                  HAT (HOORA Analysis Tool) Model 
    HAT                  Formula 1 Car Race 2001 Log Hat File 
    HAY                  MixlleniumEngine (Haydee) Web Server Module Content File 
    HAZ                  Train Simulator Hazard File (Microsoft Corporation)
    HAZ                  Flight Simulator Texture File (Microsoft)
    HBB                  HAHTtalk Object File 
    HBC                  Home Buyers Calculator Suite 
    HBK                  Humanic Software Accounting Data 
    HBK                  Auto Backup Backup Archive (Han-soft Software)
    HBK                  HandyCafe Database File (Ates Software)
    HBK                  MathCAD Handbook (Mathsoft Engineering & Education Inc.)
    HBK                  Hex Workshop Hex Editor Bookmark File (BreakPoint Software, Inc.)
    HBM                  Hotline Book Mark (Hotsprings Inc.)
    HBX                  Starmoney Macro file 
    HBZ                  Starmoney Macro file 
    HC                   High Resolution Cantone TIFF Bitmap 
    HC                   Header File 
    HC                   BI/Suite Hypercube 
    HCC                  HydroCAD Stormwater Modeling System Storage Chamber Definition (HydroCAD Software Solutions LLC)
    HCD                  C Poet Examples Advanced Inc File 
    HCE                  HydroCAD Stormwater Modeling System Hydrograph File (HydroCAD Software Solutions LLC)
    HCF                  123 Heading Creator Project File (Mark Hendricks and Hunteridge, Inc.)
    HCI                  HydroCAD Stormwater Modeling System IDF File (HydroCAD Software Solutions LLC)
    HCL                  Handwritten Claims Log 
    HCM                  IBM HCM Configuration 
    HCOM                 Macintosh Sound 
    HCP                  HydroCAD Stormwater Modeling System Project File (HydroCAD Software Solutions LLC)
    HCR                  HydroCAD Stormwater Modeling System Rainfall Definition (HydroCAD Software Solutions LLC)
    HCR                  IBM HCD/HCM Production Configuration 
    HCR                  Half-fold Card File 
    HCSP                 Content Server Web Page (Stellent, Inc.)
    HCSS                 Hyper Color Swatch Set (SodaBush)
    HCT                  Symantec Anti-Virus Live Update File (Symantec Corporation)
    HCU                  HydroCAD Stormwater Modeling System Units Definition File (HydroCAD Software Solutions LLC)
    HCW                  HTML Help Workshop Conversion Wizard Macro (Microsoft)
    HCX                  Harvard Graphics Chart XL Chart (Harvard Graphics)
    HD                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HD2                  Curious Labs Poser Hand File 
    HDA                  HotDocs Auto-Assemble File (Matthew Bender & Company, a member of the LexisNexis Group)
    HDAT                 vStaticIP Data File (Phil Doyle a.k.a. MrJolly)
    HDAT                 Objective Analysis Package Data File 
    HDB                  Nero HD-Backup (Nero AG)
    HDB                  ACT! History File (Best Software CRM Division)
    HDC                  HD Tach Benchmark File (Simpli Software Inc.)
    HDC                  HelloDOC Medical Software File (IMAGINE Éditions)
    HDD                  HelloDOC Medical Software File (IMAGINE Éditions)
    HDD                  Parallels Desktop for Mac Hard Disk Drive Image (Parallels, Inc.)
    HDF                  Hierarchical Data Format File 
    HDF                  Procomm Plus Alternate Download File Listing 
    HDF                  Amiga Hardfile Image 
    HDF                  Help Development Kit Help File 
    HDI                  HelloDOC Medical Software File (IMAGINE Éditions)
    HDI                  AutoCAD Heidi Device Interface (Autodesk, Inc.)
    HDI                  HotDocs Auto-Install File (Matthew Bender & Company, a member of the LexisNexis Group)
    HDK                  MAXI Disk Disk Image (Herne Data Systems Ltd.)
    HDK                  OS/2 Saved SKF Format File (IBM)
    HDL                  Procomm Plus Alternate Download File Listing 
    HDL                  Giza Specifier Hidden Line Drawings (20-20 Technologies)
    HDL                  HotDocs Library File (Matthew Bender & Company, a member of the LexisNexis Group)
    HDM                  Handheld Markup Language File 
    HDML                 Handheld Markup Language File 
    HDMP                 WinXP Trouble Report 
    HDO                  Helpdesk-One File 
    HDP                  Magix Music/Video 
    HDR                  MicroSim PCBoard Hole-Drill Schedule Report 
    HDR                  InstallShield Setup Header (InstallShield Software Corporation)
    HDR                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HDR                  Photomatix HDR Image (HDRsoft)
    HDR                  PC-File+ Database Header 
    HDR                  Procomm Plus Message Header 
    HDR                  Photoshop (Adobe)
    HDR                  Egret Data File 
    HDR                  ArcView Header File (ESRI)
    HDR                  ArcInfo Binary (Environmental Systems Research Institute, Inc. (ESRI))
    HDR                  Apollo HDRU 
    HDRU                 Apollo HDRU 
    HDS                  Hierarchical Data System 
    HDS                  XMap Hierarchical Dataset (Delorme)
    HDV                  Formula 1 Car Race 2001 Season01 Vehicles Arrows File 
    HDW                  Harvard Graphics Draw Vector Graphics (Harvard Graphics)
    HDX                  Help Index 
    HEA                  Fifa 2001 Data Feart Legal Screeen 
    HEAD                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HEADER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HEADERS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HEARTS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HEARTSSAVE-MS        Hearts Saved Game (Microsoft Corporation)
    HEAVYSIGH            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HEC                  Grafic Pvquan Quant File 
    HED                  ReliaSoft RG 
    HED                  HighEdit Document 
    HED                  Hi-Eddi 
    HEH                  Scarlett Expert System Compiler Facts File 
    HEIDELB              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HEL                  Hellbender Saved Game (Microsoft)
    HELI                 RealFlight Radio Control Airplane Description (Knife Edge Software)
    HELLFONT             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HELLO                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HELP                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HELPER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HELPSCREEN           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HEP                  HostExplorer Telnet Session Profile (Hummingbird Ltd.)
    HEP                  Novell NetWare Help Librarian Data File 
    HER                  Grafic GIF File 
    HERALD               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HERBS1               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HERBS2               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HERBS3               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HERBS4               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HERBS5               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HERE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HERMETICA            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HESSIAN              Insight II Hessian File (Accelrys Software Inc.)
    HESSIANX             Insight II Hessian File (Accelrys Software Inc.)
    HEX                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HEX                  Hex Dump 
    HEX                  Smart Touch Analyze and Retrieve Tool (START) Classic Operator Software Update (HySecurity)
    HEX                  Magellan GPS Receiver Flash Upgrade File (2005 Thales Navigation, Inc.)
    HEX                  SPLITGRADE Upgrade File (Heiland electronic GmbH)
    HEX                  Macintosh BinHex 2.0 File 
    HEX                  HUGO Game File 
    HEY                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HEY2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HEYCOOLMAN           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HF                   HF File 
    HFA                  ERDAS IMAGINE Hierarchical File Architecture (Leica Geosystems , LLC)
    HFB                  Ralcgm File 
    HFB                  Hell Fighter Battle File (David Peterson)
    HFD                  HotDocs Form Document (Matthew Bender & Company, a member of the LexisNexis Group)
    HFI                  HP Font Info 
    HFO                  Hollywood FX Object File (Pinnacle Systems, Inc.)
    HFS                  Macintosh Disk Image on CD 
    HFT                  HotDocs Form Template (Matthew Bender & Company, a member of the LexisNexis Group)
    HFV                  Simulated Macintosh Volumes on PC Hard Drive 
    HFX                  US Robotics Rapid Comm Voice Data File 
    HFX                  Hollywood FX Special Effects File (Pinnacle Systems, Inc.)
    HFX                  Harvard Graphics F/X File (Harvard Graphics)
    HFX                  HotFax Fax (Smith Micro Software, Inc.)
    HFZ                  Pinnacle Studio HollywoodFX Plug-In Video Editing File (Pinnacle Systems, Inc.)
    HGH                  Star Wars - Tie Fighter High Score 
    HGL                  DN Highlight Groups Definition 
    HGL                  HP Graphics Language (Plotter File) 
    HGP                  HighGrow (Legal Marijuana Growing Game) Plant File (Slick Software)
    HGS                  HighGrow Virtual Seed File 
    HGT                  Shuttle Radar Topography Mission Data (SRTM) 
    HH                   Help Topic Map File 
    HH                   C++ Header 
    HH                   HammerHead Drumloop File 
    HHA                  HyperHub Archive (Oracle)
    HHC                  HTML Help Table of Contents (Microsoft)
    HHC                  TurboTax Contents File (Intuit)
    HHH                  Power C Precompiled Header 
    HHK                  HTML Help Index (Microsoft)
    HHL                  Visual Basic File 
    HHP                  HTML Help Workshop Project (Microsoft)
    HHS                  HTML Help Samples (Microsoft)
    HHS                  HyperHub Subkit (Oracle)
    HHTML                Realmedia Adstream HTML File 
    HI                   High Scores 
    HI                   Spy-CD CD Search Index File 
    HI1                  Spy-CD CD Search Database File 
    HI2                  Spy-CD CD Search Database File 
    HI4                  Spy-CD CD Search Database File 
    HIA                  Spy-CD CD Search Database File 
    HIB                  Spy-CD CD Search Database File 
    HIC                  Spy-CD CD Search Database File 
    HIF                  Quicken On-line File (Intuit Inc.)
    HIG                  High Scores 
    HIGH                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HIH                  Spy-CD CD Search Database File 
    HIHAT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HIM                  Spy-CD CD Search Database File 
    HIN                  Spy-CD CD Search Database File 
    HIN                  HyperChem Molecule File (Hypercube, Inc.)
    HINT                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HINTS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HINTS2               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HIP                  Spy-CD CD Search Database File 
    HIPS                 Bitmap Graphic 
    HIQ                  Spy-CD CD Search Database File 
    HIR                  Hidden Icon Resource 
    HIR                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HIR                  C64 Hires Bitmap Graphic 
    HIR2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HIS                  DN History of Executing/Viewing/Editing 
    HIS                  SOBEK Simulation History (WL | Delft Hydraulics)
    HIS                  Insight II Dynamics Trajectory History File (Accelrys Software Inc.)
    HIS                  Spy-CD CD Search Database File 
    HIST                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HISTORY              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HIT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HIT                  HitPlayer Audio File 
    HIT                  The Sims Sound File (Electronic Arts (Maxis))
    HIT                  Nonags Dialog Killer Data File 
    HIV                  RegShot Registry Hive File (TiANWEi)
    HIV                  Windows NT Registry Hive (Microsoft Corporation)
    HIW                  Spy-CD CD Search Database File 
    HIX                  System SYSUTIL File 
    HIZ                  Spy-CD CD Search Database File 
    HJT                  TreePad Hypertext Database File (Freebyte)
    HK5                  ACT! Database File (Best Software CRM Division)
    HKC                  HTML-KIT Auto Complete Short-cuts 
    HKI                  WinHki Compressed Archive (Hanspeter Imp)
    HKI1                 WinHki Compressed File 
    HKI2                 WinHki Compressed File 
    HKI3                 WinHki Compressed File 
    HKT                  egseing RB HK Data 
    HL$                  Compressed DOS Help File (Microsoft)
    HLB                  VAX Help Library 
    HLD                  Structural Desktop Saved Data (Hold File) (Structural Desktop, Inc.)
    HLF                  FAR Manager Help File (FAR Group)
    HLF                  BASIC QuickBAS QuickB01 File 
    HLI                  MOJO 620 Logo (TDK)
    HLL                  Boulderdash Game File 
    HLM                  Winhelp Vbhilfe File 
    HLN                  MicroStation Hidden Line File (Bentley Systems, Incorporated)
    HLP                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HLP                  Advisor Help File (Microsoft Corporation)
    HLP                  Windows Help File (Microsoft)
    HLP                  Stata Statistical Software Help File (Stata Corporation)
    HLP                  HP-95LX Help File 
    HLX                  Visual C++ Syntax Coloring Instructions (Microsoft)
    HLX                  Hyperlatex File 
    HLX                  ATI Radeon Video Driver Support File 
    HLX                  Hollex Cartridge Atari Cartridge Image File 
    HLX                  Multi-Edit Help File 
    HLZ                  Multi-Edit Packed Help File 
    HM                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HM                   Windows Help Context IDs used by MAKEHM.EXE. 
    HM2                  Help & Manual Help File Project 
    HM3                  Help & Manual Help File Project (ver 3) 
    HMA                  ATRAC3plus Encoded Audio File (Sony)
    HME                  Smartphone Theme File 
    HMF                  HOOPS Metafile (Tech Soft America)
    HMI                  Descent Human Machine Interfaces MIDI File 
    HMM                  Procomm Plus Alternate Mail Read Option Menu 
    HMP                  Descent Human Machine Interfaces MIDI File 
    HMP                  3D GameStudio Terrain Entity (Conitec Datasystems, Inc)
    HMP                  Frontier: First Encounters Music File 
    HMP                  Technics Sx KN 6000 Keyboard Home Page File 
    HMP                  Help Magician Pro Project (Stateline Software)
    HMR                  GeoTIFF Bitmap File (Bentley Systems, Incorporated)
    HMS                  HMS Archiware File 
    HMS                  Stentura 8000 or Stentura 8000LX Deposition File 
    HMS                  MS SMS Inventory File 
    HMT                  HighMAT (Microsoft Corporation)
    HMW                  Homeword for DeskMate 
    HMX                  MixlleniumEngine (Haydee) Web Server Module Template File 
    HM~                  Help&Manual Backup Help File Project 
    HNC                  CNC Program File Heidenhain Dialog 
    HNC                  NoteCenter Note File (Domas Savickas)
    HND                  Hallo Northern Sky (HNSKY) Deep Sky Database File (Han Kleijn)
    HNE                  NoteCenter Encrypted Note File (Domas Savickas)
    HNF                  Humax Receiver Settings 
    HNS                  Hallo Northern Sky (HNSKY) Settings (Han Kleijn)
    HNV                  TheWord & Bible Companion Hebrew Names Version Bible 
    HO                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HO                   Magix Music/Video 
    HO                   Knowledge Explorer Knowledge Representation Hierarchy Outline 
    HOB                  Star Trek: Birth of the Federation Ship Graphic (Atari)
    HOC                  High Octane Race Course (Fishstick Games)
    HOCKEY               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HOD                  Homeworld Hardpoint/Object Data (Sierra Entertainment, Inc.)
    HOF                  Hall Of Fame 
    HOG                  Descent Mission File 
    HOG                  Lucas Arts Dark Forces WAD File 
    HOL                  Personal Paint Dutch Language User Interface File (Amiga)
    HOLES                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HOLLY                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HOME                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HOME1                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HOME2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HOOD                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HOR                  Horoscope 
    HORIZON1             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HORIZON2             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HORN                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HORNPIPE             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HOT                  eFax Messenger Fax (j2 Global Communications, Inc.)
    HOT                  The Sims Sound File (Electronic Arts (Maxis))
    HOT                  Mosaic Internet Address 
    HOTFONT              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HOTSPOT              Homeworld Cursor Informaiton (Sierra Entertainment, Inc.)
    HOW                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HP                   Thor Database Primary Hash File 
    HP                   HP Graphics Language (Plotter) 
    HP$                  NTgraph Visual C Wizzard File 
    HP-                  HP Distribution Binary File 
    HP-UX                HP-UNIX File 
    HP8                  HP NewWave Write ASCII Roman8 Character Set 
    HPD                  HP Document (Hewlett-Packard Development Company, L.P.)
    HPD                  Harry Potter Related 
    HPD                  HotDocs PDF Document (Matthew Bender & Company, a member of the LexisNexis Group)
    HPF                  NetHelp2 Project File (Netscape)
    HPF                  PageMaker HP LaserJet Font (Adobe)
    HPF                  HotLine Partial Download File 
    HPG                  HP Graphics Language (Plotter) 
    HPGL                 HP Graphics Language (Plotter) 
    HPH                  Designer Graphics System2 File 
    HPI                  GEM Font Information 
    HPI                  Total Annihilation Saved Game 
    HPI                  Hemera Photo-Object Image (Hemera Technologies Inc.)
    HPIC                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HPJ                  Visual Basic Help Project 
    HPJ                  Help Project File 
    HPJ                  HP LaserJet Print-to-File 
    HPK                  HPXUpgrade Service Pack Fileset (Haliplex Communication Systems)
    HPK                  Compressed Archive 
    HPL                  HP Graphics File 
    HPLJ                 Hewlett-Packerd LaserJet Vector Image 
    HPM                  Quaestor 1.x High Performance Map 
    HPM                  Procomm Plus Alternative Menu 
    HPM                  Procomm Plus Alternate Main Menu for Privileged Users 
    HPM                  HP NewWave Emm Text 
    HPP                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HPP                  Zortech C++ Header 
    HPP                  C++ Program Header 
    HPP                  C++ Builder 6 Program Header (Borland Software Corporation)
    HPP                  InfoMgr Database File 
    HPP                  Printer Control Language Vector Image (Hewlett-Packard)
    HPPCL                Hewlett-Packard Printer Control Language Vector Image 
    HPR                  ATLAS.ti 4.x Project File (Scientific Software Development)
    HPR                  RoboHELP Configuration File 
    HPR5                 ATLAS.ti 5.0 Project File (Scientific Software Development)
    HPRJ                 Heroglyph Project File (proDAD GmbH)
    HPRT                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HPT                  HotDocs PDF Template (Matthew Bender & Company, a member of the LexisNexis Group)
    HPUX                 HP-UNIX File 
    HPW                  CompuServe Home Page Wizard 
    HP_                  Winhelp Compressed File 
    HP~                  V-help File 
    HQP                  CP/M Disc Utility Information 
    HQX                  Macintosh BinHex 4 Compressed Archive 
    HR                   TRS 80 Graphic 
    HR2                  Curious Labs Poser Hair File 
    HR5                  Heredis 2000 Family History File (BSD CONCEPT)
    HRC                  SoftImage Model Geometry File 
    HRF                  Hitachi Raster Format Graphic 
    HRH                  C++ and Resources Common Header 
    HRL                  Herolab EasyWin32 
    HRM                  Procomm Plus Alternate Main Menu 
    HRM                  Netmino File 
    HRP                  Handy Recovery Recovered Partion Information (SoftLogica LLC.)
    HRT                  Hornet Packer Module/Song 
    HRU                  HRU Bitmap 
    HRV                  Personal Paint Croatian Language User Interface File (Amiga)
    HRX                  Herolab EasyWin32 
    HRZ                  Slow Scan Television (SSTV) 
    HS                   RAZR Flex File (Motorola, Inc.)
    HS                   Hugs Source Code 
    HS                   Literate Haskell Source Code File 
    HS                   HSnapshot Screen Snapshot (PCNetWork)
    HS2                  Postering Monochrome Image 
    HS2                  HyperRESEARCH (ResearchWare, Inc.)
    HS3                  HyperRESEARCH (ResearchWare, Inc.)
    HSB                  HALion Sampler File (Steinberg Media Technologies GmbH.)
    HSB                  HandStory Suite eBook File (SJ NAMO, Inc.)
    HSBOY                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HSC                  HSC Tracker 2-op FM Music File 
    HSC                  HelpScribble Project Source (JGsoft - Just Great Software)
    HSC                  HYSYS Simulation (Aspen Technology, Inc.)
    HSD                  HSD+CP Aero-HSD Configuration File (Thrane & Thrane)
    HSE                  Dreamweaver Snippet 
    HSF                  HOOPS Stream File (Tech Soft America)
    HSH                  IIS NNTP Hash File (Microsoft)
    HSI                  Handmade Software Graphic File 
    HSI                  HSI JPEG Bitmap 
    HSK                  Nimbus 7 SAMS Data File 
    HSM                  Assembly Language Header - University of Waterloo Engineering Convention 
    HSM                  The Sims Sound File (Electronic Arts (Maxis))
    HSP                  HSC Packed 
    HSP                  HomeSite WWW Page Project 
    HSQ                  CAST Application Mining Suite 
    HSQ                  Data File Associated with Qaz Trojan 
    HSR                  Hero Screen Recorder Screen Capture File (Beijing Herosoft Zongheng Network Technology Co., Ltd)
    HSR                  RezeptAssist Import and Export File 
    HSS                  Dreamweaver Snippet 
    HSS                  Photoshop Hue/Saturation Information (Adobe)
    HST                  Procomm Plus History File 
    HST                  WordPerfect (Corel Corporation)
    HST                  History File 
    HST                  Image Alchemy Handmade Software Graphics Histogram 
    HST                  Corel QuickCorrect and Speller History File 
    HST                  Host File 
    HST                  Netscape History File 
    HST                  Symantec Anti-Virus Live Update File (Symantec Corporation)
    HST                  Yahoo! Instant Messenger History 
    HSZ                  Handyscript Skin (Ventris, Inc.)
    HT                   HyperTerminal Data File (Hilgraeve Inc.)
    HT                   BrailleNote Visual Display File (Pulse Data International Ltd.)
    HT3                  HTML File 
    HTA                  Hemera Thumbnail Archive (Hemera Technologies Inc.)
    HTA                  Tonline BSW4 File 
    HTA                  Hypertext Application 
    HTACCESS             Apache Access Configuration 
    HTB                  HttpBrowser Database File 
    HTC                  HTML Component 
    HTD                  HyperTRANSCRIBE Project File (ResearchWare, Inc.)
    HTF                  HTF Sounding File 
    HTF                  Virtual Hypertext Font 
    HTF                  WebBase File 
    HTI                  Win Help Related File 
    HTI                  WebBase File 
    HTM                  Hypertext Markup Language 
    HTMENX               egis Encrypted HTM File (HiTRUST Inc.)
    HTML                 Hypertext Markup Language 
    HTML                 Mozilla (Netscape) Bookmark File (Mozilla.org)
    HTMLENX              egis Encrypted HTML File (HiTRUST Inc.)
    HTMLS                Server-side HTML File 
    HTN                  Micrografx Picture Publisher 8 printstyles file 
    HTR                  Motion Analysis Software Skeletal File 
    HTR                  People Putty Haptek Character (Haptek, Inc.)
    HTRI                 HTRI Xchanger Suite Data Input File (Heat Transfer Research, Inc.)
    HTS                  Troj/DelSpy-E 
    HTT                  Microsoft Hypertext Template 
    HTX                  MS FrontPage Database Results HTML-like query script. 
    HTX                  Extended Hypertext Template 
    HTX                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HTZ                  HTML Editor Archive 
    HU                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HUF                  Huffman Encoded/Compressed File 
    HUH                  HydroCAD Stormwater Modeling System Unit Hydrograph (HydroCAD Software Solutions LLC)
    HULL                 Halflife Map Collision Hull File 
    HUN                  Personal Paint Hungarian Language User Interface File (Amiga)
    HUN                  Dataview File 
    HUS                  Husqvarna Designer I Embroidery Machine Format (Husqvarna Viking)
    HUY                  Eto Dlia Vas 
    HVA                  Tiberian Sun Voxel Animation Control File (TiberiumSun.com)
    HVD                  chosenOS HoverNews Skin File 
    HVD                  CMT Surveyor's Assistant 1.x/2.0x Job 
    HVD                  HerdView Event Data File 
    HVD                  ISIS Hviews File 
    HVD                  HoverDesk Theme File (Hover Inc.)
    HVX                  LightWave 3D Hypervoxel (NewTek)
    HW                   Stanford University EPGY Homework 
    HWCFG                VersaPro Hardware Configuration File 
    HWD                  Hollywood Presentation File 
    HWL                  Hangman Deluxe Word List 
    HWL                  Corel Shared Writing Tools 9.0 File 
    HWL                  HttpWatch HTTP Watch Log (Simtec Limited)
    HWM                  Hillwalker/MapWise/Alpiniste/PhotoMaps Own Raster Map Parameters (ISYS Computers)
    HWP                  Hangul Word Processor File (Haansoft)
    HWPROJ               Visual Studio .NET Help Project (Microsoft)
    HWR                  Hillwalker/MapWise/Alpiniste/PhotoMaps ISYS Route Card (ISYS Computers)
    HX                   THOR Database Cross-reference Hash File 
    HX                   SurRender3D Database File 
    HX                   EARS Component File 
    HXC                  Help 2 Project/Collection File (Microsoft)
    HXCONV               Huelix Audio Converter Session Information (Huelix Solutions Private Ltd)
    HXD                  Help 2 Validator File (Microsoft)
    HXF                  Help 2 Include File (Microsoft)
    HXG                  Help 2 Filter Cache (Microsoft)
    HXH                  Help 2 Merged Hierarchy (Microsoft)
    HXI                  Help 2 Compiled Help File (Microsoft)
    HXK                  Help 2 Keyword Index (Microsoft)
    HXK                  Help TOC/Index File 
    HXL                  Help 2 Registered Namespaces List (Microsoft)
    HXM                  Procomm Plus Alternate Protocol Selection Menu for All Users 
    HXM                  Descent2 HAM File 
    HXS                  Help 2 Compiled Help File (Microsoft Corporation)
    HXT                  Help TOC/Index File 
    HXT                  Help 2 Table of Contents (Microsoft)
    HXV                  Help 2 Virtual Topic Definition (Microsoft)
    HXW                  Help 2 Merged Keyword Index (Microsoft)
    HXX                  C++ Header 
    HX_                  C Poet Compressed Disk1 File 
    HY1                  Ventura Publisher Hyphenation Algorithm 
    HY2                  Ventura Publisher Hyphenation Algorithm 
    HYC                  WordPerfect Hyphenation File (Corel Corporation)
    HYD                  WordPerfect for Windows Hyphenation Dictionary (Corel Corporation)
    HYM                  Scandata 3D Scanner File (Hymarc, Inc.)
    HYP                  Hyphenation File 
    HYP                  Hypertext File 
    HYP                  Hyper Compressed Archive 
    HYP                  Acrobat Spelling File (Adobe)
    HYPERCARD            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HYPERXCMD            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HYPHENATOR           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    HYS                  Globe HySales Client File (Globe Interactive)
    HYS                  Smart Touch Analyze and Retrieve Tool (START) New Generation Operator Software Update (HySecurity)
    HYS                  Multi-Parametric Toolbox HYSDEL File (Eidgenössische Technische Hochschule Zürich)
    HYS                  TriboIndenter Air Indent Electrostatic Force Constant Setting (Hysitron Incorporated)
    HZ                   Chinese Text 
    H_                   Winhelp Compressed File 
    H__                  C++ Header Seldom used C++ Header (same as H++ and H) 
    I                    INTERCAL File 
    I                    Integra Database (Imagina Software)
    I                    C++ Preprocessor Intermediate File (Borland Software Corporation)
    I                    Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    I                    Phase Vocorder Analysis Data 
    I                    APLAC Source Code Data File (APLAC)
    I                    Progress Database Include File (Progress Software Corporation)
    I-E                  i-Enforce Security File (ICM Conversions & Management)
    I0                   Winter Windows Scheduler File 
    I00                  (100-103) Winphone Phonebook 
    I16                  Nokia Phone Logo File (Nokia)
    I17                  Pixel Power Collage 
    I18                  Pixel Power Collage 
    I1C                  IBM Voice Type Language Newuser File 
    I1I                  IBM Voice Type Language Newuser File 
    I2C                  IBM Voice Type Language Newuser File 
    I2I                  IBM Voice Type Language Newuser File 
    I2S                  Invision for mIRC Settings 
    I3                   Modula-3 Interface Source 
    I3C                  IBM Voice Type Language Newuser File 
    I3F                  I3 Fax File (Interactive Intelligence Inc.)
    I3I                  IBM Voice Type Language Newuser File 
    I3ML                 I3ML Source Page 
    I66                  Interactive66 Graphic Design Information 
    IA                   Roland Garros Tennis Person File 
    IA6                  IApproach Data File (Workfile) (Tdata, Inc.)
    IAF                  IAFA Document 
    IAF                  Outlook 97 and 2000 E-mail Account Settings (Microsoft)
    IAF                  Quikscribe Intelligent Audio File (Quikscribe Pty Ltd)
    IAF                  Informer Configuration (Dataconomy)
    IAF                  Outlook Express Account Settings (Microsoft)
    IAFA                 IAFA Document 
    IAM                  Inventor CAD (Autodesk, Inc.)
    IAN                  Sterling Software COOL Business Team Model File 
    IAP                  TeamBG File 
    IAP                  ASM Palette 
    IAP                  Iguana Active Server Active Page Script (ILan Game)
    IAR                  CD Katalog (KRISoftware)
    IAS                  Iguana Active Server Script (ILan Game)
    IAX                  IBM Image Access Executive Bitmap 
    IB7                  Pixel Power Collage 
    IBA                  InfoHandler Infobase File (MDE Software)
    IBA                  IBasic Source Code File 
    IBA                  Image 
    IBC                  IncrediBar Cursor File (IncrediMail Ltd.)
    IBD                  Windows Installer File (Microsoft)
    IBD                  PDS Picture 
    IBF                  Instant Backup (Towodo Software)
    IBG                  iBridge Project Workspace (i2 Inc.)
    IBG                  NASA PDS Graphic Format 
    IBG                  Bert's Dinosaurs File 
    IBK                  Sound Blaster Instrument Bank (Creative Technology Ltd.)
    IBK                  National Construction Estimator 32 Picture File 
    IBK                  (Kodak) Imaging for Windows Bookmark (Microsoft Corporation)
    IBL                  IBasic Component Language File 
    IBL                  PCLTool Index-by-Location File (Page Technology Marketing, Inc.)
    IBM                  Archdos Compressed File 
    IBM                  Commodore Amiga Graphics 
    IBP                  IsoBuster CD/DVD Layout File (Smart Projects)
    IBQ                  IsoBuster Optical Media Data Container File (Smart Projects)
    IBS                  IncrediBar Skin (IncrediMail Ltd.)
    IBS                  i2 iBase File 
    IBT                  IBTracer Infiniband Link And Protocol Analyzer InfiniBand Trace File (Computer Access Technology Corporation)
    IBX                  Interscope BlackBox File 
    IBY                  ROM Image Component Include File 
    IC                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IC                   Spice Initial Condition 
    IC                   Modula-3 M3CG Intermediate Language File 
    IC1                  Imagic Bitmap Graphic 
    IC1                  ZyIMAGE Index 
    IC2                  IMagic Bitmap Graphic 
    IC3                  IMagic Bitmap Graphic 
    ICA                  Identity Compass Encrypted Answers (Identity Compass International GmbH.)
    ICA                  Image Object Content Architecture (IOCA) Graphics File 
    ICA                  Citrix Independent Computer Architecture File (Citrix)
    ICAL                 iCal Personal Calendar (Apple Computer, Inc.)
    ICB                  Image Capture Board 
    ICB                  Targa Bitmap 
    ICC                  Kodak Printer Image 
    ICC                  IronCAD Catalog 
    ICC                  Apple ColorSync 2.0 (Apple Computer, Inc.)
    ICC                  Animation Softdisk 
    ICC                  ICC Profile Format File 
    ICCP                 Identity Compass Encrypted Culture Profile (Identity Compass International GmbH.)
    ICD                  Rayman2 Game File 
    ICD                  Rate Structure File 
    ICD                  IronCAD 2D CAD File 
    ICD                  File Associated with Many Games 
    ICD                  ASTi Model Builder 
    ICD                  SafeDisc Encrypted Digital Rights Management File (Macrovision Corporation)
    ICDX                 Home Organizer Document (Imagery Concepts, LLC)
    ICDX                 FilePic Document (Imagery Concepts, LLC)
    ICE                  Cooltalk Audio 
    ICE                  Freeze! Compressed Archive (Reeve Soft)
    ICE                  Archive 
    ICE                  ICEOWS Compressed File (Raphaël et Béatrice Mounier)
    ICEICEBABY           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ICF                  Cheyenne AntiVirus 
    ICF                  RapidForm INUS Compression Format File (INUS Technology, Inc.)
    ICF                  ICOM CS-2100 Radio Programming Data 
    ICF                  INI Creator for X10 Remote Control 
    ICF                  Insane Chat Saved Conversation 
    ICL                  Clean File 
    ICL                  Icon Library File 
    ICM                  Image Color Matching Profile File (Kodak)
    ICM                  ICC Profile, Acer Monitor Drive 
    ICM                  Earl F. Glynn Color Tablecolor Table 
    ICM                  Windows Color Matching Control (Microsoft Corporation)
    ICMP                 Internet Control Message Protocol 
    ICMS                 Ragtime Extensions File 
    ICN                  HP Palmtop 100/200LX Icon 
    ICN                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ICN                  Micrografx Icon 
    ICN                  AT&T Graphic Format 
    ICN                  Bert's Dinosaurs File 
    ICN                  Icon Programming Language Source Code File 
    ICN                  RIPTerm Image 
    ICN                  SunOS Mono Icon 
    ICNS                 Macintosh OS X Icon File (Apple Inc.)
    ICO                  GIMP Icon (The GIMP Team)
    ICO                  Sun Icon/Cursor 
    ICO                  OS/2 Icon 
    ICO                  Windows Icon (Microsoft Corporation)
    ICON                 Sun Icon/Cursor 
    ICON                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ICON2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ICONES               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ICONS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ICONS1               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ICONS2               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ICP                  Identity Compass Encrypted Profile (Identity Compass International GmbH.)
    ICP                  UNIX Icontact Parameter File 
    ICP                  InterCept File 
    ICQ                  ICQ Saved File (ICQ Inc.)
    ICR                  NCSA Telnet Interactive Color Raster Graphic 
    ICS                  SwiftView ICS Command File (SwiftView, Inc.)
    ICS                  Image Cytometry 
    ICS                  IronCAD 3D CAD File 
    ICS                  iCalendar Calendar Data 
    ICS                  Outlook Calendar File (Microsoft)
    ICS                  CopyControl Generic Copier Floppy Disk Image (Microcosm Ltd.)
    ICS                  Inovate 3D CAD File 
    ICT                  TIFF and ISO Image Related File 
    ICTX                 Home Organizer Transport Container (Imagery Concepts, LLC)
    ICTX                 FilePic Transport Container (Imagery Concepts, LLC)
    ICV                  In Chair Viewing 
    ICV                  Bond .AVI Film 
    ICW                  InterCAD File 
    ICW                  Icon 
    ICW                  MS Internet Explorer Internet Connect Wizard 
    ICX                  More Than Words File 
    ICX                  VisualAge Smalltalk Enterprise Program Image File (IBM)
    IC_                  Compressed LCT File 
    ID                   AOL Instant Messenger (AIM) Buddy Icon (America Online, Inc.)
    ID                   Pegasus Mail (David Harris)
    ID                   Gamecode 
    ID                   MS Data Map (Microsoft)
    ID                   Notes ID File (IBM)
    ID                   Disk Identification File 
    ID                   3D Graphic 
    ID                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ID3                  ID3 Resource 
    IDA                  IIS Server File (Microsoft)
    IDAPI                Integrated Database Application Programming Interface (Borland Software Corporation)
    IDB                  A-B-CD Info File 
    IDB                  Database Used by Disassembler 
    IDB                  Delphi Pascal Object File 
    IDB                  Developer Intermediate File (Microsoft)
    IDB                  IDA Database 
    IDB                  MS Developer Intermediate MDPX File 
    IDC                  Structured Query Language (SQL) 
    IDC                  Core IDC 
    IDC                  IDA C-like Language Source Code 
    IDC                  Internet Database Connector Document 
    IDD                  Complete Works Database-related File (Toplevel Computing)
    IDD                  MIDI Instrument Definition 
    IDE                  ideCAD Structural Structural 3D Design Model (ideYAPI Ltd. Sti.)
    IDE                  C++ Project (Borland Software Corporation)
    IDE                  ideCAD Architectural Architectural 3D Design Model (ideYAPI Ltd. Sti.)
    IDE                  Grand Theft Auto III Model IDs and Properties (Take-Two Interactive Software, Inc.)
    IDE                  Sophos Anti-Virus Virus Identity File 
    IDE                  Integrated Development Environment Configuration File 
    IDEA                 Brainbloom Document (Gael Limited)
    IDEAS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IDEMO                Instant Demo Project File (NetPlay Software)
    IDENTIFIER           Internet Explorer Zone Identification Information (Microsoft Corporation)
    IDF                  Printed Circuit Board Data Exchange File 
    IDF                  ARTiSAN Real-time Studio ID 
    IDF                  Identification File 
    IDF                  MIDI Instruments Definition File (Microsoft)
    IDI                  i-deas Web ACCESS File 
    IDIF                 Netscape Saved Address Book 
    IDK                  OS/2 Saved SKF Format File (IBM)
    IDL                  Visual C++ Interface Definition File (Microsoft)
    IDL                  CRiSP Harvest File 
    IDL                  OMG CORBA Interface Definition Language 
    IDL                  Snappy Devdocs 
    IDLK                 InDesign Lock File (Adobe Systems Inc.)
    IDM                  Fahrenheit The Game - DAT + IDM Archive (Quantic Dream)
    IDM                  Ulead Photo Express Messages File 
    IDMEF                Snort File 
    IDQ                  Internet Data Query File 
    IDQ                  Structured Query Language (SQL) Query 
    IDR                  CompactDRAW Object(s) (MediaChance)
    IDS                  Minos 3D Data File 
    IDS                  IDA Imported Names Format 
    IDS                  Infini-D Scene File 
    IDS                  Infinity Game Engine Numbers to Description Map (BioWare Corp.)
    IDS                  WFO-A Data 
    IDT                  Text Archive Table 
    IDT                  Windows Installer File (Microsoft)
    IDT                  Identification 
    IDT                  Output of DLL2IDT Utility 
    IDU                  Interleaf Desktop Utility (BroadVision)
    IDV                  Inventor Design View Tree (Autodesk, Inc.)
    IDW                  IntelliDraw Vector Graphic (Adobe)
    IDW                  Inventor CAD (Autodesk, Inc.)
    IDWORDCOUNT          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IDX                  RCA/Lyra Handheld MP3 Player Database Index (RCA)
    IDX                  Ca Visual Objects Platform for Developer (CAVO) Index File (CA)
    IDX                  Clip Gallery 1.x Index (Microsoft)
    IDX                  Java Applet Cache Index (Sun Microsystems, Inc.)
    IDX                  Complete Works Index File (Toplevel Computing)
    IDX                  NoX 
    IDX                  LaTeX Index 
    IDX                  AOL Temporary Internet Mail File 
    IDX                  Symantec Q&A Relational Database Index File (Symantec Corporation)
    IDX                  Stork Format Header Information 
    IDX                  Outlook Express Mailbox Index (Microsoft)
    IDX                  Pro/ENGINEER Index File (PTC)
    IDX                  Index 
    IDX                  ICQ Index (ICQ Inc.)
    IDX                  FoxPro Index (Microsoft)
    IDX                  FAX File 
    IDX                  Corel QuickFinder Information 
    IDX                  ArcView Geocoding Index For Read-Only Datasets (ESRI)
    IDY                  Debug Information File 
    IDY                  Index File 
    IDZ                  i-deas Web ACCESS File 
    ID_                  C Poet Compressed Disk1 File 
    IE                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IE1-3                Internet Explorer 3 Address Book 
    IE3                  Internet Explorer 3 File 
    IEE-695              IEEE 695 Information (Hewlett Packard)
    IEF                  Image File 
    IEL                  Erazer98 File 
    IEM                  Informix Error Message 
    IES                  IESNA Photometric File Data (Illuminating Engineering Society)
    IET                  IETheme Internet Explorer Theme (Eye Can Publishing)
    IF9                  Pixel Power Collage 
    IFB                  Siebel Interface Configuration File (Siebel Systems)
    IFC                  All Image Compressed Image File (Towodo Software)
    IFD                  Jet Form Design Form 
    IFF                  Amiga Bitmap Graphic (Amiga)
    IFF                  The Sims Object/Data File (Electronic Arts (Maxis))
    IFF                  TextCraft Document 
    IFF                  TDI Explore & Alias Wavefront Image 
    IFF                  Sun TAAC Image File 
    IFF                  Simple Musical Score 
    IFF                  ProWrite Document 
    IFF                  Philips CDI File 
    IFF                  Deluxe Paint Image (Electronic Arts, Inc.)
    IFF                  Graphic, Sound 
    IFF                  Bitmap Graphic 
    IFF                  Amiga Sound (Amiga)
    IFF                  DESR VFF Greyscale Bitmap Image 
    IFF                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IFILE                Amaya i386 File 
    IFL                  Descent 3 Ship Graphic 
    IFM                  Brighterion iPrevent Fraud Model 
    IFM                  Shana Informed Filler 
    IFM                  IFS Applications Message File (IFS AB)
    IFN                  Amma Lsb v1.0 Initialization 
    IFO                  ImageForge Pro Graphic Object Layer Data 
    IFO                  DVD Info File 
    IFO                  Info File 
    IFP                  KnowledgeMan Script 
    IFR                  Formula 1 Racing Databin Cars Common File 
    IFS                  IconForge Image EXE Library 
    IFS                  OS/2 Warp Installable File System 
    IFS                  Yuvpak Compressed Fractal Image 
    IFS                  Fractint Iterated Function System Parameters 
    IFS                  Fractint Iterated Function System Fractal 
    IFS                  InfoSlips Secure Information Package (InfoSlips)
    IFU                  All Image Uncompressed Image File (Towodo Software)
    IFX                  ImageSoft v2.00 Graphic 
    IFX                  Fax File 
    IFX                  IMAGINE Effect File 
    IFX                  Rational XDE (IBM)
    IG                   Modula-3 Generic Interface Source 
    IGC                  Global Positioning System Datalog File (Fédération Aéronautique Internationale)
    IGC                  SeeYou Flight Data (Naviter)
    IGES                 Initial 2D/3D Graphics Exchange Specification 
    IGES                 IGES-Format 
    IGF                  Image Systems 
    IGF                  Vector Graphic 
    IGF                  Hijack Graphic Bitmap 
    IGF                  Sound file 
    IGF                  Inset Systems Metafile 
    IGJ                  iGrafx Published Web Content (iGrafx, a division of Corel Inc.)
    IGL                  inSORS IGLayout Layout File 
    IGL                  IGOR Saved Level File (Elmer Productions)
    IGM                  inSORS IGClient Meeting File 
    IGN                  ICQ Igonre List (ICQ Inc.)
    IGNORE               Likely a Text File Some programs write out a temporary file with an extension like .IGNORE. These are often text files. Just because it says you can ignore it does not mean you should delete it. 
    IGP                  IGOR Player File (Elmer Productions)
    IGP                  InfoGest Configuration/Profile Settings (S.C. NVN Group S.R.L.)
    IGP                  Igor Reader Sheet Music File 
    IGR                  SmartSketch Drawing (Intergraph Corporation)
    IGS                  Image Systems Grayscale, 8-bits 
    IGS                  IGES-Format 
    IGS                  CAD Overlay 
    IGV                  inSORS IGClient Venue File 
    IGX                  Process (iGrafx (a division of Corel Inc.))
    IHP                  Watcom Help File 
    IHS                  Bink Inbound History 
    IHT                  Intranet Connector Script File (SCORT)
    IHTML                Inline HyperText Markup Language (Inline Internet Systems, Inc.)
    II                   Gcc Preprocessed C++ Source Code 
    II                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IID                  Tonline BSW4 Fnstall Lid File 
    IIE                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IIF                  QuickBooks Import/Export Interchange File (Intuit, Inc.)
    IIGS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    III                  Intel IPhone Compatible File (Intel)
    III                  ARC+ Drawing (ACA)
    III                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IIL                  Norton CleanSweep Installation Log (Symantec)
    IIM                  Music Module 
    IIM                  Interscope InstallMaster Setup Project 
    IIM                  Inshape Graphic 
    IIM                  iOpus Internet Macro (iOpus GmbH)
    IIMG                 Interleaf Graphic 
    IIT                  Install Maker Install Info File (Clickteam)
    IIX                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IIX                  Rational XDE (IBM)
    IJS                  J Interpretor Script File 
    IJX                  J Language Execution Window (Source Code) (Jsoftware)
    IKL                  InkBall (Microsoft)
    IKO                  Windows Icon Resource 
    IL                   Icon Library 
    IL                   FORTRAN 90 Inline Expansion File 
    IL                   Euphoria Shrouded Source Code (Rapid Deployment Software)
    IL                   MSIL Assembler Input File (Microsoft)
    IL                   SimuPLC Electric Commands Language Instruction List (André Luiz Barbosa)
    ILB                  Scream Tracker Data 
    ILBM                 Deluxe Paint Graphic (Electronic Arts, Inc.)
    ILBM                 Amiga Interleaved Bitmap Format (Amiga)
    ILBM                 Graphic, Sound 
    ILD                  International Laser Display Association Frame Vector List for Laser Light Shows (International Laser Display Association)
    ILF                  Omniscope Application-Specific Data (Iokio Ltd)
    ILF                  SMS Insight Log File (Ag Leader Technology, Inc.)
    ILG                  TeX MakeIndex Track File 
    ILG                  InstallShield Log File (InstallShield Software Corporation)
    ILK                  Ilink Program Format Outline (Microsoft)
    ILL                  2D Graphic 
    ILL                  ILL Transaction File 
    ILL                  Agati Illustrated Image 
    ILL                  Illustrator Prologue (Adobe)
    ILM                  Satsang Quiz File 
    ILM                  Illuminatus Opus Multimedia File 
    ILM                  CAS 200 QDA v3.0 Image List Mode File 
    ILM                  Amiga IFF Image (Amiga)
    ILM                  Iomega Zip Drive Speed Configuration File 
    ILPLOPTIONS          InterLock and PowerLock 
    ILPLVOUCHERS         InterLock and PowerLock 
    ILS                  Win3 Windows Word File 
    ILS                  Internet Security And Acceleration Server Summary (Microsoft)
    ILSR                 Iomega Reader 
    ILV                  ILOG View File 
    ILVG                 iLiving P-to-P 
    IM                   Sun Raster File 
    IM                   3D Graphic 
    IM                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IM                   AOL Instant Messenger (AIM) Grey Scale Image (America Online, Inc.)
    IM04                 Image Sun Raster 
    IM1                  Sun Raster File 
    IM24                 Bitmap Graphic 
    IM30                 Image Sun Raster 
    IM32                 Bitmap Graphic 
    IM5                  Visilog (NorPix, Inc.)
    IM5                  Visilog 
    IM7                  DaVis Graphic (LaVision GmbH)
    IM8                  Sun Raster Graphic 
    IM?                  2D Graphic 
    IMA                  EGO - Chart - Autumn Mirage Vector Graphic 
    IMA                  Mirage Vector Graphics 
    IMA                  AOL Instant Messenger (AIM) Grey Scale Image (America Online, Inc.)
    IMA                  Zenographics Image 
    IMA                  ImageMixer (Pixela Corporation)
    IMA                  IncrediMail Animation (IncrediMail Ltd.)
    IMA                  WinImage File 
    IMAGE                Disk Copy, ShrinkWrap Disk Image 
    IMAGE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IMAGE                Squeak Code File 
    IMAGINE              Object & Texture Imagine 3.0 
    IMAKE                Imake File 
    IMB                  IncrediMail Address Book and Temporary Contacts (IncrediMail Ltd.)
    IMC                  MontagePlus Thumbnail Image Collection (Serif Europe Ltd.)
    IMC                  IncrediMail (IncrediMail)
    IMD                  Caseware IDEA 
    IME                  Global Input Method Editor (Microsoft)
    IME                  IncrediMail Emoticons (IncrediMail Ltd.)
    IMESHDEF             iMesh Skin (iMesh)
    IMF                  ImageForge/ImageForge Image EXE Library Saved Filtered Brush 
    IMF                  IncrediMail Letter/eCard (IncrediMail Ltd.)
    IMF                  MIDI Music File 
    IMG                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IMG                  AutoCAD CAD-Camera (Autodesk, Inc.)
    IMG                  Bitmap Graphic 
    IMG                  Disk Copy, ShrinkWrap Disk Image 
    IMG                  Apple Macintosh Disk Copy Format (Apple Computer, Inc.)
    IMG                  Alias Bitmap Graphic 
    IMG                  QEMU Qcow Disk Image (Fabrice Bellard)
    IMG                  ADEX Corporation ChromaGraph Graphics Card Bitmap Graphic 
    IMG                  All Image Raw Image (Towodo Software)
    IMG                  ERDAS IMAGINE Image Format (Leica Geosystems , LLC)
    IMG                  CloneCD Image (SlySoft Inc.)
    IMG                  Vicar 
    IMG                  Vivid Ray-tracer 
    IMG                  Planetary Data System 
    IMG                  Sharp GPB 
    IMG                  Grand Theft Auto III File (Take-Two Interactive Software, Inc.)
    IMG                  Radiance Scene Description Image 
    IMG                  Microtek Eyestar 
    IMG                  Image Whatnot / Erdas Imagine MIS / Radiance / Starbase 
    IMG                  Grand Theft Auto III .DFF and .TXD Archive (Take-Two Interactive Software, Inc.)
    IMG                  Idrisi32 Raster Image 
    IMG                  Starbase 
    IMG                  Img Software Set Bitmap 
    IMG                  Ventura Publisher/GEM VDI Image Format Bitmap 
    IMGSAO               Aleph One Images (Marathon Open Source)
    IMH                  IncrediMail Data File (IncrediMail Ltd.)
    IMI                  IncrediMail Animation (IncrediMail Ltd.)
    IMI                  Turbo Pascal DOS File (Borland Software Corporation)
    IMJ                  Pegasus Image Corp Bitmap 
    IMJ                  JFIF File With a Microsoft Windows BMP Header 
    IMK                  IncrediMail License License (IncrediMail Ltd.)
    IML                  ACT! Internet Mail Message File (Best Software CRM Division)
    IMM                  IncrediMail Trash (IncrediMail Ltd.)
    IMN                  IncrediMail Notifier (IncrediMail Ltd.)
    IMP                  WinIMP Archive (Technelysium Pty Ltd)
    IMP                  Improv Spreadsheet (IBM)
    IMP                  Urban Chaos Game File 
    IMP                  Gemstar eBook Publisher eBook (Gemstar eBook)
    IMP                  FileMaker Database Translation Filter 
    IMP                  DVD File 
    IMP                  Crayola Art Image 
    IMP                  Babya Installer Archive 
    IMP                  TEX-DESIGN Clothing Design File (Koppermann Computersysteme GmbH)
    IMP                  Adobe Audition Convolution Effect (Adobe Systems Incorporated)
    IMP                  Pascal Implementation 
    IMPLEMENT            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IMQ                  NASA Planetary Image 
    IMQ                  ImageQ Presentation 
    IMR                  Cognos Impromptu Report Writer 
    IMR                  Information Magic Data File (RM)
    IMR                  IncrediMail Notifier (IncrediMail Ltd.)
    IMS                  SiteSpinner Project (Virtual Mechanics Inc.)
    IMS                  IconForge 
    IMS                  IncrediMail Graphic Skin (IncrediMail Ltd.)
    IMS                  Music File 
    IMS                  Rebel Assault File 
    IMS                  IMS Content Package (IMS Global Learning Consortium, Inc.)
    IMSP                 Survey Pro Survey File (iMagic Software)
    IMT                  IMNET Image 
    IMT                  egseing Image Mode 
    IMU                  Star Wars Force Commander Game Audio (Lucasfilm Ltd.)
    IMV                  Yahoo! Instant Messenger IMVironment 
    IMV                  ImageMixerVCD Project File (Pixela Corporation)
    IMW                  IncrediMail Sound (IncrediMail Ltd.)
    IMW                  Imageware Surfacer 3D CAD Surface Geometry 
    IMX                  SAS SAS/IML Workshop Matrix (SAS Institute Inc.)
    IMX                  DaVis Binary Image File (LaVision GmbH)
    IMY                  iMelody Ringtone Format (Infrared Data Association)
    IMZ                  Floppy Image Compressed Image (Towodo Software)
    IMZ                  WinImage Compressed Disk Image (Gilles Vollant Software)
    IM_                  Improve Compressed Audio File 
    IN                   smartGENERATOR Template (BITPlan)
    IN                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IN                   Input File 
    IN                   McAfee Antivirus 
    IN$                  HP NewWave Install File 
    IN0                  INI Backup 
    IN1                  INI Backup 
    IN3                  Harvard Graphics Input Device Driver (Harvard Graphics)
    IN7                  Ox Object-Oriented Matrix Programming Language PcGive 7 Data File 
    INA                  DOS File 
    INB                  Rebel Assault Backup 
    INB                  Vermont HighTest Script 
    INB                  Visual Expert Source Code Documentation 
    INC                  Wincommander Include File 
    INC                  JAWS for Windows Text File (Freedom Scientific)
    INC                  Active Server Include File 
    INC                  Assembler Language or Active Server Include File 
    INC                  Astro Windows 
    INC                  Include File 
    INC                  Internet Connection File 
    INC                  MS-Backup Incremental File 
    INC                  Spice Include File Subcircuit 
    INC                  PovRay 2 or 3 
    INC                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INCD                 InCopy/InDesign Link File (Adobe)
    INCH                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IND                  dBASE IV 
    IND                  Well of Souls Game File 
    IND                  Notes List files for Compact Task on Domino Server (IBM)
    IND                  Windows Shared Database File 
    IND                  Windows/Applog Index 
    IND                  LaTeX or TeX Makeindx Index 
    IND                  Foxmail E-Mail Box Index 
    IND                  InDesign Publication (Adobe Systems Inc.)
    IND                  XMap Dataset Index (Delorme)
    INDD                 InDesign Document (Adobe Systems Inc.)
    INDENT               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INDEX                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INDEX                Index File 
    INDIGO               Indigo Graphics Format [Eddi Litvak] 
    INE                  Imagination Configuration Settings (Foxpertise)
    INF                  BLUEWAVE Information File 
    INF                  Autorun File 
    INF                  Type Manager Font Information (Adobe)
    INF                  Type 1 LaserJet Font Information File 
    INF                  Pro/ENGINEER Temporary File (PTC)
    INF                  OS/2 2.0 Information Presentation Facility (IPF) Data Format OS/2 PTR file : Cursor image file 
    INF                  OS/2 Buch Hypertext Help System File 
    INF                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INF                  Information or Setup File 
    INF1                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INFINI-D             Infini-D 3.0 Format 
    INFO                 Amiga Icon (Amiga)
    INFO                 UNIX GNU Info Reader File 
    INFO                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INFO                 ZoomBrowser Image Index (Canon, Inc.)
    INFO1                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INFO2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INFO3                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INFO4                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ING                  INGVAR Forestry Simulation (Skogforsk)
    ING                  MasterCook Ingredients 
    INGR                 Intergraph Raster File Format 
    INH                  Visual Basic Basfwin File 
    INI                  Simple Exchange Rate Information File Format (SERIFF) 
    INI                  Gravis UltraSound Bank Setup File 
    INI                  Initialization/Configuration File 
    INI                  MWave DSP Synth's GM-setup File: MWSYNTH.INI 
    INIT                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INK                  Pantone Reference File 
    INK                  Mimio Ink File 
    INK                  Tablet PC Digital Ink File (Microsoft Corporation)
    INKBL                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INL                  Visual C++ Inline Function File (Microsoft)
    INLOVE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INLX                 InDesign Library Interchange (Adobe Systems Inc.)
    INM                  Fax Master File 
    INN                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INN                  PHANTASMAGORIA INI File 
    INN                  COBOL/2 Overlay (Micro Focus)
    INO                  Inor IPRO Software File 
    INO                  Inno Setup Script Files File 
    INP                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INP                  Tramo-Seats Input Specification File (Bank of Spain)
    INP                  ABAQUS Input File (ABAQUS, Inc.)
    INP                  Self-Extracting Archive Utility Project 
    INP                  CHARMM Input File 
    INP                  FRANC2D Mesh File 
    INP                  GIS Software Text Input File 
    INP                  InPage Document (Concept Software)
    INP                  Oracle Source Code 
    INP                  MacMolPlt GAMESS Input format 
    INPUT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INQ                  Piped Technology Internet Query Configuration File (Queensgate Systems Ltd)
    INRS                 INRS-Telecommunications Sound 
    INS                  Systems Management Server (SMS) Advertising Instruction File (Microsoft Corporation)
    INS                  REAKTOR Instrument (Native Instruments)
    INS                  WordPerfect (Corel Corporation)
    INS                  Sample Cell / II Instrument 
    INS                  InstallShield Script (InstallShield Software Corporation)
    INS                  Inspiration Mind Mapping Software File 
    INS                  1st Reader Install Script 
    INS                  IIS Internet Communications Settings (Microsoft)
    INS                  Ensoniq EPS Family Instrument (E-MU Systems)
    INS                  AdLib Instrument Music File 
    INS                  PUT Compressed File Archive (MicroFox Company)
    INSCRIBE             InScribe 2004 Egyptian Hieroglyphs Document (Saqqara Technology Ltd.)
    INSID                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INSPECT              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INST                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INST                 3D Graphic 
    INST                 Object Oriented Graphics Library (OOGL) 
    INST2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INSTALL              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INSTALL              Likely a Text File 
    INSTALLER            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INSTR                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INSTRUC              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INSTRUCT             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INSURANCE            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INSURANCE2           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INT                  Delphi Interface Unit (Borland Software Corporation)
    INT                  Foxpro Code Page (Microsoft)
    INT                  Intermediate Code Used for Syntax Checking 
    INT                  Unreal Tournament Mutator 
    INT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INT                  Signature Semi-compiled Program 
    INTEGER              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INTER                Interspector Project File 
    INTEREST             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INTERIEUR            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INTERLEAV            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INTERLOCKFAVOURITES  InterLock and PowerLock 
    INTERNETMAILSEND     Internet Mail Send Powertoy File 
    INTERRUPT            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INTMATH              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INTO                 Interspector Object File 
    INTREC               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INTRO                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INTRODUCT            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INTRODUCTION         Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INU                  INSPEC Configuration File 
    INV                  Medlin Accounting Current Year Checks and Invoices (Medlin Accounting)
    INV                  Lineage Inventory File (NC Interactive, Inc.)
    INV                  dnaLIMS Invoice File (dnaTools)
    INV                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INV                  Windows Update file 
    INV                  Rainbow Six: Rogue Spear Inventory File (Red Storm Entertainment, Inc.)
    INVADERS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INVEN                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INVENTOR             Open Inventor File Translator How To [SGI] 
    INVENTORY            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INVERT               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    INV_LOWRES           Rainbow Six: Rogue Spear Inventory File (Red Storm Entertainment, Inc.)
    INX                  Foxpro Foxbase Index 
    INX                  ACL for Windows Index 
    INX                  InstallShield Compiled Rules File (InstallShield Software Corporation)
    INX                  Rational XDE (IBM)
    IN_                  Setup Information (Microsoft)
    IN~                  King's Quest 7, Grabriel Knight, Compressed File 
    IO                   IO Language Script (Steve Dekorte)
    IO                   Compressed Archive 
    IO                   Modula-3 Object File 
    IOB                  3D Object TDDDB Format 
    IOB                  Turbo Silver TDDDB Format 
    IOB                  TDDD Format 3D Graphic Database 
    IOB                  iOnly ME Website Generator Backup (iOnly)
    IOB                  Imagine Ver. 1 or 2 TDDDB Format 
    IOB                  Imagine File 
    IOC                  Instant ORGcharting! Organization Chart 
    IOC                  Winamp Io Plug-in (File: IMGORG.IOC) (Nullsoft)
    IOCA                 Image Object Content Architecture (IOCA) Graphics File 
    IOD                  DVGate DVMotion In/Out List 
    IOF                  Findit Document (Microsoft)
    IOF                  Findit Document 
    IOK                  Omniscope Data File (Iokio Ltd)
    ION                  NewsLeecher Description File (newsleecher.com)
    ION                  Mascot Data 
    ION                  4DOS File Description 
    ION                  Installation Description 
    IOP                  iOnly ME Website Generator File (iOnly)
    IP                   IconPackager Theme (Stardock Corporation)
    IP                   Interactive Physics Data 
    IP                   Internet Protocol 
    IP                   Profile 
    IP                   NRC Inspection Manual Inspection Procedures 
    IPC                  Interactive Publisher Pro Church Edition Project Data (Interactive Software Designs)
    IPD                  BlackBerry Backup File (Research In Motion Limited)
    IPD                  InstallPROG 6 EDBS Install Database 
    IPE                  Microth Stroke Set (Microth, Inc.)
    IPE                  ItsPersonal File 
    IPE                  Wacom Pen SYSTEM.INI and WIN.INI Backup 
    IPF                  SMS Installer Script (Microsoft)
    IPF                  F-Secure Incremental Policy File (F-Secure)
    IPF                  Grand Theft Auto III Animation Data (Take-Two Interactive Software, Inc.)
    IPF                  Interchangeable Preservation Format (Classic Amiga Preservation Society)
    IPG                  Mindjongg Format 
    IPG                  iPod Game Package (Apple Inc.)
    IPH                  InterPaint Hires Image 
    IPJ                  Inventor Project (Autodesk, Inc.)
    IPJ                  InfinityQS SPC Project File (InfinityQS International)
    IPK                  internet PacKage Archive (direct Netware Group)
    IPK                  Sharp Zaurus PDA Application Install File 
    IPK                  NRG iPack File 
    IPL                  Grand Theft Auto III Object Placement Data (Take-Two Interactive Software, Inc.)
    IPL                  PLATO Web Learning Network (PWLN) Temporary File (PLATO Learning, Inc.)
    IPL                  InstallPROG 6 EDBS File Library 
    IPL                  IPLab IPLab-format Image File (BD Biosciences)
    IPL                  Corel Pantone Spot Reference Palette 
    IPL                  Ink Palette 
    IPL22                PLATO Web Learning Network (PWLN) Temporary File (PLATO Learning, Inc.)
    IPLI                 IPLab Graphic 
    IPM                  Image Pro Plus Ver. 5 Macro (Media Cybernetics Inc.)
    IPN                  Inventor Presentation (Autodesk, Inc.)
    IPOD                 Rockbox Executable Application (The Rockbox Crew)
    IPP                  Help & Manual Proprietary Image 
    IPR                  SMART Ideas Document (SMART Technologies Inc.)
    IPR                  IntelliJ Project XML File 
    IPS                  IPScript File 
    IPS                  Game Patch File 
    IPS                  International Patching System Binary Patch File 
    IPS                  MENSI 3Dipsos (MENSI Corporation)
    IPT                  Inventor CAD (Autodesk, Inc.)
    IPT                  InterPaint Multicolor Image 
    IPTC                 Newswire Profile 
    IPTHEME              IconPackager Icon Theme Package (Stardock Corporation, Inc.)
    IPU                  Vocaltech Iphone Internet Phone User File 
    IPW                  UniPlot Graph 
    IPX                  Interactive Pictures Corporation AV File 
    IPX                  Ipix Spherical Panorama 
    IPX                  Viewpoint iPix File 
    IPZ                  ICQ Skin (ICQ Inc.)
    IQ                   IBM Query File 
    IQA                  AmeriCalc Application 
    IQB                  AmeriCalc Backup 
    IQC                  CADIQ Comparison Analysis Result File (ITI TranscenData)
    IQC                  AmeriCalc Client File 
    IQD                  Webstir File 
    IQD                  AmeriCalc Database 
    IQF                  Integra 3.0 Query File 
    IQI                  IBM Query 
    IQJ                  CADIQ Specific Tasks/Analysis Details (ITI TranscenData)
    IQL                  CADIQ Log File (ITI TranscenData)
    IQP                  20-20 Office Sales Exported Quote (20-20 Technologies)
    IQR                  CADIQ View Analysis Results (ITI TranscenData)
    IQR                  IBM Query 
    IQS                  CADIQ Summary Log File (ITI TranscenData)
    IQS                  AmeriCalc Security File 
    IQT                  IBM Query 
    IQU                  AmeriCalc Update 
    IQY                  Internet Inquiry (Microsoft Corporation)
    IQY                  Excel Web Query (Microsoft)
    IR                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IRC                  jIRCii Script (Raffi)
    IRC                  IRCAM Format Sound 
    IRC                  Nevo Infra-red Device Code (Universal Electronics Inc.)
    IRCD                 iArcade Game Package (Brooklyn Games, Inc.)
    IRD                  IRD Append File 
    IRD                  Money Temporary File (Microsoft)
    IRF                  RoomBox 6 IR Code File (Living Control)
    IRF                  Phonecat2 Ocp97 File 
    IRI                  Thermography Studio Infrared Image (GORATEC Technology GmbH & Co. KG)
    IRIS                 Silicon Graphics RGB 
    IRIX                 Irix Make File 
    IRIX6                Irix 6 File 
    IRIX64BIT            Irix 64 Bit Make File 
    IRK                  iArcade Game Package (Brooklyn Games, Inc.)
    IRL                  ILOG JRules ILOG Rule Language File (ILOG, Inc.)
    IRS                  WordPerfect Resource File (Corel Corporation)
    IRTP                 Graphicon-2000 Interactive Real-Time PHIGS Graphic 
    IRV                  XMap Workfile (ImageReg) (Delorme)
    IRX                  PS2 Modified Elf Object File 
    IRX                  IBM Mainframe Rule File 
    IRX                  Brewer Lamp Irradiation File 
    IRZ                  iROS Application Packaging Unit (Interactive Workspaces Project, Stanford University)
    IS                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IS                   Modula-3 Intermediate Assembly File 
    ISB                  IS-Builder Link File (NPO Computer)
    ISBN                 HM3 Tutorial Software 
    ISC                  RVS-COM Prepaid SMS Package File (Living Byte Software GmbH)
    ISC                  GRDSOZ4W ISCII File 
    ISC                  Internet Servide Providers Configuration File 
    ISC                  iScreensaver Designer Project File (Xochi Media Inc.)
    ISC                  Xilinx Integrated Software Environment Device Configuration Information (Xilinx, Inc.)
    ISD                  InstallShield Dialog (InstallShield Software Corporation)
    ISD                  GraphPad InStat 
    ISD                  RapidFile Dictionary 
    ISD                  dynaSight File 
    ISF                  ModelPress Highly-compressed CAD Model (Informative Graphics Corporation)
    ISF                  Inspiration Concept Map (Flowchart) Document (Inspiration Software, Inc.)
    ISFF                 Intergraph Standard File Format 
    ISH                  Image Speeder 
    ISH                  Image System 
    ISH                  Compressed Archive File 
    ISK                  Command File 
    ISL                  Inno Setup 1.3 File 
    ISL                  InstallShield Temporary Installer File (InstallShield Software Corporation)
    ISL                  The Universal Game Server Export File (The Universal Development Team)
    ISL                  DB-MAIN Scheme File 
    ISM                  Image System 
    ISM                  SimulationX Encrypted Model (ITI Gesellschaft für ingenieurtechnische
    Informationsverarbeitung mbH)
    ISM                  MySQL ISAM Table Handler Data (MySQL AB)
    ISML                 Intershop Enfinity 
    ISO                  Cimagraphi CAD File (Graphitech Ltd)
    ISO                  SEGA CD/Sony Playstation Game File 
    ISO                  ISO-9660 CD Disc Image 
    ISO                  RichWin 
    ISO                  ISO-9660 Table 
    ISO                  InstallShield Uninstall File (InstallShield Software Corporation)
    ISO                  CALS ISO 8613 Bitmap 
    ISO                  Easy CD Creator Disc Image (Sonic Solutions)
    ISO                  Virtual PC Virtual CD-ROM Image (Microsoft)
    ISOD                 CloneDVD ISO Image (Elaborate Bytes AG)
    ISP                  IIS Internet Service Provider Settings (Microsoft)
    ISP                  Internet/Internationnal Communication Settings parameter file 
    ISP                  Internet Communication Settings 
    ISP                  Internet Sign-up File 
    ISQ                  MicroCT Image (SCANCO Medical AG)
    ISR                  Streets & Trips Route File (Microsoft)
    ISR                  Uninstaller Text File 
    ISR                  Go-Moku Game File 
    ISS                  Caseware IDEA IDEAScript 
    ISS                  InstallShield Response File (InstallShield Software Corporation)
    ISS                  ISS Graphic 
    ISS                  Inno Setup Script (Jordan Russell's Software)
    ISS                  The Longest Journey Sound File (Funcom)
    IST                  Digitrakker Instrument File 
    IST                  IsoStar Library of Intermolecular Interactions (The Cambridge Crystallographic Data Centre)
    IST                  Systems Management Server (SMS) Remote Instruction File (Microsoft Corporation)
    IST                  Imagine Staging Stage Editor File (Impulse, Inc.)
    IST                  ImageStyler (Adobe Systems Incorporated)
    IST                  TeX MakeIndex Style Index 
    ISTG                 Imagine Staging Stage Editor File (Impulse, Inc.)
    ISTR                 IsoStar Library of Intermolecular Interactions (The Cambridge Crystallographic Data Centre)
    ISU                  Easy CD Creator 4 Uninstall File 
    ISU                  Netscape File 
    ISU                  InstallShield Uninstall Script (InstallShield Software Corporation)
    IT                   Read.it 
    IT                   Impulse Tracker Music Module 
    IT                   Intalk Settings 
    IT                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IT                   Sound 
    IT3                  Internet Timer 3 Log File 
    ITA                  IconTweaker Theme 
    ITA                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ITA                  Italian README File 
    ITA                  Personal Paint Italian Language User Interface File (Amiga)
    ITAL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ITALIC               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ITC                  Uthscsa Imagetool Calibration File 
    ITC                  iTunes CoverFlow Data File (Apple Computer, Inc.)
    ITE                  E-Tabs Reader File 
    ITEMS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ITF                  JPI TopSpeed Pascal Interface File 
    ITF                  Inside Tract File 
    ITG                  Intergraph Format 
    ITH                  InTether Receiver File (Infraworks Corporation)
    ITHMB                iPod photo Thumbnail Image (Apple Computer, Inc.)
    ITI                  Impulse Tracker Instrument 
    ITL                  IntelliTipster League File (IntelliForecast Labs)
    ITL                  iTunes Music Database File (Apple Computer, Inc.)
    ITM                  Infinity Game Engine Inventory Object (BioWare Corp.)
    ITM                  Diablo II Item (Blizzard Entertainment)
    ITM                  Item / Article / Zone 
    ITP                  riteMail Mail and Notes File (Pen&Internet, LLC)
    ITP                  IconTweaker Plug-in 
    ITR                  Icy Tower Replay (Johan Peitz, Free Lunch Design)
    ITR                  IntroCreator Project File (Klaus Schwenk)
    ITS                  Internet Document Set 
    ITS                  International Translation 
    ITS                  Impulse Tracker Sample 
    ITU                  Sheffield Dictionary File 
    ITW                  IT Works Database (Simple Solutions)
    ITW                  Icetips Wizard Editor 
    ITX                  ITRANS Multilingual Transliteration Scheme 
    ITX                  Texture IMAGINE 
    ITX                  Igor Chromatogram File 
    ITX                  Impulse Imagine for Windows 
    ITX                  ISB Raw Song Format 
    ITZ                  Impulse Tracker Music Module 
    ITZ                  Amiga and PC Tracker Module 
    IUM                  infoUSA Network Meter File (infoUSA.com, Inc.)
    IUT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IV                   VRML File 
    IV                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IV                   Open Inventor (Silicon Graphics, Inc.)
    IV-VRML              Inventor VRML Format Document 
    IVB                  Truevision Targa Format Bitmap 
    IVC                  iView MediaPro Catalog (iView Multimedia Ltd.)
    IVD                  ISPS for Windows Multi-dimensional Data Store (International Planning & Research Corporation)
    IVD                  Beyond 20/20 Microdata Dimension or Variable-level File 
    IVE                  OpenSceneGraph Native Binary Format (OSG Community)
    IVF                  Indeo Video File (Intel Corporation)
    IVI                  MSDN InfoViewer 5.0 Topic (Microsoft)
    IVN                  iView MediaPro Catalog Notepad File (iView Multimedia Ltd.)
    IVNU                 Inetba Web Page (Inetba)
    IVP                  PhotoSuite Ver 4 Image-based Virtual Reality Text File (Sonic Solutions)
    IVP                  Beyond 20/20 User Subset Profile 
    IVP                  Truevision Targa Bitmap 
    IVQ                  MSDN InfoViewer 5.0 index full-text searches 
    IVR                  Virtual Reality World Live Picture 
    IVS                  ImageView Storage File (Infinite Solutions)
    IVS                  ISPS for Windows Settings File (International Planning & Research Corporation)
    IVS                  iVisit 
    IVT                  Beyond 20/20 Table 
    IVT                  Concordance Inverted Full Text Database File 
    IVT                  MSDN InfoViewer 5.0 Information Title 
    IVU                  IVUE Live Picture Drawing 
    IVUE                 Live Picture Multi-view Format 
    IVWD                 IntelliVIEW Designer Data/Layout File (Synaptris; products division of Cybernet Software Systems, Inc.)
    IVWL                 IntelliVIEW Designer Report Layout File (Synaptris; products division of Cybernet Software Systems, Inc.)
    IVWR                 IntelliVIEW Designer Report File (Synaptris; products division of Cybernet Software Systems, Inc.)
    IVWR                 IntelliVIEW Offline Report File (Cybernet Software Systems Inc.)
    IVX                  Beyond 20/20 Extract 
    IW                   Iconware Application 
    IW                   Idlewild Screensaver 
    IW                   IconAuthor - HSC InterActive Presentation Flowchart 
    IW                   IBM Updater File 
    IW2                  Information Workshop 2000 Data File (BlackCat)
    IW4                  DjVu File 
    IWA                  IBM Writing Assistant Text 
    IWC                  WaveL Wavelet Compressed Bitmap (WaveL Software)
    IWC                  Installwatch Database 
    IWI                  Ignite Image Optimiser 
    IWM                  IconAuthor Start File (AimTech Corp)
    IWP                  Wang Text File 
    IWP                  InfinityQS SPC Web Publishing Configuration (InfinityQS International)
    IWR                  i*write 2.0 File 
    IWR_BAK              i*write 2.0 Backup File 
    IWS                  InstallWatch Scan Results (Epsilon Squared)
    IWS                  Information Workshop Data File (BlackCat)
    IWZ                  ItemWizard File 
    IX                   Modula-3 Linker Information 
    IX                   WordPerfect Office Template File (Corel Corporation)
    IX                   WordPerfect Office Template File (Corel Corporation)
    IX                   PaperPort Search File (ScanSoft Inc.)
    IX                   FrameMaker Index File (Adobe)
    IX1                  Rebel Chess Program Database File 
    IX2                  WordPerfect Office Template File (Corel Corporation)
    IX2                  Rebel Chess Program Database File 
    IX2                  WordPerfect Office Calendar File Calendar File (Corel Corporation)
    IX3                  Instant.EXE 3.0 Script (ASXperts Inc.)
    IX3                  Rebel Chess Program Database File 
    IX4                  Rebel Chess Program Database File 
    IXA                  Ulead Image File 
    IXB                  Ulead Video Image File (Ulead Systems, Inc.)
    IXC                  Index+ for Windows Code Definition File (System Simulation Ltd.)
    IXC                  ArcView Geocoding Index For Read-Write Coverages (ESRI)
    IXE                  InterEX Control File 
    IXF                  IFX for FAX FAX Document 
    IXF                  Index+ for Windows Form Defintion File (System Simulation Ltd.)
    IXL                  DB/TextWorks Database Indexed List 
    IXP                  ISIPublisher Publication Information Export (Image Solutions, Inc.)
    IXR                  Index+ for Windows Report Definition File (System Simulation Ltd.)
    IXS                  ArcView Geocoding Index For Read-Write Shapefiles (ESRI)
    IXT                  ISIPublisher Publication Template (Image Solutions, Inc.)
    IXX                  C++ Include File 
    IZD                  Intrexx Application Export Format 
    IZER                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    IZS                  1st Page 2000 JavaScript Example File (Evrsoft)
    IZT                  IZL Binary Token 
    IZX                  Intermezzon Designer E-Learning Published File (Intermezzon Learning Systems AB)
    I_I                  Eru/erd File 
    J                    JPEG / JFIF Image 
    J                    JAR Compressed File 
    J                    Java Source Code (Sun)
    J1                   EDIT Compiler JEDEC Boundary File 
    J2A                  Jazz Jackrabbit 2 Animation Library (Epic MegaGames Inc.)
    J2B                  Jazz Jackrabbit 2 Music File (Epic MegaGames, Inc.)
    J2D                  Jazz Jackrabbit Data File (Epic MegaGames, Inc.)
    J2E                  Jazz Jackrabbit 2 Episode File (Epic MegaGames, Inc.)
    J2G                  J2 Fax File 
    J2H                  Jazz Jackrabbit 2 High Scores (Epic MegaGames, Inc.)
    J2I                  Jazz Jackrabbit 2 Internet Server Link (Epic MegaGames, Inc.)
    J2K                  JPEG-2000 JP2 File 
    J2L                  Jaxx Creation Station Level File (Epic MegaGames Inc.)
    J2M                  Jazz Jackrabbit 2 Recorded Macro (Epic MegaGames, Inc.)
    J2S                  Jazz Jackrabbit 2 Language Data (Epic MegaGames, Inc.)
    J2T                  Jazz Jackrabbit 2 Tile Set (Epic MegaGames, Inc.)
    J2V                  Jazz Jackrabbit 2 Cinematic File (Epic MegaGames, Inc.)
    J62                  Ricoh Digital Camera File 
    J6I                  Ricoh Digital Camera File (Ricoh Co., Ltd.)
    J92                  Javelin Configuration File (Minisoft Inc.)
    JA                   IBM Tools Updater File 
    JACKET               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JACKSUM              Jacksum Fingerprint (Dipl.-Inf. (FH) Johann N. Löfflmann)
    JAD                  Java Application Descriptor File (Sun Microsystems, Inc.)
    JAG                  Jaguar Server File 
    JAIL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JAM                  FTN Software Jam Messagebase 
    JAM                  Image Microprose 
    JAM                  JamTracker Music Module 
    JAM                  E-mail 
    JAM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JAM                  Joaquim-Andrew-Mats Message Base Proposal 
    JAM2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JAMMIX               Jammix Enhacer Music Transform File (Danilics Tibor)
    JAN93                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JAP                  Fifa World Cup Game Data fe Art Legalscr File 
    JAP                  JAlbum Project File (Datadosen)
    JAP                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JAPAN                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JAR                  JARCS Compressed Archive (JPMdL Archival System)
    JAR                  JAR Archive (ARJ Software, Inc.)
    JAR                  Java Archive (Sun)
    JAR                  Java Applet (Sun)
    JARRE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JAS                  Paint Shop Pro Compressed Graphic (Corel Corporation)
    JASC                 JAS Compressed Graphic 
    JAU                  Jauge 
    JAV                  Java Source Code (Sun)
    JAVA                 Java Source Code (Sun)
    JAW                  IchiTaro Web Japanese Extended WordProcessor (Justsystem Corporation)
    JB2                  JBIG2 Bitmap Graphic 
    JBC                  Jam Byte-Codec File 
    JBC                  BestCrypt Container (Jetico, Inc.)
    JBD                  SigmaScan Data File 
    JBF                  Paint Shop Pro Browser Cache (Corel Corporation)
    JBG                  JBIG File 
    JBIG                 JBIG File 
    JBIG2                JBIG2 Bitmap Graphic 
    JBK                  Juno Backup File 
    JBK                  Jot + Notes Backup File for Older Version (King Stairs Software)
    JBL                  Doom Scene Add-in (Id Software, Inc.)
    JBP                  DJ Jukebox Project 
    JBR                  Paint Shop Pro Brush (Corel Corporation)
    JBS                  DesignArt (Anagraph)
    JBX                  Project Scheduler File 
    JC                   FlashGet Intermediate Download 
    JC                   Watcom Source File 
    JC!                  FlashGet (Amaze Software)
    JCAMP                Misc 
    JCB                  CB-JCBrooke Song/Module 
    JCC                  Code Crusader Project Data (New Planet Software)
    JCC                  Japan Crossword Dmitry Torshin Format File (EliteWare)
    JCD                  FlashGet Download Database File 
    JCE                  JWPce (Glenn Rosenthal)
    JCF                  JAWS for Windows Configuration File (Freedom Scientific)
    JCK                  Liquid Motion JACK Scene Description 
    JCL                  Job Control Language IBM 
    JCM                  Java Commerce Message Commerce Message (Wallet) (Sun)
    JCP                  Jesusonic Preset (Cockos Incorporated)
    JCS                  Latest Bruker WinSpecEdit v3.08 JCAMP 
    JCS                  Flashget Script HTML Table 
    JCV                  Baseball/Golf Analyzer (JC Video Systems)
    JD                   Jigsaw Terminal File 
    JDA                  Winphone Fax Send Journal 
    JDB                  Winphone Fax Receive Journal 
    JDB                  Siren Jukebox 
    JDE                  Cariverona 
    JDF                  Turok 2 Game File 
    JDF                  JAWS for Windows Voice Synth File (Freedom Scientific)
    JDF                  Job Definition Format Job File 
    JDT                  Capture Classic Filler (Accelio)
    JDX                  EDICT Index 
    JDX                  JCAMP Chemical Spectroscopic Data Exchange Format (MDL Information Systems, Inc.)
    JDX                  J Database Exchange (JDX) O/R Mapping Specification (Software Tree, Inc.)
    JDX                  JSpell UNICODE Dictionary (The Solution Cafe)
    JED                  Xilinx Integrated Software Environment Device Configuration Information (Xilinx, Inc.)
    JED                  JEDEC Programming Specification 
    JEF                  Janome NH10000 Sewing Machine Stitch File (Janome America, Inc.)
    JEQ                  Jet Audio Saved Equalizer Setting (JetAudio, Inc.)
    JET                  Hybrid JetFax Fax 
    JET                  Jetpack User-created Level Map 
    JFD                  JAWS for Windows (Freedom Scientific)
    JFD                  FormWare Job 
    JFF                  JAWS for Windows (Freedom Scientific)
    JFF                  JPEG Image 
    JFG                  jEEPers User Specific Information 
    JFI                  JPEG/JIFF Image 
    JFIF                 JPEG/JIFF Image 
    JFM                  Capture Classic (Accelio)
    JFX                  J2 Fax File 
    JG6                  BigJig Jigsaw 
    JGCSCS               EditPad Pro Custom Coloring Scheme (JGsoft - Just Great Software)
    JGD                  Paint Shop Pro Gradient (Corel Corporation)
    JGF                  Jetcam Geometry File (JETCAM International, Ltd.)
    JGF                  JAWS for Windows Configuration File (Freedom Scientific)
    JGL                  QWS3270 PLUS Session Layout File (Jolly Giant Software Inc.)
    JGP                  QWS3270 PLUS Session Parameter File (Jolly Giant Software Inc.)
    JGS                  QWS3270 PLUS Script (Jolly Giant Software Inc.)
    JHTM                 Dynamo Server Page 
    JHTML                Dynamo Server Page 
    JIF                  Jeff's Image Format Bitmap 
    JIF                  JPEG/JIFF Image 
    JIG                  Jigsaw Game (GameHouse, Inc., a division of RealNetworks, Inc.)
    JIG                  Living Scenes Jigsaw Puzzles (Carolina Road Software L.L.C.)
    JIGDO                Jigsaw Download Download Instruction File (Richard Atterer)
    JIM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JINKS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JIS                  Japanese Industrial Standard Text File 
    JJ                   JavaCC File 
    JJ                   C64 Doodle Compressed File 
    JJC                  Canvas Compressed Audio File 
    JJF                  Jester Jugglor File 
    JKL                  Jedi Knight 
    JKM                  JAWS for Windows Configuration File (Freedom Scientific)
    JKS                  Jedi Knight Backup 
    JL                   Sawfish Lisp-based Script 
    JLA                  VisualPro BMP Image 
    JLAB                 JLab Program Source Code (Linova)
    JLC                  Sawfish Byte Compiled Lisp-based Script 
    JLF                  JetLinks Bookmark Managment 
    JLP                  JLIP Video Capture (Video Producer) Index Format Display (JVC)
    JLS                  JAWS for Windows (Freedom Scientific)
    JLS                  JPEG-LS File 
    JMAN                 Illum Source Irohaserver File 
    JMD                  Paint Shop Pro Image Map Settings (Corel Corporation)
    JMF                  Java Multimedia File (Sun)
    JMG                  16 Bit Adaptive RLE Compressed Bitmap 
    JMH                  JPEG File Interchange Format 
    JMK                  Code Crusader Build Script (New Planet Software)
    JMM                  Digital Camera Video Clip (JVC)
    JMP                  SAS JMP Discovery Chart-to-Statistics File (SAS Institute Inc.)
    JMS                  UltraForce Music Format 
    JMS                  SpamProx Blocking List (Dewqs' Tribes)
    JMS                  Voxtron TeleButler Graphical Voice Response Call Flow 
    JMX                  JMeter File 
    JN1                  Epic MegaGames Jill of the Jungle Data File 
    JNB                  Sigma Plot Workbook File 
    JNC                  Communication Log File 
    JNG                  JPEG Network Graphic Bitmap 
    JNGL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JNK                  Junk 
    JNL                  Journal Macro Data File (Chosen Software)
    JNL                  Ingres Journal File (Computer Associates International, Inc.)
    JNL                  WINDEV Log File (PC SOFT)
    JNLP                 Java Web Start (Sun Microsystems, Inc.)
    JNT                  Windows XP Tablet PC Edition Journal (Microsoft Corporation)
    JOB                  Windows Task Scheduler Task Object (Microsoft)
    JOB                  QuestVision Vector Graphics File 
    JOB                  BETA 44 Job File 
    JOB                  Job File 
    JOB                  File Invesigator Script Commands 
    JOB                  AI Research 
    JOB                  Tripod Data Systmes Survey Pro Job 
    JOB                  Systems Management Server (SMS) Job File (Microsoft Corporation)
    JOB                  MicroSim PCBoard External ASCII Job File 
    JOBOPTIONS           Acrobat Job Options File (Adobe)
    JOBS                 VirtualDub Job Control File (Avery Lee)
    JOC                  Context4 DOS Word Processor File 
    JOD                  MS.Jet.OLEDB.4.0 Related File 
    JOI                  Xboard Join Conferences File 
    JOKE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JONES                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JOR                  SQL Server Journal File (Microsoft)
    JOR                  Gupta SQLWindows Journal 
    JOT                  Jot + Notes Note Data File (King Stairs Software)
    JOT                  JDesignerPro File 
    JOT                  WinJottr Desktop Note 
    JOU                  VAX Edt Editor Journal File 
    JOURNAL              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JOY                  Joystick Calibration File 
    JOY                  Trackjoy GUS Tracker Module 
    JP2                  JPEG-2000 JP2 File 
    JP3                  Electric Quilt File 
    JP4                  Electric Quilt File 
    JP5                  Electric Quilt File 
    JPC                  JPEG-2000 Code Stream Bitmap 
    JPC                  Jprobe Coverage Memory Snapshot (Quest Software Inc.)
    JPC                  Japan Picture Format 
    JPE                  JPEG/JIFF Image 
    JPEG                 JPEG/JIFF Image 
    JPEGENX              egis Encrypted JPEG File (HiTRUST Inc.)
    JPF                  Biz Plan Pro Business Plan Contents (JIAN Tools For Sales, Inc.)
    JPG                  CompactDRAW e-JPG Graphic (MediaChance)
    JPG                  JPEG/JIFF Image 
    JPGENX               egis Encrypted JPG File (HiTRUST Inc.)
    JPG_T                JPG Image File 
    JPH                  Jprobe Memory Debugger Memory Snapshot (Quest Software Inc.)
    JPI                  Jupiter Document (Elfin Sp. z o.o.)
    JPL                  Siren Jukebox 
    JPL                  Jprobe Settings File (Quest Software Inc.)
    JPM                  LuraDocument.jpm Compressed PDF File (Algo Vision LuraTech GmbH)
    JPN                  Photomodeler Lite File 
    JPOW                 JPowered Java Bar Charts 
    JPP                  Jigsaw Puzzle Player Puzzle Definition (B&P Games B.V.)
    JPP                  Jprobe Profiler Memory Snapshot (Quest Software Inc.)
    JPR                  Oracle JDeveloper Model JProject 
    JPR                  Javasockets File 
    JPS                  Print Studio Graphic (Jolly, Inc.)
    JPS                  Jigsaw Puzzle Player Puzzle Completion Status (B&P Games B.V.)
    JPS                  Stereo Image 
    JPT                  Print Studio Template (Jolly, Inc.)
    JPT                  Jprobe Threadalyzer Memory Snapshot (Quest Software Inc.)
    JPX                  JBuilder Project File (Borland Software Corporation)
    JPX                  JPEG-2000 JP2 File 
    JQX                  Windows Help Related File (Microsoft)
    JR                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JR2                  FishSim Add-on (J.C.Spooner)
    JRC                  Jrchive Compressed Archive 
    JRF                  The Jumbot Bot Level Navigation File 
    JRF                  winIDEA Workspace (Programming Project File) (iSYSTEM AG)
    JRL                  Top Secret Crypto Gold Top Secret Journal File (TAN$TAAFL Software Company)
    JRP                  Electric Quilt File 
    JS                   JACOsub Script 
    JS                   Mozilla (Netscape) Settings (Mozilla.org)
    JS                   JavaScript Source Code 
    JS                   JetScript Output File 
    JSB                  JavaScript Bean (JSB) File 
    JSB                  JAWS for Windows Script Binary File (Freedom Scientific)
    JSCRIPT              GoLive JavaScript (Adobe)
    JSD                  eFAX Jet Suite Document (j2 Global Communications, Inc.)
    JSD                  Paint Shop Pro Image Slice Settings (Corel Corporation)
    JSD                  Join-Split File 
    JSD                  JAWS for Windows Script Documentation File (Freedom Scientific)
    JSDB                 JumpStart Database (Jumpstart-it.com)
    JSE                  JScript Encoded Script File 
    JSE                  Fichier Encoded Javascript 
    JSF                  Jahshaka Scene File (The Jahshaka Project)
    JSF                  Macromedia Fireworks Batch Script 
    JSH                  JAWS for Windows Script Header File (Freedom Scientific)
    JSL                  Visual J# Code (Microsoft Corporation)
    JSL                  Paint Shop Pro Shapes File (Corel Corporation)
    JSM                  JugWin Jigsaw 
    JSM                  Jig ShapeMap (Steamboat Software)
    JSM                  JAWS for Windows Script Message File (Freedom Scientific)
    JSM                  Gray Design Associates Jigsaws Galore Jigsaw File 
    JSP                  Java Server Page (Sun)
    JSP10                Java Server Page (Sun)
    JSPF                 JSP Page Fragment (Sun)
    JSS                  JSheet 
    JSS                  Gray Design Associates Jigsaws Galore Saved Game Jigsaw File 
    JSS                  JugWin Jigsaw 
    JSS                  JAWS for Windows Script Source File (Freedom Scientific)
    JST                  Jnes Saved Game State File (JaboSoft)
    JST                  Code Crusader Symbol Database (New Planet Software)
    JSV                  Java Structure Viewer 
    JSV                  HyperRail 3D Coaster Design Construction File 
    JSV                  Japanese Flash Card File 
    JSV                  VXML JavaServer Page 
    JSW                  WML JavaServer Page 
    JT                   JT Fax 
    JTD                  Ichitaro Document (Justsystem Corporation)
    JTF                  JPEG Tagged Interchange Format Image 
    JTF                  Hayes JT Fax 
    JTIF                 JPEG Tagged Interchange Format Image 
    JTK                  Java ToolKit File (Sun)
    JTL                  JTool Data File (Behavioral Health Quality)
    JTP                  Windows XP Tablet PC Edition Journal (Microsoft)
    JTP                  JetForm File 
    JTR                  Jelbert GeoTagger Location Database (Jelbert Consulting)
    JUDI                 JudoScript Source Code File (JudoScript.COM)
    JUDO                 JudoScript Source Code File (JudoScript.COM)
    JUL                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JUL                  Inteliweb Program Registration and Validation (InVision Marketing, Inc. & Inteliweb.net)
    JUMP                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JUMP1                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JUN14                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JUP                  Code Crusader User-Specific Project Preferences (New Planet Software)
    JUST                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    JUT                  Jutvision Scene 
    JVC                  Digital Camera File 
    JVS                  Jpeg Video Sequencer Sequenced JPEG Images 
    JW                   Justwrite Text 
    JW                   Jigsaw Terminal File 
    JW                   Legacy Version 1.1 
    JW                   Q & A Write for Windows 3.0 
    JW                   MS Wordpad, Works for windows v3.0 & 4.0 
    JWC                  JewelCase Maker Saved File (NewTech Infosystems Inc.)
    JWL                  Justwrite Text File Library 
    JWL                  Easy CD Creator's CD Label (Sonic Solutions)
    JWL                  Jewel Case Creator CD Disc Case 
    JWP                  JWP Document 
    JWP                  Easy CD Creator Label File (Sonic Solutions)
    JWR                  LegaSuite GUI J Walk Runtime File (Seagull Software Systems, Inc.)
    JWS                  WebLogic Workshop Java Web Services (BEA Systems, Inc.)
    JXDM                 Multi-Channel Server (MCS) Markup Editor Java Server Page with XDIME Markup/Java Code (Volantis Systems Ltd.)
    JXX                  C++ Header File 
    JZZ                  Jazz Spreadsheet 
    K                    Desktop Color Separation Specification Black Layer 
    K                    NRJ Live File 
    K                    IRIS Editor Format File (eurobraille)
    K00                  Tonline DB File 
    K01                  Clarion DOS Database Key File (SoftVelocity)
    K01                  Tonline DB File 
    K02                  Clarion DOS Database Key File (SoftVelocity)
    K02                  Tonline DB File 
    K03                  Tonline DB File 
    K03                  Clarion DOS Database Key File (SoftVelocity)
    K04                  Clarion DOS Database Key File (SoftVelocity)
    K05                  Clarion DOS Database Key File (SoftVelocity)
    K06                  Clarion DOS Database Key File (SoftVelocity)
    K07                  Clarion DOS Database Key File (SoftVelocity)
    K08                  Clarion DOS Database Key File (SoftVelocity)
    K09                  Clarion DOS Database Key File (SoftVelocity)
    K1S                  Wave Glib19 File 
    K2                   Kar2ouche Storyboard File 
    K25                  Kurzweil 2500 Sample (Kurzweil Music Systems, Inc.)
    K25                  Kodak DC25 Digital Camera File 
    K3D                  3DS Macro Language Description 
    K4E                  Karaoke 4 Everyone.com Player/Editor File (Karaoke 4 Everyone.com)
    K7                   DCMO6 Emulator Tape Image 
    KAF                  Resurrection Matchmaker! MapPack (Wildman-Productions)
    KAL                  Sfkal Calender Kal_40 File 
    KAN                  Kwalitan Annotation (Vincent Peters, Department of Research Methodology, University of Nijmegen, The Netherlands)
    KAP                  MapInfo Sea Chart 
    KAR                  Keep-It Archive (Remember International Associates Ltd)
    KAR                  FOX+ 2.0 
    KAR                  Karaoke MIDI File 
    KAT                  Katorzer Music Composition 
    KAU                  KeyAudit Software Audit 
    KB                   Knowledge Pro Source Code 
    KB                   C++ Keyboard Script (Borland Software Corporation)
    KBD                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KBD                  Keyboard Script File Layout (Microsoft)
    KBF                  MyInfo Documnet/Outline (Milenix Software)
    KBI                  Keirsey-Bates Temperament Sorter Index 
    KBM                  Reflection Keyboard Script File Layout 
    KBM                  Scala Keyboard Mapping 
    KBN                  TeamSpeak Keybinding File (TeamSpeak Systems)
    KBP                  Keirsey-Bates Temperament Sorter Profile 
    KBR                  The Drums Pro file 
    KBR                  Keirsey-Bates Temperament Sorter Data 
    KBS                  ACE Studio Model Builder 
    KCC                  KeywordConvert Keyword/Phrase Combine List (FormatBlue Software)
    KCF                  KiSS Configuration File 
    KCH                  KChess Notation File (Ark Angles)
    KCL                  KeywordConvert Keyword/Phrase List (FormatBlue Software)
    KCL                  Kyoto Common LISP Source Code 
    KCO                  Kwalitan Codes (Vincent Peters, Department of Research Methodology, University of Nijmegen, The Netherlands)
    KCP                  Keychamp File 
    KCT                  Kwalitan Categories (Vincent Peters, Department of Research Methodology, University of Nijmegen, The Netherlands)
    KDB                  kinodb File (datb)
    KDB                  kusziDataBase Database File (kusziNET)
    KDB                  Doobs Database (Kyuran's World)
    KDB                  KeePass Password Database (Dominik Reichl)
    KDC                  Kwalitan Document (Vincent Peters, Department of Research Methodology, University of Nijmegen, The Netherlands)
    KDC                  Kodak RAW Bitmap Image (Eastman Kodak Company)
    KDELNK               KDE Desktop 
    KDF                  VisiQuest (Khoros) Standard Data Format (AccuSoft)
    KDF                  Kinkos Photo (Kinko's, Inc.)
    KDH                  CAJViewer 
    KDK                  Kodak Proprietary Decimated TIFF Format 
    KDO                  Kudo Picture Browser 
    KDT                  Medbasin Evaporation Data (National Technical University of Athens, Laboratory of Reclamation Works & Water Resources Management)
    KDW                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KE$                  Modem Bitware Fax Disk2 File 
    KED                  KEDIT Profile and Macro File 
    KEN                  Player CDcheck Compressed File 
    KEP                  SapShow Training File (SAP)
    KEP                  Turbo Pascal DOS TP19 Compressed Kepler File (Borland Software Corporation)
    KERNEL               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KES                  Kurzweil 1000 and 3000 Accessible Documents (Kurzweil Educational Systems, Inc.)
    KEV                  JAWS for Windows Preload File (Freedom Scientific)
    KEX                  KEDIT Macro File 
    KEXI                 Kexi Native Database File (KDE e.V.)
    KEXIS                Kexi Database Connection Parameter File (KDE e.V.)
    KEXT                 Mac OS X Kernel Extension (Apple)
    KEY                  Diablo II Key File (Blizzard Entertainment)
    KEY                  Forecast Pro Datafile 
    KEY                  FractInt Demo Script 
    KEY                  Keynote Presentation Wrapper Folder (Apple Computer)
    KEY                  Program Security or Registration File 
    KEY                  Infinity Game Engine Directory of Resources, Locations and Types (BioWare Corp.)
    KEY                  Keyboard Definition File 
    KEY                  National Construction Estimator 32 Costbook Key File 
    KEY                  Sentry 2020 Encryption File 
    KEY                  TBAV File 
    KEY                  Business Key Accounting File (Universal Systems)
    KEY                  DataCAD Icon Toolbar File (DATACAD LLC)
    KEY                  ArcView Geocoding Matching Keys (ESRI)
    KEY                  AntiVir PersonalEdition License Key File (AntiVir PersonalProducts GmbH)
    KEY                  Top Secret Crypto Gold (TAN$TAAFL Software Company)
    KEY                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KEY                  Kaspersky Anti-virus Toolkit Key File (Kaspersky Lab)
    KEY                  OS/2 Archive .KEY-file 
    KEY1                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KEYBOARD             Amaya Configuration 
    KEYCHANGER           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KEYS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KE_                  Ntgraph Ze32v215 File 
    KF2                  Max Payne Model File (Rockstar Games)
    KFA                  IRIS Shortened Format (eurobraille)
    KFA                  Dark Age of Camelot Animation 
    KFI                  IRIS Native Data File (eurobraille)
    KFL                  Pegasus Mail Auto-Reply Memory File (David Harris)
    KFL                  IRIS Literary (eurobraille)
    KFL                  KFLog Airfield File 
    KFL                  Kwalitan Filter (Vincent Peters, Department of Research Methodology, University of Nijmegen, The Netherlands)
    KFM                  KEY Form Designer Form Design (Broderbund)
    KFN                  KaraFun Karaoke Song (RECISIO S.A.R.L.)
    KFP                  Kinkos File Prep Tool 
    KFS                  Max Payne Model File (Rockstar Games)
    KFS                  Pegasus Mail Static Kill File (David Harris)
    KFT                  Matrox X Tools Keyframe (Matrox Electronic Systems Ltd.)
    KFX                  2D Graphic 
    KFX                  KoFax Group 4 Image 
    KGB                  Kremlin Encrypt File 
    KGB                  KGB Archiver Compressed Archive (Tomasz Pawlak)
    KGB                  Z80 Spectrum Emulator Snap/File Format 
    KGP                  Image 
    KHF                  Kwalitan Tree Structure (Vincent Peters, Department of Research Methodology, University of Nijmegen, The Netherlands)
    KHS                  Kwalitan History (Vincent Peters, Department of Research Methodology, University of Nijmegen, The Netherlands)
    KHTMP                Google Earth Temporary Object Location (Google)
    KIC                  Kodak Image Compression File (Eastman Kodak Company)
    KICK                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KICK3                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KID                  Kidspiration (Inspiration Software, Inc.)
    KID                  Tonline Bsw4 Install Mdmimp File 
    KID                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KIF                  Knowledge Interchange Format Draft Proposition 
    KIF                  AutoCAD Key Index (Autodesk, Inc.)
    KIK                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KIL                  Interview File 
    KILLER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KIM                  Commodore KIM-1 Program Image File 
    KIN                  Kinemage Chemical Protein Language 
    KIP                  OA-KIP Client Data Transfer File (van Ovost Automatisering b.v.)
    KIT                  Raven Toolkit File 
    KIT                  Battery Drum Kit (Native Instruments)
    KIT                  20-20 Design Kitchen Design (20-20 Technologies)
    KIT                  Power Chord Drum Kit 
    KITCHEN              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KITTEN               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KIX                  KiXtart Script 
    KIZ                  UUencoded File 
    KIZ                  Kodak Digital Postcard File 
    KJV                  TheWord & Bible Companion King James Version Bible 
    KKT                  TTSKONE 
    KKW                  RoboHELP Keyword Help Project Index File 
    KLB                  Klemmbretter 
    KLB                  GraphicWorks Album 
    KLD                  Dungeon Keeper 2 Playable Map (Electronic Arts, Inc.)
    KLI                  KalKulator Linear Algebra File 
    KLIP                 KlipFolio Klip (serence inc.)
    KLN                  COLOGNE Multilingual Transliteration Scheme 
    KM                   BMC Software Patrol File 
    KM2                  ActiveModeler Professional Process Map (KAISHA-Tec Co. Ltd.)
    KMA                  Kodak Memory Album (Kodak)
    KMA                  Correlate K-Map Structure (Correlate AS.)
    KMA                  Video Hoster Karaoke File (Micro Technology Unlimited)
    KMC                  Correlate (Correlate AS.)
    KMD                  Model Browser File (Metis)
    KMD                  ABSTAT for Windows Batch File (AndersonBell Corp)
    KMD                  Media Desktop File (Kaaza)
    KMD                  Clicker Module Output 
    KMD                  AIMAX Keyboard Macro (TA Engineering)
    KMF                  Kwalitan Memo (Vincent Peters, Department of Research Methodology, University of Nijmegen, The Netherlands)
    KML                  HP48 Emulator Win48 Script 
    KML                  Google Earth Placemark (Google)
    KML                  Keyhole Placemark (Google)
    KML                  Kedit Macro Library 
    KML                  Emu48 Emulator Keyboard Configuration File 
    KMN                  SIL Hebrew Font System Keyboard 
    KMP                  Korg Trinity KeyMaP File 
    KMP                  Knowledge Manager Knowledge Base Compressed Map (Hypersoft-net)
    KMP                  Correlate K-Map KnowledgeMap (Correlate AS)
    KMP                  Ketchum Media Player Playlist (Ketchum Software Inc.)
    KMS                  Model Browser Shortcut (Metis)
    KMS                  Correlate Solution Topics (Correlate AS.)
    KMV                  Model Browser File (Metis)
    KMZ                  Google Earth Saved Working Session (Google)
    KMZ                  Correlate K-Map Zipped KnowledgeMap (Correlate AS)
    KM_                  Seqmax Seqprest Compressed Audio File 
    KNF                  Orchida Knitting System (Orchida Soft)
    KNF                  Personal Knowbase Custom Import Filter (Bitsmith Software)
    KNF                  Universal Flat Knitting Machine Pattern Design 
    KNN                  Clarion for Windows Database Key (SoftVelocity)
    KNO                  Personal Knowbase Data (Bitsmith Software)
    KNOW                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KNT                  KeyNote Note File (Marek Jedlinski)
    KNW                  Known Problems 
    KO                   IBM/Tools/Updater File 
    KOA                  Koala Paint C64 Bitmap 
    KOD                  Code 
    KOE                  Turbo Pascal DOS File (Borland Software Corporation)
    KOK                  DEL MP3 Karaoke 
    KOM                  Kit-O-Mat 3D Model File (KingMill Enterprises LLC)
    KON                  Konfabulator Widget Code (Konfabulator)
    KOR                  Korean Text File 
    KOR                  Correction File 
    KOS                  MicroType Pro Document 
    KOSIMA               Kosima File 
    KOURSE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KOZ                  Sprint Music Store Music Download fof Phone (Sprint Nextel)
    KPD                  Kapital Database File (K Software)
    KPF                  Kapere 
    KPG                  Kai'S Power Goo Graphic 
    KPJ                  K Database Magic Project File (Sandor Kovacs)
    KPL                  Kazaa Playlist 
    KPL                  KPL Source Code 
    KPP                  SmartPad Toolpad 
    KPP                  Kudo Place Pack, Place Module 
    KPP                  Kid Pix Deluxe Slide Show (Broderbund)
    KPR                  Kwalitan Project (Vincent Peters, Department of Research Methodology, University of Nijmegen, The Netherlands)
    KPR                  KPresenter Slide Show (KOffice Web Team)
    KPR                  Fifa World Cup Game Data Comon be File 
    KPS                  IBM KIPS Bitmap 
    KPT                  KPresenter Template (KOffice Web Team)
    KPX                  Kid Pix Deluxe Data File (Broderbund)
    KQL                  Kyle's Quest Level Editor Graphic (Crimson Fire Entertainment)
    KQP                  Konica Quality Picture 
    KR1                  Kurzweil 2000 Sample (Kurzweil Music Systems, Inc.)
    KRA                  Krita Image (The KDE Project)
    KRA                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KRACK                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KRACKS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KRAK                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KRAW                 Macintosh OS X Kernel-Raw Image Format (Apple Computer, Inc.)
    KRC                  Mouse and Key Recorder Recorded File (Kra-Tronic Corporation)
    KRIS                 Kris Tracker Format Song 
    KRISTAL              KRISTAL Audio Engine Project File (Kreatives.org)
    KRK                  Xbox Music Mixer Karaoke-enabled Music File (Microsoft Corporation)
    KRK                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KRN                  NBA99 Ice Hockey Foreground Font 
    KROMHE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KRP                  HiCAD 3D Part File (ISD Benelux B.V.)
    KRX                  Karaoke MIDI File 
    KRY                  Herolab Krypten 
    KRZ                  Kurzweil 2000 Sample (Kurzweil Music Systems, Inc.)
    KS                   SOF Game File Log 
    KS                   Works Sheet (Microsoft)
    KSC                  Korg Trinity Script 
    KSE                  Keepsake Studio Export File 
    KSF                  Korg Trinity Sample 
    KSF                  Kick it Up Stepfiles 
    KSG                  Kwalitan Segment (Vincent Peters, Department of Research Methodology, University of Nijmegen, The Netherlands)
    KSH                  Prodigy Internet Registration File 
    KSH                  Dazzle Dazzle Realization 
    KSH                  UNIX Shell Script 
    KSHRC                UNIX Kom Shell Environment File 
    KSL                  Karaoke Song List Creator Song List (Airwer)
    KSM                  Kefrens-Sound Machine Format Song 
    KSM                  Embroidery File Format for Pfaff Machines 
    KSM                  Ken Silverman Music Format 
    KSS                  KDE [UNIX/Linux Desktop] Screen Saver (KDE)
    KSS                  MSX Music File 
    KSS                  K-ForCE Scoresheet File (Stephan Rakonza)
    KST                  Olivetti Olitext Plus Script File 
    KSW                  Kepler Scientific Workflows (Kepler Project)
    KT2                  BATTERY 2 Drum Kit (Native Instruments)
    KTI                  Kelvin TOP-SET Investigator Investigation Data (The Kelvin Consultants Limited)
    KTK                  Kutoka's Mia 
    KTP                  Clarion for Windows Temporary Key File (SoftVelocity)
    KTT                  KeyText Data File 
    KTX                  Kwalitan Text (Vincent Peters, Department of Research Methodology, University of Nijmegen, The Netherlands)
    KU                   Knowledge Explorer Knowledge Representation Knowledge Unit 
    KUMAC                Physics Analysis Workstation Macro File 
    KUP                  Karaoke Song List Creator (Airwer)
    KV3                  KM-NET Viewer Database (Kyocera Mita Corporation)
    KVB                  Klick-N-View Business Cards 
    KVQ                  Sega Smash Pack PC Version File (Sega)
    KVT                  BASIC QuickBAS QuickB03 File 
    KVW                  Google Earth Workspace (Google)
    KW$                  NTgraph Visual C Wizzard File 
    KW3                  KanjiWORD Document (Pacific Software Publishing, Inc.)
    KWB                  Kwalitan Workfile (Vincent Peters, Department of Research Methodology, University of Nijmegen, The Netherlands)
    KWB                  KeyWord Braille File (Pulse Data International Ltd)
    KWD                  Dungeon Keeper 2 Playable Map (Electronic Arts, Inc.)
    KWD                  Keyword 
    KWD                  KWord File (KWord Team)
    KWF                  Kwalitan Word List (Vincent Peters, Department of Research Methodology, University of Nijmegen, The Netherlands)
    KWF                  Kartex 
    KWF                  Delphi Pascal Chart 
    KWI                  Navigation System Data File (KIWI Format) 
    KWIC                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    KWS                  Kyocera Ringtone 
    KXS                  Kexis Lossless Compressed Audio 
    KY                   Peachtext List Manager (Peachtree Software (Sage Software SB, Inc.))
    KYB                  Keyboard Layout 
    KYE                  Kye Game Data 
    KYF                  Visual-Voice Mouth Movement File 
    KYS                  Photoshop CS Keyboard Shortcut (Adobe)
    KYS                  SourceOffSite Server Keys 
    KYS                  Premiere Keys File (Adobe Systems Inc.)
    KZP                  Kazoo3D or KazooStudio KazooPicture 
    L                    Link Instructions 
    L                    LEX Program File 
    L                    LISP Program File 
    L                    WATCOM Wlink Directive 
    L                    Xcode Core Lex Program (Apple Computer, Inc.)
    L                    Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    L                    IRIS Editor Literary Format File (eurobraille)
    L                    Game of Life Pattern File 
    L                    Cliq Accessories Datebook Lock File 
    L2L                  IBM Voice Type Language Addword File 
    L2P                  IBM Voice Type Language Addword File 
    L2R                  Lineage II Replay (NC Interactive, Inc.)
    L3G                  Compressed Archive 
    L4D                  Cinema 4D Layout (MAXON Computer GmbH)
    L5K                  RSLogix 5000 (Rockwell Automation)
    L64                  64LAN Image File 
    L95                  Library File 
    LA                   Netscape Packetized Audio 
    LA                   GNU Libtool Library (GNU Project - Free Software Foundation)
    LA                   Lightscape Animation 
    LA                   LA Lossless Compressed Audio 
    LA1                  Liquid Audio Player Resource DLL (Geneva Media, LLC)
    LAA                  LucasArts AdLib Audio File Format 
    LAB                  NCSS/SOLO Data File 
    LAB                  NGSofts.com LAB Editor (NGSofts.com LAB Editor)
    LAB                  Mailing Labels 
    LAB                  Q+E Mailing Label 
    LAB                  NCSS Data File 
    LAB                  Complete Works Label (Toplevel Computing)
    LAB                  Outlaws Container File (LucasArts Entertainment Company LLC)
    LABELS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LABLES               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LAD                  Daylon Leveller Animation Data 
    LAF                  Essential Taceo User License Activation File (Essential Security Software, Inc.)
    LAM                  Netscape Media Player Streaming Audio Metafile 
    LAMP                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LAN                  ERDAS IMAGINE LAN Sensor Data (Leica Geosystems , LLC)
    LAN                  Novell NetWare LAN Drivers 
    LAND                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LANDSAT              EOSAT Landsat Thematic Mapper Bitmap Data 
    LANG                 Skype Language File (Skype Technologies S.A.)
    LANG                 CDex Language File 
    LANGUAGE             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LAO                  EPSON Smart Panel File 
    LAPTOP               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LAR                  Liquid Audio Player Registration File (Geneva Media, LLC)
    LARG                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LARGE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LARRY                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LAS                  Log ASCII Standard (LAS) File (The Canadian Well Logging Society (CWLS))
    LASE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LASER                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LASSO                Lasso Studio Database Driven Web Page (Blue World Communications, Inc.)
    LAST                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LAT                  LATEX 
    LAT                  Crossword Express Lattice File (AUS-PC-SOFT)
    LATEST               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LATEX                LaTeX Source Document (LaTeX3 Project)
    LAUNCH               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LAUNCHER             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LAV                  Liquid Audio Player Voucher File (Geneva Media, LLC)
    LAV                  Listlabl File 
    LAVAITTE             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LAVS                 Liquid Audio Player Secure Streaming Voucher File (Geneva Media, LLC)
    LAW                  eLawForms Form File (E*Law Publishing Limited)
    LAWS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LAY                  Mastercook Layout File (Sierra)
    LAY                  MiTek Engineering File 
    LAY                  Track Builder Simulation Compiler Code (Signal Computer Consultants)
    LAY                  Rhino 3D Layer Text File (Robert McNeel & Associates)
    LAY                  Lightscape Layer State 
    LAY                  Pro/ENGINEER (PTC)
    LAY                  Applause Word Chart Layout 
    LAYER1               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LAYOUT               Premiere Layout (Adobe Systems Inc.)
    LAYOUT               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LAYUP                MSC.Patran Laminate Modeler Composites Ply Layup Information (MSC.Software Corporation)
    LA_                  TI Databook File 
    LB                   Novanew File 
    LB                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LB                   LB Packer Sound File 
    LBA                  Little Billy's Album Photo Album Project File (Little Billy's Album, Inc.)
    LBA                  Liberty BASIC File 
    LBD                  Life Balance User Data (Llamagraphics, Inc.)
    LBD                  Grafic FLI File 
    LBE                  Life Balance XML-based Data Exchange (Llamagraphics, Inc.)
    LBG                  dBASE IV Label Generator 
    LBI                  Dreamweaver Library File 
    LBI                  Fifa 2001 Data Feart Legal Screeen 
    LBK                  VersaPro Local Logic Block Backup 
    LBL                  Markin Label File (Markin)
    LBL                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LBL                  British National Bibliography File 
    LBL                  Label 
    LBL                  Creatacard Label Project (Broderbund)
    LBM                  Amiga Interleaved Bitmap Format (Amiga)
    LBM                  Deluxe Paint (Electronic Arts, Inc.)
    LBM                  XLib Linear Bitmap Graphic 
    LBM                  Deluxe Paint Bitmap 
    LBM                  Graphic, Sound 
    LBO                  dBASE IV Ordered Labels 
    LBR                  Compressed Archive File 
    LBR                  1-2-3 Display Driver (IBM)
    LBR                  Omnis7 Application File (Omnis Software)
    LBR                  CP/M Information File 
    LBS                  Omnis Studio Application Library (Raining Data Corporation)
    LBT                  Foxpro Label Memo (Microsoft)
    LBV                  Tonline Ob4hbci Layout Print File 
    LBW                  CCP Log Server Utility Log File (CCP)
    LBX                  Foxpro Label (Microsoft)
    LBXADD               Libronix DLS Addin 
    LBXALV               Logos Bible Software Series X Passage In All Versions 
    LBXCBV               Logos Bible Software Series X Copy Bible Verses Style 
    LBXCIT               Libronix DLS Citation Format 
    LBXCMP               Libronix DLS Company Information 
    LBXCOL               Libronix DLS Resource Collection 
    LBXCTB               Libronix DLS Custom Toolbar 
    LBXDAT               Libronix DLS Data File 
    LBXDTP               Libronix DLS Data Type 
    LBXGYQ               Libronix DLS Graphical Query 
    LBXLBK               Libronix DLS License Backup 
    LBXLIC               Libronix DLS Licenses File 
    LBXLLS               Libronix DLS Resource 
    LBXNOT               Libronix DLS Notes 
    LBXOEB               Libronix DLS Resource 
    LBXPDF               Libronix DLS Resource 
    LBXPLN               Logos Bible Software Series X Bible Reading Plan 
    LBXPLP               Logos Bible Software Series X Parallel Passage 
    LBXPLV               Logos Bible Software Series X Parallel Bible Versions 
    LBXPRF               Libronix DLS Preferences 
    LBXPRL               Logos Bible Software Series X Prayer List 
    LBXRAS               Libronix DLS Resource Association 
    LBXRTP               Libronix DLS Resource Type 
    LBXSDG               Logos Bible Software Series X Sentence Diagram 
    LBXSHL               Libronix DLS Shell 
    LBXUPD               Libronix DLS Update Script 
    LBXVRS               Logos Bible Software Series X Verse List 
    LBXWDL               Libronix DLS Word List 
    LBXWKS               Libronix DLS Workspace 
    LBZ                  Leibniz Mathematical Word Processor 
    LC                   Textbridge Classic Bin File 
    LCA                  SimpleFormat Changeables Graphic 
    LCB                  Concept2 LogCard Utility LogCard Backup Image (Concept2, Inc.)
    LCB                  Living Cookbook File 
    LCC                  Capture One PRO Lens Cast Calibration File (Phase One)
    LCC                  IBM Voice Type Language Newuser File 
    LCD                  LeoCAD Project (BT Software)
    LCD                  CDspace CD Emulator 
    LCD                  ABBYY Finereader 4.0 Sprint (ABBYY Software House)
    LCE                  SimuPLC Electric Commands Language Source Program (André Luiz Barbosa)
    LCF                  Linker Control File 
    LCF                  HP-95LX Datacomm Script File 
    LCH                  PagePath Technologies LAUNCH! File 
    LCH                  IBM Works for OS/2 Chart 
    LCI                  IBM Voice Type Language Newuser File 
    LCK                  Paradox Lock File (Corel)
    LCK                  PageMaker Database (Adobe)
    LCK                  Frontpage-related File (Microsoft)
    LCK                  Help 2 Lock File (Microsoft)
    LCK                  Lock File 
    LCK                  Dreamweaver Lock File (Adobe Systems Incorporated)
    LCK                  Pegasus Mail Mailbox Lock File (David Harris)
    LCK                  ACT! Database Locking File (Best Software CRM Division)
    LCK                  Inventor Locking File (Autodesk, Inc.)
    LCL                  FTP Software Data 
    LCL                  Checklist Long-term Checklist (Checklist à long terme) (Francis Gagné)
    LCN                  WordPerfect Dictionary File (Corel Corporation)
    LCO                  Railroad Tycoon 3 Locomotive Properties (PopTop Software Inc.)
    LCS                  L0phtCrack (@stake)
    LCS                  ACT! History File (Best Software CRM Division)
    LCT                  Elsie Electrical Filter Design (Tonne Software)
    LCW                  Lucid 3-D Spreadsheet 
    LD                   Paradox Language Driver (Corel)
    LD                   Telix Long Distance Codes 
    LD                   PC-Bibliothek User Options (Bibliographisches Institut & F.A. Brockhaus AG)
    LD1                  dBASE Overlay File 
    LDA                  LaserData 
    LDB                  Access Lock File (Microsoft)
    LDB                  Settler for Windows Temporary File 
    LDB                  TrueVector Log Database 
    LDB                  RPG Maker 2000/2003 Database (Enterbrain)
    LDB                  VisiTrax Music Library Database (Synapsa Productions)
    LDB                  IBM Works for OS/2 Data Filer Database 
    LDC                  Steuer2001 File 
    LDD                  Laser Dispatch Data 
    LDF                  ProWORX Nxt Logic Data (Schneider Electric)
    LDF                  IBM Works for OS/2 Data Filer Form 
    LDF                  LuraDocument File 
    LDF                  TriboIndenter Air Indent Electrostatic Force Constant Calibration (Hysitron Incorporated)
    LDF                  enotate Layered Data Format File (Informal Software, D.I.P., Ischian Systems)
    LDF                  SQL Server Transaction Log File (Microsoft)
    LDF                  Geoworks Glue Library Definition 
    LDI                  LDIF File 
    LDIF                 LDAP Data Interchange Format 
    LDL                  Corel Paradox Delivered Library 
    LDL                  OzWin CompuServe E-mail/Forum Access Download Request 
    LDP                  Label Designer Plus DELUXE Data File (CAM Development)
    LDR                  Yamaha Live Drum Kits (Yamaha Corporation of America)
    LDR                  Symantec Ghost Template File (Symantec Corporation)
    LDR                  LDraw Model File (Estate of James Jessiman)
    LDS                  LOUDNESS Sound System 
    LDS                  Laser Dispatch Script 
    LDS                  Trane Trace Load 
    LDS                  Corel40 Programs Data File 
    LDX                  Liquid Studio Liquid Executable Bytecode (Global Heavy Industries)
    LD_                  Visc15 Adressen Setup File 
    LE                   DOS/4GW PM 32X DOS Extender Executable Image 
    LE$                  BASIC VB 
    LEECHGET             LeechGet Download Manager Temporary File (Martin Albiez)
    LEF                  Lenex XML Data (GeoLogix AG)
    LEFT                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LEFTY                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LEG                  Legacy Graphic Format 
    LEGIT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LEMONS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LEN                  XNBC V8 Biological Neural Network Simulation Workstation Axon Lengths 
    LEN                  PlatePrep 
    LEN                  Sheffield Dictionary File 
    LENGTH               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LENTH                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LEP                  Label and Envelope Designer Plus 2000 
    LES                  Mklesson Lesson File 
    LES                  Logitech Entertainment Game Profile 
    LES                  Lesson File 
    LES                  Learn Lesson File 
    LESPAUL              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LET                  Nuke Randomic Life Generator 
    LET                  Creatacard Letterhead Project (Broderbund)
    LET                  Genesis 2000 (Genesis 2000, Inc.)
    LET                  Maestro Icons File 
    LET                  Letter 
    LETTER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LETTERS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LEV                  Worms Armageddon Map 
    LEV                  NetHack Level File 
    LEV                  Worms World Party Map 
    LEV                  Liero Level 
    LEVEL                Homeworld Level Data (Sierra Entertainment, Inc.)
    LEVEL                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LEX                  Lexicon (Dictionary) 
    LEX                  Lexmark Printer Install File 
    LEX                  HP-Basic Language Extension 
    LEX                  Acrobat (Adobe)
    LEY                  Leylines Map File or Saved Game (Crystal Shard)
    LE_                  BASIC VB Compressed Disk1 File 
    LF                   SoftwareKey License File 
    LFA                  LifeForm File 
    LFA                  IncrediMail Default Animations (IncrediMail Ltd.)
    LFD                  LucasArts Games Resource (Lucasfilm Entertainment Company Ltd.)
    LFF                  LucasFilm Format 
    LFG                  LensForge Lens Design Specification (Ripplon Software Inc.)
    LFL                  LucasFilm Library 
    LFM                  LifeForm File 
    LFP                  LifeForm File 
    LFQ                  LeechFTP Queue File 
    LFT                  3D Studio Loft File (Autodesk, Inc.)
    LFT                  ChiWriter Laser Printer Font 
    LFXM                 FaceMail Text to Speech E-mail 
    LG                   Logo Procedure Definition 
    LG                   Archive Handler by George Lyapko 
    LG                   Arhangel Compressed Archive 
    LGA                  Windows Applog File 
    LGC                  Screen Saver Toolkit (Stardust Software)
    LGC                  Logicator File 
    LGC                  Program Use Log File 
    LGC                  Corel Layout Designer Logic File for SGML 
    LGC                  Windows Application Log 
    LGC                  Audio 
    LGD                  Windows Application Log 
    LGE                  Windows Application Log 
    LGF                  ProWORX Nxt Log Book Note (Schneider Electric)
    LGF                  Legal Eagle Text File 
    LGF                  LeechGet Download Manager FTP Queue (Martin Albiez)
    LGF                  Logomotion Graphic File (Logotron Ltd.)
    LGF                  Windows Application Log 
    LGG                  Windows Application Log 
    LGH                  Windows Application Log 
    LGH                  Pro/ENGINEER (PTC)
    LGI                  Windows Application Log 
    LGJ                  Windows Application Log 
    LGK                  Windows Application Log 
    LGL                  Windows Application Log 
    LGL                  LeechGet Download Manager Download Queue (Martin Albiez)
    LGM                  Windows Application Log 
    LGN                  Windows Application Log 
    LGO                  MSWLogo Saved Logo Design (Softronics, Inc.)
    LGO                  Modern ListGeo Output 
    LGO                  Windows Logo Driver (Microsoft)
    LGO                  SuperFax Header/Footer Logo 
    LGP                  Logitech G15 Gaming Keyboard Profile File (Logitech)
    LGP                  Final Fantasy Game Archive (Square Enix Co., Ltd)
    LGP                  Windows Application Log 
    LGP                  Logotron SuperLogo File 
    LGQ                  Windows Application Log 
    LGR                  Windows Application Log 
    LGR                  Elasto Mania Character Skin 
    LGS                  Windows Application Log 
    LGT                  The Sims Lighting File (Electronic Arts (Maxis))
    LGT                  Urban Chaos Game File 
    LGX                  Logistix Spreadsheet 
    LGZ                  Windows Application Log 
    LH                   Morph File 
    LHA                  Compressed Archive File 
    LHP                  Electric Butterfly 
    LHS                  Literate Haskell Source File 
    LHTS                 The Movies Editor Text String Database (The Movies Workshop)
    LHW                  Compressed Amiga Archive File 
    LHZ                  LHA Compressed Archive File 
    LI$                  MS Compressed Library 
    LIA                  P-CAD Schematic Library (Altium Limited)
    LIB                  AS/400 Physical File Library (IBM)
    LIB                  Sibelius House Style File (Sibelius Software Ltd)
    LIB                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LIB                  Game Maker Drag & Drop Actions Library (YoYo Games Limited)
    LIB                  Scholar's Aid Library File 
    LIB                  Greeting Card Library 
    LIB                  Library Used by an Application 
    LIB                  Program Library Common Object File Format (COFF) (Microsoft)
    LIB1                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LIB2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LIB3                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LIBRARIES            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LIBRARY              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LIBS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LIC                  License File 
    LIC                  Borland License File (Borland Software Corporation)
    LIC                  FLEXnet Manager License File (Macrovision Corporation)
    LICENSE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LICK4                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LID                  Dylan Library Interchange Description 
    LID                  Light Field Description File 
    LID                  Maple V Setup File (Maplesoft, a Division of Waterloo Maple Inc.)
    LID                  Quicken 2002 Deluxe File (Intuit Inc.)
    LID                  Scholar's Aid Backup File 
    LID                  WinDVD File 
    LID                  LabelVision Auto Incrementing Value File 
    LIF                  HP Logical Interchange Format File 
    LIF                  Knowledge Dynamics Install Archive 
    LIF                  TurboTax Information Archive (Intuit)
    LIF                  Compressed Archive File 
    LIF                  Homeworld Layered Image Format File (Sierra)
    LIF                  CA Clipper Installer Compressed File Archive 
    LIF                  Game of Life Pattern File 
    LIFE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LIFE                 Game of Life Pattern File 
    LIGH                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LIGHT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LIGHTNING            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LILLIES              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LIM                  Limit Compressed Archive 
    LIN                  AutoCAD Linetype Definition (Autodesk, Inc.)
    LIN                  Medlin Accounting Current Period Invoice Detail (Medlin Accounting)
    LIN                  VAX Program Listing 
    LIN                  ArcView ARC/INFO Lineset Symbol File (ESRI)
    LIN                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LINDBERG             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LINEEDIT             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LINES                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LINK                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LINKTP               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LINO                 LinoLab Source Code (Linova)
    LINUX                File for Linux 
    LIP                  SLIP 
    LIQ                  Liquid Tracker Song/Module 
    LIQ                  CollegeBAR Bartending Software Liquor Cabinet 
    LIQCRYS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LIR                  Liram Application Save Application Setting (Liram Ltd.)
    LIS                  Structured Query Reporting (SQR) Output 
    LIS                  VAX Program Listing 
    LIS                  Compiler Listing File 
    LISP                 LISP Translators File 
    LISPM                LISP Machine File Bitmap 
    LIST                 BlindWrite (VSO-Software)
    LIST                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LISTER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LISTING              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LISTS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LIT                  AMOS Executable File (Alpha Micro)
    LIT                  Reader eBook File (Microsoft Corporation)
    LIT                  Movie 
    LIV                  Living Actor Actor File (Cantoche)
    LIV                  Yamaha Live Voice File (Yamaha Corporation of America)
    LIV                  LiveMotion SWF Movie 
    LIVEREG              Symantec Anti-Virus Update Session (Symantec Corporation)
    LIVESUBSCRIBE        Symantec Anti-Virus Update Catalog (Symantec Corporation)
    LIVEUPDATE           Norton Anti-Virus Update Settings File (Symantec)
    LIX                  Extend Simulation Library File 
    LIX                  Libronix DLS Resource 
    LIZ                  Comprressed List File 
    LI_                  Rukus File 
    LJ                   HP LaserJet Graphic Bitmap 
    LJ                   Text File for HP LJ II Printer 
    LK                   Advanced Revelation and OpenInsight Primary Data Storage Space (Revelation Software)
    LKB                  Faverly Data Recorder Output 
    LKD                  Pioneer OEL Screensaver (Pioneer Electronics (USA) Inc.)
    LKO                  Outlook Express Linked Object (Microsoft)
    LKP                  Systems Management Server (SMS) Lookup File (Microsoft Corporation)
    LKS                  WinAmp Links File 
    LKT                  Lookup Table 
    LKU                  T9000 Linkable Unit 
    LL                   Links Language Template File 
    LL                   List and Label (combit GmbH)
    LL3                  LapLink III Document File 
    LLB                  LabVIEW Application Program Library (National Instruments Corporation)
    LLC                  LapLink Saved Connection File 
    LLD                  Links Language Data File 
    LLG                  LADSIM Ladder Logic Diagram (Bytronic Ltd)
    LLP                  Lotto Logic 2002 Lottery Configuration File (Timersoft)
    LLP                  VersaPro Local Logic Block 
    LLP                  Digital Chart of the World Data File 
    LLT                  Digital Chart of the World Data File 
    LLX                  List and Label Extension Object (combit GmbH)
    LLX                  Laplink Exchange Agent 
    LLX                  Tonline Ob4hbci File 
    LM8                  Picture File 
    LMA                  Learning Mobile Author (LMA) Project File (Hot Lava Software)
    LMA                  Netscape Packetized Audio 
    LMD                  Tune Smithy List of Modes for a Musical Scale 
    LMD                  ABBYY Finereader 4.0 Sprint (ABBYY Software House)
    LME                  LegglessMusicEditor Sound/Module 
    LMI                  HomePortal Firmware Recovery File (2Wire, Inc.)
    LMK                  Sprint-Layout Macro (ABACOM Ingenieurbüro)
    LML                  LOGML File 
    LMP                  Quake File 
    LMP                  Doom Recorded Demo File (Id Software, Inc.)
    LMS                  Light-O-Rama Lighting Control File (Light-O-Rama)
    LMT                  RPG Maker 2000/2003 Map Tree (Enterbrain)
    LMU                  RPG Maker 2000/2003 Map File (Enterbrain)
    LMV                  LiveSwif Movie File 
    LMV                  LEGO CAD Movie (The LEGO Group)
    LMW                  LMW Examples File 
    LMX                  Lomax Software Game File 
    LN03                 DEC LN03+ Sixel Bitmap 
    LNB                  Logomanager File 
    LND                  3D Landscape Data 
    LNED                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LNF                  Oyoaha lookandfeel Interface Elements (Oyoaha)
    LNG                  Launch-n-Go Shared Keyboard Shortcuts (Tethys Solutions, LLC.)
    LNG                  Language File 
    LNG                  DeepBurner Language File (ASTONSOFT Co.)
    LNG                  SourceEdit Language Definition 
    LNG                  Acrobat Language Plug-in (Adobe)
    LNG                  Diablo II (Blizzard Entertainment)
    LNG                  Jasmine Language File 
    LNG                  NRG SDR Language File 
    LNG                  Lahey FORTRAN Language Extension 
    LNG                  Lingo Script File 
    LNGTH                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LNK                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LNK                  Linker File 
    LNK                  Advanced Revelation Database File (Revelation Software)
    LNK                  Windows Shortcut File (Microsoft Corporation)
    LNK                  XNBC V8 Biological Neural Network Simulation Workstation Link Configuration File 
    LNK2                 Program Neighborhood Explorer Shortcut (Citrix Systems, Inc.)
    LNM                  WordPerfect SGML Alias (Corel Corporation)
    LNR                  Lunar Invasion Saved Game 
    LNS                  LinkStash Bookmark File 
    LNS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LNT                  ARMACON Data File (LNT Engenharia [LNTinfopage])
    LNT                  LiveNote Case File (LiveNote Technologies)
    LNX                  Atari Lynx ROM Image 
    LNX                  Lynx Project File (Zippety Software)
    LO                   GNU Libtool Library Object File (GNU Project - Free Software Foundation)
    LO                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LO4                  SeeYou Flight Data (Naviter)
    LOAD                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOADER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOADS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOAN                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOANS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOANX                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOC                  EasyGPS TerraByte Location File (TopoGrafix)
    LOC                  Suppose Locations File 
    LOC                  MicroSim PCBoard Component Locations Report 
    LOC                  Localisation String Resource Header File 
    LOCAL                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOCAL                Xset Setup Program Bin File 
    LOCATOR              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOCK                 .NET Framework Database Lock File (Microsoft)
    LOCKED               Softwrap Locked and Encrypted EXE File (Softwrap)
    LOCKED               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOCKSMITHING         Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOD                  Load File 
    LOD                  Homeworld Level Of Detail (Sierra Entertainment, Inc.)
    LOF                  LaTeX Auxiliary File for List Of Figures 
    LOG                  Log File 
    LOG                  MS HTMLHelp Log 
    LOG                  SeeYou Flight Data (Naviter)
    LOG                  ProWORX Nxt Text Log File (Schneider Electric)
    LOG                  Pro/ENGINEER (PTC)
    LOG                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOGIN                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOGO                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOGO2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOGO3                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOGO4                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOGOFF               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOGON                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOGONXP              LogonStudio Document (Stardock Corporation, Inc.)
    LOK                  OmniPage Ver 9 File (ScanSoft, Inc.)
    LOK                  FileWrangler/ZipWrangler/SecurDesk! Encrypted Archive 
    LOLLY                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LONG                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOOK                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOOP                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOOP                 Soundtrack Hinted Audio Loop (Apple)
    LOOP1                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOOP2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOOPA6               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOP                  Sierra Mastercook Layout File 
    LOSANGELE            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOT                  LaTeX Auxiliary File for List Of Figures 
    LOUD                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOUD3                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOVE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOW                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LOWSTRINGS           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LO_                  Systems Management Server (SMS) Renamed Old Log File (Microsoft Corporation)
    LP                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LP                   FlexRip/B 
    LP                   LP88 Linear Programming for the IBM-PC File 
    LP                   Lightscape Preparation File 
    LP                   NASCAR Racing 2002 Track Configuration Options (Sierra)
    LP1                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LP3                  Hansen Software Label Printer 
    LP4                  Hansen Software Label Printer 
    LP7                  LP7Signer Digital Signed File (Lex Persona)
    LPA                  LPA-View Laser Particle Analyser Test Analysis Software 
    LPB                  LP88 Linear Programming for the IBM-PC File 
    LPB                  ProWORX Nxt Logic Pointer (Schneider Electric)
    LPC                  LivePC Engine Subscription Information (moka5)
    LPC                  Teko Printer Driver 
    LPC                  Linear Predictive Coding Sound 
    LPD                  Label Pro Ver. 2.0-3.02 Data File (Avery)
    LPD                  LECTURNITY Multimedia Audio/Video/Clips Presentation Document (IMC AG)
    LPD                  Helix Nuts and Bolts File 
    LPD                  LogonProcessor Configuration Database (VAI)
    LPF                  Lytec's Direct Electronic Medical Claims ClaimsDirect 
    LPF                  ProWORX Nxt Logic Pointer (Schneider Electric)
    LPF                  LogicPort LA1034 Logic Analyzer Project (Intronix Test Instruments, Inc.)
    LPG                  Lowrey Organ Registration (Setup) File (Lowrey Organ Company)
    LPI                  Live Pictures 
    LPJ                  PASSOLO Localization Project 
    LPJ                  SoftwareShield License Manager License Project File (SoftwareShield Technologies Inc.)
    LPK                  License Package 
    LPL                  Darkstone Character Information File 
    LPN                  LVPlan - Planification of Low Voltage Aerial Distribution Networks 
    LPP                  Let's Photo Photo Slideshow (Canopus Co., Ltd.)
    LPR                  LAPRO Source Code File (Marco Mugnatto)
    LPR                  Homeworld (Sierra Entertainment, Inc.)
    LPROJ                Mac OS X Localized Project Folder 
    LPT                  LogPlot/LogView Compiled Boring Log (RockWare, Inc.)
    LPU                  PASSOLO Localization Project 
    LQ                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LQF                  Leech FTP Save Queue File 
    LQP                  Liquid Audio Player Passport File (Geneva Media, LLC)
    LQT                  Liquid Audio Player File (Geneva Media, LLC)
    LQT                  Winamp File 
    LR                   Chaos Desktop Layer List (Chaos systems AB)
    LRC                  Dance With Intensity 
    LRC                  Intel Video Phone File 
    LRF                  C/C++ Linker Response File (Microsoft)
    LRG                  Macromedia XRes Multi-resolution Bitmap 
    LRM                  Encarta Class Server Learning Resource (Microsoft)
    LRP                  IBM Works for OS/2 Report 
    LRP                  Linux Router Project Package 
    LRR                  Mercury LoadRunner Test Results File (Mercury Interactive Corporation)
    LRS                  Mercury LoadRunner Scenario (Mercury Interactive Corporation)
    LRS                  Language Resource File (Corel)
    LRS                  Daylon Leveller Rendition Style 
    LRT                  Yacc: Yet Another Compiler-Compiler Log File 
    LRU                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LRZ                  Lotto Hat File 
    LS                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LS                   Unix ls Directory File 
    LS                   Freeware Dictionary Project Data File 
    LS                   Links 98 Game File 
    LS                   Lightscape Solution File 
    LS1                  Winhelp Source file 
    LS1                  Cokriging Software VARIO Output File 
    LS1                  OzWin CompuServe E-mail/Forum Access Library Listing 
    LS2                  Cokriging Software COREG Output File 
    LSA                  LightScape Technologies ASCII Data 
    LSA                  Liquid Audio Player Skin File (Geneva Media, LLC)
    LSB                  LightScape Technologies Binary Data 
    LSC                  Tune Smithy List of Muscial Scales 
    LSC                  LOGO!Soft Comfort Circuit File (Siemens AG)
    LSD                  Fifa World Cup Game Data Environment File 
    LSD                  LECTURNITY Presentation File Source Format (IMC AG)
    LSD                  RPG Maker 2000/2003 Saved Game File (Enterbrain)
    LSD                  ABBYY Lingvo Non-changaeble System Dictionary (ABBYY Software House)
    LSE                  Audio Manager (Nokia)
    LSF                  Libronix DLS Resource 
    LSF                  Streaming Audio/Video File 
    LSF                  Totem Cash Profile File 
    LSG                  LocStudio Glossary (Microsoft)
    LSI                  Audio 
    LSI                  Corel Layout Specification Instance SGML 
    LSL                  Liquid Audio Player Resource DLL (Geneva Media, LLC)
    LSL                  Lotus Script Library (IBM)
    LSL                  Corel Paradox Saved Library 
    LSL                  Daylon Leveller Shape List 
    LSLIDE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LSN                  Works File (Microsoft)
    LSO                  BHV Tonstudio Platin File 
    LSO                  Logic Audio File 
    LSP                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LSP                  Living Scenes Jigsaw Puzzles 
    LSP                  LISP Program File 
    LSP                  AutoLISP Application Source Code (Autodesk, Inc.)
    LSQ                  IBM Voice Type Language Addword File 
    LSQ                  CalTech PaleoMag Least Squares Format 
    LSQ                  VAX CRYSTALS Chemical Crystallogaphy File 
    LSQ                  MRS-802 Phrase Loop Formula Data (Zoom Corporation)
    LSR                  Logos Library System 2.x Standard Reference Definition 
    LSS                  Legato Spreadsheet 
    LSS                  LotusScript Program File 
    LST                  OzWin CompuServe E-mail/Forum Access Library Listing 
    LST                  Listing or List 
    LST                  Ingres Dump List File (Computer Associates International, Inc.)
    LST                  Family Tree Maker Error File 
    LST                  SignIQ Field List (ImpulseLogic)
    LST                  Pro/ENGINEER Temporary File (PTC)
    LST                  Oracle Spool File 
    LST                  PowerPoint View File (Microsoft)
    LST                  Orchida Knitting System (Orchida Soft)
    LST                  SAS Procedure Output (SAS Institute Inc.)
    LST                  BPM Studio Play List (ALCATech)
    LST                  1st Reader Keyboard Macro 
    LST                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LST                  IMail Server List Server Mail (Ipswitch, Inc.)
    LSW                  Technics Keyboard Current Panel File 
    LSW                  Golf Swapfiles File 
    LSX                  Streaming Audio/Video Shortcut 
    LSZ                  WinFax (Symantec)
    LSZ                  Adressmn File 
    LSZ                  LiteStep Installer Theme File (beyondconvention)
    LS_                  Unix ls Directory File 
    LT                   PC LTAS Printer Configuration File 
    LT2                  Curious Labs Poser Lighting File 
    LTB                  Amma Lsb v1.0 
    LTD                  LivingTime Data File (Lifescope Inc.)
    LTE                  Adressmn 
    LTF                  Frogans Short-cut (STG Interactive SA)
    LTG                  Lasertank Bitmap Graphics 
    LTH                  Sierra Print Artist Letterhead 
    LTL                  Linear Temporal Logic Formula 
    LTM                  LiteMail Database/Project (Sprika Software)
    LTM                  Lotus Form (IBM)
    LTM                  EOSAT Landsat Thematic Mapper Bitmap Data 
    LTPL                 Insight II Layout Template File (Accelrys Software Inc.)
    LTR                  Letter 
    LTR                  ACT! Letter (Best Software CRM Division)
    LTT                  SDR99 Speech Recognition Task Lexical TREC Transcription 
    LTT                  HP Library and Tape Tools Log File 
    LTT                  MicroImages Text Layout (MicroImages, Inc.)
    LTX                  LaTeX Mathematical Text Document 
    LU                   ThoughtWing Library Unit File 
    LUA                  Lua Source (Tecgraf - Computer Graphics Technology Group)
    LUA                  Homeworld Generic Lua File (Sierra Entertainment, Inc.)
    LUCS                 Amaya Tools 
    LUDI_PSEUDO_PROTEIN  Insight II Exclusion Shell File (Accelrys Software Inc.)
    LUL                  OzWin CompuServe E-mail/Forum Access Upload 
    LUMPSUM              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LUN                  DB-MAIN Project File (REVER S.A.)
    LUNCH                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LUP                  OzWin CompuServe E-mail/Forum Access Library Update 
    LUT                  Osiris Prefs Colordir File 
    LUT                  LUT File 
    LUTE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LUX                  Spectramap Database (Coloritto BV)
    LUXB                 Lux Saved Map File (Sillysoft)
    LV1                  Boulder Rash Original Single Level 
    LVA                  Logitech Video Effects Avatar and Face Accessory (Logitech)
    LVB                  Outlaws Binary Mission Files (LucasArts Entertainment Company LLC)
    LVD                  LPA-View Data File (MP Filtri U.K. Ltd.)
    LVF                  Logitech Video Effects Avatar and Face Accessory (Logitech)
    LVL                  Boulder Rash Original Full Levels 
    LVL                  Parallax Miner Descent/D2 Level Extension Level Extension 
    LVL                  Max Payne Uncompiled Level File (Rockstar Games)
    LVL                  Lasertank Level 
    LVL                  Snood Game Level 
    LVL                  Continuum Zone Data (SubSpaceDownloads.com)
    LVP                  Avaya Voice Player Compressed Voice Audio (Avaya Inc.)
    LVP                  Lucent Voice Player 
    LVP                  LView Pro Image 
    LVS                  Daylon Leveller Selection Mask 
    LVT                  Outlaws Text Mission Files (LucasArts Entertainment Company LLC)
    LVU                  Livid Union Project File (Livid Instruments)
    LVW                  Daylon Leveller View File 
    LVZ                  Continuum Zone Data Graphics (SubSpaceDownloads.com)
    LW                   LightWave (ClearJump)
    LW                   Line Work TIFF Bitmap 
    LW                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LW3                  Lightwright File 
    LWA                  LightWorks Archive (LWA) Material/Scene File (LightWork Design Ltd.)
    LWC                  Living Word Commentaries Bible File 
    LWD                  LotusWorks Text Document (IBM)
    LWF                  LuraWave-compressed Graphic File 
    LWI                  LightWorks Image (LightWork Design Ltd.)
    LWK                  LonghornMax Data File (Cattlesoft, Inc.)
    LWM                  LittleWays Map (hfrmobile)
    LWO                  LightWave 3D 3D Object (NewTek)
    LWO                  Tstream Editor Studio Translation Package (Xplanation Language Services)
    LWP                  Wordpro 96/97 Document (IBM)
    LWP                  IBM Works for OS/2 Document 
    LWS                  LightWave 3D 3D Scene File (NewTek)
    LWTP                 LimeWire Theme File (Lime Wire LLC.)
    LWV                  Linguistically Enhanced Sound File (Microsoft)
    LX                   LPEX Customization Command Profile 
    LXA                  Office Speech Lexicon File (Microsoft)
    LXD                  eDictionary Reader Dictionary Data File (PPCLINK)
    LXD                  Lexacom Data File (Aprobrium Limited)
    LXF                  LEGO CAD Scene File (The LEGO Group)
    LXF                  Lenex Compressed XML Data (GeoLogix AG)
    LXFML                LEGO CAD Digital Designer XML Data (The LEGO Group)
    LXG                  LEGO CAD PC/Macintosh Exchange Format (The LEGO Group)
    LXL                  LPEX File Open Command Profile 
    LXM                  Lexacom Mobile PDA Dictation File (Aprobrium Limited)
    LXN                  SeeYou Flight Data (Naviter)
    LXO                  LaserMaster Font 
    LXS                  LPEX File Save Command Profile 
    LXW                  Lexacom Talk & Type Dictation File 
    LY                   GNU LilyPond 
    LY+                  DataCAD Plus Layer (DATACAD LLC)
    LYNX                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LYR                  Song Lyric File 
    LYR                  DataCAD Layer File (DATACAD LLC)
    LYR                  Lyrics for Audio CD 
    LYRIX                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LYS                  DataCAD Layer Control File (DATACAD LLC)
    LYT                  DESQview (Symantec)
    LYT                  MicroImages Text Layout (MicroImages, Inc.)
    LYT                  TurboTax Install Log File (Intuit)
    LYT                  SignMaking Application Extension 
    LYX                  LyX 
    LZA                  LZA Animation/Video 
    LZD                  Binary Difference File 
    LZF                  3ds Max Lens Effect Flare (Autodesk, Inc.)
    LZH                  Compressed Archive File 
    LZMA                 7-Zip Compressed File (Igor Pavlov)
    LZO                  lzop Compressed Archive 
    LZS                  LARC Compressed File Archive 
    LZS                  Skyroads Data File 
    LZS                  Compressed Archive File 
    LZSS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    LZW                  LHWarp Compressed Amiga File Archive 
    LZW                  Compressed Amiga Archive File 
    LZX                  Compressed File 
    LZZZZZZZ             Unknown 
    M                    Objective-C Source 
    M                    Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    M                    Winamp3 Uncompiled Script 
    M                    Miranda language Script File 
    M                    MATLAB Function or Script (The MathWorks, Inc.)
    M                    Mathematica Functions and Commands (Wolfram Research, Inc.)
    M                    Maple Common Binary File (Maplesoft, a Division of Waterloo Maple Inc.)
    M                    Limbo Interface Declaration 
    M                    Desktop Color Separation Specification Magenta Layer 
    M                    Cliq Accessories Datebook Modification File 
    M                    Brief Macro Source 
    M$S                  MindStor Firmware Update File 
    M0D                  Malete Options (Record 0) File (OpenIsis)
    M11                  Farallon's MacRecorder Format Sampling Rate 11k 
    M11                  MASS11 Text File 
    M12                  SHARP MZ-series Emulator File 
    M12                  Proprietary Data Format for Interchange of Defense Form Data with Mil-Pac Technology Software Products 
    M12                  Hydrographs Synthesis Project File 
    M12                  S-BASIC File 
    M14                  Steuer2001 File 
    M15                  Apple MPEG-1 Video (Apple Computer, Inc.)
    M15                  thinEdge Model (Petr Schreiber)
    M18                  Accolade Golf Course File 
    M1A                  MPEG-1 Audiostream 
    M1A                  Apple MPEG-1/2 Audio (Apple Computer, Inc.)
    M1C                  IBM Voice Type Language Newuser File 
    M1I                  IBM Voice Type Language Newuser File 
    M1PG                 iFinish Video Clip (Media 100 Inc.)
    M1S                  MPEG Media File 
    M1V                  MPEG-1 Video File 
    M2                   Modula 2 Language Source 
    M22                  Farallon's MacRecorder Format Sampling Rate 22k 
    M2A                  MPEG-2 Audio 
    M2A                  Apple MPEG-1/2 Audio (Apple Computer, Inc.)
    M2C                  IBM Voice Type Language Newuser File 
    M2I                  IBM Voice Type Language Newuser File 
    M2O                  Modula Fitted Modula2 File 
    M2P                  MPEG-2 Program Stream Format File 
    M2S                  MPEG-2 Audio and Video 
    M2TS                 Blu-ray BDAV MPEG-2 Transport Stream (Blu-ray Disc Association)
    M2V                  MPEG-2 Video Only File 
    M3                   Modula 3 Program File 
    M32                  Dr.T MIDI File 
    M3A                  MPEG Archive Enhanced .M3U Playlist File 
    M3C                  IBM Voice Type Language Newuser File 
    M3D                  3D Animations Macro 
    M3D                  Corel Motion 3D Animation 
    M3D                  EverQuest File 
    M3G                  Nokia Phone Screen Saver (Nokia)
    M3I                  IBM Voice Type Language Newuser File 
    M3S                  Myst III Exile Saved Game 
    M3U                  MP3 Playlist File 
    M3U8                 MP3 Playlist File (UTF-8) 
    M3URL                MP3 Playlist File 
    M4                   m4 Macro 
    M4                   Meta4 Source Code (Meta4)
    M4A                  MPEG-4 Audio Layer 
    M4B                  MPEG-4 Audio Layer Bookmarkable File 
    M4C                  IBM Voice Type Language Newuser File 
    M4I                  IBM Voice Type Language Newuser File 
    M4P                  Protected AAC File (Apple Computer, Inc.)
    M4U                  Ogg Media Redirector File 
    M4V                  MPEG-4 Video File 
    M68                  Turbo Pascal DOS File (Borland Software Corporation)
    M7                   Farallon's MacRecorder Format Sampling Rate 7k 
    M75                  Apple MPEG-1 Video (Apple Computer, Inc.)
    M8                   Grlib Scry PCserv File 
    M8                   Heretic II MipMap 
    M??                  Ingres Modify Table/Index File (Computer Associates International, Inc.)
    MA                   Homeworld Maya Scene File (Sierra Entertainment, Inc.)
    MA                   Diablo II Map File (Blizzard Entertainment)
    MA                   Mathematica ASCII File (Wolfram Research, Inc.)
    MA                   Maya ASCII Scene (Alias Systems, a division of Silicon Graphics Limited)
    MA                   hDC Products for MicroApp Executable Files 
    MA0                  Diablo II (Blizzard Entertainment)
    MA1                  SKY Mobile Phone Icon/Ringtone 
    MA1                  Diablo II (Blizzard Entertainment)
    MA2                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MA2                  Diablo II (Blizzard Entertainment)
    MA3                  Harvard Graphics Macro (Harvard Graphics)
    MA3                  Diablo II (Blizzard Entertainment)
    MAB                  Mozilla (Netscape) Personal Addressbook or Collected Addressbook (Mozilla.org)
    MAC                  Geometry II Plus Macro File (Cabrilog)
    MAC                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAC                  Dynamics GP Macro (Microsoft Corporation)
    MAC                  MINITAB Macro (Minitab, Inc.)
    MAC                  Macro 
    MAC                  Access Shortcut (Microsoft)
    MAC                  TASM Macro (Borland Software Corporation)
    MAC                  MacPaint Bitmap Graphic 
    MAC2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAC3                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAC4                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MACAPPL              MacDrive (Mediafour Corporation)
    MACHINE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MACHTYPE             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MACP                 MacPaint Bitmap Graphic 
    MACR                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MACRO                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MACRO2               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MACRO3               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MACROS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MACS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MACWRITE             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAD                  Mlat AdLib Tracker 
    MAD                  Access Module Shortcut (Microsoft)
    MAD                  Game Movie File (Electronic Arts, Inc.)
    MAD                  Homeworld Model Animation Data (Sierra Entertainment, Inc.)
    MADNESS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MADONNA              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MADSTATE             Homeworld Model Animation Script (Sierra Entertainment, Inc.)
    MAE                  Maestro File 
    MAF                  Access (Microsoft)
    MAFF                 Firefox Mozilla Archive Format Extension Saved Web Page Archive (Mozilla)
    MAG                  Magenta Color Separation (Adobe)
    MAG                  Access Diagram Shortcut (Microsoft)
    MAG                  MagIchan Graphic 
    MAG                  Magic Mail Monitor Mailbox File 
    MAG                  Fortis and File Magic Document File (Westbrook Technologies, Inc.)
    MAG                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAG                  Magic 1 2 3 Acounting File (Universal Systems)
    MAG                  MPS Magro Paint System Graphic 
    MAGIC                Magic Mail Monitor Mailbox 
    MAGNETICFIELDSIV     Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAGS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAH                  Mahjongg Solitaire Settings 
    MAHJONGTITANSSAVE-MS Mahjong Titans Saved Game (Microsoft Corporation)
    MAI                  Pegasus Mail Message Directory (David Harris)
    MAI                  MS Mail Message 
    MAI                  Sony Video Thumbnail (Sony Corporation of America)
    MAI                  VAX Mail 
    MAIL                 NeoPlanet Mail Client 
    MAIL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAILABVIEW           MSN AddressBook View (Microsoft Corporation)
    MAILBOX              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAILHOST             E-mail Server Preferences File (Microsoft)
    MAILORDER            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAILSRV              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAILVIEW             MSN Mail Message List (Microsoft Corporation)
    MAIN                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAJ                  Profile Master 2000 Job Data (MAP Ltd.)
    MAK                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAK                  Make File 
    MAK                  Visual Basic Ver. 3.0 Project (Microsoft)
    MAK                  Visual C++ Project (Microsoft)
    MAKE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAKE                 Makefile 
    MAKE2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAKEFILE             Makefile 
    MAKER                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAKETAIL             Sourcecontrol File 
    MAKI                 Winamp3 Compiled Script 
    MAL                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAL                  Connect Server Configuration (AvantGo, Inc.)
    MAL                  Webalizer.01 Language File 
    MALE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAM                  Access Macro Shortcut (Microsoft)
    MAN                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAN                  Windows 2000 Mandatory User Profile 
    MAN                  Manual 
    MANAGER              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MANGER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MANIFEST             Windows Visual Stylesheet XML File (Microsoft Corporation)
    MANNERLY             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MANTECA              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MANUAL               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MANUALS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MANX                 Manual File 
    MAP                  3Com Network Supervisor Network Topolgy Map (3Com)
    MAP                  MapInfo Map (MapInfo Corporation)
    MAP                  Offline Explorer (MetaProducts Corporation)
    MAP                  General Surfing Map 
    MAP                  Carnivores / Ice Age Map File (Action Forms, Ltd.)
    MAP                  SpellForce Level Map (JoWooD Productions Software AG)
    MAP                  Pro/ENGINEER (PTC)
    MAP                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAP                  Command & Conquer Red Alert 2 Mission File (Westwood Studios)
    MAP                  Diablo II Map (Blizzard Entertainment)
    MAP                  OziExplorer Map (Des & Lorraine Newman)
    MAP                  Hex Workshop Hex Editor Character Map (BreakPoint Software, Inc.)
    MAP                  Atlas MapMaker Map 
    MAP                  Geography Maps 
    MAP                  Ingres Termcap Mapping File (Computer Associates International, Inc.)
    MAP                  Visual Studio .NET 2002 Linker Map (Microsoft)
    MAP                  Manifold Map 
    MAP                  MAPIMail 
    MAP                  Micrografx Picture Publisher Format Data 
    MAP                  Poser 3D Map File 
    MAP                  PROGNOSIS Map Document File 
    MAP                  Server-side Image Map 
    MAP                  WorldCraft Quake/Half-Life Game Level File 
    MAP                  Image 
    MAP                  Fenix Map 
    MAP                  AccView Network Map 
    MAP                  Blood Game File Map 
    MAP                  Route 66 Card Document 
    MAP                  Color Palette 
    MAP                  ColorMAP Intensities and Indices Color Palette 
    MAP                  Debug Info Used by Some Programming Languages 
    MAP                  DIV Game Studio Map 
    MAP                  Duke Nukem Game Level Data File 
    MAP2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAPF                 MainActor Project (MainConcept AG)
    MAPIMAIL             Mail SentTo Drop (Microsoft)
    MAPIMAIL             Outlook Express Mail File (Microsoft)
    MAPISEND             MAPISEND File (Microsoft)
    MAPLET               Maplet Design File (Blitz Research Ltd.)
    MAPS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAPSAO               Aleph One Maps (Marathon Open Source)
    MAQ                  Access Query Shortcut (Microsoft)
    MAR                  Bibliographic Data Format 
    MAR                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAR                  Access Report Shortcut (Microsoft)
    MAR                  Mozilla Archive (Mozilla)
    MAR                  MSN Archive (Microsoft Corporation)
    MAR                  VAX Macro Assembly Program 
    MAR                  Pegasus Mail Attributes Records Per User (David Harris)
    MARCH                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MARIMBA              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MARIO                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MARK                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MARKE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MARKER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MARM                 EPOC Makefile for MARM Platform 
    MAS                  Focus Master Data Definition 
    MAS                  Make A Storybook Storybook (Kennard Consulting)
    MAS                  Freelance Graphics Smartmaster File (IBM)
    MAS                  QuickCard File 
    MAS                  Access Stored Procedures (Microsoft)
    MASCOT               Otaku Mascot Mascot (Accursed Toys Inc.)
    MASK                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MASKA                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MASON                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MASS                 Mass Downloader Partial Download (MetaProducts Corporation)
    MASSAGESOAPBOOK      WinCity Massage SOAP Notes Soap Book (WinCity Custom Software, Inc.)
    MASTER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MASTERANDSERVANT     Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAT                  Sound File 
    MAT                  Pro/ENGINEER Material Properties File (PTC)
    MAT                  Poser 3D Pose File 
    MAT                  Ox Object-Oriented Matrix Programming Language Matrix File 
    MAT                  MATLAB Variables Binary File (The MathWorks, Inc.)
    MAT                  CRiSP Harvest File 
    MAT                  ArcView Geocoding Matching Parameters File (ESRI)
    MAT                  Access Table Shortcut (Microsoft)
    MAT                  Freelancer Model Textures File (Microsoft)
    MAT                  3D Studio (Autodesk, Inc.)
    MAT                  Animation:Master Material (Hash)
    MATH                 AnalyticMath (Paul Hunter)
    MATH                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MATH                 Amaya 
    MATHCALC             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MATILDA              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAUC                 Multi-Channel Server (MCS) Component Policy Editor Audio Component (Volantis Systems Ltd.)
    MAUD                 Sound File 
    MAV                  Access View Shortcut (Microsoft)
    MAW                  Access Data Access Page (Microsoft)
    MAX                  PaperPort Scanned Image (ScanSoft Inc.)
    MAX                  Maximizer Contact & E-business Program 
    MAX                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAX                  OmniPage Image (ScanSoft, Inc.)
    MAX                  Max Source Code 
    MAX                  Kinetix's 3D Studio MAX Scene 
    MAX                  3D Studio Scene (Autodesk, Inc.)
    MAX                  PS2 Powersave File 
    MAX                  OrCAD Layout File (Cadence Design Systems, Inc.)
    MAYBE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MAZ                  Hover Maze Data 
    MAZ                  3ds Max Archived Network Rendering Job (Autodesk, Inc.)
    MAZ                  Division's dVS/dVISE File 
    MAZE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MB                   Mathematica Binary File (Wolfram Research, Inc.)
    MB                   MapBasic Program (MapInfo Corporation)
    MB                   Paradox Memo Holder 
    MB                   Maya Binary Scene (Alias Systems, a division of Silicon Graphics Limited)
    MB1                  Apogee Monster Bash Data 
    MBA                  Martus Human Rights Bulletin System (Benetech)
    MBB                  MannaBook (MannaBook Software, Inc.)
    MBD                  Multimedia Builder MP3 Authoring File 
    MBD                  Revolution EDX Catalog (4pi Analysis, Inc.)
    MBF                  Corel Visual DTD Document Type Definition for SGML (Corel Corporation)
    MBF                  Money Backup File (Microsoft Corporation)
    MBFAVS               Stardent AVS X Image 
    MBFS                 Stardent AVS X Image 
    MBG                  MS Mail Mailbag 
    MBG                  Mailbag Assistant E-mail Archive 
    MBI                  MBasic Source Code 
    MBK                  Atari 
    MBK                  dBASE IV Multiple Index Backup 
    MBK                  VersaPro Motion Block Backup 
    MBL                  Logger Pro Experiment Data File (Vernier Software & Technology)
    MBM                  SymbianOS Multi BitMap Image 
    MBOX                 Unix Mbox Format Mail File 
    MBP                  Mediabase Package (Digilink B.V.)
    MBP                  Medbasin Project File (National Technical University of Athens, Laboratory of Reclamation Works & Water Resources Management)
    MBP                  Balder Multiboot Project (HCData)
    MBR                  AS/400 Physical File Member (IBM)
    MBS                  Mailbag Assistant Script 
    MBS                  Micrografx Picture Publisher Brush Shape 
    MBT                  HASLEMERE / Other MBT Applications MBT Data File (Modern Business Technology Limited)
    MBT                  ecBuilder Template 
    MBTN                 Multi-Channel Server (MCS) Component Policy Editor Button Image Component (Volantis Systems Ltd.)
    MBU                  Neverwinter Nights Music File 
    MBV                  Corel.visualDTD.7 
    MBW                  MarketBrowser 
    MBX                  Mailbox Message File 
    MBX                  Database Index; dBASE Index 
    MBX                  MapBasic Compiled Program (MapInfo Corporation)
    MC                   PlaySaver and PSEmuPro Memory Card Save and Single Game Save 
    MC                   Bricolage Page Template (Bricolage Development Team)
    MC                   MetaCard Script 
    MC                   Modula-3 M3CG intermediate Language File 
    MC                   NRC Inspection Manual Chapter 
    MC                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MC                   Windows Localized Message Table Resource 
    MC$                  Magic Xchange Package 
    MC1                  MedCalc Data File 
    MC2                  Mastercook File (Sierra)
    MC3                  Bridge Design Modeler 3D Match-Casting Geometry Control 
    MC3E                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MC4D                 Cinema 4D v4.x Object (MAXON Computer GmbH)
    MC5                  Grafic Jpg Bcc file/microsoft C v5 related file 
    MC6                  C File/Makefile (Microsoft)
    MC7                  Mastercam 7 File (CNC Software, Inc.)
    MC8                  Mastercam 8 File (CNC Software, Inc.)
    MC9                  Mastercam 9 File (CNC Software, Inc.)
    MCA                  HFS Medicare Auditor Medicare Cost Audit (Health Financial Systems)
    MCA                  i-deas Web ACCESS File 
    MCA                  ModCA IOCA 
    MCB                  Grlib Ravekit File 
    MCB                  Merak Email Server Configuration Backup (IceWarp Ltd.)
    MCC                  Amiga MUI External Class/Library (SASG)
    MCC                  VectorWorks/MiniCAD File 
    MCC                  Dialer10 Calling Card 
    MCC                  MathCAD Configuration (Mathsoft Engineering & Education Inc.)
    MCC                  The Microsoft Network Shortcut (Microsoft)
    MCC                  Incite Media Assistant File 
    MCC                  Monu-Cad Pro CAD File 
    MCC                  Mylar City Character Editor 
    MCD                  MediaCD Configuration File (Bot Productions)
    MCD                  Monu-Cad Pro CAD File 
    MCD                  MathCaD Document (Mathsoft Engineering & Education Inc.)
    MCD                  MinCAD 7 Drawing/Texture (Nemetschek North America Inc.)
    MCD                  Craig McDowall.com NotePad 
    MCD                  Bleem! Memory Card Save 
    MCD                  VectorWorks Drawing (Nemetschek)
    MCF                  Public Browser Platform Configuration File (Myriad Communications Ltd)
    MCF                  Master Command File 
    MCF                  Sierra Mastercook Cookbook File 
    MCF                  TMPGEnc Conversion Template (Pegays Inc.)
    MCF                  Hotsauce Graphics 
    MCF                  SignIQ Media Collection File (ImpulseLogic)
    MCF                  Media Container Format 
    MCF                  Meta Content Format File 
    MCF                  MAX II Configuration File (Max B.V.)
    MCF                  MathCAD Font (Mathsoft Engineering & Education Inc.)
    MCH                  Xtra Cache File (Macromedia)
    MCHT                 Multi-Channel Server (MCS) Component Policy Editor Chart Component (Volantis Systems Ltd.)
    MCI                  Media Control Interface Command Set 
    MCIF                 MacroMolecular Crystallographic Information File 
    MCL                  Multimedia Manager Collection File 
    MCL                  VectorWorks Template (Nemetschek)
    MCL                  Macro Command Language 
    MCL                  MUSHclient World Information File 
    MCL                  MultiEdit Macro Library 
    MCL                  MCell Cellular Automaton Ffile (Mirek Wojtowicz (Miroslaw Wójtowicz))
    MCM                  MacMolecule File 
    MCM                  Enable Macro 
    MCO                  MSN Messenger Wink File (MSN)
    MCP                  MUSICompress Encoded Audio (Soundspace Audio LLC)
    MCP                  WaveZIP Lossless Compressed Audio 
    MCP                  Master Compiler Profile 
    MCP                  Amiga MUI Preference Class/Library (SASG)
    MCP                  Capsule Application Script 
    MCP                  MSN Content Plus Installer File Archive (MSN Content Plus Inc.)
    MCP                  MathCAD Printer Driver (Mathsoft Engineering & Education Inc.)
    MCP                  Metrowerks CodeWarrior Project 
    MCP                  NetMC Player Composed Media Multimedia File 
    MCQ                  McAfee Quarantined File 
    MCR                  DataCAD Keyboard Macro File (DATACAD LLC)
    MCR                  Generic CADD Macro 
    MCR                  HFS Medicare Auditor Medicare Cost Report (Health Financial Systems)
    MCR                  CuteFTP Script (GlobalSCAPE)
    MCR                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MCR                  BricsCad Macro (BricsCad nv)
    MCR                  PlaySaver and PSEmuPro Memory Card Save and Single Game Save 
    MCR                  Freemagic Source Imageanalysis File 
    MCR                  Macro Magic Macro (iolo technologies, LLC.)
    MCR                  Perfect Keyboard Macro (Pitrinec Software)
    MCR3                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MCR3E                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MCRE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MCS                  VirusScan TC Installation File (Network Associates)
    MCS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MCS                  MathCAD Format File (Mathsoft Engineering & Education Inc.)
    MCS                  Mimics Project File (Materialise NV)
    MCS                  HalfLife MapCycle - Editor 
    MCT                  MSN Messenger Packaged Wink File (MSN)
    MCT                  MapChart Genetic Map Data (Plant Research International)
    MCT                  Musicator 
    MCW                  MacWrite II Macintosh Text File 
    MCW                  Word for Macintosh Document (Microsoft)
    MCX                  Mastercook File (Sierra)
    MCX                  MacroX Macro File (Millsoft Software Development)
    MCX                  Intel Connection Fax File 
    MCX                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MC_                  Musicato Compressed File 
    MD                   Mayura Draw Vector Drawing File (Mayura Software)
    MD                   Chaos Desktop Metadata File (Chaos systems AB)
    MD                   Teknical eLearning Objects Meta-Data File (Teknical Ltd.)
    MD                   Technics Keyboard User MIDI Settings File 
    MD                   MDCD Compressed Archive File 
    MD1                  Gear CD Suite File 
    MD2                  Quake 2 Player Model 
    MD3                  Quake III Arena Model File (ID Software)
    MD5                  Checksum File for .SHN Audio 
    MD5                  MD5summer Checksum 
    MD5                  IsoBuster MD5 Checksum (Smart Projects)
    MD5                  fsum MD5 Checksum File (SlavaSoft Inc.)
    MD8                  Mediator Project (MatchWare)
    MD8                  CDrom Database File 
    MDA                  MicroDesign Area File (Creative Technology)
    MDA                  Access Add-in (Microsoft)
    MDA                  Access 2 Workgroup (Microsoft)
    MDA                  Meridian Data Slingshot 
    MDB                  Access Database (Microsoft)
    MDB                  Mediator Multimedia Data 
    MDB                  PTC Pro/MECHANICA File 
    MDB                  MS File Viewer 
    MDB                  ProWORX Nxt Memory Data (Schneider Electric)
    MDB                  Access Application (Microsoft)
    MDC                  PowerPlay Power Cube (Cognos Incorporated)
    MDC                  MedDC Mail Pouch (Asier Technology Corporation)
    MDE                  Access MDE Database File (Microsoft)
    MDE                  Office File (Microsoft)
    MDE                  Tonline Database File 
    MDF                  Star Trek Armada Map File Title & Image Information 
    MDF                  ProWORX Nxt Memory Data (Schneider Electric)
    MDF                  SQL Server Master Database File (Microsoft Corporation)
    MDF                  Alcohol 120% CD Image File (Alcohol Software)
    MDF                  Insight II Molecular Data File (Accelrys Software Inc.)
    MDF                  Vector CANape / CANgraph / CANdito Measurement Data (Vector Informatik GmbH)
    MDF                  Menu Definition File 
    MDF                  Capture Classic Filler (Accelio)
    MDF                  I-deas Master Drafting Machine Data File 
    MDF                  Fldigi Macro (Dave Freese, W1HKJ)
    MDHTML               Access HTML File (Microsoft)
    MDI                  MIDI-sequention Sound 
    MDI                  Borland Multiple Document Interface (Borland Software Corporation)
    MDI                  Office Document Imaging File (Microsoft Corporation)
    MDK                  Schedule File (Ad Astra)
    MDL                  Symbian OS MIME Type Library 
    MDL                  Simulink Simulation Model 
    MDL                  Half-life Model 
    MDL                  Moray Wireframe Model (SoftTronics)
    MDL                  Quake Model File 
    MDL                  MapShots Software 
    MDL                  CA-Compete! Spreadsheet 
    MDL                  ASTi Model Builder 
    MDL                  Animation:Master 3D Model (Hash)
    MDL                  Flight Simulator 3D Model File (Microsoft)
    MDL                  Mighty Draw DOS Library (theliquidateher Software, An MCS Investments, Inc. Company)
    MDL                  Digitrakker Music Module 
    MDL                  3D GameStudio Model Entity (Conitec Datasystems, Inc)
    MDL                  Steam Source SDK Model Content File (Valve Corporation)
    MDL                  Rational Rose Object Design Model (IBM)
    MDL                  RapidForm Proprietary Format (INUS Technology, Inc.)
    MDL                  3D Design Plus Model 
    MDM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MDM                  Telix Modem Definition 
    MDM                  Mario Dash Map File 
    MDMP                 WinXP Trouble Report (Microsoft Corporation)
    MDN                  Access Blank Database Template (Microsoft)
    MDO                  Alcohol 120% Split Image File (Alcohol Software)
    MDO                  Internet Information Server Configuration Backup (Microsoft)
    MDP                  MicroDesign Page File (Creative Technology)
    MDP                  Developer Studio Project (Microsoft)
    MDP                  Visual Studio .NET Project Converter (VC 4.x) (Microsoft)
    MDP                  Visual C++ MAK File and Visual J++ Project Workspace (Microsoft)
    MDPR                 Multi-Channel Server (MCS) Device Policy Editor Device Profile Repository (Volantis Systems Ltd.)
    MDR                  ZX Spectrum-Emulator Microdrive File 
    MDR                  MAX II Recovery File (Max B.V.)
    MDR                  FaxTalk Modem Doctor Modem Report File 
    MDS                  SeeYou Raster Map (Naviter)
    MDS                  Media Descriptor CD Image File 
    MDS                  Directx Mid2stream File 
    MDT                  Ilink Incremental Linker Data Table (Microsoft)
    MDT                  Access Add-in Data (Microsoft)
    MDT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MDV                  QLAY MDV Image 
    MDV                  M1 Pattern File (H. Stoll GmbH & Co.)
    MDW                  Access Workgroup Information (Microsoft Corporation)
    MDX                  Rational XDE Model (IBM)
    MDX                  dBASE IV Multiple Index 
    MDX                  MidiEx MIDI System Exclusive Dump 
    MDX                  Borland Database Engine Index (Borland Software Corporation)
    MDX                  Label Pro Re-Index (Avery)
    MDYV                 Multi-Channel Server (MCS) Component Policy Editor Dynamic Visual Component (Volantis Systems Ltd.)
    MDZ                  Protracker/Module File 
    MDZ                  Access Wizard Template (Microsoft)
    MDZ                  Cubic Player/Cross-View Music Module Description 
    MD_                  Visc15 Adressen Setup File 
    ME                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ME                   Information 
    ME                   MultiEdit Configuration 
    ME                   Formatted Manual Page with ME Macros 
    ME!                  Text (Generally a README File) 
    ME1                  Zortech 21 Sample Winexam File 
    ME2                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ME3                  Multimedia Email Message (ArcSoft, Inc.)
    MEA                  Meazure File 
    MEA                  Mastersizer 2000 Sample Analysis/Measurement (Malvern Instruments Ltd)
    MEANT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MEB                  Open eBook (International Digital Publishing Forum)
    MEB                  WordPerfect Macro Editor Bottom Overflow File (Corel Corporation)
    MEC                  Maximus MECCA Source 
    MED                  WordPerfect Macro Editor Delete Save (Corel Corporation)
    MED                  Medline Document 
    MED                  A2B Player Purchase (DigitalPhono.com)
    MED                  MOBILedit! Phone Information (COMPELSON Laboratories)
    MED                  Amiga MED/OctaMED Tracker Module Sound File 
    MED                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MEDIACONTROL         Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MEDIACTRL            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MEDIAONEPROJECT      MediaOne Project (Corel Corporation)
    MEDICAL              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MEDIU                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MEDLINE              Medline Document 
    MEEP                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MEF                  Jaws PDF Server Exported Settings (Global Graphics Software Ltd.)
    MEFP                 ExerciseRT Sample Data Record (Morgan Medical Ltd.)
    MEFS                 ExerciseRT Secondary Data Record (Morgan Medical Ltd.)
    MEG                  MEG 101 B Database File (MEGBase)
    MEGABLASTER          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MEGAMISQUE           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MEI                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MEK                  Gearhead Garage Game File (Mekada)
    MEL                  Maya Embedded Language Script (Alias Systems, a division of Silicon Graphics Limited)
    MEL                  Ericsson Phone Melody 
    MELL                 Mellel Document (RedleX)
    MEM                  Medlin Accounting Additional Vendor Data (Medlin Accounting)
    MEM                  Context4 File 
    MEM                  Memory File of Variables 
    MEM                  FoxPro Memory Variable Save File (Microsoft)
    MEM                  WordPerfect Macro Editor Macro (Corel Corporation)
    MEM                  Clarion DOS Database Memo File (SoftVelocity)
    MEM                  Fanuc Parameter File (FANUC Robotics America, Inc.)
    MEM                  BPM Studio Saved Files for Sampler Functions (ALCATech)
    MEM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MEM                  Mio Technology Mio Object (Mioplanet Technologies, Inc.)
    MEMB                 Pro/ENGINEER Temporary File (PTC)
    MEMBERS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MEMO                 Unknown Found in Multiple GNU/Linux Packages 
    MEMORY               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MEN                  Menu 
    MEN                  Geometry II Plus Utility (Cabrilog)
    MENTAL               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MENU                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MENU1                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MENU2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MENUS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MEP                  Inor MePro Software File 
    MEQ                  WordPerfect Macro Editor Print Queue (Corel Corporation)
    MER                  Data Interchange Format 
    MER                  WordPerfect Macro Editor Resident Area (Corel Corporation)
    MERLIN               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MERMACK              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MERR                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MERRITT              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MES                  Message File 
    MES                  WordPerfect Macro Editor Work Space (Corel Corporation)
    MES                  SpectraMagic Data File (Konica Minolta Holdings, Inc.)
    MES                  A2B Player File (DigitalPhono.com)
    MESA                 Opengl Glut-3.6 File 
    MESSAGE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MESSAGES             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MESSENGER            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MET                  WordPerfect Macro Editor Top Overflow File (Corel Corporation)
    MET                  eDonkey Cached Information (MetaMachine)
    MET                  eMule Cached Information (eMule Team)
    MET                  ActiveSky Export File (HiFi Simulation Software)
    MET                  OS/2 PM Meta File 
    MET                  Presentation Manager Meta File 
    MET                  OmniPage Document (ScanSoft, Inc.)
    METAL                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    METALFONT            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    METALINK             Metalink File (Anthony Bryan)
    METHODS              Likely a Text File 
    MEU                  DOS Shell Menus 
    MEX                  WordPerfect Macro Editor Expound File (Corel Corporation)
    MEX                  MATLAB Macintosh Dynamically Linked Subroutines (The MathWorks, Inc.)
    MEX                  MATLAB Executable Command (The MathWorks, Inc.)
    MEX                  Macro Express Macro (Insight Software Solutions)
    MEX                  mexReader Multimedia File 
    MEX4                 MATLAB Sun-4/SPARC Dynamically Linked Subroutines (The MathWorks, Inc.)
    MEXDS                MATLAB DECStation Dynamically Linked Subroutines (The MathWorks, Inc.)
    MEXHP3               MATLAB HP9000/Series 300 Dynamically Linked Subroutines (The MathWorks, Inc.)
    MEXHP7               MATLAB HP9000/Series 700 Dynamically Linked Subroutines (The MathWorks, Inc.)
    MEXN                 MATLAB NeXT Dynamically Linked Subroutines (The MathWorks, Inc.)
    MEXRS6               MATLAB IBM RS/6000 Dynamically Linked Subroutines (The MathWorks, Inc.)
    MEXSG                MATLAB SGI Dynamically Linked Subroutines (The MathWorks, Inc.)
    MEZ                  RealArcade Game Information 
    MEZ                  MusicEase Music Notation File (MusicEase Software)
    MF                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MF                   Manifest Graphic 
    MF                   METAFONT Font Description File 
    MF                   MoonFish Track File (Bram Bos)
    MF                   Java Manifest (Sun Microsystems, Inc.)
    MF                   FrameMaker MakerFile (Adobe)
    MF2                  Authorware Mediator5 Multimedia File (Adobe Systems Incorporated)
    MFA                  Site Server 3.0 Report Definition (Microsoft)
    MFASL                Mozilla (Netscape) XUL FastLoad File (Mozilla.org)
    MFC                  Mankind File 
    MFD                  Multiframe Structural Frame Model (Formation Design Systems Pty Ltd)
    MFD                  ABC Programming Language Monadic Function 
    MFD                  F1RC Relay Game Code File 
    MFD                  Adobe Multiple Master Font Metrics Directory File (Adobe)
    MFF                  MIDI File Format 
    MFG                  MediaForge Runtime Player Distribution Project (XMLAuthor, Inc.)
    MFI                  Win98rk File 
    MFJ                  MetaFont Job Batch Input 
    MFK                  Mafuka Audio 
    MFL                  Mozilla (Netscape) XUL FastLoad File (Mozilla)
    MFM                  DMP Music Format 
    MFM                  mTropolis mPire Interactive Multimedia Title (Quark)
    MFM                  VTeX Multiple Master Font Metrics 
    MFMOD                Mobile Forces Game MOD (Rage Software)
    MFT                  MetaFont Source Formatter Style File 
    MFW                  mTropolis mPire Interactive Multimedia Title (Quark)
    MFX                  mTropolis mPire Interactive Multimedia Title (Quark)
    MG                   Modula-3 Generic Module Source 
    MG1                  MusiGenesis Project File (Kenneth Adams)
    MGA                  Migrys IMR Call Flow (Migrys Innovations)
    MGC                  Equation Editor File (MGCSoft)
    MGC                  Clipart Collection Catalog (Microsoft)
    MGCF                 MagicInstall Compressed File 
    MGD                  Master Grade Gradebook 
    MGF                  Materials and Geometry Format File 
    MGF                  Micrografx Font 
    MGF                  DataModeler XML Schema Project Data (URL Integration, Inc.)
    MGF                  Image File 
    MGK                  ImageMagick Ver. 5.3.2 Coders File (ImageMagick Studio LLC)
    MGL                  Age of Kings (Microsoft)
    MGL                  MosASCII Graphics Library 
    MGM                  MGCSoft Equation Editor Macro File 
    MGMAP                Ygnius MindGenius Mind Map (Ygnius)
    MGR                  MGR Bitmap 
    MGR                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MGRP                 Multi-Channel Server (MCS) Component Policy Editor Asset Group (Volantis Systems Ltd.)
    MGS                  Intelligence Tracking System File 
    MGTPL                Ygnius MindGenius Template (Ygnius)
    MGU                  Band-in-a-Box MIDI File 
    MGV                  Yamaha Mega Voices (Yamaha Corporation of America)
    MGX                  Micrografx Picture Publisher Clipart 
    MGX                  Rise of Nations Saved Game (Microsoft)
    MGZ                  Metal Gear Solid Archive 
    MH                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MH                   Graphic File 
    MHF                  Maestro ERP Hot Fix (Maestro Technologies Inc.)
    MHG                  Multimedia File 
    MHK                  Riven Movie Data File 
    MHK                  Broderbund Mohawk Archive 
    MHP                  Home Publishing Project (Microsoft)
    MHT                  MHTML Document (Microsoft Corporation)
    MHTENX               egis Encrypted HTM File (HiTRUST Inc.)
    MHTM                 MHTML Document (Microsoft)
    MHTML                MHTML Document (Microsoft)
    MHTMLENX             egis Encrypted MHTML File (HiTRUST Inc.)
    MI                   Winamp3 MAKI Scripting Library 
    MI                   Miscellaneous 
    MI                   CoCreate CAD File 
    MI4                  HDD 100/120 Audio Jukebox Firmware File (Koninklijke Philips Electronics N.V.)
    MIA                  Babya bMix/Logic Music Project File (Babya Software Group)
    MIA                  MusicIndiaOnline Trident Player Music File (MusicIndiaOnLine)
    MIB                  Management Information Base File 
    MIB                  SNMP MIB File 
    MIB                  MyInfo Topic File Backup (Milenix Software Ltd.)
    MIBFONT              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MIC                  Image Composer File (Microsoft)
    MICE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MID                  Musical Instrument Digital Interface MIDI-sequention Sound 
    MID                  MapInfo Interchange Format 
    MIDI                 Musical Instrument Digital Interface MIDI-sequention Sound 
    MIDIFILE             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MIDISYNTH            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MIF                  Software License Tracking Tool 
    MIF                  MapInfo Interchange Format File (MapInfo Corporation)
    MIF                  EarthLink E-mail Message Index (EarthLink, Inc.)
    MIF                  Molecular Information Format 
    MIF                  MIDI Instrument 
    MIF                  Systems Management Server (SMS) Management Information Format (Microsoft Corporation)
    MIF                  MicroHof .MIF Assembler File 
    MIF                  Watcom Make Script 
    MIF                  FrameMaker Interchange Format (Adobe)
    MIF                  Management Information Format (Distributed Management Task Force, Inc.)
    MIF                  ImageMagick Machine Independent File Format Bitmap (ImageMagick Studio LLC)
    MIFF                 ImageMagick Machine Independent File Format Bitmap (ImageMagick Studio LLC)
    MIG                  Mighty Draw Drawing (theliquidateher Software, An MCS Investments, Inc. Company)
    MII                  PageWiz Image (Microtek)
    MII                  MicroStat II Data File 
    MII                  Wii Avatar File (Nintendo)
    MIL                  Micro Illustrator Uncompressed Graphic 
    MIL                  Group 4 Fax File 
    MILK                 MilkDrop Winamp Visualization Plug-in Preset File (Nullsoft, Inc.)
    MIM                  Multi-Purpose Internet Mail Extensions (MIME) File 
    MIME                 Multi-Purpose Internet Mail Extensions (MIME) File 
    MIMG                 Multi-Channel Server (MCS) Component Policy Editor Image Component (Volantis Systems Ltd.)
    MIN                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MINDTEASER1          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MINESWEEPERSAVE-MS   Minesweeper Saved Game (Microsoft Corporation)
    MINGW                Mingw Make File 
    MINGW32              Mingw Compiler File 
    MINI                 miniControl Data File (LucaNet)
    MINIPSF              Playstation Sound Format (Miniature Playstation Sound Format) (Sony Corporation of America)
    MINIPSF2             Playstation 2 Sound Format (Miniature Playstation Sound Format) (Sony)
    MIO                  Iphoto4 Programs Fio File 
    MIO                  Mio Technology Generic Communications Format (Mioplanet Technologies, Inc.)
    MIO                  MIO Compressed Audio 
    MIO                  MyInfo Topic File (Milenix Software)
    MIO                  mexReader Multimedia File 
    MIP                  MODDE Project (Umetrics AB)
    MIP                  Paint Shop Pro Multiple Image Print File (Corel Corporation)
    MIPSEL               Mips File 
    MIR                  Bauhaus Mirage Project File (Bauhaus Software, Inc.)
    MIS                  Delta Force Land Warrior Mission (NovaLogic, Inc.)
    MIS                  MagicInstall Installation Script 
    MIS                  Game Mission File 
    MIS                  Tom Clancey's Ghost Recon Mission File (Red Storm Entertainment)
    MIS                  3WayPack Missing Data Pointers (The Three-Mode Company)
    MIS                  Tribes 2 Game File (Sierra)
    MISC                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MISCTOOL             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MISS                 Homeworld Missle Data (Sierra Entertainment, Inc.)
    MISTRALLOGO          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MISTY                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MITSU                Mitsubishi S340-10 Color Sublimation Printer Bitmap (Mitsubishi Companies)
    MIX                  Techno eJay Music Mix File 
    MIX                  Picture It! Multilayer Picture File (Microsoft)
    MIX                  PhotoDraw 2000 Multilayer Picture File (Microsoft)
    MIX                  Project Dogwaffle Type 2 Mixing Palette (TheBest3D.com)
    MIX                  Windows Sound MIX (Microsoft)
    MIX                  OzWin CompuServe E-mail/Forum Access Index File 
    MIX                  Don Knuth's MIX Emulator 
    MIX                  Command & Conquer Picture/Sound File 
    MIX                  Power C Object File 
    MIZ                  Winamp Compressed MIDI File 
    MJ                   MovieJack DVD Ripper File 
    MJB                  PDA Mystery Juke Box Generation Profile (Paul Duggan - PDAWare)
    MJF                  Mjuice Secure Music Files 
    MJP                  Media Center Plug-in (J. River, Inc.)
    MJP                  Media Jukebox 
    MJPG                 SMJPEG Video 
    MJR                  Media Jukebox Registration File (J. River)
    MK                   Make File 
    MK                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MKA                  Matroska Audio Stream (Matroska Association)
    MKD                  Malz++Kassner CAD5 Drawing 
    MKE                  Make File 
    MKG                  Make File 
    MKH                  WinMark Pro Mark File (Synrad, Inc.)
    MKI                  MagView Japanese Makichan Format Graphic 
    MKI                  MediaKnife Animated Internet Picture 
    MKI                  Picture File 
    MKJ                  MKJamz 
    MKL                  Créatoria 
    MKL                  Malz++Kassner CAD5 Drawing Library 
    MKM                  Matroska Media File (Matroska Association)
    MKR                  McCullough Knowledge Explorer Knowledge Representation (Richard H. McCullough, Knowledge Engineer)
    MKR                  Tonline Bsw4 Internet File 
    MKR                  Train Simulator Marker File (Microsoft Corporation)
    MKS                  Sierra Mastercook Cookbook File 
    MKS                  Matroska Subtitles Stream (Matroska Association)
    MKS                  Tact Data 
    MKTS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MKV                  Matroska Video Stream (Matroska Association)
    MKW                  Compressed Audio File (Michael K. Weise)
    MKZ                  PokerOffice Backed Up Database (PokerOffice.com)
    ML                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ML                   Camomile Configuration/Module File 
    ML                   ML language Source Code File 
    ML0                  Multiledger Company Name (CheckMark Software)
    ML2                  Tripos MOL2 Molecule File 
    ML20                 MSN Webcam Recorder Video File 
    ML3                  Milestones 3.x Schedule (KIDASA Software, Inc.)
    ML4                  Milestones ver 4.0 and 4.5 Schedule (KIDASA Software, Inc.)
    ML5                  Milestones ver 5.0 and 5.01 Schedule (KIDASA Software, Inc.)
    ML6                  Milestones Professional 2000 Schedule (KIDASA Software, Inc.)
    ML6                  PowerPoint Milestone File (Microsoft)
    ML7                  Milestones Professional 2002 Schedule (KIDASA Software, Inc.)
    ML8                  Milestones Professional 2004 Schedule (KIDASA Software, Inc.)
    ML9                  Milestones Professional 2004U Schedule (KIDASA Software, Inc.)
    MLA                  Milestones Professional 2006 Schedule (KIDASA Software, Inc.)
    MLB                  MyLittleBase Database 
    MLB                  Symphony Macro Library File 
    MLB                  Peachtext List Manager (Peachtree Software (Sage Software SB, Inc.))
    MLB                  FoxPro for Macintosh Library 
    MLB                  OpenVAX Macro Library 
    MLC                  mAirList Cart Set (Torben Weibert)
    MLD                  Multilizer Dictionary 
    MLD                  mAirList Desktop (Torben Weibert)
    MLD                  Polyphonic Ringtone File for Phones 
    MLE                  mAirList Event List (Torben Weibert)
    MLF                  UpdateEXPERT Machine List File (St. Bernard Software)
    MLI                  AutoCAD Material-Library File (Autodesk, Inc.)
    MLI                  3D Studio Material-Library File (Autodesk, Inc.)
    MLI                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MLI                  Camomile Individual Module File 
    MLI                  Encyclopedia Media List 
    MLINEFONT            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MLK                  Sierra Mastercook Look File 
    MLM                  Novel Groupwise E-mail File 
    MLN                  AutoCAD Multiline Definition (Autodesk, Inc.)
    MLNK                 Multi-Channel Server (MCS) Component Policy Editor Link Component (Volantis Systems Ltd.)
    MLP                  Meridian Lossless Packing Audio File (Meridian Audio)
    MLP                  mAirList Playlist (Torben Weibert)
    MLS                  Miles Sound Tools Compressed DLS 
    MLS                  Milestones 1.x and 2.x Schedule and Simplicity (all ver) (KIDASA Software, Inc.)
    MLS                  mAirList Script (Torben Weibert)
    MLT                  Forecast Pro Multivariate Forecast File (Business Forecast Systems, Inc.)
    MLT                  mAirList Desktop Template (Torben Weibert)
    MLTV                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MLYT                 Multi-Channel Server (MCS) Layout Policy Editor Layout Policy (Volantis Systems Ltd.)
    MM                   Multimate Advantage File 
    MM                   Xcode Core C Program (Apple Computer, Inc.)
    MM                   Rise of Nations Campaign Media File (Microsoft)
    MM                   Lightscape Material Map 
    MM                   Macro Mania Macro 
    MM                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MM                   FreeMind Mind Map 
    MM                   Meme Virtual Reality World (Immersive Systems Inc.)
    MM3                  Mortgage Minder Ver. 3 Saved Mortgage Configuration File (CW Technical Services)
    MM4                  Mortgage Minder Ver. 4 Saved Mortgage Configuration File (CW Technical Services)
    MM5                  Makromat Macroeconomic Model Data (University of Muenster, Germany)
    MMA                  Master Album Maker Album File (Intrepid Mouse Limited)
    MMA                  Grlib Cppima Scilimag File 
    MMAP                 MindManager Brainstorm and Process Control Map (Mindjet)
    MMB                  MultiMedia Builder 
    MMB                  Oracle Forms Menu Binary Source Code 
    MMBK                 MannaBook (MannaBook Software, Inc.)
    MMBM                 MannaBook (MannaBook Software, Inc.)
    MMC                  Office Media Content 
    MMC                  Media Catalog 
    MMC                  Clipart DLLbackup Autoshape File (Microsoft)
    MMD                  Nigarva Downloader Download List File (K&A Corporation)
    MMD                  Summit Membership Manager Database (MIE Software Pty Ltd)
    MMD                  Cumulate Draw Editable MMD Format (Cumulate Labs)
    MMD                  Media Processing Server Studio Data (Nortel Networks)
    MMD                  MangoData Multi-User Database (MightyMango)
    MMD                  mAirList Metadata (Torben Weibert)
    MMD                  DocsAlive Multimedia Document (Softedge-Systems Ltd)
    MMD                  MacroModel Molecular Mechanics File (Columbia University Chemistry)
    MME                  Multi-Purpose Internet Mail Extensions (MIME) File 
    MMF                  AppMind Message Management File (Appmind Software AB)
    MMF                  Mail File (Microsoft)
    MMF                  Media Processing Server Studio Reference Name (Nortel Networks)
    MMF                  McAfee VirusScan Configuration (Networks Associates Technology, Inc.)
    MMF                  Yamaha SMAF Synthetic Music Mobile Application Format (Yamaha Corporation)
    MMF                  Meal Master Format 
    MMG                  mkvmerge GUI Saved Muxing Settings (Moritz Bunkus)
    MMH                  MS Media Manager Helper DLL 
    MMI                  Media Processing Server Studio Index (Nortel Networks)
    MMI                  Memo Open Client 
    MMJB_MIME            MusicMatch Jukebox MIME ID File (MUSICMATCH, Inc.)
    MML                  FrameMaker (Adobe)
    MML                  Mail Meta Language 
    MML                  MySoftware or ProVenture MailList Program Mail List (Elibrium, LLC)
    MML                  MathML Math Embedded HTML File 
    MML                  MolMol Molecule Description File 
    MML                  Aleph One Marathon Markup Language (Marathon Open Source)
    MMLD                 EdGCM Data File (The EdGCM Cooperative Project of Columbia University)
    MMM                  MacroMind Directory RIFF/RMMP Format Movie 
    MMM                  Multiple Master Font File (Adobe)
    MMM                  Multimedia Movie (Microsoft)
    MMO                  RapidFile Memo Writer 
    MMO                  Memory-Map Navigator Overlay (Memory-Map, Inc.)
    MMO                  Musicmatch On Demand Playlist (Musicmatch, Inc.)
    MMOD                 MacroModel Molecular Mechanics File (Columbia University Chemistry)
    MMP                  EPOC Makmake Makefile Project 
    MMP                  Grand Theft Auto Multiplayer Map File (Take-Two Interactive Software, Inc.)
    MMP                  CoCo Systems VisiMap 
    MMP                  MixMeister Playlist (MixMeister Technology)
    MMP                  MicroMedia Playlist (Demonic Software)
    MMP                  Bravado Board Video Output Format 
    MMP                  MindManager Map File (Mindjet)
    MMP                  Gammadyne Mailer Project File 
    MMP                  Bravado Board Output Video Format 
    MMP                  MS Music Producer 
    MMP                  Master Album Maker Album Project File (Intrepid Mouse Limited)
    MMP                  Multimedia Player Movie Clip 
    MMPBT                MindManager Branch Template File 
    MMPT                 MindManager Template 
    MMR                  LOKI Character Template (Synthetic Realms)
    MMS                  JPEG-6b File 
    MMS                  Miles Sound Tools 
    MMS                  NT MidiMagic Song Information (NovaTech Design)
    MMS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MMSCRIPT             MindManager Script File 
    MMT                  Nimbus 7 SAMS Data File 
    MMT                  CoCo Systems VisiMap 
    MMT                  egseing Memory Dump Data 
    MMV                  Sony MovieShaker File 
    MMW                  Multimedia World Magazine 
    MMW                  AceMoney Financial Data (MechCAD Software LLC)
    MMW                  Media Content 
    MMX                  Oracle Forms Compiled Menu 
    MMX                  MarkMagic Import/Export 
    MMX                  Command & Conquer Red Alert 2 Map File 
    MMZ                  MusicMatch Theme File (MUSICMATCH, Inc.)
    MN1                  Money Ver 1 Data File (Microsoft)
    MN2                  Money Ver 2 Data File (Microsoft)
    MN2                  Descent2 Mission File 
    MN3                  Descent 3 
    MN3                  Descent3 Mission File 
    MN3                  Money Ver 3 Data File (Microsoft)
    MN4                  Money Database (Microsoft)
    MN4                  Money Ver 4 Data File (Microsoft)
    MN5                  Money Ver 5 Data File (Microsoft)
    MN6                  Money Ver 6 Data File (Microsoft)
    MN7                  Money Ver 7 Data File (Microsoft)
    MN8                  Money Ver 8 Data File (Microsoft)
    MN9                  Money Ver 9 Data File (Microsoft)
    MNB                  MuPAD File 
    MNC                  My Notes Center File 
    MNC                  AutoCAD Compiled Menu (Autodesk, Inc.)
    MND                  Fractal Forge Ver. 2.x Fractal Parameters 
    MND                  AutoCAD Menu Program (Autodesk, Inc.)
    MND                  Cerious Mandelbrot Image 
    MND                  MNI Mandelbrot for Windows 
    MND                  MusicNet Music Download (MusicNet)
    MNF                  Saved MSN Search 
    MNG                  DeLorme Map'n'Go Map 
    MNG                  Keyboard File Manager MIDI/MID Files Floppy Database (Yahama)
    MNG                  Animation Shop Animation (Corel Corporation)
    MNG                  Multi-image Network Graphic Animation 
    MNI                  Cerious Mandelbrot Image 
    MNL                  Zortech 3 Binary File 
    MNL                  AutoCAD AutoLISP Functions Related to Menu (Autodesk, Inc.)
    MNL                  OrCAD Netlist File (Cadence Design Systems, Inc.)
    MNO                  Monopoly Saved Game 
    MNO                  Macromedia Design Notes 
    MNPF                 NoteStar v1.2a + Graphic (Jayden Carr)
    MNR                  AutoCAD Compiled Menu (Autodesk, Inc.)
    MNR                  IBM Voice Type Language Newuser File 
    MNS                  AutoCAD ASCII Menu (Autodesk, Inc.)
    MNT                  Foxpro Menu Memo (Microsoft)
    MNU                  HP NewWave Advanced Macro 
    MNU                  Menu File 
    MNU                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MNU                  AutoCAD Menu Template (Autodesk, Inc.)
    MNU                  FoxPro Menu 
    MNW                  Menu File 
    MNX                  AutoCAD Compiled Menu (Autodesk, Inc.)
    MNX                  Foxpro Menu (Microsoft)
    MNY                  Money Data File (Microsoft Corporation)
    MO                   Dymola Simulation 
    MO                   Modula-3 Object File 
    MO                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MO                   GNU Gettext Machine Object File (Free Software Foundation)
    MO1                  Odyssey Disk/ROM Image 
    MO3                  MO3 MOD Audio Format (un4seen developments)
    MOB                  Device Definitions 
    MOB                  Master of Orion 3 
    MOBAPP               MOBILedit! Application (COMPELSON Laboratories)
    MOBDRV               MOBILedit! Driver (COMPELSON Laboratories)
    MOBSKIN              MOBILedit! Skin (COMPELSON Laboratories)
    MOC                  MOCHA Javascript 
    MOCHA                MOCHA Javascript 
    MOD                  Peachtext Database (Peachtree Software (Sage Software SB, Inc.))
    MOD                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MOD                  Femap (UGS Corp.)
    MOD                  CADdy++ Mechanical 2D/3D Model Data (DataSolid GmbH)
    MOD                  WinBoard Import Module (Ivex Design International)
    MOD                  JVC Everio GZ-MG20U Digital Video File (JVC Americas Corp)
    MOD                  PHP Source Code 
    MOD                  Music File 
    MOD                  Multiplan Spreadsheet (Microsoft)
    MOD                  Monarch File 
    MOD                  Module 
    MOD                  Modula-2 Program File 
    MOD                  Model File 
    MOD                  File Implementing DOS Support for Windows DLL's 
    MOD                  EPOC Model Definition 
    MOD                  Digital Music Sound Clip 
    MOD                  Connectivity Memory Model Cell Specific Plasticity Functions on Incoming Synapses Input File 
    MOD                  CATIA Export File (Dassault Systèmes)
    MOD                  Amiga and PC Tracker Module 
    MOD                  Music Module 
    MOD                  Windows Kernel Module 
    MOD                  Multiplan Spreadsheet 
    MOD                  Clarion for Modula-2 Source Code File (SoftVelocity)
    MOD                  ProModel Optimization Software Suite Model 
    MODEL                PlantWALK Model 
    MODEL                CATIA Model (Dassault Systèmes)
    MODEL                LEGO CAD Model File (The LEGO Group)
    MODEL                Ribbons Molecular Model 
    MODEM                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MODEM1               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MODEM2               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MODEM3               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MODEM4               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MODEM5               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MODEM6               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MODEM7               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MODEM8               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MODEM9               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MODERNTALKING        Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MODS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MODULAE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MODULE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MODULES              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MOF                  MSinfo (Microsoft)
    MOF                  Systems Management Server (SMS) Managed Object Format (Microsoft Corporation)
    MOHO                 Anime Studio Project File (Lost Marble)
    MOI                  French Text File 
    MOI                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MOL                  Easy Palette Object Library 
    MOL                  MDL Molfile Molecule File (MDL Information Systems, Inc.)
    MOL                  MOdule (play)List 
    MOL2                 Portable Representation of a SYBYL Molecule 
    MOL2                 WebLab Molecular Viewer File 
    MOM                  Xcode Compiled Core Data-based Data Model (Apple Computer, Inc.)
    MOM                  CDrom Runtime Database File 
    MOM                  SPEFO Stellar Spectra Analysis File 
    MON                  Virgin Games Monopoly Deluxe 
    MON                  Firefox Bookmark Backup File 
    MON                  ReadMail Monitor 
    MONDAY               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MONEY                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MONICA               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MONO                 Bi-level Bitmap in Least-Significant-Byte First Order 
    MONO                 Monolith Munged Encoded File 
    MONS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MONSTER2             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MONTE                CRiSP Harvest File 
    MOO                  QuickTime Movie Clip (Apple Computer, Inc.)
    MOO                  MoonRock Language Source Code File 
    MOOD                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MOON                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MOOV                 QuickTime Movie (Apple Computer, Inc.)
    MOP                  M.Fujimiya's Morpher for Windows 1.4 Work File 
    MOP                  MDL Molfile MOPAC Input Format (MDL Information Systems, Inc.)
    MOP                  Mopac Input File 
    MOP                  NS/Elite Display 3.12 Macro Object Path File 
    MOR                  Poser 3D Morph Target File 
    MOR                  Corel Shared Writing Tools 9.0 File 
    MORE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MORTBATS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MOS                  Infinity Game Engine Background and GUI Tile Info (BioWare Corp.)
    MOS                  MACAOS Compressed Tranismission File (Elprint AS)
    MOS                  System DOS 6.2 File 
    MOS                  Dymola Simulation 
    MOS                  Infinity Game Engine Compressed Background and GUI Tile Info (BioWare Corp.)
    MOT                  Sharp cellphone GX Flash File (Sharp)
    MOT                  Linux Load Instructions 
    MOT                  Phoneme Translation File 
    MOTIF                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MOTIF                Microangelo Motif 
    MOUNT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MOUSE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MOUSETEXT            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MOV                  AutoCAD AutoFlix Movie (Autodesk, Inc.)
    MOV                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MOV                  QuickTime Video Clip (Apple Computer, Inc.)
    MOVE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MOVEMON              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MOVERADDIN           Mover Add-in (Nicodemous Bytes)
    MOVERBACKUP          Mover Backup Set (Nicodemous Bytes)
    MOVERCONFIG          Mover Configuration (Nicodemous Bytes)
    MOVERDB              Mover Database (Nicodemous Bytes)
    MOVERREG             Mover Registration Info (Nicodemous Bytes)
    MOVERTEMPLATE        Mover Template (Nicodemous Bytes)
    MOVES                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MOVIE                QuickTime Movie (Apple Computer, Inc.)
    MOVIE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MOX                  Extend Simulation Model File 
    MOZ                  Netscape Temp File 
    MOZART               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MP                   cGPSmapper Polski Format Mapy -- Polish Map Format (Stanislaw Kozicki)
    MP                   Multiplan File 
    MP                   MetaPost Macro 
    MP                   Monochrome Picture TIFF Bitmap 
    MP                   Maya Personal Learning Edition (Alias Systems, a division of Silicon Graphics Limited)
    MP                   EARS Component File 
    MP+                  MPEG Plus Audio File 
    MP1                  MPEG Audio Stream, Layer I 
    MP2                  MPEG Audio Stream, Layer II 
    MP2                  Masterpoint Red Points File (ABF Masterpoint Centre)
    MP2S                 Max Payne 2 Saved Game (Rockstar Games)
    MP2S                 MPEG-2 Video 
    MP2V                 MPEG Audio Stream, Layer II 
    MP3                  Wrapster Wrapped File 
    MP3                  AlbumWrap Wrapped Music File Archive (Infamus Software LLC)
    MP3                  mp3PRO Audio File 
    MP3                  Masterpoint Green Points File (ABF Masterpoint Centre)
    MP3                  MPEG Audio Stream, Layer III 
    MP3                  SHARP MZ-series Emulator File 
    MP3PRO               mp3PRO Enhanced MP3 File 
    MP4                  MPEG-4 Video File 
    MP4V                 MPEG-4 Video File 
    MP5                  Maple V Release 3 Windows/DOS File (Maplesoft, a Division of Waterloo Maple Inc.)
    MP5                  Masterpoint Red and Green Points File (ABF Masterpoint Centre)
    MPA                  Palm Memo File 
    MPA                  MPEG Audio Stream, Layer I, II or III 
    MPA                  E-Designer (Beijer Electronics)
    MPB                  Maintenance Parts Bin Database (Nhuntsoftware)
    MPB                  TruVideo Video-only MPB Stream 
    MPB                  MyPhoneExplorer Backup File (Franz Josef Wechselberger)
    MPC                  Collectorz.com MP3 Collector MP3 Collection Data (Collectorz.com)
    MPC                  Musepack Audio File 
    MPC                  Project Calender (Microsoft Corporation)
    MPC                  VersaPro Compiled Motion Block 
    MPCB                 ReliaSoft MPC 3 
    MPCF                 MarketPlace Pro Configuration Settings (Claritas)
    MPCPL                Media Player Classic Playlist 
    MPD                  File List Creator MP3 Playlist (WenSoftware)
    MPD                  MLCad LDraw Multipart Model File (LM-Software)
    MPD                  SCSI MiniPort Driver (Microsoft)
    MPD                  Windows Mini-port Driver 
    MPD                  Project Database File (Microsoft)
    MPD                  My Digital Photos Filmstrip (Super Win Software)
    MPD                  ABC Programming Language Monadic Predicate 
    MPDEX                MPdex MP3 Database Utility 
    MPE                  MPEG Movie Clip 
    MPE.EXE              MP3 Digital Player Wrapper 
    MPEG                 MPEG 1 System Stream 
    MPF                  MosASCII Project Workspace File 
    MPF                  Design Gallery (Microsoft)
    MPF                  MainActor Project (MainConcept AG)
    MPF                  MPF Audio File (Mobijet Science & Technology Co., Ltd.)
    MPG                  MPEG 1 System Stream 
    MPG                  MPEG Animation 
    MPG                  Salt Lake City Game Movies File 
    MPG                  ProgDVBR MPEG2 Video File 
    MPG4                 MPEG-4 Media File 
    MPGA                 Mpeg-1 Layer3 Audio Stream 
    MPH                  Photoline4 Morphing File 
    MPH                  Publisher (Microsoft)
    MPH                  COMSOL Multiphysics Model File (COMSOL, Inc.)
    MPI                  MPlayer Control File 
    MPI                  Links98 Golf Game File 
    MPJ                  MINITAB Project File (Minitab Inc.)
    MPJ                  STOIK Morph Man Project Information File 
    MPJ                  Mountain Video Editing File 
    MPJ                  Sometimes .MP3 File 
    MPJ                  RoboHTML Project 
    MPK                  Augmented Microsoft-Project (Microsoft)
    MPKG                 Meta Package File 
    MPLS                 Blu-ray Movie Playlist Information (Blu-ray Disc Association)
    MPM                  Max Payne MOD Archive (Rockstar Games)
    MPM                  MPEG Movie 
    MPM                  WordPerfect MathPlan Macro (Corel Corporation)
    MPN                  mophun Game File (morphun)
    MPNT                 MacPaint Bitmap File 
    MPO                  Augmented Microsoft-Project (Microsoft)
    MPO                  Sheffield Dictionary File 
    MPP                  BX-24 Basic Express Compiler Map 
    MPP                  Musepack Audio File 
    MPP                  CAD Drawing File 
    MPP                  Project Project (Microsoft Corporation)
    MPP                  Zortech 3 README File 
    MPQ                  Blizzard Game Data File (Blizzard Entertainment)
    MPR                  Multilizer Project File (Multilizer, Inc.)
    MPR                  Music Publisher Backup Data File (Braeburn Software)
    MPR                  Westwood Studios Game File 
    MPR                  FileMaker Spelling Dictionary 
    MPR                  Live for Speed Multiple Player Replay 
    MPR                  Visual Foxpro Solution Menus File (Microsoft)
    MPR                  AutoCAD Mass Properties Report (Autodesk, Inc.)
    MPS                  Casio PDL Pocket Streets Map 
    MPS                  MapSource GPS Mapping Interface Waypoints (Garmin Ltd.)
    MPS                  Max Payne Saved Game (Rockstar Games)
    MPS                  MPEG-1 Audio and Video File 
    MPS                  Pocket Streets Map File (Microsoft Corporation)
    MPS                  MySweetNix! 
    MPS                  Multimedia File 
    MPT                  Project Template File (Microsoft Corporation)
    MPT                  Multipage TIFF Bitmap 
    MPV                  MyPhotoZip File 
    MPV                  MPEG-1 Video File 
    MPV                  Project View (Microsoft Corporation)
    MPV2                 MPEG Audio Stream, Layer II 
    MPW                  MosASCII Project Workspace File 
    MPW                  Oc2.316s Cakit File 
    MPX                  Alpha Five Temporary Index Information (Alpha Software, Inc.)
    MPX                  RCA/Lyra Handheld MP3 Player File (RCA)
    MPX                  Foxpro Compiler Menu (Microsoft)
    MPX                  BX-24 Basic Express Compiler Map 
    MPX                  Project Export File (Microsoft Corporation)
    MPY                  RCA/Lyra Handheld MP3 Player File (RCA)
    MPZ                  SoundSlimmer Compressed Audio (Sound Genetics Inc.)
    MPZ                  SignIQ Project (ImpulseLogic)
    MPZ                  MyPhotoZip File 
    MPZ                  Moove Resource Description File (moove Corporation)
    MQC                  CP/M Disk Drive Information File 
    MQD                  Malete Query Data (OpenIsis)
    MQL                  MetaTrader Indicator File (MetaQuotes Software Corp.)
    MQP                  Multiquence Project 
    MQV                  Sony Movie Player Format (Sony Corporation of America)
    MQX                  Malete Query Access File (OpenIsis)
    MRA                  VilagRadio Recorded Radio Program (MAVIP Ltd)
    MRB                  C++ Multiple Resolution Bitmap Graphic 
    MRC                  MIRC Script File 
    MRC                  Bibliographic Data Format 
    MRD                  Infinity Document (MathRevolt)
    MRD                  Malete Record Data File (OpenIsis)
    MRES                 Homeworld (Sierra Entertainment, Inc.)
    MRF                  MeterReading Data (Dataconomy)
    MRF                  WinMorph Project Information File 
    MRF                  Macintosh Font 
    MRF                  Marks Russel File 
    MRF                  Grade Busters Teacher Tools File 
    MRF                  GPS Pathfinder Office Geoid Grid File (Trinble Navigation Limited)
    MRF                  MultiEx Commander Resource File (XeNTaX)
    MRF                  Morpheus Saved Game 
    MRG                  DESQview (Symantec)
    MRG                  Merge File 
    MRI                  MRI Scan 
    MRK                  Windows 95 User Benchmark 
    MRK                  Informative Graphics Markup File 
    MRK                  SleepScan and Ceegraph Stores Technician (User) Comments (Bio-logic Systems Corp.)
    MRK                  MissionRisk 
    MRK                  CSI Markup File 
    MRK                  TAPCIS Marked Messsage Commands 
    MRLV                 Multi-Channel Server (MCS) Component Policy Editor Rollover Image Component (Volantis Systems Ltd.)
    MRO                  SharpEye Music Reader File 
    MRP                  Infinity Solution Plot (MathRevolt)
    MRS                  WordPerfect Macro Resource File (Corel Corporation)
    MRW                  Sony (Minolta) Raw Image File (Sony)
    MRX                  Malete Record Access File (OpenIsis)
    MS                   Modula-3 Intermediate Assembly File 
    MS                   Formatted Manual Page with MS Macros 
    MS                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MS                   Maple Worksheet 
    MS                   Checksum File for Anti-Virus 
    MS1                  OzWin CompuServe E-mail/Forum Access Message File 
    MS3D                 MilkShape 3D Model 
    MS8                  Multisim Design (National Instruments Corporation)
    MS9                  Multisim Design (National Instruments Corporation)
    MSA                  Atari MSA Disk Image (Atari)
    MSA                  Magic Shadow Archive 
    MSBCM                Business Contact Manager Backup Database (Microsoft Corporation)
    MSC                  CRiSP Harvest File 
    MSC                  C Makefile 
    MSC                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MSC                  eYaMosaic Graphic Mosaic (ESRI China (Beijing))
    MSC                  Microsoft Management Console Snap-in Control File (Microsoft Corporation)
    MSCR                 Multi-Channel Server (MCS) Component Policy Editor Script Component (Volantis Systems Ltd.)
    MSD                  Multimedia Studio Document (Burton Software)
    MSD                  Macro eXpert Macro (GrassSoftware)
    MSD                  Advanced Key and Mouse Recorder Macro (GrassSoftware)
    MSD                  Diagnostic Utility Report (Microsoft)
    MSD                  [Unknown] 
    MSD                  Maxsurf Design File (Formation Design Systems Pty Ltd.)
    MSDEV                Msvc Make File 
    MSDL                 Manchester Scene Description Language 
    MSE                  Bwsb120a Demo File 
    MSF                  Multiple Sequence File 
    MSF                  Thunderbird Mail Summary File (Mozilla Corporation)
    MSF                  EarthLink E-mail Message File (EarthLink, Inc.)
    MSF                  Netscape Mail Mail Summary File (Netscape)
    MSG                  Program Message 
    MSG                  OzWin CompuServe E-mail/Forum Access Message File 
    MSG                  Exchange Mail Message (Microsoft Corporation)
    MSG                  PMMail 2000 E-mail Message File (Blueprint Software Works)
    MSG                  Pegasus Mail Stored Messages To Be Sent (David Harris)
    MSG                  FoxPro FoxDoc Message 
    MSG                  TAPCIS Captured Messages 
    MSG                  PMMail/2 E-mail Message File (V.O.I.C.E. (Virtual OS/2 International Consumer Education))
    MSG                  OS/2 Help Message File (IBM)
    MSG1                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MSG2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MSG3                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MSH                  Orbiter Mesh File (Martin Schweiger)
    MSH                  FLUENT Mesh (Fluent Incorporated)
    MSH                  MASH 
    MSHR                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MSI                  TruNEST Nesting Data (Magestic Systems, Inc.)
    MSI                  Cerius2 (Accelrys Software Inc.)
    MSI                  Windows Installer File (Microsoft Corporation)
    MSK                  BMC Software Patrol Icon File 
    MSK                  Paint Shop Pro Mask File (Corel Corporation)
    MSK                  CARA-Script-Interpreter-Engine Mask 
    MSK                  Animator Mask Format (Autodesk, Inc.)
    MSL                  MAINSAIL Source Code (XIDAK, Inc.)
    MSL                  ImageMagick Scripting Language File (ImageMagick Studio LLC)
    MSL                  ProWORX Nxt MSL Loadables (Schneider Electric)
    MSM                  Multisim Circuit Diagram (National Instruments Corporation)
    MSM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MSM                  Windows Installer Merge Module (Microsoft)
    MSN                  Network Document 
    MSN                  MSN Content Plus Installer File (MSN Content Plus Inc.)
    MSN                  Descent Mission File 
    MSN                  Microsoft Network Document (Microsoft)
    MSN                  Microsoft Network MSN Central (Microsoft)
    MSNBAK               MSN Mail Backup File (Microsoft)
    MSO                  MedlySound Sound/Module 
    MSO                  Word OLE Web Page Storage Stream (Microsoft Corporation)
    MSO                  Orgchart File (Microsoft)
    MSO                  Math Script Object File 
    MSO                  FrontPage File (Microsoft)
    MSP                  Windows Installer Patch 
    MSP                  Paint Bitmap (Microsoft)
    MSP                  Technics Sx KN 6000 Keyboard Performance Pads File 
    MSP                  Technics Sx KN 5000 Keyboard Manual Sequence Pads File 
    MSP                  Magix Music Studio File 
    MSPLG                Modem Spy Plug-in File (SoftCab Inc.)
    MSPX                 XML-based Web Page (Microsoft)
    MSQ                  MIDI File 
    MSR                  OzWin CompuServe E-mail/Forum Access SYSOP File 
    MSRCINCIDENT         Remote Assistance Request (Microsoft)
    MSS                  Manuscript Text File 
    MSS                  Game Score 
    MSS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MSS                  Miles XMIDI Sequence with DLS Data 
    MSS                  MediaShout Presentation Script (MediaComplete)
    MSSTYLES             Windows XP Visual Style (Microsoft)
    MSSYTLES             Windows Visual Style File 
    MST                  Windows SDK Setup Transform Script (Microsoft Corporation)
    MST                  Visual Test Source File 
    MST                  Minispecification 
    MST                  Room Arranger Design File (Jan Adamec)
    MST                  Pension System Master File (DATAIR Employee Benefit Systems, Inc.)
    MST                  Presentation Master Presentation Document (Corel)
    MST                  ChemFinder Chemical Structure Index 
    MSU                  Vista Hotfix Installer Package (Microsoft Corporation)
    MSV                  MyStars! Saved View (Relative Data, Inc.)
    MSV                  Memory Stick Compressed Voice File (Sony Corporation of America)
    MSVC                 Msvc Make File 
    MSW                  Word Text File (Microsoft Corporation)
    MSWMM                Windows Movie Maker Project (Microsoft Corporation)
    MSX                  Age of Empires II: The Conquerors Expansion Multi-player Saved Game (Microsoft Corporation)
    MSX                  Magic Splitter Split File Rebuild Information (Kynosoft, Brescia)
    MSX                  Image Format 
    MSX                  Compressed CP/M File Archive 
    MSX                  SuperGraphic Studio Message File (Time-O-Matic, Inc.)
    MSZ                  MiSZle Data/Parameter File (Mystic Fractal)
    MS_                  Winhelp Compressed File 
    MT                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MT2                  MadTracker Ver. 2.0 Module/Song (Yannick Delwiche)
    MTA                  Flight Simulator Maintenance File (Microsoft)
    MTB                  MINITAB Exec File 
    MTD                  Digital Sheet Music 
    MTE                  MadTracker Envelope File 
    MTE                  TargetExpress Target (MCW Technologies Inc.)
    MTF                  Darkstone Archive 
    MTF                  Mathe Vektor MTF File 
    MTF                  MicroSurfer File 
    MTG?                 Jay Klein Making the Grade Teacher Gradebook Program 
    MTGA                 Homeworld (Sierra Entertainment, Inc.)
    MTH                  Derive Math File 
    MTHM                 Multi-Channel Server (MCS) Theme Policy Editor Theme Policy (Volantis Systems Ltd.)
    MTI                  Musitron MUSILI.ZIT Compressed File 
    MTI                  MadTracker Instrument File 
    MTK                  MPU-401 Trakker 
    MTK                  Matrikon Software License File 
    MTL                  Wavefront 
    MTL                  Mastercam Mill Tool Library (CNC Software, Inc.)
    MTL                  MathLook File 
    MTM                  MultiTracker Music Module 
    MTOOL                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MTP                  MadTracker Pattern File 
    MTP                  L&H Power Translator Pro 
    MTR                  RealFlight Radio Control Airplane Description (Knife Edge Software)
    MTR                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MTS                  Viewpoint iPix File 
    MTS                  Encore Samples File 
    MTS                  MadTracker Sample File 
    MTT                  Messenger Saved Contact File (Microsoft)
    MTV                  MTV Music Generator 
    MTV                  MTV Ray-tracer 
    MTW                  TRADOS MultiTerm Termbase 
    MTW                  Minitab Data File 
    MTX                  Viewpoint iPix File 
    MTX                  M-Tx Music-From-Text Translator Source File 
    MTX                  Marked Text Source File 
    MTX                  File WSOCK32.MTX Associated with MTX Worm 
    MTX                  MadTracker Extension File 
    MTX                  Temporary File often used by a Browser or TWAIN Device 
    MTX                  GROMACS: Molecular Dynamics Hessian Matrix Format 
    MTX                  Max Magic Microtuner Tuning Interchange Document (Victor Cerullo)
    MTX                  Adobe Atmosphere File (Adobe)
    MTX                  Matrix Spreadsheet (Grove Business Software)
    MTX                  Anjal2000 Tamil Text File Format Description 
    MTXDEF               Gestionnaire d'extensions Multiple Extension Definition (Définition d'extension multiple) (Francis Gagné)
    MTXT                 Multi-Channel Server (MCS) Component Policy Editor Text Component (Volantis Systems Ltd.)
    MT_                  Encore Compressed Audio File 
    MU                   Quattro Menu (Corel)
    MUC                  Collectorz.com Music Collector Collection Data (Collectorz.com)
    MUD                  ACT! (Best Software CRM Division)
    MUD                  Yahoo! Mail Saved Attachment 
    MUD                  zMUD Settings File (Zugg Software)
    MUF                  ProtoMuck Multi User Forth Program 
    MUF                  Alpha Five Temporary Access Control File (Alpha Software, Inc.)
    MUI                  Configuration Resource File (Microsoft)
    MUI                  Opengl Glut-3.6 File 
    MUI                  Amiga MUI Default Class/Library (SASG)
    MUL                  SeeYou Flight Data (Naviter)
    MUL                  Ultima Online Game 
    MULTICALC            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MULTICL2             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MULTICL3             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MULTICLR             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MUP                  Music Publisher Data File (Braeburn Software)
    MUS                  Finale Music Score (MakeMusic!)
    MUS                  MusicTime/Doom Sound 
    MUS                  SCORE Music Publishing System Music Notation File (San Andreas Press)
    MUS                  Infinity Game Engine Music Control File (BioWare Corp.)
    MUS                  Harmony Assistant, Melody Assistant, Myriad Music plug-in, Melody Player Music File (MYRIAD)
    MUS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MUSIC                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MUTE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MUX                  Trackmania Music File (NADEO)
    MUZ                  Hover Game MIDI File 
    MV                   IRIX makemovie Command Compressed Movie (Silicon Graphics, Inc.)
    MV                   Miva Script (Prior to Ver. 4.14) (Miva Corp.)
    MV3                  Oozic 3D MusicVideo 
    MV3                  AUPEC Audio File 
    MVA                  Matrox Video Card Driver 
    MVA                  Setup Program Archive 
    MVB                  Manual Storage Format 
    MVB                  Multimedia Viewer (Microsoft)
    MVB                  Database 
    MVC                  Collectorz.com Movie Collector Data File (Collectorz.com)
    MVC                  Miva Compiled Script (Ver. 4.14 and Later) (Miva Corp.)
    MVD                  MicroVision Database 
    MVD                  MicroDVD DVD Movie 
    MVE                  muvee autoProducer Project File (muvee Technologies Pte. Ltd.)
    MVE                  Infinity Game Engine (BioWare Corp.)
    MVE                  Wing Commander III MVE Movie File Format 
    MVE                  Interplay Video File 
    MVE                  4DVista Geological Model (Midland Valley Exploration)
    MVF                  AutoCAD Stop Frame File (Autodesk, Inc.)
    MVF                  Production Studio Video (Babya)
    MVG                  ImageMagick Vector Graphic (ImageMagick Studio LLC)
    MVHLINK              MyVirtualHome Link to Online Features (Creative Designer Software Pty Ltd)
    MVI                  AutoCAD Movie Command (Autodesk, Inc.)
    MVL                  WinMaven Saved Log File (David R. Snow)
    MVL                  Interplay Multimedia File 
    MVP                  MediaView Project 
    MVR                  AutoCut Configuration File (Autometrix)
    MVS                  MUSICMATCH Jukebox Visualization (MUSICMATCH, Inc.)
    MVU                  EchoView Adaptive Binary Optimization Format File (MatrixView Pte. Ltd.)
    MVW                  Saber LAN Log File 
    MVX                  Intentia International Movex 
    MVX                  Mixer File 
    MW                   Mail Wiper Mail Control File (Mail Wiper, Inc.)
    MW                   Maple Worksheet (Maplesoft, a Division of Waterloo Maple Inc.)
    MW                   MacWrite II Text Document (Apple)
    MW1                  SILK Spreadsheet 
    MW4                  MechWarrior 4 Map (Game Structure Archive) (Microsoft Corporation)
    MWB                  MySQL Workbench Database Structure (MySQL AB)
    MWD                  Photo Soap2 File 
    MWF                  MapGuide Author Map Window (Autodesk, Inc.)
    MWF                  ProMotion Animation 
    MWI                  Medlin Accounting Invoice Layout Information (Medlin Accounting)
    MWII                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MWII                 MacWrite II Text Document 
    MWL                  Mighty Draw Windows Library (theliquidateher Software, An MCS Investments, Inc. Company)
    MWM                  Mail Warrior File 
    MWP                  MegaWorks Pack File 
    MWP                  Word Pro 97 Smartmaster File (IBM)
    MWS                  Medlin Accounting Statement Layout Information (Medlin Accounting)
    MWS                  Maple Spreadsheet (Maplesoft, a Division of Waterloo Maple Inc.)
    MWT                  MetaStock Stock Analysis Template (Equis International)
    MWV                  MovieWorks File 
    MW_                  Audio Utility Midimach Compressed File 
    MX                   Modula-3 Linker Information 
    MX                   MATLAB Common Matrix Routines (The MathWorks, Inc.)
    MX                   MX Designer Remote Control File 
    MX3                  Bitbop Tuner Locked Sound File 
    MX3                  MP3 Encoded File 
    MX3                  MATLAB Windows Dynamically Linked Subroutines (The MathWorks, Inc.)
    MXC                  ArcView Geocoding Index For Read-Write Coverages (ESRI)
    MXD                  MX Editor Remote Control Device File 
    MXD                  ArcMap GIS Project File (Map) (ESRI)
    MXE                  Macro Express 
    MXE                  Mindex Effect Album 
    MXE                  MXEdit File 
    MXE                  VCom Error Report (V Communications, Inc.)
    MXE                  Quatro Pro Startup Macro (Corel)
    MXF                  MX Editor Remote Control File 
    MXF                  SimCity Maxis Font File (Electronic Arts, Inc.)
    MXF                  Material Exchange Format File 
    MXI                  Proteus File 
    MXI                  Mindex Indices File for Indexed Images 
    MXI                  Macromedia Extension Information 
    MXL                  Moxcel Spreadsheet 
    MXL                  PackRat 5.0 Support File 
    MXM                  Mindex Master Index File for a Category 
    MXM                  Project/Outlook Team Assign Task (Microsoft)
    MXMF                 Mobile XMF File Format 
    MXP                  Macromedia Extension Manager 
    MXP                  ArcReader Published Map 
    MXS                  ArcView Geocoding Index For Read-Write Shapefiles (ESRI)
    MXT                  ArcMap Map Template (ESRI)
    MXT                  C Data (Microsoft)
    MXV                  Movie Edit Pro Video Format (MAGIX)
    MYB                  MYOB Data File (MYOB Technology Pty Ltd.)
    MYC                  Simply Safe Backup Archive (Joshua Cannon Butcher)
    MYD                  MySQL Database (MySQL AB)
    MYDOCS               MyDocs Drop Target (Microsoft)
    MYI                  MySQL Database Index (MySQL AB)
    MYL                  MYLAI Multilingual Transliteration Scheme 
    MYO                  MYOB Data File (MYOB Technology Pty Ltd.)
    MYP                  Make Your Point Presentation 
    MYR                  Harmony Assistant, Melody Assistant, Myriad Music plug-in, Melody Player Music File (MYRIAD)
    MYS                  Myst Saved Game 
    MYT                  MyTraffic X AI Aircraft Traffic Database File (BK & simMarket.com)
    MYTHS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MYV                  Vivaldi Studio Score File (goVivaldi Inc.)
    MYW                  MyWorld Education Drawing 
    MZ                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    MZ                   MOZART Music Document (David Webber)
    MZB                  MegaZeux Board 
    MZC                  MOZART Chord Dictionary (David Webber)
    MZE                  MidiMaze File 
    MZE                  Maze Creator File 
    MZF                  SHARP MZ-series Emulator File 
    MZF                  MediaZip Compressed File (ACE)
    MZM                  MegaZeux Dynamic Board Section 
    MZP                  MOZART Percussion File (David Webber)
    MZQ                  SHARP MZ-series Emulator QuickDisk File 
    MZS                  SHARP MZ-series Emulator File 
    MZT                  SHARP MZ-series Emulator File 
    MZT                  MOZART Template File (David Webber)
    MZX                  MegaZeux World 
    M_C                  MailCall 2 File 
    M_P                  Pro/ENGINEER Temporary File (PTC)
    M_U                  MazeGold Critical File Backup 
    N                    Nuke Randomic Life Generator 
    N                    Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    N                    Neko Bytecode (Motion-Twin)
    N1                   Nuke Randomic Life Generator 
    N1E                  OM2 OMALI NENA Encrypted Document Version 1 
    N2E                  OM2 OMALI NENA Encrypted Document Version 2 
    N2P                  Net2Phone File 
    N3                   Notation 3 Document 
    N3D2                 The Nebula Device 2 Mesh Resource File in ASCII Format (RadonLabs GmbH)
    N3E                  OM2 OMALI NENA Encrypted Document Version 3 
    N4E                  OM2 OMALI NENA Encrypted Document Version 4 
    N64                  MiMiX Emulator 
    N64                  Nintento 64 Emulation ROM Image 
    NA                   NoAdware Data File (NoAdware.net)
    NA2                  Netscape Communicator Address Book 
    NAB                  Netscape Communicator Address Book 
    NAB                  Novell Groupwise Address Book 
    NACHTMUSIK           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NAD                  Noder File 
    NAF                  Compressed Archive 
    NAI                  WinINSTALL File 
    NAIL                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NAK                  Audio 
    NAL                  ArrayScribe (NimbleGen Systems Inc.)
    NAL                  Systems Management Server (SMS) Network Abstraction Layer (Microsoft Corporation)
    NAM                  Office Name File 
    NAME                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NAMEONCLIPBOARD      NAMEONCLIPBOARD Powertoy File 
    NAMES                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NAN                  Nanoscope File 
    NAP                  Electrical Network Analysis and Planification 
    NAP                  NAP Metafile Vector Image 
    NAP                  EnerGraphics VideoShow File 
    NAP                  Napster Music Security Definition File 
    NAP                  VideoShow/EnerGraphics NAPLPS Vector Image Metafile 
    NAP                  ePolicy Orchestrator Network Associates Package (Policy File) (Network Associates Technology, Inc.)
    NAPLPS               North American Presentation Layer Protocol Syntax = Vector image 
    NAR                  .NET Archive 
    NAR                  Nikon Capture Advanced RAW File (Nikon Inc.)
    NAS                  NASTRAN File 
    NASB                 TheWord & Bible Companion New American Standard Bible 
    NAV                  Counter-Strike: Source CSBot Navigation Mesh (Valve Corporation)
    NAV                  Navigator Desktop File (Tecepe)
    NAV                  Microsoft Network Component (Microsoft)
    NAV                  Application Extension (Microsoft)
    NAV                  MSN Application Extension 
    NAVGTR               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NAY                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NB                   Mathematica Notebook (Wolfram Research, Inc.)
    NB                   WordPerfect Notebook (Corel Corporation)
    NB                   Nota Bene Document File (Nota Bene Associates, Inc.)
    NB7                  NovaBACKUP System Configuration  File (NovaStor Corporation)
    NBA                  NBA Game Data 
    NBA                  Nota Bene Configuration Information File (Nota Bene Associates, Inc.)
    NBATTRS              NetBeans Project Attributes File (Collab.Net, Inc.)
    NBE                  Nessus File 
    NBF                  Backup NOW! Backup Archive (NewTech Infosystems)
    NBG                  NBG Clean Registry File 
    NBI                  BackItUp Document (Ahead Software)
    NBK                  i-Brarian 
    NBK                  Nick-O-Matic Design Factory Project (Riverdeep Interactive Learning Limited)
    NBK                  NovaBACKUP Script (NovaStor Corporation)
    NBM                  NetBeans Module (Sun)
    NBP                  NeoBook Plug-in (NeoSoft Corp.)
    NC                   Geopath Machine Part File (SolutionWare Corporation)
    NC                   X-editor Program for CNC Machines (X-net ApS)
    NC                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NC                   Unidata netCDF Graphics 
    NC                   Preprosessed GOC Source 
    NC                   Numerical Control Machine Instructions 
    NC                   neatComponents 
    NC                   Mastercam NC Programs (CNC Software, Inc.)
    NC                   Graphics File 
    NCA                  NetCam Supervisor Archive 
    NCA                  Nikon Capture Color Adjustment File (Nikon Inc.)
    NCB                  Nikon Capture Color Booster File (Nikon Inc.)
    NCB                  Nick-O-Matic Design Factory Project (Riverdeep Interactive Learning Limited)
    NCB                  Developer Studio File (Microsoft)
    NCC                  NetCam Supervisor Directory 
    NCC                  Computer Numeric Control File 
    NCC                  Nokia PC Suite Content Copier File (Nokia)
    NCC                  NetControl V1 1999 
    NCC                  Nikon Capture Camera File (Nikon Inc.)
    NCD                  Xilinx Integrated Software Environment Native Circuit Description (Xilinx, Inc.)
    NCD                  CoverDesigner CD Cover Picture (Nero)
    NCD                  Norton Change Directory 
    NCD                  NTI CD-Maker File 
    NCE                  Nick-O-Matic Design Factory Project (Riverdeep Interactive Learning Limited)
    NCF                  GibbsCAM CNC Program Listing (Gibbs and Associates, Inc.)
    NCF                  Notes Internal Clipboard (IBM)
    NCF                  Steam Game Configuration File (Valve Corporation)
    NCF                  Netware Command File (Novell, Inc.)
    NCF                  Systems Management Server (SMS) Network Configuration File (Microsoft Corporation)
    NCF                  Nick-O-Matic Design Factory Project (Riverdeep Interactive Learning Limited)
    NCH                  On Hold Message/Music File 
    NCH                  Outlook Express Folder File (Microsoft)
    NCI                  NetCam Supervisor Meta Data 
    NCI                  Mastercam Toolpath File (CNC Software, Inc.)
    NCL                  RHS Nibbles Custom Level (Redwood High School)
    NCM                  Communicator Configuration Message (Nokia)
    NCM                  NetCam Supervisor Video 
    NCO                  Nero BackItUp Compressed Backup File (Nero AG)
    NCP                  NHL Ice Hockey 2002 User NHLcards File 
    NCR                  NCR Image 
    NCS                  Netscape Conference Call File 
    NCSA                 NCSA Hierarchical Data File Bitmap 
    NCT                  Nero Cover Designer Template (Nero AG)
    NCV                  Nikon Capture Curves File (Nikon Inc.)
    NCX                  SecurDataStor Secure Archive (encryptX Corporation)
    NDA                  Nick-O-Matic Design Factory Project (Riverdeep Interactive Learning Limited)
    NDA                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NDAS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NDB                  SeeYou Waypoint (Naviter)
    NDB                  ProWORX Nxt Short Comment (Schneider Electric)
    NDB                  Intellicom Network Database 
    NDB                  NetOp Various Information Files 
    NDB                  ACT! Notes Data File (Best Software CRM Division)
    NDB                  Nick-O-Matic Design Factory Project (Riverdeep Interactive Learning Limited)
    NDC                  Nick-O-Matic Design Factory Project (Riverdeep Interactive Learning Limited)
    NDCT                  DALiM LiTHO Continuous Tone Bitmap (Blanchard Systems Inc.)
    NDD                  Nikon Capture Digital DEE File (Nikon Inc.)
    NDF                  ProWORX Nxt Short Comment (Schneider Electric)
    NDF                  NeoPlanet Browser File 
    NDF                  NimbleScan NimbleGen Design File (NimbleGen Systems Inc.)
    NDF                  SQL Server Secondary Data File (Microsoft)
    NDG                  Nick-O-Matic Design Factory Project (Riverdeep Interactive Learning Limited)
    NDK                  OS/2 Saved Disk Format (IBM)
    NDK                  Notes Database Design Elements (IBM)
    NDL                  Notes Link (IBM)
    NDL                  Nick-O-Matic Design Factory Project (Riverdeep Interactive Learning Limited)
    NDL                  Nodelist File 
    NDL                  Inmos Hardware Network Description File 
    NDO                  3D Low-polygon Modeler 
    NDP                  Nick-O-Matic Design Factory Project (Riverdeep Interactive Learning Limited)
    NDQ                  Totem Cash Data File 
    NDR                  Nikon Capture Image Dust Off File (Nikon Inc.)
    NDR                  RAVE Reports Report File (Nevrona Designs)
    NDS                  Nick-O-Matic Design Factory Project (Riverdeep Interactive Learning Limited)
    NDS                  Nintendo DS Game ROM Image (Nintendo)
    NDS                  Indiana Jones and the Infernal Machine Saved Game (Lucasfilm Entertainment Company Ltd.)
    NDW                  Allplan Drawing File (Nemetschek A.G.)
    NDX                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NDX                  Rational XDE (IBM)
    NDX                  Index 
    NDX                  ArcView Fonts Index File (ESRI)
    NDX                  WINDEV Index (PC SOFT)
    NE1                  OM2 OMALI NENA Document Version 1 
    NE2                  OM2 OMALI NENA Document Version 2 
    NE3                  OM2 OMALI NENA Document Version 3 
    NE4                  OM2 OMALI NENA Document Version 4 
    NEB                  Nortec H.E.L.P. 
    NEBULA               Homeworld Nebula Data (Sierra Entertainment, Inc.)
    NEBULOS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NEC                  NEC JIS Encoded File 
    NEC                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NEC                  PIMS Notes for Windows CE Audio Record File (Nacetech Co., Ltd.)
    NED                  MSN Application Extension 
    NEEDS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NEF                  Nikon Digital SLR Camera Raw Image File (Nikon Inc.)
    NEN                  Nick-O-Matic Design Factory Project (Riverdeep Interactive Learning Limited)
    NEO                  NeoPhoto NeoPhoto Album (Archive) (DTLink)
    NEO                  Atari NeoChrome Bitmap Image 
    NEO40SFONT           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NES                  Nintendo Entertainment System ROM Image 
    NET                  FileMaker Pro Networking Module 
    NET                  OrCAD Netlist Output File (Cadence Design Systems, Inc.)
    NET                  netViz Project (netViz Corporation)
    NET                  Network Configuration 
    NETA                 Netica Bayes Net File (Norsys Software Corp.)
    NETCDF               Scientific data NETwork Common Data Format 
    NEU                  MENSI 3D Laser Scanner File 
    NEUALL2              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NEUCHEAT1            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NEW                  New Information 
    NEW                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NEWFONT              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NEWORDER             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NEWPLR               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NEWS                 News Bitmap Image 
    NEWS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NEWSLETTER           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NEWTECH              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NEX                  Nexcon System File 
    NEX                  Neurological Data for Statistical Analysis 
    NEXTURN              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NEZ                  Nintento NES ROM Image 
    NF                   CVIP NF_help File 
    NF2                  NeverForget Reminder File (National Creative Registry, Inc.)
    NFB                  Nokia PC Suite Content Copier (Backup) File (Nokia)
    NFC                  Nokia PC Suite Content Copier File (Nokia)
    NFC                  Netware Command File (Typo) 
    NFE                  Nikon Capture Fisheye Lens Info File (Nikon Inc.)
    NFF                  Haines Neutral File Format and WorldToolkit minimal 3D scene description language 
    NFF                  WorldToolKit Neutral File Format (Sense8)
    NFG                  RHS Nibbles Finished Game File (Redwood High School)
    NFG                  Gambit Normal Form Representation (The Gambit Project)
    NFL                  Sierra Mastercook Layout File 
    NFL                  AutoCAD Multiline Filter List (Autodesk, Inc.)
    NFM                  NFM Archival Data Format File (Prevo Technologies, Inc.)
    NFO                  Folio Infobase (Fast Search & Transfer (FAST))
    NFO                  System Info File (Microsoft)
    NFO                  JCanyon: Grand Canyon for Java DEM Companion File 
    NFO                  Text File 
    NFO                  Warez Group Info File 
    NFT                  Netobjects Fusion Netobjects System File 
    NFX                  Netobjects Fusion Components BBScomp File 
    NG                   Norton Guide Online Documentation 
    NGB                  Nonogram Puzzle 
    NGD                  NimbleScan NimbleGen Gene Description File (NimbleGen Systems Inc.)
    NGD                  Xilinx Integrated Software Environment Native Generic Database (Xilinx, Inc.)
    NGF                  Enterasys Networks NetSight Generated Format File 
    NGG                  Nokia Phone Group Graphics/Icons (Nokia)
    NGL                  Sheffield Dictionary File 
    NGO                  Nick-O-Matic Design Factory Project (Riverdeep Interactive Learning Limited)
    NGO                  NG Linker Object (NGML) 
    NGO                  numberGo Dynamic Report (numberGo, LLC)
    NGP                  NeoGeo Pocket ROM Image 
    NGR                  Nokia PC Suite Install Archive (Nokia)
    NGS                  NG Compiler Source (NGC) 
    NGS                  NGSSniff File 
    NH                   NetHack Defaults File 
    NHB                  No Hands Backup Backup Description (Anders Petersson's Free Software)
    NHF                  Nero HFS-CD Compilation (Nero AG)
    NHL                  NHL Ice Hockey Game 2002 User Teams File 
    NHM                  Systems Management Server (SMS) No History MIF File (Microsoft Corporation)
    NHV                  Nero HD-BURN-Video Compilation (Nero AG)
    NI                   Netobjects Fusion Preview File 
    NI                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NIA                  Nikon Capture Image Adjustment File (Nikon Inc.)
    NIB                  Interface Builder User Interface Resources (Apple Computer, Inc.)
    NIB                  Corel Graphics10 Photopoint File 
    NIC                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NIC                  Nici Encrypted Image/Graphic File (The DiScosel Group)
    NICHOLS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NICK                 Office Data File (Microsoft)
    NICO                 Nolan's Icon Maker Icon File (Nolan's Games)
    NID                  Nikon Capture IPTC Data File (Nikon Inc.)
    NIF                  Notation Interchange File Format (NIFF) 
    NIF                  NetShift Interface File (NetShift Software Ltd.)
    NIF                  GameBryo Format File (Numerical Design Limited)
    NIF                  NetImmerse Format File (Numerical Design Limited)
    NIF                  Navy Image File Format 
    NIF                  Network Initialization File 
    NIFF                 Navy Interchange File Format Bitmap 
    NIGHT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NIGHTS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NIL                  Norton Icon Library File 
    NIM                  Nokia Phone Game File (Nokia)
    NIP                  Network Interface Plug-in 
    NIRVANA              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NIS                  Homeworld Non-Interactive Sequence (Sierra Entertainment, Inc.)
    NIT                  ArcView INFO Table Definitions File (ESRI)
    NITF                 National Imagery Transmission Format 
    NIV                  TheWord & Bible Companion New International Version Bible 
    NIX                  Rational XDE (IBM)
    NJP                  njPipes Compiler File 
    NJX                  NJStar Document 
    NK2                  Outlook AutoComplete File (Microsoft Corporation)
    NKC                  ECG Viewer (Nihon Kohden Corporation)
    NKI                  KONTAKT Instruments (Native Instruments)
    NKJV                 TheWord & Bible Companion New King James Version Bible 
    NKM                  KONTAKT (Native Instruments)
    NKY                  NUMROTOplus Keyfile (NUM Guettinger)
    NL                   Norton Desktop Icon Library 
    NL                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NLB                  Oracle 7 Data 
    NLB                  Colin Mc Rae Car Race Network Leader Board 
    NLC                  NeroLINUX Compilation (Nero AG)
    NLD                  ATI Radeon Video Driver 
    NLD                  Windows Media Audio/Video File (Microsoft Corporation)
    NLF                  ReliaSoft Weibull++ 6 
    NLF                  MicroSim PCBoard Netlist File for Schematics 
    NLIST                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NLM                  Nokia Phones LogoManager Bitmap (Nokia)
    NLM                  Netware Loadable Module (Novell)
    NLO                  NetG Skill Builder 
    NLP                  Newsletter Profi 2001 
    NLR                  NLREG Nonlinear Regression Control File (Phillip H. Sherrod)
    NLS                  Code Page National Language Support 
    NLS                  W32/Alman-B Virus 
    NLTRACK              NoLimits Rollercoaster Simulator Track File 
    NLU                  Norton Live Update E-Mail Trigger File 
    NLV                  Nikon Capture LCHE Editor File (Nikon Inc.)
    NLX                  FormWorx Form 
    NM                   RM Number Magic Data File (RM plc)
    NMA                  GPS Log Data 
    NMA                  Normalised Extinction Corrected Differential Photometry 
    NMAT                 NeoEngine (Reality Rift Studios)
    NMD                  SwordSearcher File 
    NMD                  Nero miniDVD Compilation (Nero AG)
    NMDAR                Connectivity Memory Model Input File When Tau1=0.0 
    NMEA                 GPS Log Data 
    NMF                  Lightwave Narmal Map 
    NMF                  Nuzzler Basic Message File (Securepoint Security Solutions)
    NMF                  Note-Mate Question Set (StudySoft)
    NMF                  Nuzzler Basic Rule Definition File (Securepoint Security Solutions)
    NMI                  SwordSearcher File 
    NMK                  Oc2.316s Cakit File 
    NML                  NEURO ONE Static Neuron Model (Netral)
    NML                  NewsMail 2.0 E-mail Newsletter (ESU-Soft)
    NML                  .nugthat Compiler Source Code (.nugthat Mark-up Language) (Nuggets Unlimited, Inc.)
    NML                  NML Language Source File (NevoSoft)
    NMO                  Virtools Behavioral Objects 
    NMS                  Virtools Graphical Scripts 
    NMS                  Numega Softice's Loader File 
    NMS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NMS                  Note-Mate Save Position (StudySoft)
    NMW                  Netmeeting T126 Compatable Whiteboard Document (Microsoft)
    NNB                  newnovelist Story Outline 
    NNI                  ANNI Artificial Neural Network Investing Single Investment File (Neural Science)
    NNR                  Nikon Capture Noise Reduction File (Nikon Inc.)
    NOA                  Cell Phone Video Clip 
    NOA                  Nancy Video CODEC 
    NOB                  Vampire: La Mascarade Archive 
    NOB                  VersaPro Word Exchange File 
    NOB                  Silver Sprites and Data 
    NOD                  Noder File 
    NOD                  NetObjects Fusion 
    NOFIN                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NOI                  TGIF-3.0 File 
    NOL                  Nokia Phone Operator (Manager) Logo (Nokia)
    NOM                  Aonix Nomad 
    NOMFICHIER           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NON                  Star Wars - Tie Fighter Mouse Options 
    NON                  Nonogram Puzzle 
    NOR                  Personal Paint Norwegian Language User Interface File (Amiga)
    NOR                  ATI Radeon Video Driver 
    NORMAL               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NORUN                Unknown 
    NOT                  Notation or Note 
    NOT                  Acrobat Spelling File (Adobe)
    NOT                  Broderbund Note Card Type (Riverdeep Interactive Learning Limited)
    NOT                  Note-It Note File (Veign)
    NOT                  e-Sword Study Note File (Rick Meyers)
    NOT                  ActiveNote Post-It-Notes 
    NOTE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NOTE                 WordPerfect for Macintosh Help Document (Corel Corporation)
    NOTE91               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NOTEPAD              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NOTES                Réviseur de notes Notes File (Fichier de notes) (Francis Gagné)
    NOTES                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NOTESEQ              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NOTESYN              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NOTESYNTH            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NOTH                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NOTICE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NOW                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NOW                  Text File 
    NP                   Planner Project Schedule (Nokia)
    NP                   Cliq Accessories Notepad Text 
    NP                   Portfolio NetPublish Web Template (Extensis, Inc.)
    NPA                  ReliaSoft Weibull++ 6 
    NPAC                 NeoEngine Resource Package (Reality Rift Studios)
    NPC                  Nick-O-Matic Design Factory Project (Riverdeep Interactive Learning Limited)
    NPC                  Jagged Alliance 2 Non-Player Character File (Sir-tech Canada Ltd.)
    NPD                  Ultra Network Sniffer Captured Packet (GJPSoft, Inc.)
    NPD                  Noder File 
    NPDOC                Newspilot Document (Infomaker Scandinavia AB)
    NPF                  Nick-O-Matic Design Factory Project (Riverdeep Interactive Learning Limited)
    NPI                  Nikon Capture Filters File (Nikon Inc.)
    NPI                  dBASE Application Generator Source 
    NPI                  Nick-O-Matic Design Factory Project (Riverdeep Interactive Learning Limited)
    NPK                  MikroTik RouterOS Upgrade Package 
    NPL                  Core Media Player Nefertiti Playlist (CoreCodec, Inc.)
    NPL                  DART Pro 98 Noiseprint Info 
    NPL                  Xilinx Integrated Software Environment Project File (Xilinx, Inc.)
    NPL                  Nokia Audio File (Nokia)
    NPL                  Allplan Plot Layout File (Nemetschek A.G.)
    NPM                  Corel Graphics Ver 10 Draw Media Lines File (Corel)
    NPM                  Nokia Phone Picture Message (Nokia)
    NPM                  Neopaint Mask 
    NPP                  Art Explosion Publisher Pro Document (Nova Development Corporation)
    NPS                  NeroMix (Nero AG)
    NPS                  Lotus Agenda File 
    NPSEW                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NPT                  DART Pro 98 Noiseprint Archive Import 
    NPT                  Portfolio NetPublish Pre-cursor Templates (Extensis, Inc.)
    NQC                  Not Quite C Source Code (Dave Baum)
    NR3                  Nero MP3 CD-ROM Compilation (Nero AG)
    NR4                  Nero AAC CD Compilation (Nero AG)
    NRA                  Nero Audio-CD Compilation (Nero AG)
    NRB                  Nero CD-ROM Boot Compilation (Nero AG)
    NRC                  Nero UDF/ISO CD-ROM Compilation (Nero AG)
    NRD                  Nero DVD-Video Compilation (Nero AG)
    NRE                  Nero CD Extra Compilation (Nero AG)
    NRF                  Neutral Raster File Bitmap 
    NRF                  Nicolet DataViewer Multi-segment Waveform Data (Nicolet Instrument Technologies, Inc.)
    NRF                  NEC Scanner File 
    NRG                  Norton Registration Entries (Symantec)
    NRG                  Nero CD-Image File (Nero AG)
    NRH                  Nero Hybrid CD-ROM Compilation (Nero AG)
    NRI                  Trickster Online Image (NTREEV SOFT)
    NRI                  Nero ISO CD-ROM Compilation (Nero AG)
    NRL                  iManage File 
    NRM                  Nero Mixed-Mode-CD Compilation (Nero AG)
    NRS                  Nero CD-ROM (EFI Boot) Compilation (Nero AG)
    NRSV                 TheWord & Bible Companion New Revised Standard Version Bible 
    NRT                  Nokia Phone Ring Tone (Nokia)
    NRU                  Nero UDF CD-ROM Compilation (Nero AG)
    NRV                  Nero Video-CD Compilation (Nero AG)
    NRW                  Nero WMA CD-ROM Compilation (Nero AG)
    NRX                  NeatReceipts SCANALIZER Expense Report File (NeatReceipts)
    NS1                  NetStumbler NS1 Log 
    NS2                  Notes Ver 2 Database (Older Form) (IBM)
    NS3                  Notes Database (Older Form) (IBM)
    NS4                  Notes Database (Older Form) (IBM)
    NS5                  Notes/Domino (IBM)
    NSB                  Novaschem School Schedule 
    NSC                  Noder File 
    NSC                  Windows Media Station File 
    NSCE                 NeoEngine Scene File (Reality Rift Studios)
    NSD                  NRG Site Database 
    NSD                  Nero Super Video CD Compilation (Nero AG)
    NSD                  Norton System Doctor Sensors Configuration (Symantec)
    NSDF                 NevoSoftDB Standard Format Database File (NevoSoft)
    NSF                  NES Sound File 
    NSF                  Notes Database (IBM)
    NSG                  Notes Database (Older Form) (IBM)
    NSH                  Notes Database (Older Form) (IBM)
    NSH                  Nullsoft Scriptable Install System (NSIS) Header/Include File 
    NSI                  Symantec Shared File (Symantec Corporation)
    NSI                  Install System Script (Nullsoft)
    NSI                  NewsStand Reader (NewsStand, Inc.)
    NSIS                 Compressed Archive 
    NSK                  NaShrinK Archive (NashSoft Systems)
    NSL                  Nokia Phone Startup Logo (Nokia)
    NSM                  NESticle Movie 
    NSO                  Nick-O-Matic Design Factory Project (Riverdeep Interactive Learning Limited)
    NSO                  NetStudio Easy Web Graphics File 
    NSOF                 NoteStar v1.2a + Options Data (Jayden Carr)
    NSP                  Computerized Speech Lab Audio Speech Format (Kay Elemetrics Corp.)
    NSP                  Multi-Speech Sound File (Kay Elemetrics Corp.)
    NSR                  Nessus Project 
    NSR                  Nikon Capture Size and Resolution File (Nikon Inc.)
    NSRD                 SH&E NetWorks Report Design (Simat, Helliesen and Eichner, Inc.)
    NSS                  Norton ScreenSaver Module 
    NST                  Audio 
    NST                  Noise Tracker Music Module 
    NSV                  Nullsoft (Streaming|S as in Soft) Video (Nullsoft, Inc.)
    NSV                  Winamp3 Video Format File 
    NSX                  SuccessWare SIX 3.00 Compound Index File 
    NSX                  Apollo Database Engine Index 
    NSX                  HiPer-Six Index File 
    NT                   Windows NT Command Script (Microsoft Corporation)
    NT0                  Adressmn 
    NT1                  Adressmn 
    NT2                  Adressmn 
    NT3                  Adressmn 
    NT6                  Orchida Knitting System Universal Flat Knitting Machine Pattern (Orchida Soft)
    NT7                  Orchida Knitting System Universal Flat Knitting Machine Pattern (Orchida Soft)
    NTC                  Nikon Capture Camera Curves Information File (Nikon Inc.)
    NTE                  Holonote Note 
    NTF                  Notes Database Template (IBM)
    NTF                  National Imagery Transmission Format (NITF) 
    NTF                  National Transfer Format Map File 
    NTH                  ImageMagick Ver. 5.3.2 Visualmagick JBIG File (ImageMagick Studio LLC)
    NTH                  Series 60 Theme Studio for Symbian OS Theme File (Nokia)
    NTN                  NOTION Performance Score (VirtuosoWorks, Inc.)
    NTO                  NTOScanner 
    NTP                  Neato CD Labels 
    NTR                  Netrun Executable Text File 
    NTRK                 IL2 Track File (Ubisoft Entertainment)
    NTS                  Psion Organiser Notepad File (Psion PLC)
    NTS                  Norton Tutorial 
    NTS                  Netsend Executable Text File 
    NTS                  Norman TurboStore Database Data Storage (Norman ASA)
    NTT                  CD Label Template 
    NTV                  PatternSmith Import File (Autometrix)
    NTW                  LVNet - Low Voltage Networks Analysis 
    NTW                  Noteworthy Composer Song File (NoteWorthy Software, Inc.)
    NTX                  yEnc32 yEncoded File 
    NTX                  Clipper Index 
    NTX                  PIMS Notes for Windows CE Text Note File (Nacetech Co., Ltd.)
    NTX                  NTX Web Audio File 
    NTX                  Network Terminal Accelerator Adapter Card Configuration Information 
    NTX                  CARIS Marine Navigation Info 
    NTZ                  InVircible Directory Integrity Information 
    NU4                  Norton Utilities DLL Root File (Symantec)
    NU6                  Norton Utilities System DLL File 
    NUF                  Procomm Plus New User Message 
    NUG                  .nugthat Player Nugget File (Nuggets Unlimited, Inc.)
    NUL                  CCS-lib.2 ACS File 
    NUL                  Grand Theft Auto (Playstation 2) Game File (Take-Two Interactive Software, Inc.)
    NULL                 NULL Image 
    NUM                  Guitar Chords in Nashville Numbering System Format 
    NUM                  DOS 7 File 
    NUM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NUM                  Nikon Capture Unsharp Mask File (Nikon Inc.)
    NUMBER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NUMBERWHEEL          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NUMBRWHEEL           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NUOVO                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NUP                  NOD32 Digitally Signed Update File (Eset)
    NUT                  .nugthat Player (Nuggets Unlimited, Inc.)
    NUV                  MythTV NuppelVideo Video File 
    NUWAGE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NV                   Juno Access Number Info File (Juno Online Services, Inc.)
    NV                   Visicontrol MVS-20 Non-Volatile Memory Backup File 
    NV!                  Net Transport Partial Download (Exciting Software)
    NV3                  Visicontrol MVS-30/35/40 Flash-Memory Backup File 
    NVC                  NeroVision Express Project File (Nero AG)
    NVC                  Nikon Capture Vignette Correction Information File (Nikon Inc.)
    NVD                  AOLpress Help 
    NVD                  NetworkView Network Discovery 
    NVF                  Nomad II Voice File (Creative Technology, Ltd.)
    NVF                  Visicontrol MVS-20 Non-Volatile Memory Backup File 
    NVM                  AOLpress Help 
    NVR                  NeverForget Reminder File (National Creative Registry, Inc.)
    NVV                  Nvidia Vertex Shader (Nvidia)
    NVX2                 The Nebula Device 2 Mesh Resource File in Binary Format (RadonLabs GmbH)
    NW                   NoteWorthy 2 File 
    NW                   Graal Level (Cyberjoueurs)
    NW                   Dispatch Northwest Timetable File (Scott Andrew Borton)
    NW3                  Netware.3x File 
    NW4                  Netware.4x File 
    NW8                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NWB                  Nikon Capture White Balance File (Nikon Inc.)
    NWC                  NavisWorks File 
    NWC                  Noteworthy Composer Song File (NoteWorthy Software, Inc.)
    NWD                  NavisWorks File 
    NWF                  NavisWorks File 
    NWID                 NeoEngine (Reality Rift Studios)
    NWK                  Nano Wiki  Engine Wiki Data Source (Silvio Torre)
    NWP                  NEO Content File 
    NWS                  Creatacard Newsletter or Photo Album or Presentation or Report Cover or Resume Project (Broderbund)
    NWS                  Outlook Express News File (Microsoft)
    NWS                  Newsmaster or Newsmaster II Data File 
    NWV                  Dragon NaturallySpeaking User Profile File (Scansoft, Inc.)
    NWX                  NWX Editor (TheOutlawDad)
    NWX                  Outlaws Choreography (LucasArts Entertainment Company LLC)
    NW_                  Noteworthy Compressed Audio File 
    NXD                  ndxCards Index Card Project (TruTamil LLC)
    NXP                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NXP2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NXS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    NXT                  NeXT Sound File 
    NY                   Nyquist Plug-in 
    NYF                  Mybase Database (WJJ Software)
    NYM                  Sheffield Dictionary File 
    NYT                  Times Reader Article (New York Times)
    NZ                   Audio Utility Tuningwrench Compressed File 
    NZB                  newzBin USENET Index (newzBin)
    NZL                  Corel Painter Nozzle File 
    NZN                  NoteZilla Sticky Note (Conceptworld Corporation)
    O                    Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    O                    Object File 
    O                    UNIX/GCC Object File 
    O                    Origin C Compiled Program File (OriginLab)
    O                    Free Pascal Object FIle for Linux or DOS Version (FPC Development Team)
    O                    Xcode Compiled Core C or Assembler Program (Apple Computer, Inc.)
    O$$                  Sprint Outfile 
    O$$                  Output File 
    O01                  Typhoon Voice File 
    O1                   Free Pascal Object File for GO321v1 Platform (FPC Development Team)
    O1                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    O2C                  Objects to See 3D File Format (mb Software AG)
    O3                   Ozone Messenger Program Support Files (htmlland.net)
    O8O                  Mediator5 Repositorytabs Vorlagen.006 File 
    OAB                  Outlook Address Book (Microsoft)
    OAF                  ETH Oberon Applet File 
    OAF                  Outrage Animation Rows 
    OAF                  Flight Simulator Texture File (Microsoft)
    OAH2DSK              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OAZ                  NetFax Manager OAZ Fax File 
    OB                   ABF Outlook Backup Backup File (ABF software, Inc.)
    OB                   IMB LinkWay Object File 
    OB$                  Compressed OBJ (Microsoft)
    OB1                  OzWin CompuServe E-mail/Forum Access Saved Outbox File 
    OBA                  Timespace X100 Digital Video Recorder Video File (Electronics Services Agency)
    OBD                  Oracle Forms Online Help File (Oracle)
    OBD                  Office Binder Document (Microsoft)
    OBE                  Out Of Box Experience 
    OBE                  System DLL Oobe File 
    OBJ                  Wavefront 3D Object 
    OBJ                  Relocatable Object Code 
    OBJ                  Metatools Object Bryce Support File 
    OBJ                  LightWave 3D Object (NewTek)
    OBJ0                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OBJ1                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OBJECTS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OBK                  Oxygen Phone Manager Phone Book (Oxygen Software)
    OBK                  Ubi Soft Chessmaster Opening Book File 
    OBN                  Oberon-2 Module Source File 
    OBP                  Bryce 5 Object Preset (Corel Corporation)
    OBPACK               ObjectBar Object Package (Stardock Corporation, Inc.)
    OBR                  Objective Reference Document Link (Objective Corporation Limited)
    OBR                  C++ Object Browser Data File (Borland Software Corporation)
    OBS                  ObjectScript Script 
    OBS                  OzWin CompuServe E-mail/Forum Access Saved Outbox File 
    OBSKIN               ObjectBar Skin File (Stardock Corporation, Inc.)
    OBT                  Office Binder Template (Microsoft)
    OBTHEME              ObjectBar Theme File (Stardock Corporation, Inc.)
    OBTHEME              ObjectBar Theme File 
    OBV                  ObjectScript Visual Interface 
    OBX                  Rational XDE (IBM)
    OBZ                  Office Binder Wizard (Microsoft)
    OC8                  OrgCon 8 Organization (EcoMerc)
    OC9                  OrgCon 9 Organization (EcoMerc)
    OCA                  OLE Custom Control Library Information 
    OCA                  Control Typelib Cache (Microsoft)
    OCA                  Visual Basic Control Typelib Cache 
    OCB                  Origin Precompiled Origin C File (OriginLab Corporation)
    OCC                  DB/TextWorks Database Terms and Words 
    OCC                  OptControlCentre Model File 
    OCCSAN               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OCD                  OrgCon 7 Organization (EcoMerc)
    OCD                  OCAD Digitized Map (Steinegger Software)
    OCE                  Brio Query (Hyperion Solutions Corporation)
    OCEN                 EdGCM Data File (The EdGCM Cooperative Project of Columbia University)
    OCF                  Object Craft File 
    OCF                  OM2Cipher File 
    OCM                  AOL Advertising Control File (America Online, Inc.)
    OCM                  Internet Odyssey 2 Update 
    OCM                  Netscape Communicator Aim File 
    OCP                  Offline Commander Project File 
    OCP                  Advanced Art Studio 
    OCR                  FAXGrapper FAX Text File 
    OCS                  eDoc DigiSign Digitally Signed Document (Fujitsu Denmark)
    OCT                  Octalizer Musical File 
    OCT                  Radiance Octree Format 
    OCTOPUSFONT          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OCX                  Object Linking and Embedding (OLE) Control Extension (Microsoft)
    OC_                  Doko Compressed Install File 
    OD1                  Omnis5 Database File 
    OD2                  Omnis5 Database File 
    OD3                  Omnis5 Database File 
    OD4                  Omnis5 Database File 
    OD5                  Omnis5 Database File 
    OD6                  Omnis5 Database File 
    OD7                  Omnis5 Database File 
    OD8                  Omnis5 Database File 
    OD9                  Omnis5 Database File 
    ODA                  AIBO Audio Voice (Sony)
    ODA                  CALS Group IV Type 2 Raster Image 
    ODA                  Used with OCX 
    ODA                  ODA Document 
    ODB                  OpenOffice/StarOffice OpenDocument (Ver 2) Database (Sun Microsystems, Inc.)
    ODB                  Psion Organiser Datafile and LZ Diary File (Psion PLC)
    ODB                  Organizers Database (ODB) Database (Organizers' Collaborative)
    ODB                  ArcView Object Database ASCII File (ESRI)
    ODB                  ABAQUS File (ABAQUS, Inc.)
    ODC                  OpenOffice/StarOffice OpenDocument (Ver 2) Chart (Sun Microsystems, Inc.)
    ODC                  Office Data Connection 
    ODCCUBEFILE          Office Data Connection 
    ODCDATABASEFILE      Office Data Connection 
    ODCNEWFILE           Office Data Connection 
    ODCTABLEFILE         Office Data Connection 
    ODDS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ODE                  Office Object Data Embedding File (Microsoft)
    ODF                  Star Trek Armada Ship/Structure Infomation 
    ODF                  Open Document Interchange (ODIF) 
    ODF                  OnkoS Data (OnkoS)
    ODF                  BattleZone Cartographers Guild File 
    ODF                  OpenOffice/StarOffice OpenDocument (Ver 2) Formula (Sun Microsystems, Inc.)
    ODG                  OpenOffice/StarOffice OpenDocument (Ver 2) Graphics Document (Sun Microsystems, Inc.)
    ODH                  Odigo History File (Comverse, Inc. a part of Comverse Technology)
    ODI                  OpenOffice/StarOffice OpenDocument (Ver 2) Image (Sun Microsystems, Inc.)
    ODIF                 Open Document Interchange Format 
    ODL                  Visual C++ Type Library Source (Interface Definition Language File) (Microsoft)
    ODL                  Object Definition Language 
    ODM                  OpenOffice/StarOffice OpenDocument (Ver 2) Global Text Document (Sun Microsystems, Inc.)
    ODM                  Opendisc Software (Opendisc)
    ODP                  OpenOffice/StarOffice OpenDocument (Ver 2) Presentation (Sun Microsystems, Inc.)
    ODP                  Oedipus 
    ODS                  OpenOffice/StarOffice OpenDocument (Ver 2) Spreadsheet (Sun Microsystems, Inc.)
    ODS                  NCompass Labs Internet Script 
    ODS                  Outlook Express Mailbox (Microsoft)
    ODT                  OpenOffice/StarOffice OpenDocument (Ver 2) Text Document (Sun Microsystems, Inc.)
    ODT                  MindRender Virtual Reality Explorer Kit (VREK) Object File Format (Themekit Systems Ltd.)
    ODU                  Odigo User File (Comverse, Inc. a part of Comverse Technology)
    ODZ                  IBM OpenDoc Editor Package File 
    OEB                  Open eBook Document (International Digital Publishing Forum (IDPF))
    OEB                  Outlook Express Backup Wizard 
    OEM                  SB_AWE Keys SB_NT File 
    OEM                  OEM Data Used During Device Install 
    OEM                  Text Listing of Supported OEM Devices for WinNT 
    OF                   LessDATA Video System OneFile File 
    OF                   Audio Utility Tuningwrench Compressed File 
    OF                   MicroImages Outline Font (MicroImages, Inc.)
    OFASTER              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OFC                  Open Financial Connectivity File 
    OFD                  20-20 Office Sales E-magine File (20-20 Technologies)
    OFD                  ObjectView Form Defintion 
    OFD                  OfficeForms Form Data File (Toplevel Computing)
    OFF                  3D Mesh Object File Format 
    OFF                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OFF                  Vector Graphic 
    OFFENSE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OFFICES              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OFM                  OfficeForms Blank Form File (Toplevel Computing)
    OFM                  Embroidery Design File 
    OFM                  PostScript Font Description File 
    OFM                  Font (Adobe)
    OFM                  OmniForm Form File 
    OFN                  Systems Management Server (SMS) Offer Creation Notification (Microsoft Corporation)
    OFN                  Office FileNew File (Microsoft)
    OFR                  OptimFROG Encoded Audio (Florin Ghido)
    OFR                  Systems Management Server (SMS) Offer Description File (Microsoft Corporation)
    OFT                  Outlook Item Template (Microsoft)
    OFX                  Olicom Fax 
    OFX                  Open Financial Exchange File 
    OGF                  OnkoS Graphic File (OnkoS)
    OGF                  Outrage Graphic Rows 
    OGG                  Ogg Vorbis Codec Compressed Multimedia File (Xiph.Org Foundation)
    OGG                  Origin Graphic File (OriginLab)
    OGG                  Salt Lake City Game File 
    OGI                  OpenPBX Gateway Interface File (OpenPBX.org)
    OGL                  Overlay Source Code 
    OGM                  Ogg Vorbis Compressed Video File 
    OGM                  Origin Matrix File (OriginLab)
    OGNC                 Web Page Source Code 
    OGO                  OriginPro Graphic Object File (OriginLab)
    OGS                  Origin Script File (OriginLab)
    OGW                  Origin Worksheet File (OriginLab)
    OGX                  C++ Component Gallery Exported Classes and Resources (Borland Software Corporation)
    OH                   Overhoor ( DrillAssistant) List (Efkasoft)
    OH4                  Overhoor ( DrillAssistant) List (Efkasoft)
    OHC                  OpenHelp Table of Contents File (Borland Software Corporation)
    OHI                  OpenHelp Index (Borland Software Corporation)
    OHL                  OpenHelp Link (Borland Software Corporation)
    OHP                  OpenHelp Project File (Borland Software Corporation)
    OHP                  DOS 7 File 
    OHS                  Bink Outbound History 
    OHSD                 HealthFrame OpenHealth Service Plug-in (Records For Living, Inc.)
    OHW                  Overhoor ( DrillAssistant) List (Efkasoft)
    OID                  Prosa/OM Object Interaction Diagram 
    OIF                  QuickBase Web-sharable Database Interchange File 
    OIK                  Lenné3D Plant Distribution File (Lenné3D GmbH i.G.)
    OIL                  Open Image Library Format 
    OIT                  SleepScan and Ceegraph Offset Index File (Bio-logic Systems Corp.)
    OIX                  Lenné3D Plant Distribution File (Lenné3D GmbH i.G.)
    OJP                  3WayPack (The Three-Mode Company)
    OK                   Dot OK File Digest (John Oberschelp)
    OK                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OKS                  OKScript File (Mike Mardesich)
    OKT                  Oktalyzer Tracker Module 
    OKX                  OKScript File (Mike Mardesich)
    OLB                  Medical Terminology Challenge Data File (Delmar Learning)
    OLB                  PSpice Capture Symbols Models (OrCAD)
    OLB                  Automation Object Library (Microsoft Corporation)
    OLB                  AUTOGEN 
    OLB                  VAX Object Library 
    OLD                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OLD                  Old Version 
    OLDEST               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OLE                  Object Linking and Embedding (OLE) Object 
    OLF                  OyezForms Legal Documents (UK) (Oyez)
    OLI                  Olivetti Text File 
    OLK                  Outlook Address Book (Microsoft)
    OLK                  MS Mail Mailbag Lock 
    OLL                  TrialDirector Object Load List 
    OLLIEFONT            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OLN                  Visual C++ Outline Examples (Microsoft)
    OLR                  Openlab Raw Format (Improvision)
    OLRW                 Openlab Raw Format (Improvision)
    OLT                  Visual C++ Outline Examples (Microsoft)
    OLT                  Orbit Downloader List File (OrbitDownloader.com)
    OLV                  OyezForms Legal Documents (UK) (Oyez)
    OLV                  OnLive! Traveler Sound File (The Digital Space Commons)
    OM                   Omicron Compiler File 
    OMA                  Sony OpenMG Music Format File (Sony Corporation of America)
    OMD                  Prosa/OM Class Model 
    OMF                  Open Media Format File 
    OMG                  OpenMG Jukebox (Sony)
    OMN                  Omnipotent, Inc. Collection File 
    OMO                  Oracle Media Objects File 
    OMS                  Briggs Softworks Order Maven 
    OMS                  Macintosh MP3 Music Format 
    OMS                  Omega Downloader Configuration File 
    OMX                  Ots Media Library (Ots Labs Pty Ltd)
    OMX                  OMAX Make Tool Path File (OMAX Corporation)
    ON                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ON                   Audio Utility Tuningwrench Compressed File 
    OND                  Notes Database (IBM)
    ONE                  docPointer Visual ReadMe File (docPointer)
    ONE                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ONE                  EdgeCAM Licensing Data Ffile (Pathtrace Engineering Systems)
    ONE                  OneNote Note File (Microsoft Corporation)
    ONE                  SEMONE Compressed Archive 
    ONF                  Open Navigation Format File 
    ONLINE               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ONX                  Onyx Graphics Postershop File 
    OO                   Free Pascal Object File for OS/2 Version (FPC Development Team)
    OOM                  Shroom Swap File 
    OOP                  OOP Packer Archive (TonyTown Software Ltd.)
    OP                   Rescue Disk File 
    OP                   Origin Preprocessed Origin C File (OriginLab Corporation)
    OP2                  Musplay 1.72 File 
    OP2                  Outpost 2 Saved Game 
    OP73                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OPA                  Psion OPL Object File (Psion PLC)
    OPC                  Beta 44 Job File 
    OPC                  Office Upgrade Control File (Microsoft)
    OPC                  Omron OPC Client File 
    OPD                  Durango Interferometry Software Document (Diffraction International)
    OPD                  OmniPage Document (ScanSoft, Inc.)
    OPEN                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OPENYOURMIND         Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OPF                  Visual OMP Project File (DNA Software, Inc.)
    OPF                  KEPServerEX Project Configuration File (Kepware Technologies Inc.)
    OPF                  Flip Album File 
    OPF                  Obsidium Project File (Obsidium Software)
    OPF                  Gemstar eBook Publisher Package (Gemstar eBook)
    OPI                  OmniPass Exported User (Softex, Inc.)
    OPJ                  OmniPage Schedule (ScanSoft, Inc.)
    OPJ                  Origin Project File (OriginLab)
    OPJ                  OrCAD Project File (Cadence Design Systems, Inc.)
    OPK                  Origin Origin Pack File (OriginLab)
    OPL                  Psion Organiser Programming Language Source File (Psion PLC)
    OPM                  Quadrox Digital CCTV System Components 
    OPML                 Outline Processor Markup Language 
    OPN                  Strip Poker Deluxe Opponent File (Eagle Rock Enterprises, Inc.)
    OPN                  Court Opinion File (Tennessee Administrative Office of the Courts)
    OPN                  Rebel Strip Poker Opponent File (Rebel Software)
    OPN                  Exact Active Options 
    OPO                  Psion OPL Output Executable File (Psion PLC)
    OPO                  Psion Organiser OPL Code (Psion PLC)
    OPOS                 VirtuaSys Opos Script (VirtuaSys :: Sistemas Virtuais)
    OPP                  IPhoto File (Keronsoft)
    OPS                  Dynamite VSP Object Placement Style (3am Solutions (UK) Limited)
    OPS                  Office Profile Settings File (Microsoft)
    OPT                  Psion Organiser OPL Source (Psion PLC)
    OPT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OPT                  FLEXnet Manager Options File (Macrovision Corporation)
    OPT                  Delphi Project Options (Borland Software Corporation)
    OPT                  pcAnywhere Option Set File 
    OPT                  Business Integrator Option File (Pervasive Software Inc.)
    OPT                  Segue SilkTest Option Set File 
    OPT                  QEMM Optimize 
    OPT                  ProWORX Nxt List Module Option File (Schneider Electric)
    OPT                  Developer Studio File Workspace Options (Microsoft)
    OPT                  Options/Configuration 
    OPTIONAL             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OPTNS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OPUS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OPW                  OrgPlus Organization Chart (HumanConcepts, LLC)
    OPX                  Flip Album File 
    OPX                  OrgChart Organization Diagram (Microsoft Corporation)
    OPX                  OrgPlus Organization Chart (HumanConcepts, LLC)
    OPX                  Psion Organiser Programming Language (Psion PLC)
    OPX                  Exact Inactive Options 
    OPX                  OPL Extension DLL 
    OQY                  Excel OLAP Query File (Microsoft)
    OR                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OR2                  Organizer Ver 2 File (IBM)
    OR3                  Organizer 97 File (IBM)
    OR4                  Organizer (IBM)
    OR5                  Organizer (IBM)
    OR6                  Organizer Data File (IBM)
    OR8                  Rational Rose Oracle8 Database Model Report (IBM)
    ORA                  Oracle 7 Configuration 
    ORC                  cSound Synthesis Instruments (Csounders)
    ORC                  Digital Orchestrator Plus Sound File (Voyetra Turtle Beach, Inc.)
    ORC                  Oracle 7 Script 
    ORC                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ORCA                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ORD                  BC45 Owl File 
    ORD                  OMAX Make Tool Path File (OMAX Corporation)
    ORE                  3WayPack (The Three-Mode Company)
    ORF                  Descent 3 Outrage Room Format 
    ORF                  Olympus Digital Camera Raw Image File 
    ORG                  Organiser (IBM)
    ORG                  Odds Ratio Generator Data File (Psytek Ltd.)
    ORG                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ORG                  Yamaha Organ Flutes (Yamaha Corporation of America)
    ORG                  Netmino File 
    ORG                  Hollywood FX Effects Organizer File (Pinnacle Systems, Inc.)
    ORG                  Microcal Origin Plot 
    ORG                  Origin File 
    ORGAN                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ORI                  Need for Speed-Hot Pursuit 2 File 
    ORI                  Original 
    ORIG                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    ORIG                 Gen Original File 
    ORIGINAL             Netobjects Fusion Components BBScomp Message File 
    ORO                  3WayPack (The Three-Mode Company)
    ORV                  Oracom Video Format (Oracom Co. Ltd.)
    OS                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OS2                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OS3                  OS/2 Warp 3 File 
    OSA                  W32/Sober-AD 
    OSC                  OptionScope Worksheet (Equis International)
    OSC                  Formula One 2001 Options File 
    OSC                  Fruitypro Samples SS_effects File 
    OSC                  Remote Installation Services Client Installation Wizard File (Microsoft)
    OSD                  Open Software Description File (Microsoft)
    OSD                  SoftGrid Program Description (Softricity, Inc.)
    OSF                  ICZ OriSAFE 
    OSF                  Descent 3 Sound File (Outrage Entertainment)
    OSF                  Distribution Bin File 
    OSF                  Opera Skin (Opera Software ASA)
    OSG                  OpenSceneGraph ASCII Native Format (OSG Community)
    OSI                  Osiris Database (The Shmoo Group)
    OSI                  SunNet OSI Description File (Sun)
    OSL                  OckamSoft Log File (Ockam Instruments, Inc.)
    OSMALLER             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OSS                  Office Saved Search (Microsoft)
    OST                  Exchange Offline File (Microsoft)
    OST                  Outlook Offline Storage File (Microsoft Corporation)
    OST                  Tivoli Storage Manager File (IBM)
    OSTANDARD            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OTB                  Nokia Phone Graphics Editor (Nokia)
    OTB                  On-the-air Bitmap 
    OTC                  OpenOffice/StarOffice OpenDocument (Ver 2) Chart Template (Sun Microsystems, Inc.)
    OTF                  Open Type Font Format 
    OTF                  Blu-ray Text Subtitle Font Information (Blu-ray Disc Association)
    OTF                  OnkoS Text Data File (OnkoS)
    OTF                  OpenOffice/StarOffice OpenDocument (Ver 2) Formula Template (Sun Microsystems, Inc.)
    OTG                  OpenOffice/StarOffice OpenDocument (Ver 2) Graphics Document Template (Sun Microsystems, Inc.)
    OTH                  Origin Graph Properties File (OriginLab Corporation)
    OTH                  OpenOffice/StarOffice OpenDocument (Ver 2) HTML Text Template (Sun Microsystems, Inc.)
    OTH                  Power Toys Other Folder Explorer Tool 
    OTHERFOLDER          Power Toys Other Folder Explorer Tool 
    OTHERS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OTI                  OpenOffice/StarOffice OpenDocument (Ver 2) Image Template (Sun Microsystems, Inc.)
    OTL                  Fookes Super NoteTab Template File 
    OTL                  NoteTab Outline 
    OTL                  Ventura Publisher's Type Foundry Outline Editor format 
    OTL                  Z-Soft Type Foundry Font 
    OTL                  OpenTrap Event Log 
    OTM                  IMSI Micro Cookbook File (IMSI)
    OTM                  Outlook VBA Module (Microsoft)
    OTM                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OTM                  Origin Matrix Template File (OriginLab)
    OTM2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OTN                  OmniPage Training File (ScanSoft, Inc.)
    OTP                  Origin Graphic Template File (OriginLab)
    OTP                  Top Secret Crypto Gold One Time Pad Key File (TAN$TAAFL Software Company)
    OTP                  OpenOffice/StarOffice OpenDocument (Ver 2) Presentation Template (Sun Microsystems, Inc.)
    OTRKEY               OnlineTvRecorder Encoded Video (OnlineTvRecorder.com)
    OTS                  OtsJuke Sound File 
    OTS                  OpenOffice/StarOffice OpenDocument (Ver 2) Spreadsheet Template (Sun Microsystems, Inc.)
    OTS                  PSR-9000 (Yamaha Corporation of America)
    OTT                  Nokia Phone Ring Tone (Nokia)
    OTT                  OpenOffice/StarOffice OpenDocument (Ver 2) Text Document Template (Sun Microsystems, Inc.)
    OTW                  Origin Worksheet Template File (OriginLab)
    OTX                  Olivetti Olitext Plus File 
    OU!                  OzWin CompuServe E-mail/Forum Access Outbound Messages Online 
    OUT                  Outlines or Output File 
    OUT                  OzWin CompuServe E-mail/Forum Access Outbound Messages 
    OUT                  NovaBACKUP Backup Set (NovaStor Corporation)
    OUT                  Pro/ENGINEER Temporary File (PTC)
    OUT                  MyInfo Documnet/Outline (Milenix Software)
    OUT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OUT                  3WayPack (The Three-Mode Company)
    OUTPUT               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OUV                  Sphinx Open Ended Responses (Le Sphinx Développement)
    OV                   OpenInsight Database Overflow File (Revelation Software)
    OV                   OrgPlus OrgViewer File (HumanConcepts, LLC)
    OV1                  Program File - Overlay 
    OV2                  Program File - Overlay 
    OV2                  TomTom Point of Interest File (TomTom International BV)
    OV3                  Program File - Overlay 
    OVA                  Web Studio Project Page (Back To The Beach Software, LLC)
    OVA                  Sierra Web Studio Graphic 
    OVB                  OnkoS Vocabulary Backup File (OnkoS)
    OVD                  ObjectVision Datafile (Application) (Borland Software Corporation)
    OVE                  Cakewalk Overture Score (Cakewalk)
    OVER                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OVL                  Program File - Overlay 
    OVL                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OVL                  RollerCoaster Tycoon Program Sprite (Atari)
    OVLDEF               Gestionnaire d'extensions Overload Definition (Définition de surcharge) (Francis Gagné)
    OVM                  BHV Tonstudio Platin File 
    OVR                  Program File - Overlay 
    OVW                  Cubase .WAV File Image (Steinberg)
    OVW                  DIANA Overview File 
    OVW                  Cool Edit Pro Overview File 
    OVX                  Psion Oval Control (Psion PLC)
    OW                   Free Pascal Object File for Windows Version (FPC Development Team)
    OWD                  OpenWire Project 
    OWF                  OnkoS Wave Data File (OnkoS)
    OWL                  BC45 Owl File 
    OWLING               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    OWS                  Web Studio Project File (Back To The Beach Software, LLC)
    OX                   Ox Object-Oriented Matrix Programming Language File 
    OXB                  OlapX Report Portfolio (OlapX Software)
    OXC                  OlapX Cube Definition File (OlapX Software)
    OXL                  OMEGA Product Suite File 
    OXO                  Ox Object-Oriented Matrix Programming Language File 
    OXO                  DB-MAIN Compiled Voyager-2 Program 
    OXV                  OlapX Local View File (OlapX Software)
    OYB                  Organize Your Beanie Babies Data File 
    OYZ                  Lotus Approach Alternate dBASE Index 
    OZF                  Mozart Programming System Functor (Mozart Board)
    OZF2                 OziExplorer Map (Des & Lorraine Newman)
    OZFX3                OziExplorer Image File (Des & Lorraine Newman)
    OZM                  Sharp Organizer Memo Bank 
    OZP                  Sharp Organizer Telephone Bank 
    P                    ZX81 Image File 
    P                    Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    P                    Applause Picture 
    P                    Cliq Accessories Datebook Permissions 
    P                    PASCAL Program File 
    P                    Progress Database Procedure File (Progress Software Corporation)
    P                    ReaGeniX Code Generator Application Parameter File 
    P00                  C64 Emulator Program File 
    P01                  Parity Volume Set 
    P01                  Toast CD Image 
    P02                  Parity Volume Set 
    P03                  Parity Volume Set 
    P04                  Parity Volume Set 
    P05                  Parity Volume Set 
    P06                  Parity Volume Set 
    P07                  Parity Volume Set 
    P08                  Parity Volume Set 
    P09                  Parity Volume Set 
    P1                   MicroImages Print Driver File (MicroImages, Inc.)
    P10                  Tektronix Plot 10 Drawing 
    P10                  PKCS #7 Certificate 
    P10                  Certificate Request 
    P12                  Personal Information Exchange File 
    P16                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    P16                  ProTracker Studio Music 
    P1C                  IBM Voice Type Language Newuser File 
    P1I                  IBM Voice Type Language Newuser File 
    P2000                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    P22                  Patch22 Patch File 
    P2A                  IBM Voice Type Language Vocabels File 
    P2B                  PeerGuardian 2 IP Blocking List (Phoenix Labs)
    P2C                  IBM Voice Type Language Newuser File 
    P2C                  pic2clik Photo Album 
    P2D                  PointLineCAD 2D Drawing - 2D Document (Point Line, Inc.)
    P2I                  IBM Voice Type Language Newuser File 
    P2L                  PointLineCAD 2D Drawing Layer (Point Line, Inc.)
    P2M                  Peer2Mail Archive Part and Definition (SoftBulldog)
    P2M                  GameShark Hexidecimal Code Storage (Mad Catz, Inc.)
    P2P                  Picture to Part Data File (GraphiTech Ltd.)
    P2P                  PeerGuardian IP Blocking List (Phoenix Labs)
    P3                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    P3                   Primavera Project Planner Project File (Primavera Systems, Inc.)
    P34                  P34 Lotto File 
    P38                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    P3C                  IBM Voice Type Language New User File 
    P3C                  Primavera Project Planner Compressed Backup (Primavera Systems, Inc.)
    P3D                  Operation Flashpoint Game Model (Bohemia Interactive and The Codemasters Software Company Limited)
    P3D                  3D Graphic 
    P3D                  PointLineCAD 3D Drawing - 3D Document Manager File (Point Line, Inc.)
    P3E                  PC-Doctor File 
    P3I                  PC-Doctor File 
    P3L                  PointLineCAD 3D Drawing Layer (Point Line, Inc.)
    P3P                  Platform for Privacy Preferences 
    P40                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    P4200                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    P56                  Patch 
    P5W                  Ressource 
    P64                  H.261 Encoded Video File 
    P64                  Picasso 64 
    P65                  PageMaker Version 6.5 File (Adobe)
    P7                   XV Visual Schnauzer Thumbnail Format 
    P7B                  PKCS #7 Certificate 
    P7C                  PKCS #7 Certificate 
    P7M                  PKCS #7 MIME Message 
    P7R                  PKCS #7 Certificate 
    P7S                  PKCS #7 Signature 
    P8                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    P83                  GAEB DA 2000 File (Bundesamt für Bauwesen und Raumordnung)
    PA                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PA                   Print Artist Project (Nova Development Corporation)
    PA1                  PageAhead Worktable 
    PA2                  STABCAL (stability calculation for aqueous systems) File 
    PA3                  Turbo Pascal DOS File (Borland Software Corporation)
    PA3                  STABCAL (stability calculation for aqueous systems) File 
    PA4                  Turbo Pascal DOS File (Borland Software Corporation)
    PA5                  Turbo Pascal DOS File (Borland Software Corporation)
    PA5                  STABCAL (stability calculation for aqueous systems) File 
    PAA                  Personal Ancestral File (Corporation of the President of The Church of Jesus Christ of Latter-day Saints)
    PAB                  Personal Address Book (Microsoft Corporation)
    PAC                  Windows Applications Manager Added or Changed Package 
    PAC                  LPAC Lossless Compressed Audio (Tilman Liebchen)
    PAC                  Desperados: Wanted D|A Archive 
    PAC                  Photo-CD Multi-resolution Image File 
    PAC                  Pacman File 
    PAC                  Sound Blaster Studio II Package 
    PAC                  STAD Graphic 
    PAC                  Win2020 / Polistream Video Subtitle Data (Screen Subtitling Systems Ltd.)
    PAC                  SBStudio Song + Sound Package 
    PAC                  CrossePAC Compressed Archive (Digital Strategies, Inc.)
    PACK                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PACKAGE              Autopackage Software Installation Package (Autopackage)
    PACKAGE              The Sims 2 DBPF Resource Package (Electronic Arts (Maxis))
    PAD                  Scanner Output 
    PAD                  Telemate Keypad Definition 
    PAD                  Top Secret Crypto Gold True One Time Pad File (TAN$TAAFL Software Company)
    PAD                  MicroSim PCBoard PADS-compatible Netlist File 
    PAD                  Digital Paper Description 
    PAD                  Visual Expert Source Code Documentation 
    PAE                  PowerArchiver Encrypted Archive (ConeXware, Inc.)
    PAE                  Projet Personal AVI Editor 
    PAF                  Personal Ancestral File Family Database (Corporation of the President of The Church of Jesus Christ of Latter-day Saints)
    PAF                  Ensoniq PARIS Audio Format (E-MU Systems)
    PAG                  Psion Organiser Pager File (Psion PLC)
    PAG                  RealTick Page (Townsend Analytics, Ltd.)
    PAG                  Visual Basic Property Page File (Microsoft Corporation)
    PAG                  Lmsoft-Hypermedia Page 
    PAG                  ActivePerl Pro Studio Hash Database (ActiveState, a division of Sophos)
    PAGE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAGE1                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAGE10               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAGE11               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAGE12               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAGE2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAGE3                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAGE4                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAGE5                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAGE6                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAGE7                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAGE8                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAGE9                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAINT                MacPaint Bitmap Graphic 
    PAJ                  Pajek Project File 
    PAK                  Lands of Lore Archive 
    PAK                  Quake 2 Archive 
    PAK                  Quake Archive 
    PAK                  Trickstyle Archive 
    PAK                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAK                  Hexen 2 Archive 
    PAK                  Heretic 2 Archive 
    PAK                  Hand of Fate Archive 
    PAK                  Daikatana 
    PAK                  Quake Archive (ID Software)
    PAK                  Infonie File 
    PAK                  Compressed Archive File 
    PAK                  Dune 2 Archive 
    PAK                  Final Fantasy VIII (Square Enix Co., Ltd)
    PAK                  Half Life Archive 
    PAL                  Rise of Nations Campaign Media File (Microsoft)
    PAL                  ZX Spectrum-Emulator 
    PAL                  Tree Professional Palm Creator File 
    PAL                  Protos Process Model (Pallas Athena)
    PAL                  Personal Ancestral File 
    PAL                  PageMaker Library Palette (Adobe)
    PAL                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAL                  Dr. Halo Palette 
    PAL                  Color Palette (Microsoft)
    PAL                  Command & Conquer File 
    PAL                  Compressed File 
    PAL                  Lotus Freelance 
    PAL                  Harvard Graphics Palette (Harvard Graphics)
    PAL1                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAL2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PALM                 Programming 
    PALM                 Palm Pixmap 
    PAM                  PAM Property Appraisal Manager Project (PAM Software (UK) Ltd)
    PAM                  Tonline Ob4hbci Smartupdate File 
    PAM                  Personal Ancestral File (Corporation of the President of The Church of Jesus Christ of Latter-day Saints)
    PAN                  Panorama Database (ProVUE Development)
    PAN                  Chromeleon Control Panel 
    PAN                  The Panorama Factory Project (Smoky City Design, LLC)
    PAN                  Infinite Pictures, Inc. iMove Viewer 
    PAN                  CorelDraw Printer-specific File 
    PAN                  SmoothMove Pan Viewer 
    PANDO                Pando Package (Pando Networks)
    PANEL                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PANI                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PANO                 Quicktime VR Panorama File (Apple Inc.)
    PAP                  Tree Professional Palm Creator Image File 
    PAP                  Corel Painter Pattern, Selection or Texture File 
    PAQ                  Quantum Archive 
    PAQ                  HP System Recovery File (Hewlett-Packard Development Company, L.P.)
    PAQ6                 PAQ Data Compression Program Archive 
    PAQ7                 PAQ Data Compression Program Archive 
    PAQ8                 PAQ Data Compression Program Archive 
    PAQ8JC               PAQ PAQ8JC Compressed Archive 
    PAR                  Solid Edge Document (UGS PLM Solutions)
    PAR                  SAP Enterprise Portal Component 
    PAR                  FMS Flight Characteristics File (Roman & Michael Möller)
    PAR                  Digitalk Parts Application 
    PAR                  SolidEdge 3D CAD Model 
    PAR                  Windows 3.x Permanent Output (Swap) File 
    PAR                  SmartPar 
    PAR                  Pov-ray for Windows Parameter File (Persistence of Vision Raytracer Pty Ltd.)
    PAR                  Parity Volume Sets (.P01,...) 
    PAR                  Model Parameter File 
    PAR                  Mirror Archive Restoration File 
    PAR                  ImageZoom Parameter File (Scalado)
    PAR                  Fractint Parameter File 
    PAR2                 Parity Archive Volume Set 
    PARACOUNT            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PARAM                Kosima Scripts File 
    PARAMS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PARMS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PARMS3               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PARSER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PART                 eMule Plus Partial Download (eMule Team)
    PART                 Partial Download 
    PART                 eDonkey Partial Download (MetaMachine)
    PART                 Partial Go!Zilla Download 
    PART1                Netobjects Fusion Netobjects System File 
    PART1                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PART1OF2             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PART2                Netobjects Fusion Netobjects System File 
    PART2OF2             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PARTIAL              Partial Download 
    PARTY                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAS                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAS                  C++ Builder 6 Source (Borland Software Corporation)
    PAS                  Delphi Source (Borland Software Corporation)
    PAS                  Pascal Source Code 
    PASCAL               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PASM                 Parrot Source Code (The Perl Foundation)
    PASS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PASS                 Quiz Editor Password Protected File (Minimouse)
    PASSAGE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PASSERS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAT                  Palace Server Rooms Configuration File (Communities.com)
    PAT                  Pattern Maker Cross-Stitch Pattern File (HobbyWare, Inc.)
    PAT                  Corel Vector Pattern (Corel)
    PAT                  AutoCAD Hatch Pattern (Autodesk, Inc.)
    PAT                  ArcView Geocoding Pattern Recognition File (ESRI)
    PAT                  Arcinfo Coverage 
    PAT                  Advanced Gravis Ultrasound Patch File 
    PAT                  MultiMate 3.3 
    PAT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAT                  Smash Simulator Test Pattern 
    PAT                  Fast-Talk Phonetic Audio Track (Fast-Talk Communications, Inc.)
    PAT                  Train Simulator Track Path (Microsoft Corporation)
    PAT                  PCStitch Pattern (M&R Technologies, Inc.)
    PAT                  File Investigator Pattern File (RobWare)
    PAT                  DART Pro 98 Pattern 
    PAT                  Orchida Embroidery System (Orchida Soft)
    PAT                  Patch File 
    PAT                  CADTERNS Sloper File (CADTERNS Custom Clothing Inc.)
    PAT                  Polygon Attribute Table 
    PAT                  Paint Shop Pro Pattern (Corel Corporation)
    PAT                  US Patent and Trademark Office Bitmap 
    PAT                  Hatch Patterns Photostyler 
    PAT                  Vector Pattern 
    PAT                  WarHack Warcraft2 exePatch Utility 
    PAT                  PatternSmith Cutting Pattern File (Autometrix)
    PAT                  Photoshop Pattern File (Adobe)
    PAT                  Forte Technologies Patch File 
    PAT                  GIMP Pattern (The GIMP Team)
    PAT                  Sound 
    PATCH                General Patch File 
    PATCH                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PATCHDOCS            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PATCHER              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PATH                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PATHNAME             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PATHS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PATTERN              Photoline5 Defaults File 
    PAU                  OzWin CompuServe E-mail/Forum Access Paused Message 
    PAUSE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAV                  Panda Antivirus File 
    PAV                  MediaMatrix Peavey Architectural View (Peavey Electronics)
    PAV????              Panda Antivirus File 
    PAX                  Pick Ax Secure Graphic File (Smaller Animals Software, Inc.)
    PAX                  Project A.R.S.E Brain File 
    PAX                  Personal Ancestral File (Corporation of the President of The Church of Jesus Christ of Latter-day Saints)
    PAY                  PriorityRate (Priority Data Sytems)
    PAYMENTS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAYROLL              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PAZ                  Project A.R.S.E Brain File 
    PB                   FAXability Fax 
    PB                   PocketCache Backup File (Kadena Systems, Inc.)
    PB                   PixBase Setup 
    PB                   PureBasic Source Code (Fantaisie Software)
    PB                   PowerBASIC Configuration (PowerBASIC, Inc.)
    PB                   WinFax Pro Phone Book (Symantec)
    PB                   PuffBOMB Extra Level File (Sykhronics)
    PB                   Proboard Configuration 
    PB                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PB1                  First Publisher Document 
    PB2                  STABCAL (stability calculation for aqueous systems) File 
    PB2                  OrindaBuild Configuration (Orinda Software Ltd)
    PBA                  Process Panel Builder (ABB Ltd)
    PBA                  PowerBASIC Source Code (PowerBASIC, Inc.)
    PBB                  MS Mail Address Information File 
    PBC                  Parrot Compiled Byte Code File (The Perl Foundation)
    PBD                  Palm Database (Palm, Inc.)
    PBD                  Epicentric Portal Server Module Java Bean (Vignette Corporation)
    PBD                  iSilo Document (DC & Co.)
    PBD                  Graphic Format (Adobe)
    PBD                  Faxit Phone Book 
    PBD                  PowerBuilder Dynamic Library 
    PBD                  Axialis Image 
    PBF                  Turtle Beach Pinnacle Bank File 
    PBF                  Sprite Backup Mobile Device Image (Sprite Software)
    PBF                  Portable Bitmap Format File 
    PBF                  PBook E-book Format 
    PBF                  Grand Prix Legends BMAP (Bitmap) File 
    PBF                  Paragon Disk Image 
    PBH                  PowerBASIC Help File (PowerBASIC, Inc.)
    PBI                  Profiler Binary Input (Microsoft)
    PBI                  PowerBASIC Include File (PowerBASIC, Inc.)
    PBI                  Axialis Image JPEG JFIF 
    PBI                  PC-BSD Setup File (PC-BSD Software)
    PBK                  HP-95LX Phone Book File 
    PBK                  Phonebook (Microsoft)
    PBK                  ElectraSoft Fax Phone Book 
    PBL                  Publitor Data File (Soho Soho Ltd.)
    PBL                  PowerBASIC Library (PowerBASIC, Inc.)
    PBL                  PowerBuilder Source Code Library (Sybase)
    PBM                  UNIX Portable Bitmap Graphic 
    PBMP                 Pilot Bitmap 
    PBMV                 Portable Bitmap File 
    PBN                  Portable Bridge Notation 
    PBO                  Profiler Binary Output 
    PBO                  Operation Flashpoint Game Mission Folder (Bohemia Interactive and The Codemasters Software Company Limited)
    PBOB                 Vbox Preview (Aladdin Knowledge Systems)
    PBP                  Phoenix Visual Designer Project (Prometheus Software)
    PBP                  PlaybackPro Video Database (DTvideolabs)
    PBP                  Perl Builder File 
    PBP                  PSP Firmware Image (Sony Corporation of America)
    PBPROJ               Project Builder Project (Apple Computer, Inc.)
    PBQ                  PunkBuster Query Script 
    PBQ                  Audio 
    PBR                  BI/Suite Hypercube Presentation 
    PBR                  PowerBuilder Resource 
    PBR                  PowerBuilder Resource File 
    PBR                  PowerBASIC Resource File (PowerBASIC, Inc.)
    PBR                  Publisher Autosave/Autorecovery/Backup File (Microsoft)
    PBR                  BorderManager 3.5 Support Pack 3 File 
    PBS                  PowerArchiver Backup Script 
    PBS                  Paint Shop Pro Brush Stroke Preset (Corel Corporation)
    PBT                  Micro Dynamics MARS 
    PBT                  Profiler Binary Table 
    PBTPL                PowerBASIC Template (PowerBASIC, Inc.)
    PBV                  Paint Shop Pro Bevel Preset (Corel Corporation)
    PBX                  Outlook Express Message Folder 
    PC                   Personal Composer Musical Notation (Personal Composer Inc.)
    PC                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PC                   Oracle Pro*C Source Code (Oracle)
    PC                   Sierra Print Artist Postcard 
    PC                   Postcard 
    PC                   PC-specific Text File 
    PC1                  Degas & Degas Elite 
    PC2                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PC2                  PrivateChat! 
    PC2                  STABCAL (stability calculation for aqueous systems) File 
    PC2                  AutoCAD R14 Plotter Configuration (Autodesk, Inc.)
    PC2                  Degas & Degas Elite 
    PC3                  Harvard Graphics Custom Palette (Harvard Graphics)
    PC3                  Degas & Degas Elite 
    PC3                  AutoCAD R2000 Plotter Configuration (Autodesk, Inc.)
    PC3                  STABCAL (stability calculation for aqueous systems) File 
    PC5                  Rockwell Software Logix 5 File 
    PC5                  STABCAL (stability calculation for aqueous systems) File 
    PC8                  NewWave Write ASCII IBM8 Character Set 
    PC8                  ASCII Text IBM-8 
    PC?                  2D Graphic 
    PCA                  MicroSim PCBoard Layout Database File 
    PCA                  PCAnywhere Registry Backup 
    PCA                  Sonic Foundry Perfect Clarity Audio File (Sony Corporation of America)
    PCAP                 Ethernet Packet Sniffer File 
    PCB                  ExpressPCB Printed Circuit Board Layout (ExpressPCB)
    PCB                  PCB Layout Check Design Rule Template (Top Tec PCB Ltd.)
    PCB                  Protel Technology Advanced PCB Design 
    PCB                  PowerPoint Application Data File (Microsoft)
    PCB                  Print Shop Deluxe Business Card 
    PCB                  PC-Doctor File 
    PCB                  P-CAD Database (Altium Limited)
    PCB                  PADS Printed Circuit Board (Mentor Graphics Corp.)
    PCB                  Ivex Winboard Design File 
    PCB                  Broderbund Print Shop Business Card 
    PCB                  ACCEL or Protel Printed Circuit Board File 
    PCC                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PCC                  VideoCraft GIF 
    PCC                  PC Paintbrush Image File 
    PCC                  PC Paintbrush Cutout Picture Vector Graphic (Zsoft)
    PCC                  PC Checkup System Information 
    PCD                  GX IEC Developer Backup File (Mitsubishi Electric Corporation)
    PCD                  Images CD Creator Corel Adaptec 
    PCD                  Visual Test (Microsoft)
    PCD                  Photo-CD Image (Kodak)
    PCD                  P-Code Compiled Test Scripts 
    PCDS                 Photo-CD Image 
    PCE                  Borland Package Collection Editor File (Borland Software Corporation)
    PCE                  PC Engine 
    PCE                  Mail Signature 
    PCE                  Print Shop Envelope 
    PCE                  PC-Doctor File 
    PCE                  Eudora Mailbox Name Map 
    PCE-LIST             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PCF                  Cisco VPN Client Configuration (Cisco Systems, Inc.)
    PCF                  Process Charter Data (Scitor Corporation)
    PCF                  PowerToy Calculator Saved Functions and Variables (Microsoft Corporation)
    PCF                  Systems Management Server (SMS) Patron Configuration File (Microsoft Corporation)
    PCF                  DD PrintCenter Document (Imagetech Corporation, Ltd.)
    PCF                  UNIX Font File 
    PCF                  Paint Shop Pro Colored Foil Preset (Corel Corporation)
    PCF                  Profiler Command File 
    PCF                  HP-95LX Printer Configuration File 
    PCF                  Pro/ENGINEER (PTC)
    PCF                  Calyx Point 
    PCG                  Photo CD Graphic File 
    PCG                  TRITON Le or Karma Music Workstation Ver. 1.x and 2.x Synthesizer File (KORG)
    PCH                  Pixeur Color History File (Veign)
    PCH                  Patch File 
    PCH                  Visual C PreCompiled Header (Microsoft Corporation)
    PCH2                 Nord Modular G2 Patch (Clavia DMI AB)
    PCI                  PC-Doctor File 
    PCI                  Windows PCI Miniport file (Microsoft)
    PCI1                 WinControl DCX-AT3000 Motion Control Command Language File (Precision MicroControl Corp.)
    PCI3                 WinControl DCX-AT3000 Motion Control Command Language File (Precision MicroControl Corp.)
    PCJ                  IBM Linkaway-Live Multimedia Tool Graphics 
    PCK                  Turbo Pascal Pick File (Borland Software Corporation)
    PCK                  Package 
    PCK                  Commandos II File 
    PCK                  ARIA PIPP  Spectrum 
    PCK                  Systems Management Server (SMS) Received Package File (Microsoft Corporation)
    PCL                  2D Graphic Page Control Language 
    PCL                  HP Printer Control Language (Hewlett-Packard Development Company, L.P.)
    PCLF                 PC MightyMax Offline Licensing File (PC MightyMax, Inc.)
    PCM                  Sound File 
    PCM                  OKI MSM6376 Synth Chip PCM Format 
    PCM                  LaserJet Printer Cartridge Metric (Hewlett-Packard Development Company, L.P.)
    PCM                  Image Photoline 
    PCM                  Embroidery Design File 
    PCM                  Bliss for Windows (Handicom)
    PCN                  Paint Shop Pro Contour Preset (Corel Corporation)
    PCO                  PC-Outline Outline 
    PCP                  Symantec Live Update Pro (Symantec Corporation)
    PCP                  PC Paint (DOS) Bitmap Image 
    PCP                  AutoCAD R13 and Prior Plotter Configuration (Autodesk, Inc.)
    PCP                  RCF EnCoder/DeCoder Encoded File (RCF Group - Reborn Code Freaks)
    PCP                  Windows Installer Database Editor (Orca) Patch Creation Properties File (Microsoft Corporation)
    PCR                  Creatacard Post Card Project (Broderbund)
    PCR                  PCMark Benchmark File (Futuremark Corp.)
    PCS                  PICS Animation 
    PCS                  Pfaff Home Embroidery Format (Pfaff)
    PCS                  Painter4 PICT File 
    PCS                  Yamaha Piano Combo Style (Yamaha Corporation of America)
    PCS                  Apple Macintosh PICT File Animation (Apple Computer, Inc.)
    PCS                  PowerPoint ClipArt Gallery Picture Storage (Microsoft)
    PCS                  Ultra Zip Password Cracker (UZPC) 
    PCT                  Bitmap Graphic 
    PCT                  Clipart ClarisWorks 
    PCT                  Honeywell GUS Display Builder 
    PCT                  Macintosh Quickdraw/PICT Drawing 
    PCT                  NIST IHDR 
    PCT                  PC Paint Bitmap 
    PCU                  XProfan Compiled Unit (Roland G. Hülsmann)
    PCU                  PC MightyMax Undo File (PC MightyMax, Inc.)
    PCV                  MozBackup Backup (Pavel Cvrcek)
    PCV                  PC Upgrade Commander (V Communications, Inc.)
    PCV                  Move Me Moving Van (Spearit Software)
    PCW                  PC Write Text File 
    PCX                  PC Paintbrush Bitmap Graphic 
    PD                   Pure Data Patchfile (PD Community)
    PD                   Paradox Table 
    PD                   Perl Data Language PreProcessor File 
    PD                   Male MRI 
    PD1                  DFSee Partition Table and Boot Sector Backup (Fsys Software)
    PD2                  DFSee Partition Table and Boot Sector Backup (Fsys Software)
    PD3                  ICDD X-ray Powder Diffraction Patterns and Digitized Diffractogram 
    PD3                  Denso BHT-7000 Terminal Executable Program File 
    PD3                  DFSee Partition Table and Boot Sector Backup (Fsys Software)
    PD3                  PodFlow (Ikon AVS Ltd.)
    PD3                  3ds Max (Autodesk, Inc.)
    PD4                  DFSee Partition Table and Boot Sector Backup (Fsys Software)
    PD5                  DFSee Partition Table and Boot Sector Backup (Fsys Software)
    PDA                  Print Shop Bitmap Graphic 
    PDAS                 PDAStore Store (Harald-René Flasch)
    PDB                  Protein Databank File 
    PDB                  TealPaint 
    PDB                  Tact File 
    PDB                  Pegasus DataBase 
    PDB                  Powerproject Teamplan 
    PDB                  Ribbons Data 
    PDB                  PhotoDeluxe Image (Adobe)
    PDB                  QuickPOS Database File 
    PDB                  Visual C++/.NET Program Database File (Microsoft Corporation)
    PDB                  PowerDesigner Physical Model Backup 
    PDB                  PowerBuilder Dynamic Library 
    PDB                  Pilot Image Format 
    PDB                  Photo Deluxe Image 
    PDB                  Palmpilot Database/Document File (Palm, Inc.)
    PDB                  MolMol Molecule Description File 
    PDB                  PowerBASIC Debugger Symbols (PowerBASIC, Inc.)
    PDB                  BGBlitz Position Database (Frank Berger)
    PDB                  C64 Emulator File 
    PDB                  MDL Molfile Protein DataBank (MDL Information Systems, Inc.)
    PDB                  Insight II Brookhaven Protein Databand File (Accelrys Software Inc.)
    PDB                  MonkeyCard/MonkeyLogo 
    PDBX                 Insight II X-PLOR Coordinate File (Accelrys Software Inc.)
    PDC                  PowerDivX NextGen Media Player Chapter File 
    PDC                  PReS Print Control 
    PDD                  PhotoDeluxe Image (Adobe)
    PDE                  Principalm Student Information Extract (Discovery Software Ltd.)
    PDES                 PIPE-FLO Professional Design File Template (Engineered Software)
    PDF                  P-CAD Database Interchange Format (Altium Limited)
    PDF                  Package Definition File 
    PDF                  Netware Printer Definition File 
    PDF                  Ventura Publisher EPS-variation Page (Corel Corporation)
    PDF                  Ed-Scan 24-bit Graphic File 
    PDF                  Analyser Protocol Definition 
    PDF                  Acrobat Portable Document Format (Adobe Systems Inc.)
    PDF                  ArcView Preferences Definition File (ESRI)
    PDF                  Systems Management Server (SMS) Package Description File (Microsoft Corporation)
    PDFENX               egis Encrypted PDF File (HiTRUST Inc.)
    PDG                  PrintShop Deluxe File 
    PDG                  Print Designer GOLD Data File (CAM Development)
    PDI                  IBM VideoTex 
    PDI                  PI ProcessBook Display Definition File (OSIsoft, Inc.)
    PDI                  InstantCopy Disc Image (Pinnacle Systems, Inc.)
    PDI                  PowerPoint Import/Export (Microsoft)
    PDI                  PReS Document Creation Subroutines 
    PDL                  SymbianOS Printer Driver 
    PDL                  Programmable Driver Language (Quovadx, Inc.)
    PDL                  C++ Project Description Language (Borland Software Corporation)
    PDL                  Print Shop Project 
    PDL                  WITE32 File 
    PDM                  Sybase Power Designer File 
    PDM                  PowerDivX NextGen Media Player 
    PDN                  Plan de Negocio 
    PDN                  Paint.NET Image 
    PDO                  Pepakura Designer Paper Craft Pattern (Tama Software Ltd.)
    PDO                  Access Package Deployment Script (Microsoft)
    PDP                  Photoshop PDF Format (Adobe)
    PDP                  Broderbund Print Shop Deluxe File 
    PDQ                  Patton & Patton Flowcharting PDQ Lite File 
    PDQ                  BLUEWAVE Sound File 
    PDQ                  PDQ Lite Flowchart 
    PDR                  SymbianOS Printer Resource File 
    PDR                  Port Driver (Microsoft)
    PDS                  NASA Planetary Data System Format Space Mission Data 
    PDS                  Planetary Data System 
    PDS                  Print Shop Graphic 
    PDS                  Source Code File 
    PDS                  PLDASM Source Code 
    PDS                  Telsis HiCall Program File 
    PDS                  VICAR: Video Image Communication and Retrieval Graphic 
    PDS                  IncrediMail (IncrediMail Ltd.)
    PDT                  InkWriter/Note Taker Template (Microsoft)
    PDT                  ProCite Primary Database 
    PDT                  VersaPro Compiled Block 
    PDV                  Paintbrush Printer Driver 
    PDW                  Professional Draw Document 
    PDW                  HiJaak Vector Graphics 
    PDX                  ProCite Database Keys 
    PDX                  Acrobat Catalog Index (Adobe)
    PDX                  PageMaker Printer Description (Adobe)
    PDX                  Mayura Draw 
    PDX                  PDXplorer IPC-2570 Product Data Exchange Information (Active Sensing, Inc.)
    PDX                  Paradox Files 
    PDZ                  GZipped Brookhaven Protein Databank File 
    PD_                  Visc15 Images Setup File 
    PE                   PatentEase File 
    PE                   Microsoft Symbol and Type Information 
    PE                   Portable Executable File 
    PE3                  QuickViewer Image Archive 
    PE3                  PhotoImpact Image Archive (Ulead Systems, Inc.)
    PE4                  Photo Explorer Thumbnail 
    PE4                  PhotoImpact Image Archive (Ulead Systems, Inc.)
    PEA                  P-Encryption Suite Archive (CadabraSoftware, Inc.)
    PEA                  Peach Text File 
    PEB                  WordPerfect Program Editor Bottom Overflow File (Corel Corporation)
    PEBPRJ               PEBundle File 
    PEC                  Brother/Babylock/Bernina Home Embroidery Format 
    PED                  WordPerfect Program Editor Delete Save (Corel Corporation)
    PEEKER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PEEKIT               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PEF                  Pentax RAW Bitmap Graphic (PENTAX Corporation)
    PEG                  Red Faction II Game Data 
    PEH                  Power Email Harvester (YesGoal LLC)
    PEI                  Performer Software Encrypted Image (Performer Software)
    PEK                  Premiere Graphic Waveform Data (Adobe Systems Inc.)
    PEM                  Audio Module 
    PEM                  WordPerfect Program Editor Macro (Corel Corporation)
    PEM                  Privacy Enhanced Mail Security Certificate 
    PEN                  Paint Shop Pro Enamel Preset (Corel Corporation)
    PEN                  io Software Digital Pen Ink File (Logitech, Inc.)
    PEN                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PEO                  Homeworld MESH File (Sierra)
    PEP                  TurboProject Project File 
    PEPPER               Pepper Keeper Shared Package File (Pepper Computer, Inc.)
    PEPSI                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PEQ                  WordPerfect Program Editor Print Queue File (Corel Corporation)
    PER                  Rise of Nations AI File (Microsoft)
    PER                  WordPerfect Program Editor Resident Area (Corel Corporation)
    PERCENTAGES          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PERFORMANCE          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PERL                 Perl Source File 
    PES                  WordPerfect Program Editor Work Space File (Corel Corporation)
    PES                  Brother/Babylock/Bernina Home Embroidery Format 
    PET                  WordPerfect Program Editor Top Overflow File (Corel Corporation)
    PET                  Magic Petz File 
    PET                  Macintosh PICT Image 
    PET                  Custom CD Menu 
    PETERGUN             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PEW                  IAR Embedded Workbench 
    PEX                  PodFlow Pod Export (Ikon AVS Ltd.)
    PEX                  Proboard Executable Program 
    PF                   Aladdin Systems Private Files Encrypted File 
    PF                   Archive 
    PF                   ICC Profile 
    PF                   Prefetch (also called Scenario) File (Microsoft)
    PF                   Progress Database Parameter File (Progress Software Corporation)
    PF                   Monitor or Printer Profile File 
    PF                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PF?                  SmartWare Project Source File 
    PFA                  Formdef or Pagedef Source Code 
    PFA                  PostScript Unhinted Type 3 Font 
    PFA                  Postscript Type 1 Font 
    PFB                  PostScript Type 1 Font (Adobe)
    PFB                  DESQview (Symantec)
    PFC                  PF Component 
    PFC                  AOL Preferences/Favorites/Mail (America Online, Inc.)
    PFC                  New Soft Presto! Page Manager File 
    PFC                  WordPerfect Perfect Fit Filter (Corel Corporation)
    PFC                  First Choice Text File 
    PFD                  ProForm Database Data Entry File (SoftPro)
    PFD                  Process (iGrafx (a division of Corel Inc.))
    PFDB                 Password Fortress Secure Database File (Dunning Software)
    PFDT                 Password Fortress Database Template (Dunning Software)
    PFE                  Programmers File Editor 
    PFF                  Delta Force Archive (NovaLogic, Inc.)
    PFF                  Tachyon Archive 
    PFF                  Armored Fist 3 Archive 
    PFF                  Zoomify Image Stream (Zoomify, Inc.)
    PFF                  Comanche 4 
    PFF                  Formatta Filler Portable Form Files (Formatta Corporation)
    PFF                  F22 Lightning 3 
    PFG                  jEEPers Program File 
    PFK                  XTree Programmable Function Keys 
    PFL                  Lenné3D-Player (Lenné3D-Plantage) Plant File (Lenné3D GmbH)
    PFL                  Quicken Lawyer Data File (Intuit Inc.)
    PFL                  Paint Shop Pro Fine Leather Preset (Corel Corporation)
    PFL                  SmartStation AppsView File 
    PFL                  Freelance Graphics DOS Version 4.0 Portfolio Presentation (IBM)
    PFL                  PhotoFiltre Plug-in (Antonio Da Cruz)
    PFL                  Family Lawyer Data File (Broderbund)
    PFM                  Printer Font Metrics (Adobe)
    PFP                  The Panorama Factory Project 
    PFR                  Paint Shop Pro Frame (Corel Corporation)
    PFR                  WebFont Wizard Portable Font Resource 
    PFS                  BIOCCELERATOR Homologous Sequence Names 
    PFS                  pfs:File Database / pfs:Write Text 
    PFS                  First Publisher ART File 
    PFT                  ChiWriter Printer Font 
    PFW                  Symantec Liveadvisor Shared File (Symantec Corporation)
    PFX                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PFX                  Rapfile 
    PFX                  PFXplus Application Source Code (POWERflex Corporation)
    PFX                  Personal Information Exchange File 
    PFX                  Amiga Pageflipper Plus F/X Animation (Amiga)
    PF_                  Encore Compressed Audio File 
    PG                   2D Graphic 
    PG                   Printfox/Pagefox 
    PG                   IBM LinkWay Cut/Paste File 
    PG1                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PG2                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PG3                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PGA                  IBM Professional Graphics Adapter Image 
    PGA                  PowerGREP Action File (JGsoft - Just Great Software)
    PGA                  Solitaire Aztec Peg Backup 
    PGB                  Rayman2 File 
    PGC                  Portable Gaming Notation (PGN) Compressed Format 
    PGC                  Egypt Solitaire Back 
    PGC                  Compressed Portfolio Graphic 
    PGD                  Pretty Good Privacy (PGP) Virtual Disk File (PGP Corporation)
    PGE                  Solitaire Peg Back 
    PGF                  PowerGREP File Selection (JGsoft - Just Great Software)
    PGF                  PGC Portfolio Graphics Compressed Bitmap 
    PGF                  GPS Pathfinder Office Geoid Grid File (Trinble Navigation Limited)
    PGF                  Progressive Graphics File (xeraina GmbH)
    PGI                  PGraph Library Printer Device Driver 
    PGL                  HP Plotter Language 
    PGL                  PowerGREP Library (JGsoft - Just Great Software)
    PGM                  Signature Program 
    PGM                  Opentech Digital STB Main Software (Opentech, Inc.)
    PGM                  MegaBasic Apple II Program File 
    PGM                  Portable Graymap Graphic 
    PGM                  CGI Program 
    PGM                  RS7000 OS Image (Yahama Corporation)
    PGM                  Chromeleon Batch Program 
    PGMEURS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PGN                  Portable Game Notation (PGN) 
    PGN                  Picatinny Arsenal Electronic Formstore Form in TIFF Format 
    PGP                  AutoCAD Program Parameter (Autodesk, Inc.)
    PGP                  Pretty Good Privacy (PGP) Key/Signature File (PGP Corporation)
    PGP                  Program Parameter 
    PGP                  TBAV File 
    PGR                  Pretty Good Privacy (PGP) PGP Groups (PGP Corporation)
    PGR                  PowerGREP Results File (JGsoft - Just Great Software)
    PGS                  PGSuper 
    PGS                  Man4DOS Manual Page 
    PGS                   DALiM LiTHO Vector Page (Blanchard Systems Inc.)
    PGS                  PageStream Document (Grasshopper LCC)
    PGT                  Sothink Glenda Widget Tools File 
    PGU                  PowerGREP Undo History (JGsoft - Just Great Software)
    PGX                  Visual Basic Binary Property Page File 
    PGZ                  TPDesign ver 1.23 Control System Touch Panel Design (AMX Corporation)
    PGZ                  StreamLync Download Archive (PalmGear)
    PG_                  Improve Compressed Audio File 
    PH                   PERL Header 
    PH                   Optimized Geoworks .GOH File 
    PH                   Help Compiler Phrase Table 
    PH                   MRS-802 Phrase Audio Data (Mono or Stereo Left Channel) (Zoom Corporation)
    PH                   AOL Phone Home Data File (America Online, Inc.)
    PH                   Profan Header File (rgh-soft.de)
    PH3                  PHP Script 
    PH4                  PHP Script 
    PHANTASM             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PHAPACK              PhaPacker Format 
    PHASER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PHB                  PhoneB Phonebook File 
    PHB                  NewLeaf PhraseBook 
    PHB                  ClustaW Tree File 
    PHB                  TreeView File 
    PHB                  PhotoBase (ArcSoft, Inc.)
    PHC                  Home Embroidery Format 
    PHD                  PolyHedra Database 
    PHD                  PC Help Desk File 
    PHE                  WinChess Game Record 
    PHIST                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PHL                  PHP Source File 
    PHL                  Icon Collection File 
    PHL                  Database Configuration File 
    PHM                  DN - Lync Phone Book 
    PHM                  Phorm 
    PHN                  UltraFax/QmodemPro Phone List 
    PHN                  Phormation 
    PHN                  PhoneFree Internet Telephone 
    PHN                  Compuserve Signup File 
    PHO                  Gerber Photoplot File 
    PHO                  Metz Phone Phone List 
    PHP                  Creatacard Quick Prints Project (Broderbund)
    PHP                  PhotoParade Slideshow (Callisto Corporation)
    PHP                  PHP Script (The PHP Group)
    PHP                  Picture It! Publishing Project File (Microsoft)
    PHP                  Presentation 
    PHP3                 PHP Script 
    PHP4                 PHP Script 
    PHP5                 PHP Script 
    PHPS                 PHP Source (The PHP Group)
    PHPT                 PHP: Hypertext Preprocessor Make Test Test Suite (The PHP Group)
    PHPW                 GTK+ Application 
    PHQ                  Handheld Quake Language File 
    PHR                  MRS-802 Phrase Audio Data (Stereo Right Channel) (Zoom Corporation)
    PHR                  NoExcuse Phrase Checker 
    PHR                  LocoScript Phrases (LocoScript Software)
    PHS                  xfit Phase File 
    PHS                  XtalView Phase File (The Computational Center for MacroMolecular Structures (CCMS))
    PHS                  PHScript File 
    PHT                  Partial Hypertext File 
    PHTM                 PHP Script 
    PHTML                Embedded Perl (ePerl) File 
    PHTML                web-iPerl Document 
    PHTML                PHP Script 
    PHW                  PhotoWebber Session File (Media Lab)
    PHX                  PodFlow Firmware Update (Ikon AVS Ltd.)
    PHX                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PHX                  Phoenix Image Transfer Format 
    PHY                  Spicemod Asic File 
    PHY2                 Marathon 2 Game Physics File (Bungie)
    PHYS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PHYSAO               Aleph One Physics Model (Marathon Open Source)
    PHZ                  Point Horizon Spreadsheet with Methods (Logic Scientific)
    PH_                  C Poet Compressed Disk1 File 
    PI                   Extension Associated with W32.Sobig.D@mm Worm 
    PI                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PI                   iPi Pi Calculation Storage (10101 Software)
    PI                   Blazing Paddles 
    PI                   Image Japan PI 
    PI$                  MS Compressed PIF 
    PI1                  Degas & Degas Elite 
    PI2                  Studio2 High Resolution Encrypted Image (Portrait Innovations, Inc.)
    PI2                  Degas & Degas Elite 
    PI2SHR               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PI3                  Degas & Degas Elite 
    PI3SHR               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PI4                  Degas & Degas Elite 
    PI5                  Degas & Degas Elite 
    PI6                  Degas & Degas Elite 
    PIANO                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PIC                  Advanced Art Studio 
    PIC                  Bio-Rad Confocal Image (Bio-Rad)
    PIC                  Delcam Picture File 
    PIC                  Animator PIC/CEL Bitmap (Autodesk, Inc.)
    PIC                  Handy Scaner Cameron Image 
    PIC                  IBM Storyboard Bitmap 
    PIC                  Picture 
    PIC                  Pixar Picture 
    PIC                  Psion Series 3 Bitmap (Psion PLC)
    PIC                  QuickTime Picture (Apple Computer, Inc.)
    PIC                  Radiance Scene Description Image 
    PIC                  MTV & Rayshade Image 
    PIC                  Rayshade Graphic 
    PIC                  SDSC Image Tool Pixar Picture 
    PIC                  Movie BYU File 
    PIC                  SoftImage 3D Image 
    PIC                  Stad File 
    PIC                  Macintosh Quickdraw/PICT Drawing 
    PIC                  Micrografx Draw 
    PIC                  PC Paint Bitmap Graphic 
    PIC                  Lotus Picture (IBM)
    PIC                  Pictor Picture 
    PIC                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PIC                  Softimage 
    PIC1                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PIC2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PIC3                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PIC4                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PIC5                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PIC6                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PIC8                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PICCATA              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PICHAND              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PICIO                Pixar Picture 
    PICON                Personal Icon 
    PICS                 PICT Drawing Sequence 
    PICT                 Macintosh Quickdraw/PICT Drawing 
    PICT1                Macintosh QuickDraw/PICT Image (Apple Computer, Inc.)
    PICT2                Macintosh Quickdraw/PICT Drawing 
    PICTCLIPPING         Macintosh OS X Picture Clipping (Apple Inc.)
    PICTOR               PC Paint Image File 
    PICTURE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PID                  W32/Yayin-A Worm Data File 
    PID                  UNIX Process ID File 
    PIE                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PIE                  Studio2 High Resolution Encrypted Image (Portrait Innovations, Inc.)
    PIECE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PIECES               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PIF                  IBM PIF Drawing 
    PIF                  Macintosh Compressed File Archive 
    PIF                  OS/2 Graphics Metafile 
    PIF                  Windows Program Information File (Microsoft Corporation)
    PIF                  GDF Format Vector Image 
    PIG                  Descent Texture File 
    PIG                  Lucas Arts Dark Forces WAD File 
    PIG                  Ricoh IS30 
    PIG                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PIGG                 City of Heros Virtual Drive Game File (Cryptic Studios)
    PII                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PIL                  LABpilot Trend Data (CMA Microdialysis AB)
    PIL                  ICUpilot Trend Data (CMA Microdialysis AB)
    PILE                 chum Test Results (Piletest.com Limited)
    PIM                  PIMPLE Compressed File (Ilia Muraviev)
    PIM                  PixMaker Project File 
    PIM                  Personal Information Manager File 
    PIM                  Ultimate Draw Pascal Text Mode Image 
    PIM1                 Pinnacle Systems Video File (Pinnacle Systems, Inc.)
    PIN                  ArchiCAD (Graphisoft R&D Software Development Rt.)
    PIN                  Compton's 3D World Atlas 
    PIN                  Atari ST Graphics Format 
    PIN                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PIN                  Links Games Player Information File 
    PIN                  PUDL Coded Split File Particle 
    PIN                  Epic Pinball Data File 
    PIN                  Pushpin Set 
    PINBALL1             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PINBALL2             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PINE                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PIP                  JPEG,JPG,JPE, JFIF,PJPEG Compressed Bitmap Picture 
    PIP                  Office Personalized Menu and Toolbar (Microsoft)
    PIPE                 PIPE-FLO Professional System Model (Engineered Software)
    PIPL                 Photoshop 5.0 SDK Samplecode Colorpicker File (Adobe)
    PIQ                  Piped Technology Information Query (Queensgate Systems Ltd)
    PIQT                 QuickTime Still (Apple Computer, Inc.)
    PIRA                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PISI                 Pardus Linux Package (Scientific & Technological Research Council of Turkey)
    PISTE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PIT                  Macintosh Compressed Archive File 
    PIV                  Pivotal Sales Suite Activity File (http://www.pivotal.com/)
    PIV                  Pivot Stickfigure Animator Cartoon (Peter Bone)
    PIW                  PI ProcessBook Display Definition File (OSIsoft, Inc.)
    PIX                  LUMENA .PIX and .BPX File Formats 
    PIX                  Island Graphics 
    PIX                  SDSC Alias/Wavefront RLE Image 
    PIX                  Truevision Targa Bitmap 
    PIX                  Vort 
    PIX                  PrintMaster Graphic File 
    PIX                  Roland Scanner File 
    PIX                  PABX Background Bitmap 
    PIX                  ESM Software Pix 
    PIX                  Inset Systems Bitmap or Vector Graphic 
    PIX                  Alias PIX Bitmap 
    PIXAR                Pixar Picture 
    PIXELPAINT           PixelPaint Professional Ver. 1.0/2.0 Image File 
    PIZ                  Renamed Zip File 
    PIZ                  Pizzicato Music Score (ARPEGE sprl)
    PIZ                  Pizza Connection 2 Saved Game (Virgin Interactive Entertainment)
    PIZZA                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PI_                  Compressed PIC or PIF File 
    PJ                   SuperProject Project File 
    PJ                   Project64 Game Progress File 
    PJ                   MKS Source Integrity File 
    PJ                   PaintJet PCL Bitmap (Hewlett-Packard)
    PJD                  Portfolio 7 Exported Categories File (Extensis, Inc.)
    PJF                  iAddress Postal Job File (Flagship Software Ltd.)
    PJF                  Photojunction Format File (Photojunction Ltd.)
    PJG                  Photo Assistant Image 
    PJL                  ProCite Term Lists and Journal Title Lists 
    PJP                  JPEG Image 
    PJPEG                JPEG Image 
    PJR                  DVDx Project (labDV)
    PJS                  Purejapan Viewer 
    PJT                  Rational Rose Project (IBM)
    PJT                  Codewright Editor Project (Starbase)
    PJT                  Photojunction Template (Photojunction Ltd.)
    PJT                  Foxpro Project Memo (Microsoft)
    PJT                  AcuBench Project File (Acucorp, Inc.)
    PJX                  Foxpro Project (Microsoft)
    PJX                  Foxpro Project index 
    PJXL                 PaintJet XL PCL Bitmap (Hewlett-Packard)
    PK                   TeX DVI Driver Packed Bitmap Font 
    PK                   Audition Graphical Waveform (Adobe Systems Inc.)
    PK                   LaTeX Compressed Font 
    PK                   Text 
    PK3                  Return to Castle Wolfenstein (Id Software, Inc.)
    PK3                  Quake III Engine PAK File (id Software)
    PK3                  Medal of Honor: Allied Assault 
    PK3                  Heavy Metal: F.A.K.K.2 Archive 
    PK3                  American McGee Alice Archive (Electronic Arts, Inc.)
    PK4                  Railroad Tycoon 3 Packaged Graphics (PopTop Software Inc.)
    PKA                  Compressed Archive File 
    PKA                  PointLineCAD 3D Keyframe Animation (Point Line, Inc.)
    PKB                  QuickPOS (Point-Of-Sale) Keyboard Layout File 
    PKB                  Oracle Package Body 
    PKB                  The Matrix Online Archive (Sony Online Entertainment Inc.)
    PKC                  ForeignDesk Project Archive 
    PKC                  Jython JAR Index File 
    PKD                  PowerKaraoke Project File (PAW)
    PKD                  Turbo Pascal Compressed DOS Batch File (Borland Software Corporation)
    PKD                  Top Secret Crypto Gold Compressed File (TAN$TAAFL Software Company)
    PKF                  ARTiSAN Real-time Studio 
    PKF                  SecretAgent Key File 
    PKG                  Playstation 3 Installation Package (Sony Computer Entertainment, Inc.)
    PKG                  OneSpace Designer Package (CoCreate)
    PKG                  Automise ActionStudio Action Design (VSoft Technologies Pty Ltd.)
    PKG                  MicroSim PCBoard External ASCII Package Definition File 
    PKG                  Newton File/Application (Apple Computer, Inc.)
    PKG                  AppleLink Package Compression Format 
    PKG                  Developer Studio Application Extension (Microsoft)
    PKG                  Systems Management Server (SMS) Package Description File (Microsoft Corporation)
    PKG                  Next Installer Script 
    PKG                  P-CAD Database (Altium Limited)
    PKGLIST              Pepper Keeper Package Installation File (Pepper Computer, Inc.)
    PKN                  Systems Management Server (SMS) Package Creation Notification (Microsoft Corporation)
    PKO                  PublicKey Security Object 
    PKP                  MS Development Common IDE Pakage Project File 
    PKPAK                Archive 
    PKR                  Tony Hawk's Pro Skater 2 Archive (Neversoft Entertainment)
    PKR                  Pretty Good Privacy (PGP) Public Keyring (PGP Corporation)
    PKS                  Oracle Package Specification 
    PKS                  Insight II NMR Peak Intensity/Integral (Accelrys Software Inc.)
    PKT                  TTC FireBerd 500 Capture File 
    PKT                  TeX Font 
    PKT                  EtherPeek Collected Packets 
    PKT                  Fidonet Packet 
    PKX                  Rational XDE (IBM)
    PKY                  PockeTTY Security Certificate (DejaVu Software)
    PKZ                  Winoncd Images Mask File 
    PL                   PROLOG Program File 
    PL                   PERL Program File 
    PL                   Linux Shell Executable Binary 
    PL                   Interleaf Printerleaf or WorldView Format 
    PL                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PL                   TeX Property List Font Metric 
    PL                   Harvard Graphics Palette (Harvard Graphics)
    PL                   Fractal Zplot Palette File (Mystic Fractal)
    PL1                  3D Home Architect Room Plan 
    PL3                  Harvard Graphics Chart Palette (Harvard Graphics)
    PL4                  Pathloss Network Data File 
    PL4                  Micrografx Designer Palette 
    PLA                  APT Assistant 
    PLA                  Connectivity Memory Model Plasticity Function Additional Precision Input File 
    PLA                  STABCAL (stability calculation for aqueous systems) File 
    PLA                  ArchiCAD Plan Archive (Graphisoft R&D Software Development Rt.)
    PLA                  Total Club Manager Training Routine (Electronic Arts Inc.)
    PLASMA               Plasma Fractal Image 
    PLAT                 Platinum Web Package Compiled Web File (Platinum Inc.)
    PLATFORM             Topsy File 
    PLATO                PLATO Web Learning Network (PWLN) Temporary File (PLATO Learning, Inc.)
    PLAY                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PLAY2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PLAYER               Asle / ReDoX 6.1 Format 
    PLAYER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PLAYER               The Player 4.0 
    PLAYLIST             Cyberlink DVD Player Playlist (CyberLink Corp.)
    PLAYSOUND            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PLB                  Insight II Pseudoatom Library (Accelrys Software Inc.)
    PLB                  LogoShow Screensaver 
    PLB                  MSC.Fatigue v9.0 Improved Performance Patch 
    PLB                  DESQview (Symantec)
    PLB                  MicroSim PCBoard Package Library File 
    PLB                  The Print Shop Multimedia Organizer 
    PLB                  AAICW Log File 
    PLB                  STABCAL (stability calculation for aqueous systems) File 
    PLB                  Foxpro Library (Microsoft)
    PLB                  PCAD Library Management Module File 
    PLC                  Lotus Add-in (IBM)
    PLC                  PL/B Object File (Sunbelt Computer Systems, Inc.)
    PLC                  P-CAD Database (Altium Limited)
    PLD                  PINO 3 
    PLD                  PhotoDeluxe PhotoLine Image Document 
    PLD                  Messenger Plus! Backup Configuration (Patchou)
    PLD                  MIE People Database (MIE Software Pty Ltd)
    PLD                  PLD2 Source File 
    PLE                  Phone2PC Sound File (Konexx)
    PLE                  Messenger Plus! Encrypted Logfile (Patchou)
    PLEASURE             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PLENTY2              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PLEX                 Visual Perl File 
    PLEZE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PLF                  ProLife Cellular Automata Binary Data File 
    PLF                  InterVideo WinDVD Playlist File 
    PLF                  SalesLogix Pick List Text File (Best Software)
    PLF                  PatchLink Patch Developers Kit Exported Patch Archive (PatchLink Corporation)
    PLG                  ButtonGadget 
    PLG                  Aston Shell Plug-in (Gladiators Software)
    PLG                  BPM Studio Play List Categories (ALCATech)
    PLG                  Developer Studio Build Log (Microsoft)
    PLG                  REND386/AVRIL File 
    PLG                  Sibelius Plug-in (Sibelius Software Ltd)
    PLH                  Paint Shop Pro Light Preset (Corel Corporation)
    PLI                  Oracle 7 Data Description 
    PLIST                Property List XML File 
    PLJ                  PlayJ Music Format 
    PLK                  Op9630 Am32 File 
    PLK                  ATI Radeon Video Driver 
    PLL                  Clipper Prelinked Library 
    PLL                  PAKLEO Compressed Archive 
    PLM                  Panasonic SD Voice Editor File 
    PLM                  DisorderTracker2 Module 
    PLN                  RealFlight Radio Control Airplane Description (Knife Edge Software)
    PLN                  ArchiCAD 3D Model File (Graphisoft R&D Software Development Rt.)
    PLN                  Home Plan Pro Architecture Design (Home Plan Software)
    PLN                  WordPerfect Spreadsheet File (Corel Corporation)
    PLN                  Psion Organiser Spreadsheet File (Psion PLC)
    PLN                  Flight Simulator Flight Plan (Microsoft)
    PLN                  InDesign 3rd Party Plug-in (Adobe Systems Inc.)
    PLN                  CTP Scene Definition File (Crater Software)
    PLO                  STABCAL (stability calculation for aqueous systems) File 
    PLOT                 UNIX Plot(5) Format 
    PLOTTER              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PLP                  PicaLoader Project File (VOWSoft, Ltd.)
    PLP                  protexIP Project License Profile (Black Duck Software, Inc.)
    PLP                  Messenger Plus! Sound Pack (Patchou)
    PLR                  Descent Pilot File 
    PLR                  Player File 
    PLRSTAT              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PLS                  MYOB Data File (MYOB Technology Pty Ltd.)
    PLS                  Shoutcast File 
    PLS                  MattBatt iAM-player (MattBatt Productions)
    PLS                  Real MP3 Playlist 
    PLS                  Generic Playlist File 
    PLS                  Napster MPEG PLayList File 
    PLS                  WinAmp MPEG PlayList File (Nullsoft)
    PLS                  DisorderTracker2 Sample 
    PLS                  Pro/ENGINEER Temporary File (PTC)
    PLT                  HP Graphics Language 
    PLT                  OziExplorer Plot Track (Des & Lorraine Newman)
    PLT                  Page Magic Ver. 2.0 Paper Format 
    PLT                  Infinity Game Engine Bitmap Graphic (BioWare Corp.)
    PLT                  X-wing vs. Tie Fighter Pilot File 
    PLT                  Windows Update 
    PLT                  Software Platform File 
    PLT                  dnaLIMS 3700 Sample Sheet (dnaTools)
    PLT                  Palette 
    PLT                  Clipper 5 Pre-linked Transfer File 
    PLT                  Gerber Sign-making Software File 
    PLT                  AutoCAD Plot drawing (Autodesk, Inc.)
    PLT                  PLATO Risk Management Data (Unisys)
    PLT                  Organiser Paper Layout Definition (IBM)
    PLT                  MicroStation Driver Configuration for Plotting (Bentley Systems, Incorporated)
    PLT                  Clipper Ver. 5 Pre-linked Transfer File 
    PLUGIN               Amaya Plug-in Library File 
    PLUIE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PLUS                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PLW                  PicoLog Data File 
    PLX                  PERL Programming Language Script 
    PLX                  iriver Plus Playlist (iriver)
    PLX                  Neurological Data for Statistical Analysis 
    PLX                  Executable Perl Script 
    PLY                  PopMail Data 
    PLY                  Polygon Model Format (Stanford University, Georgia Tech)
    PLY                  Harvard Graphics Spotlight Presentation Screen (Harvard Graphics)
    PLY                  Blaze Media Pro File 
    PLY                  AIRO (Odetics Broadcast)
    PLY                  MiDi PoWeR Playlist File (ACCORG)
    PLY                  Cyberware Scanner File 
    PLZ                  Lotus Freelance Presentation 
    PM                   Presentation Manager Graphic 
    PM                   Amiga Power Music Song/Module 
    PM                   Perl Code Module 
    PM                   Rebel Assault File 
    PM                   Pegasus Mail Frequently Used File (David Harris)
    PM                   PageMaker Document (Adobe)
    PM                   X Window PixelMap Bitmap 
    PM                   PrintMaster Graphic 
    PM!                  Pegasus Mail Address Book Index (David Harris)
    PM#                  Pegasus Mail Status/Index File (David Harris)
    PM$                  Pegasus Mail Temporary File (David Harris)
    PM0                  Pegasus Mail State File (David Harris)
    PM1                  Pegasus Mail Auxiliary Address Book File (David Harris)
    PM2                  Pegasus Mail UID Information (David Harris)
    PM3                  PageMaker Version 3 Document (Adobe)
    PM3                  Pegasus Mail (David Harris)
    PM4                  PageMaker Version 4 Document (Adobe)
    PM4                  Pegasus Mail General Filtering Rules (David Harris)
    PM5                  PageMaker Version 5 Document (Adobe)
    PM5                  Pegasus Mail New Messasge Template Script (David Harris)
    PM6                  Pegasus Mail Reply Template Script (David Harris)
    PM6                  PageMaker Version 6 Document (Adobe)
    PM7                  Pegasus Mail Folder State File (David Harris)
    PMA                  Pegasus Mail Version 2.2 Data Archive (David Harris)
    PMA                  Windows Performance Monitor File (Microsoft)
    PMA                  MSX Computers Archive Format 
    PMAP                 Pocket Mindmap Mindmap File (JKRB Software)
    PMAPZ                Pocket Mindmap Compressed Mindmap File (JKRB Software)
    PMB                  Pegasus Mail Version 2.2 Addressbook (David Harris)
    PMB                  Bitmap Image 
    PMC                  Pegasus Mail ASCII Format Mail Filter Rule (David Harris)
    PMC                  Windows Performance Monitor File (Microsoft)
    PMC                  A4Tech Scanner Graphic 
    PMC                  Perl Compiled Code Module 
    PMD                  Pegasus Mail User Dictionary (David Harris)
    PMD                  PageMaker (Adobe)
    PMD                  PlanMaker Spreadsheet 
    PMD                  Polyphonic Ringtone File for Phones 
    PME                  Plazmic Media Engine File 
    PME                  Pixela Digital Picture (Pixela Corporation)
    PME                  Pegasus Mail Filter Rule File (David Harris)
    PMF                  Pyrism Map File (Two Brothers Software)
    PMF                  Pegasus Mail Message Attachment (David Harris)
    PMF                  ArcReader GIS Mapping 
    PMF                  PCLTool PageTech Metafile File Format (Page Technology Marketing, Inc. (PageTech))
    PMF                  PSP Movie Format (PSMF) (Sony Corporation of America)
    PMF                  AppMind Process Management File (Appmind Software AB)
    PMG                  Pegasus Mail Folder Index (David Harris)
    PMG                  PageMaker Group File (Adobe)
    PMG                  Paint Magic 
    PMH                  Pegasus Mail Phonemic Homonym Dictionary (David Harris)
    PMI                  Pegasus Mail Message Folder Index File (David Harris)
    PMI                  OS/2 Program Manager Information 
    PMJ                  3D Digital Scanner File 
    PMJ                  Pegasus Mail WinPMail State File (David Harris)
    PMJX                 3D Digital Scanner File 
    PMK                  Pegasus Mail Keyboard File (David Harris)
    PML                  PageMaker Library (Adobe)
    PML                  PADGen Program Information 
    PML                  Windows Performance Monitor File (Microsoft)
    PML                  Pegasus Mail Distribution List (David Harris)
    PML                  Spin Promela Source Code File 
    PML                  Process Monitor Process Capture File (Microsoft Corporation)
    PMM                  Amaris BTX/2 Program 
    PMM                  Pegasus Mail Message Folder (David Harris)
    PMN                  Pegasus Mail Annotation File (David Harris)
    PMO                  Print Master Gold Text 
    PMO                  Pegasus Mail Saved Message (David Harris)
    PMOD                 Pike Module (Linköpings universitet)
    PMP                  Pegasus Mail Notepad File (David Harris)
    PMP                  VideoCraft GIF Animator Project 
    PMP                  AutoCAD R2000 Plotter Model Parameters (Autodesk, Inc.)
    PMP                  Photomorph Project 
    PMP                  Project Manager Pro Schedule 
    PMP                  Soldat Map Maker/Soldat Polymap File 
    PMP                  ThumbsPlus Graphic 
    PMP                  Sony DSC-F1 Cyber-shot 
    PMP                  PowerMedia Color Palette 
    PMP                  PHPMaker Project File (e.World Technology Limited)
    PMPPKG               Pegasus Mail Pegasus Mail Preferences PacKaGe (David Harris)
    PMQ                  Pegasus Mail Mail Filter Rules (David Harris)
    PMR                  Windows Performance Monitor File (Microsoft)
    PMR                  Pegasus Mail Address Book (David Harris)
    PMR                  PhotoModeler Project (Eos Systems)
    PMS                  Pegasus Mail Signature (David Harris)
    PMS                  Password Master Encrypted Password File (dreameesoft.com)
    PMS                  Personal Media Suite Encoded File (Freecom Technologies)
    PMS                  AliceSoft PMS Bitmap 
    PMS                  Pop'n Music Script 
    PMT                  Technics Keyboard Panel Memory File 
    PMT                  Pegasus Mail Tray File (David Harris)
    PMT                  PageMaker Template (Adobe Systems Inc.)
    PMTOOL               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PMU                  Pegasus Mail Pmail Keyboard Macro-related File (David Harris)
    PMV                  Pegasus Mail Filter Rules (David Harris)
    PMW                  Pegasus Mail Draft Stored Messages To Be Sent (David Harris)
    PMW                  Windows Performance Monitor File (Microsoft)
    PMX                  Pegasus Mail Final Stored Messages To Be Sent (David Harris)
    PMZ                  Pegasus Mail Pmail Scrap Buffer (David Harris)
    PM_                  Musicato MUSICAT.ZIT Compressed File 
    PN2                  International Systems Consultancy ParsNegar II Word Processor 
    PN3                  Harvard Graphics Printer Driver (Harvard Graphics)
    PNA                  PhatNoise Media/Music Manager Audio File (PhatNoise, Inc.)
    PNA                  Panoramide Java 3D Panoramic Animations (Ki Solutions Group)
    PNAGENT              Program Neighborhood Agent (Citrix Systems, Inc.)
    PNB                  Pokémon NetBattle Trainer/Team File (Smogon)
    PNC                  IBM Voice Type Language Scripts Data File 
    PNC                  Pegasus Mail Content Control Set Definition (David Harris)
    PND                  Pension System Pending File (DATAIR Employee Benefit Systems, Inc.)
    PND                  Pegasus Mail Connection Information (David Harris)
    PND                  NstisApp.exe Hyperspectral Data File (Plain Sight Systems)
    PNF                  Precompiled Setup Information 
    PNF                  Portable Network Graphics Frame Bitmap 
    PNF                  Windows Precompiled Setup Information (Microsoft)
    PNG                  Fireworks Image File (Macromedia)
    PNG                  Portable (Public) Network Graphic 
    PNG                  Paint Shop Pro Browser Catalog (Corel Corporation)
    PNL                  ProCon Control Panel Panel Control File (Roche AV)
    PNL                  PEERNET Label Designer File 
    PNM                  Pegasus Mail IMAP Message Flag Status (David Harris)
    PNM                  PBM Portable Any Map Graphic Bitmap 
    PNP                  Pegasus Mail POP3 Server Side Filter Rules (David Harris)
    PNP                  PEERNET Publisher (PEERNET Inc.)
    PNQ                  ICQ Instant Message File (ICQ Inc.)
    PNR                  PEERNET Label Designer File 
    PNS                  PEERNET Label Designer File 
    PNS                  Pegasus Mail Formatted Signatures (David Harris)
    PNT                  ARC Format Vector Point Data 
    PNT                  MacPaint Graphic File 
    PNT                  FTN Software Pointlist Segment 
    PNT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PNT                  MacPaint Bitmap Graphic 
    PNT                  World Data Bank II Country Data File 
    PNT                  Pro/ENGINEER Pen Table Plot (PTC)
    PNT                  PaperPort Thumbsview (ScanSoft Inc.)
    PNT                  MarkMail Qwk Reader Pointers 
    PNTG                 MacPaint Bitmap Graphic 
    PNU                  PEERNET Label Designer File 
    PNV                  PCschematic PowerDistribution Project (DpS CAD-center ApS)
    PNV                  CRiSP Harvest File 
    PNW                  Poles 'n' Wires Project Data File (PowerMation)
    PNX                  Pegasus Mail RTF Version of Queued/Draft Message (David Harris)
    PNY                  QV Map Image File (Touratech)
    PNZ                  Panorama Shortcut to Open Multiple Databases (ProVUE Development)
    PO                   GNU Gettext Portable Object (Free Software Foundation)
    PO2                  STABCAL (stability calculation for aqueous systems) File 
    PO3                  STABCAL (stability calculation for aqueous systems) File 
    PO5                  STABCAL (stability calculation for aqueous systems) File 
    POA                  Fifa World Cup Game Data fe Art Legalscr File 
    POB                  Punch CAD Program 3D Object (Punch! Software)
    POC                  PocoMail Script (Poco Systems, Inc.)
    POD                  Terminal Velocity Archive 
    POD                  Punch CAD Program (Punch! Software)
    POD                  Bloodrayne 
    POD                  Text 
    POD                  PodFlow Project (Ikon AVS Ltd.)
    POEM                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    POEMS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    POF                  Programming Object File 
    POF                  Tajima DG/ML Embroidery Design File (Pulse Microsystems Ltd.)
    POG                  Descent 2 Alternative Texture Set 
    POH                  Optimized Geoworks .GOH File 
    POK                  ZX Spectrum-Emulator 
    POKER                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    POL                  AscToHTM Policy File 
    POL                  Windows Policy File 
    POL                  Personal Paint Polish Language User Interface File (Amiga)
    POL                  Policy Windows 95 Network Setup 
    POL                  InnovMetric Software 3D Polygon Models Format 
    POL                  Music File 
    POLICY               Java Policy File (Sun)
    POLKA                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    POLYGONIA            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PONGLIFE             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    POO                  INITeX String Pool 
    POO                  8-bit Commodore 64 Machine Code 
    POOL                 TeX Messages File 
    POP                  dBASE Popup Menu 
    POP                  PopMail Message Index 
    POR                  Corel Painter Portfolio File 
    POR                  SPSS Portable Data File (SPSS Inc.)
    PORT                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PORT1                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PORT2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PORT3                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PORT4                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PORT5                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    POS                  POSTER Save File (Poster Software)
    POS                  WinHex Position Data (X-Ways Software Technology AG)
    POS                  QuickPOS IIF File 
    POS                  Paint Shop Pro Polished Stone Preset (Corel Corporation)
    POS                  ProCite Output Styles 
    POSTSCRIPT           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    POT                  Animation 
    POT                  PowerPoint Template (Microsoft)
    POT                  Fractint Continuous Potential Bitmap 
    POT                  GNU Gettext Portable Object Base Translation (Free Software Foundation)
    POTHTML              Powerpoint HTML Template (Microsoft Corporation)
    POTM                 Power Point Microsoft Office Open XML Format Presentation Template with Macros Enabled (Microsoft Corporation)
    POTMENX              egis Encrypted POTM (Powerpoint 2007) File (HiTRUST Inc.)
    POTX                 Power Point Microsoft Office Open XML Format Presentation Template (Microsoft Corporation)
    POTXENX              egis Encrypted POTX (Powerpoint 2007) File (HiTRUST Inc.)
    POV                  Persistence of Vision Ray-tracer 
    POW                  PowerChords Chord Chart 
    POWER                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    POWER                Goo Picture Library 
    POWERCHORD           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    POWERFIFTH           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    POWERFONT            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    POWERGS              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    POWER_OF_AMERICAN    Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PP                   Topocad Polygon Point Coordinate File (Chaos Systems AB)
    PP                   Free Pascal Source Code File (Free Pascal Development Team)
    PP                   Compressed Amiga Archive File 
    PP                   Perl Data Language PreProcessor File 
    PP                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PP$                  Modem Bitware Fax Disk6 File 
    PP1                  Free Pascal Compiled Unit for GO321v1 Platform (FPC Development Team)
    PP2                  PingPlotter Sample File (Nessoft, LLC)
    PP2                  Ping Plotter Saved Trace Data (Nessoft, LLC)
    PP2                  Visual Passage Planner Passage File (Digital Wave)
    PP2                  Curious Labs Poser Prop File 
    PP3                  PlanetPress Suite 3 Form File (Objectif Lune)
    PP3NSCALE            PlanetPress File 
    PP4                  Picture Publisher 
    PP5                  Picture Publisher 
    PPA                  PowerPoint Add-in (Microsoft)
    PPAMENX              egis Encrypted PPAM (Powerpoint 2007) File (HiTRUST Inc.)
    PPB                  WordPerfect Print Preview Button Bar (Corel Corporation)
    PPC                  XMap Flight Plan or Flight Plan Statistic File (Delorme)
    PPC                  Breeze Plug-in for PowerPoint Sync File (Macromedia)
    PPC                  Roxio Easy CD Creator File (Sonic Solutions)
    PPC                  Uefa Champions League 
    PPC                  Percussion Studio Compiled Rhythmic Patterns (Henry Kellner)
    PPD                  PostScript Printer Description 
    PPD                  PageMaker PostScript Printer Description (Adobe)
    PPF                  Turtle Beach Pinnacle Program File 
    PPF                  PlayStation Patch File 
    PPF                  MENSI PointScape 3D Laser Scanner Project File 
    PPF                  EdgeCAM Part Data (Pathtrace Engineering Systems)
    PPF                  Micrografx Picture Publisher File 
    PPF                  Paint Shop Pro Soft Plastic Preset File (Corel Corporation)
    PPG                  ProPixel2D Image (Pacific Software)
    PPG                  PowerPoint Presentation (Microsoft)
    PPG                  Professor Franklin's Photo Print Gold 
    PPI                  PowerPoint Graphics File (Microsoft)
    PPJ                  Premiere Video Editing File (Adobe Systems Inc.)
    PPK                  PPK Archive 
    PPK                  PuTTY Win32 Telnet/SSH Client 
    PPL                  Harvard Graphics Polaroid Palette Plus ColorKey Driver (Harvard Graphics)
    PPM                  PBM Portable Pixelmap Graphic 
    PPM                  Insight II Proton Chemical Shifts (Accelrys Software Inc.)
    PPMA                 Portable Bitmap 
    PPN                  Lexmark Firmware Flash File (Lexmark International, Inc.)
    PPO                  Clipper Preprocessor Output 
    PPO                  Free Pascal Compiled Unit for OS/2 Version (FPC Development Team)
    PPP                  Punk Productions Picture 
    PPP                  Point to Point Protocol 
    PPP                  Parson Power Publisher 
    PPP                  Enfocus Preflight Profile 
    PPP                  Picture Pump Project File 
    PPP                  PagePlus Publication (Serif)
    PPR                  PowerPlay OLAP Multidimensional Cube (Cognos Incorporated)
    PPRINT               Pics Print File 
    PPS                  Personal Producer Storyboard 
    PPS                  ArcView Processing Set Codes (ESRI)
    PPS                  PowerPoint Slideshow (Microsoft Corporation)
    PPSENX               egis Encrypted Powerpoint PPS File (HiTRUST Inc.)
    PPSM                 Power Point Microsoft Office Open XML Format Presentation Slide Show with Macros Enabled (Microsoft Corporation)
    PPSMENX              egis Encrypted PPSM (Powerpoint 2007) File (HiTRUST Inc.)
    PPSX                 Power Point Microsoft Office Open XML Format Presentation Slide Show (Microsoft Corporation)
    PPSXENX              egis Encrypted PPSX (Powerpoint 2007) File (HiTRUST Inc.)
    PPSXENX              egis Encrypted Powerpoint Open XML PPSX File (HiTRUST Inc.)
    PPT                  BIFF File (Microsoft)
    PPT                  PowerPoint Presentation (Microsoft)
    PPTENX               egis Encrypted Powerpoint PPT File (HiTRUST Inc.)
    PPTHTML              Powerpoint HTML Document (Microsoft Corporation)
    PPTM                 Power Point Microsoft Office Open XML Format Presentation with Macros Enabled (Microsoft Corporation)
    PPTMENX              egis Encrypted PPTM (Powerpoint 2007) File (HiTRUST Inc.)
    PPTX                 Power Point Microsoft Office Open XML Format Presentation (Microsoft Corporation)
    PPTXENX              egis Encrypted Powerpoint Open XML PPTX File (HiTRUST Inc.)
    PPTXENX              egis Encrypted PPTX (Powerpoint 2007) File (HiTRUST Inc.)
    PPU                  Free Pascal Compiled Unit for Linux or DOS Version (FPC Development Team)
    PPV                  Pocket PowerPoint Presentation (Microsoft)
    PPV                  Vox Proxy Animated Power Point (Right Seat Software, Inc.)
    PPV                  Pogle Platinum Software Version Archive (Pandora International)
    PPV                  Punch CAD Program Thumbnail (Punch! Software)
    PPW                  Free Pascal Compiled Unit for Windows Version (FPC Development Team)
    PPW                  Micrografx Picture Publisher Wizard 
    PPX                  Serif PagePlus Publication 
    PPX                  PowerPlay OLAP Reporting (Cognos Incorporated)
    PPX                  PingPlotter Script (Nessoft, LLC)
    PPZ                  PowerPoint Packaged Presentation (Microsoft)
    PQ                   Progress Quest Saved Character 
    PQ                   PageMaker Default Printer Style (Adobe)
    PQA                  Palm Query Application File 
    PQB                  PowerQuest Batch File 
    PQE                  Easy Recovery Recovered Data (Kroll Ontrack Inc.)
    PQF                  Corel Presentations File (Corel Corporation)
    PQG                  Rescue ME/OS2/DOS File 
    PQI                  PhreeqcI 
    PQI                  PowerQuest Drive Imaging Software 
    PQO                  PhreeqcI 
    PQW                  Corel Presentations 9 Runtime 
    PQX                  Power Quest Drive Image Index 
    PQZ                  QuaSZ Palette File (Mystic Fractal)
    PR                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PR                   EROSION 3D Precipitation Parameters 
    PR                   Sun Icon/Cursor 
    PR0                  Payroll (CheckMark Software)
    PR1                  Payroll (CheckMark Software)
    PR1                  Packrat Ver 4.0 Data File 
    PR1                  OzWin CompuServe E-mail/Forum Access Purged Message File 
    PR2                  Proteus Session 
    PR2                  Payroll (CheckMark Software)
    PR2                  Persuasion Presentation 
    PR2                  Packrat 4.x data file 
    PR2                  dBASE IV Printer Driver 
    PR3                  Persuasion Presentation (Adobe)
    PR3                  PFXplus Resource File (POWERflex Corporation)
    PR3                  Payroll (CheckMark Software)
    PR3                  dBASE IV PostScript Printer Driver 
    PR4                  Harvard Graphics Presentation (Harvard Graphics)
    PRAY                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRB                  xyALGEBRA File 
    PRC                  PathAway Map (MuskokaTech.)
    PRC                  Palmpilot Resource File (Palm, Inc.)
    PRC                  Profan Compiled P-Code (rgh-soft.de)
    PRC                  filePro Data Table (fp Technologies)
    PRC                  Rational Rose Processes (IBM)
    PRC                  Picture Gear Pocket 
    PRC                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRC                  Corel Presentation 
    PRC                  Medlin Accounting Current Year Paychecks (Medlin Accounting)
    PRC                  Hillwalker/MapWise/Alpiniste/PhotoMaps List of Parked Route Cards (ISYS Computers)
    PRD                  Product Designer Data File (ProductFoundry)
    PRD                  Lode Runner Game 
    PRD                  Printer Driver 
    PRE                  Stork Format CMYK Bitmap 
    PRE                  Programmer's WorkBench Settings 
    PRE                  NeroWave Editor Presets (Nero AG)
    PRE                  Freelance Graphics (IBM)
    PRE                  Insight II Dynamics Scratch File (Accelrys Software Inc.)
    PRECSTLE             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PREF                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PREFERENCES          Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PREFERRED            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PREFIX               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PREFS                Preferences File 
    PREFS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PREL                 Premiere Elements Project File (Adobe Systems Incorporated)
    PREP                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PREQ                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PREQUAL              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRESENT              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRESENTS             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRESREL              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRESS                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PREXPORT             Premiere Preset (Adobe Systems Inc.)
    PRF                  Fastgraph Pixel Run Format Graphics 
    PRF                  ClarisWorks Preferences File 
    PRF                  CheckIt Pro 1.0 Data 
    PRF                  Preferences File 
    PRF                  Profiler Output 
    PRF                  Rapfile 
    PRF                  Profan (rgh-soft.de)
    PRF                  Windows System File 
    PRF                  Plot Reference File 
    PRF                  dBASE IV Printer Driver 
    PRF                  Polychrome Recursive Format Bitmap 
    PRF                  McAfee Viruscan Profile 
    PRF                  MicroImages Print Driver File (MicroImages, Inc.)
    PRF                  PICS Rules File 
    PRF                  Monarch Portable Report File 
    PRF                  Director Settings (Macromedia)
    PRF2                 Nord Modular G2 Performance (Clavia DMI AB)
    PRG                  Atari ST Program (Infogrames Entertainment SA)
    PRG                  RPG Toolkit Program Script (Christopher Matthews)
    PRG                  SHARP MZ-series Emulator File 
    PRG                  Program File 
    PRG                  SpecEm Snapshot 
    PRG                  WAVmaker Program 
    PRG                  GEM Executable Program 
    PRG                  OzWin CompuServe E-mail/Forum Access Purged Message File 
    PRG                  Commodore PET/VIC20/64/128 Executable Programs in BASIC 
    PRG2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRH                  Cold Fusion Studio 3.1 Project 
    PRH                  HomeSite 3.0 Project 
    PRI                  LocoScript Printer Definitions (LocoScript Software)
    PRI                  Prisms 
    PRIMER               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRINCIPAL            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRINT                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRINT2               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRINTBUF16           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRINTBUF32           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRINTBUF48           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRINTBUF62           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRINTBUF8            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRINTER              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRISM                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRIVY                PrivyPad (SlavaSoft Inc.)
    PRJ                  MyCad IC Layout Architecture 
    PRJ                  Monarch File 
    PRJ                  Stylus Studio Project FIle (DataDirect Technologies, a division of Progress Software Corporation)
    PRJ                  ArcView ESRI Coordinate System Definition (ESRI)
    PRJ                  AwpHelp Project 
    PRJ                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRJ                  3D Studio Project (Autodesk, Inc.)
    PRJ                  MultiEdit Editor Project (MultiEdit Software)
    PRJ                  Project File 
    PRJ                  FireFly SDK Designer Project (PlanetSquires Software)
    PRJ                  HitPlayer Advanced Playlist File 
    PRJ                  Visual Café Project (Symantec)
    PRJ                  Visual Basic Project (Microsoft)
    PRK                  Tony Hawk's Pro Skater 2 Saved Park (Neversoft Entertainment)
    PRK                  Campground Master/Resort Master Database (Cottonwood Software)
    PRKLOG               Campground Master/Resort Master Log File (Cottonwood Software)
    PRL                  Process Revolution Object Library File 
    PRL                  Paint Shop Pro Rough Leather Preset (Corel Corporation)
    PRL                  Perl Script 
    PRM                  Premiere Plug-in (Adobe Systems Inc.)
    PRM                  MicroImages Parameters File for Radar Slant to Ground Conversion (MicroImages, Inc.)
    PRM                  MYOB Premier 2004 Data File (MYOB Technology Pty Ltd.)
    PRM                  Parameter File 
    PRM                  Prolog Module 
    PRM                  BullsEye Style File 
    PRN                  HP Printer Control Language 
    PRN                  PostScript File 
    PRN                  Printer Text File 
    PRN                  Calcomp Raster Bitmap 
    PRN                  Signature Printer Driver 
    PRN                  XYWrite Printer Driver 
    PRN                  DataCAD Windows Printer File (DATACAD LLC)
    PRO                  Atari Disk Image (Atari)
    PRO                  Pro/ENGINEER Configuration (PTC)
    PRO                  Sid Meier's SimGolf 
    PRO                  IDL Source Code (Research Systems, Inc.)
    PRO                  Punch Pro Home Design File 
    PRO                  Infinity Game Engine Projectile Type Description (BioWare Corp.)
    PRO                  Euphoria Profile (Rapid Deployment Software)
    PRO                  DOS Graphics Profile File 
    PRO                  Lorenz Graf HTML Tool Project 
    PRO                  Pagis Pro Ver. 3.0 Graphic 
    PRO                  Configuration File (Profile) 
    PRO                  Chord Pro Song Format (Mussoft)
    PRO                  Guitar Chord File 
    PRO                  Fast-Talk Pronunciation File (Fast-Talk Communications, Inc.)
    PRO                  PROLOG Program File 
    PRO                  PCschematic ELautomation Project (DpS CAD-center ApS)
    PRO                  Pegasus Mail Profile (David Harris)
    PRO                  Creamware Pulsar Audio File 
    PRO                  Qt Project Configuration File (Trolltech AS)
    PRO                  PV-WAVE Procedure File 
    PRO                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRO                  Terramodel Project File 
    PROCESSOR            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRODOS               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRODUCT              Norton Ghost File 
    PROFILE              FastTrack Client Profile 
    PROFILE              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PROFILE              UNIX Bourne or Kom Shell Environment File 
    PROFILES             Amaya Configuration 
    PROG                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PROGRAM              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PROGRAMMER           Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PROJ                 Poseidon for UML Project Information (Gentleware AG)
    PROJ                 Insight II NMR Project (Accelrys Software Inc.)
    PROJ                 NeXT Interface Builder Project File 
    PROJECT              Project Builder for WebObjects Project (Apple Computer, Inc.)
    PROLAMP              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PROMIZER             Promizer v0.1/v1.0c/v1.8a/v2.0/v4.0 Formats 
    PROP                 Application Visualization System AVS High-end Visualization Environment. 
    PROPACKER            ProPacker v1.0/v2.1/v3.0 Formats [Asle / ReDoX] 
    PROPERTIES           Netscape Communicator Java Classes File 
    PROPERTIES           Java Properties File (Sun)
    PRORUNNER            ProRunner v1/v2 Formats [Asle / ReDoX] 
    PROTECT              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PROTO                Windows Amaya File 
    PRP                  Compressed GIMP Image with Properties of GIMP Properties File 
    PRP                  Oberon Prospero Data Conversion Saved Project 
    PRP                  InstantDB Database File 
    PRP                  Movie 
    PRP                  Palace Server Server Configuration (Communities.com)
    PRP                  Uru: Ages Beyond Myst Page Resource Package (PRP) File (Cyan Worlds, Inc.)
    PRP                  Rational Rose Model Properties (IBM)
    PRPRESET             Premiere Preset (Adobe Systems Inc.)
    PRPROJ               Premiere Pro Project File (Adobe)
    PRR                  Perfect Resume Data 
    PRS                  Norton Viewer DLL 
    PRS                  Yamaha Pro Style (Yamaha Corporation of America)
    PRS                  LodeRunner Game 
    PRS                  Harvard Graphics (Harvard Graphics)
    PRS                  Printer Resource File 
    PRS                  WordPerfect Printer Resource Font File (Corel Corporation)
    PRS                  dBASE Procedure 
    PRS                  Lotus Freelance Presentation for OS/2 
    PRSL                 Premiere Style (Adobe Systems Inc.)
    PRT                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRT                  Presentations Template (Corel)
    PRT                  Unigraphics Part File 
    PRT                  P-CAD Component (Altium Limited)
    PRT                  Personal Paint Portuguese Language User Interface File (Amiga)
    PRT                  Printer-formatted File 
    PRT                  Pro/ENGINEER Model File (PTC)
    PRT                  Process Revolution Template File 
    PRT                  Pro/ENGINEER Parts File (PTC)
    PRT                  SCEdit Part File 
    PRT                  Schedule+ Print File (Microsoft)
    PRT                  IRMA Workstation for Windows Printer Configuration 
    PRT                  InterComm Pin Use Definition 
    PRT                  CADKEY Parts File (Kubotek USA Inc.)
    PRT                  Printer Configuration 
    PRT                  Dr.Halo Printer Driver 
    PRT                  W32/Parrot-A Worm-related 
    PRT2                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PRTL                 Premiere Title (Adobe Systems Inc.)
    PRV                  Previous Version 
    PRV                  ClarisWorks File 
    PRV                  psiMail ISP Template 
    PRV                  PsiMail Internet Provider Template 
    PRV                  SecretAgent Private Key File 
    PRV                  Backup 
    PRVKR                Pretty Good Privacy (PGP) Private Keyring (PGP Corporation)
    PRW                  PR Weaver Saved Data File (PR Weaver Software)
    PRW                  Art-lantis Shader's Preview (ABVENT)
    PRX                  Foxpro Compiler Program (Microsoft)
    PRX                  Atari ST Executable 
    PRX                  Primavera Project Planner Compressed Project (Primavera Systems, Inc.)
    PRX                  Process Revolution Diagram File 
    PRX                  Media Player Ver. 9 XML/Media File (Microsoft Corporation)
    PRZ                  Freelance Graphics 97 File (IBM)
    PR_                  Compressed Project File 
    PS                   PostScript 
    PS                   Works File (Microsoft)
    PS                   PaulShields Song/Module 
    PS                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PS0                  PS/2 File 
    PS1                  PostScript File 
    PS16                 Protracker Studio 16 Format 
    PS2                  Level II PostScript File (Adobe)
    PS3                  Level III PostScript File (Adobe)
    PS5                  ProSteel 5 Data File (Survey Design Associates Ltd.)
    PSA                  Pretty Simple Archive Compressed Archive 
    PSA                  Photoshop Album Photo Album File (Adobe)
    PSA                  Intactix pro/space Plus Planogram (Floorplan) (JDA Software Group, Inc.)
    PSAR                 PSP Update Data (Sony Corporation of America)
    PSB                  Paint Shop Pro Sunburst Preset (Corel Corporation)
    PSB                  Pinnacle Sound Bank 
    PSB                  Project Scheduler Configuration File 
    PSB                  Photoshop Large Document Format (Adobe Systems Inc.)
    PSC                  Paint Shop Pro Sculpture Preset (Corel Corporation)
    PSCOLLECTION         PornSnatcher Collection Data (PORNSnatcher.com)
    PSD                  DESIGN II for Windows General Flowsheet 
    PSD                  Apple Viewer (Apple Computer, Inc.)
    PSD                  Periscope Debugger Definition File 
    PSD                  Photoshop Format (Adobe Systems Inc.)
    PSE                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PSE                  IBM Printer Page Segment Bitmap 
    PSEG                 IBM Printer Page Segment Bitmap 
    PSEG3820             IBM Printer Page Segment Bitmap 
    PSEG38PP             IBM Printer Page Segment Bitmap 
    PSETUP               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PSF                  PowerSoft Report File 
    PSF                  MicroStation Plot Specification (Bentley Systems, Incorporated)
    PSF                  PhotoStudio Graphic (ArcSoft, Inc.)
    PSF                  PostScript Support File 
    PSF                  Insight II X-PLOR Molecular Structure File (Accelrys Software Inc.)
    PSF                  ChiWriter PostScript Printer Font 
    PSF                  Photoshop Proof Setup (Adobe Systems Inc.)
    PSF                  Playstation Sound Format (Sony Corporation of America)
    PSF2                 Playstation 2 Sound Format (Sony)
    PSFLIB               Playstation Sound Format (Playstation Sound Format Library) (Sony Corporation of America)
    PSFLIB2              Playstation 2 Sound Format (Playstation Sound Format Library) (Sony)
    PSH                  Lexmark Firmware Flash File (Lexmark International, Inc.)
    PSH                  Punch Home Design Series CAD Data File 
    PSI                  Psion A-law Audio (Psion PLC)
    PSI                  Adesign Image File (Pierresoft.com)
    PSID                 PostScript Image Data 
    PSID                 Word Sidtune File Format 
    PSION                Psion A-law Audio (Psion PLC)
    PSL                  Sierra Mastercook Layout File 
    PSL                  MicroSim PCBoard Padstack Library File 
    PSL                  Accellera Property Specification Language Source Code File (Accellera)
    PSL                  PATROL Script Language Script (BMC Software, Inc.)
    PSL                  PowerTerm Script Language (Ericom Software)
    PSM                  Solid Edge Document (UGS PLM Solutions)
    PSM                  PrintShop Mail Data File (Atlas Software BV)
    PSM                  Epic Games Sound Data 
    PSM                  Protracker Studio Module Format 
    PSM                  SolidEdge CAD File 
    PSM                  Turbo Pascal Symbol Table (Borland Software Corporation)
    PSM5                 PrintShop Mail Data File (Atlas Software BV)
    PSMD                 PrintShop Mail Document (Atlas Software B.V.)
    PSMODEL              Delcam Powershape 
    PSMT                 PrintShop Mail Template (Atlas Software B.V.)
    PSN                  Post-it Software Notes (3M Company)
    PSN                  Sound Script Presentation 
    PSO                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PSOD                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PSP                  Streets & Trips PushPin File (Microsoft)
    PSP                  Scenicsoft Preps 
    PSP                  Project Scheduler Planning File 
    PSP                  Prodea Synergy Procedure 
    PSP                  Paint Shop Pro Image (Corel Corporation)
    PSP                  PL/SQL Server Page (Oracle)
    PSPAUTOSAVE          Paint Shop Pro Ver. 8+ Auto Save File (Corel Corporation)
    PSPBRUSH             Paint Shop Pro Ver. 8+ Brush (Corel Corporation)
    PSPBUMPMAP           Paint Shop Pro Ver. 8+ Bump Map (Corel Corporation)
    PSPCACHE             Paint Shop Pro Ver. 8+ Cache (Corel Corporation)
    PSPCMYKPROFILE       Paint Shop Pro Ver. 8+ CMYK Profile (Corel Corporation)
    PSPD                 PhotoSuite Ver. 5 File (Sonic Solutions)
    PSPDEFORMATIONMAP    Paint Shop Pro Ver. 8+ Deformation Map (Corel Corporation)
    PSPENVIRONMENTMAP    Paint Shop Pro Ver. 8+ Environment Map (Corel Corporation)
    PSPFRAME             Paint Shop Pro Ver 8+ Frame (Corel Corporation)
    PSPGRADIENT          Paint Shop Pro Ver. 8+ Gradient (Corel Corporation)
    PSPIMAGE             Paint Shop Pro Ver 8+ Image (Corel Corporation)
    PSPMASK              Paint Shop Pro Ver 8+ Mask (Corel Corporation)
    PSPPALETTE           Paint Shop Pro Ver. 8+ Palette (Corel Corporation)
    PSPSCRIPT            Paint Shop Pro Ver 8+ Script (Corel Corporation)
    PSPSELECTION         Paint Shop Pro Ver. 8+ Saved Selection (Corel Corporation)
    PSPSHAPE             Paint Shop Pro Ver 8+ Shape (Corel Corporation)
    PSPSTYLEDLINE        Paint Shop Pro Ver. 8+ Styled Line (Corel Corporation)
    PSPTUBE              Paint Shop Pro Ver 8+ Tube (Corel Corporation)
    PSPWORKSPACE         Paint Shop Pro Ver 8+ Workspace (Corel Corporation)
    PSQ                  Postscript Grafic 
    PSR                  PowerSoft Report (Sybase, Inc.)
    PSR                  Project Scheduler Resource File 
    PSS                  Playstation Game File (Sony Computer Entertainment America Inc.)
    PSS                  Paint Shop Pro Sandstone Preset (Corel Corporation)
    PSS                  Axialis Screen Saver Producer 
    PSS                  Type Manager PostScript Stub (Adobe)
    PSS                  Pocket SlideShow Converted PowerPoint Presentation (CNetX)
    PSS                  AutoCAD Plot Stamp Settings (Autodesk, Inc.)
    PSSD                 PhotoSuite Ver. 5 Slideshow (Sonic Solutions)
    PST                  Mastercam Post Processor File (CNC Software, Inc.)
    PST                  Outlook Personal Folder File (Microsoft Corporation)
    PST                  Ulead Pattern 
    PST                  Lahey Fortran Paste Buffer 
    PST                  BMail Postbox File 
    PST                  Medbasin Parameter File (National Technical University of Athens, Laboratory of Reclamation Works & Water Resources Management)
    PST                  MIDI Studio Preset File (MAGIX)
    PST                  Yamaha Pianist (Solo Piano) Style (Yamaha Corporation of America)
    PST                  LightWave 3D Preset (NewTek)
    PSV                  Psychedelic Screen Saver Graphic (Synthesoft Corporation)
    PSW                  Paint Shop Pro Straw Wall Preset (Corel Corporation)
    PSW                  Print Shop Deluxe Ver. 6 File (Broderbund)
    PSW                  Windows XP Backup Password File (Microsoft Corporation)
    PSW                  Pocket Word Document (Microsoft)
    PSX                  Dirty Little Helper Chat File 
    PSX                  Playstation Single Game Save (Sony Corporation of America)
    PSY                  PSYCLE Tune/Song 
    PSY                  Psychedelic Screen Saver Saved Settings (Synthesoft Corporation)
    PSY                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PSY                  PLD2 System File 
    PSY                  Psychedelic Windows Products 
    PSZ                  Creaf Awedip Doc File 
    PT                   Pitch Track Sound 
    PT                   Kodak Precision Transform 
    PT                   Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PT                   Player Tools Game Crack 
    PT                   Pop!site HTML 
    PT                   PTab Spreadsheet (Z4Soft)
    PT                   PassMark PerformanceTest File 
    PT3                  Protext, Protext///e, and DBFsvr Formatted Text Data File (A. Dombi Engineering /A/D/E/)
    PT3                  PageMaker Version 3 Document Template (Adobe)
    PT3                  Harvard Graphics Device Driver (Harvard Graphics)
    PT4                  ProtoTRAK Design Control File (Southwestern Industries, Inc.)
    PT4                  PageMaker Version 4 Document Template (Adobe)
    PT5                  PageMaker Version 5 Document Template (Adobe)
    PT6                  PageMaker Version 6 Document Template (Adobe)
    PTB                  PubTach Batch Works Script 
    PTB                  Pro/ENGINEER Table (PTC)
    PTB                  Power Tab Guitar Tablature Editor 
    PTB                  Peachtree Complete Accounting Backup Data File (Best Software SB, Inc.)
    PTBL                 PIPE-FLO Professional Pipe Data Table (Engineered Software)
    PTC                  NRG Symphonie Patch File 
    PTC                  PFXplus Compiled Program (POWERflex Corporation)
    PTC                  ABBYY Finereader 5.0 Pro (ABBYY Software House)
    PTD                  PTCS Data File (DATAIR Employee Benefit Systems, Inc.)
    PTD                  Pro/ENGINEER Temporary File (PTC)
    PTD                  Pro/ENGINEER Table (PTC)
    PTDB                 Peachtree Accounting Database 
    PTE                  Pop!site 
    PTE                  Picture to EXE Project 
    PTF                  ProTracker Tennis Match File (Fieldtown Software)
    PTF                  LiveNote Legal Transcript File (LiveNote Technologies)
    PTG                  ArtRage Project (Ambient Design Ltd.)
    PTH                  VisualGPSXP Path File (VisualGPS, LLC.)
    PTH                  4th Dimension Database Windows Pathname Document (4D, Inc.)
    PTI                  PTI-Plug-in Graphic 
    PTI                  Pop!site 
    PTI                  Punch! Professional Home Design Texture Info File (Punch! Software)
    PTI                  IBM Configurator Configuration 
    PTIF                 Pyramid Encoded TIFF 
    PTK                  Puntotek Cross-Stitch Pattern (Mediatek Sistemas S.C.)
    PTK                  Red Alert 2 File 
    PTK                  Quicken On-line Data File (Intuit Inc.)
    PTL                  Python Template Language File 
    PTL                  Visual Modeler Petal File (Microsoft)
    PTL                  Corel Painter Pattern, Selection or Texture File 
    PTL                  Paint Shop Pro Tile Preset (Corel Corporation)
    PTL                  Painter4 File 
    PTL                  Site Builder Path Translation List 
    PTL                  Rational Rose Export/Import File (IBM)
    PTM                  Page Magic Ver. 2.0 Template 
    PTM                  PolyTracker Music Module 
    PTM                  PubTech BatchWorks Macro 
    PTM                  PTMac Project File (Kekus Digital)
    PTM                  Module audio Poly Tracker 
    PTM                  MapPoint Map (Microsoft Corporation)
    PTM                  Polynomial Texture Map 
    PTMC                 PTMac Project File (Kekus Digital)
    PTN                  PaperPort Thumbnail Images (ScanSoft)
    PTN                  Amaya Dicopar File 
    PTN                  CADKEY Pattern File (Kubotek USA Inc.)
    PTN                  Housecall Virus Pattern File (Trend Micro, Inc.)
    PTP                  C Poet Examples Advanced File 
    PTP                  ToolBook Instructor/Assistant Book (SumTotal Systems, Inc.)
    PTP                  ACT! Modem Sync File (Best Software CRM Division)
    PTP                  Primo Computer Emulator Tape File 
    PTR                  OS/2 Cursor Image 
    PTR                  QMail Qwk Reader Pointer 
    PTRON                Proxomitron Configuration File (Scott R. Lemmon (deceased))
    PTS                  Halflife Map Creation Debug File 
    PTS                  Infinity Engine Game Tileset 
    PTS                  3D Points File 
    PTS                  ABBYY Finereader 5.0 Pro (ABBYY Software House)
    PTT                  MapPoint Template (Microsoft)
    PTT                  Context4 DOS Word Processor Print File 
    PTU                  Performer Terrain Utilities 
    PTX                  Punch! Professional Home Design Texture File (Punch! Software)
    PTX                  Pentax RAW Bitmap Graphic (PENTAX Corporation)
    PTX                  PTCS Index File (DATAIR Employee Benefit Systems, Inc.)
    PTX                  Printronix Bitmap 
    PTX                  E-Transcript of Court Proceedings 
    PTX                  Pop!site 
    PTX                  Paint Shop Pro Texture Preset (Corel Corporation)
    PTX                  Rayman2 File 
    PTY                  Rational Rose 98 Properties (IBM)
    PTY                  PockeTTY Saved Session (DejaVu Software)
    PUB                  Pretty Good Privacy (PGP) Public Key Ring (PGP Corporation)
    PUB                  Ventura Publisher (Corel Corporation)
    PUB                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PUB                  The Sims Family Data (Electronic Arts (Maxis))
    PUB                  NeoBook NBW Publication Source File (NeoSoft Corp.)
    PUB                  Print Perfect Gold Template 
    PUB                  Publisher Document (Microsoft Corporation)
    PUBHTML              Publisher HTML Document (Microsoft)
    PUBKR                Pretty Good Privacy (PGP) Public Keyring (PGP Corporation)
    PUBLIC               Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PUBLISHED            Project Server-Published Project (Microsoft Corporation)
    PUBMHTML             Publisher Archived HTML Document (Microsoft)
    PUD                  Warcraft III PUD Map (Blizzard Entertainment)
    PUF                  Puffer Encrypted File 
    PUI                  Publish-iT Save File (Poster Software)
    PUL                  Dinfo File 
    PULLDOWN             Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PUMPUPJAM            Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PUNC                 Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PUNC2                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PUP                  Playstation System Software Update (Sony Corporation of America)
    PURBLEPAIRSSAVE-MS   Purble Place Pairs Save File (Microsoft Corporation)
    PURBLESHOPSAVE-MS    Purble Place Shop Save File (Microsoft Corporation)
    PURGE                Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PUT                  Wscan Graphic 
    PUT                  PUT Compressed File Archive (MicroFox Company)
    PUT                  Instalit Script 
    PUZ                  BrainsBreaker Puzzle 
    PUZ                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PUZ                  Puzzle 
    PUZ                  Packed Publisher File (Microsoft)
    PUZ                  Across Lite Crossword Puzzle 
    PUZ                  General Name for a Puzzle File 
    PUZZ                 X11 Puzzle Bitmap 
    PUZZLE               X11 Puzzle Bitmap 
    PV                   Phase Vocorder Analysis Data 
    PV                   CONSEL Output P-Value File (Dept. of Mathematical and Computing Sciences Tokyo Institute of Technology)
    PVA                  Process Panel Builder Application File (Rockwell Automation)
    PVA                  Hauppauge DVB-Software 
    PVA                  PVA Video (TechnoTrend AG)
    PVB                  VirtualBoss PocketPC Data (VirtualBoss Development Co.)
    PVC                  Process Panel Builder Application File (Rockwell Automation)
    PVD                  PageRecall (Authentica)
    PVD                  PacketView Pro Packet Trace Data (Klos Technologies, Inc.)
    PVD                  ParaView Data Object (Kitware, Inc.)
    PVD                  Packet Analyzer Data File (Klos Technologies)
    PVD                  Picture Browser Cache (Canon Software Publishing)
    PVD                  Install-It Script 
    PVE                  GoBeProductive Document (Gobe Corporation, Inc.)
    PVF                  PageRecall (Authentica)
    PVG                  Encarta World Atlas Pushpins (Microsoft)
    PVI                  POD-Bot Knowledge File 
    PVI                  Picture Browser Cache Index (Canon Software Publishing)
    PVI                  Wave Corrector Anti-piracy File (Ganymede Test & Measurement)
    PVI                  PVElite Pressure Vessel Analysis File (COADE, Inc.)
    PVK                  PVK Private Key (Microsoft)
    PVL                  Instalit Library 
    PVM                  Sonic Adventure DX Game Object (Sega of America, Inc.)
    PVM                  OSTA.org MusicPhotoVideo (OSTA (Optical Storage Technology Association))
    PVP                  Pan&Vision Project (Akromon)
    PVR                  PVR-CONV 
    PVR                  Sonic Adventure DX (Possibly) Level File (Sega of America, Inc.)
    PVS                  Scarlett Expert System Compiler Reserved Words File 
    PVS                  PolyView Slide Show Script (Polybytes, Inc.)
    PVS                  Parallels Desktop for Mac Virtual Machine Configuration File (Parallels, Inc.)
    PVT                  Local Fidonet Pointlist 
    PVTI                 ParaView Parallel (Partitioned) VTK Image Data (Kitware, Inc.)
    PVTK                 ParaView Parallel (Partitioned) Legacy VTK File (Kitware, Inc.)
    PVTP                 ParaView Parallel (Partitioned) VTK Polygonal Data (Kitware, Inc.)
    PVTR                 ParaView Parallel (Partitioned) VTK Rectilinear Grid (Kitware, Inc.)
    PVTS                 ParaView Parallel (Partitioned) VTK Structured Grid (Kitware, Inc.)
    PVTU                 ParaView Parallel (Partitioned) VTK Unstructured Data (Kitware, Inc.)
    PVU                  PaleoVu Paleoenvironmental Data 
    PVW                  Portfolio Saved View (Extensis, Inc.)
    PVX                  ParaView Configuration XML File (Kitware, Inc.)
    PW                   Professional Write Text File 
    PWA                  Password Agent File 
    PWB                  Password Boss Data File (Ammosoft Software)
    PWB                  PhotoWeb Image File (Magnitude)
    PWC                  Pulse Web Content (Pulse Entertainment, Inc.)
    PWC                  COSMI Street Maps and Vacation Planner 
    PWC                  Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PWCHECK              Unknown Apple II File (found on Golden Orchard Apple II CD Rom) 
    PWD                  TBAV File 
    PWD                  Password Pro File 
    PWD                  Pocket Word Document (Microsoft)
    PWD                  AutoCAD Password (Autodesk, Inc.)
    PWD                  Solid Edge Document (UGS PLM Solutions)
    PWDER                Personal Passworder Database (DVAsoft)
    PWF                  ProCite Workforms 
    PWF                  POD-Bot Waypoint File 
    PWI                  InkWriter/Note Taker/Pocket Word Document (Microsoft Corporation)
    PWI                  PowerINSPECT Saved Format File (Delcam plc.)
    PWINSURED            Policy Works Insured File (Tec4 Systems Inc.)
    PWK                  Password Keeper File 
    PWL                  Windows Password List (Microsoft)
    PWP                  Smith Corona Word Processor File (Smith Corona)
    PWP                  Seattle Film Works Multi-image File 
    PWP                  Photoworks Image File 
    PWP                  Professional WritePlus Document 
    PWRK                 Apple IIGS Image (Apple Computer, Inc.)
    PWS                  ProWorks(DB) Start File (Everett Charles Technologies)
    PWT                  Password Tracker Deluxe Tracking List (Carl L. Roth)
    PWT                  HotDog PageWiz Template 
    PWT                  Pocket Word Template File (Microsoft)
    PWT                  AutoCAD Publish-to-Web Template (Autodesk, Inc.)
    PWX                  Poles 'n' Wires Registration Data File (PowerMation)
    PWZ                  PowerPoint Wizard (Microsoft)
    PX                   Primary Index 
    PX                   Paradox Primary Database Index (Corel)
    PX                   PC-Axis Family Statistical Data (Statistics Sweden)
    PX0                  Production Studio Patch File (Babya)
    PX1                  Image Pixel Paint 
    PX1                  Production Studio Melody Sequence File (Babya)
    PXA                  Pegs 
    PXA                  Pixia Color Painter File 
    PXB                  Pixibox 
    PXC                  PX-Edit Classifications (Statistics Finland)
    PXD                  ProxyDesigner Design File (ProxySource, Inc.)
    PXD                  eJay Mixer File 
    PXE                  Partial XML Envelope 
    PXF                  Tajima DG/ML Embroidery Design File (Pulse Microsystems Ltd.)
    PXF                  STAR Transit Translation Project (STAR Medien AG)
    PXI                  $$D2K.PXI associated with a Trojan 
    PXI                  Pixie File 
    PXI                  PlexTools CD Image (Plextor)
    PXJ                  Pixtran File 
    PXJ                  Sonic RecordNow! Project (Hewlett-Packard Development Company, L.P.