/*
 * Decompiled with CFR 0.152.
 */
package org.alfresco.repo.lock;

import java.util.TreeSet;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.alfresco.repo.domain.locks.LockDAO;
import org.alfresco.repo.lock.JobLockService;
import org.alfresco.repo.lock.LockAcquisitionException;
import org.alfresco.repo.transaction.AlfrescoTransactionSupport;
import org.alfresco.repo.transaction.RetryingTransactionHelper;
import org.alfresco.repo.transaction.TransactionListenerAdapter;
import org.alfresco.repo.transaction.TransactionalResourceHelper;
import org.alfresco.service.namespace.QName;
import org.alfresco.util.GUID;
import org.alfresco.util.TraceableThreadFactory;
import org.alfresco.util.VmShutdownListener;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class JobLockServiceImpl
implements JobLockService {
    private static final String KEY_RESOURCE_LOCKS = "JobLockServiceImpl.Locks";
    private static Log logger = LogFactory.getLog(JobLockServiceImpl.class);
    private LockDAO lockDAO;
    private RetryingTransactionHelper retryingTransactionHelper;
    private int defaultRetryCount = 10;
    private long defaultRetryWait = 20L;
    private ScheduledExecutorService scheduler;
    private VmShutdownListener shutdownListener;
    private final LockTransactionListener txnListener = new LockTransactionListener();

    public JobLockServiceImpl() {
        TraceableThreadFactory threadFactory = new TraceableThreadFactory();
        threadFactory.setThreadDaemon(false);
        threadFactory.setNamePrefix("JobLockService");
        this.scheduler = Executors.newSingleThreadScheduledExecutor((ThreadFactory)threadFactory);
        this.shutdownListener = new VmShutdownListener("JobLockService");
    }

    public void shutdown() {
        if (logger.isInfoEnabled()) {
            logger.info((Object)"shutting down.");
        }
        this.scheduler.shutdown();
    }

    public void setLockDAO(LockDAO lockDAO) {
        this.lockDAO = lockDAO;
    }

    public void setRetryingTransactionHelper(RetryingTransactionHelper retryingTransactionHelper) {
        this.retryingTransactionHelper = retryingTransactionHelper;
    }

    public void setDefaultRetryCount(int defaultRetryCount) {
        this.defaultRetryCount = defaultRetryCount;
    }

    public void setDefaultRetryWait(long defaultRetryWait) {
        this.defaultRetryWait = defaultRetryWait;
    }

    @Override
    public void getTransactionalLock(QName lockQName, long timeToLive) {
        this.getTransactionalLock(lockQName, timeToLive, this.defaultRetryWait, this.defaultRetryCount);
    }

    @Override
    public void getTransactionalLock(QName lockQName, long timeToLive, long retryWait, int retryCount) {
        String txnId = AlfrescoTransactionSupport.getTransactionId();
        if (txnId == null) {
            throw new IllegalStateException("Locking requires an active transaction");
        }
        TreeSet heldLocks = TransactionalResourceHelper.getTreeSet(KEY_RESOURCE_LOCKS);
        TreeSet heldLocksTemp = new TreeSet(heldLocks);
        boolean added = heldLocksTemp.add(lockQName);
        if (!added) {
            this.refreshLock(txnId, lockQName, timeToLive);
        } else {
            QName lastLock = (QName)heldLocksTemp.last();
            if (lastLock.equals((Object)lockQName)) {
                if (logger.isDebugEnabled()) {
                    logger.debug((Object)("Attempting to acquire ordered lock: \n   Lock:     " + lockQName + "\n" + "   TTL:      " + timeToLive + "\n" + "   Txn:      " + txnId));
                }
                this.getLockImpl(txnId, lockQName, timeToLive, retryWait, retryCount);
            } else {
                if (logger.isDebugEnabled()) {
                    logger.debug((Object)("Attempting to acquire UNORDERED lock: \n   Lock:     " + lockQName + "\n" + "   TTL:      " + timeToLive + "\n" + "   Txn:      " + txnId));
                }
                this.getLockImpl(txnId, lockQName, timeToLive, retryWait, 1);
            }
        }
        heldLocks.add(lockQName);
    }

    @Override
    public String getLock(QName lockQName, long timeToLive) {
        return this.getLock(lockQName, timeToLive, this.defaultRetryWait, this.defaultRetryCount);
    }

    @Override
    public String getLock(QName lockQName, long timeToLive, long retryWait, int retryCount) {
        String lockToken = GUID.generate();
        this.getLockImpl(lockToken, lockQName, timeToLive, retryWait, retryCount);
        return lockToken;
    }

    @Override
    public void refreshLock(final String lockToken, final QName lockQName, final long timeToLive) {
        RetryingTransactionHelper.RetryingTransactionCallback<Object> refreshLockCallback = new RetryingTransactionHelper.RetryingTransactionCallback<Object>(){

            @Override
            public Object execute() throws Throwable {
                JobLockServiceImpl.this.lockDAO.refreshLock(lockQName, lockToken, timeToLive);
                return null;
            }
        };
        try {
            this.retryingTransactionHelper.doInTransaction(refreshLockCallback, false, true);
            if (logger.isDebugEnabled()) {
                logger.debug((Object)("Refreshed Lock: \n   Lock:     " + lockQName + "\n" + "   TTL:      " + timeToLive + "\n" + "   Txn:      " + lockToken));
            }
        }
        catch (LockAcquisitionException e) {
            if (logger.isDebugEnabled()) {
                logger.debug((Object)("Lock refresh failed: \n   Lock:     " + lockQName + "\n" + "   TTL:      " + timeToLive + "\n" + "   Txn:      " + lockToken + "\n" + "   Error:    " + e.getMessage()));
            }
            throw e;
        }
    }

    @Override
    public void refreshLock(final String lockToken, final QName lockQName, final long timeToLive, final JobLockService.JobLockRefreshCallback callback) {
        if (this.scheduler.isShutdown() || this.scheduler.isTerminated()) {
            if (logger.isDebugEnabled()) {
                logger.debug((Object)("Lock refresh failed: \n   Lock:     " + lockQName + "\n" + "   TTL:      " + timeToLive + "\n" + "   Txn:      " + lockToken + "\n" + "   Error:    " + "Lock refresh scheduler has shut down.  The VM may be terminating."));
            }
            throw new LockAcquisitionException(lockQName, lockToken);
        }
        final long delay = timeToLive / 2L;
        if (delay < 1L) {
            throw new IllegalArgumentException("Very small timeToLive: " + timeToLive);
        }
        Runnable runnable = new Runnable(){

            @Override
            public void run() {
                if (logger.isDebugEnabled()) {
                    logger.debug((Object)("Initiating timed Lock refresh: \n   Lock:     " + lockQName + "\n" + "   TTL:      " + timeToLive + "\n" + "   Txn:      " + lockToken));
                }
                if (JobLockServiceImpl.this.shutdownListener.isVmShuttingDown()) {
                    JobLockServiceImpl.this.callLockReleased(callback);
                    return;
                }
                boolean isActive = false;
                try {
                    isActive = JobLockServiceImpl.this.callIsActive(callback, delay);
                }
                catch (Throwable e) {
                    logger.error((Object)("Lock isActive check failed: \n   Lock:     " + lockQName + "\n" + "   TTL:      " + timeToLive + "\n" + "   Txn:      " + lockToken), e);
                    JobLockServiceImpl.this.callLockReleased(callback);
                    return;
                }
                if (!isActive) {
                    if (logger.isDebugEnabled()) {
                        logger.debug((Object)("Lock callback is inactive.  Releasing lock: \n   Lock:     " + lockQName + "\n" + "   TTL:      " + timeToLive + "\n" + "   Txn:      " + lockToken));
                    }
                    try {
                        JobLockServiceImpl.this.releaseLock(lockToken, lockQName);
                        JobLockServiceImpl.this.callLockReleased(callback);
                    }
                    catch (LockAcquisitionException e) {}
                } else {
                    try {
                        JobLockServiceImpl.this.refreshLock(lockToken, lockQName, timeToLive);
                        JobLockServiceImpl.this.scheduler.schedule(this, delay, TimeUnit.MILLISECONDS);
                    }
                    catch (LockAcquisitionException e) {
                        JobLockServiceImpl.this.callLockReleased(callback);
                    }
                }
            }
        };
        this.scheduler.schedule(runnable, delay, TimeUnit.MILLISECONDS);
    }

    private boolean callIsActive(JobLockService.JobLockRefreshCallback callback, long delay) throws Throwable {
        long t1 = System.nanoTime();
        boolean isActive = callback.isActive();
        long t2 = System.nanoTime();
        double timeWastedMs = (double)(t2 - t1) / 1.0E7;
        if (timeWastedMs > (double)delay || timeWastedMs > 1000.0) {
            throw new RuntimeException("isActive check took " + timeWastedMs + " to return, which is too long.");
        }
        return isActive;
    }

    private void callLockReleased(JobLockService.JobLockRefreshCallback callback) {
        try {
            callback.lockReleased();
        }
        catch (Throwable ee) {
            logger.error((Object)"Callback to lockReleased failed.", ee);
        }
    }

    @Override
    public void releaseLock(final String lockToken, final QName lockQName) {
        RetryingTransactionHelper.RetryingTransactionCallback<Void> releaseCallback = new RetryingTransactionHelper.RetryingTransactionCallback<Void>(){

            @Override
            public Void execute() throws Throwable {
                JobLockServiceImpl.this.lockDAO.releaseLock(lockQName, lockToken);
                return null;
            }
        };
        this.retryingTransactionHelper.doInTransaction(releaseCallback, false, true);
    }

    private void getLockImpl(final String lockToken, final QName lockQName, final long timeToLive, long retryWait, int retryCount) {
        if (retryCount < 0) {
            throw new IllegalArgumentException("Job lock retry count cannot be negative: " + retryCount);
        }
        RetryingTransactionHelper.RetryingTransactionCallback<Object> getLockCallback = new RetryingTransactionHelper.RetryingTransactionCallback<Object>(){

            @Override
            public Object execute() throws Throwable {
                JobLockServiceImpl.this.lockDAO.getLock(lockQName, lockToken, timeToLive);
                return null;
            }
        };
        try {
            int iterations = this.doWithRetry((RetryingTransactionHelper.RetryingTransactionCallback<? extends Object>)getLockCallback, retryWait, retryCount);
            if (AlfrescoTransactionSupport.getTransactionReadState() != AlfrescoTransactionSupport.TxnReadState.TXN_NONE) {
                AlfrescoTransactionSupport.bindListener(this.txnListener);
            }
            if (logger.isDebugEnabled()) {
                logger.debug((Object)("Acquired Lock: \n   Lock:     " + lockQName + "\n" + "   TTL:      " + timeToLive + "\n" + "   Txn:      " + lockToken + "\n" + "   Attempts: " + iterations));
            }
        }
        catch (LockAcquisitionException e) {
            if (logger.isDebugEnabled()) {
                logger.debug((Object)("Lock acquisition failed: \n   Lock:     " + lockQName + "\n" + "   TTL:      " + timeToLive + "\n" + "   Txn:      " + lockToken + "\n" + "   Error:    " + e.getMessage()));
            }
            throw e;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int doWithRetry(RetryingTransactionHelper.RetryingTransactionCallback<? extends Object> callback, long retryWait, int retryCount) {
        int maxAttempts = retryCount > 0 ? retryCount : 1;
        int lockAttempt = 0;
        LockAcquisitionException lastException = null;
        while (++lockAttempt <= maxAttempts) {
            try {
                this.retryingTransactionHelper.doInTransaction(callback, false, true);
                lastException = null;
                break;
            }
            catch (LockAcquisitionException e) {
                if (logger.isDebugEnabled()) {
                    logger.debug((Object)("Lock attempt " + lockAttempt + " of " + maxAttempts + " failed: " + e.getMessage()));
                }
                lastException = e;
                if (lockAttempt >= maxAttempts) break;
                RetryingTransactionHelper.RetryingTransactionCallback<? extends Object> retryingTransactionCallback = callback;
                synchronized (retryingTransactionCallback) {
                    try {
                        callback.wait(retryWait);
                    }
                    catch (InterruptedException e2) {
                        // empty catch block
                    }
                }
            }
        }
        if (lastException == null) {
            return lockAttempt;
        }
        throw lastException;
    }

    private class LockTransactionListener
    extends TransactionListenerAdapter {
        private LockTransactionListener() {
        }

        @Override
        public void beforeCommit(boolean readOnly) {
            final String txnId = AlfrescoTransactionSupport.getTransactionId();
            final TreeSet heldLocks = TransactionalResourceHelper.getTreeSet(JobLockServiceImpl.KEY_RESOURCE_LOCKS);
            if (heldLocks.size() == 0) {
                return;
            }
            RetryingTransactionHelper.RetryingTransactionCallback<Object> releaseCallback = new RetryingTransactionHelper.RetryingTransactionCallback<Object>(){

                @Override
                public Object execute() throws Throwable {
                    for (QName lockQName : heldLocks) {
                        JobLockServiceImpl.this.lockDAO.releaseLock(lockQName, txnId);
                    }
                    return null;
                }
            };
            JobLockServiceImpl.this.retryingTransactionHelper.doInTransaction(releaseCallback, false, true);
            heldLocks.clear();
        }

        @Override
        public void afterRollback() {
            final String txnId = AlfrescoTransactionSupport.getTransactionId();
            TreeSet<QName> heldLocks = TransactionalResourceHelper.getTreeSet(JobLockServiceImpl.KEY_RESOURCE_LOCKS);
            if (heldLocks.size() == 0) {
                return;
            }
            for (final QName lockQName : heldLocks) {
                RetryingTransactionHelper.RetryingTransactionCallback<Object> releaseCallback = new RetryingTransactionHelper.RetryingTransactionCallback<Object>(){

                    @Override
                    public Object execute() throws Throwable {
                        JobLockServiceImpl.this.lockDAO.releaseLock(lockQName, txnId);
                        return null;
                    }
                };
                try {
                    JobLockServiceImpl.this.retryingTransactionHelper.doInTransaction(releaseCallback, false, true);
                }
                catch (Throwable e) {
                    logger.warn((Object)("Failed to release a lock in 'afterRollback':\n   Lock Name:  " + lockQName + "\n" + "   Lock Token: " + txnId), e);
                }
            }
        }
    }
}

