/*
 * Copyright (C) 2005-2010 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */
package org.alfresco.ibatis;

import java.io.InputStream;
import java.util.Properties;

import org.alfresco.util.resource.HierarchicalResourceLoader;
import org.apache.ibatis.builder.BaseBuilder;
import org.apache.ibatis.builder.BuilderException;
import org.apache.ibatis.builder.xml.XMLConfigBuilder;
import org.apache.ibatis.builder.xml.XMLMapperBuilder;
import org.apache.ibatis.builder.xml.XMLMapperEntityResolver;
import org.apache.ibatis.datasource.DataSourceFactory;
import org.apache.ibatis.executor.ErrorContext;
import org.apache.ibatis.io.Resources;
import org.apache.ibatis.mapping.Environment;
import org.apache.ibatis.parsing.XNode;
import org.apache.ibatis.parsing.XPathParser;
import org.apache.ibatis.plugin.Interceptor;
import org.apache.ibatis.reflection.MetaClass;
import org.apache.ibatis.reflection.factory.ObjectFactory;
import org.apache.ibatis.reflection.wrapper.ObjectWrapperFactory;
import org.apache.ibatis.session.AutoMappingBehavior;
import org.apache.ibatis.session.Configuration;
import org.apache.ibatis.session.ExecutorType;
import org.apache.ibatis.transaction.TransactionFactory;
import org.apache.ibatis.type.TypeHandler;
import org.mybatis.spring.SqlSessionFactoryBean;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;

/**
 * Extends the MyBatis XMLConfigBuilder to allow the selection of a {@link ResourceLoader}
 * that will be used to load the resources specified in the <b>mapper</b>'s <b>resource</b>.
 * <p>
 * By using the <b>resource.dialect</b> placeholder with hierarchical resource loading,
 * different resource files can be picked up for different dialects. This reduces duplication
 * when supporting multiple database configurations.
 * <pre>
 * &lt;configuration&gt;
 *    &lt;mappers&gt;
 *        &lt;mapper resource=&quot;org/x/y/#resource.dialect#/View1.xml&quot;/&gt;
 *        &lt;mapper resource=&quot;org/x/y/#resource.dialect#/View2.xml&quot;/&gt;
 *    &lt;/mappers&gt;
 * &lt;/configuration&gt;
 * <p/>
 * 
 * Much of the implementation is a direct copy of the MyBatis {@link XMLConfigBuilder}; some
 * of the <tt>protected</tt> methods do not have access to the object's state and can therefore
 * not be overridden successfully: <a href=https://issues.apache.org/jira/browse/IBATIS-589>IBATIS-589</a>

 * Pending a better way to extend/override, much of the implementation is a direct copy of the MyBatis 
 * {@link SqlSessionFactoryBean}; some of the <tt>protected</tt> methods do not have access to the object's state 
 * and can therefore not be overridden successfully. 
 * 
 * This is equivalent to HierarchicalSqlMapConfigParser which extended iBatis (2.x).
 * See also: <a href=https://issues.apache.org/jira/browse/IBATIS-589>IBATIS-589</a>
 * and: <a href=http://code.google.com/p/mybatis/issues/detail?id=21</a>
 * 
 * @author Derek Hulley, janv
 * @since 4.0
 */
// note: effectively extends XMLConfigBuilder to use hierarchical resource loader
public class HierarchicalXMLConfigBuilder extends BaseBuilder
{
    final private HierarchicalResourceLoader resourceLoader;
    
    public HierarchicalXMLConfigBuilder(HierarchicalResourceLoader resourceLoader, InputStream inputStream, String environment, Properties props, boolean useLocalCaches)
    {
        super(new Configuration());
        
        // EXTENDED
        this.resourceLoader = resourceLoader;
        
        ErrorContext.instance().resource("SQL Mapper Configuration");
        this.configuration.setVariables(props);
        this.parsed = false;
        this.environment = environment;
        this.useLocalCaches = useLocalCaches;
        this.parser = new XPathParser(inputStream, true, props, new XMLMapperEntityResolver());
    }

    private boolean parsed;
    private XPathParser parser;
    private String environment;
    private boolean useLocalCaches = false;

    public Configuration parse() {
      if (parsed) {
        throw new BuilderException("Each MapperConfigParser can only be used once.");
      }
      parsed = true;
      parseConfiguration(parser.evalNode("/configuration"));
      return getConfiguration();
    }

    private void parseConfiguration(XNode root) {
      try {
        typeAliasesElement(root.evalNode("typeAliases"));
        pluginElement(root.evalNode("plugins"));
        objectFactoryElement(root.evalNode("objectFactory"));
        objectWrapperFactoryElement(root.evalNode("objectWrapperFactory"));
        propertiesElement(root.evalNode("properties"));
        settingsElement(root.evalNode("settings"));
        environmentsElement(root.evalNode("environments"));
        typeHandlerElement(root.evalNode("typeHandlers"));
        mapperElement(root.evalNode("mappers"));

        configuration.setUseLocalCaches(useLocalCaches);
      } catch (Exception e) {
        throw new BuilderException("Error parsing SQL Mapper Configuration. Cause: " + e, e);
      }
    }

    private void typeAliasesElement(XNode parent) {
      if (parent != null) {
        for (XNode child : parent.getChildren()) {
          String alias = child.getStringAttribute("alias");
          String type = child.getStringAttribute("type");
          try {
            Class<?> clazz = Resources.classForName(type);
            if (alias == null) {
              typeAliasRegistry.registerAlias(clazz);
            } else {
              typeAliasRegistry.registerAlias(alias, clazz);
            }
          } catch (ClassNotFoundException e) {
            throw new BuilderException("Error registering typeAlias for '" + alias + "'. Cause: " + e, e);
          }
        }
      }
    }

    private void pluginElement(XNode parent) throws Exception {
      if (parent != null) {
        for (XNode child : parent.getChildren()) {
          String interceptor = child.getStringAttribute("interceptor");
          Properties properties = child.getChildrenAsProperties();
          Interceptor interceptorInstance = (Interceptor) resolveClass(interceptor).newInstance();
          interceptorInstance.setProperties(properties);
          configuration.addInterceptor(interceptorInstance);
        }
      }
    }


    private void objectFactoryElement(XNode context) throws Exception {
      if (context != null) {
        String type = context.getStringAttribute("type");
        Properties properties = context.getChildrenAsProperties();
        ObjectFactory factory = (ObjectFactory) resolveClass(type).newInstance();
        factory.setProperties(properties);
        configuration.setObjectFactory(factory);
      }
    }
    
    private void objectWrapperFactoryElement(XNode context) throws Exception {
      if (context != null) {
        String type = context.getStringAttribute("type");
        ObjectWrapperFactory factory = (ObjectWrapperFactory) resolveClass(type).newInstance();
        configuration.setObjectWrapperFactory(factory);
      }
    }

    private void propertiesElement(XNode context) throws Exception {
      if (context != null) {
        Properties defaults = context.getChildrenAsProperties();
        String resource = context.getStringAttribute("resource");
        String url = context.getStringAttribute("url");
        if (resource != null && url != null) {
          throw new BuilderException("The properties element cannot specify both a URL and a resource based property file reference.  Please specify one or the other.");
        }
        if (resource != null) {
          defaults.putAll(Resources.getResourceAsProperties(resource));
        } else if (url != null) {
          defaults.putAll(Resources.getUrlAsProperties(url));
        }
        Properties vars = configuration.getVariables();
        if (vars != null) {
          defaults.putAll(vars);
        }
        parser.setVariables(defaults);
        configuration.setVariables(defaults);
      }
    }

    private void settingsElement(XNode context) throws Exception {
      if (context != null) {
        Properties props = context.getChildrenAsProperties();
        // Check that all settings are known to the configuration class
        MetaClass metaConfig = MetaClass.forClass(Configuration.class);
        for (Object key : props.keySet()) {
          if (!metaConfig.hasSetter(String.valueOf(key))) {
            throw new BuilderException("The setting " + key + " is not known.  Make sure you spelled it correctly (case sensitive).");
          }
        }
        configuration.setAutoMappingBehavior(AutoMappingBehavior.valueOf(stringValueOf(props.getProperty("autoMappingBehavior"), "PARTIAL")));
        configuration.setCacheEnabled(booleanValueOf(props.getProperty("cacheEnabled"), true));
        configuration.setLazyLoadingEnabled(booleanValueOf(props.getProperty("lazyLoadingEnabled"), false));
        configuration.setAggressiveLazyLoading(booleanValueOf(props.getProperty("aggressiveLazyLoading"), true));
        configuration.setMultipleResultSetsEnabled(booleanValueOf(props.getProperty("multipleResultSetsEnabled"), true));
        configuration.setUseColumnLabel(booleanValueOf(props.getProperty("useColumnLabel"), true));
        configuration.setUseGeneratedKeys(booleanValueOf(props.getProperty("useGeneratedKeys"), false));
        configuration.setDefaultExecutorType(ExecutorType.valueOf(stringValueOf(props.getProperty("defaultExecutorType"), "SIMPLE")));
        configuration.setDefaultStatementTimeout(integerValueOf(props.getProperty("defaultStatementTimeout"), null));
      }
    }

    private void environmentsElement(XNode context) throws Exception {
      if (context != null) {
        if (environment == null) {
          environment = context.getStringAttribute("default");
        }
        for (XNode child : context.getChildren()) {
          String id = child.getStringAttribute("id");
          if (isSpecifiedEnvironment(id)) {
            TransactionFactory txFactory = transactionManagerElement(child.evalNode("transactionManager"));
            DataSourceFactory dsFactory = dataSourceElement(child.evalNode("dataSource"));
            Environment.Builder environmentBuilder = new Environment.Builder(id)
                .transactionFactory(txFactory)
                .dataSource(dsFactory.getDataSource());
            configuration.setEnvironment(environmentBuilder.build());
          }
        }
      }
    }

    private TransactionFactory transactionManagerElement(XNode context) throws Exception {
      if (context != null) {
        String type = context.getStringAttribute("type");
        Properties props = context.getChildrenAsProperties();
        TransactionFactory factory = (TransactionFactory) resolveClass(type).newInstance();
        factory.setProperties(props);
        return factory;
      }
      throw new BuilderException("Environment declaration requires a TransactionFactory.");
    }

    private DataSourceFactory dataSourceElement(XNode context) throws Exception {
      if (context != null) {
        String type = context.getStringAttribute("type");
        Properties props = context.getChildrenAsProperties();
        DataSourceFactory factory = (DataSourceFactory) resolveClass(type).newInstance();
        factory.setProperties(props);
        return factory;
      }
      throw new BuilderException("Environment declaration requires a DataSourceFactory.");
    }


    private void typeHandlerElement(XNode parent) throws Exception {
      if (parent != null) {
        for (XNode child : parent.getChildren()) {
          String javaType = child.getStringAttribute("javaType");
          String jdbcType = child.getStringAttribute("jdbcType");
          String handler = child.getStringAttribute("handler");

          Class<?> javaTypeClass = resolveClass(javaType);
          TypeHandler typeHandlerInstance = (TypeHandler) resolveClass(handler).newInstance();

          if (jdbcType == null) {
            typeHandlerRegistry.register(javaTypeClass, typeHandlerInstance);
          } else {
            typeHandlerRegistry.register(javaTypeClass, resolveJdbcType(jdbcType), typeHandlerInstance);
          }
        }
      }
    }


    private void mapperElement(XNode parent) throws Exception {
      if (parent != null) {
        for (XNode child : parent.getChildren()) {
          String resource = child.getStringAttribute("resource");
          String url = child.getStringAttribute("url");
          InputStream inputStream;
          if (resource != null && url == null) {
            ErrorContext.instance().resource(resource);
            
            // // EXTENDED
            // inputStream = Resources.getResourceAsStream(resource);
            inputStream = null;
            Resource res = resourceLoader.getResource(resource);
            if (res != null && res.exists())
            {
                inputStream = res.getInputStream();
            }
            else {
                throw new BuilderException("Failed to get resource: "+resource);
            }
            
            XMLMapperBuilder mapperParser = new XMLMapperBuilder(inputStream, configuration, resource, configuration.getSqlFragments());
            mapperParser.parse();
          } else if (url != null && resource == null) {
            ErrorContext.instance().resource(url);
            inputStream = Resources.getUrlAsStream(url);
            XMLMapperBuilder mapperParser = new XMLMapperBuilder(inputStream, configuration, url, configuration.getSqlFragments());
            mapperParser.parse();
          } else {
            throw new BuilderException("A mapper element may only specify a url or resource, but not both.");
          }
        }
      }
    }

    private boolean isSpecifiedEnvironment(String id) {
      if (environment == null) {
        throw new BuilderException("No environment specified.");
      } else if (id == null) {
        throw new BuilderException("Environment requires an id attribute.");
      } else if (environment.equals(id)) {
        return true;
      }
      return false;
    }
}
