/*
 * Copyright (C) 2005-2010 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */
package org.alfresco.web.extensibility;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.springframework.extensions.surf.RequestContext;
import org.springframework.extensions.surf.extensibility.ExtensionModuleEvaluator;

import java.util.Map;


/**
 * <p>
 * Evaluator used to decide if an extension module (and its {@code<components>} & {@code<customizations>}) shall be
 * used for this request.
 * </p>
 *
 * <p>
 * Makes it possible to decide if we are viewed specific sites based on their ids and sitePreset ids by matching them
 * against the regexps inside the {@code<sites>} and {@code<sitePresets>} parameters.
 * </p>
 *
 * <p>
 * Note! If we are outside a side (i.e. a "global/non-site-page"  page: i.e. the "Repository browser", A users dashboard or the
 * "My Workflows" page the evaluator will return <code>true</code> by default. To change this behaviour you can set
 * {@code<applyForNonSites>} to false, which means the evaluator will return true ONLY when inside a site.
 * </p>
 * <p>
 * Note! The regexp is expressed without using the surrounding // characters.
 * </p>
 *
 * <p>
 * Example:
 * </p>
 *
 * <pre>{@code
 * <evaluator>site.module.evaluator</evaluator>
 * <evaluatorProperties>
 *     <sitePresets>rm-site-dashboard</sitePresets>
 * </evaluatorProperties>
 * }</pre>
 *
 * <p>
 * Will return true if we are inside a site with a sitePreset id of "rm-site-dashboard".
 * </p>
 *
 * <p>
 * Example 2:
 * </p>
 *
 * <pre>{@code
 * <evaluator>site.module.evaluator</evaluator>
 * <evaluatorProperties>
 *     <sites>rm|photos</sitePresets>
 * </evaluatorProperties>
 * }</pre>
 *
 * <p>
 * Will return true if we are inside a site with a site id of "rm" or "photos".
 * </p>
 *
 * @author ewinlof
 */
public class SlingshotSiteModuleEvaluator implements ExtensionModuleEvaluator
{
    private static Log logger = LogFactory.getLog(SlingshotSiteModuleEvaluator.class);

    /* Evaluator parameters */
    public static final String SITE_PRESET_FILTER = "sitePresets";
    public static final String SITE_FILTER = "sites";
    public static final String APPLY_FOR_NON_SITES = "applyForNonSites";

    protected SlingshotEvaluatorUtil util = null;

    public void setSlingshotEvaluatorUtil(SlingshotEvaluatorUtil slingshotExtensibilityUtil)
    {
        this.util = slingshotExtensibilityUtil;
    }

    public String[] getRequiredProperties()
    {
        String[] properties = new String[2];
        properties[0] = SITE_PRESET_FILTER;
        properties[1] = SITE_FILTER;
        return properties;
    }

    /**
     * Will return true if we are outside a site OR inside a site with a sitePreset id of "rm-site-dashboard".
     *
     * @param context
     * @param params
     * @return true if we are outside a site OR inside a site with a sitePreset id of "rm-site-dashboard".
     */
    public boolean applyModule(RequestContext context, Map<String, String> params)
    {
        String siteId = util.getSite(context);

        // If we are in a site use site filters
        if (siteId != null)
        {
            // SITE FILTER
            if (!siteId.matches(util.getEvaluatorParam(params, SITE_FILTER, ".*")))
            {
                return false;
            }

            // SITE PRESET FILTER
            String sitePreset = util.getSitePreset(context, siteId);
            if (sitePreset == null || !sitePreset.matches(util.getEvaluatorParam(params, SITE_PRESET_FILTER, ".*")))
            {
                return false;
            }

            // SITE PASSED BOTH SITE ID & SITE PRESET FILTERS
            return true;
        }

        // We are not inside a site, see if we shall apply the module anyhow
        return util.getEvaluatorParam(params, APPLY_FOR_NON_SITES, "true").equals("true");
    }

}
