package com.wewebu.ow.server.ecmimpl;

import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import com.wewebu.ow.server.ao.OwAOProvider;
import com.wewebu.ow.server.ao.OwAOType;
import com.wewebu.ow.server.ao.OwRoleManagedAOType;
import com.wewebu.ow.server.ecm.OwNetwork;
import com.wewebu.ow.server.exceptions.OwAccessDeniedException;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.exceptions.OwServerException;
import com.wewebu.ow.server.role.OwRoleManager;
import com.wewebu.ow.server.util.OwString1;

/**
 *<p>
 *  AOProvider bound to the current network implementation.
 *  Will work like it was defined by delegating to network
 *  to provide the application objects. 
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 *@since 4.0.0.0
 */
@SuppressWarnings({ "unchecked", "rawtypes" })
public class OwBackwardsCompatibilityAOProvider implements OwAOProvider
{
    private OwNetwork network;
    private OwRoleManager roleManager;

    public OwBackwardsCompatibilityAOProvider(OwNetwork adaptor, OwRoleManager roleManager)
    {
        this.network = adaptor;
        this.roleManager = roleManager;
    }

    public <T> List<T> getApplicationObjects(OwAOType<T> type, String name, boolean forceSpecificObj) throws OwException
    {
        Collection col = null;
        try
        {
            col = this.network.getApplicationObjects(type.getType(), getDefaultLocation(type, name, null), forceSpecificObj);
            if (type instanceof OwRoleManagedAOType)
            {
                Collection<T> allowedCollection = new LinkedList<T>();
                OwRoleManagedAOType<T> roleManagedType = (OwRoleManagedAOType<T>) type;
                for (Iterator i = col.iterator(); i.hasNext();)
                {
                    T object = (T) i.next();
                    if (roleManagedType.isAllowed(object, roleManager))
                    {
                        allowedCollection.add(object);
                    }
                }
                col = allowedCollection;
            }
        }
        catch (OwException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new OwServerException("Unknown Problem during application object retrieval", e);
        }

        return new LinkedList<T>(col);
    }

    public <T> T getApplicationObject(OwAOType<T> type, String name, boolean forceSpecificObj, boolean createNonExisting) throws OwException
    {
        Object answer = null;
        try
        {
            answer = this.network.getApplicationObject(type.getType(), name, forceSpecificObj, createNonExisting);
        }
        catch (OwException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new OwServerException("Unknown Problem during application object retrieval", e);
        }

        T tAnswer = (T) answer;
        if (type instanceof OwRoleManagedAOType)
        {
            OwRoleManagedAOType<T> roleManagedType = (OwRoleManagedAOType<T>) type;
            if (!roleManagedType.isAllowed(tAnswer, roleManager))
            {
                throw new OwAccessDeniedException(new OwString1("app.OwAOProvider.acces.denied", "Access to the application object %1 was denied", name));
            }
        }

        return tAnswer;
    }

    public <T> T getApplicationObject(OwAOType<T> aoType, String name, List<Object> params, boolean forceSpecificObj, boolean createNonExisting) throws OwException
    {
        Object answer = null;
        try
        {
            answer = this.network.getApplicationObject(aoType.getType(), name, params, forceSpecificObj, createNonExisting);
        }
        catch (OwException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new OwServerException("Unknown Problem during application object retrieval", e);
        }
        return (T) answer;
    }

    /**
     * Return the default location for specific application object
     * @param aoType OwAOType specific application object type
     * @param name String name or location defined by retrieval (can be null)
     * @param params List of objects for additional information (can be null)
     * @return String location/path for specific objects
     */
    protected String getDefaultLocation(OwAOType<?> aoType, String name, List<Object> params)
    {
        String location = name;
        if (name == null)
        {
            switch (aoType.getType())
            {
                case OwAOConstants.AO_TYPE_SEARCHTEMPLATE:
                    location = "owsearchtemplates";
                    break;
                case OwAOConstants.AO_TYPE_VIRTUAL_FOLDER:
                    location = "other";
                    break;
                default:
                    location = name;
            }
        }
        return location;
    }

}
