package com.wewebu.ow.server.ecmimpl.opencmis.auth;

import java.util.Locale;
import java.util.Map;

import org.apache.chemistry.opencmis.client.runtime.SessionFactoryImpl;
import org.apache.chemistry.opencmis.commons.SessionParameter;
import org.apache.chemistry.opencmis.commons.exceptions.CmisPermissionDeniedException;
import org.apache.chemistry.opencmis.commons.spi.AuthenticationProvider;
import org.apache.log4j.Logger;

import com.wewebu.ow.server.app.OwConfiguration;
import com.wewebu.ow.server.auth.OwAuthentication;
import com.wewebu.ow.server.auth.OwAuthenticator;
import com.wewebu.ow.server.auth.OwCredentialsAuthentication;
import com.wewebu.ow.server.conf.OwBaseInitializer;
import com.wewebu.ow.server.ecmimpl.opencmis.OwCMISNetwork;
import com.wewebu.ow.server.ecmimpl.opencmis.conf.OwCMISNetworkCfg;
import com.wewebu.ow.server.exceptions.OwAuthenticationException;
import com.wewebu.ow.server.exceptions.OwConfigurationException;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.exceptions.OwNotSupportedException;
import com.wewebu.ow.server.exceptions.OwServerException;
import com.wewebu.ow.server.log.OwLogCore;
import com.wewebu.ow.server.util.OwAuthenticationConfiguration;
import com.wewebu.ow.server.util.OwString;
import com.wewebu.ow.server.util.OwXMLUtil;
import com.wewebu.ow.server.util.ldap.OwLdapConnector;

/**
 *<p>
 * CMIS credentials based authenticator. 
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 */
public class OwCMISCredentialsAuthenticator implements OwAuthenticator
{
    private static final String ECM_ADAPTER = "EcmAdapter";

    /** package logger for the class */
    private static final Logger LOG = OwLogCore.getLogger(OwCMISCredentialsAuthenticator.class);

    /**optional configuration node name containing the full qualified java class name*/
    public static final String CONF_NODE_SECURITY_HANDLER = "SecurityHandler";

    private OwBaseInitializer baseInitializer;

    private OwCMISNetwork network;

    /**
    * @param initializer OwBaseInitializer
    */
    public OwCMISCredentialsAuthenticator(OwBaseInitializer initializer)
    {
        super();
        this.baseInitializer = initializer;
    }

    /**
     * @param network
     */
    public OwCMISCredentialsAuthenticator(OwCMISNetwork network)
    {
        super();
        this.network = network;
    }

    public OwAuthentication authenticate(OwAuthentication authentication_p) throws OwAuthenticationException, OwConfigurationException, OwServerException
    {
        String userName = authentication_p.getUserName();
        String password = authentication_p.getPassword();

        if (userName != null && password != null)
        {
            OwCMISCredentials credentials = null;

            try
            {
                SessionFactoryImpl sessionFactory = SessionFactoryImpl.newInstance();
                Map<String, String> opencmisParameters = buildOpenCmisParameters(userName, password, null);
                OwCMISNetworkCfg cfg = getNetworkConfiguration();
                AuthenticationProvider authProv = null;
                if (cfg.getAuthenticationInterceptor() != null)
                {
                    authProv = retrieveAuthenticationProvider(opencmisParameters, cfg);
                    sessionFactory.getRepositories(opencmisParameters, null, authProv, null);
                }
                else
                {
                    sessionFactory.getRepositories(opencmisParameters);
                }
                credentials = createCredentials(userName, password, authProv);
            }
            catch (CmisPermissionDeniedException e)
            {
                LOG.debug("OwCMISCredentialsAuthenticator.authenticate(): Cmis exception unallowed!", e);
                throw new OwAuthenticationException(new OwString("opencmis.auth.OwCMISCredentialsAuthenticator.err.login.failed", "Login failed: Wrong username or password!"), e);
            }
            catch (OwException owe)
            {
                LOG.error("Could not create credentials !", owe);
                throw new OwServerException("Invalid configuration exception!", owe);
            }

            OwCredentialsAuthentication credentialsAuthentication = new OwCredentialsAuthentication(credentials, userName, password);

            return credentialsAuthentication;
        }
        else
        {
            LOG.debug("OwCMISCredentialsAuthenticator.authenticate(): Login failed: Wrong username or password!");
            throw new OwAuthenticationException(new OwString("opencmis.auth.OwCMISCredentialsAuthenticator.err.login.failed", "Login failed: Wrong username or password!"));
        }
    }

    /**
     * Get configuration helper instance.
     * @return OwCMISNetworkCfg
     * @throws OwConfigurationException
     */
    protected OwCMISNetworkCfg getNetworkConfiguration() throws OwConfigurationException
    {
        if (this.network != null)
        {
            return this.network.getNetworkConfiguration();
        }
        else
        {
            OwXMLUtil bootstrapx = OwConfiguration.getBootstrap(baseInitializer);

            if (bootstrapx == null)
            {
                String msg = "No bootstrap.";
                LOG.error("OwCMISRepositoryAuthenticator.authenticate :" + msg);
                throw new OwConfigurationException(msg);
            }

            try
            {
                return new OwCMISNetworkCfg(bootstrapx.getSubUtil(ECM_ADAPTER));
            }
            catch (Exception e)
            {
                String msg = "Invalid bootstrap.";
                LOG.error("OwCMISRepositoryAuthenticator.authenticate :" + msg);
                throw new OwConfigurationException(msg, e);
            }
        }
    }

    /**
     * Create the OpenCmis parameters which are used for authentication verification. 
     * @param user_p String
     * @param pwd_p String
     * @param currentLocale_p Locale (can be null)
     * @return Map of key and value Strings
     * @throws OwConfigurationException
     */
    public Map<String, String> buildOpenCmisParameters(String user_p, String pwd_p, Locale currentLocale_p) throws OwConfigurationException
    {
        Map<String, String> opencmisParameters = getNetworkConfiguration().getBindingConfig();

        opencmisParameters.put(SessionParameter.USER, user_p);
        opencmisParameters.put(SessionParameter.PASSWORD, pwd_p);

        if (currentLocale_p != null && !"debugmode".equals(currentLocale_p.toString()))
        {
            opencmisParameters.put(SessionParameter.LOCALE_ISO3166_COUNTRY, currentLocale_p.getCountry());
            opencmisParameters.put(SessionParameter.LOCALE_ISO639_LANGUAGE, currentLocale_p.getLanguage());
            opencmisParameters.put(SessionParameter.LOCALE_VARIANT, currentLocale_p.getVariant());
        }

        return opencmisParameters;
    }

    /**
     * Factory method for credentials objects.
     * @param user_p String user login name
     * @param password_p String login password
     * @param nativeAuthProvider AuthenticationProvider which should be used for authentication (can be null)
     * @return OwCMISCredentials
     * @throws OwException
     */
    protected OwCMISCredentials createCredentials(String user_p, String password_p, AuthenticationProvider nativeAuthProvider) throws OwException
    {
        return new OwCMISCredentials(user_p, password_p, nativeAuthProvider);
    }

    /**
     * Depending on initialization
     * returning the current network or null
     * @return OwCMISNetwork (can return null)
     */
    protected OwCMISNetwork getNetwork()
    {
        return this.network;
    }

    /**
     * Depending on initialization
     * will return the OwBaseInitializer or null 
     * @return OwBaseInitializer (can return null)
     */
    protected OwBaseInitializer getBaseInitialzier()
    {
        return this.baseInitializer;
    }

    /**
     * Retrieve the (prepared) AuthenticationProvider which should be used for authentication verification and for credentials.
     * <p>By default the provider is created using the network instance, if not available null is returned</p>
     * @param openCmisParams Map of Strings representing configuration parameters 
     * @param netConfig OwCMISNetworkCfg current configuration object 
     * @return AuthenticationProvider which was found, retrieved or created
     * @throws OwException
     */
    protected AuthenticationProvider retrieveAuthenticationProvider(Map<String, String> openCmisParams, OwCMISNetworkCfg netConfig) throws OwException
    {
        if (this.network != null)
        {
            return this.network.getAuthInterceptor().createAuthenticationProvider(openCmisParams);
        }
        else
        {
            throw new OwNotSupportedException("Cannot get prepared org.apache.chemistry.opencmis.commons.spi.AuthenticationProvider without valid Network instance.");
        }
    }

    /**
     * Factory for LDAP connector instances.
     * @param authenticationConf_p OwAuthenticationConfiguration
     * @return OwLdapConnector
     * @throws OwException
     */
    protected OwLdapConnector createLDAPConnector(OwAuthenticationConfiguration authenticationConf_p) throws OwException
    {
        return new OwLdapConnector(authenticationConf_p.getConfiguration().getNode());
    }
}