package com.wewebu.ow.server.ecmimpl.opencmis.auth;

import org.apache.log4j.Logger;

import com.wewebu.ow.server.ecm.OwUserInfo;
import com.wewebu.ow.server.ecm.ui.OwUIUserSelectModul;
import com.wewebu.ow.server.ecmimpl.opencmis.OwCMISNetwork;
import com.wewebu.ow.server.ecmimpl.opencmis.ui.OwCMISUserSelectionModule;
import com.wewebu.ow.server.ecmimpl.opencmis.users.OwUsersRepositoryLDAP;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.exceptions.OwServerException;
import com.wewebu.ow.server.log.OwLogCore;
import com.wewebu.ow.server.util.OwAuthenticationConfiguration;
import com.wewebu.ow.server.util.OwString;
import com.wewebu.ow.server.util.ldap.OwLdapConnector;

/**
 *<p>
 * LDAP based authentication provider for the Open CMIS adapter.
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 */
public class OwCMISLDAPAuthenticationProvider extends OwCMISDefaultAuthenticationProvider
{
    private static final Logger LOG = OwLogCore.getLogger(OwCMISLDAPAuthenticationProvider.class);
    private OwLdapConnector ldapconnector;
    private OwUsersRepositoryLDAP userRepository;

    /**
     * @param network
     */
    public OwCMISLDAPAuthenticationProvider(OwCMISNetwork network)
    {
        super(network);
    }

    /* (non-Javadoc)
     * @see com.wewebu.ow.server.ecmimpl.opencmis.auth.OwCMISDefaultAuthenticationProvider#loginDefault(java.lang.String, java.lang.String)
     */
    @Override
    public void loginDefault(String username_p, String password_p) throws OwException
    {
        try
        {
            OwAuthenticationConfiguration authConf = this.network.getNetworkConfiguration().getAuthenticationConfiguration();
            OwLdapConnector ldapconnector = createLDAPConnector(authConf);
            this.ldapconnector = ldapconnector;
            this.userRepository = new OwUsersRepositoryLDAP(this.ldapconnector);
        }
        catch (OwException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            LOG.error("Invalid Credentials - maybe there are problems on reading the network config node/subnode \"(Authentication)\"...", e);
            throw new OwServerException(new OwString("opencmis.OwCMISLDAPAuthenticationProvider.invalid.authentication.configuration.error", "Invalid CMIS authentication configuration!"), e);
        }

        super.loginDefault(username_p, password_p);
    }

    /* (non-Javadoc)
     * @see com.wewebu.ow.server.ecmimpl.opencmis.auth.OwCMISDefaultAuthenticationProvider#canUserSelect()
     */
    @Override
    public boolean canUserSelect() throws OwException
    {
        return true;
    }

    /* (non-Javadoc)
     * @see com.wewebu.ow.server.ecmimpl.opencmis.auth.OwCMISDefaultAuthenticationProvider#getUserSelectSubModul(java.lang.String, int[])
     */
    @SuppressWarnings("unchecked")
    @Override
    public OwUIUserSelectModul getUserSelectSubModul(String strID_p, int[] types_p) throws OwException
    {
        OwCMISUserSelectionModule module = new OwCMISUserSelectionModule(this.userRepository);
        module.setFilter(types_p);
        module.setCurrentUserID(strID_p);
        try
        {
            module.init(this.network);
        }
        catch (OwException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            String msg = "Cannot initialize the User selection module";
            LOG.error(msg, e);
            throw new OwServerException(this.network.getContext().localize("opencmis.OwCMISNetwork.err.initUserSelectEx", "Cannot initialize the User Selection dialog."), e);
        }
        return module;
    }

    protected OwLdapConnector createLDAPConnector(OwAuthenticationConfiguration authenticationConf_p) throws OwException
    {
        return new OwLdapConnector(authenticationConf_p.getConfiguration().getNode());
    }

    /* (non-Javadoc)
     * @see com.wewebu.ow.server.ecmimpl.opencmis.auth.OwCMISDefaultAuthenticationProvider#createAuthenticator()
     */
    @Override
    protected OwCMISCredentialsAuthenticator createAuthenticator() throws OwException
    {
        return new OwCMISLDAPCredentialsAuthenticator(network, this.ldapconnector);
    }

    /* (non-Javadoc)
     * @see com.wewebu.ow.server.ecmimpl.opencmis.auth.OwCMISDefaultAuthenticationProvider#getUserFromID(java.lang.String)
     */
    @Override
    public OwUserInfo getUserFromID(String strID_p) throws OwException
    {
        return this.userRepository.findUserByID(strID_p);
    }
}