package com.wewebu.ow.server.ecmimpl.opencmis.object;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.chemistry.opencmis.client.api.CmisObject;
import org.apache.chemistry.opencmis.client.api.ObjectFactory;
import org.apache.chemistry.opencmis.client.api.OperationContext;
import org.apache.chemistry.opencmis.client.api.Policy;
import org.apache.chemistry.opencmis.client.api.Property;
import org.apache.chemistry.opencmis.client.api.Relationship;
import org.apache.chemistry.opencmis.client.api.Rendition;
import org.apache.chemistry.opencmis.client.api.Session;
import org.apache.chemistry.opencmis.client.api.TransientCmisObject;
import org.apache.chemistry.opencmis.commons.PropertyIds;
import org.apache.chemistry.opencmis.commons.data.Acl;
import org.apache.chemistry.opencmis.commons.data.CmisExtensionElement;
import org.apache.chemistry.opencmis.commons.data.ObjectData;
import org.apache.chemistry.opencmis.commons.data.Properties;
import org.apache.chemistry.opencmis.commons.data.RenditionData;
import org.apache.chemistry.opencmis.commons.enums.ExtensionLevel;
import org.apache.chemistry.opencmis.commons.enums.IncludeRelationships;
import org.apache.chemistry.opencmis.commons.enums.Updatability;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.ObjectDataImpl;
import org.apache.chemistry.opencmis.commons.impl.dataobjects.PolicyIdListImpl;
import org.apache.chemistry.opencmis.commons.spi.ObjectService;
import org.apache.log4j.Logger;

import com.wewebu.ow.server.ecmimpl.opencmis.log.OwLog;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.exceptions.OwServerException;

/**
 *<p>
 *A transient object that refreshes cached properties using bulk fetch through {@link ObjectService#getProperties(String, String, String, org.apache.chemistry.opencmis.commons.data.ExtensionsData)}.
 *</p>
 *
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 */
public class OwCMISBatchTransientObject<N extends TransientCmisObject> extends OwCMISAbstractTransientObject<N>
{
    private static final Logger LOG = OwLog.getLogger(OwCMISBatchTransientObject.class);

    private static final HashSet<Updatability> ALL_UPDATABILITIES = new HashSet<Updatability>(Arrays.asList(Updatability.values()));

    public OwCMISBatchTransientObject(N transientCmisObject, OperationContext creationContext, Session session)
    {
        super(transientCmisObject, creationContext, session);
    }

    @Override
    protected OwCMISContextBoundObject<N> retrieveProperties(Set<String> nativePropertyFilter) throws OwException
    {
        return fetchProperties(nativePropertyFilter);
    }

    /**
     * Fetch the native properties and add them to the cache.
     * @param nativePropertyFilter native properties' names (no qualifier)
     * @throws OwServerException 
     */
    private OwCMISContextBoundObject<N> fetchProperties(Set<String> nativePropertyFilter) throws OwException
    {
        LOG.info("Fetching batch-object-properties from content repository!");
        if (LOG.isDebugEnabled())
        {
            LOG.debug(String.format("OwCMISBatchTransientObject.fetchProperties: Fetched batch-object-properties  = %s", nativePropertyFilter.toString()));
        }

        StringBuilder filterStringBuilder = new StringBuilder();

        //TODO: use operationContext for filter string generation

        if (nativePropertyFilter.contains("*"))
        {
            filterStringBuilder.append("*");
        }
        else
        {
            for (String oid : nativePropertyFilter)
            {
                if (filterStringBuilder.length() > 0)
                {
                    filterStringBuilder.append(",");
                }

                filterStringBuilder.append(oid);
            }
        }

        String nativeId = this.contextBoundObject.object.getId();
        Properties fetchedNativeProperties = this.session.getBinding().getObjectService().getProperties(this.session.getRepositoryInfo().getId(), nativeId, filterStringBuilder.toString(), null);
        Map<String, Property<?>> fetchedPropertiesMap = this.session.getObjectFactory().convertProperties(this.contextBoundObject.object.getType(), fetchedNativeProperties);

        return mergeProperties(fetchedPropertiesMap, null, fetchedNativeProperties.getExtensions(), nativePropertyFilter);

    }

    protected OwCMISContextBoundObject<N> mergeProperties(Map<String, Property<?>> add, Collection<String> remove, List<CmisExtensionElement> extensions, Set<String> fetchFilter)
    {
        CmisObject cmisObject = this.contextBoundObject.object.getCmisObject();

        ObjectData objectData = mergeProperties(cmisObject, add, remove, extensions);

        ObjectFactory objectFactory = this.session.getObjectFactory();

        OperationContext creationContext = this.contextBoundObject.context;

        if (fetchFilter != null)
        {
            //merge contexts
            Set<String> creationFilter = creationContext.getFilter();
            if (creationFilter == null)
            {
                creationFilter = new HashSet<String>();
            }
            else
            {
                creationFilter = new HashSet<String>(creationFilter);
            }

            creationFilter.addAll(fetchFilter);

            boolean includeAcls = creationContext.isIncludeAcls();
            boolean includeAllowableActions = creationContext.isIncludeAllowableActions();
            boolean includePolicies = creationContext.isIncludePolicies();
            IncludeRelationships includeRelationships = creationContext.getIncludeRelationships();
            Set<String> renditionFilter = creationContext.getRenditionFilter();
            boolean includePathSegments = creationContext.isIncludePathSegments();
            String orderBy = creationContext.getOrderBy();
            boolean cacheEnabled = creationContext.isCacheEnabled();
            int maxItemsPerPage = creationContext.getMaxItemsPerPage();

            creationContext = this.session.createOperationContext(creationFilter, includeAcls, includeAllowableActions, includePolicies, includeRelationships, renditionFilter, includePathSegments, orderBy, cacheEnabled, maxItemsPerPage);
        }

        CmisObject newCmisObject = objectFactory.convertObject(objectData, creationContext);

        return new OwCMISContextBoundObject<N>((N) newCmisObject.getTransientObject(), creationContext);
    }

    protected ObjectData mergeProperties(CmisObject cmisObject, Map<String, Property<?>> add, Collection<String> remove, List<CmisExtensionElement> extensions)
    {
        ObjectDataImpl objectData = new ObjectDataImpl();
        ObjectFactory objectFactory = this.session.getObjectFactory();

        List<Property<?>> cmisObjectProperties = cmisObject.getProperties();
        Map<String, Object> propertiesMap = new HashMap<String, Object>();
        for (Property<?> property : cmisObjectProperties)
        {
            String id = property.getId();
            if (remove == null || !remove.contains(id))
            {
                propertiesMap.put(id, property);
            }
        }

        if (add != null)
        {
            propertiesMap.putAll(add);
        }

        Property<String> typeId = (Property<String>) propertiesMap.get(PropertyIds.OBJECT_TYPE_ID);
        propertiesMap.put(PropertyIds.OBJECT_TYPE_ID, typeId.getValue());

        Properties properties = objectFactory.convertProperties(propertiesMap, null, ALL_UPDATABILITIES);

        objectData.setProperties(properties);
        objectData.setAcl(cmisObject.getAcl());
        objectData.setAllowableActions(cmisObject.getAllowableActions());
        objectData.setExtensions(allExtensionsOf(cmisObject));
        Acl acl = cmisObject.getAcl();
        if (acl != null)
        {
            objectData.setIsExactAcl(acl.isExact());
            objectData.setAcl(acl);
        }

        List<Policy> policies = cmisObject.getPolicies();
        if (null != policies)
        {
            List<String> policyIds = new LinkedList<String>();
            for (Policy policy : policies)
            {
                policyIds.add(policy.getId());
            }
            PolicyIdListImpl policyIdList = new PolicyIdListImpl();
            policyIdList.setPolicyIds(policyIds);
            objectData.setPolicyIds(policyIdList);
        }

        List<Rendition> renditions = cmisObject.getRenditions();
        if (renditions != null)
        {
            List<RenditionData> renditionDatas = new LinkedList<RenditionData>(renditions);
            objectData.setRenditions(renditionDatas);
        }

        //        ChangeEventInfo changeEventInfo = null;
        //        objectData.setChangeEventInfo(changeEventInfo);

        List<Relationship> relationships = cmisObject.getRelationships();
        if (relationships != null)
        {
            List<ObjectData> relationshipDatas = new LinkedList<ObjectData>();
            for (Relationship relationship : relationships)
            {
                ObjectData realtionshipData = mergeProperties(relationship, null, null, null);
                relationshipDatas.add(realtionshipData);
            }
            objectData.setRelationships(relationshipDatas);
        }

        return objectData;

    }

    private List<CmisExtensionElement> allExtensionsOf(CmisObject cmisObject)
    {
        List<CmisExtensionElement> extensions = new LinkedList<CmisExtensionElement>();
        ExtensionLevel[] allLevels = ExtensionLevel.values();

        for (ExtensionLevel extensionLevel : allLevels)
        {
            List<CmisExtensionElement> levelEextensions = cmisObject.getExtensions(extensionLevel);
            if (levelEextensions != null)
            {
                extensions.addAll(levelEextensions);
            }
        }

        return extensions;
    }
}
