package com.wewebu.ow.server.ecmimpl.cmis;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.List;

import javax.xml.ws.Binding;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.handler.Handler;
import javax.xml.ws.handler.soap.SOAPHandler;
import javax.xml.ws.handler.soap.SOAPMessageContext;
import javax.xml.ws.soap.MTOMFeature;

import org.apache.log4j.Logger;
import org.oasis_open.docs.ns.cmis.ws._200908.ACLService;
import org.oasis_open.docs.ns.cmis.ws._200908.ACLServicePort;
import org.oasis_open.docs.ns.cmis.ws._200908.DiscoveryService;
import org.oasis_open.docs.ns.cmis.ws._200908.DiscoveryServicePort;
import org.oasis_open.docs.ns.cmis.ws._200908.MultiFilingService;
import org.oasis_open.docs.ns.cmis.ws._200908.MultiFilingServicePort;
import org.oasis_open.docs.ns.cmis.ws._200908.NavigationService;
import org.oasis_open.docs.ns.cmis.ws._200908.NavigationServicePort;
import org.oasis_open.docs.ns.cmis.ws._200908.ObjectService;
import org.oasis_open.docs.ns.cmis.ws._200908.ObjectServicePort;
import org.oasis_open.docs.ns.cmis.ws._200908.PolicyService;
import org.oasis_open.docs.ns.cmis.ws._200908.PolicyServicePort;
import org.oasis_open.docs.ns.cmis.ws._200908.RelationshipService;
import org.oasis_open.docs.ns.cmis.ws._200908.RelationshipServicePort;
import org.oasis_open.docs.ns.cmis.ws._200908.RepositoryService;
import org.oasis_open.docs.ns.cmis.ws._200908.RepositoryServicePort;
import org.oasis_open.docs.ns.cmis.ws._200908.VersioningService;
import org.oasis_open.docs.ns.cmis.ws._200908.VersioningServicePort;

import com.wewebu.ow.server.ecm.OwResource;
import com.wewebu.ow.server.ecmimpl.OwAbstractCredentials;
import com.wewebu.ow.server.ecmimpl.OwCredentialsConstants;
import com.wewebu.ow.server.ecmimpl.cmis.log.OwLog;
import com.wewebu.ow.server.ecmimpl.cmis.wshandler.OwCMISContentHandler;
import com.wewebu.ow.server.ecmimpl.cmis.wshandler.OwCMISSecurityHandlerInterface;
import com.wewebu.ow.server.exceptions.OwConfigurationException;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.mandator.OwMandator;
import com.wewebu.ow.server.util.OwString;
import com.wewebu.ow.server.util.OwXMLUtil;

/**
 *<p>
 * Credentials implementation. Containing all references to the CMIS services, 
 * which are also initialized in this class.Credentials implementation. <br/>
 * Containing all references to the CMIS services, which are also initialized
 * in this class.
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 */
public class OwCMISCredentials extends OwAbstractCredentials
{
    private static final Logger LOG = OwLog.getLogger(OwCMISCredentials.class);

    /**Configuration node name for RepositoryService URL*/
    public static final String CONF_NODE_WSDL_REPOSITORY = "WSDLRepository";
    /**Configuration node name for NavigationService URL*/
    public static final String CONF_NODE_WSDL_NAVIGATION = "WSDLNavigation";
    /**Configuration node name for ObjectService URL*/
    public static final String CONF_NODE_WSDL_OBJECT = "WSDLObject";
    /**Configuration node name for MultifilingService URL*/
    public static final String CONF_NODE_WSDL_MULTIFILING = "WSDLMultifiling";
    /**Configuration node name for DiscoveryService (search service) URL*/
    public static final String CONF_NODE_WSDL_DISCOVERY = "WSDLDiscovery";
    /**Configuration node name for VersioningService URL*/
    public static final String CONF_NODE_WSDL_VERSIONING = "WSDLVersioning";
    /**Configuration node name for RelationshipService URL*/
    public static final String CONF_NODE_WSDL_RELATIONSHIP = "WSDLRelationship";
    /**Configuration node name for PolicyService URL*/
    public static final String CONF_NODE_WSDL_POLICY = "WSDLPolicy";
    /**Configuration node name for ACL-Service URL*/
    public static final String CONF_NODE_WSDL_ACL = "WSDLACL";

    /**Basic authentication of CMIS, only user name and password*/
    public static final String AUTH_NONE = "NONE";
    /**LDAP authentication, bypassing CMIS and connecting to the LDAP system of the CMIS repository*/
    public static final String AUTH_LDAP = "LDAP";

    private OwCMISUserInfo m_userinfo;

    /** CMIS repository service port */
    private RepositoryServicePort m_repositoryServicePort;
    /** CMIS object service port */
    private ObjectServicePort m_objectServicesPort;
    /** CMIS navigation service port */
    private NavigationServicePort m_navigationServicePort;
    /** CMIS discovery service port */
    private DiscoveryServicePort m_discoveryServicePort;
    /** CMIS multiple filing service port */
    private MultiFilingServicePort m_multiFilingServicePort;
    /** CMIS policy service port */
    private PolicyServicePort m_policyServicePort;
    /** CMIS ACL service port
     *@since 3.2.0.0 */
    private ACLServicePort m_aclServicePort;
    /** CMIS relationship service port*/
    private RelationshipServicePort m_relationshipServicePort;
    /** CMIS versioning service port */
    private VersioningServicePort m_versioningServicePort;

    /** WSS handler */
    private OwCMISSecurityHandlerInterface m_securityHandler;

    private SOAPHandler<SOAPMessageContext> m_contentHandler;

    private OwMandator m_mandator;

    private OwXMLUtil configuration;

    /** map<confNode,URL> of URL's which are loaded from configuration node*/
    private HashMap<String, URL> m_serviceURLs;

    /** create credentials
     * 
     * @param username_p String log on name
     * @param password_p String or null
     * @param securityhandler_p OwCMISSecurityHandlerInterface
     * @param mandator_p OwMandator
     * @throws OwException
     */
    public OwCMISCredentials(OwXMLUtil configuration_p, String username_p, String password_p, OwCMISSecurityHandlerInterface securityhandler_p, OwMandator mandator_p) throws OwException
    {
        super(username_p, password_p);
        m_securityHandler = securityhandler_p;
        m_mandator = mandator_p;
        m_contentHandler = createContentHandler();
        this.configuration = configuration_p;

        m_serviceURLs = new HashMap<String, URL>();

    }

    public OwMandator getMandator()
    {
        return m_mandator;
    }

    public void setMandator(OwMandator madator_p)
    {
        m_mandator = madator_p;
    }

    public ObjectServicePort getObjectServicePort() throws OwException
    {
        return getObjectServicePort(getServiceURL(CONF_NODE_WSDL_OBJECT));
    }

    /** Get the CMIS object service, which handle
     * create, copy and update objects, and also retrieving objects
     * by ID or path.
     * @param wsdlLocation_p URL to create the service port if not connected
     * @return ObjectServicePort
     */
    public ObjectServicePort getObjectServicePort(URL wsdlLocation_p)
    {
        if (m_objectServicesPort == null)
        {
            ObjectService s = new ObjectService(wsdlLocation_p);
            m_objectServicesPort = s.getObjectServicePort(new MTOMFeature());

            // insert security handler
            setSecurity((BindingProvider) m_objectServicesPort);
        }

        return m_objectServicesPort;
    }

    /**
     * Get the URL from configuration, and cache it
     * for further request.
     * @param confNode_p String one of CONF_NODE_WSDL_...
     * @return URL for the specified configuration node
     * @throws OwException if URL in configuration node is malformed
     */
    protected URL getServiceURL(String confNode_p) throws OwException
    {
        if (this.m_serviceURLs.containsKey(confNode_p))
        {
            return this.m_serviceURLs.get(confNode_p);
        }
        else
        {
            URL url = null;
            try
            {
                url = configuration.getURLFromNode(confNode_p);
            }
            catch (MalformedURLException e)
            {
                LOG.error("OwCMISNetwork.getURLFromNode(): Invalid CMIS configuration! The request node " + confNode_p + " does not contain a valid URL.", e);
                throw new OwConfigurationException(new OwString("ecmimpl.cmis.OwCMISNetwork.invalid.configuration.error", "Invalid CMIS configuration!"), e);

            }
            if (url != null)
            {
                this.m_serviceURLs.put(confNode_p, url);
            }
            return url;
        }
    }

    public RepositoryServicePort getRepositoryServicePort() throws OwConfigurationException
    {
        RepositoryServicePort result = null;
        try
        {
            result = getRepositoryServicePort(getServiceURL(CONF_NODE_WSDL_REPOSITORY));
        }
        catch (Exception e)
        {
            LOG.error("Cannot obtain repository service port", e);
            throw new OwConfigurationException("Cannot obtain repository service port", e);
        }
        return result;

    }

    /**
     * Get the CMIS repository service, which is used
     * to retrieve additional Information about repositories,
     * or existent object class definitions.
     * @param wsdlLocation_p URL of the WSDL to create service port
     * @return RepositoryServicePort
     */
    public RepositoryServicePort getRepositoryServicePort(URL wsdlLocation_p)
    {
        if (null == m_repositoryServicePort)
        {
            RepositoryService s = new RepositoryService(wsdlLocation_p);
            m_repositoryServicePort = s.getRepositoryServicePort();

            // insert security handler
            setSecurity((BindingProvider) m_repositoryServicePort);
        }
        return m_repositoryServicePort;
    }

    public NavigationServicePort getNavigationServicePort() throws OwException
    {
        return getNavigationServicePort(getServiceURL(CONF_NODE_WSDL_NAVIGATION));
    }

    /** Get the CMIS service which handles
     *  the browsing and/or traversing through
     *  the file structure of the CMIS back-end system.
     *  @param wsdlLocation_p URL to create a service port if still not connected
     * @return NavigationServicePort
     */
    public NavigationServicePort getNavigationServicePort(URL wsdlLocation_p)
    {
        if (null == m_navigationServicePort)
        {
            NavigationService s = new NavigationService(wsdlLocation_p);
            m_navigationServicePort = s.getNavigationServicePort();

            // insert security handler
            setSecurity((BindingProvider) m_navigationServicePort);
        }

        return m_navigationServicePort;
    }

    public DiscoveryServicePort getDiscoveryServicePort() throws OwException
    {
        return getDiscoveryServicePort(getServiceURL(CONF_NODE_WSDL_DISCOVERY));
    }

    /**
     * Get the CMIS service which handles the
     * searches against the CMIS back-end system.
     * @param url_p URL to create a service port if not connected
     * @return DiscoverServicePort
     */
    public DiscoveryServicePort getDiscoveryServicePort(URL url_p)
    {
        if (this.m_discoveryServicePort == null)
        {
            DiscoveryService s = new DiscoveryService(url_p);
            m_discoveryServicePort = s.getDiscoveryServicePort();

            setSecurity((BindingProvider) m_discoveryServicePort);
        }

        return this.m_discoveryServicePort;
    }

    public MultiFilingServicePort getMultiFilingservicePort() throws OwException
    {
        return getMultiFilingservicePort(getServiceURL(CONF_NODE_WSDL_MULTIFILING));
    }

    /**
     * Get the Service for multiple filing of documents, in different
     * folders.
     * <p>Before calling this method, the capability of the repository should be checked</p>
     * @param wsdlLocation_p URL location of WSDL for MultiFilingService
     * @return MultiFilingServicePort
     */
    public MultiFilingServicePort getMultiFilingservicePort(URL wsdlLocation_p)
    {
        if (this.m_multiFilingServicePort == null)
        {
            MultiFilingService s = new MultiFilingService(wsdlLocation_p);
            m_multiFilingServicePort = s.getMultiFilingServicePort();

            setSecurity((BindingProvider) m_multiFilingServicePort);
        }

        return this.m_multiFilingServicePort;
    }

    public VersioningServicePort getVersionServicePort() throws OwException
    {
        return getVersionServicePort(getServiceURL(CONF_NODE_WSDL_VERSIONING));
    }

    /**
     * Get the service port for handling versions functionality 
     * of CMIS repository.
     * <p>Before calling this method, the capability of the repository
     * should be checked for versioning.</p>
     * @param wsdlLocation_p URL location of WSDL for creation of service port if not connected
     * @return VersioningServicePort
     */
    public VersioningServicePort getVersionServicePort(URL wsdlLocation_p)
    {
        if (this.m_versioningServicePort == null)
        {
            VersioningService s = new VersioningService(wsdlLocation_p);
            this.m_versioningServicePort = s.getVersioningServicePort(new MTOMFeature());

            setSecurity((BindingProvider) this.m_versioningServicePort);
        }

        return this.m_versioningServicePort;
    }

    public RelationshipServicePort getRelationshipServicePort() throws OwException
    {
        return getRelationshipServicePort(getServiceURL(CONF_NODE_WSDL_RELATIONSHIP));
    }

    /**
     * Get CMIS relationship service port, for handling of 
     * document relationships.
     * <p>The relationship capability should be checked before
     * calling this method</p>
     * @param wsdlLocation_p
     * @return RelationshipServicePort
     */
    public RelationshipServicePort getRelationshipServicePort(URL wsdlLocation_p)
    {
        if (this.m_relationshipServicePort == null)
        {
            RelationshipService s = new RelationshipService(wsdlLocation_p);
            this.m_relationshipServicePort = s.getRelationshipServicePort();

            setSecurity((BindingProvider) this.m_relationshipServicePort);
        }

        return this.m_relationshipServicePort;
    }

    public PolicyServicePort getPolicyServicePort() throws OwException
    {
        return getPolicyServicePort(getServiceURL(CONF_NODE_WSDL_POLICY));
    }

    /**
     * Get the policy service port, for applying and requesting
     * policies from objects.
     * @param wsdlLocation_p URL location of WSDL for policy service
     * @return PolicyServicePort
     */
    public PolicyServicePort getPolicyServicePort(URL wsdlLocation_p)
    {
        if (this.m_policyServicePort == null)
        {
            PolicyService s = new PolicyService(wsdlLocation_p);
            this.m_policyServicePort = s.getPolicyServicePort();

            setSecurity((BindingProvider) this.m_policyServicePort);
        }

        return this.m_policyServicePort;
    }

    public ACLServicePort getACLServicePort() throws OwException
    {
        return getACLServicePort(getServiceURL(CONF_NODE_WSDL_ACL));
    }

    /**
     * Get the ACL service port, for applying and requesting
     * policies from objects.
     * @param wsdlLocation_p URL location of WSDL for the ACL service
     * @return ACLServicePort
     * @since 3.2.0.0
     */
    public ACLServicePort getACLServicePort(URL wsdlLocation_p)
    {
        if (this.m_aclServicePort == null)
        {
            ACLService s = new ACLService(wsdlLocation_p);
            this.m_aclServicePort = s.getACLServicePort();

            setSecurity((BindingProvider) this.m_aclServicePort);
        }

        return this.m_aclServicePort;
    }

    /** enable WSS for CMIS web service ports
     * 
     * @param bp_p BindingProvider
     */
    @SuppressWarnings("rawtypes")
    private void setSecurity(BindingProvider bp_p)
    {
        Binding binding = bp_p.getBinding();
        //HTTP-Authentication
        bp_p.getRequestContext().put(BindingProvider.PASSWORD_PROPERTY, getAuthInfo(OwCredentialsConstants.LOGIN_PWD));
        bp_p.getRequestContext().put(BindingProvider.USERNAME_PROPERTY, getAuthInfo(OwCredentialsConstants.LOGIN_USR));

        List<Handler> handlerChain = binding.getHandlerChain();
        handlerChain.add(getContentHandler());
        handlerChain.add(getSecurityHandler());
        binding.setHandlerChain(handlerChain);
    }

    /** Get the class which is used as WebService Security Handler
     * @return OwCMISSecurityHandler
     */
    public OwCMISSecurityHandlerInterface getSecurityHandler()
    {
        return this.m_securityHandler;
    }

    public String getSecurityToken(OwResource resource_p)
    {
        return null;
    }

    /**(overridable)
     * 
     * @param userName_p
     * @return an new {@link OwCMISUserInfo} for the given user
     */
    protected synchronized OwCMISUserInfo createUserInfo(String userName_p)
    {
        return new OwCMISSimpleUserInfo(userName_p, m_mandator);
    }

    public OwCMISUserInfo getUserInfo() throws OwException
    {
        if (m_userinfo == null)
        {
            m_userinfo = createUserInfo(getAuthInfo(OwCredentialsConstants.LOGIN_USR));
        }
        return m_userinfo;
    }

    public void invalidate() throws OwException
    {
        if (this.m_relationshipServicePort != null)
        {
            removeHandler((BindingProvider) this.m_relationshipServicePort);
            this.m_relationshipServicePort = null;
        }
        if (this.m_versioningServicePort != null)
        {
            removeHandler((BindingProvider) this.m_versioningServicePort);
            this.m_versioningServicePort = null;
        }
        if (this.m_policyServicePort != null)
        {
            removeHandler((BindingProvider) this.m_policyServicePort);
            this.m_policyServicePort = null;
        }
        if (this.m_multiFilingServicePort != null)
        {
            removeHandler((BindingProvider) this.m_multiFilingServicePort);
            this.m_multiFilingServicePort = null;
        }
        if (this.m_aclServicePort != null)
        {
            removeHandler((BindingProvider) this.m_aclServicePort);
            this.m_aclServicePort = null;
        }
        if (this.m_discoveryServicePort != null)
        {
            removeHandler((BindingProvider) this.m_discoveryServicePort);
            this.m_discoveryServicePort = null;
        }
        if (this.m_objectServicesPort != null)
        {
            removeHandler((BindingProvider) this.m_objectServicesPort);
            this.m_objectServicesPort = null;
        }
        if (this.m_repositoryServicePort != null)
        {
            removeHandler((BindingProvider) this.m_repositoryServicePort);
            this.m_repositoryServicePort = null;
        }
        this.m_securityHandler = null;
        this.m_contentHandler = null;
        this.m_userinfo = null;
    }

    public boolean isContainerBasedAuthenticated() throws OwException
    {
        return false;
    }

    /**
     * Remove all handler which were added with the current
     * class to the given BindingProvider
     * @param bp_p BindingProvider
     * @since 3.1.0.0
     */
    protected void removeHandler(BindingProvider bp_p)
    {
        bp_p.getBinding().getHandlerChain().remove(getContentHandler());
        bp_p.getBinding().getHandlerChain().remove(getSecurityHandler());
    }

    /**
     * Get a SOAPHandler which will be add
     * as first handler into the handler chain.
     * <p>The first handler should be responsible for the content handling</p>. 
     * @return SOAPHandler
     * @since 3.1.0.0
     */
    protected SOAPHandler<SOAPMessageContext> getContentHandler()
    {
        return this.m_contentHandler;
    }

    /**
     * Create a handler which will be assigned as the content handler.
     * <p>Per default it will create an instance of the {@link OwCMISContentHandler}.</p>
     * @return SOAPHandler
     * @since 3.1.0.0
     */
    protected SOAPHandler<SOAPMessageContext> createContentHandler()
    {
        return new OwCMISContentHandler();
    }

    /**
     * Getter for the login name, which was used for authentication against repository.
     * @return String login name
     * @since 3.2.0.0
     * @deprecated since 4.0.0.0 use {@link #getAuthInfo(String)} instead
     */
    public String getLoginName()
    {
        return getAuthInfo(OwCredentialsConstants.LOGIN_USR);
    }

    /**
     * Getter for the password, which was used to authenticate against repository.
     * @return String password
     * @since 3.2.0.0
     * @deprecated since 4.0.0.0 use {@link #getAuthInfo(String)} instead
     */
    public String getLoginPassword()
    {
        return getAuthInfo(OwCredentialsConstants.LOGIN_PWD);
    }
}