package com.wewebu.ow.server.ecmimpl.cmis;

import java.util.Locale;

import org.oasis_open.docs.ns.cmis.messaging._200908.CmisFaultType;
import org.oasis_open.docs.ns.cmis.ws._200908.CmisException;

import com.wewebu.ow.server.exceptions.OwAccessDeniedException;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.exceptions.OwInvalidOperationException;
import com.wewebu.ow.server.exceptions.OwNotSupportedException;
import com.wewebu.ow.server.exceptions.OwObjectNotFoundException;
import com.wewebu.ow.server.exceptions.OwServerException;
import com.wewebu.ow.server.util.OwString;

/**
 *<p>
 * {@link CmisException} catching block helper.<br>
 * Creates specific fault type exceptions and log-statements.
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 */
public class OwCMISExceptionCatcher
{
    private CmisException cmisException;
    private Locale locale;
    private String cmisFaultTypeLogMessage;
    private String cmisFaultTypeLocalMessage;

    /**
     * Constructor
     * @param cmisException_p 
     */
    public OwCMISExceptionCatcher(CmisException cmisException_p)
    {
        this(cmisException_p, null);
    }

    /**
     * Constructor
     * @param cmisException_p
     * @param locale_p current locale used in fault type exception message localization 
     */
    public OwCMISExceptionCatcher(CmisException cmisException_p, Locale locale_p)
    {
        super();
        this.cmisException = cmisException_p;
        this.locale = locale_p;
        createCMISFaultTypeMessage();
    }

    private void createCMISFaultTypeMessage()
    {
        CmisFaultType fault = this.cmisException.getFaultInfo();

        switch (fault.getType())
        {
            case CONSTRAINT:
                this.cmisFaultTypeLogMessage = "CMIS constraint fault!";
                this.cmisFaultTypeLocalMessage = localize("ecmimpl.cmis.OwCMISException.fault.message.constraint", this.cmisFaultTypeLogMessage);
                break;
            case NAME_CONSTRAINT_VIOLATION:
                this.cmisFaultTypeLogMessage = "CMIS name contstraint fault!";
                this.cmisFaultTypeLocalMessage = localize("ecmimpl.cmis.OwCMISException.fault.message.name.constraint.violation", this.cmisFaultTypeLogMessage);
                break;
            case CONTENT_ALREADY_EXISTS:
                this.cmisFaultTypeLogMessage = "Content already exists!";
                this.cmisFaultTypeLocalMessage = localize("ecmimpl.cmis.OwCMISException.fault.message.content.alreay.exists", this.cmisFaultTypeLogMessage);
                break;
            case FILTER_NOT_VALID:
                this.cmisFaultTypeLogMessage = "Invalid filter!";
                this.cmisFaultTypeLocalMessage = localize("ecmimpl.cmis.OwCMISException.fault.message.filter.not.valid", this.cmisFaultTypeLogMessage);
                break;
            case INVALID_ARGUMENT:
                this.cmisFaultTypeLogMessage = "Invalid argument!";
                this.cmisFaultTypeLocalMessage = localize("ecmimpl.cmis.OwCMISException.fault.message.invalid.argument", this.cmisFaultTypeLogMessage);
                break;
            case NOT_SUPPORTED:
                this.cmisFaultTypeLogMessage = "Not supported!";
                this.cmisFaultTypeLocalMessage = localize("ecmimpl.cmis.OwCMISException.fault.message.not.supported", this.cmisFaultTypeLogMessage);
                break;
            case OBJECT_NOT_FOUND:
                this.cmisFaultTypeLogMessage = "Object not found!";
                this.cmisFaultTypeLocalMessage = localize("ecmimpl.cmis.OwCMISException.fault.message.object.not.found", this.cmisFaultTypeLogMessage);
                break;
            case PERMISSION_DENIED:
                this.cmisFaultTypeLogMessage = "Permission denied!";
                this.cmisFaultTypeLocalMessage = localize("ecmimpl.cmis.OwCMISException.fault.message.permission.denied", this.cmisFaultTypeLogMessage);
                break;
            case RUNTIME:
                this.cmisFaultTypeLogMessage = "Runtime error!";
                this.cmisFaultTypeLocalMessage = localize("ecmimpl.cmis.OwCMISException.fault.message.runtime", this.cmisFaultTypeLogMessage);
                break;
            case STORAGE:
                this.cmisFaultTypeLogMessage = "CMIS storage problems!";
                this.cmisFaultTypeLocalMessage = localize("ecmimpl.cmis.OwCMISException.fault.message.storage", this.cmisFaultTypeLogMessage);
                break;
            case STREAM_NOT_SUPPORTED:
                this.cmisFaultTypeLogMessage = "ContentStream is not supported in this context (type/class)!";
                this.cmisFaultTypeLocalMessage = localize("ecmimpl.cmis.OwCMISException.fault.message.not.supported", this.cmisFaultTypeLogMessage);
                break;
            case UPDATE_CONFLICT:
                this.cmisFaultTypeLogMessage = "Update conflict!";
                this.cmisFaultTypeLocalMessage = localize("ecmimpl.cmis.OwCMISException.fault.message.update.conflict", this.cmisFaultTypeLogMessage);
                break;
            case VERSIONING:
                this.cmisFaultTypeLogMessage = "Versioning error!";
                this.cmisFaultTypeLocalMessage = localize("ecmimpl.cmis.OwCMISException.fault.message.versioning", this.cmisFaultTypeLogMessage);
                break;
            default:
                this.cmisFaultTypeLogMessage = "Unspecified CMIS fault info type!";
                this.cmisFaultTypeLocalMessage = localize("ecmimpl.cmis.OwCMISException.fault.message.unspecified", "Unspecified CMIS fault info type!");
        }
    }

    private String localize(String key_p, String defaultText_p)
    {
        if (this.locale != null)
        {
            return OwString.localize(this.locale, key_p, defaultText_p);
        }
        else
        {
            return defaultText_p;
        }
    }

    /**
     * 
     * @return fault type specific log message  (fault-type message and fault message) 
     */
    public String getLogMessage()
    {
        CmisFaultType fault = this.cmisException.getFaultInfo();
        return this.cmisFaultTypeLogMessage + " CMIS fault message is : " + fault.getMessage();
    }

    /**
     * 
     * @return the localized fault type message 
     */
    public String getLocalizedFaultTypeMessage()
    {
        return this.cmisFaultTypeLocalMessage;
    }

    /**
     * 
     * @return an {@link OwException} wrapper of the current {@link CmisException}
     */
    public OwException toOwException()
    {
        return toOwException(getLocalizedFaultTypeMessage());
    }

    /**
     * 
     * @param message_p
     * @return an {@link OwException} wrapper of the current {@link CmisException} with the given message
     */
    public OwException toOwException(String message_p)
    {
        return toOwException(message_p, false);
    }

    /**
     * 
     * @param message_p
     * @param addCMISFaultTypeMessage_p if <code>true</code> the localized fault type message is added to the 
     *                                  given message in the generated exception object
     * @return an {@link OwException} wrapper of the current {@link CmisException} with the given message  
     */
    public OwException toOwException(String message_p, boolean addCMISFaultTypeMessage_p)
    {

        CmisFaultType fault = this.cmisException.getFaultInfo();
        String message = message_p;
        if (addCMISFaultTypeMessage_p)
        {
            String faultLink = localize("ecmimpl.cmis.OwCMISException.cmis.fault.link", "CMIS fault is :");
            message = message + " " + faultLink + " " + getLocalizedFaultTypeMessage();
            if (fault != null && fault.getMessage() != null && !fault.getMessage().equals(""))
            {
                message = message + " [" + fault.getMessage() + "]";
            }
        }
        if (fault != null)
        {
            switch (fault.getType())
            {
                case CONSTRAINT:
                    return new OwInvalidOperationException(message, this.cmisException);
                case NAME_CONSTRAINT_VIOLATION:
                    return new OwInvalidOperationException(message, this.cmisException);
                case CONTENT_ALREADY_EXISTS:
                    return new OwInvalidOperationException(message, this.cmisException);
                case FILTER_NOT_VALID:
                    return new OwInvalidOperationException(message, this.cmisException);
                case INVALID_ARGUMENT:
                    return new OwInvalidOperationException(message, this.cmisException);
                case NOT_SUPPORTED:
                    return new OwNotSupportedException(message, this.cmisException);
                case OBJECT_NOT_FOUND:
                    return new OwObjectNotFoundException(message, this.cmisException);
                case PERMISSION_DENIED:
                    return new OwAccessDeniedException(message, this.cmisException);
                case RUNTIME:
                    return new OwServerException(message, this.cmisException);
                case STORAGE:
                    return new OwServerException(message, this.cmisException);
                case STREAM_NOT_SUPPORTED:
                    return new OwNotSupportedException(message, this.cmisException);
                case UPDATE_CONFLICT:
                    return new OwInvalidOperationException(message, this.cmisException);
                case VERSIONING:
                    return new OwInvalidOperationException(message, this.cmisException);
                default:
                    return new OwServerException(message, this.cmisException);
            }
        }
        else
        {
            return new OwServerException(message, this.cmisException);
        }
    }

    public OwException toOwException(OwString message_p)
    {
        CmisFaultType fault = this.cmisException.getFaultInfo();
        switch (fault.getType())
        {
            case CONSTRAINT:
                return new OwInvalidOperationException(message_p, this.cmisException);
            case NAME_CONSTRAINT_VIOLATION:
                return new OwInvalidOperationException(message_p, this.cmisException);
            case CONTENT_ALREADY_EXISTS:
                return new OwInvalidOperationException(message_p, this.cmisException);
            case FILTER_NOT_VALID:
                return new OwInvalidOperationException(message_p, this.cmisException);
            case INVALID_ARGUMENT:
                return new OwInvalidOperationException(message_p, this.cmisException);
            case NOT_SUPPORTED:
                return new OwNotSupportedException(message_p, this.cmisException);
            case OBJECT_NOT_FOUND:
                return new OwObjectNotFoundException(message_p, this.cmisException);
            case PERMISSION_DENIED:
                return new OwAccessDeniedException(message_p, this.cmisException);
            case RUNTIME:
                return new OwServerException(message_p, this.cmisException);
            case STORAGE:
                return new OwServerException(message_p, this.cmisException);
            case STREAM_NOT_SUPPORTED:
                return new OwNotSupportedException(message_p, this.cmisException);
            case UPDATE_CONFLICT:
                return new OwInvalidOperationException(message_p, this.cmisException);
            case VERSIONING:
                return new OwInvalidOperationException(message_p, this.cmisException);
            default:
                return new OwServerException(message_p, this.cmisException);
        }
    }

}