package com.wewebu.ow.server.ecmimpl.cmis;

import java.lang.reflect.Constructor;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.w3c.dom.Node;

import com.wewebu.ow.server.ecmimpl.cmis.log.OwLog;
import com.wewebu.ow.server.ecmimpl.cmis.util.OwCMISIDDMSIDConverter;
import com.wewebu.ow.server.exceptions.OwConfigurationException;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.exceptions.OwServerException;
import com.wewebu.ow.server.util.OwNetworkConfiguration;
import com.wewebu.ow.server.util.OwStandardXMLUtil;
import com.wewebu.ow.server.util.OwXMLUtil;

/**
 *<p>
 * Helper to wrap the CMIS specific owbootstrap.xml.
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 * @since 3.2.0.0
 */
public class OwCMISNetworkConfiguration extends OwNetworkConfiguration
{
    private static final Logger LOG = OwLog.getLogger(OwCMISNetworkConfiguration.class);
    public static final String EL_DEFAULTVERSIONHANDLING = "DefaultVersionHandling";
    public static final String EL_PROPERTY_ORDER = "PreferedPropertyOrder";
    public static final String EL_ID_DMSID_CONVERTER_CLASS = "IDDMSIDConverterClass";

    private Map<String, String> versionHandling;

    private static final String FOLDERCHILDREN_CONTEXT = "FolderChildren";
    private static final String NORMALIZE_VERSION = "normalize";
    private static final String PRESERVE_VERSION = "preserve";

    private Boolean preservVersion;
    private OwCMISPreferredPropertyTypeCfg preferredPropertyTypeCfg;
    private Map<String, OwCMISIDDMSIDConverter> idDMSIDConverters = new HashMap<String, OwCMISIDDMSIDConverter>();
    private OwCMISDMSIDDecoder dmsidDecoder;

    public OwCMISNetworkConfiguration(OwXMLUtil config_p)
    {
        super(config_p);
    }

    /**
     * @return a {@link Map} of String context names mapped to version handling modes (see CMIS bootstrap documentation) 
     */
    @SuppressWarnings("unchecked")
    public synchronized Map<String, String> getDefaultVersionHandling()
    {
        if (versionHandling == null)
        {
            OwXMLUtil configNode = getConfigNode();
            versionHandling = new HashMap<String, String>();

            List<Node> contexts = configNode.getSafeNodeList(EL_DEFAULTVERSIONHANDLING);
            for (Node context : contexts)
            {
                versionHandling.put(context.getNodeName(), context.getTextContent());
            }
        }

        return versionHandling;
    }

    /**
     * 
     * @param context_p
     * @return a String version handling mode for the given context
     */
    public String getDefaultVersionHandling(String context_p)
    {
        Map<String, String> versionHandligMap = getDefaultVersionHandling();
        return versionHandligMap.get(context_p);
    }

    /**
     * Is network configured to preserve version information of object.
     * @return true if version handling set to &quot;preserve&quot;
     * @since 3.2.0.1
     */
    public boolean isPreservedVersion()
    {
        if (preservVersion == null)
        {
            String folderChildrenVersionHandling = getDefaultVersionHandling(FOLDERCHILDREN_CONTEXT);
            if (LOG.isDebugEnabled())
            {
                if (folderChildrenVersionHandling == null)
                {
                    LOG.debug("No default version hanling configured for " + FOLDERCHILDREN_CONTEXT);
                }
                else if (!NORMALIZE_VERSION.equalsIgnoreCase(folderChildrenVersionHandling) && !PRESERVE_VERSION.equals(folderChildrenVersionHandling))
                {
                    LOG.debug("Invalid version hanling configured for " + FOLDERCHILDREN_CONTEXT + " : " + folderChildrenVersionHandling);
                }
            }
            preservVersion = NORMALIZE_VERSION.equalsIgnoreCase(folderChildrenVersionHandling) ? Boolean.FALSE : Boolean.TRUE;
        }
        return preservVersion.booleanValue();
    }

    public OwXMLUtil getBpmNode() throws OwConfigurationException
    {
        OwXMLUtil result = null;
        try
        {
            result = getConfigNode().getSubUtil("BPM");
        }
        catch (Exception e)
        {
            throw new OwConfigurationException("Could not get the BPM configuration!", e);
        }

        if (null == result)
        {
            throw new OwConfigurationException("Could not get the BPM configuration!");
        }
        return result;
    }

    public OwCMISPreferredPropertyTypeCfg getPreferredPropertyTypeCfg()
    {
        if (null == this.preferredPropertyTypeCfg)
        {
            this.preferredPropertyTypeCfg = new OwCMISPreferredPropertyTypeCfg(getConfigNode());
        }

        return this.preferredPropertyTypeCfg;
    }

    /**
     * get a list of property names which represents the preferred order (sequence) of the properties
     * @return a {@link List} of property names
     * @throws OwException
     * @since 4.0.0.0
     */
    @SuppressWarnings("unchecked")
    public List<String> getPreferedPropertyOrder() throws OwException
    {
        Node preferedOrderNode = null;
        try
        {
            preferedOrderNode = getConfigNode().getSubNode(EL_PROPERTY_ORDER);
        }
        catch (Exception e)
        {
            LOG.warn("OwCMISNetworkConfig.getPerferedPropertyOrder: Configuration \"PreferedPropertyOrder\" could not be read.");
        }
        if (preferedOrderNode == null)
        {
            return new LinkedList<String>();
        }
        else
        {
            try
            {
                return (new OwStandardXMLUtil(preferedOrderNode)).getSafeStringList();
            }
            catch (Exception e)
            {
                LOG.error("Parsing of PreferedPropertyOrder node failed", e);
                throw new OwServerException("Could not parse configuration of PreferedPropertyOrder", e);
            }
        }
    }

    public synchronized OwCMISDMSIDDecoder getDMSIDDecoder()
    {
        if (dmsidDecoder == null)
        {
            dmsidDecoder = new OwCMISStandardDMSIDDecoder();
        }
        return dmsidDecoder;
    }

    public synchronized OwCMISIDDMSIDConverter getIdDMSIDConverter(String resourceID_p) throws OwException
    {
        OwCMISIDDMSIDConverter idDMSIDConverter = idDMSIDConverters.get(resourceID_p);
        if (idDMSIDConverter == null)
        {
            try
            {
                String javaClassName = getConfigNode().getSafeTextValue(EL_ID_DMSID_CONVERTER_CLASS, OwCMISIDDMSIDConverter.class.getName());

                Class<OwCMISIDDMSIDConverter> javaClass = (Class<OwCMISIDDMSIDConverter>) Class.forName(javaClassName);

                Constructor<OwCMISIDDMSIDConverter> constructor = javaClass.getConstructor(new Class[] { OwCMISDMSIDDecoder.class, String.class });
                idDMSIDConverter = constructor.newInstance(new Object[] { getDMSIDDecoder(), resourceID_p });

                idDMSIDConverters.put(resourceID_p, idDMSIDConverter);
            }
            catch (Exception e)
            {
                throw new OwConfigurationException("Error accessing the ID to DMSID converter configuration!", e);
            }
        }

        return idDMSIDConverter;
    }
}
