package com.wewebu.ow.server.ecmimpl.cmis;

import java.lang.reflect.Proxy;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.log4j.Logger;
import org.oasis_open.docs.ns.cmis.core._200908.CmisACLCapabilityType;
import org.oasis_open.docs.ns.cmis.core._200908.CmisPermissionDefinition;
import org.oasis_open.docs.ns.cmis.core._200908.CmisRepositoryCapabilitiesType;
import org.oasis_open.docs.ns.cmis.core._200908.CmisRepositoryInfoType;
import org.oasis_open.docs.ns.cmis.messaging._200908.CmisRepositoryEntryType;

import com.wewebu.ow.server.ecm.OwPrivilege;
import com.wewebu.ow.server.ecm.OwResource;
import com.wewebu.ow.server.ecm.OwUserInfo;
import com.wewebu.ow.server.ecmimpl.cmis.alfresco.OwCMISALExtensionLibrary;
import com.wewebu.ow.server.ecmimpl.cmis.extension.OwCMISExtensionLibrary;
import com.wewebu.ow.server.ecmimpl.cmis.log.OwLog;
import com.wewebu.ow.server.ecmimpl.cmis.objectclasses.OwCMISObjectClassFactory;
import com.wewebu.ow.server.ecmimpl.cmis.permissions.OwCMISPrivilege;
import com.wewebu.ow.server.exceptions.OwException;

/**
 *<p>
 * Resource implementation for CMIS repository.
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 */
public class OwCMISResource implements OwResource
{
    private static Logger LOG = OwLog.getLogger(OwCMISResource.class);
    /** CMIS reference to the repository / resource */
    private CmisRepositoryEntryType repositoryEntry;
    private CmisRepositoryInfoType repositoryInfo;

    private OwCMISObjectModel objectModel;
    private OwCMISConnection cmisConnection;
    private OwCMISObjectClassFactory objectClassFactory;
    private OwCMISExtensionLibrary extensionLibrary;

    /** create a resource for the given CMIS repository
     * 
     * @param repositoryEntryType_p
     * @throws OwException 
     */
    public OwCMISResource(CmisRepositoryEntryType repositoryEntryType_p, CmisRepositoryInfoType repositoryInfo_p, OwCMISConnection cmisConnection_p, OwCMISObjectClassFactory objectClassFactory_p) throws OwException
    {
        this.repositoryEntry = repositoryEntryType_p;
        this.repositoryInfo = repositoryInfo_p;
        this.cmisConnection = cmisConnection_p;
        this.objectClassFactory = objectClassFactory_p;
        this.extensionLibrary = createExtensionLibrary(this.repositoryInfo);
    }

    public OwCMISConnection getConnection()
    {
        return this.cmisConnection;
    }

    public synchronized OwCMISObjectModel getObjectModel() throws OwException
    {
        if (this.objectModel == null)
        {
            this.objectModel = createResourceObjectModel();
        }

        return this.objectModel;
    }

    protected OwCMISObjectModel createResourceObjectModel() throws OwException
    {
        return new OwCMISResourceObjectModel(this, this.objectClassFactory);
    }

    /*
     * (non-Javadoc)
     * @see com.wewebu.ow.server.ecm.OwResource#getDescription(java.util.Locale)
     */
    public String getDescription(Locale locale_p)
    {
        try
        {
            return getRepositoryInfo().getRepositoryDescription();
        }
        catch (OwException owEx)
        {
            LOG.warn("Can not request the repository service port", owEx);
        }
        return "";
    }

    /*
     * (non-Javadoc)
     * @see com.wewebu.ow.server.ecm.OwResource#getDisplayName(java.util.Locale)
     */
    public String getDisplayName(Locale locale_p)
    {
        return this.repositoryEntry.getRepositoryName();
    }

    /*
     * (non-Javadoc)
     * @see com.wewebu.ow.server.ecm.OwResource#getID()
     */
    public String getID() throws OwException
    {
        return this.repositoryEntry.getRepositoryId();
    }

    /**
     * Returns the native object, to retrieve information
     * about the current repository.
     * @return CmisRepositoryInfoType the native CMIS info type
     * @throws OwException if there are problem to retrieve the information type
     */
    public synchronized CmisRepositoryInfoType getRepositoryInfo() throws OwException
    {
        return this.repositoryInfo;
    }

    /**
     * Returns a string, representing the supported
     * CMIS implementation version.<br />
     * Simple call the {@link #getRepositoryInfo()}.getCmisVersionSupported().
     * @return BigDecimal
     * @throws OwException if service communication problem exist
     * @see #getRepositoryInfo()
     */
    public String getSupportedCMISVersion() throws OwException
    {
        return getRepositoryInfo().getCmisVersionSupported();
    }

    /**
     * Returns a string which contains the name
     * of vendor, which is behind the CMIS layer.
     * @return String name of vendor
     * @throws OwException
     * @see #getRepositoryInfo()
     */
    public String getCMISVendor() throws OwException
    {
        return getRepositoryInfo().getVendorName();
    }

    /**
     * Returns the vendor product version number,
     * which is behind the CMIS layer.
     * @return String representing ECM product version
     * @throws OwException
     * @see #getRepositoryInfo()
     */
    public String getVendorProductVersion() throws OwException
    {
        return getRepositoryInfo().getProductVersion();
    }

    /**
     * Returns the Capabilities of the Repository which are 
     * supported through the CMIS layer.
     * @return CmisRepositoryCapabilitiesType
     * @throws OwException
     * @see #getRepositoryInfo()
     */
    public CmisRepositoryCapabilitiesType getCMISCapabilities() throws OwException
    {
        return getRepositoryInfo().getCapabilities();
    }

    /**
     * Returns the root folder ID of current repository.
     * @return String representing the root folder ID.
     * @throws OwException if {@link #getRepositoryInfo()} fails
     * @see #getRepositoryInfo()
     */
    public String getRepsitoryRootFolderId() throws OwException
    {
        return getRepositoryInfo().getRootFolderId();
    }

    /**(overridable)
     * Creates the extension library for this resource with consideration to 
     * the underlying repository.
     * @param info_p 
     * @return the extension library of this resource
     */
    protected OwCMISExtensionLibrary createExtensionLibrary(CmisRepositoryInfoType info_p)
    {
        String productName = info_p.getProductName();
        String productVersion = info_p.getProductVersion();
        LOG.debug("OwCMISResource.createExtensionLibrary : resource info " + productName + " " + productVersion);
        if (productName != null && productName.contains("Alfresco"))
        {
            if (productVersion != null)
            {
                String[] versionSplit = productVersion.split("\\.");
                if (versionSplit.length > 1)
                {
                    try
                    {
                        int major = Integer.parseInt(versionSplit[0]);
                        int minor = Integer.parseInt(versionSplit[1]);
                        if (major > 3 || (major == 3 && minor >= 4))
                        {
                            LOG.debug("OwCMISResource.createExtensionLibrary : detected extensible Alfresco repository");
                            return new OwCMISALExtensionLibrary();
                        }

                    }
                    catch (NumberFormatException e)
                    {
                        LOG.debug("Unsupported Alfresco version string : " + productVersion, e);
                    }
                }
            }
        }

        LOG.debug("OwCMISResource.createExtensionLibrary : unextensible repository - null extension library");
        return null;
    }

    /**
     * 
     * @param <E>
     * @param extensionId_p
     * @param extensionInterface_p extension interface type 
     * @param nullExtensionResult_p default return value - can be null 
     * @return the extension interface implementor of the requested extension
     */
    @SuppressWarnings("unchecked")
    public <E> E getExtension(String extensionId_p, Class<E> extensionInterface_p, Object nullExtensionResult_p)
    {
        E extension = null;
        if (this.extensionLibrary != null)
        {
            extension = this.extensionLibrary.getExtension(extensionId_p, extensionInterface_p);
        }
        E extensionPointProxy = (E) Proxy.newProxyInstance(extensionInterface_p.getClassLoader(), new Class[] { extensionInterface_p }, new OwCMISExtensionPoint(extensionId_p, extensionInterface_p, extension, nullExtensionResult_p));
        return extensionPointProxy;

    }

    /**
     * 
     * @return the available {@link OwPrivilege}s mapped by their names   
     * @since 3.2.0.0
     */
    public Map<String, OwPrivilege> getAvailablePrivileges()
    {
        CmisACLCapabilityType aclCapability = repositoryInfo.getAclCapability();
        List<CmisPermissionDefinition> permissionDefinitions = aclCapability.getPermissions();
        Map<String, OwPrivilege> privileges = new HashMap<String, OwPrivilege>();
        for (CmisPermissionDefinition permissionDefinition : permissionDefinitions)
        {
            OwCMISPrivilege privilege = new OwCMISPrivilege(permissionDefinition);
            privileges.put(privilege.getName(), privilege);
        }

        return privileges;
    }

    /**
     * 
     * @param principal_p
     * @return the available {@link OwPrivilege}s mapped by their names  for the given user  
     * @since 3.2.0.0
     */
    public Map<String, OwPrivilege> getAvailablePrivileges(OwUserInfo principal_p)
    {
        return getAvailablePrivileges();
    }
}