package com.wewebu.ow.server.ecmimpl.cmis;

import java.math.BigInteger;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Stack;
import java.util.StringTokenizer;

import org.apache.log4j.Logger;
import org.oasis_open.docs.ns.cmis.core._200908.CmisRepositoryCapabilitiesType;
import org.oasis_open.docs.ns.cmis.core._200908.CmisRepositoryInfoType;
import org.oasis_open.docs.ns.cmis.core._200908.CmisTypeDefinitionType;
import org.oasis_open.docs.ns.cmis.core._200908.EnumBaseObjectTypeIds;
import org.oasis_open.docs.ns.cmis.messaging._200908.CmisTypeContainer;
import org.oasis_open.docs.ns.cmis.messaging._200908.CmisTypeDefinitionListType;
import org.oasis_open.docs.ns.cmis.ws._200908.CmisException;
import org.oasis_open.docs.ns.cmis.ws._200908.RepositoryServicePort;

import com.wewebu.ow.server.ecm.OwNetwork;
import com.wewebu.ow.server.ecm.OwObjectReference;
import com.wewebu.ow.server.ecm.OwResource;
import com.wewebu.ow.server.ecmimpl.cmis.log.OwLog;
import com.wewebu.ow.server.ecmimpl.cmis.objectclasses.OwCMISObjectClass;
import com.wewebu.ow.server.ecmimpl.cmis.objectclasses.OwCMISObjectClassFactory;
import com.wewebu.ow.server.ecmimpl.cmis.propertyclasses.OwCMISPropertyClass;
import com.wewebu.ow.server.ecmimpl.cmis.version.OwCMISStandardVersionModel;
import com.wewebu.ow.server.ecmimpl.cmis.version.OwCMISVersionModel;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.exceptions.OwInvalidOperationException;
import com.wewebu.ow.server.exceptions.OwNotSupportedException;
import com.wewebu.ow.server.exceptions.OwObjectNotFoundException;
import com.wewebu.ow.server.util.OwString;
import com.wewebu.ow.server.util.OwString1;

/**
 *<p>
 * OwCMISResourceObjectModel.
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 */
public class OwCMISResourceObjectModel implements OwCMISObjectModel
{
    private static Logger LOG = OwLog.getLogger(OwCMISResourceObjectModel.class);

    private OwCMISObjectClassFactory objectClassFactory;
    private OwCMISResource resource;
    private Map<String, OwCMISObjectClass> objectClasses = new HashMap<String, OwCMISObjectClass>();
    private OwCMISObjectClass baseClass = null;
    private final boolean createBaseClass = true;
    private int cachedHierarchyDepth = 0;

    protected static final OwCMISACLModel MANAGED = new OwCMISACLModel() {
        public boolean canManageACL()
        {
            return true;
        }
    };

    protected static final OwCMISACLModel DISCOVERED = new OwCMISACLModel() {
        public boolean canManageACL()
        {
            return false;
        }
    };

    public OwCMISResourceObjectModel(OwCMISResource resource_p, OwCMISObjectClassFactory objectClassFactory_p) throws OwException
    {
        super();
        this.objectClassFactory = objectClassFactory_p;
        this.resource = resource_p;
        createTypes();
    }

    public OwCMISObjectClass getObjectClass(String strClassName_p) throws OwException
    {
        if (strClassName_p == null || strClassName_p.length() == 0)
        {
            return this.baseClass;
        }
        else
        {

            OwCMISObjectClass ret = this.objectClasses.get(strClassName_p);

            if (null == ret)
            {
                createType(strClassName_p);
                ret = objectClasses.get(strClassName_p);
                if (null == ret)
                {
                    LOG.error("OwCMISResourceObjectModel.getObjectClass(): Object class not found " + strClassName_p);
                    throw new OwObjectNotFoundException(new OwString1("ecmimpl.cmis.OwCMISResourceObjectModel.object.class.not.found.error", "Could not find object class %1!", strClassName_p));
                }
            }

            return ret;
        }
    }

    /**
     *This method is called by network to request the map.
     *@param types_p array of types or null,
     *@param excludeHiddenAndNonInstantiable_p boolean to filter hidden or not instantiable classes
     *@param rootOnly_p requesting only root classes
     *@see OwNetwork#getObjectClassNames(int[],boolean, boolean, OwResource)
     *@return a map of symbolic class name to display name
     */
    public Map<String, String> getObjectClassNames(int[] types_p, boolean excludeHiddenAndNonInstantiable_p, boolean rootOnly_p, Locale locale_p) throws OwException
    {

        HashMap<String, String> filteredTypes = new HashMap<String, String>();
        if (rootOnly_p)
        {
            if (types_p != null)
            {
                for (int type : types_p)
                {
                    switch (type)
                    {
                        case OwObjectReference.OBJECT_TYPE_DOCUMENT:
                        {
                            OwCMISObjectClass objClass = getObjectClass(EnumBaseObjectTypeIds.CMIS_DOCUMENT.value());
                            filteredTypes.put(objClass.getClassName(), objClass.getDisplayName(locale_p));
                        }
                            break;
                        case OwObjectReference.OBJECT_TYPE_FOLDER:
                        {
                            OwCMISObjectClass objClass = getObjectClass(EnumBaseObjectTypeIds.CMIS_FOLDER.value());
                            filteredTypes.put(objClass.getClassName(), objClass.getDisplayName(locale_p));
                        }
                            break;
                        default:
                            ;//nothing should be returned
                    }
                }
            }
            else
            {
                for (EnumBaseObjectTypeIds id : EnumBaseObjectTypeIds.values())
                {
                    OwCMISObjectClass objClass = getObjectClass(id.value());
                    filteredTypes.put(objClass.getClassName(), objClass.getDisplayName(locale_p));
                }
            }
        }
        else
        {
            createDescendantTypes(-1, false);
            for (OwCMISObjectClass clazz : this.objectClasses.values())
            {
                if (types_p != null)
                {
                    for (int type : types_p)
                    {
                        if (type == clazz.getType())
                        {
                            filteredTypes.put(clazz.getClassName(), clazz.getDisplayName(locale_p));
                            break;
                        }
                    }
                }
                else
                {
                    filteredTypes.put(clazz.getClassName(), clazz.getDisplayName(locale_p));
                }
            }
        }
        return filteredTypes;
    }

    /**
     * Get OwPropertyClass by full qualified name of property.<br />
     * A full qualified name of a property is a concatenation of 
     * object class name and property class name using a separator character.
     * <pre> String strFullQualifiedName = objectClassName + "." + propertyClassName;</pre>
     * <p>If strFullQualifiedName_p does not contains the separator character, the
     * CMIS base classes are searched for the property  definition.</p>
     * @param strFullQualifiedName_p String qualified property name
     * @return OwFieldDefinition of the requested property/field
     * @throws OwException if strFullQualifiedName_p is null a OwInvalidOperationException
     *  or if OwFieldDefinition was not found a OwObjectNotFoundException is thrown. 
     */
    public OwCMISPropertyClass getPropertyClass(String strFullQualifiedName_p) throws OwException
    {
        if (strFullQualifiedName_p == null)
        {//precondition
            LOG.error("OwCMISResourceObjectModel.getPropertyClass(): null property class name!");
            throw new OwInvalidOperationException(new OwString("ecmimpl.cmis.OwCMISResourceObjectModel.invalid.object.model.request.error", "Invalid object model request!"));
        }

        OwCMISPropertyClass propertyClass = null;
        if (strFullQualifiedName_p.contains("."))
        {// it is a class specific field
            StringTokenizer tok = new StringTokenizer(strFullQualifiedName_p, ".");
            String classToken = tok.nextToken();
            try
            {
                OwCMISObjectClass objClass = getObjectClass(classToken);
                propertyClass = objClass.getPropertyClass(strFullQualifiedName_p);
            }
            catch (OwObjectNotFoundException e)
            {
                throw new OwObjectNotFoundException("Property '" + strFullQualifiedName_p + "' was not found in Object class '" + classToken + "'.", e);
            }

        }
        else
        {// must be a virtual base class field
            if (this.baseClass != null)
            {
                propertyClass = this.baseClass.getPropertyClass(strFullQualifiedName_p);
            }
        }

        if (propertyClass == null)
        { //postcondition

            LOG.debug("OwCMISResourceObjectModel.getPropertyClass(): property class not found '" + strFullQualifiedName_p + "'.");
            throw new OwObjectNotFoundException(new OwString1("ecmimpl.cmis.OwCMISResourceObjectModel.property.class.not.found.error", "No such property class %1 !", strFullQualifiedName_p));
        }
        return propertyClass;
    }

    public OwCMISVersionModel getVersionModel() throws OwException
    {
        return new OwCMISStandardVersionModel();
    }

    /**
     * Creates and adds to {@link #objectClasses} the object class corresponding to the
     * CMIS type with the given ID.
     * @param typeID_p CMIS type ID
     * @throws OwException 
     */
    private void createType(String typeID_p) throws OwException
    {
        try
        {
            OwCMISConnection resourceConnection = this.resource.getConnection();
            RepositoryServicePort repositoryServicePort = resourceConnection.getRepositoryServicePort();
            CmisTypeDefinitionType type = repositoryServicePort.getTypeDefinition(this.resource.getID(), typeID_p, null);
            createType(type);
        }
        catch (CmisException ex)
        {
            OwCMISExceptionCatcher catcher = new OwCMISExceptionCatcher(ex);
            LOG.debug("OwCMISResourceObjectModel.createType(): Could not initialize a class description! A CMIS error occurred :  " + catcher.getLogMessage(), ex);
            throw catcher.toOwException(new OwString("ecmimpl.cmis.OwCMISResourceObjectModel.createType.error", "An error occurred while initialising a class description!"));
        }
    }

    /**
     * Creates and adds to {@link #objectClasses} the object class corresponding to the
     * given CMIS type .
     * @param type_p CMIS type ID
     * @throws OwException
     */
    private void createType(CmisTypeDefinitionType type_p) throws OwException
    {
        OwCMISObjectClass objectClass = this.objectClassFactory.createClass(type_p, this);

        this.objectClasses.put(objectClass.getClassName(), objectClass);

        if (LOG.isTraceEnabled())
        {
            LOG.trace("OwCMISResourceObjectModel.createType(): #> added type " + type_p.getId());
        }
    }

    /** create object class and property class descriptions
     * 
     * @throws OwException
     */
    private void createTypes() throws OwException
    {

        if (this.createBaseClass)
        {
            this.baseClass = this.objectClassFactory.createVirtualBaseClass(this);
            this.objectClasses.put(this.baseClass.getClassName(), this.baseClass);
        }

        createDescendantTypes(2, true);
    }

    /**
     * Caches base type descendant types up to the given hierarchy depth.
     * 
     * @param depth_p hierarchy depth. Must be greater than 0 or equal to -1.<br/>
     *               If -1 the whole class hierarchy is cached. 
     * @param includePropertyDefinitions_p
     * @throws OwException
     * @since 3.2.0.1
     */
    private void createDescendantTypes(int depth_p, boolean includePropertyDefinitions_p) throws OwException
    {
        if (this.cachedHierarchyDepth != -1 && (depth_p == -1 || (depth_p > 0 && this.cachedHierarchyDepth < depth_p)))
        {
            try
            {
                OwCMISConnection resourceConnection = this.resource.getConnection();
                RepositoryServicePort repositoryServicePort = resourceConnection.getRepositoryServicePort();
                //get supported base list
                CmisTypeDefinitionListType lstBase = repositoryServicePort.getTypeChildren(resource.getID(), null, Boolean.TRUE, BigInteger.TEN, BigInteger.ZERO, null);

                for (CmisTypeDefinitionType baseType : lstBase.getTypes())
                {
                    if (LOG.isTraceEnabled())
                    {
                        LOG.trace("OwResources.createDescendantTypes: Basetype : " + baseType.getId());
                    }

                    Stack<CmisTypeContainer> typeStack = new Stack<CmisTypeContainer>();

                    //we already have the base type level and have to check -1 
                    int serviceDepth = depth_p == -1 ? depth_p : depth_p - 1;
                    if (serviceDepth > 0 || serviceDepth == -1)
                    {
                        List<CmisTypeContainer> descendats = repositoryServicePort.getTypeDescendants(resource.getID(), baseType.getId(), BigInteger.valueOf(serviceDepth), includePropertyDefinitions_p, null);
                        typeStack.addAll(descendats);
                    }

                    CmisTypeContainer baseContainer = new CmisTypeContainer();
                    baseContainer.setType(baseType);
                    typeStack.push(baseContainer);

                    while (!typeStack.isEmpty())
                    {
                        CmisTypeContainer type = typeStack.pop();

                        if (!objectClasses.containsKey(type.getType().getId()))
                        {
                            createType(type.getType());
                        }

                        List<CmisTypeContainer> typeChildren = type.getChildren();
                        if (typeChildren != null)
                        {
                            typeStack.addAll(typeChildren);
                        }
                    }
                }

                this.cachedHierarchyDepth = depth_p;
            }
            catch (CmisException ex)
            {
                OwCMISExceptionCatcher catcher = new OwCMISExceptionCatcher(ex);
                LOG.error("OwCMISResourceObjectModel.createDescendantTypes(): Error occurred while initialising class description!. A CMIS error occurred :  " + catcher.getLogMessage(), ex);
                throw catcher.toOwException(new OwString("ecmimpl.cmis.OwCMISResourceObjectModel.createType.error", "An error occurred while initialising a class description!"));
            }
        }
    }

    public OwCMISACLModel getACLModel() throws OwException
    {
        CmisRepositoryInfoType repoInfo = this.resource.getRepositoryInfo();
        CmisRepositoryCapabilitiesType repoCapabilities = repoInfo.getCapabilities();
        switch (repoCapabilities.getCapabilityACL())
        {
            case MANAGE:
            {
                return MANAGED;
            }

            case DISCOVER:
            {
                return DISCOVERED;
            }
            default:
            {
                LOG.error("OwCMISResourceObjectModel.getACLModel():ACL not supported!");
                throw new OwNotSupportedException(new OwString("ecmimpl.cmis.OwCMISResourceObjectModel.acl.error", "ACL not supported !"));
            }
        }
    }

    public OwCMISResource getResource()
    {
        return this.resource;
    }

}