package com.wewebu.ow.server.ecmimpl.cmis;

import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;
import javax.xml.namespace.QName;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPEnvelope;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPHeader;
import javax.xml.soap.SOAPHeaderElement;
import javax.xml.soap.SOAPMessage;
import javax.xml.ws.handler.MessageContext;
import javax.xml.ws.handler.soap.SOAPMessageContext;

import org.apache.log4j.Logger;

import com.wewebu.ow.server.ecmimpl.cmis.log.OwLog;
import com.wewebu.ow.server.ecmimpl.cmis.wshandler.OwCMISSecurityHandlerInterface;

/**
 *<p>
 * WSS handler for CMIS user PasswordText authentication.
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 */
public class OwCMISSecurityHandler implements OwCMISSecurityHandlerInterface
{
    private static final Logger LOG = OwLog.getLogger(OwCMISSecurityHandler.class);
    /**Security Header*/
    protected static final QName SECURITY = new QName(WS_SECEXT, "Security", WS_SECEXT_PREFIX);
    /**Set of headers which are supported by this security handler.*/
    protected static final Set<QName> HEADER_SUPPORT = new HashSet<QName>();
    static
    {
        HEADER_SUPPORT.add(SECURITY);
    }

    private String password;
    private String username;
    private Locale locale;
    private String wsi18nTimeZone;
    private TimeZone wssecTimeZone;

    /**
     * New Constructor to define directly the parameter of the security handler.
     * @param username_p String
     * @param password_p String 
     * @param locale_p Locale  can be null
     * @param wssi18nTimeZone_p TimeZone can be null
     * @param wssecTimeZone_p TimeZone if null will be set to UTC
     * @since 4.0.0.0
     */
    public OwCMISSecurityHandler(String username_p, String password_p, Locale locale_p, TimeZone wssi18nTimeZone_p, TimeZone wssecTimeZone_p)
    {
        super();
        this.password = password_p;
        this.username = username_p;
        this.locale = locale_p;
        setWsi18nTimeZone(wssi18nTimeZone_p);
        this.wssecTimeZone = wssecTimeZone_p == null ? TimeZone.getTimeZone("UTC") : wssecTimeZone_p;
    }

    /**
     * Create a handler with UTC time zone settings, without any locale information.
     * @param username_p String
     * @param password_p String
     */
    public OwCMISSecurityHandler(String username_p, String password_p)
    {
        this(username_p, password_p, null, TimeZone.getTimeZone("UTC"), TimeZone.getTimeZone("UTC"));
    }

    public boolean handleMessage(SOAPMessageContext messageContext_p)
    {
        if ((Boolean) messageContext_p.get(MessageContext.MESSAGE_OUTBOUND_PROPERTY))
        {
            try
            {
                SOAPMessage msg = messageContext_p.getMessage();
                SOAPEnvelope envelope = msg.getSOAPPart().getEnvelope();
                SOAPHeader soapHeader;
                if (envelope.getHeader() != null)
                {
                    soapHeader = envelope.getHeader();
                }
                else
                {
                    soapHeader = envelope.addHeader();
                }

                createWssecHeader(soapHeader);
                createWsi18nHeader(soapHeader);

            }
            catch (Exception ex)
            {
                LOG.error("OwCMISSecurityHandler.handleMessage(): SOAP message handling failed!", ex);
                return false;
            }
        }
        return true;
    }

    public Set<QName> getHeaders()
    {
        return HEADER_SUPPORT;
    }

    public boolean handleFault(SOAPMessageContext context_p)
    {
        return true;
    }

    public void close(MessageContext context_p)
    {

    }

    public void setLocale(Locale locale_p)
    {
        this.locale = locale_p;
    }

    public Locale getLocale()
    {
        return this.locale;
    }

    public void setWsi18nTimeZone(TimeZone timeZone_p)
    {
        this.wsi18nTimeZone = timeZone_p == null ? null : createTimezoneString(timeZone_p);
    }

    /**
     * Simple helper to create GMT representation
     * of the client time zone.
     * @return String representing client time zone.
     * @since 3.2.0.0
     */
    protected String createTimezoneString(TimeZone ctz)
    {
        StringBuffer tz = new StringBuffer("GMT");
        tz.append(ctz.getRawOffset() < 0 ? "-" : "+");
        int min = Math.abs(ctz.getRawOffset() / 60000);
        int hour = min / 60;
        min = min % 60;
        tz.append(hour < 10 ? "0" : "");
        tz.append(Integer.toString(hour));
        tz.append(min < 10 ? "0" : "");
        tz.append(Integer.toString(min));
        return tz.toString();
    }

    public void init(Map<String, String> arr_p)
    {
        if (this.password == null)
        {
            password = arr_p.get(OwCMISSecurityHandlerInterface.PWD);
        }

        if (this.username == null)
        {
            username = arr_p.get(OwCMISSecurityHandlerInterface.USER);
        }
    }

    /**(overridable)
     * Create the depending header structure depending on the implemented
     * security handling.
     * @param soapHeader_p SOAPHeader to attach to
     * @throws SOAPException if problems with creation of the attaching new child elements
     * @throws DatatypeConfigurationException if could not create DatatypeFactory for XMLGregorianCalendar
     * @since 3.1.0.0
     */
    protected void createWssecHeader(SOAPHeader soapHeader_p) throws SOAPException, DatatypeConfigurationException
    {
        DatatypeFactory factory = DatatypeFactory.newInstance();
        GregorianCalendar now = new GregorianCalendar(wssecTimeZone);
        GregorianCalendar fiveMinutesLater = new GregorianCalendar(wssecTimeZone);
        fiveMinutesLater.add(Calendar.MINUTE, 5);

        XMLGregorianCalendar xmlNow = factory.newXMLGregorianCalendar(now);
        XMLGregorianCalendar xmlFiveMinutesLater = factory.newXMLGregorianCalendar(fiveMinutesLater);

        SOAPHeaderElement headerElement = soapHeader_p.addHeaderElement(SECURITY);
        headerElement.setMustUnderstand(false);//set this to true if you want a strict WS_SEC handling

        SOAPElement timestamp = headerElement.addChildElement(new QName(WS_SECUTILITY, "Timestamp", WS_SECUTILITY_PREFIX));
        timestamp.addAttribute(new QName(WS_SECUTILITY, "Id", WS_SECUTILITY_PREFIX), "Timestamp");

        SOAPElement created = timestamp.addChildElement(new QName(WS_SECUTILITY, "Created", WS_SECUTILITY_PREFIX));
        created.addTextNode(xmlNow.toXMLFormat());

        SOAPElement expired = timestamp.addChildElement(new QName(WS_SECUTILITY, "Expires", WS_SECUTILITY_PREFIX));
        expired.addTextNode(xmlFiveMinutesLater.toXMLFormat());

        SOAPElement usernameToken = headerElement.addChildElement(new QName(WS_SECEXT, "UsernameToken"));
        SOAPElement username = usernameToken.addChildElement(new QName(WS_SECEXT, "Username"));
        username.addTextNode(this.username);

        SOAPElement password = usernameToken.addChildElement(new QName(WS_SECEXT, "Password"));
        password.addAttribute(new QName("Type"), WS_USER_TOKEN_PROFILE + "#PasswordText");
        password.addTextNode(this.password);
    }

    /**(overridable)
     * Create an i18n-header if a locale was set for this security handler.
     * @param soapHeader_p SOAPHeader where to attach the new i18n-elements
     * @throws SOAPException if problem occurs during attaching/creating header element
     * @since 3.1.0.0
     */
    protected void createWsi18nHeader(SOAPHeader soapHeader_p) throws SOAPException
    {
        if (getLocale() != null)
        {
            SOAPHeaderElement headerElement = soapHeader_p.addHeaderElement(new QName(WS_I18N, "international", WS_I18N_PREFIX));
            headerElement.setMustUnderstand(false);

            SOAPElement locElem = headerElement.addChildElement(new QName(WS_I18N, "locale", WS_I18N_PREFIX));
            locElem.addTextNode(getLocale().toString());

            if (getWsi18nTimeZone() != null)
            {
                SOAPElement timeElem = headerElement.addChildElement(new QName(WS_I18N, "tz", WS_I18N_PREFIX));
                timeElem.addTextNode(getWsi18nTimeZone());
            }
        }
    }

    @Deprecated
    public void setTimezoneString(String timeZone_p)
    {
        this.wsi18nTimeZone = timeZone_p;
    }

    @Deprecated
    public String getTimezoneString()
    {
        return getWsi18nTimeZone();
    }

    public void setWsi18nTimeZone(String timeZone_p)
    {
        this.wsi18nTimeZone = timeZone_p;
    }

    public String getWsi18nTimeZone()
    {
        return this.wsi18nTimeZone;
    }

    /**
     * Get the WsSecTimeZone to be used
     * for security header TTL (time to live) definition. 
     * @return Timezone
     * @since 4.0.0.0
     */
    protected TimeZone getWsSecTimeZone()
    {
        return this.wssecTimeZone;
    }
}