package com.wewebu.ow.server.ecmimpl.cmis;

import org.apache.log4j.Logger;
import org.oasis_open.docs.ns.cmis.core._200908.CmisObjectType;
import org.oasis_open.docs.ns.cmis.core._200908.EnumIncludeRelationships;
import org.oasis_open.docs.ns.cmis.ws._200908.CmisException;
import org.oasis_open.docs.ns.cmis.ws._200908.ObjectServicePort;

import com.wewebu.ow.server.ecmimpl.cmis.log.OwLog;
import com.wewebu.ow.server.ecmimpl.cmis.objectclasses.OwCMISObject;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.util.OwObjectIDCodeUtil;

/**
 *<p>
 * OwCMISSimpleDMSID.
 * This class simple creates from the given strings a semicolon separated
 * DMSID representation, starting always with the DMS-prefix.
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 */
public class OwCMISSimpleDMSID implements OwCMISDMSID
{
    private static final Logger LOG = OwLog.getLogger(OwCMISSimpleDMSID.class);

    private String dmsPrefix;
    private String cmisID;
    private String resourceID;

    public OwCMISSimpleDMSID(String dmsPrefix_p, String resourceID_p, String cmisID_p)
    {
        super();
        this.dmsPrefix = dmsPrefix_p;
        this.cmisID = cmisID_p;
        this.resourceID = resourceID_p;
    }

    public String getCMISID()
    {
        return this.cmisID;
    }

    /**
     * This method concatenate the given String using semicolon &quot;;&quot; as separator,
     * DMSID syntax is:
     * <p>
     * <code>DMS-prefix + &quot;,&quot; + resourceID + &quot;,&quot; + cmisObjectID</code>
     * </p>
     * @return String representing DMSID
     */
    public String getDMSIDString()
    {
        return createDMSID(this.dmsPrefix, this.resourceID, this.cmisID);
    }

    public OwCMISObject getObject(OwCMISNetwork network_p) throws OwException
    {
        CmisObjectType obj;
        String decCmisID = OwObjectIDCodeUtil.decode(this.cmisID);
        try
        {
            ObjectServicePort objectServicePort = network_p.getObjectServicePort();
            obj = objectServicePort.getObject(this.resourceID, decCmisID, null, Boolean.TRUE, EnumIncludeRelationships.NONE, null, Boolean.FALSE, Boolean.FALSE, null);
        }
        catch (CmisException e)
        {
            OwCMISExceptionCatcher catcher = new OwCMISExceptionCatcher(e);
            LOG.error("OwCMISSimpleDMSID.getObject(): Could not retrieve object for DMSID (" + getDMSIDString() + ")! A CMIS error occurred :  " + catcher.getLogMessage(), e);
            throw catcher.toOwException(network_p.getContext().localize("ecmimpl.cmis.OwCMISSimpleDMSID.getObject.error", "Could not retrieve object!"));
        }

        OwCMISResource resource = network_p.getResource(this.resourceID);

        return network_p.createCMISObject(obj, resource);
    }

    public String getResourceID()
    {
        return this.resourceID;
    }

    /**
     * Static method which is used for creation of DMSID without an instance of OwCMISSimpleDMSID.
     * <p>Create a DMSID which looks like: <br />
     * <code>DMS-prefix + &quot;,&quot; + resourceID + &quot;,&quot; + cmisObjectID</code><br />
     * the cmisID_p will be escaped using the {@link OwObjectIDCodeUtil#encode(String)} method.
     * </p>
     * @param dmsPrefix_p String prefix to be used
     * @param resourceID_p String resource ID (object store or repository)
     * @param cmisID_p String object id
     * @return String representing a DMSID
     * @since 3.1.0.0
     */
    public static String createDMSID(String dmsPrefix_p, String resourceID_p, String cmisID_p)
    {
        /*StringBuilder will save up to 10 microseconds for DMSID creation
         * compared to the OwEscapedStringTokenizer.createDelimitedString(Collection)*/
        StringBuilder builder = new StringBuilder(dmsPrefix_p);
        builder.append(OwCMISDMSID.DMSID_SEPARATOR);
        builder.append(resourceID_p);
        builder.append(OwCMISDMSID.DMSID_SEPARATOR);
        if (cmisID_p.indexOf(OwObjectIDCodeUtil.PATH_SEPARATOR_CHAR) > 0)
        {
            builder.append(OwObjectIDCodeUtil.encode(cmisID_p));
        }
        else
        {
            builder.append(cmisID_p);
        }

        return builder.toString();
    }

}