package com.wewebu.ow.server.ecmimpl.cmis.alfresco;

import java.io.IOException;
import java.util.List;

import org.apache.log4j.Logger;
import org.oasis_open.docs.ns.cmis.core._200908.CmisTypeDefinitionType;

import com.wewebu.ow.server.ecmimpl.cmis.OwCMISNetwork;
import com.wewebu.ow.server.ecmimpl.cmis.log.OwLog;
import com.wewebu.ow.server.ecmimpl.cmis.objectclasses.OwCMISObjectClass;
import com.wewebu.ow.server.exceptions.OwConfigurationException;
import com.wewebu.ow.server.exceptions.OwException;

/**
 *<p>
 * Helper class for mandatory aspect definition.
 * This class is needed during creation process to verify and retrieve 
 * mandatory aspect definitions from Alfresco system.
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 *@since 3.2.0.0
 */
public class OwCMISALAspectsService
{
    private static final Logger LOG = OwLog.getLogger(OwCMISALAspectsService.class);

    /**
     * Method which will use a OwCMISALAspectRetriever to get the information about Aspects of provided class.
     * Will create a cache object and attach it to the native representation. In case that the OwCMISObjectClass
     * does not have a native object type, the method will ignore the call and simply return.
     * <p>Attention: Existing cache will be overwritten, this method will not verify the cache ({@link #ensureMandatroyAspects(OwCMISNetwork, OwCMISObjectClass)} at all.</p> 
     * @param network OwCMISNetwork to be used for additional information
     * @param objClass OwCMISObjectClass to retrieve Aspect information
     * @throws OwException if problem occur with creating instance of retriever
     * @throws IOException if problem occur during network communication to (ECM) back-end system
     */
    public void retrieveManadatoryAspects(OwCMISNetwork network, OwCMISObjectClass objClass) throws OwException, IOException
    {
        CmisTypeDefinitionType nativeType = objClass.getNativeObject();
        if (nativeType != null)
        {
            OwCMISALAspectRetriever retriever = getAspectRetriever(network);
            List<String> lst = retriever.retrieveAspectIds(network, objClass);

            List<Object> nativeAny = nativeType.getAny();
            OwCMISALMandatoryAspectCache cache = OwCMISALUtil.findFirst(nativeAny, OwCMISALMandatoryAspectCache.class);
            if (cache == null)
            {//create and add cache object
                cache = new OwCMISALMandatoryAspectCache(lst);
                nativeAny.add(cache);
            }
            else
            {//reset existing cache
                cache.getMandatoryAspectIds().clear();
                cache.getMandatoryAspectIds().addAll(lst);
            }
        }
        else
        {
            LOG.debug("OwCMISALAspectsService.retrieveManadatoryAspects: Unknown OwCMISObjectClass could not retrieve native type representation");
        }

    }

    /**
     * Verify cache initialized/existent, before retrieve of information is requested.
     * If no cache object was found, the method will delegate to {@link #retrieveManadatoryAspects(OwCMISNetwork, OwCMISObjectClass)}.
     * @param network OwCMISNetwork
     * @param objClass OwCMISObjectClass to ensure aspects cache
     * @throws OwException see {@link #retrieveManadatoryAspects(OwCMISNetwork, OwCMISObjectClass) retrieveManadatoryAspects(...)}
     * @throws IOException see {@link #retrieveManadatoryAspects(OwCMISNetwork, OwCMISObjectClass) retrieveManadatoryAspects(...)}
     * @see #retrieveManadatoryAspects(OwCMISNetwork, OwCMISObjectClass) 
     */
    public void ensureMandatroyAspects(OwCMISNetwork network, OwCMISObjectClass objClass) throws OwException, IOException
    {
        CmisTypeDefinitionType nativeType = objClass.getNativeObject();
        if (nativeType != null)
        {
            if (!isCacheInitialised(objClass))
            {
                retrieveManadatoryAspects(network, objClass);
            }
        }
        else
        {
            LOG.debug("OwCMISALAspectsService.ensureMandatroyAspects: Unknown OwCMISObjectClass could not retrieve native type representation");
        }
    }

    /**
     * Check if mandatory aspects information already exist in cache or not.
     * @param objClass OwCMISObjectClass to be verified
     * @return true if respective cache exist.
     */
    public static boolean isCacheInitialised(OwCMISObjectClass objClass)
    {
        CmisTypeDefinitionType nativeType = objClass.getNativeObject();
        if (nativeType != null)
        {
            List<Object> lst = nativeType.getAny();
            OwCMISALMandatoryAspectCache cache = OwCMISALUtil.findFirst(lst, OwCMISALMandatoryAspectCache.class);
            return cache != null;
        }
        else
        {
            LOG.debug("OwCMISALAspectsService.isCacheInitialised: Unknown OwCMISObjectClass could not retrieve native type representation");
            return false;
        }
    }

    /**
     * Read the configuration and create the OwCMISALAspectRetriever,
     * which will be used to get Mandatory Aspects information for specific types.
     * <p>A basic retriever is instantiated if no explicit configuration is found,
     * else an instance from defined retriever is created using default constructor.</p>
     * @param network OwCMISNetwork to be used for configuration information
     * @return OwCMISALAspectRetriever which is configured
     * @throws OwException if fail to create instance of retriever
     */
    protected OwCMISALAspectRetriever getAspectRetriever(OwCMISNetwork network) throws OwException
    {
        OwCMISALAspectRetriever instance = null;
        String retClass = network.getNetworkConfiguration().getConfigNode().getSafeTextValue(OwCMISALAspectRetriever.RETRIEVER_CLASS, "com.wewebu.ow.server.ecmimpl.cmis.alfresco.OwCMISALBasicAspectRetriever");
        try
        {
            Class<OwCMISALAspectRetriever> clazz = (Class<OwCMISALAspectRetriever>) Class.forName(retClass);
            instance = clazz.newInstance();
        }
        catch (InstantiationException ex)
        {
            LOG.error("Could not instantiate OwCMISALAspectRetriever: " + retClass, ex);
            throw new OwConfigurationException("Instantiation failed for class = " + retClass, ex);
        }
        catch (IllegalAccessException e)
        {
            LOG.error("Class cannot be accessed, class = " + retClass, e);
            throw new OwConfigurationException("Instantiation failure (not accssible) for class = " + retClass, e);
        }
        catch (ClassNotFoundException e)
        {
            LOG.fatal("Could not find specified class for OwCMISALAspectRetriever", e);
            throw new OwConfigurationException("Could not find class = " + retClass, e);
        }
        return instance;
    }
}
