package com.wewebu.ow.server.ecmimpl.cmis.alfresco;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.apache.commons.codec.binary.Base64;
import org.apache.log4j.Logger;
import org.json.simple.JSONObject;
import org.json.simple.parser.JSONParser;
import org.json.simple.parser.ParseException;

import com.wewebu.ow.server.ecmimpl.OwCredentialsConstants;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISCredentials;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISNetwork;
import com.wewebu.ow.server.ecmimpl.cmis.log.OwLog;
import com.wewebu.ow.server.ecmimpl.cmis.objectclasses.OwCMISObjectClass;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.exceptions.OwInvalidOperationException;

/**
 *<p>
 * Mandatory Aspects Retrieve with Basic HTTP Authentication.
 * Password and user name will be base64 encoded and provided
 * in a HTTP Authorization header value.<br />
 * Implementation is based on the Alfresco Rest-API
 * http://&lt;server&gt;/alfresco/service/api/classes/
 *</p>
 *
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 *@since 3.2.0.0
 */
public class OwCMISALBasicAspectRetriever implements OwCMISALAspectRetriever
{
    private static final Logger LOG = OwLog.getLogger(OwCMISALBasicAspectRetriever.class);

    public List<String> retrieveAspectIds(OwCMISNetwork network, OwCMISObjectClass clazz) throws IOException, OwException
    {
        URL url = getURL(network, clazz.getClassName());
        URLConnection uc = url.openConnection();
        uc.setRequestProperty("Authorization", createAuthValue(network));

        return processInputStream(uc.getInputStream(), network);
    }

    /**
     * Read connection URL from network configuration, if
     * not available use RepositoryService WSDL definition and 
     * extract location depending on that setting.  
     * @param network OwCMISNetwork to read configuration
     * @param classId String representing the OwObjectClass.getClassName() Id
     * @return URL from where to retrieve Mandatory Aspect definition
     * @throws MalformedURLException if URL could not be created/constructed
     * @throws OwException problem retrieving configuration from network 
     */
    protected URL getURL(OwCMISNetwork network, String classId) throws MalformedURLException, OwException
    {
        String strURL = network.getNetworkConfiguration().getConfigNode().getSafeTextValue(OwCMISALAspectRetriever.RETRIEVER_URL, null);
        if (strURL == null)
        {
            URL repo = network.getNetworkConfiguration().getConfigNode().getURLFromNode(OwCMISCredentials.CONF_NODE_WSDL_REPOSITORY);
            StringBuilder str = new StringBuilder(repo.getProtocol());
            str.append("://").append(repo.getAuthority());
            strURL = repo.getPath();
            strURL = strURL.substring(0, strURL.indexOf('/', 1));
            str.append(strURL);
            str.append("/service/api/classes/");
            strURL = str.toString();
        }

        return new URL(strURL + getRequestTypeId(network, classId));
    }

    /**
     * Transform the class Id into a representation which can be used
     * for the service call. Will transform every &quot;cmis:&quot;
     * specific namespace into the following types:
     * <ul>
     * <li>cmis:document == cm:content</li>
     * <li>cmis:folder == cm:folder</li>
     * <li>cmis:* == cm:cmobject</li>
     * </ul>
     * In case of other namespace the leading prefix will be removed
     * to represent native definition. At the end By default all colon (:) 
     * occurrence will be removed by an underscore (_) and the String returned.
     * @param network OwCMISNetwork
     * @param classId String
     * @return String which should be used to request Aspect information 
     */
    protected String getRequestTypeId(OwCMISNetwork network, String classId)
    {
        String typeId;
        if (classId.startsWith("cmis:"))
        {
            switch (classId.charAt(5))
            {
                case 'D':
                case 'd':
                    typeId = "cm:content";
                    break;

                case 'F':
                case 'f':
                    typeId = "cm:folder";
                    break;

                default:
                    typeId = "cm:cmobject";
            }
        }
        else
        {
            typeId = classId.substring(classId.indexOf(':') + 1);
        }
        return typeId.replace(':', '_');
    }

    /**
     * Create the Value for the HTTP &quot;Authorization&quot; header, 
     * which will be attached during retrieve REST-call.
     * @param network OwCMISNetwork
     * @return String representing the authorization information 
     * @throws OwException if could not retrieve the credentials
     */
    protected String createAuthValue(OwCMISNetwork network) throws OwException
    {
        StringBuilder authValue = new StringBuilder(network.getCredentials().getAuthInfo(OwCredentialsConstants.LOGIN_USR));
        authValue.append('\u003a');//colon
        authValue.append(network.getCredentials().getAuthInfo(OwCredentialsConstants.LOGIN_PWD));

        return "Basic " + new String(Base64.encodeBase64(authValue.toString().getBytes()));
    }

    /**
     * Process input stream and retrieve only the id's which are defined as &quot;defaultAspects&quot;
     * @param stream InputStream from service call
     * @param network OwCMISNetwork instance for localization or other information
     * @return List of String's representing the Id's
     * @throws IOException in case of error while reading input stream
     * @throws OwException if parsing of service answer was not successful/ could not proceed
     */
    protected List<String> processInputStream(InputStream stream, OwCMISNetwork network) throws IOException, OwException
    {
        LinkedList<String> lst = new LinkedList<String>();
        StringBuilder builder = new StringBuilder();
        InputStreamReader reader = null;
        try
        {
            reader = new InputStreamReader(new BufferedInputStream(stream));
            char[] buffer = new char[256];
            int read;
            while ((read = reader.read(buffer)) >= 0)
            {
                builder.append(buffer, 0, read);
            }
        }
        finally
        {
            if (reader != null)
            {
                reader.close();
            }
        }
        JSONParser parser = new JSONParser();
        Object obj = null;
        try
        {
            obj = parser.parse(builder.toString());
        }
        catch (ParseException e)
        {//no logging, CMISALAspectsServices will be the main controller and "LOGGER"
            throw new OwInvalidOperationException(network.getContext().localize("OwCMISALBasicAspectRetriever.json.parse.ex", "Could not parse delivered JSON string"), e);
        }
        JSONObject jsonObj = (JSONObject) obj;
        jsonObj = (JSONObject) jsonObj.get("defaultAspects");

        Iterator<?> it = jsonObj.keySet().iterator();
        while (it.hasNext())
        {
            String name = it.next().toString();
            if (!(name.startsWith("sys") || name.startsWith("cm")))
            {
                lst.add(OwCMISALAspectRetriever.PREFIX_ASPECT + name);
            }
            else
            {
                LOG.debug("OwCMISALBasicAspectRetriever.processInputStream : Aspect " + name + " filtered out.");
            }
        }

        return lst;
    }
}
