package com.wewebu.ow.server.ecmimpl.cmis.alfresco;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;
import org.oasis_open.docs.ns.cmis.core._200908.CmisObjectType;
import org.oasis_open.docs.ns.cmis.core._200908.CmisTypeDefinitionType;

import com.wewebu.ow.server.ecm.OwContentCollection;
import com.wewebu.ow.server.ecm.OwNetwork;
import com.wewebu.ow.server.ecm.OwObject;
import com.wewebu.ow.server.ecm.OwPermissionCollection;
import com.wewebu.ow.server.ecm.OwPropertyCollection;
import com.wewebu.ow.server.ecm.OwResource;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISNetwork;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISObjectModel;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISResource;
import com.wewebu.ow.server.ecmimpl.cmis.log.OwLog;
import com.wewebu.ow.server.ecmimpl.cmis.object.OwCMISQueryContext;
import com.wewebu.ow.server.ecmimpl.cmis.objectclasses.OwCMISObject;
import com.wewebu.ow.server.ecmimpl.cmis.objectclasses.OwCMISObjectClass;
import com.wewebu.ow.server.ecmimpl.cmis.propertyclasses.OwCMISPropertyClass;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.exceptions.OwInvalidOperationException;
import com.wewebu.ow.server.exceptions.OwObjectNotFoundException;
import com.wewebu.ow.server.util.OwString3;

/**
 *<p>
 * Base Alfresco specific object class for Aspects handling.
 * Will do a pre- and post-processing during 
 * {@link #createNewObject(OwCMISNetwork, boolean, Object, OwResource, OwPropertyCollection, OwPermissionCollection, OwContentCollection, OwObject, String, String, boolean) createNewObject(...)}
 * to handle Aspect values correct. 
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 *@since 3.2.0.0
 */
@SuppressWarnings("rawtypes")
public class OwCMISALObjectClass implements OwCMISObjectClass
{
    private static final Logger LOG = OwLog.getLogger(OwCMISALObjectClass.class);

    private OwCMISNetwork network;

    private OwCMISObjectClass primaryClass;

    /** preferred property order list */
    private List<String> preferredPropertyOrder;

    public OwCMISALObjectClass(OwCMISObjectClass primaryClass, OwCMISNetwork network)
    {
        this.network = network;
        this.primaryClass = primaryClass;
    }

    /**
     * Return a list of Id's representing the Aspects applied to this type.
     * @return List of Strings representing the Id of an Aspect
     * @throws OwException
     */
    public List<String> getAppliedAspectTypes() throws OwException
    {
        OwCMISALMandatoryAspectCache cache = OwCMISALUtil.findFirst(getNativeObject().getAny(), OwCMISALMandatoryAspectCache.class);
        return cache == null ? new LinkedList<String>() : cache.getMandatoryAspectIds();
    }

    /**
     * Check if an Aspect is applied to this type.
     * @param aspectName_p String Id 
     * @return boolean true only if it exist in the {@link #getAppliedAspectTypes()} collection
     * @throws OwException
     */
    public boolean isAspectApplied(String aspectName_p) throws OwException
    {
        List<String> aspectTypes = getAppliedAspectTypes();
        return aspectTypes.contains(aspectName_p);
    }

    public boolean canCreateNewObject() throws OwException
    {
        return getPrimaryClass().canCreateNewObject();
    }

    public OwCMISObject createCMISObject(OwCMISNetwork network_p, CmisObjectType object_p, boolean preserveVersion_p) throws OwException
    {
        return getPrimaryClass().createCMISObject(network_p, object_p, preserveVersion_p);
    }

    @SuppressWarnings("unchecked")
    public String createNewObject(OwCMISNetwork network_p, boolean promote_p, Object mode_p, OwResource resource_p, OwPropertyCollection properties_p, OwPermissionCollection permissions_p, OwContentCollection content_p, OwObject parent_p,
            String strMimeType_p, String strMimeParameter_p, boolean keepCheckedOut_p) throws OwException
    {

        //basic creation
        String id = getPrimaryClass().createNewObject(network_p, promote_p, mode_p, resource_p, properties_p, permissions_p, content_p, parent_p, strMimeType_p, strMimeParameter_p, keepCheckedOut_p);

        return id;
    }

    public Map getChildNames(OwNetwork network_p, boolean excludeHiddenAndNonInstantiable_p) throws Exception
    {
        return getPrimaryClass().getChildNames(network_p, excludeHiddenAndNonInstantiable_p);
    }

    public List getChilds(OwNetwork network_p, boolean excludeHiddenAndNonInstantiable_p) throws Exception
    {
        List base = getPrimaryClass().getChilds(network_p, excludeHiddenAndNonInstantiable_p);
        if (base == null || base.isEmpty())
        {
            return base;
        }
        else
        {
            LinkedList<OwCMISObjectClass> lst = new LinkedList<OwCMISObjectClass>();
            Iterator<?> it = base.iterator();
            while (it.hasNext())
            {
                OwCMISObjectClass toWrap = (OwCMISObjectClass) it.next();
                lst.add(new OwCMISALObjectClass(toWrap, getNetwork()));
            }
            return lst;
        }
    }

    public String getClassName()
    {
        return getPrimaryClass().getClassName();
    }

    public String getDescription(Locale locale_p)
    {
        return getPrimaryClass().getDescription(locale_p);
    }

    public String getDisplayName(Locale locale_p)
    {
        return getPrimaryClass().getDisplayName(locale_p);
    }

    public String getMimetype()
    {
        return getPrimaryClass().getMimetype();
    }

    public List getModes(int operation_p) throws Exception
    {
        return getPrimaryClass().getModes(operation_p);
    }

    public String getNamePropertyName() throws OwException
    {
        return getPrimaryClass().getNamePropertyName();
    }

    public OwCMISObjectClass getParent() throws OwException
    {
        return getPrimaryClass().getParent();
    }

    public OwCMISPropertyClass getPropertyClass(String propertyClassName_p) throws OwException
    {
        try
        {
            return getPrimaryClass().getPropertyClass(propertyClassName_p);
        }
        catch (OwObjectNotFoundException e1)
        {
            //avoiding over-logging : we are trying to find the property definition class
            //TODO: a check property class definition method is needed !!!

            OwCMISResource resource = getResource();
            OwCMISObjectModel objectModel = resource.getObjectModel();

            List<String> appliedAspects = getAppliedAspectTypes();
            for (String aspect : appliedAspects)
            {
                OwCMISObjectClass aspectClass = objectModel.getObjectClass(aspect);
                try
                {
                    return aspectClass.getPropertyClass(propertyClassName_p);
                }
                catch (OwObjectNotFoundException e2)
                {
                    //avoiding over-logging : we are trying to find the property definition class
                    //TODO: a check property class definition method is needed !!!
                }
            }
            LOG.debug("OwCMISALObjectClass.getPropertyClass : no such property " + propertyClassName_p + " in primary class " + getClassName() + " or applied aspects " + appliedAspects);
            throw new OwObjectNotFoundException(new OwString3("ecmimpl.cmis.OwCMISAlfrescoObjectClass.invalid.property.class.tree", "Property %1 is neither defined in object class %2 nor in the applied Alfresco aspects %3", propertyClassName_p,
                    getClassName(), appliedAspects.toString()));
        }
    }

    public Map<String, OwCMISPropertyClass> getPropertyClasses() throws OwException
    {
        Map<String, OwCMISPropertyClass> propertyClasses = new HashMap<String, OwCMISPropertyClass>(getPrimaryClass().getPropertyClasses());

        Map<String, OwCMISPropertyClass> aspectProps = getAllAspectProperties();

        if (!aspectProps.isEmpty())
        {
            aspectProps.remove("cmis:policy.cmis:name");

            propertyClasses.putAll(aspectProps);
        }
        return propertyClasses;
    }

    public Collection<String> getPropertyClassNames() throws OwException
    {
        OwCMISObjectClass myPrimaryClass = getPrimaryClass();

        Collection<String> propertyClassNames = myPrimaryClass.getPropertyClassNames();

        Collection<String> propertyClasses = new LinkedList<String>(propertyClassNames);

        Map<String, OwCMISPropertyClass> aspectProps = getAllAspectProperties();

        if (!aspectProps.isEmpty())
        {
            aspectProps.remove("cmis:policy.cmis:name");

            propertyClasses.addAll(aspectProps.keySet());
        }

        return reorderProperties(propertyClasses);

    }

    public Set<OwCMISPropertyClass> getQueryablePropertyClasses(OwCMISQueryContext context_p) throws OwException
    {
        return getPrimaryClass().getQueryablePropertyClasses(context_p);
    }

    public String getQueryName()
    {
        return getPrimaryClass().getQueryName();
    }

    public OwCMISResource getResource()
    {
        return getPrimaryClass().getResource();
    }

    public int getType()
    {
        return getPrimaryClass().getType();
    }

    public boolean hasChilds(OwNetwork network_p, boolean excludeHiddenAndNonInstantiable_p, int context_p) throws Exception
    {
        return getPrimaryClass().hasChilds(network_p, excludeHiddenAndNonInstantiable_p, context_p);
    }

    public boolean hasVersionSeries() throws Exception
    {
        return getPrimaryClass().hasVersionSeries();
    }

    public boolean isContentRequired() throws OwException
    {
        return getPrimaryClass().isContentRequired();
    }

    public boolean isHidden() throws Exception
    {
        return getPrimaryClass().isHidden();
    }

    public boolean isQueryable()
    {
        return getPrimaryClass().isQueryable();
    }

    public void subclassedBy(OwCMISObjectClass subclass_p) throws OwInvalidOperationException
    {
        getPrimaryClass().subclassedBy(subclass_p);
    }

    public boolean isAssignableFrom(OwCMISObjectClass class_p) throws OwException
    {
        return getPrimaryClass().isAssignableFrom(class_p);
    }

    public CmisTypeDefinitionType getNativeObject()
    {
        return getPrimaryClass().getNativeObject();
    }

    /**
     * Getter method to the native/primary class definition.
     * @return OwCMISObjectClass
     */
    protected OwCMISObjectClass getPrimaryClass()
    {
        return primaryClass;
    }

    /**
     * Getter method for the current network.
     * @return OwCMISNetwork
     */
    protected OwCMISNetwork getNetwork()
    {
        return network;
    }

    /**
     * Helper used to iterate over the {@link #getAppliedAspectTypes()} calling
     * the {@link #getAspectProperties(String)}.
     * @return Map with specific properties or empty map.
     * @throws OwException
     */
    protected Map<String, OwCMISPropertyClass> getAllAspectProperties() throws OwException
    {
        List<String> appliedAspects = getAppliedAspectTypes();
        Map<String, OwCMISPropertyClass> aspectProps = null;
        for (String appliedAspect : appliedAspects)
        {
            if (aspectProps == null)
            {
                aspectProps = getAspectProperties(appliedAspect);
            }
            else
            {
                aspectProps.putAll(getAspectProperties(appliedAspect));
            }
        }
        return aspectProps == null ? new LinkedHashMap<String, OwCMISPropertyClass>() : aspectProps;
    }

    /**
     * Helper method to filter out only editable Aspect properties.
     * All system properties will be removed from returned collection.
     * <p>May be some properties must be removed in post processing
     * of the returned map! (cmis:name or cmis:policy.cmis:name)</p>
     * @param aspectClassId_p String id/symbolic name of the Aspect
     * @return Map of property-id to property class
     * @throws OwException
     */
    protected Map<String, OwCMISPropertyClass> getAspectProperties(String aspectClassId_p) throws OwException
    {
        OwCMISResource resource = getResource();
        OwCMISObjectModel objectModel = resource.getObjectModel();

        OwCMISObjectClass aspDef = objectModel.getObjectClass(aspectClassId_p);
        LinkedHashMap<String, OwCMISPropertyClass> props = new LinkedHashMap<String, OwCMISPropertyClass>();
        for (Map.Entry<String, OwCMISPropertyClass> entry : aspDef.getPropertyClasses().entrySet())
        {
            try
            {
                if (!entry.getValue().isSystemProperty())
                {
                    props.put(entry.getKey(), entry.getValue());
                }
            }
            catch (Exception e)
            {
                if (LOG.isDebugEnabled())
                {
                    LOG.debug("PropertyClass " + entry.getKey() + " skiped caused by error", e);
                }
            }
        }

        return props;
    }

    /**
     * Get the preferred property order.
     * @return List of Strings (property names)
     * @since 4.0.0.0
     */
    public List<String> getPreferredPropertyOrder()
    {
        if (preferredPropertyOrder == null)
        {
            preferredPropertyOrder = new LinkedList<String>();
            try
            {
                preferredPropertyOrder = network.getNetworkConfiguration().getPreferedPropertyOrder();
            }
            catch (OwException e)
            {
                LOG.warn("OwCMISNetworkConfig.getPerferedPropertyOrder: Configuration \"PreferedPropertyOrder\" could not be read.");
            }
        }

        return preferredPropertyOrder;
    }

    /**
     * Sort provided properties regarding definition of {@link #getPreferredPropertyOrder()} list.
     * @param allProperties_p Collection to be sorted
     * @return Collection with preferred property order
     * @throws OwException
     * @since 4.0.0.0
     */
    protected Collection<String> reorderProperties(Collection<String> allProperties_p) throws OwException
    {

        Collection<String> orderedProperties = new LinkedList<String>();
        for (String temp : getPreferredPropertyOrder())
        {
            if (allProperties_p.contains(temp))
            {
                orderedProperties.add(temp);
            }
        }

        orderedProperties.addAll(allProperties_p);
        return orderedProperties;
    }

}
