package com.wewebu.ow.server.ecmimpl.cmis.alfresco;

import java.util.List;

import org.alfresco.Aspects;
import org.apache.log4j.Logger;
import org.oasis_open.docs.ns.cmis.core._200908.CmisObjectType;
import org.oasis_open.docs.ns.cmis.core._200908.CmisPropertiesType;
import org.oasis_open.docs.ns.cmis.ws._200908.CmisException;
import org.oasis_open.docs.ns.cmis.ws._200908.ObjectServicePort;

import com.wewebu.ow.server.ecm.OwNetworkContext;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISExceptionCatcher;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISNetwork;
import com.wewebu.ow.server.ecmimpl.cmis.log.OwLog;
import com.wewebu.ow.server.ecmimpl.cmis.property.OwCMISPropertyNames;
import com.wewebu.ow.server.exceptions.OwException;

/**
 *<p>
 * Alfresco CMIS data manipulation utility class. 
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 *@since 3.1.0.0
 */
public class OwCMISALUtil
{
    private static final Logger LOG = OwLog.getLogger(OwCMISALUtil.class);

    /**
     * Find first occurrence of given type (class).
     * @param objectList_p List to search through
     * @param class_p Class which type is requested
     * @return first occurrence of type <T> in or null if not found
     */
    public static <T> T findFirst(List<Object> objectList_p, Class<T> class_p)
    {
        for (Object object : objectList_p)
        {
            if (object != null)
            {
                if (class_p.isAssignableFrom(object.getClass()))
                {
                    return (T) object;
                }
            }
        }

        return null;
    }

    /**
     * Caches the {@link Aspects} (applied aspects and aspects properties) for 
     * the given {@link CmisObjectType}.
     * The {@link Aspects} object is cached as an element of the any properties extension
     * of the given {@link CmisObjectType}.
     * If the {@link Aspects} of the given object have already been cached the applied aspects list 
     * will be replaced with the latest version obtained from the web service. 
     * @param object_p CmisObjectType whose aspects will be cached
     * @param network_p OwCMISNetwork the current network 
     * @param resourceID_p String id of the object's residing repository/resource 
     * @throws OwException
     */
    public synchronized static void cacheAspects(CmisObjectType object_p, OwCMISNetwork network_p, String resourceID_p) throws OwException
    {
        try
        {

            String objectId = OwCMISPropertyNames.OBJECT_ID.getIdValue(object_p);
            /* Workaround for implementations where object-id is incorrect, 
             * the object service cannot work with the provided objectId use version-series Id
             * String vId = null;// OwCMISPropertyNames.VERSION_SERIES_ID.getIdValue(object_p);
             * CmisPropertiesType properties = osp.getProperties(resourceID_p, vId != null ? vId : objectId, "none", null);*/
            ObjectServicePort osp = network_p.getObjectServicePort();

            CmisPropertiesType properties = osp.getProperties(resourceID_p, objectId, "none", null);
            Aspects aspects = OwCMISALUtil.findFirst(properties.getAny(), Aspects.class);

            if (aspects != null)
            {
                CmisPropertiesType aspectsProperties = aspects.getProperties();
                List<Object> aspectsPropertiesAny = aspectsProperties.getAny();
                aspectsPropertiesAny.add(new OwCMISALCacheInfo(true));
            }

            CmisPropertiesType objectProperties = object_p.getProperties();
            List<Object> objectAny = objectProperties.getAny();
            if (aspects != null)
            {
                Aspects cachedAspects = OwCMISALUtil.findFirst(objectAny, Aspects.class);
                if (cachedAspects == null)
                {
                    objectAny.add(aspects);
                }
                else
                {
                    List<String> cachedAList = cachedAspects.getAppliedAspects();
                    cachedAList.clear();

                    List<String> aList = aspects.getAppliedAspects();

                    cachedAList.addAll(aList);
                }
            }

            OwCMISALCacheInfo cacheInfo = OwCMISALUtil.findFirst(objectAny, OwCMISALCacheInfo.class);
            if (cacheInfo == null)
            {
                cacheInfo = new OwCMISALCacheInfo(true);
                objectAny.add(cacheInfo);
            }
            else
            {
                cacheInfo.setAppliedAspectsCached(true);
            }

        }
        catch (CmisException e)
        {
            OwNetworkContext context = network_p.getContext();
            OwCMISExceptionCatcher catcher = new OwCMISExceptionCatcher(e, context.getLocale());
            LOG.error("OwCMISALUtil.cacheAspects(): Error while caching aspects! A CMIS error occurred : " + catcher.getLogMessage(), e);
            throw catcher.toOwException(context.localize("ecmimpl.cmis.OwCMISObjectBase.setProperties.error", "Error while updating properties!"), true);
        }

    }

    /**
     * Caches and returns the {@link Aspects} object associated with the given CmisObjectType.
     * If the {@link Aspects} of the given object have already been cached they will not be retrieved from
     * the web service again.
     * @see #cacheAspects(CmisObjectType, OwCMISNetwork, String)
     * @param object_p CmisObjectType objects whose aspects will be retrieved
     * @param network_p OwCMISNetwork the current network 
     * @param resourceID_p String id of the object's residing repository/resource 
     * @return the {@link Aspects} object associated with the given CmisObjectType obtained by performing a one time
     *         web service cache  
     * @throws OwException
     */
    public synchronized static Aspects resolveAspects(CmisObjectType object_p, OwCMISNetwork network_p, String resourceID_p) throws OwException
    {
        CmisPropertiesType properties = object_p.getProperties();

        List<Object> any = properties.getAny();

        Aspects aspects = null;
        OwCMISALCacheInfo cacheInfo = null;

        aspects = OwCMISALUtil.findFirst(any, Aspects.class);

        if (aspects == null)
        {
            cacheAspects(object_p, network_p, resourceID_p);
            aspects = OwCMISALUtil.findFirst(any, Aspects.class);
        }

        cacheInfo = OwCMISALUtil.findFirst(any, OwCMISALCacheInfo.class);

        if (cacheInfo == null || !cacheInfo.areAppliedAspectsCached())
        {
            cacheAspects(object_p, network_p, resourceID_p);
            aspects = OwCMISALUtil.findFirst(any, Aspects.class);
        }

        return aspects;
    }
}