package com.wewebu.ow.server.ecmimpl.cmis.auth;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;

import javax.xml.ws.WebServiceException;
import javax.xml.ws.soap.SOAPFaultException;

import org.apache.log4j.Logger;
import org.oasis_open.docs.ns.cmis.ws._200908.CmisException;
import org.oasis_open.docs.ns.cmis.ws._200908.RepositoryServicePort;
import org.w3c.dom.Node;

import com.wewebu.ow.server.app.OwConfiguration;
import com.wewebu.ow.server.auth.OwAuthentication;
import com.wewebu.ow.server.auth.OwAuthenticator;
import com.wewebu.ow.server.auth.OwCredentialsAuthentication;
import com.wewebu.ow.server.conf.OwBaseInitializer;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISCredentials;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISLDAPCredentials;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISSecurityHandler;
import com.wewebu.ow.server.ecmimpl.cmis.wshandler.OwCMISSecurityHandlerInterface;
import com.wewebu.ow.server.exceptions.OwAuthenticationException;
import com.wewebu.ow.server.exceptions.OwConfigurationException;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.exceptions.OwServerException;
import com.wewebu.ow.server.log.OwLogCore;
import com.wewebu.ow.server.mandator.OwMandator;
import com.wewebu.ow.server.util.OwString;
import com.wewebu.ow.server.util.OwString1;
import com.wewebu.ow.server.util.OwXMLDOMUtil;
import com.wewebu.ow.server.util.OwXMLUtil;
import com.wewebu.ow.server.util.ldap.OwLdapConnector;

/**
 *<p>
 * CMIS credentials based authenticator. 
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 *@since 4.0.0.0
 */
public class OwCMISCredentialsAuthenticator implements OwAuthenticator
{
    private static final String ECM_ADAPTER = "EcmAdapter";

    private static final String AUTHENTICATION = "Authentication";

    /** package logger for the class */
    private static final Logger LOG = OwLogCore.getLogger(OwCMISCredentialsAuthenticator.class);

    /**optional configuration node name containing the full qualified java class name*/
    public static final String CONF_NODE_SECURITY_HANDLER = "SecurityHandler";

    private OwBaseInitializer baseInitializer;
    private OwXMLUtil configNode;

    public OwCMISCredentialsAuthenticator(OwBaseInitializer baseInitializer_p)
    {
        super();
        this.baseInitializer = baseInitializer_p;
    }

    public OwCMISCredentialsAuthenticator(OwXMLUtil configNode)
    {
        super();
        this.configNode = configNode;
    }

    public OwAuthentication authenticate(OwAuthentication authentication_p) throws OwAuthenticationException, OwConfigurationException, OwServerException
    {

        String userName = authentication_p.getUserName();
        String password = authentication_p.getPassword();

        if (userName != null && password != null)
        {
            OwCMISCredentials credentials = null;

            try
            {
                credentials = createCredentials(userName, password, getNetworkConfiguration(), TimeZone.getDefault(), Locale.getDefault(), null);

                RepositoryServicePort repositoryServicePort = credentials.getRepositoryServicePort();//try port
                repositoryServicePort.getRepositories(null);//try to retrieve

            }
            catch (SOAPFaultException ex)
            {
                LOG.debug("Login failed: Wrong username or password", ex);
                throw new OwAuthenticationException(new OwString("ecmimpl.cmis.OwCMISNetwork.login.failed", "Login failed: Wrong username or password"), ex);
            }
            catch (WebServiceException serviceEx)
            {
                LOG.fatal("Can not create communication to the Webservices", serviceEx);
                throw new OwServerException(new OwString("ecmimpl.cmis.OwCMISNetwork.login.communication.error", "Cannot create communication to the Webservices"), serviceEx);
            }
            catch (OwConfigurationException owcfg)
            {
                throw owcfg;
            }
            catch (OwAuthenticationException owa)
            {
                LOG.debug("Authentication Exception", owa);
                throw new OwAuthenticationException(new OwString("ecmimpl.cmis.OwCMISNetwork.login.failed", "Login failed: Wrong username or password"), owa);
            }
            catch (OwServerException ows)
            {
                throw ows;
            }
            catch (CmisException e)
            {
                throw new OwServerException(e.getMessage(), e);
            }
            catch (OwException owe)
            {
                throw new OwServerException(owe.getMessage(), owe);
            }

            OwCredentialsAuthentication credentialsAuthentication = new OwCredentialsAuthentication(credentials, userName, password);

            return credentialsAuthentication;
        }
        else
        {
            LOG.debug("OwCMISCredentialsAuthenticator.authenticate(): Login failed: Wrong username or password");
            throw new OwAuthenticationException(new OwString("ecmimpl.cmis.OwCMISNetwork.login.failed", "Login failed: Wrong username or password"));
        }
    }

    /**
     * Get the current configuration node.
     * @return OwXMLUtil
     * @throws OwConfigurationException
     */
    protected OwXMLUtil getNetworkConfiguration() throws OwConfigurationException
    {
        if (this.configNode != null)
        {
            return this.configNode;
        }
        else
        {
            OwXMLUtil bootstrapx = OwConfiguration.getBootstrap(baseInitializer);

            if (bootstrapx == null)
            {
                String msg = "No bootstrap.";
                LOG.error("OwCMISCredentialsAuthenticator.getNetworkConfiguration: " + msg);
                throw new OwConfigurationException(msg);
            }

            try
            {
                return bootstrapx.getSubUtil(ECM_ADAPTER);
            }
            catch (Exception e)
            {
                String msg = "Invalid bootstrap.";
                LOG.error("OwCMISRepositoryAuthenticator.getNetworkConfiguration: " + msg);
                throw new OwConfigurationException(msg, e);
            }
        }
    }

    /**(overridable)
     * Creates credentials for given name and password with respect to 
     * the &lt;Authentication&gt; bootstrap configuration.  
     * @param strUser_p
     * @param strPassword_p
     * @return an {@link OwCMISCredentials} 
     * @throws OwServerException
     */
    protected OwCMISCredentials createCredentials(String strUser_p, String strPassword_p, OwXMLUtil configuration_p, TimeZone wsiTimeZone_p, Locale locale_p, OwMandator mandator_p) throws OwException
    {
        try
        {
            OwCMISSecurityHandlerInterface handler = createSecurityHandler(getLoginMap(strUser_p, strPassword_p), locale_p, configuration_p);
            handler.setWsi18nTimeZone(wsiTimeZone_p);
            Node authenticationNode = configuration_p.getSubNode(AUTHENTICATION);

            String authenticationMode = OwXMLDOMUtil.getSafeStringAttributeValue(authenticationNode, "mode", "");

            if (OwCMISCredentials.AUTH_LDAP.equals(authenticationMode))
            {
                return createLDAPCredentials(configuration_p, authenticationNode, strUser_p, strPassword_p, handler, mandator_p);
            }
            else if (OwCMISCredentials.AUTH_NONE.equals(authenticationMode))
            {
                return new OwCMISCredentials(configuration_p, strUser_p, strPassword_p, handler, mandator_p);
            }
            else
            {
                String msg = "Invalid CMIS authentication configuration! Invalid <Athentication> @mode : " + authenticationMode;
                LOG.error("OwCMISCredentialsAuthenticator.createCredentials(): " + msg);
                throw new OwServerException(new OwString("ecmimpl.cmis.OwCMISNetwork.invalid.authentication.configuration.error", "Invalid CMIS authentication configuration!"));

            }
        }
        catch (OwException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            LOG.error("Invalid Credentials - maybe there are problems on reading the network config node/subnode \"(Authentication)\"...", e);
            throw new OwServerException(new OwString("ecmimpl.cmis.OwCMISNetwork.invalid.authentication.configuration.error", "Invalid CMIS authentication configuration!"), e);
        }
    }

    /**(overridable)
     * Creates a security handler which will be used for the web service communication.
     * <p>Will check if the the configuration contains the node {@link #CONF_NODE_SECURITY_HANDLER}
     * and try to instantiate the defined class. If the node is missing, it will create by default
     * an instance of {@link OwCMISSecurityHandler}.</p>
     * @param initValues_p Map of String key and values to be used for init-method
     * @return OwCMISSecurityHandlerInterface instance
     * @throws OwException
     */
    @SuppressWarnings("unchecked")
    protected OwCMISSecurityHandlerInterface createSecurityHandler(Map<String, String> initValues_p, Locale locale_p, OwXMLUtil configuration_p) throws OwException
    {
        OwCMISSecurityHandlerInterface retHandler = null;
        String strClass = configuration_p.getSafeTextValue(CONF_NODE_SECURITY_HANDLER, null);
        if (strClass != null && !"".equals(strClass))
        {
            try
            {
                Class<? extends OwCMISSecurityHandlerInterface> clazz = (Class<? extends OwCMISSecurityHandlerInterface>) Class.forName(strClass);
                retHandler = clazz.newInstance();
            }
            catch (SecurityException e)
            { //permission denied to access the defined class/constructor
                LOG.error("SecurityManager restrict the instantiation of type " + strClass + "!", e);
                throw new OwServerException(new OwString1("ecmimpl.cmis.OwCMISNetwork.createSecHandler.securityEx", "Instantiation of type %1 is restricted by security.", strClass), e);
            }
            catch (InstantiationException e)
            { //cannot instantiate defined class
                LOG.error("Cannot Instantiate the defined class " + strClass, e);
                throw new OwConfigurationException(new OwString1("ecmimpl.cmis.OwCMISNetwork.createSecHandler.instanceEx", "The defined class %1 is not allowed to be instantiated.", strClass), e);
            }
            catch (IllegalAccessException e)
            { //default constructor is not accessible
                LOG.fatal("Cannot Instantiate defined class " + strClass + " because it is not accessible.", e);
                throw new OwServerException(new OwString1("ecmimpl.cmis.OwCMISNetwork.createSecHandler.illegalAccessEx", "The default constructor of class %1 is not accessible or does not exist.", strClass), e);
            }
            catch (ClassNotFoundException e)
            { //defined class could not be found
                LOG.error("The class " + strClass + " could not be found by current ClassLoader.", e);
                throw new OwConfigurationException(new OwString1("ecmimpl.cmis.OwCMISNetwork.createSecHandler.classNotFoundEx", "The defined class %1 could not be found.", strClass), e);
            }
        }
        else
        {
            retHandler = new OwCMISSecurityHandler(initValues_p.get(OwCMISSecurityHandlerInterface.USER), initValues_p.get(OwCMISSecurityHandlerInterface.PWD));
        }
        retHandler.init(initValues_p);
        retHandler.setLocale(locale_p);
        return retHandler;
    }

    /**(overridable)
     * Creates a LDAP credentials object for a given bootstrap configuration node 
     * (usually the &lt;Authentication&gt; node) , user name and password and current mandator
     * @param username_p
     * @param password_p
     * @param mandator_p current {@link OwMandator} , <code>null</code> for no mandator  
     * @return an {@link OwCMISLDAPCredentials} for the given bootstrap configuration node (usually the &lt;Authentication&gt; node) , 
     *         user name and password and current mandator 
     * @throws OwException
     */
    protected OwCMISLDAPCredentials createLDAPCredentials(OwXMLUtil configuration_p, Node authenticationNode_p, String username_p, String password_p, OwCMISSecurityHandlerInterface handler_p, OwMandator mandator_p) throws OwException
    {
        OwLdapConnector ldapconnector = createLDAPConnector(authenticationNode_p);
        ldapconnector.authenticate(username_p, password_p);

        return new OwCMISLDAPCredentials(configuration_p, ldapconnector, username_p, password_p, handler_p, mandator_p);
    }

    /**(overridable)
     * Creates an LDAP connection for a given bootstrap configuration node 
     * (usually the &lt;Authentication&gt; node).<br>
     * The returned connector is not authenticated.
     * @param authenticationNode_p
     * @return an {@link OwLdapConnector} for the given configuration node
     */
    protected OwLdapConnector createLDAPConnector(Node authenticationNode_p)
    {
        return new OwLdapConnector(authenticationNode_p);
    }

    /**
     * Method called to create/retrieve the hash map which should 
     * be used for initialization of the {@link OwCMISSecurityHandlerInterface}.
     * <p>By default this method try to 
     * @param user_p String representing the current user name
     * @param pwd_p String representing the current user password
     * @return HashMap of String key to String value
     */
    protected Map<String, String> getLoginMap(String user_p, String pwd_p)
    {
        HashMap<String, String> initLoginMap_p = new HashMap<String, String>();
        initLoginMap_p.put(OwCMISSecurityHandlerInterface.PWD, pwd_p);
        initLoginMap_p.put(OwCMISSecurityHandlerInterface.USER, user_p);
        return initLoginMap_p;
    }

}
