package com.wewebu.ow.server.ecmimpl.cmis.object;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.xml.ws.Holder;

import org.apache.log4j.Logger;
import org.oasis_open.docs.ns.cmis.core._200908.CmisAllowableActionsType;
import org.oasis_open.docs.ns.cmis.core._200908.CmisObjectType;
import org.oasis_open.docs.ns.cmis.core._200908.CmisRepositoryCapabilitiesType;
import org.oasis_open.docs.ns.cmis.core._200908.EnumIncludeRelationships;
import org.oasis_open.docs.ns.cmis.messaging._200908.CmisExtensionType;
import org.oasis_open.docs.ns.cmis.messaging._200908.CmisObjectParentsType;
import org.oasis_open.docs.ns.cmis.ws._200908.CmisException;
import org.oasis_open.docs.ns.cmis.ws._200908.NavigationServicePort;

import com.wewebu.ow.server.ecm.OwContentCollection;
import com.wewebu.ow.server.ecm.OwNetworkContext;
import com.wewebu.ow.server.ecm.OwObject;
import com.wewebu.ow.server.ecm.OwObjectCollection;
import com.wewebu.ow.server.ecm.OwStandardObjectCollection;
import com.wewebu.ow.server.ecm.OwStatusContextDefinitions;
import com.wewebu.ow.server.ecm.OwVersion;
import com.wewebu.ow.server.ecm.OwVersionSeries;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISExceptionCatcher;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISNetwork;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISObjectModel;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISResource;
import com.wewebu.ow.server.ecmimpl.cmis.broker.OwCMISListenerHandler;
import com.wewebu.ow.server.ecmimpl.cmis.broker.OwCMISObjectBroker;
import com.wewebu.ow.server.ecmimpl.cmis.broker.OwCMISVersionEventListener;
import com.wewebu.ow.server.ecmimpl.cmis.log.OwLog;
import com.wewebu.ow.server.ecmimpl.cmis.objectclasses.OwCMISObject;
import com.wewebu.ow.server.ecmimpl.cmis.property.OwCMISProperty;
import com.wewebu.ow.server.ecmimpl.cmis.property.OwCMISPropertyNames;
import com.wewebu.ow.server.ecmimpl.cmis.version.OwCMISVersionModel;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.exceptions.OwInvalidOperationException;
import com.wewebu.ow.server.exceptions.OwObjectNotFoundException;
import com.wewebu.ow.server.exceptions.OwServerException;

/**
 *<p>
 * Class representing Document in CMIS environments,
 * which implements also the OwVersion interface.
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 */
public class OwCMISDocumentObject extends OwCMISObjectBase implements OwCMISVersionEventListener
{
    /** package logger for the class */
    private static final Logger LOG = OwLog.getLogger(OwCMISDocumentObject.class);

    private OwCMISListenerHandler<OwCMISVersionEventListener> versionListenerHandler;
    private OwVersion version;

    public OwCMISDocumentObject(OwCMISNetwork network_p, OwCMISObjectModel objectModel_p, CmisObjectType type_p) throws OwException
    {
        super(network_p, objectModel_p, type_p);
        if (hasVersionSeries())
        {
            OwCMISObjectBroker objectBroker = network_p.getObjectBroker();
            this.versionListenerHandler = new OwCMISListenerHandler<OwCMISVersionEventListener>(this);
            objectBroker.register(getResourceID(), getVersionSeriesId(), this.versionListenerHandler);
        }
    }

    public boolean canGetContent(int contentType_p, int context_p) throws OwException
    {
        return m_cmisObjType.getAllowableActions().isCanGetContentStream();
    }

    @Override
    public boolean canSetProperties(int contextStatus_p) throws OwException
    {
        boolean basicCanSet = super.canSetProperties(contextStatus_p);
        if (basicCanSet)
        {
            if (!isImmutable())
            {
                OwVersion myVersion = getVersion();
                OwVersionSeries versionSeries = getVersionSeries();
                OwCMISResource resource = getResource();
                CmisRepositoryCapabilitiesType resourceCapabilities = resource.getCMISCapabilities();
                try
                {
                    OwVersion pwc = versionSeries.getReservation();
                    if (pwc != null && pwc.equals(myVersion))
                    {
                        return resourceCapabilities.isCapabilityPWCUpdatable();
                    }
                    else
                    {
                        return true;
                    }
                }
                catch (OwException e)
                {
                    throw e;
                }
                catch (Exception e)
                {
                    LOG.fatal("OwCMISDocumentObject.canSetProperties(): Error while checking the set properties capability!", e);
                    throw new OwInvalidOperationException(getNetwork().getContext().localize("ecmimpl.cmis.OwCMISDocumentObject.canSetProperties.error", "Error while checking the set properties capability!"), e);
                }

            }
            else
            {
                return false;
            }
        }
        else
        {
            return false;
        }
    }

    public boolean isImmutable() throws OwException
    {
        OwCMISProperty immutableProperty = getProperty(OwCMISPropertyNames.IS_IMMUTABLE.getId());
        Object immutableValue = immutableProperty.getValue();
        if (immutableValue instanceof Boolean)
        {
            Boolean immutableBoolValue = (Boolean) immutableValue;
            return immutableBoolValue;
        }
        else
        {
            String errClass = immutableValue == null ? "<null>" : immutableValue.getClass().toString();
            String msg = "Invalid immutable value " + immutableValue + " of java-class " + errClass;
            LOG.fatal("OwCMISDocumentObject.isImmutable():" + msg);
            throw new OwInvalidOperationException(msg);

        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public OwObjectCollection getParents() throws OwException
    {
        try
        {
            OwStandardObjectCollection parentsCollection = new OwStandardObjectCollection();
            //WARNING  see getObjectParents CMIS specification   :
            //         object ID  should be used instead but it seems that Alfresco is 
            //         misinterpreting its own objectId - version series ID seems to work
            NavigationServicePort ns = getNetwork().getNavigationServicePort();
            List<CmisObjectParentsType> lst = ns.getObjectParents(getResourceID(), getVersionSeriesId(), null, Boolean.TRUE, EnumIncludeRelationships.NONE, null, Boolean.TRUE, null);
            for (CmisObjectParentsType parent : lst)
            {
                OwCMISObject parentObject = getNetwork().createCMISObject(parent.getObject(), getResource());
                parentsCollection.add(parentObject);
            }
            return parentsCollection.isEmpty() ? null : parentsCollection;
        }
        catch (CmisException e)
        {
            OwCMISNetwork network = getNetwork();
            OwNetworkContext context = network.getContext();
            OwCMISExceptionCatcher catcher = new OwCMISExceptionCatcher(e, context.getLocale());
            LOG.error("OwCMISDocumentObject.getParents(): Could not retrieve parents of document!A CMIS error occurred : " + catcher.getLogMessage(), e);
            throw catcher.toOwException(context.localize("ecmimpl.cmis.OwCMISDocumentObject.getParents.error", "Can not retrieve parents of document!"), true);
        }
    }

    public String getMIMEType() throws OwException
    {
        try
        {
            //            return getContentCollection().getContentElement(OwContentCollection.CONTENT_TYPE_DOCUMENT, 0).getMIMEType();
            OwCMISProperty contentMimeType = getProperty(OwCMISPropertyNames.CONTENT_STREAM_MIME_TYPE.getId());
            Object mimeTypeValue = contentMimeType.getValue();
            return mimeTypeValue == null ? "" : mimeTypeValue.toString();
        }
        catch (OwException ex)
        {
            //re-throw, do not wrap the exception, if thrown by the Workdesk framework
            throw ex;
        }
        catch (Exception e)
        {
            LOG.fatal("OwCMISDocumentObject.getMIMEType():Can not retrieve ContentStreamMimeType property of Document!", e);
            throw new OwObjectNotFoundException(getNetwork().getContext().localize("ecmimpl.cmis.OwCMISDocumentObject.getMIMEType.error", "Can not retrieve the content stream MIME-type of document!"), e);
        }
    }

    @Override
    public String getMIMEParameter() throws OwException
    {
        try
        {
            StringBuffer buf = new StringBuffer("name=\"");
            buf.append(getProperty(OwCMISPropertyNames.CONTENT_STREAM_FILENAME.getId()).getValue().toString());
            buf.append('"');

            return buf.toString();
        }
        catch (Exception ex)
        {
            LOG.debug("OwCMISDocumentObject.getMIMEParameter(): Problem retrieving contentstream filename!", ex);
            return "";
        }
    }

    @Override
    public boolean hasContent(int context_p) throws OwException
    {
        try
        {
            OwCMISProperty contentMimeType = getProperty(OwCMISPropertyNames.CONTENT_STREAM_ID.getId());
            return contentMimeType.getValue() != null;
        }
        catch (OwException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            LOG.fatal("OwCMISDocumentObject.hasContent():Can not retrieve ContentStreamLentgh property of Document!", e);
            throw new OwObjectNotFoundException(getNetwork().getContext().localize("ecmimpl.cmis.OwCMISDocumentObject.hasContent.error", "Can not retrieve the document content information!"), e);
        }
    }

    @Override
    public void delete() throws OwException
    {
        Holder<CmisExtensionType> holder = new Holder<CmisExtensionType>();
        try
        {
            getNetwork().getObjectServicePort().deleteObject(getResourceID(), getDecodedID(), Boolean.TRUE, holder);
        }
        catch (CmisException e)
        {
            OwCMISNetwork network = getNetwork();
            OwNetworkContext context = network.getContext();
            OwCMISExceptionCatcher catcher = new OwCMISExceptionCatcher(e, context.getLocale());
            LOG.error("OwCMISDocumentObject.delete(): Could not delete object!A CMIS error occurred : " + catcher.getLogMessage(), e);
            throw catcher.toOwException(context.localize("ecmimpl.cmis.OwCMISDocumentObject.delete.error", "Can not delete document!"), true);
        }
    }

    public OwVersion getVersion() throws OwException
    {
        if (this.version == null)
        {
            OwCMISVersionModel versionModel = getObjectModel().getVersionModel();
            this.version = versionModel.createVersion(getNetwork(), this);
        }
        return version;
    }

    public String getVersionSeriesId() throws OwException
    {
        OwCMISProperty versionSeriesId = getProperty(OwCMISPropertyNames.VERSION_SERIES_ID.getId());
        Object versionSeriesIdValue = versionSeriesId.getValue();
        return versionSeriesIdValue.toString();
    }

    public CmisAllowableActionsType getAllowableActions()
    {
        return m_cmisObjType.getAllowableActions();
    }

    public OwVersionSeries getVersionSeries() throws OwException
    {
        OwCMISVersionModel versionModel = getObjectModel().getVersionModel();
        return versionModel.createVersionSeries(this, getNetwork());
    }

    public boolean hasVersionSeries() throws OwException
    {
        try
        {
            return getObjectClass().hasVersionSeries();
        }
        catch (OwException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            LOG.error("Error checking version series!", e);

            OwCMISNetwork network = getNetwork();
            OwNetworkContext context = network.getContext();
            throw new OwInvalidOperationException(context.localize("ecmimpl.cmis.OwCMISDocumentObject.version.series.error", "Version series error!"), e);
        }
    }

    public boolean canSetContent(int contentType_p, int context_p) throws Exception
    {
        return getVersion().canSave(OwStatusContextDefinitions.STATUS_CONTEXT_CORRECT_STATUS);
    }

    public void setContentCollection(OwContentCollection content_p) throws Exception
    {
        getVersion().save(content_p, null, null);
    }

    public void versionCheckedOut(String resourceId_p, String versionSeriesId_p, CmisObjectType pwcObject_p, String oldVersionId_p) throws OwException
    {
        if (getDecodedID().equals(oldVersionId_p))
        {
            resetNativeObject(pwcObject_p);
        }
    }

    public void versionCanceledChenckedOut(String resourceId_p, String versionSeriesId_p, CmisObjectType currentObject_p, String pwcVersionId_p) throws OwException
    {
        if (getDecodedID().equals(pwcVersionId_p))
        {
            resetNativeObject(currentObject_p);
        }
    }

    public void versionCheckedIn(String resourceId_p, String versionSeriesId_p, CmisObjectType currentObject_p, String pwcVersionId_p) throws OwException
    {
        if (getDecodedID().equals(pwcVersionId_p))
        {
            resetNativeObject(currentObject_p);
        }
    }

    public void versionSaved(String resourceId_p, String versionSeriesId_p, CmisObjectType currentObject_p, String oldVersionId_p) throws OwException
    {
        if (getDecodedID().equals(oldVersionId_p))
        {
            resetNativeObject(currentObject_p);
        }
    }

    public boolean hasChilds(int[] objectTypes_p, int context_p) throws Exception
    {
        return false;
    }

    @Override
    protected void resetNativeObject(CmisObjectType object_p) throws OwException
    {
        super.resetNativeObject(object_p);
        this.version = null;
    }

    @Override
    public String getPath() throws OwException
    {
        StringBuilder path = null;
        OwObjectCollection col = getParents();
        if (col != null)
        {
            OwCMISObject parent = (OwCMISObject) col.get(0);
            try
            {
                path = new StringBuilder(parent.getPath());
            }
            catch (OwException e)
            {
                throw e;
            }
            catch (Exception e)
            {
                throw new OwServerException(getNetwork().getContext().localize("OwCMISDocumentObject.getPath.error.parent", "Could not retrieve path information."), e);
            }
            if (path.charAt(path.length() - 1) != '/')
            {
                path.append("/");
            }
            try
            {
                if (getVersion().isCheckedOut(OwStatusContextDefinitions.STATUS_CONTEXT_TIME_CRITICAL))
                {
                    OwVersionSeries series = getVersionSeries();
                    if (getVersion().isMajor(OwStatusContextDefinitions.STATUS_CONTEXT_TIME_CRITICAL))
                    {
                        OwObject obj = series.getObject(series.getReleased());
                        path.append(obj.getName());
                    }
                    else
                    {
                        Collection<?> versionCol = series.getVersions(null, null, 5);
                        Iterator<?> it = versionCol.iterator();
                        OwObject previous = null;
                        while (it.hasNext())
                        {
                            OwVersion now = (OwVersion) it.next();
                            if (now.isLatest(OwStatusContextDefinitions.STATUS_CONTEXT_TIME_CRITICAL))
                            {
                                if (previous == null)
                                {
                                    if (it.hasNext())
                                    {
                                        now = (OwVersion) it.next();
                                        previous = series.getObject(now);
                                    }
                                    else
                                    {//we don't have any other versions
                                        previous = series.getObject(now);
                                    }
                                    break;
                                }
                            }
                        }

                        if (previous != null)
                        {
                            path.append(previous.getName());
                        }
                        else
                        {
                            LOG.error("OwCMISDocumentObject.getPath: Could not retrieve last available version.");
                            throw new OwServerException(getNetwork().getContext().localize("OwCMISDocumentObject.getPath.error.version", "Could not retrieve last available version."));
                        }
                    }
                }
                else
                {
                    path.append(getName());
                }
            }
            catch (OwException e)
            {
                throw e;
            }
            catch (Exception e)
            {
                throw new OwServerException(getNetwork().getContext().localize("OwCMISDocumentObject.getPath.error.name", "Could not retrieve cmis:name information."), e);
            }
        }
        return path == null ? null : path.toString();
    }
}