package com.wewebu.ow.server.ecmimpl.cmis.object;

import java.math.BigInteger;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.xml.ws.Holder;

import org.apache.log4j.Logger;
import org.oasis_open.docs.ns.cmis.core._200908.CmisObjectType;
import org.oasis_open.docs.ns.cmis.core._200908.EnumBaseObjectTypeIds;
import org.oasis_open.docs.ns.cmis.core._200908.EnumIncludeRelationships;
import org.oasis_open.docs.ns.cmis.core._200908.EnumUnfileObject;
import org.oasis_open.docs.ns.cmis.messaging._200908.CmisExtensionType;
import org.oasis_open.docs.ns.cmis.messaging._200908.CmisObjectInFolderContainerType;
import org.oasis_open.docs.ns.cmis.messaging._200908.CmisObjectInFolderListType;
import org.oasis_open.docs.ns.cmis.messaging._200908.CmisObjectInFolderType;
import org.oasis_open.docs.ns.cmis.messaging._200908.DeleteTreeResponse.FailedToDelete;
import org.oasis_open.docs.ns.cmis.messaging._200908.EnumServiceException;
import org.oasis_open.docs.ns.cmis.ws._200908.CmisException;
import org.oasis_open.docs.ns.cmis.ws._200908.NavigationServicePort;
import org.oasis_open.docs.ns.cmis.ws._200908.ObjectServicePort;
import org.perf4j.StopWatch;
import org.perf4j.log4j.Log4JStopWatch;

import com.wewebu.ow.server.ecm.OwContentCollection;
import com.wewebu.ow.server.ecm.OwNetworkContext;
import com.wewebu.ow.server.ecm.OwObject;
import com.wewebu.ow.server.ecm.OwObjectClass;
import com.wewebu.ow.server.ecm.OwObjectCollection;
import com.wewebu.ow.server.ecm.OwObjectReference;
import com.wewebu.ow.server.ecm.OwPermissionCollection;
import com.wewebu.ow.server.ecm.OwPropertyCollection;
import com.wewebu.ow.server.ecm.OwStandardObjectCollection;
import com.wewebu.ow.server.ecm.OwStatusContextDefinitions;
import com.wewebu.ow.server.ecm.OwStatusContextException;
import com.wewebu.ow.server.ecm.OwVersion;
import com.wewebu.ow.server.ecm.OwVersionSeries;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISExceptionCatcher;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISNetwork;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISObjectModel;
import com.wewebu.ow.server.ecmimpl.cmis.log.OwLog;
import com.wewebu.ow.server.ecmimpl.cmis.objectclasses.OwCMISObject;
import com.wewebu.ow.server.ecmimpl.cmis.objectclasses.OwCMISObjectClass;
import com.wewebu.ow.server.ecmimpl.cmis.property.OwCMISProperty;
import com.wewebu.ow.server.ecmimpl.cmis.property.OwCMISPropertyNames;
import com.wewebu.ow.server.ecmimpl.cmis.propertyclasses.OwCMISPropertyClass;
import com.wewebu.ow.server.ecmimpl.cmis.util.OwCMISPropertiesFilter;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.exceptions.OwInvalidOperationException;
import com.wewebu.ow.server.exceptions.OwNotSupportedException;
import com.wewebu.ow.server.exceptions.OwObjectNotFoundException;
import com.wewebu.ow.server.exceptions.OwServerException;
import com.wewebu.ow.server.field.OwFieldDefinition;
import com.wewebu.ow.server.field.OwSearchNode;
import com.wewebu.ow.server.field.OwSort;
import com.wewebu.ow.server.field.OwSort.OwSortCriteria;

/**
 *<p>
 * Class representing a Folder in CMIS environments.
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 */
public class OwCMISFolderObject extends OwCMISObjectBase
{

    private static final Logger LOG = OwLog.getLogger(OwCMISFolderObject.class);

    private static final int DEFAUL_MAX_SORT_CRITERIA = 1; //limited by Alfresco sort capability

    /**parameter used for client side sorting 
     * @since 3.1.0.0 */
    private OwSort clientSortCollection;

    public OwCMISFolderObject(OwCMISNetwork network_p, OwCMISObjectModel objectModel_p, CmisObjectType cmistype_p) throws OwException
    {
        super(network_p, objectModel_p, cmistype_p);
    }

    @Override
    public int getChildCount(int[] objectTypes_p, int context_p) throws OwException
    {
        //omits child count display through exception handling, no logging needed here 
        throw new OwStatusContextException("");
        //TODO: implement child count with good performance (nice to have)
    }

    @SuppressWarnings({ "unchecked", "rawtypes" })
    public OwObjectCollection getChilds(int[] objectTypes_p, Collection propertyNames_p, OwSort sort_p, int maxSize_p, int versionSelection_p, OwSearchNode filterCriteria_p) throws OwException
    {
        OwStandardObjectCollection ret = createObjectCollection();
        clientSortCollection = null;
        // iterate over the child folders
        NavigationServicePort np = getNetwork().getNavigationServicePort();

        BigInteger depth = BigInteger.ONE;
        BigInteger skip = BigInteger.ZERO;
        BigInteger max = new BigInteger(Integer.toString(maxSize_p));
        StopWatch stopWatchGetChilds = new Log4JStopWatch("getChilds", "OwCMISFolder.getChildren()");
        if (objectTypes_p == null)
        {
            objectTypes_p = new int[0];
        }
        for (int iType = 0; iType < objectTypes_p.length; iType++)
        {
            switch (objectTypes_p[iType])
            {
                case OwObjectReference.OBJECT_TYPE_ALL_CONTAINER_OBJECTS:
                case OwObjectReference.OBJECT_TYPE_FOLDER:
                {
                    try
                    {
                        OwCMISObjectClass clazz = getNetwork().getObjectClass(EnumBaseObjectTypeIds.CMIS_FOLDER.value(), getResource());
                        if (getResource().getCMISCapabilities().isCapabilityGetFolderTree() && (sort_p == null || sort_p.getCriteriaCollection().isEmpty()))
                        {
                            List<CmisObjectInFolderContainerType> lst;
                            StopWatch stopWatch = new Log4JStopWatch("getFolderTree(folder)", "OwCMISFolderObject.getChilds: service getFolderTree(folder)");
                            lst = np.getFolderTree(getResourceID(), getDecodedID(), depth, getChildrenPropertiesAsString(propertyNames_p, clazz), Boolean.TRUE, EnumIncludeRelationships.NONE, OwCMISPropertyNames.NONE.getId(), Boolean.FALSE, null);
                            stopWatch.stop();

                            for (CmisObjectInFolderContainerType conType : lst)
                            {
                                CmisObjectInFolderType objectInFolder = conType.getObjectInFolder();
                                CmisObjectType theObject = objectInFolder.getObject();
                                OwCMISObject child = getNetwork().createCMISObject(theObject, getResource());
                                ret.add(child);
                            }
                        }
                        else
                        {
                            StopWatch stopWatch = new Log4JStopWatch("getChildren(folder)", "OwCMISFolderObject.getChilds: service getChildren(folder)");
                            CmisObjectInFolderListType lst = np.getChildren(getResourceID(), getDecodedID(), getChildrenPropertiesAsString(propertyNames_p, clazz), getSortString(sort_p, DEFAUL_MAX_SORT_CRITERIA), Boolean.TRUE,
                                    EnumIncludeRelationships.NONE, OwCMISPropertyNames.NONE.getId(), Boolean.TRUE, max, skip, null);
                            stopWatch.stop();
                            for (CmisObjectInFolderType objInFolder : lst.getObjects())
                            {
                                CmisObjectType cmisChild = objInFolder.getObject();
                                String objectTypeID = OwCMISPropertyNames.OBJECT_TYPE_ID.getIdValue(cmisChild);
                                OwObjectClass classDesc = getNetwork().getObjectClass(objectTypeID, getResource());
                                if (classDesc.getType() == OwObjectReference.OBJECT_TYPE_FOLDER)
                                {
                                    objInFolder.getObject();
                                    CmisObjectType object = objInFolder.getObject();
                                    OwCMISObject cmisObject = getNetwork().createCMISObject(object, getResource());
                                    ret.add(cmisObject);
                                }
                            }
                        }

                    }
                    catch (CmisException e)
                    {
                        OwCMISNetwork network = getNetwork();
                        OwNetworkContext context = network.getContext();
                        OwCMISExceptionCatcher catcher = new OwCMISExceptionCatcher(e, context.getLocale());
                        LOG.error("OwCMISFolderObject.getChilds(): Could retrieve folder children! A CMIS error occurred : " + catcher.getLogMessage(), e);
                        throw catcher.toOwException(context.localize("ecmimpl.cmis.OwCMISFolderObject.getChilds.error", "Could retrieve folder children!"), true);
                    }

                }
                    break;
                case OwObjectReference.OBJECT_TYPE_DOCUMENT:
                case OwObjectReference.OBJECT_TYPE_ALL_CONTENT_OBJECTS:
                {
                    try
                    {
                        StopWatch stopWatch = new Log4JStopWatch("getChildren(document)", "OwCMISFolderObject.getChilds: service getChildren(document)");
                        OwCMISObjectClass clazz = getNetwork().getObjectClass(EnumBaseObjectTypeIds.CMIS_DOCUMENT.value(), getResource());
                        CmisObjectInFolderListType lstType = np.getChildren(getResourceID(), getDecodedID(), getChildrenPropertiesAsString(propertyNames_p, clazz), getSortString(sort_p, DEFAUL_MAX_SORT_CRITERIA), Boolean.TRUE,
                                EnumIncludeRelationships.NONE, OwCMISPropertyNames.NONE.getId(), Boolean.TRUE, max, skip, null);
                        stopWatch.stop();
                        for (CmisObjectInFolderType objectInFolder : lstType.getObjects())
                        {
                            CmisObjectType cmisChild = objectInFolder.getObject();
                            String objectTypeID = OwCMISPropertyNames.OBJECT_TYPE_ID.getIdValue(cmisChild);
                            OwObjectClass classDesc = getNetwork().getObjectClass(objectTypeID, getResource());
                            if (classDesc.getType() == OwObjectReference.OBJECT_TYPE_DOCUMENT)
                            {
                                CmisObjectType object = objectInFolder.getObject();
                                OwCMISObject cmisObject = getNetwork().createCMISObject(object, getResource());
                                ret.add(cmisObject);
                            }
                        }
                    }
                    catch (CmisException e)
                    {
                        OwCMISNetwork network = getNetwork();
                        OwNetworkContext context = network.getContext();
                        OwCMISExceptionCatcher catcher = new OwCMISExceptionCatcher(e, context.getLocale());
                        LOG.error("OwCMISFolderObject.getChilds(): Could retrieve folder children! A CMIS error occurred : " + catcher.getLogMessage(), e);
                        throw catcher.toOwException(context.localize("ecmimpl.cmis.OwCMISFolderObject.getChilds.error", "Could retrieve folder children!"), true);
                    }

                }
                    break;
            }
        }
        stopWatchGetChilds.stop();

        //check if the collection is need to be sorted on client side.
        if (getClientSortCollection() == null || getClientSortCollection().getCriteriaCollection().isEmpty())
        {
            return ret;
        }
        else
        {
            StopWatch stopWatchSort = new Log4JStopWatch("sort", "OwCMISFolder.getChildren(): client sorting");
            doClientsideSorting(ret);
            stopWatchSort.stop();
            setClientSortCollection(null);
            return ret;
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public OwObjectCollection getParents() throws OwException
    {
        OwStandardObjectCollection col = new OwStandardObjectCollection();
        try
        {
            if (!getNetwork().getResource(getResourceID()).getRepsitoryRootFolderId().equals(getDecodedID()))
            {
                CmisObjectType parent = getNetwork().getNavigationServicePort().getFolderParent(getResourceID(), getDecodedID(), null, null);
                if (parent.getAllowableActions() == null)
                {//we need allowable actions to resolve the behavior of parent folder
                    parent = getNetwork().getObjectServicePort().getObject(getResourceID(), OwCMISPropertyNames.OBJECT_ID.getIdValue(parent), null, Boolean.TRUE, EnumIncludeRelationships.NONE, OwCMISPropertyNames.NONE.getId(), Boolean.FALSE,
                            Boolean.FALSE, null);
                }
                OwCMISObject cmisObject = getNetwork().createCMISObject(parent, getResource());
                col.add(cmisObject);
            }
        }
        catch (CmisException e)
        {
            OwCMISNetwork network = getNetwork();
            OwNetworkContext context = network.getContext();
            OwCMISExceptionCatcher catcher = new OwCMISExceptionCatcher(e, context.getLocale());
            LOG.error("OwCMISFolderObject.getParents(): Could retrieve folder parents! A CMIS error occurred : " + catcher.getLogMessage(), e);
            throw catcher.toOwException(context.localize("ecmimpl.cmis.OwCMISFolderObject.getParents.error", "Could retrieve folder parents!"), true);
        }
        return col.isEmpty() ? null : col;
    }

    public boolean canAdd(OwObject object_p, int context_p) throws OwException
    {
        try
        {
            if (context_p == OwStatusContextDefinitions.STATUS_CONTEXT_CORRECT_STATUS)
            {
                String objID = object_p.getClassName();
                OwCMISProperty prop = this.getProperty(OwCMISPropertyNames.ALLOWED_CHILD_OBJECTTYPE_IDS.getId());
                String[] childIDs = (String[]) prop.getValue();
                if (childIDs != null && childIDs.length > 0)
                {
                    try
                    {
                        for (String id : childIDs)
                        {
                            if (id != null && isSubtypeOf(objID, id))
                            {
                                return Boolean.TRUE.booleanValue();
                            }
                        }
                    }
                    catch (OwObjectNotFoundException e)
                    {
                        //id could not be found, maybe different repository
                    }
                    return Boolean.FALSE.booleanValue();
                }
                else
                {// we don't have any limitations, allow add operation
                    return Boolean.TRUE.booleanValue();
                }
            }
            else
            {//it's time critical, so we allow filling and may be get error on addObject call
                return Boolean.TRUE.booleanValue();
            }
        }
        catch (OwException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            LOG.fatal("OwCMISFolderObject.canAdd(): Error while checking the add object capability!", e);
            throw new OwServerException(getNetwork().getContext().localize("ecmimpl.cmis.OwCMISFolderObject.canAdd.error", "Error while checking the add object capability!"), e);
        }
    }

    public boolean canGetContent(int contentType_p, int context_p) throws OwException
    {
        return false;
    }

    /**
     * Convert the Collection of Strings into a String of comma separated
     * property names.
     * <p>if Collection is null, then all possible properties are retrieved
     * using the SQL operator *, <br /> else the collection is transformed
     * into a String of comma separated property names.</p>
     * @param propertyNames_p Collection, can be null
     * @param defaultObjClass_p OwCMISObjectClass default object class for filtering use
     * @return String of comma separated property names, or &quot;*&quot; if Collection is null
     * @throws OwException 
     */
    protected String getChildrenPropertiesAsString(Collection<String> propertyNames_p, OwCMISObjectClass defaultObjClass_p) throws OwException
    {
        String properties;
        if (propertyNames_p == null || propertyNames_p.isEmpty())
        {
            properties = "*";
        }
        else
        {
            OwCMISPropertiesFilter propertiesFilter = new OwCMISPropertiesFilter();
            propertiesFilter.addAll(OwCMISQueryContext.GET_PROPERTY, propertyNames_p, defaultObjClass_p, getResource());
            propertiesFilter.add(OwCMISPropertyNames.OBJECT_TYPE_ID.getId());
            propertiesFilter.add(OwCMISPropertyNames.OBJECT_ID.getId());
            propertiesFilter.add(OwCMISPropertyNames.IS_IMMUTABLE.getId());
            propertiesFilter.add(OwCMISPropertyNames.CHANGE_TOKEN.getId());
            if (defaultObjClass_p != null && defaultObjClass_p.getType() == OwObjectReference.OBJECT_TYPE_DOCUMENT)
            {
                propertiesFilter.add(OwCMISPropertyNames.IS_VERSION_SERIES_CHECKED_OUT.getId());
                propertiesFilter.add(OwCMISPropertyNames.VERSION_SERIES_ID.getId());
                propertiesFilter.add(OwCMISPropertyNames.VERSION_SERIES_CHECKED_OUT_BY.getId());
                propertiesFilter.add(OwCMISPropertyNames.VERSION_LABEL.getId());
                propertiesFilter.add(OwCMISPropertyNames.CONTENT_STREAM_ID.getId());
                propertiesFilter.add(OwCMISPropertyNames.IS_MAJOR_VERSION.getId());
                propertiesFilter.add(OwCMISPropertyNames.CONTENT_STREAM_MIME_TYPE.getId());
            }

            properties = propertiesFilter.getFilterString();
        }

        return properties;
    }

    /**
     * Return a String of the defined OwSort criteria in correct order.
     * If given OwSort is null or empty <code>null</code> is returned.
     * <p>Method will check every property if it isOrderable, regarding
     * it's property definition. If the property can be used for sorting (isOrderable == true)
     * the property will be add to returned String with the specific cmis:queryname, else
     * the property will silently ignored and the next one will be processed</p>
     * @param sortingOrder_p OwSort sort definition
     * @param maxSortCriteria_p maxim number of sort criteria supported by the native CMIS system 
     * @return String representing the sorting, or null
     * @throws OwException if cannot resolve sort criteria
     */
    protected String getSortString(OwSort sortingOrder_p, int maxSortCriteria_p) throws OwException
    {
        if (sortingOrder_p != null && !sortingOrder_p.getCriteriaCollection().isEmpty())
        {
            StringBuilder sortStatements = new StringBuilder();
            boolean notOrderableCriteria = false;
            if (maxSortCriteria_p > 0 && maxSortCriteria_p < sortingOrder_p.getCriteriaCollection().size())
            {
                notOrderableCriteria = true;
            }
            else
            {
                Iterator<?> itSortCrit = sortingOrder_p.getCriteriaCollection().iterator();
                while (itSortCrit.hasNext())
                {
                    OwSortCriteria crit = (OwSortCriteria) itSortCrit.next();
                    StringBuilder sortOrder = new StringBuilder();
                    boolean orderAble = false;
                    try
                    {
                        OwFieldDefinition def = getNetwork().getFieldDefinition(crit.getPropertyName(), getResourceID());
                        if (def instanceof OwCMISPropertyClass)
                        {
                            OwCMISPropertyClass propClazz = (OwCMISPropertyClass) def;
                            if (propClazz.isOrderable())
                            {
                                sortOrder.append(propClazz.getQueryName(OwCMISQueryContext.SQL));
                                orderAble = true;
                            }
                        }

                        if (!orderAble)
                        {
                            if (LOG.isDebugEnabled())
                            {
                                LOG.debug("OwCMISFolderObject.getSortString(OwSort): The property " + crit.getPropertyName() + " cannot be used for sorting.");
                            }
                            notOrderableCriteria = Boolean.TRUE.booleanValue();
                        }
                    }
                    catch (OwObjectNotFoundException ex)
                    {
                        LOG.warn("Could not retrieve the field definition, we will use the name as defined = " + crit.getPropertyName(), ex);
                        throw new OwInvalidOperationException(getNetwork().getContext().localize1("ecmimpl.cmis.OwCMISFolderObject.invalidSearchCriteria", "The property %1 could not be found!", crit.getPropertyName()), ex);
                    }

                    if (orderAble)
                    {
                        sortOrder.append(" ");
                        sortOrder.append(crit.getAscFlag() ? "ASC" : "DESC");
                        if (itSortCrit.hasNext())
                        {
                            sortOrder.insert(0, ",");
                        }
                    }

                    sortStatements.insert(0, sortOrder);
                }
            }
            if (notOrderableCriteria)
            {/* the whole sort definition must be provided, or the information 
              * about defined sort order is lost during client sorting*/
                setClientSortCollection(sortingOrder_p);
            }
            LOG.info("OwCMISFolderObject.getSortString: Sorting will be: " + sortStatements.toString());
            String sortString = sortStatements.length() > 0 ? sortStatements.toString() : null;
            return sortString;
        }
        else
        {
            return null;
        }
    }

    @Override
    public void delete() throws OwException
    {
        try
        {
            FailedToDelete failedToDelete = getNetwork().getObjectServicePort().deleteTree(getResourceID(), getDecodedID(), Boolean.TRUE, EnumUnfileObject.DELETE, Boolean.TRUE, null);
            List<String> objectIds = failedToDelete.getObjectIds();
            if (objectIds != null && !objectIds.isEmpty())
            {
                StringBuilder errorBuilder = new StringBuilder();
                for (String objectId : objectIds)
                {
                    errorBuilder.append("\n\t[");
                    errorBuilder.append(objectId);
                    errorBuilder.append("]");
                }

                LOG.error("OwCMISFolderObject.delete(): could not delete the folder tree objects with the following IDs : " + errorBuilder.toString());

                OwCMISNetwork network = getNetwork();
                OwNetworkContext context = network.getContext();
                throw new OwServerException(context.localize("ecmimpl.cmis.OwCMISFolderObject.delete.tree.error", "Could not delete all of the folder tree elements!"));
            }
        }
        catch (CmisException e)
        {
            OwCMISNetwork network = getNetwork();
            OwNetworkContext context = network.getContext();
            OwCMISExceptionCatcher catcher = new OwCMISExceptionCatcher(e, context.getLocale());
            LOG.error("OwCMISFolderObject.delete(): Could not delete folder!A CMIS error occurred : " + catcher.getLogMessage(), e);
            throw catcher.toOwException(context.localize("ecmimpl.cmis.OwCMISFolderObject.delete.error", "Can not delete folder!"), true);
        }
    }

    @Override
    public void removeReference(OwObject object_p) throws OwException
    {
        try
        {
            boolean multi = getResource().getCMISCapabilities().isCapabilityMultifiling();
            if (object_p.getType() == OwObjectReference.OBJECT_TYPE_DOCUMENT && multi)
            {
                getNetwork().getMultiFilingServicePort().removeObjectFromFolder(getResourceID(), decodeID(object_p.getID()), getDecodedID(), null);
            }
            else
            {
                getNetwork().getObjectServicePort().deleteTree(getResourceID(), decodeID(object_p.getID()), Boolean.FALSE, EnumUnfileObject.UNFILE, Boolean.TRUE, null);
            }
        }
        catch (CmisException e)
        {
            OwCMISNetwork network = getNetwork();
            OwNetworkContext context = network.getContext();
            OwCMISExceptionCatcher catcher = new OwCMISExceptionCatcher(e, context.getLocale());
            LOG.error("OwCMISFolderObject.removeReference(): Could not remove reference !A CMIS error occurred : " + catcher.getLogMessage(), e);
            throw catcher.toOwException(context.localize("ecmimpl.cmis.OwCMISFolderObject.removeReference.error", "Can not remove reference !"), true);
        }
    }

    @Override
    public boolean canRemoveReference(OwObject object_p, int context_p) throws OwException
    {
        if (object_p.getType() == OwObjectReference.OBJECT_TYPE_DOCUMENT)
        {
            return getResource().getCMISCapabilities().isCapabilityMultifiling();
        }
        else
        {
            return super.canRemoveReference(object_p, context_p);
        }
    }

    @Override
    public void add(OwObject object_p) throws OwException
    {
        OwCMISNetwork network = getNetwork();
        OwNetworkContext context = network.getContext();

        OwObjectClass objectClass = object_p.getObjectClass();
        int objectType = objectClass.getType();

        if (objectType == OwObjectReference.OBJECT_TYPE_FOLDER)
        {
            LOG.error("OwCMISFolderObject.add(): Folder multifilling (INSERT_MODE_REFERENCE) is not supported by CMIS! ");
            throw new OwNotSupportedException(context.localize("ecmimpl.cmis.OwCMISFolderObject.add.folders.not.supported", "Folder multifiling is not supported!"));
        }

        if (getResource().getCMISCapabilities().isCapabilityMultifiling())
        {
            try
            {
                Holder<CmisExtensionType> holder = new Holder<CmisExtensionType>();
                network.getMultiFilingServicePort().addObjectToFolder(getResourceID(), decodeID(object_p.getID()), getDecodedID(), null, holder);
            }
            catch (CmisException e)
            {
                OwCMISExceptionCatcher catcher = new OwCMISExceptionCatcher(e, context.getLocale());
                LOG.error("OwCMISFolderObject.add(): Could not add object!A CMIS error occurred : " + catcher.getLogMessage(), e);

                if (e.getFaultInfo().getType().equals(EnumServiceException.CONSTRAINT))
                {
                    /* constraint: The Repository MUST throw this exception if the cmis:objectTypeId property value
                     * of the given object is NOT in the list of AllowedChildObjectTypeIds of the parent-folder specified
                     * by folderId. 
                     */
                    throw new OwInvalidOperationException(context.localize("ecmimpl.cmis.OwCMISFolderObject.add.cmis.error.constraint", "It is not allowed to add an object of this type!"), e);
                }
                else
                {
                    throw catcher.toOwException(context.localize("ecmimpl.cmis.OwCMISFolderObject.add.cmis.error", "Cannot add the object!"), true);
                }
            }
        }
        else
        {
            LOG.error("OwCMISFolderObject.add(): Repository does not support multifiling");
            throw new OwNotSupportedException(context.localize("ecmimpl.cmis.OwCMISFolderObject.add.multifiling.not.supported", "Multifiling is not supported by this system!"));
        }
    }

    /**
     * Helper method to add a property name to a StringBuilder, which contains/represents 
     * a String of comma (&quot;,&quot;) separated property names.
     * <p>Check if the <b>existingProp</b> is ending with a comma,
     * and appending a comma if missing before appending new property name.
     * </p>
     * <p>Does not check if existingProp already contains the newPropName!</p>
     * @param existingProp_p StringBuilder which is non-empty
     * @param newPropName_p String new property name to be add.
     */
    protected static void addPropertyName(StringBuilder existingProp_p, String newPropName_p)
    {
        if (existingProp_p.length() != 0 && !(existingProp_p.lastIndexOf(",") == existingProp_p.length()))
        {
            existingProp_p.append(',');
        }
        existingProp_p.append(newPropName_p);
    }

    public OwVersion getVersion() throws OwException
    {
        return null;
    }

    public OwVersionSeries getVersionSeries() throws OwException
    {
        return null;
    }

    public boolean hasVersionSeries() throws OwException
    {
        return false;
    }

    public boolean canSetContent(int contentType_p, int context_p) throws OwException
    {
        return false;
    }

    public void setContentCollection(OwContentCollection content_p) throws OwException
    {

    }

    public boolean hasChilds(int[] objectTypes_p, int context_p) throws OwException
    {
        return true;
    }

    @Override
    public OwCMISObject createCopy(OwCMISObject copyParent_p, OwPropertyCollection properties_p, OwPermissionCollection permissions_p, int[] childTypes_p) throws OwException
    {
        OwCMISObject myCopy = super.createCopy(copyParent_p, properties_p, permissions_p, childTypes_p);
        if (childTypes_p == null || childTypes_p.length == 0)
        {
            childTypes_p = new int[] { OwObjectReference.OBJECT_TYPE_DOCUMENT, OwObjectReference.OBJECT_TYPE_FOLDER, OwObjectReference.OBJECT_TYPE_CUSTOM };
        }
        OwObjectCollection children = getChilds(childTypes_p, null, null, 0, 0, null);

        for (Iterator<?> i = children.iterator(); i.hasNext();)
        {
            OwCMISObject cmisChild = (OwCMISObject) i.next();
            cmisChild.createCopy(myCopy, properties_p, permissions_p, childTypes_p);
        }

        return myCopy;
    }

    /**
     * Set the collection which should be used for client side
     * sorting.
     * @param sort_p OwSort
     * @since 3.1.0.0
     */
    protected void setClientSortCollection(OwSort sort_p)
    {
        this.clientSortCollection = sort_p;
    }

    /**
     * Return an OwSort containing properties which cloud not be used for sorting on server side.
     * <p>Can return null if server side can sort every defined properties for 
     * {@link #getChilds(int[], Collection, OwSort, int, int, OwSearchNode)} method.</p>
     * @return OwSort or null 
     * @since 3.1.0.0
     */
    protected OwSort getClientSortCollection()
    {
        return this.clientSortCollection;
    }

    /** (overridable)
     * Factory method for the ObjectCollection which will be used in
     * the {@link #getChilds(int[], Collection, OwSort, int, int, OwSearchNode)} method.
     * @return OwStandardObjectCollection
     * @since 3.1.0.0
     */
    protected OwStandardObjectCollection createObjectCollection()
    {
        return new OwStandardObjectCollection();
    }

    /**
     * (overridable)
     * Called to do a client side sorting, because the 
     * sorting for the given property does not work on server side.
     * <p>Sorting is incomplete because the list of object is 
     * not the whole list which exist on server side.</p>
     * @param col_p OwObjectCollection which should be sorted
     * @since 3.1.0.0
     */
    protected void doClientsideSorting(OwObjectCollection col_p) throws OwException
    {
        try
        {
            col_p.sort(getClientSortCollection());
        }
        catch (Exception e)
        {
            throw new OwInvalidOperationException("Could not sort object collection.", e);
        }
    }

    @Override
    public String getPath() throws OwException
    {
        StringBuilder path = new StringBuilder(OwObject.STANDARD_PATH_DELIMITER);
        path.append(getResourceID());
        path.append(getProperty(OwCMISPropertyNames.PATH.getId()).getValue());
        return path.toString();
    }

    @Override
    public boolean canMove(OwObject object_p, OwObject oldParent_p, int context_p) throws OwException
    {//available since 3.2.0.0
        if (object_p != null)
        {
            return canAdd(object_p, OwStatusContextDefinitions.STATUS_CONTEXT_CORRECT_STATUS);
        }
        else
        {
            return super.canMove(object_p, oldParent_p, context_p);
        }
    }

    @Override
    public void move(OwObject object_p, OwObject oldParent_p) throws OwException
    {//available since 3.2.0.0
        if (object_p != null)
        {
            if (canAdd(object_p, OwStatusContextDefinitions.STATUS_CONTEXT_CORRECT_STATUS))
            {
                ObjectServicePort osp = getNetwork().getObjectServicePort();

                Holder<String> objId = new Holder<String>();
                objId.value = decodeID(getCurrentObjectId(object_p));
                String target = getDecodedID();
                String src = oldParent_p == null ? null : decodeID(oldParent_p.getID());
                try
                {
                    osp.moveObject(getResourceID(), objId, target, src, null);
                }
                catch (CmisException e)
                {
                    LOG.error("Could not move object, srcFolder = " + src + " target = " + target + " moveObj = " + objId.value, e);
                    OwCMISExceptionCatcher transformer = new OwCMISExceptionCatcher(e);
                    OwNetworkContext ctx = getNetwork().getContext();
                    String msg = ctx.localize("ecmimpl.cmis.OwCMISFolderObject.move", "Moving operation failed.");
                    throw transformer.toOwException(msg, false);
                }
            }
            else
            {
                LOG.warn("OwCMISFolderObject.move: Moving defined type is not allowed into target folder.");
                throw new OwInvalidOperationException(getNetwork().getContext().localize("ecmimpl.cmis.OwCMISFolderObject.move.typeRestriction", "Move operation is not allowed for provided object type."));
            }
        }
        else
        {
            LOG.warn("OwCMISFodlerObject.move: Called with null for moving object");
            throw new OwInvalidOperationException(getNetwork().getContext().localize("ecmimpl.cmis.OwCMISFolderObject.move.nullObj", "No object defined for move operation"));
        }
    }

    /**
     * Helper Method which will recursively traverse bottom-up the
     * object-class/-type tree, searching for matching Id's.
     * @param typeId String current id/symbolic name of type
     * @param isParentId String parent type id/symbolic name
     * @return boolean true if both are equals, or typeId is sub type of parent id.
     * @throws OwException could not find object type or parent object type
     * @since 3.2.0.0
     */
    protected boolean isSubtypeOf(String typeId, String isParentId) throws OwException
    {
        if (typeId.equals(isParentId))
        {
            return Boolean.TRUE.booleanValue();
        }
        else
        {
            OwCMISObjectClass clazz = getObjectModel().getObjectClass(typeId);
            OwCMISObjectClass pClass = clazz.getParent();
            if (pClass == null)
            {
                return Boolean.FALSE.booleanValue();
            }
            else
            {
                return isSubtypeOf(pClass.getClassName(), isParentId);
            }
        }
    }

    /**
     * Helper method to get the correct object id, which is used in move operation 
     * (like latest object id, to avoid errors for version dependent filing). 
     * @param object_p OwObject to be moved
     * @return String id which should be used for move operation
     * @throws OwException if could not retrieve corresponding id
     * @since 3.2.0.0
     */
    protected String getCurrentObjectId(OwObject object_p) throws OwException
    {
        /* Need special handling for different repositories,
         * currently not sure which is right!
         * This will provide easy extension for processing move-operation*/
        return object_p.getID();
    }
}