package com.wewebu.ow.server.ecmimpl.cmis.objectclasses;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.log4j.Logger;

import com.wewebu.ow.server.ecmimpl.cmis.log.OwLog;
import com.wewebu.ow.server.ecmimpl.cmis.object.OwCMISQueryContext;
import com.wewebu.ow.server.ecmimpl.cmis.propertyclasses.OwCMISPropertyClass;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.exceptions.OwInvalidOperationException;
import com.wewebu.ow.server.exceptions.OwObjectNotFoundException;
import com.wewebu.ow.server.util.OwString;
import com.wewebu.ow.server.util.OwString2;

/**
 *<p>
 * Workdesk CMIS object class information holder.<br>
 * Holds class-defined property information data and implements property-class resolving methods.      
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 */
public class OwCMISClassDescription
{

    private static final Logger LOG = OwLog.getLogger(OwCMISClassDescription.class);

    private LinkedHashMap<String, OwCMISPropertyClass> m_localQualifiedPropertyClasses = new LinkedHashMap<String, OwCMISPropertyClass>();

    private LinkedHashMap<String, OwCMISPropertyClass> m_localShortNamePropertyClasses = new LinkedHashMap<String, OwCMISPropertyClass>();

    private Set<OwCMISPropertyClass> queryableProperties = null;

    private OwCMISObjectClass clazz;

    private OwCMISObjectClass parent;

    private List<OwCMISObjectClass> subclasses = new LinkedList<OwCMISObjectClass>();

    private List<OwCMISObjectClass> instantiableSubclasses = new LinkedList<OwCMISObjectClass>();

    private List<String> preferredPropertyOrder;

    public OwCMISClassDescription(OwCMISObjectClass objectClass_p, OwCMISObjectClass parent_p)
    {
        super();
        this.clazz = objectClass_p;
        this.parent = parent_p;
    }

    public OwCMISObjectClass getParentClass()
    {
        return this.parent;
    }

    public OwCMISObjectClass getDescribedClass()
    {
        return this.clazz;
    }

    public void addLocalPropertyClass(OwCMISPropertyClass propertyClass_p)
    {
        m_localQualifiedPropertyClasses.put(propertyClass_p.getClassName(), propertyClass_p);
        m_localShortNamePropertyClasses.put(propertyClass_p.getNonQualifiedName(), propertyClass_p);
    }

    /**
     * Retrieve all property classes locally-defined by this description (ancestors defined excluded).
     * @return a {@link Map} of property classes fully qualified names mapped to {@link OwCMISPropertyClass} 
     *         for all properties defined by this object class (the inherited properties are omitted). 
     * @throws OwException
     */
    public Map<String, OwCMISPropertyClass> getLocalPropertyClasses() throws OwException
    {
        return m_localQualifiedPropertyClasses;
    }

    /**
     * Retrieve all property classes defined by this description (ancestors defined included).
     * @return a {@link Map} of property classes fully qualified names mapped to {@link OwCMISPropertyClass} 
     *         for all properties defined by this object class (inherited properties included).
     * @throws OwException
     */
    public Map<String, OwCMISPropertyClass> getPropertyClasses() throws OwException
    {
        Map<String, OwCMISPropertyClass> hierarchyProperties = null;
        //first request from parent then the specific of current class
        if (this.parent != null)
        {
            hierarchyProperties = this.parent.getPropertyClasses();

            //overwrite properties

            Set<String> hierarchyKeys = new HashSet<String>(hierarchyProperties.keySet());
            for (String classKey : hierarchyKeys)
            {
                OwCMISPropertyClass property = hierarchyProperties.get(classKey);
                String shortName = property.getNonQualifiedName();
                if (m_localShortNamePropertyClasses.containsKey(shortName))
                {
                    hierarchyProperties.remove(classKey);
                    OwCMISPropertyClass localProperty = m_localShortNamePropertyClasses.get(shortName);
                    hierarchyProperties.put(localProperty.getFullQualifiedName(), localProperty);
                }

            }
        }
        else
        {//it is a topmost class, no parent exist
            hierarchyProperties = new LinkedHashMap<String, OwCMISPropertyClass>();
        }
        hierarchyProperties.putAll(getLocalPropertyClasses());

        return reorderProperties(hierarchyProperties);
    }

    public OwCMISPropertyClass getPropertyClass(String strClassName_p) throws OwException
    {
        OwCMISPropertyClass propertyClass = getLocalPropertyClasses().get(strClassName_p);
        if (propertyClass == null)
        {
            propertyClass = m_localShortNamePropertyClasses.get(strClassName_p);
            if (propertyClass == null)
            {

                if (this.parent != null)
                {
                    try
                    {
                        propertyClass = parent.getPropertyClass(strClassName_p);
                    }
                    catch (OwObjectNotFoundException e)
                    {
                        //no error logging here : we just collect exception message information to 
                        //                  enable tracing of the inspected class tree
                        throw new OwObjectNotFoundException(new OwString2("ecmimpl.cmis.OwCMISClassDescription.invalid.property.class.tree", "Property %1 is not defined by object class tree of %2", strClassName_p, this.clazz.getClassName()), e);
                    }
                }
                else
                {
                    //no error logging here : we just collect exception message information to 
                    //                  enable tracing of the inspected class tree (this 
                    //                  is the base class-no parent)
                    throw new OwObjectNotFoundException(new OwString2("ecmimpl.cmis.OwCMISClassDescription.invalid.property.class.tree", "Property %1 is not defined by object class tree of %2", strClassName_p, this.clazz.getClassName()));
                }
            }
        }
        return propertyClass;
    }

    public Set<OwCMISPropertyClass> getQueryablePropertyClasses(OwCMISQueryContext context_p) throws OwException
    {
        if (this.queryableProperties == null)
        {
            this.queryableProperties = new HashSet<OwCMISPropertyClass>();
            Map<String, OwCMISPropertyClass> propertyClasses = getPropertyClasses();

            Set<Entry<String, OwCMISPropertyClass>> propClasseseEntries = propertyClasses.entrySet();
            for (Entry<String, OwCMISPropertyClass> propertyEntry : propClasseseEntries)
            {
                OwCMISPropertyClass propertyClass = propertyEntry.getValue();
                if (propertyClass.isQueryable(context_p))
                {
                    this.queryableProperties.add(propertyClass);
                }
            }
        }
        return this.queryableProperties;
    }

    public Map<String, OwCMISObjectClass> getNamedSubclasses(boolean excludeHiddenAndNonInstantiable_p)
    {
        List<OwCMISObjectClass> subclasses = getSubclasses(excludeHiddenAndNonInstantiable_p);
        Map<String, OwCMISObjectClass> namedSubclasses = new HashMap<String, OwCMISObjectClass>();
        for (OwCMISObjectClass subclass : subclasses)
        {
            namedSubclasses.put(subclass.getClassName(), subclass);
        }
        return namedSubclasses;
    }

    public List<OwCMISObjectClass> getSubclasses(boolean excludeHiddenAndNonInstantiable_p)
    {
        if (excludeHiddenAndNonInstantiable_p)
        {
            return new LinkedList<OwCMISObjectClass>(this.instantiableSubclasses);
        }
        else
        {
            return new LinkedList<OwCMISObjectClass>(this.subclasses);
        }
    }

    public void subclassedBy(OwCMISObjectClass subclass_p) throws OwInvalidOperationException
    {
        OwCMISObjectClass subclassParent = null;

        try
        {
            subclassParent = subclass_p.getParent();
        }
        catch (OwException e)
        {
            LOG.error("OwCMISClassDescription.subclassedBy(): Subclass error! Could not create subclass description information " + this.clazz.getClassName() + "<-" + subclass_p.getClassName(), e);
            throw new OwInvalidOperationException(new OwString("ecmimpl.cmis.OwCMISClassDescription.sublcass.retreiveParentError", "Parent class could not be found."), e);
        }

        if (subclassParent != this.clazz)
        {
            LOG.error("OwCMISClassDescription.subclassedBy(): Subclass error!" + this.clazz.getClassName() + " is not the parent of " + subclass_p.getClassName());
            throw new OwInvalidOperationException(new OwString("ecmimpl.cmis.OwCMISClassDescription.sublcass.error", "Requested class is not the appropriate parent class."));
        }

        this.subclasses.add(subclass_p);
        if (subclass_p.getClass().isAssignableFrom(OwCMISNativeObjectClass.class))
        {
            try
            {
                ((OwCMISNativeObjectClass) subclass_p).getClassDescription().setPreferredPropertyOrder(getPreferredPropertyOrder());
            }
            catch (OwException e)
            {
                LOG.error("OwCMISClassDescription.subclassedBy(): Subclass error!" + this.clazz.getClassName() + " is not the parent of " + subclass_p.getClassName());
            }
        }

        try
        {
            if (subclass_p.canCreateNewObject())
            {
                this.instantiableSubclasses.add(subclass_p);
            }
        }
        catch (OwException e)
        {
            LOG.error("OwCMISClassDescription.subclassedBy(): Subclass error! Could not create subclass description information " + this.clazz.getClassName() + "<-" + subclass_p.getClassName(), e);
            throw new OwInvalidOperationException(new OwString("ecmimpl.cmis.OwCMISClassDescription.sublcass.error", "Requested class is not the appropriate parent class."), e);
        }
    }

    /**
     * Sort provided properties regarding definition of {@link #getPreferredPropertyOrder()} list.
     * @param allProperties_p Map to be sorted
     * @return Map with preferred property order
     * @throws OwException
     * @since 4.0.0.0
     */
    protected Map<String, OwCMISPropertyClass> reorderProperties(Map<String, OwCMISPropertyClass> allProperties_p) throws OwException
    {
        Map<String, OwCMISPropertyClass> orderedProperties = new LinkedHashMap<String, OwCMISPropertyClass>();
        for (String temp : getPreferredPropertyOrder())
        {
            OwCMISPropertyClass propertyClass = allProperties_p.get(temp);
            if (propertyClass != null)
            {
                orderedProperties.put(temp, propertyClass);
            }
            else
            {
                for (Map.Entry<String, OwCMISPropertyClass> entry : allProperties_p.entrySet())
                {
                    propertyClass = entry.getValue();

                    if (propertyClass.getClassName().equalsIgnoreCase(temp) || propertyClass.getNonQualifiedName().equalsIgnoreCase(temp))
                    {
                        orderedProperties.put(entry.getKey(), propertyClass);
                        break;
                    }
                }
            }
        }

        orderedProperties.putAll(allProperties_p);
        return orderedProperties;
    }

    /**
     * Get the preferred property order.
     * @return List of Strings (property names)
     * @since 4.0.0.0
     */
    public List<String> getPreferredPropertyOrder()
    {
        return preferredPropertyOrder == null ? new LinkedList<String>() : preferredPropertyOrder;
    }

    /**
     * Set preferred property order, which is represented by 
     * provided list.
     * @param preferedPropertyOrder List of property names
     * @since 4.0.0.0
     */
    public void setPreferredPropertyOrder(List<String> preferedPropertyOrder)
    {
        this.preferredPropertyOrder = preferedPropertyOrder;
    }

}
