package com.wewebu.ow.server.ecmimpl.cmis.objectclasses;

import java.math.BigInteger;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;
import org.oasis_open.docs.ns.cmis.core._200908.CmisAccessControlListType;
import org.oasis_open.docs.ns.cmis.core._200908.CmisPropertiesType;
import org.oasis_open.docs.ns.cmis.core._200908.CmisPropertyDefinitionType;
import org.oasis_open.docs.ns.cmis.core._200908.CmisTypeDefinitionType;
import org.oasis_open.docs.ns.cmis.core._200908.EnumBaseObjectTypeIds;
import org.oasis_open.docs.ns.cmis.core._200908.EnumVersioningState;
import org.oasis_open.docs.ns.cmis.messaging._200908.CmisContentStreamType;
import org.oasis_open.docs.ns.cmis.messaging._200908.CmisTypeContainer;
import org.oasis_open.docs.ns.cmis.ws._200908.CmisException;

import com.wewebu.ow.server.app.OwMimeManager;
import com.wewebu.ow.server.ecm.OwContentCollection;
import com.wewebu.ow.server.ecm.OwNetwork;
import com.wewebu.ow.server.ecm.OwNetworkContext;
import com.wewebu.ow.server.ecm.OwObject;
import com.wewebu.ow.server.ecm.OwObjectReference;
import com.wewebu.ow.server.ecm.OwPermissionCollection;
import com.wewebu.ow.server.ecm.OwProperty;
import com.wewebu.ow.server.ecm.OwPropertyClass;
import com.wewebu.ow.server.ecm.OwPropertyCollection;
import com.wewebu.ow.server.ecm.OwResource;
import com.wewebu.ow.server.ecm.OwStandardPropertyCollection;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISExceptionCatcher;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISNetwork;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISObjectModel;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISResource;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISSimpleDMSID;
import com.wewebu.ow.server.ecmimpl.cmis.content.OwCMISStandardContentConverter;
import com.wewebu.ow.server.ecmimpl.cmis.log.OwLog;
import com.wewebu.ow.server.ecmimpl.cmis.object.OwCMISObjectBase;
import com.wewebu.ow.server.ecmimpl.cmis.object.OwCMISObjectExtension;
import com.wewebu.ow.server.ecmimpl.cmis.object.OwCMISQueryContext;
import com.wewebu.ow.server.ecmimpl.cmis.permissions.OwCMISPermissionCollection;
import com.wewebu.ow.server.ecmimpl.cmis.property.OwCMISConvertFilter;
import com.wewebu.ow.server.ecmimpl.cmis.property.OwCMISFilterInternal;
import com.wewebu.ow.server.ecmimpl.cmis.property.OwCMISFilterReadonly;
import com.wewebu.ow.server.ecmimpl.cmis.property.OwCMISProperty;
import com.wewebu.ow.server.ecmimpl.cmis.property.OwCMISPropertyNames;
import com.wewebu.ow.server.ecmimpl.cmis.property.OwCMISStandardPropertyCollectionConverter;
import com.wewebu.ow.server.ecmimpl.cmis.propertyclasses.OwCMISNativePropertyClass;
import com.wewebu.ow.server.ecmimpl.cmis.propertyclasses.OwCMISPropertyClass;
import com.wewebu.ow.server.ecmimpl.cmis.propertyclasses.OwCMISPropertyClassFactory;
import com.wewebu.ow.server.ecmimpl.cmis.util.OwPropertyDefinitionHelper;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.exceptions.OwInvalidOperationException;
import com.wewebu.ow.server.exceptions.OwObjectNotFoundException;
import com.wewebu.ow.server.util.OwObjectIDCodeUtil;
import com.wewebu.ow.server.util.OwString;

/**
 *<p>
 * Object class wrapper for CMIS native object type definition.<br>
 * It relays on {@link CmisTypeDefinitionType} data to implement the 
 * Workdesk object-class interface.
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 */
public abstract class OwCMISNativeObjectClass implements OwCMISObjectClass
{

    private static final Logger LOG = OwLog.getLogger(OwCMISNativeObjectClass.class);

    private OwCMISClassDescription classDescription = null;

    private OwCMISObjectModel objectModel;

    private CmisTypeDefinitionType m_type;
    /**representing the object type, one of the OwObjectReference.OBJECT_TYPE_...*/
    private int m_itype;
    /**String representing the MIME-Type of this object class*/
    private String m_mimetype;

    private OwCMISPropertyClassFactory propertyClassFactory;

    private boolean hierarchyInitialized = false;

    /** create class description
     * 
     * @param objectModel_p
     * @param propertyClassFactory_p 
     * @param type_p
     * @throws OwException 
     */
    public OwCMISNativeObjectClass(OwCMISObjectModel objectModel_p, OwCMISPropertyClassFactory propertyClassFactory_p, CmisTypeDefinitionType type_p) throws OwException
    {
        m_type = type_p;

        this.propertyClassFactory = propertyClassFactory_p;
        this.objectModel = objectModel_p;

        setObjectType(type_p.getBaseId());

        String parentId = m_type.getParentId();
        OwCMISObjectClass parent = this.objectModel.getObjectClass(parentId);

        this.classDescription = new OwCMISClassDescription(this, parent);

        initializeProperties();

        if (parent != null)
        {
            parent.subclassedBy(this);
        }
    }

    public String getMimetype()
    {
        return m_mimetype;
    }

    protected void setObjectType(EnumBaseObjectTypeIds base_p)
    {
        switch (base_p)
        {
            case CMIS_DOCUMENT:
                m_itype = OwObjectReference.OBJECT_TYPE_DOCUMENT;
                m_mimetype = null;
                break;

            case CMIS_FOLDER:
                m_itype = OwObjectReference.OBJECT_TYPE_FOLDER;
                m_mimetype = OwMimeManager.MIME_TYPE_PREFIX_OW_FOLDER + this.m_type.getId();
                break;

            case CMIS_POLICY:
                m_itype = OwObjectReference.OBJECT_TYPE_CUSTOM;
                m_mimetype = OwCMISObjectClass.MIME_TYPE_PREFIX_OW_POLICY + this.m_type.getId();
                break;

            case CMIS_RELATIONSHIP:
                m_itype = OwObjectReference.OBJECT_TYPE_CUSTOM;
                m_mimetype = OwCMISObjectClass.MIME_TYPE_PREFIX_OW_RELATIONSHIP + this.m_type.getId();
                break;
        }
    }

    protected void initializeProperties() throws OwException
    {
        // load property classes
        List<CmisPropertyDefinitionType> propdefinitions = m_type.getPropertyDefinition();
        boolean trace = LOG.isTraceEnabled();
        for (CmisPropertyDefinitionType proptype : propdefinitions)
        {
            //TODO: check if virtual base direct child!!!
            if (getParent() == null || !proptype.isInherited())
            {
                OwCMISPropertyClass propertyClass = this.propertyClassFactory.createPropertyClass(proptype, this);

                this.classDescription.addLocalPropertyClass(propertyClass);

                if (trace)
                {
                    LOG.trace("ADDED PROPERTY " + getClassName() + " @ " + propertyClass.toString());
                }
            }
            else if (proptype.isInherited())
            {
                try
                {
                    OwCMISNativePropertyClass parentProperty = (OwCMISNativePropertyClass) getParent().getPropertyClass(proptype.getId());

                    if (OwPropertyDefinitionHelper.isInheritanceDifferent(proptype, parentProperty.getNativeType()))
                    {
                        if (LOG.isTraceEnabled())
                        {
                            LOG.trace("OwCMISAbstractNativeObjectClass.getNativePropertyClasses: Inherited but different prop-definition: " + proptype);
                        }
                        OwCMISPropertyClass propertyClass = this.propertyClassFactory.createPropertyClass(proptype, this);
                        this.classDescription.addLocalPropertyClass(propertyClass);
                    }
                    else
                    {
                        if (trace)
                        {
                            LOG.trace("INHERITED PROPERTY " + getClassName() + " @ " + parentProperty.toString());
                        }
                    }
                }
                catch (OwObjectNotFoundException e)
                {
                    if (trace)
                    {
                        String msg = "property " + proptype.getId() + " SHOULD BE INHERITED from object class " + getParent().getClassName() + " in " + getClassName() + "(exception " + e.getMessage() + ")";
                        LOG.trace("OwCMISNativeObjectClass.initializeProperties:" + msg);
                    }
                    // normally we should throw an exception here but we try to manage the situation... 
                    OwCMISPropertyClass propertyClass = this.propertyClassFactory.createPropertyClass(proptype, this);

                    this.classDescription.addLocalPropertyClass(propertyClass);

                    if (trace)
                    {
                        LOG.trace("ADDED MISSINHERITED PROPERTY  " + getClassName() + " @ " + propertyClass.toString());
                    }
                }
            }

        }

        Set<OwCMISPropertyClass> customPropertyClasses = createCustomPropertyClasses();
        for (OwCMISPropertyClass propertyClass : customPropertyClasses)
        {
            this.classDescription.addLocalPropertyClass(propertyClass);
            if (trace)
            {
                LOG.trace("ADDED INTERNAL PROPERTY " + getClassName() + " @ " + propertyClass.toString());
            }
        }

    }

    protected Set<OwCMISPropertyClass> createCustomPropertyClasses() throws OwException
    {
        if (getParent() == null)
        {
            //            String namePropertyName = getNamePropertyName();
            //            OwCMISPropertyClass nativeNamePropertyClass = getPropertyClass(namePropertyName);
            //            OwCMISPropertyClass namePropertyClass = new OwCMISMappedInternalPropertyClass(OwResource.m_ObjectNamePropertyClass, this, nativeNamePropertyClass);

            Set<OwCMISPropertyClass> customProperties = new HashSet<OwCMISPropertyClass>();
            //            customProperties.add(namePropertyClass);

            return customProperties;
        }
        else
        {
            return new HashSet<OwCMISPropertyClass>();
        }
    }

    protected OwCMISClassDescription getClassDescription() throws OwException
    {
        return this.classDescription;
    }

    /**
     * 
     * @return a {@link Map} of property classes fully qualified names mapped to {@link OwCMISPropertyClass} 
     *         for all properties defined by this object class (inherited properties included).
     * @throws OwException
     */
    public Map<String, OwCMISPropertyClass> getPropertyClasses() throws OwException
    {
        OwCMISClassDescription description = getClassDescription();
        return description.getPropertyClasses();
    }

    public boolean canCreateNewObject() throws OwException
    {
        return getNativeObject().isCreatable();
    }

    @SuppressWarnings("unchecked")
    public Map<?, ?> getChildNames(OwNetwork network_p, boolean excludeHiddenAndNonInstantiable_p) throws OwException
    {
        List<OwCMISObjectClass> lst = (List<OwCMISObjectClass>) getChilds(network_p, excludeHiddenAndNonInstantiable_p);
        Map<String, OwCMISObjectClass> map = new HashMap<String, OwCMISObjectClass>();
        for (OwCMISObjectClass child : lst)
        {
            map.put(child.getClassName(), child);
        }
        return map;
    }

    public synchronized List<?> getChilds(OwNetwork network_p, boolean excludeHiddenAndNonInstantiable_p) throws OwException
    {

        if (!hierarchyInitialized)
        {
            OwCMISNetwork cmisNetwork = (OwCMISNetwork) network_p;
            try
            {
                List<CmisTypeContainer> lst = cmisNetwork.getRepositoryServicePort().getTypeDescendants(getResource().getID(), getClassName(), BigInteger.ONE, false, null);
                Hashtable<String, OwCMISObjectClass> objClassChildren = new Hashtable<String, OwCMISObjectClass>();
                for (CmisTypeContainer con : lst)
                {
                    OwCMISObjectClass child = cmisNetwork.getObjectClass(con.getType().getId(), getResource());
                    //check that this class is also added to children collection
                    if (!this.getClassName().equals(child.getClassName()))
                    {
                        objClassChildren.put(child.getClassName(), child);
                    }
                }
                hierarchyInitialized = true;
            }
            catch (CmisException e)
            {
                OwNetworkContext context = cmisNetwork.getContext();
                OwCMISExceptionCatcher catcher = new OwCMISExceptionCatcher(e, context.getLocale());
                LOG.error("OwCMISNativeObjectClass.getChilds(): Could not retrieve class descendants! A CMIS error occurred : " + catcher.getLogMessage());
                throw catcher.toOwException(context.localize("ecmimpl.cmis.OwCMISNativeObjectClass.getChilds.error", "Could not retrieve class descendants!"), true);
            }
        }

        return this.classDescription.getSubclasses(excludeHiddenAndNonInstantiable_p);
    }

    public String getClassName()
    {
        return m_type.getId();
    }

    public String getDescription(Locale locale_p)
    {
        return OwString.localize(locale_p, getClassName() + ".Desc", m_type.getDescription());
    }

    public String getDisplayName(Locale locale_p)
    {
        return OwString.localizeLabel(locale_p, getClassName(), m_type.getDisplayName());
    }

    public List<?> getModes(int operation_p) throws OwException
    {
        return null;
    }

    public String getNamePropertyName() throws OwException
    {
        return getPropertyClass(OwCMISPropertyNames.NAME.getId()).getClassName();
    }

    public OwCMISObjectClass getParent() throws OwException
    {
        return this.classDescription.getParentClass();
    }

    public OwCMISPropertyClass getPropertyClass(String strClassName_p) throws OwException
    {
        OwCMISClassDescription description = getClassDescription();
        return description.getPropertyClass(strClassName_p);
    }

    public Collection<String> getPropertyClassNames() throws OwException
    {
        return getPropertyClasses().keySet();
    }

    public int getType()
    {
        return m_itype;
    }

    public boolean hasChilds(OwNetwork network_p, boolean excludeHiddenAndNonInstantiable_p, int context_p) throws OwException
    {
        return !getChilds(network_p, excludeHiddenAndNonInstantiable_p).isEmpty();
    }

    public boolean isHidden() throws OwException
    {
        return false;
    }

    /**
     * Helper method to retrieve the current OwCMISResource
     * reference for this object
     * @return OwCMISResource
     */
    public OwCMISResource getResource()
    {
        return this.objectModel.getResource();
    }

    public OwCMISObjectModel getObjectModel()
    {
        return this.objectModel;
    }

    /* Public since 3.2.0.0
     * Helper method to retrieve the back end based native object.
     * @return CmisTypeDefinitionType
     */
    public CmisTypeDefinitionType getNativeObject()
    {
        return this.m_type;
    }

    public boolean isContentRequired() throws OwException
    {
        return false;
    }

    public String getQueryName()
    {
        String queryName = this.m_type.getQueryName();
        //work-around for alfresco before V3.4 query name problem 
        if ((queryName.equals("cmis:document") && !"cmis:document".equals(this.m_type.getId())) || (queryName.equals("cmis:folder") && !"cmis:folder".equals(this.m_type.getId())))
        {
            LOG.warn("OwCMISNativeObjectClass.getQueryName():Inavlid query name : " + queryName + " for cmis object class " + this.m_type.getId() + ". Ingnoring the query name and proceeding with the id " + this.m_type.getId() + ".");
            queryName = this.m_type.getId();
            if ((queryName.startsWith("D:") || queryName.startsWith("F:")) && queryName.length() > 2 && queryName.indexOf(':', 2) != -1)
            {
                LOG.warn("OwCMISNativeObjectClass.getQueryName():Inavlid ID name : " + queryName + " for cmis object class " + this.m_type.getId() + ". Removing Alfresco-like 2 character type mark!");
                queryName = queryName.substring(2);
            }
        }
        return queryName;
    }

    public boolean isQueryable()
    {
        return this.m_type.isQueryable();
    }

    public Set<OwCMISPropertyClass> getQueryablePropertyClasses(OwCMISQueryContext context_p) throws OwException
    {
        OwCMISClassDescription description = getClassDescription();
        return description.getQueryablePropertyClasses(context_p);
    }

    /**
     * Method to verify the existence of object-type id property (which must exist). 
     * Since CMIS 1.0 does not have parameter to provide the object-type explicit to
     * classify the new/created object.
     * @param network_p OwCMISNetwork current network
     * @return OwPropertyCollection
     * @throws OwException
     */
    @SuppressWarnings("unchecked")
    public OwPropertyCollection getClassProperties(OwCMISNetwork network_p) throws OwException
    {
        OwCMISPropertyClass propClass = getPropertyClass(OwCMISPropertyNames.OBJECT_TYPE_ID.getId());

        OwCMISProperty objectIDProperty = propClass.newProperty(null, m_type.getId(), network_p);
        OwCMISPropertyClass idClass = objectIDProperty.getPropertyClass();
        OwPropertyCollection classProperties = new OwStandardPropertyCollection();

        classProperties.put(idClass.getClassName(), objectIDProperty);

        return classProperties;
    }

    protected abstract String createNewObjectCMISObject(OwCMISNetwork network_p, EnumVersioningState state_p, CmisPropertiesType properties_p, CmisAccessControlListType addedACL_p, CmisAccessControlListType removedACL_p,
            CmisContentStreamType content_p, String parentID_p, boolean keepCheckedOut_p) throws OwException;

    @SuppressWarnings("unchecked")
    public String createNewObject(OwCMISNetwork network_p, boolean promote_p, Object mode_p, OwResource resource_p, OwPropertyCollection properties_p, OwPermissionCollection permissions_p, OwContentCollection content_p, OwObject parent_p,
            String strMimeType_p, String strMimeParameter_p, boolean keepCheckedOut_p) throws OwException
    {

        OwPropertyCollection propertyCollectionCopy = new OwStandardPropertyCollection();
        propertyCollectionCopy.putAll(properties_p);

        OwPropertyCollection classPropertiesMap = getClassProperties(network_p);
        Collection<?> classProperties = classPropertiesMap.values();
        for (Iterator<?> i = classProperties.iterator(); i.hasNext();)
        {
            OwProperty classProperty = (OwProperty) i.next();
            OwPropertyClass classPropertyClass = null;
            try
            {
                classPropertyClass = classProperty.getPropertyClass();
            }
            catch (OwException e)
            {
                throw e;
            }
            catch (Exception e)
            {
                OwNetworkContext context = network_p.getContext();
                LOG.fatal("OwCMISNativeObjectClass.createNewObject():Invalid property found while creating object! Could not retrieve property class!", e);
                throw new OwInvalidOperationException(context.localize("ecmimpl.cmis.OwCMISNativeObjectClass.createNewObject.invalid.property.error", "Invalid property found while creating object!"), e);
            }

            propertyCollectionCopy.put(classPropertyClass.getClassName(), classProperty);
        }

        try
        {

            EnumVersioningState state = promote_p ? EnumVersioningState.MAJOR : EnumVersioningState.MINOR;

            state = keepCheckedOut_p ? EnumVersioningState.CHECKEDOUT : state;

            OwCMISResource resource = getResource();

            OwCMISObjectExtension objectExtension = resource.getExtension(OwCMISObjectBase.FILTER_PROPERTY_EXTENSIONS_EP, OwCMISObjectExtension.class, null);
            OwPropertyCollection propertyExtensions = objectExtension.filterPropertyExtensions(getResource(), getClassName(), propertyCollectionCopy);

            OwStandardPropertyCollection basicProperties = new OwStandardPropertyCollection();
            basicProperties.putAll(propertyCollectionCopy);
            if (propertyExtensions != null)
            {
                Set<?> extensionKeys = propertyExtensions.keySet();
                for (Object object : extensionKeys)
                {
                    basicProperties.remove(object);
                }
            }

            OwCMISConvertFilter convertFilter = new OwCMISFilterReadonly(OwPropertyClass.CONTEXT_ON_CREATE);
            convertFilter = new OwCMISFilterInternal(convertFilter);
            CmisPropertiesType nativeProperties = new OwCMISStandardPropertyCollectionConverter(convertFilter).createCmisProperties(basicProperties, null, resource_p.getID(), network_p);

            objectExtension = resource.getExtension(OwCMISObjectBase.ADD_PROPERTY_EXTENSIONS_EP, OwCMISObjectExtension.class, nativeProperties);
            nativeProperties = objectExtension.addPropertyExtensions(nativeProperties, propertyExtensions, null, resource.getID(), network_p);

            CmisContentStreamType contendDataHandler = new OwCMISStandardContentConverter().createCmisContentStream(content_p);

            CmisAccessControlListType addedACL = null;
            CmisAccessControlListType removedACL = null;

            if (permissions_p != null)
            {
                if (permissions_p instanceof OwCMISPermissionCollection)
                {
                    OwCMISPermissionCollection cmisPermission = (OwCMISPermissionCollection) permissions_p;
                    CmisAccessControlListType zeroACL = new CmisAccessControlListType();
                    CmisAccessControlListType[] operations = cmisPermission.diff(zeroACL);

                    removedACL = operations[0];
                    addedACL = operations[1];
                }
                else
                {
                    LOG.error("OwCMISNativeObjectClass.createNewObject : Invalid permissions class: " + permissions_p.getClass() + " instead of " + OwCMISPermissionCollection.class);
                    throw new OwInvalidOperationException("Internal error on create object");
                }
            }

            String newID = createNewObjectCMISObject(network_p, state, nativeProperties, addedACL, removedACL, contendDataHandler, getParentId(parent_p), keepCheckedOut_p);

            return OwCMISSimpleDMSID.createDMSID(network_p.getDMSPrefix(), resource.getID(), newID);
        }
        catch (OwException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            OwNetworkContext context = network_p.getContext();
            LOG.fatal("OwCMISNativeObjectClass.createNewObject():Could not create object class!", e);
            throw new OwInvalidOperationException(context.localize("ecmimpl.cmis.OwCMISNativeObjectClass.createNewObject.error", "Could not create object!"), e);
        }
    }

    public void subclassedBy(OwCMISObjectClass subclass_p) throws OwInvalidOperationException
    {
        this.classDescription.subclassedBy(subclass_p);
    }

    public boolean isAssignableFrom(OwCMISObjectClass class_p) throws OwException
    {
        OwCMISResource resource = getResource();
        OwCMISResource classResource = class_p.getResource();
        if (resource.getID().equals(classResource.getID()))
        {
            if (getClassName().equals(class_p.getClassName()))
            {
                return true;
            }
            else
            {
                OwCMISObjectClass cpParent = class_p.getParent();
                while (cpParent != null)
                {
                    if (cpParent.getClassName().equals(getClassName()))
                    {
                        return true;
                    }
                    else
                    {
                        cpParent = cpParent.getParent();
                    }
                }

                return false;
            }
        }
        else
        {
            return false;
        }
    }

    /**
     * Retrieve the id which is used for filling.
     * @param parent_p OwObject (can be null)
     * @return String id or null if parent is null
     * @throws OwException
     * @since 4.0.0.0
     */
    protected String getParentId(OwObject parent_p) throws OwException
    {
        if (parent_p == null)
        {
            return null;
        }
        String id = parent_p.getID();
        if (id.startsWith(getResource().getID()))
        {
            id = getResource().getRepsitoryRootFolderId();
        }
        return OwObjectIDCodeUtil.decode(id);
    }
}