package com.wewebu.ow.server.ecmimpl.cmis.permissions;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;
import org.oasis_open.docs.ns.cmis.core._200908.CmisAccessControlEntryType;
import org.oasis_open.docs.ns.cmis.core._200908.CmisAccessControlListType;
import org.oasis_open.docs.ns.cmis.core._200908.CmisAccessControlPrincipalType;

import com.wewebu.ow.server.ecm.OwPermissionCollection;
import com.wewebu.ow.server.ecm.OwPolicy;
import com.wewebu.ow.server.ecm.OwPrivilege;
import com.wewebu.ow.server.ecm.OwPrivilegeSet;
import com.wewebu.ow.server.ecm.OwReason;
import com.wewebu.ow.server.ecm.OwUserInfo;
import com.wewebu.ow.server.ecmimpl.OwStandardReason;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISResource;
import com.wewebu.ow.server.ecmimpl.cmis.log.OwLog;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.exceptions.OwInvalidOperationException;
import com.wewebu.ow.server.exceptions.OwServerException;
import com.wewebu.ow.server.util.OwString;

/**
 *<p>
 * CMIS {@link CmisAccessControlListType} based permission collection implementation. 
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 *@since 3.2.0.0
 */
public class OwCMISPermissionCollection implements OwPermissionCollection
{
    private static final Logger LOG = OwLog.getLogger(OwCMISPermissionCollection.class);

    private CmisAccessControlListType acl;
    private OwCMISResource resource;

    /**
     * Constructor
     * 
     * @param acl_p netive ACL object of this permissions. The given native acl is deep cloned.
     * @param resource_p
     */
    public OwCMISPermissionCollection(CmisAccessControlListType acl_p, OwCMISResource resource_p)
    {
        super();
        this.resource = resource_p;
        reset(acl_p);
    }

    private CmisAccessControlListType deepCopy(CmisAccessControlListType acl_p)
    {
        CmisAccessControlListType copy = new CmisAccessControlListType();

        List<CmisAccessControlEntryType> permissionsCopy = copy.getPermission();
        List<CmisAccessControlEntryType> permissions = acl_p.getPermission();

        for (CmisAccessControlEntryType ace : permissions)
        {
            CmisAccessControlEntryType aceCopy = new CmisAccessControlEntryType();

            aceCopy.setPrincipal(ace.getPrincipal());
            aceCopy.setDirect(ace.isDirect());
            List<String> acePermissionsCopy = aceCopy.getPermission();
            acePermissionsCopy.addAll(ace.getPermission());

            permissionsCopy.add(aceCopy);
        }

        return copy;

    }

    /**
     * Sets the current ACL. The given ACL is deep-cloned.  
     * 
     * @param acl_p
     * @since 3.2.0.1
     */
    public void reset(CmisAccessControlListType acl_p)
    {
        this.acl = deepCopy(acl_p);
    }

    private Map<String, Set<String>> mergePermissions(List<CmisAccessControlEntryType> aceList_p, Map<String, CmisAccessControlPrincipalType> principals_p)
    {
        Map<String, Set<String>> mergedPermissions = new HashMap<String, Set<String>>();

        for (CmisAccessControlEntryType ace : aceList_p)
        {
            CmisAccessControlPrincipalType principal = ace.getPrincipal();
            String principalDirectKey = principal.getPrincipalId() + (ace.isDirect() ? "D" : "I");
            Set<String> mergedSet = mergedPermissions.get(principalDirectKey);

            if (mergedSet == null)
            {
                mergedSet = new HashSet<String>();
                mergedPermissions.put(principalDirectKey, mergedSet);
                principals_p.put(principalDirectKey, principal);
            }

            mergedSet.addAll(ace.getPermission());
        }

        return mergedPermissions;
    }

    /**
     * Obtain the operations (additions and removals of native ACLs) to get from the given native ACL 
     * to this permissions native representation.
     * 
     * @param acl_p 
     * @return an {@link CmisAccessControlListType} array of size 2 :  <br/>
     *         index 0 the {@link CmisAccessControlListType} needed to be removed from the given ACL  
     *         index 1 the {@link CmisAccessControlListType} needed to be added to the given ACL
     * @since 3.2.0.1
     */
    public CmisAccessControlListType[] diff(CmisAccessControlListType acl_p)
    {
        CmisAccessControlListType removedACL = new CmisAccessControlListType();
        List<CmisAccessControlEntryType> removedACEList = removedACL.getPermission();

        CmisAccessControlListType addedACL = new CmisAccessControlListType();
        List<CmisAccessControlEntryType> addedACEList = addedACL.getPermission();

        Map<String, CmisAccessControlPrincipalType> principals = new HashMap<String, CmisAccessControlPrincipalType>();
        List<CmisAccessControlEntryType> permissions = acl.getPermission();
        Map<String, Set<String>> mergedPermissions = mergePermissions(permissions, principals);

        Map<String, CmisAccessControlPrincipalType> principalsReference = new HashMap<String, CmisAccessControlPrincipalType>();
        List<CmisAccessControlEntryType> permissionsReference = acl_p.getPermission();
        Map<String, Set<String>> mergedPermissionsReference = mergePermissions(permissionsReference, principalsReference);

        Set<String> principalsSet = new HashSet<String>(mergedPermissionsReference.keySet());
        principalsSet.addAll(mergedPermissions.keySet());

        for (String principalDirectKey : principalsSet)
        {
            Set<String> currentPermissions = mergedPermissions.get(principalDirectKey);
            Set<String> referencePermissionsList = mergedPermissionsReference.get(principalDirectKey);

            if (currentPermissions != null && referencePermissionsList != null)
            {
                Set<String> difference = new HashSet<String>(referencePermissionsList);
                Set<String> added = new HashSet<String>(currentPermissions);

                HashSet<String> currentReference = new HashSet<String>(referencePermissionsList);
                currentReference.retainAll(currentPermissions);
                difference.removeAll(currentReference);

                added.removeAll(referencePermissionsList);

                if (!difference.isEmpty())
                {
                    CmisAccessControlEntryType removedACE = new CmisAccessControlEntryType();
                    removedACE.setPrincipal(principalsReference.get(principalDirectKey));
                    removedACE.getPermission().addAll(difference);
                    removedACE.setDirect(principalDirectKey.endsWith("D"));
                    removedACEList.add(removedACE);
                }

                if (!added.isEmpty())
                {
                    CmisAccessControlEntryType addedACE = new CmisAccessControlEntryType();
                    addedACE.setPrincipal(principals.get(principalDirectKey));
                    addedACE.getPermission().addAll(added);
                    addedACE.setDirect(principalDirectKey.endsWith("D"));
                    addedACEList.add(addedACE);
                }

            }
            else if (currentPermissions == null)
            {
                CmisAccessControlEntryType removedACE = new CmisAccessControlEntryType();
                removedACE.setPrincipal(principalsReference.get(principalDirectKey));
                removedACE.getPermission().addAll(referencePermissionsList);
                removedACE.setDirect(principalDirectKey.endsWith("D"));
                removedACEList.add(removedACE);
            }
            else
            {
                CmisAccessControlEntryType addedACE = new CmisAccessControlEntryType();
                addedACE.setPrincipal(principals.get(principalDirectKey));
                addedACE.getPermission().addAll(currentPermissions);
                addedACE.setDirect(principalDirectKey.endsWith("D"));
                addedACEList.add(addedACE);
            }

        }

        return new CmisAccessControlListType[] { removedACL, addedACL };
    }

    public boolean canGetPrivileges()
    {
        return true;
    }

    public boolean canSetPrivileges()
    {

        return true;
    }

    public boolean canDenyPrivileges()
    {
        return false;
    }

    public Collection<OwPrivilege> getAvailablePrivileges(OwUserInfo principal_p)
    {
        Map<String, OwPrivilege> privilegesMap = resource.getAvailablePrivileges(principal_p);
        return privilegesMap.values();
    }

    public Map getAvailableInheritanceDepths()
    {
        // TODO Auto-generated method stub
        return null;
    }

    public OwPrivilegeSet addPrivilegeSet(OwUserInfo principal_p, Collection privileges_p, boolean deny_p, int inheritancedepth_p) throws OwException
    {
        try
        {
            List<CmisAccessControlEntryType> aces = acl.getPermission();
            CmisAccessControlEntryType ace = new CmisAccessControlEntryType();
            ace.setDirect(true);
            CmisAccessControlPrincipalType principal = new CmisAccessControlPrincipalType();
            principal.setPrincipalId(principal_p.getUserName());
            ace.setPrincipal(principal);
            List<String> permissions = ace.getPermission();
            for (OwPrivilege privilege : (Collection<OwPrivilege>) privileges_p)
            {
                permissions.add(privilege.getName());
            }
            aces.add(ace);
            return new OwCMISPrivilegeSet(ace, resource.getAvailablePrivileges());
        }
        catch (OwException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new OwInvalidOperationException("Could not add privileges", e);
        }
    }

    public Collection<OwCMISPrivilegeSet> getAppliedPrivilegeSets()
    {
        List<OwCMISPrivilegeSet> privilegeSets = new LinkedList<OwCMISPrivilegeSet>();

        List<CmisAccessControlEntryType> accessControlEntries = acl.getPermission();

        Map<String, OwPrivilege> availablePrivileges = resource.getAvailablePrivileges();
        for (CmisAccessControlEntryType cmisAccessControlEntry : accessControlEntries)
        {
            OwCMISPrivilegeSet privilegeSet = new OwCMISPrivilegeSet(cmisAccessControlEntry, availablePrivileges);
            privilegeSets.add(privilegeSet);

        }

        return privilegeSets;
    }

    public void removePrivilegeSet(OwPrivilegeSet privilegeset_p) throws OwException
    {
        List<String> removedPermissions = new LinkedList<String>();
        Collection<OwPrivilege> privileges = privilegeset_p.getPrivileges();

        for (OwPrivilege privilege : privileges)
        {
            removedPermissions.add(privilege.getName());
        }

        OwUserInfo principalUserInfo = privilegeset_p.getPrincipal();
        String principalName;
        try
        {
            principalName = principalUserInfo.getUserName();
        }
        catch (OwException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            throw new OwServerException(new OwString("ecmimpl.cmis.Permissions.error", "An error occured during permissions handling."), e);
        }
        List<CmisAccessControlEntryType> accessControlEntries = acl.getPermission();
        List<CmisAccessControlEntryType> emptyACEs = new LinkedList<CmisAccessControlEntryType>();
        for (CmisAccessControlEntryType accessControlEntry : accessControlEntries)
        {
            CmisAccessControlPrincipalType principal = accessControlEntry.getPrincipal();
            if (principal.getPrincipalId().equals(principalName))
            {
                List<String> livePermissions = accessControlEntry.getPermission();
                livePermissions.removeAll(removedPermissions);
                if (livePermissions.isEmpty())
                {
                    emptyACEs.add(accessControlEntry);
                }

            }
        }
        accessControlEntries.removeAll(emptyACEs);

    }

    public boolean canGetPolicies()
    {
        // TODO Auto-generated method stub
        return false;
    }

    public boolean canSetPolicies()
    {
        // TODO Auto-generated method stub
        return false;
    }

    public boolean canAddMultiPolicy()
    {
        // TODO Auto-generated method stub
        return false;
    }

    public Collection getAvailablePolicies(OwUserInfo principal_p)
    {
        // TODO Auto-generated method stub
        return null;
    }

    public void addPolicy(OwPolicy policy_p) throws OwException
    {
        // TODO Auto-generated method stub

    }

    public Collection getAppliedPolicies()
    {
        // TODO Auto-generated method stub
        return null;
    }

    public void removePolicy(OwPolicy policy_p) throws OwException
    {
        // TODO Auto-generated method stub
    }

    public OwReason canEditPermissions()
    {
        return OwStandardReason.ALLOWED;
    }

}
