package com.wewebu.ow.server.ecmimpl.cmis.property;

import java.util.LinkedList;
import java.util.List;

import org.oasis_open.docs.ns.cmis.core._200908.CmisObjectType;
import org.oasis_open.docs.ns.cmis.core._200908.CmisPropertiesType;
import org.oasis_open.docs.ns.cmis.core._200908.CmisProperty;
import org.oasis_open.docs.ns.cmis.core._200908.CmisPropertyDefinitionType;

import com.wewebu.ow.server.exceptions.OwInvalidOperationException;

/**
 *<p>
 * OwCMISPropertyNames enumeration and helper class.
 * Enumeration for the frequently used properties, 
 * with helper methods to retrieve that property from native CmisObjectType. 
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 */
public enum OwCMISPropertyNames
{
    /**Constant for the cmis:path property*/
    PATH("cmis:path", true),
    /**Constant for the cmis:allowedChildObjecttypeIDs property*/
    ALLOWED_CHILD_OBJECTTYPE_IDS("cmis:allowedChildObjectTypeIds", true),
    /**Constant for the cmis:baseTypeId property*/
    BASE_TYPE_ID("cmis:baseTypeId", true),
    /**Constant for the cmis:isImmutable property*/
    IS_IMMUTABLE("cmis:isImmutable", true),
    /**Constant for the cmis:checkinComment property*/
    CHECKIN_COMMENT("cmis:checkinComment", true),
    /**Constant for the cmis:isVersionSeriesCheckedOut property*/
    IS_VERSION_SERIES_CHECKED_OUT("cmis:isVersionSeriesCheckedOut", true),
    /**Constant for the cmis:isLatestMajorVersion property*/
    IS_LATEST_MAJOR_VERSION("cmis:isLatestMajorVersion", true),
    /**Constant for the cmis:isMajorVersion property*/
    IS_MAJOR_VERSION("cmis:isMajorVersion", true),
    /**Constant for the cmis:versionSeriesCheckedOutBy property*/
    VERSION_SERIES_CHECKED_OUT_BY("cmis:versionSeriesCheckedOutBy", true),
    /**Constant for the cmis:versionSeriesCheckedOutId property*/
    VERSION_SERIES_CHECKED_OUT_ID("cmis:versionSeriesCheckedOutId", true),
    /**Constant for the cmis:isLatestVersion property*/
    IS_LATEST_VERSION("cmis:isLatestVersion", true),
    /**Constant for the cmis:changeToken property*/
    CHANGE_TOKEN("cmis:changeToken", true),
    /**Constant for the cmis:versionLabel property*/
    VERSION_LABEL("cmis:versionLabel", true),
    /**Constant for the cmis:contentStreamId property*/
    CONTENT_STREAM_ID("cmis:contentStreamId", true),
    /**Constant for the cmis:versionSeriesId property*/
    VERSION_SERIES_ID("cmis:versionSeriesId", true),
    /**Constant for the cmis:parentId property*/
    PARENT_ID("cmis:parentId", true),
    /**Constant for the cmis:objectId property*/
    OBJECT_ID("cmis:objectId", true),
    /**Constant for the cmis:name property*/
    NAME("cmis:name", true),
    /**Constant for the cmis:objectTypeId property*/
    OBJECT_TYPE_ID("cmis:objectTypeId", true),
    /**Constant for the cmis:contentStreamMimeType property*/
    CONTENT_STREAM_MIME_TYPE("cmis:contentStreamMimeType"),
    /**Constant for the cmis:contentStreamFileName property*/
    CONTENT_STREAM_FILENAME("cmis:contentStreamFileName"),
    /**Constant for the cmis:contentStreamLength property*/
    CONTENT_STREAM_LENGTH("cmis:contentStreamLength"),
    /**Constant for the value cmis:none which is used very often, not a real property*/
    NONE("cmis:none");

    private static SyncList LST = null;

    /**
     * Return an object-type independent list of properties which are
     * system properties.
     * @return List of String, containing the names of the system properties
     */
    public static List<String> getSystemPropertiesCmisNames()
    {
        if (LST == null)
        {
            LST = SyncList.getInstance();
        }
        return LST.getList();
    }

    private String cmisId;
    private boolean systemProperty;

    private OwCMISPropertyNames(String cmisName_p)
    {
        this(cmisName_p, false);
    }

    private OwCMISPropertyNames(String cmisName_p, boolean systemProperty_p)
    {
        this.cmisId = cmisName_p;
        this.systemProperty = systemProperty_p;
    }

    /**
     * Return boolean if this
     * property is an object type independent system property.
     * @return boolean
     */
    public boolean isSystemProperty()
    {
        return this.systemProperty;
    }

    /**
     * Return a String which represents
     * the id/name of the property.
     * @return String
     */
    public String getId()
    {
        return this.cmisId;
    }

    /**
     * Try to retrieve the property form native CmisObjectType
     * calling {@link #fromPropertyList(List)}, can return null
     * if the CmisObjectType does not contain that property.
     * @param cmisObject_p CmisObjectType, non-null value
     * @return CmisProperty or null
     */
    public CmisProperty fromObject(CmisObjectType cmisObject_p)
    {
        CmisPropertiesType properties = cmisObject_p.getProperties();
        List<CmisProperty> propertyList = properties.getProperty();
        return fromPropertyList(propertyList);
    }

    /**
     * Trying to find the property in given collection,
     * comparing the property definition id with {@link #getId()}.
     * @param propertyList_p List of native CmisProperty
     * @return CmisProperty or null if not found
     */
    public CmisProperty fromPropertyList(List<CmisProperty> propertyList_p)
    {
        for (CmisProperty cmisProperty : propertyList_p)
        {
            if (this.cmisId.equals(cmisProperty.getPropertyDefinitionId()))
            {
                return cmisProperty;
            }
        }
        return null;
    }

    /**
     * Retrieve the single String value from the property which should be contained
     * in the CmisObjectType.
     * <p>ATTENTION: This method will not dynamically make a retrieve from 
     * back-end system and could end up in a NullPointerException.</p>
     * @param cmisObject_p
     * @return String or null
     * @throws OwInvalidOperationException if type of property is not matching CmisPropertyId
     */
    public String getIdValue(CmisObjectType cmisObject_p) throws OwInvalidOperationException
    {
        CmisProperty property = fromObject(cmisObject_p);//FIXME what if the property not exist
        return OwCMISStaticProperty.getIdValue(property);
    }

    /**
     * Retrieve the single Boolean value form the current selected property,
     * which should be retrieved with the given CmisObjectType.
     * <p>ATTENTION: This method will not dynamically make a retrieve from 
     * back-end system and could end up in a NullPointerException.</p>
     * @param cmisObject_p CmisObjectType from where to retrieve
     * @return Boolean or null
     * @throws OwInvalidOperationException if type of property is not matching CmisPropertyBoolean
     */
    public Boolean getBooleanValue(CmisObjectType cmisObject_p) throws OwInvalidOperationException
    {
        CmisProperty property = fromObject(cmisObject_p);
        return OwCMISStaticProperty.getBooleanValue(property);
    }

    /**
     * Retrieve the single String value from the property which should be contained
     * in the CmisObjectType.
     * <p>ATTENTION: This method will not dynamically make a retrieve from 
     * back-end system and could end up in a NullPointerException.</p>
     * @param cmisObject_p
     * @return String or null
     * @throws OwInvalidOperationException if type of property is not matching CmisPropertyString
     */
    public String getStringValue(CmisObjectType cmisObject_p) throws OwInvalidOperationException
    {
        CmisProperty property = fromObject(cmisObject_p);
        return OwCMISStaticProperty.getStringValue(property);
    }

    /**
     * Comparing given id with current selected property id,
     * using String.equals(..).
     * @param id_p String representing Id to compare with
     * @return boolean true only if matching with current property id
     */
    public boolean isSameId(String id_p)
    {
        return cmisId.equals(id_p);
    }

    /**
     * Comparing given CmisPropertyDefinitionType.getId with current selected property id,
     * delegating to {@link #isSameId(String)}.
     * @param propertyDefinitionType_p CmisPropertyDefinitionType to use for comparison
     * @return boolean true only if matching with current property id
     */
    public boolean isSameId(CmisPropertyDefinitionType propertyDefinitionType_p)
    {
        return isSameId(propertyDefinitionType_p.getId());
    }

    /**
     *<p>
     * SyncList is a helper class for singleton implementation.
     * Used by the OwCMISPropertyNames, to create a list of object independent system properties.
     *</p>
     *
     *<p><font size="-2">
    * Alfresco Workdesk<br/>
    * Copyright (c) Alfresco Software, Inc.<br/>
     * All rights reserved.<br/>
    * <br/>
     * For licensing information read the license.txt file or<br/>
    * go to: http://wiki.alfresco.com<br/>
     *</font></p>
     */
    protected static class SyncList
    {
        LinkedList<String> lst;
        private static SyncList singleton;

        private SyncList()
        {
            init();
        }

        public static synchronized SyncList getInstance()
        {
            if (singleton == null)
            {
                singleton = new SyncList();
            }
            return singleton;
        }

        protected synchronized void init()
        {
            if (lst == null)
            {
                lst = new LinkedList<String>();
                OwCMISPropertyNames[] allProperties = OwCMISPropertyNames.values();
                for (int i = 0; i < allProperties.length; i++)
                {
                    if (allProperties[i].isSystemProperty())
                    {
                        lst.add(allProperties[i].getId());
                    }
                }
            }
        }

        protected List<String> getList()
        {
            return lst;
        }
    }
}