package com.wewebu.ow.server.ecmimpl.cmis.property;

import java.util.LinkedList;
import java.util.List;

import org.apache.log4j.Logger;
import org.oasis_open.docs.ns.cmis.core._200908.CmisProperty;
import org.oasis_open.docs.ns.cmis.core._200908.CmisPropertyBoolean;
import org.oasis_open.docs.ns.cmis.core._200908.CmisPropertyId;
import org.oasis_open.docs.ns.cmis.core._200908.CmisPropertyString;

import com.wewebu.ow.server.ecmimpl.cmis.log.OwLog;
import com.wewebu.ow.server.exceptions.OwInvalidOperationException;
import com.wewebu.ow.server.util.OwString2;

/**
 *<p>
 * OwCMISStaticProperty, helper class.
 * This class contains methods for checks and conversion of property values and class type.
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 */
public class OwCMISStaticProperty
{
    private static final Logger LOG = OwLog.getLogger(OwCMISStaticProperty.class);

    /**
     * Check if the types are matching, and if a cast can be done without an class cast exception.
     * @param cmisProperty_p CmisProperty which will be casted to the given class
     * @param expectedClass_p Class to which will be used for cast-operation
     * @throws OwInvalidOperationException
     */
    private static void checkType(CmisProperty cmisProperty_p, Class<? extends CmisProperty> expectedClass_p) throws OwInvalidOperationException
    {
        if (cmisProperty_p != null)
        {
            if (!expectedClass_p.isAssignableFrom(cmisProperty_p.getClass()))
            {
                String msg = "Invalid property class " + cmisProperty_p.getClass() + " - " + expectedClass_p + " was expected!";
                LOG.fatal("OwCMISStaticProperty.checkType():" + msg);
                throw new OwInvalidOperationException(new OwString2("ecmimpl.cmis.OwCMISStaticProperty.checkType.error", "Invalid property class %1 - %2 was expected!", cmisProperty_p.getClass().getName(), expectedClass_p.getName()));
            }
        }
    }

    /**
     * Get first entry or null, of type &lt;T&gt; from list.
     * @param <T> the generic type
     * @param valueList_p List of types to check, can be null
     * @return the first entry of type &lt;T&gt;, or null if given list is empty or equals <code>null</code>
     */
    private static <T> T getOne(List<T> valueList_p)
    {
        if (valueList_p == null || valueList_p.isEmpty())
        {
            return null;
        }
        else
        {
            return valueList_p.get(0);
        }
    }

    /**
     * Casting and extracting Id values as list from given CmisProperty.
     * @param cmisProperty_p CmisProperty of type CmisPropertyId, non-null value
     * @return List of Strings which representing the Id values.
     * @throws OwInvalidOperationException if given property is not a CmisPropertyId
     */
    public static List<String> getIdValues(CmisProperty cmisProperty_p) throws OwInvalidOperationException
    {
        if (cmisProperty_p == null)
        {
            return new LinkedList<String>();//FIXME How should that be processed?
        }
        checkType(cmisProperty_p, CmisPropertyId.class);
        List<String> values = ((CmisPropertyId) cmisProperty_p).getValue();
        return values;
    }

    /**
     * Returning the first Id as String from given CmisProperty, if the 
     * property is from type CmisPropertyId and has a non-empty value list.
     * @param cmisProperty_p CmisProperty from where to request the value 
     * @return String representing the Id value, or null
     * @throws OwInvalidOperationException if CmisProperty is not from type CmisPropertyId
     * @see #getIdValues(CmisProperty)
     */
    public static String getIdValue(CmisProperty cmisProperty_p) throws OwInvalidOperationException
    {
        List<String> ids = getIdValues(cmisProperty_p);
        return getOne(ids);
    }

    /**
     * Casting and extracting Boolean values as list from given CmisProperty.
     * @param cmisProperty_p CmisProperty of type CmisPropertyBoolean, non-null value
     * @return List of Boolean values or null if cmisProperty_p is null.
     * @throws OwInvalidOperationException if given property is not a CmisPropertyBoolean
     */
    public static List<Boolean> getBooleanValues(CmisProperty cmisProperty_p) throws OwInvalidOperationException
    {
        checkType(cmisProperty_p, CmisPropertyBoolean.class);
        List<Boolean> values = null;
        if (cmisProperty_p != null)
        {
            values = ((CmisPropertyBoolean) cmisProperty_p).getValue();
        }
        return values;
    }

    /**
     * Returning the first Boolean value of CmisProperty or null.
     * @param cmisProperty_p CmisProperty of type CmisPropertyBoolean, non-null value
     * @return Boolean or null if property value list is empty or null.
     * @throws OwInvalidOperationException if given property is not a CmisPropertyBoolean
     */
    public static Boolean getBooleanValue(CmisProperty cmisProperty_p) throws OwInvalidOperationException
    {
        List<Boolean> booleans = getBooleanValues(cmisProperty_p);
        return getOne(booleans);
    }

    /**
     * Casting and extracting String values as list from given CmisProperty.
     * @param cmisProperty_p CmisProperty of type CmisPropertyString, non-null value
     * @return List of String values or null if cmisProperty_p is null.
     * @throws OwInvalidOperationException if given property is not a CmisPropertyString
     */
    public static List<String> getStringValues(CmisProperty cmisProperty_p) throws OwInvalidOperationException
    {
        checkType(cmisProperty_p, CmisPropertyString.class);
        List<String> values = null;
        if (cmisProperty_p != null)
        {
            values = ((CmisPropertyString) cmisProperty_p).getValue();
        }
        return values;
    }

    /**
     * Returning the first String value of CmisProperty or null.
     * @param cmisProperty_p CmisProperty of type CmisPropertyString, non-null value
     * @return String or null if property value list is empty or null.
     * @throws OwInvalidOperationException if given property is not a CmisPropertyString
     */
    public static String getStringValue(CmisProperty cmisProperty_p) throws OwInvalidOperationException
    {
        List<String> strings = getStringValues(cmisProperty_p);
        return getOne(strings);
    }

}