package com.wewebu.ow.server.ecmimpl.cmis.ui;

import java.io.IOException;
import java.io.Writer;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;

import com.wewebu.ow.server.app.OwMainAppContext;
import com.wewebu.ow.server.conf.OwBaseUserInfo;
import com.wewebu.ow.server.ecm.OwObjectCollection;
import com.wewebu.ow.server.ecm.OwUserInfo;
import com.wewebu.ow.server.ecm.ui.OwUIUserSelectModul;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISLDAPCredentials;
import com.wewebu.ow.server.ecmimpl.cmis.OwCMISNetwork;
import com.wewebu.ow.server.ecmimpl.cmis.log.OwLog;
import com.wewebu.ow.server.exceptions.OwException;
import com.wewebu.ow.server.exceptions.OwServerException;
import com.wewebu.ow.server.ui.OwAppContext;
import com.wewebu.ow.server.util.OwHTMLHelper;
import com.wewebu.ow.server.util.ldap.OwLdapConnector;

/**
 *<p>
 * OwCMISUserSelectionModul.<br />
 * A UI component which is used to retrieve and display
 * different groups, user and/or roles.
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 * @since 3.1.0.0
 */
public class OwCMISUIUserSelectionModul extends OwUIUserSelectModul
{
    protected static final String SEARCH_FIELD_ID = "OwCMISUserSelect";
    protected static final String SEARCH_USER_LST = "OwCMISUserList";

    private static final Logger LOG = OwLog.getLogger(OwCMISUIUserSelectionModul.class);

    private String currentUserID;
    private int[] filter;

    private String lastEnteredString;
    private List<OwCMISInfoItem> infoList;

    public String getCurrentUserID()
    {
        return currentUserID;
    }

    @Override
    protected void init() throws Exception
    {
        super.init();
        infoList = new LinkedList<OwCMISInfoItem>();
    }

    public void setCurrentUserID(String currentUserID_p)
    {
        this.currentUserID = currentUserID_p;
    }

    /**
     * Return an array of int values, which represents
     * filter for the search.
     * <p>If the filter is <code>null</code> or an empty array, no
     * search will be executed.</p>
     * @return array of int, can be null
     * @see OwUIUserSelectModul#TYPE_GROUP
     * @see OwUIUserSelectModul#TYPE_ROLE
     * @see OwUIUserSelectModul#TYPE_USER
     */
    public int[] getFilter()
    {
        return filter;
    }

    /**
     * Set the filter value for the next search.
     * @param filter_p array of int representing the types
     * @see OwUIUserSelectModul#TYPE_GROUP
     * @see OwUIUserSelectModul#TYPE_ROLE
     * @see OwUIUserSelectModul#TYPE_USER
     */
    public void setFilter(int[] filter_p)
    {
        this.filter = filter_p;
    }

    public void onRender(Writer w_p) throws Exception
    {
        w_p.write("<div id=\"OwMainContent\" class=\"OwUIUserSelectModul OwCMISUserSelectModul\">\n");

        renderSearchCriteriaBox(w_p);

        renderResultList(w_p);

        renderUserSelectButton(w_p);
        w_p.write("</div>");
    }

    /**
     * Render the button for acceptance of User selection.
     * @param w_p Writer to execute rendering
     * @throws IOException 
     */
    protected void renderUserSelectButton(Writer w_p) throws IOException
    {
        w_p.write("<div class=\"block\"><input type=\"button\" onclick=\"");
        w_p.write(getFormEventURL("SelectUser", null));
        w_p.write("\" onKeyDown=\"event.cancelBubble=true\" value=\"");
        w_p.write(getContext().localize("cmis.ui.OwCMISUserSelectionModul.select", "Select"));
        w_p.write("\" name=\"selectUser\"/></div>\n");

    }

    protected void renderSearchCriteriaBox(Writer w_p) throws Exception
    {
        // === render search criteria box
        w_p.write("<div class=\"block\">\n");
        w_p.write("     <input type=\"text\" value=\"");

        if (this.lastEnteredString != null)
        {
            OwHTMLHelper.writeSecureHTML(w_p, this.lastEnteredString);
        }

        w_p.write("\" name=\"");
        w_p.write(SEARCH_FIELD_ID);
        w_p.write("\"/>&nbsp;&nbsp;");

        String sSearchEventURL = getFormEventURL("Search", null);

        w_p.write("<input type=\"button\" name=\"Search\" onClick=\"");
        w_p.write(sSearchEventURL);
        w_p.write("\" value=\"");
        w_p.write(getContext().localize("cmis.ui.OwCMISUserSelectionModul.search", "Search"));//TODO create new localized ID
        w_p.write("\" /></div>\n");
        // set focus
        getContext().setFocusControlID(SEARCH_FIELD_ID);

        // register return key to submit search
        getContext().registerKeyEvent(OwAppContext.KEYBOARD_KEY_RETURN, OwAppContext.KEYBOARD_CTRLKEY_NONE, sSearchEventURL, getContext().localize("fncm.ui.OwFNCMUIUserSelectModul.search", "Search"));
    }

    protected void renderResultList(Writer w_p) throws OwException
    {
        String selectUserEvent = getFormEventURL("SelectUser", null);
        try
        {
            w_p.write("<div class=\"block\" ondblclick=\"");
            w_p.write(selectUserEvent);
            w_p.write("\">");

            // === render result list
            w_p.write("<select class=\"OwInputfield_minWidth\" size=\"10\" ");
            w_p.write("onkeydown=\"onKey(event,13,true,function(event){");
            w_p.write(selectUserEvent);
            w_p.write("});\" name=\"");
            w_p.write(SEARCH_USER_LST);
            w_p.write("\"");
            if (getMultiselect())
            {
                w_p.write(" multiple>");
            }

            w_p.write(">");

            List<OwCMISInfoItem> lst = getInfoList();
            if (lst != null)
            {
                for (int i = 0; i < lst.size(); i++)
                {
                    OwCMISInfoItem item = lst.get(i);

                    w_p.write("<option value=\"");
                    w_p.write(Integer.toString(i));
                    w_p.write("\">");
                    w_p.write(item.getUserDisplayName());
                    w_p.write("</option>");
                }
            }

            w_p.write("</select>");
            w_p.write("</div>");
        }
        catch (IOException e)
        {
            LOG.fatal("Cannot write to outputstream.", e);
            throw new OwServerException(getContext().localize("cmis.ui.OwCMISUserSelectionModul.ioEx", "Cannot write to outputstream."), e);
        }
        catch (Exception e)
        {
            LOG.error("Cannot get displayname for rendering.", e);
            throw new OwServerException(getContext().localize("cmis.ui.OwCMISUserSelectionModul.renderEx", "Cannot get displayname for rendering."), e);
        }
    }

    @Override
    protected String usesFormWithAttributes()
    {
        return "";//render new form
    }

    /**
     * Return the last entered string, which
     * was used for searching. Can return null
     * if no search was executed.
     * @return String or null
     */
    protected String getLastEnteredString()
    {
        return this.lastEnteredString;
    }

    /**
     * Set the value which should be used for
     * search.
     * @param content_p String
     */
    protected void setLastEneteredString(String content_p)
    {
        this.lastEnteredString = content_p;
    }

    /**
     * List of OwBaseUserInfo containing the 
     * users which were retrieved with last
     * search, and filtered by defined {@link #getFilter()}.
     * @return List of OwBaseUserInfo
     */
    protected List<OwCMISInfoItem> getInfoList()
    {
        return this.infoList;
    }

    public void onSearch(HttpServletRequest request_p) throws OwException
    {
        // === clear result list
        getInfoList().clear();
        setLastEneteredString(request_p.getParameter(SEARCH_FIELD_ID));

        if (getLastEnteredString() != null)// && getLastEnteredString().length() > 0)
        {
            int[] filter = getFilter();
            if (filter != null && filter.length > 0)
            {
                for (int i = 0; i < filter.length; i++)
                {
                    switch (filter[i])
                    {
                        case OwUIUserSelectModul.TYPE_USER:
                            getInfoList().addAll(getTypeUser());
                            break;
                        case OwUIUserSelectModul.TYPE_ROLE:
                            getInfoList().addAll(getTypeRole());
                            break;
                        case OwUIUserSelectModul.TYPE_GROUP:
                            getInfoList().addAll(getTypeGroup());
                            break;
                        default:
                            getInfoList().addAll(getExtendedType(filter[i]));
                    }
                }
            }
        }
    }

    public void onSelectUser(HttpServletRequest request_p) throws Exception
    {
        String selection = request_p.getParameter(SEARCH_USER_LST);
        if (selection.length() == 0)
        {
            return;
        }

        OwCMISInfoItem info = getInfoList().get(Integer.parseInt(selection));

        if (isRoleSelection())
        {
            // === when selecting roles, return the role names only
            LinkedList<String> roles = new LinkedList<String>();
            roles.add(info.getUserName());
            getEventListner().onSelectRoles(roles);
        }
        else
        {
            // === when selecting users / groups, return the user object
            LinkedList<OwBaseUserInfo> users = new LinkedList<OwBaseUserInfo>();

            OwUserInfo selectedUser = this.getNetwork().getUserFromID(info.getUserName());
            users.add(selectedUser);

            getEventListner().onSelectUsers(users);
        }
    }

    /**
     * Method to return a list of user matching the {@link #getLastEnteredString()},
     * which will be called if {@link #getFilter()} contains  {@link OwUIUserSelectModul#TYPE_USER}.
     * <p>Must return an empty list if no matching entries could be found!</p>
     * @return List of OwBaseUserInfo
     * @throws OwException 
     * @see OwUIUserSelectModul#TYPE_USER
     */
    public List<OwCMISInfoItem> getTypeUser() throws OwException
    {
        String pattern = getLastEnteredString().trim();
        if (!pattern.contains("*"))
        {
            if (0 != pattern.length())
            {
                pattern = "*" + pattern + "*";
            }
        }
        return this.findUsersMatching(pattern);
    }

    /**
     * Method to return a list of user matching the {@link #getLastEnteredString()},
     * which will be called if {@link #getFilter()} contains  {@link OwUIUserSelectModul#TYPE_GROUP}.
     * <p>Must return an empty list if no matching entries could be found!</p>
     * @return List of OwBaseUserInfo
     * @throws OwException 
     * @see OwUIUserSelectModul#TYPE_GROUP
     */
    public List<OwCMISInfoItem> getTypeGroup() throws OwException
    {
        return requestLdap(OwCMISInfoItem.GROUP);
    }

    private List<OwCMISInfoItem> requestLdap(int type_p) throws OwException
    {
        OwCMISNetwork network = getCurrentNetwork();
        OwCMISLDAPCredentials cred = (OwCMISLDAPCredentials) network.getCredentials();
        OwLdapConnector con = cred.getLdapConnector();

        String val = con.findDistinguishedNameAnonymously(getLastEnteredString());
        LinkedList<OwCMISInfoItem> lst = new LinkedList<OwCMISInfoItem>();
        if (val != null)
        {
            if (type_p == OwCMISInfoItem.GROUP)
            {
                lst.add(createGroupInfo(val));
            }
            else
            {
                lst.add(createUserInfo(val));
            }
        }

        return lst;

    }

    private List<OwCMISInfoItem> findUsersMatching(String pattern) throws OwException
    {
        OwCMISNetwork network = getCurrentNetwork();
        OwCMISLDAPCredentials cred = (OwCMISLDAPCredentials) network.getCredentials();
        OwLdapConnector con = cred.getLdapConnector();

        OwObjectCollection dnames = con.findUserDNsMatching(pattern);
        LinkedList<OwCMISInfoItem> lst = new LinkedList<OwCMISInfoItem>();
        for (Object dname : dnames)
        {
            String displayName = con.getUserDisplayName((String) dname);
            lst.add(new OwCMISInfoItem((String) dname, displayName, OwCMISInfoItem.USER));
        }

        try
        {
            if (!dnames.isComplete())
            {
                OwMainAppContext ctx = (OwMainAppContext) getContext();
                String msg = ctx.localize1("owsearch.OwResultListView.notcomplete", "The result list does not contain all results. Only the first %1 are displayed.", String.valueOf(dnames.size()));
                ctx.postMessage(msg);
            }
        }
        catch (RuntimeException re)
        {
            throw re;
        }
        catch (Exception e)
        {
            throw new OwServerException("Fata; error!", e);
        }
        return lst;
    }

    /**
     * Method to return a list of user matching the {@link #getLastEnteredString()},
     * which will be called if {@link #getFilter()} contains  {@link OwUIUserSelectModul#TYPE_ROLE}.
     * <p>Must return an empty list if no matching entries could be found!</p>
     * @return List of OwBaseUserInfo
     * @throws OwException 
     * @see OwUIUserSelectModul#TYPE_ROLE
     */
    @SuppressWarnings("unchecked")
    public List<OwCMISInfoItem> getTypeRole() throws OwException
    {
        OwCMISNetwork network = getCurrentNetwork();
        OwCMISLDAPCredentials cred = (OwCMISLDAPCredentials) network.getCredentials();
        OwLdapConnector con = cred.getLdapConnector();
        try
        {
            LinkedList<OwCMISInfoItem> lst = new LinkedList<OwCMISInfoItem>();
            //get last entered string and replace wildcard character
            String last = getLastEnteredString().replaceAll("[:*:]", ".+");
            //always return include role names
            for (String val : getDefaultRoleNames())
            {
                if (val.matches(last))
                {
                    lst.add(createRoleInfo(val));
                }
            }

            Collection<String> col = con.getAllShortGroupNames();
            if (col != null)
            {
                for (String val : col)
                {
                    if (val.matches(last))
                    {
                        OwCMISInfoItem group = createGroupInfo(val);
                        if (!lst.contains(group))
                        {
                            lst.add(group);
                        }
                    }
                }
            }
            return lst;
        }
        catch (Exception e)
        {
            //TODO localize
            throw new OwServerException("Cannot retrieve RoleInformation", e);
        }
    }

    public boolean isRoleSelection()
    {
        int[] filter = getFilter();
        for (int i = 0; i < filter.length; i++)
        {
            if (filter[i] == OwUIUserSelectModul.TYPE_ROLE)
            {
                return Boolean.TRUE.booleanValue();
            }
        }
        return Boolean.FALSE.booleanValue();
    }

    /**
     * (overridable)
     * Method for additional type search which should be added
     * to the info list ({@link #getInfoList()}).
     * <p>This method will return by default an empty list.</p>
     * @param extendedType_p int extended type search
     * @return List of OwBaseUserInfo
     */
    public List<OwCMISInfoItem> getExtendedType(int extendedType_p)
    {
        return new LinkedList<OwCMISInfoItem>();
    }

    /**
     * Helper to return a OwCMISNetwork,
     * which cast the {@link #getNetwork()} return 
     * value to OwCMISNetwork.
     * @return OwCMISNetwork
     */
    protected OwCMISNetwork getCurrentNetwork()
    {
        return (OwCMISNetwork) getNetwork();
    }

    /**
     * Factory method to create a instance representing
     * a group with given name.
     * @param name_p String name of group
     * @return OwBaseUserInfo
     */
    protected OwCMISInfoItem createGroupInfo(String name_p)
    {
        return new OwCMISInfoItem(name_p, OwCMISInfoItem.GROUP);
    }

    /**
     * Factory method to create a instance representing
     * a user with given name.
     * @param name_p String name of group
     * @return OwBaseUserInfo
     */
    protected OwCMISInfoItem createUserInfo(String name_p)
    {
        return new OwCMISInfoItem(name_p, OwCMISInfoItem.USER);
    }

    /**
     * Factory method to create a instance representing
     * a role with given name.
     * @param name_p String name of group
     * @return OwBaseUserInfo
     */
    protected OwCMISInfoItem createRoleInfo(String name_p)
    {
        return new OwCMISInfoItem(name_p, OwCMISInfoItem.ROLE);
    }

    /**
     * Used only for rendering the list.
     */
    protected static class OwCMISInfoItem
    {
        public static final int ROLE = 1;
        public static final int USER = 2;
        public static final int GROUP = 3;

        private int type;
        private String name;
        private String displayName;

        public OwCMISInfoItem(String name_p, int type)
        {
            this(name_p, name_p, type);
        }

        public OwCMISInfoItem(String name_p, String displayName, int type)
        {
            this.name = name_p;
            this.type = type;
            this.displayName = displayName;
        }

        public String getUserDisplayName() throws Exception
        {
            return this.displayName;
        }

        public String getUserName() throws Exception
        {
            return this.name;
        }

        public boolean isGroup() throws Exception
        {
            return this.type == GROUP;
        }

        public int getType()
        {
            return this.type;
        }

        public int hashCode()
        {
            return 31 * (31 + ((name == null) ? 0 : name.hashCode())) + type;
        }

        public boolean equals(Object obj)
        {
            if (this == obj)
            {
                return true;
            }
            if (obj == null || getClass() != obj.getClass())
            {
                return false;
            }
            OwCMISInfoItem other = (OwCMISInfoItem) obj;
            if (getType() != other.getType())
            {
                return false;
            }
            if (name == null)
            {
                return other.name == null;
            }
            else
            {
                return name.equals(other.name);
            }
        }
    }
}