package com.wewebu.ow.server.ecmimpl.alfresco.bpm.rest;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Date;
import java.util.Locale;

import org.restlet.Request;
import org.restlet.Response;
import org.restlet.Uniform;
import org.restlet.data.ChallengeScheme;
import org.restlet.data.Form;
import org.restlet.data.Language;
import org.restlet.data.Parameter;
import org.restlet.data.Preference;
import org.restlet.resource.ClientResource;

import com.wewebu.ow.server.ecm.OwCredentials;
import com.wewebu.ow.server.ecm.OwNetwork;
import com.wewebu.ow.server.ecm.OwNetworkContext;
import com.wewebu.ow.server.ecmimpl.OwCredentialsConstants;
import com.wewebu.ow.server.ecmimpl.alfresco.bpm.rest.beans.WorkflowStatus;
import com.wewebu.ow.server.ecmimpl.alfresco.bpm.rest.converters.NativeValueConverter;
import com.wewebu.ow.server.ecmimpl.alfresco.bpm.rest.converters.NativeValueConverterFactoryRest;
import com.wewebu.ow.server.field.OwFieldDefinition;
import com.wewebu.ow.server.field.OwSearchCriteria;
import com.wewebu.ow.server.field.OwSearchNode;
import com.wewebu.ow.server.field.OwSearchOperator;

/**
 *<p>
 * Factory to create RESTful resources.
 *</p>
 *
 *<p><font size="-2">
 * Alfresco Workdesk<br/>
 * Copyright (c) Alfresco Software, Inc.<br/>
 * All rights reserved.<br/>
 * <br/>
 * For licensing information read the license.txt file or<br/>
 * go to: http://wiki.alfresco.com<br/>
 *</font></p>
 *@since  4.0.0.0
 */
public class AlfrescoRESTFulFactory
{
    private String baseURL;
    private OwNetwork network;
    private NativeValueConverterFactoryRest converterFactory;
    private ArrayList<Preference<Language>> acceptedLanguages;

    /**
     * @param baseURL baseURL to use for accessing Alfresco RESTful services.
     * @param network the current {@link OwNetwork}.
     */
    public AlfrescoRESTFulFactory(String baseURL, OwNetwork network)
    {
        this.baseURL = baseURL;
        this.network = network;
        this.converterFactory = new NativeValueConverterFactoryRest();

        OwNetworkContext ctx = this.network.getContext();
        Locale currentLocale = ctx.getLocale();
        Language language = Language.valueOf(currentLocale.getLanguage());

        Preference<Language> currentLanguage = new Preference<Language>(language);
        this.acceptedLanguages = new ArrayList<Preference<Language>>();
        acceptedLanguages.add(currentLanguage);
    }

    /**
     * @return a REST resource for accessing the workflow-definitions service.
     * @throws OwRestException thrown if something goes wrong with the creation of the REST resource.
     */
    public WorkflowDefinitionsResource workflowDefinitionsResource() throws OwRestException
    {
        //TODO: Add support for sharing the Client(HTTPClient) instance.
        return createResourceFor(this.baseURL + "/service/api/workflow-definitions", WorkflowDefinitionsResource.class);
    }

    /**
     * @return a REST resource for accessing the workflow-definitions service.
     * @throws OwRestException thrown if something goes wrong with the creation of the REST resource.
     */
    public WorkflowDefinitionDetailsResource workflowDefinitionDetailsResource(String wfDefinitionId) throws OwRestException
    {
        String wfDefinitionIdEncoded;
        try
        {
            wfDefinitionIdEncoded = URLEncoder.encode(wfDefinitionId, "UTF-8");
        }
        catch (UnsupportedEncodingException e)
        {
            throw new OwRestException("Could not encode the workflow definition ID!", e);
        }

        String uri = String.format("%s/service/api/workflow-definitions/%s", this.baseURL, wfDefinitionIdEncoded);
        return createResourceFor(uri, WorkflowDefinitionDetailsResource.class);
    }

    public StartWorkflowResource startWorkflowResource(String definitionId) throws OwRestException
    {
        String definitionIdEncoded = null;
        try
        {
            definitionIdEncoded = URLEncoder.encode(definitionId, "UTF-8");
        }
        catch (UnsupportedEncodingException e)
        {
            throw new OwRestException("Could not encode the workflow definition ID!", e);
        }
        String resourceUri = String.format("%s/service/ow/api/workflow-definitions/%s/start", this.baseURL, definitionIdEncoded);
        return createResourceFor(resourceUri, StartWorkflowResource.class);
    }

    public TaskInstancesResource taskInstancesResource(String owner, WorkflowStatus status, OwSearchNode filterCriteria_p, int iMaxSize_p) throws OwRestException
    {
        //TODO implement a custom WebScript for listing task instances with more details (TaskInstance.definition)
        Form form = new Form();
        form.add(new Parameter("authority", owner));
        form.add(new Parameter("state", status.toString()));
        if (iMaxSize_p >= 0)
        {
            form.add(new Parameter("maxItems", Integer.toString(iMaxSize_p)));
        }

        try
        {
            if (null != filterCriteria_p)
            {
                if (filterCriteria_p.getOperator() != OwSearchNode.SEARCH_OP_AND)
                {
                    throw new OwRestException("Only simple filters combined with the AND operator are supported for the moment.");
                }

                for (Object objSearchNode : filterCriteria_p.getChilds())
                {
                    OwSearchNode searchNode = (OwSearchNode) objSearchNode;
                    OwSearchCriteria criteria = searchNode.getCriteria();
                    OwFieldDefinition fieldDefinition = criteria.getFieldDefinition();
                    String propClassName = fieldDefinition.getClassName();
                    int operator = criteria.getOperator();

                    if (propClassName.endsWith("bpm:dueDate"))
                    {
                        Date dateValue = (Date) criteria.getValue();
                        if (null != dateValue)
                        {
                            NativeValueConverter converter = this.converterFactory.converterFor(fieldDefinition);

                            String date = (String) converter.fromJava(dateValue);
                            switch (operator)
                            {
                                case OwSearchOperator.CRIT_OP_LESS:
                                case OwSearchOperator.CRIT_OP_LESS_EQUAL:
                                    form.add(new Parameter("dueBefore", date));
                                    break;
                                case OwSearchOperator.CRIT_OP_GREATER:
                                case OwSearchOperator.CRIT_OP_GREATER_EQUAL:
                                    form.add(new Parameter("dueAfter", date));
                                    break;
                                default:
                                    throw new OwRestException("This operator is not yet supported for this property.");
                            }
                        }
                    }
                }
            }
        }
        catch (RuntimeException re)
        {
            throw re;
        }
        catch (Exception e)
        {
            throw new OwRestException("Could not process search filter.", e);
        }

        String resourceURI;
        try
        {
            resourceURI = String.format("%s/service/api/task-instances?%s", this.baseURL, form.encode());
        }
        catch (IOException e)
        {
            throw new OwRestException("Could not encode query parameters.", e);
        }

        TaskInstancesResource result = createResourceFor(resourceURI, TaskInstancesResource.class);
        return result;
    }

    public TaskInstancesResource taskInstancesForWorkflowResource(String workflowId) throws OwRestException
    {
        String resourceURI = String.format("%s/service/api/workflow-instances/%s/task-instances", this.baseURL, workflowId);
        return createResourceFor(resourceURI, TaskInstancesResource.class);
    }

    public WorkflowInstanceResource workflowInstance(String workflowInstanceId) throws OwRestException
    {
        String resourceURI = String.format("%s/service/api/workflow-instances/%s?forced=true", this.baseURL, workflowInstanceId);
        return createResourceFor(resourceURI, WorkflowInstanceResource.class);
    }

    public TaskInstanceResource taskInstanceResource(String taskInstanceId) throws OwRestException
    {
        String taskInstanceIdEncoded = null;
        try
        {
            taskInstanceIdEncoded = URLEncoder.encode(taskInstanceId, "UTF-8");
        }
        catch (UnsupportedEncodingException e)
        {
            throw new OwRestException("Could not encode the workflow definition ID!", e);
        }
        String resourceUri = String.format("%s/service/api/task-instances/%s", this.baseURL, taskInstanceIdEncoded);
        return createResourceFor(resourceUri, TaskInstanceResource.class);
    }

    public TaskInstanceEndResource taskInstanceEndResource(String taskInstanceId) throws OwRestException
    {
        String taskInstanceIdEncoded = null;
        try
        {
            taskInstanceIdEncoded = URLEncoder.encode(taskInstanceId, "UTF-8");
        }
        catch (UnsupportedEncodingException e)
        {
            throw new OwRestException("Could not encode the workflow definition ID!", e);
        }
        String resourceUri = String.format("%s/service/api/workflow/task/end/%s", this.baseURL, taskInstanceIdEncoded);
        return createResourceFor(resourceUri, TaskInstanceEndResource.class);
    }

    private String getPasswd() throws OwRestException
    {
        try
        {
            OwCredentials credentials = this.network.getCredentials();
            return credentials.getAuthInfo(OwCredentialsConstants.LOGIN_PWD);
        }
        catch (RuntimeException re)
        {
            throw re;
        }
        catch (Exception e)
        {
            throw new OwRestException("Could not get user password!", e);
        }
    }

    private String getUserName() throws OwRestException
    {
        try
        {
            OwCredentials credentials = this.network.getCredentials();
            return credentials.getAuthInfo(OwCredentialsConstants.LOGIN_USR);
        }
        catch (RuntimeException re)
        {
            throw re;
        }
        catch (Exception e)
        {
            throw new OwRestException("Could not get user ID!", e);
        }
    }

    private <T> T createResourceFor(String resourceURI, Class<T> wrappedClass) throws OwRestException
    {
        ClientResource cr = new ClientResource(resourceURI);
        cr.setChallengeResponse(ChallengeScheme.HTTP_BASIC, getUserName(), getPasswd());
        cr.getConverterService().setEnabled(true);

        T resource = cr.wrap(wrappedClass);

        final Uniform originalNext = cr.getNext();
        cr.setNext(new Uniform() {

            public void handle(Request request, Response response)
            {
                request.getClientInfo().setAcceptedLanguages(acceptedLanguages);
                if (null != originalNext)
                {
                    originalNext.handle(request, response);
                }
            }
        });

        cr.getClientInfo().setAcceptedLanguages(acceptedLanguages);
        return resource;
    }
}